.186

NAME boot
TITLE boot.asm
;------------------------------------------------------------
; Name: boot.asm
;
; Description: boot code for MCE16C board.
;
; COPYRIGHT (C) 1992 MICROTEK INTERNATIONAL, INC.
;
;
;------------------------------------------------------------

; PCB control registers ports definition:
; Chip select controller
_UMCS           EQU     0FFA0H
_LMCS           EQU     0FFA2H
_MMCS           EQU     0FFA6H  
_MPCS           EQU     0FFA8H
_PACS           EQU     0FFA4H

; Interrupt controller
_I0CON          EQU     0FF38H
_I1CON          EQU     0FF3AH
_I2CON          EQU     0FF3CH
_I3CON          EQU     0FF3CH
_TCUCON         EQU     0FF32H
_DMA0CON        EQU     0FF34H
_DMA1CON        EQU     0FF36H
_INTSTS         EQU     0FF30H
_IMASK          EQU     0FF28H
_PRIMSK         EQU     0FF2AH
_EOI            EQU     0FF22H
EOI_DATA        EQU     08000H
MASK_DMA0       EQU     0004H
MASK_DMA1       EQU     0008H
MASK_INT0       EQU     0010H
MASK_INT1       EQU     0020H
UNMASK_DMA0     EQU     0FFFBH
UNMASK_INT0     EQU     0FFEFH
UNMASK_INT1     EQU     0FFDFH

; DMA port 0 controller
_DMA0SL         EQU     0FFC0H
_DMA0SH         EQU     0FFC2H
_DMA0DL         EQU     0FFC4H
_DMA0DH         EQU     0FFC6H
_DMA0TC         EQU     0FFC8H
_DX0CON         EQU     0FFCAH

; Timer controller for WatchDog 
_T0CNT          EQU     0FF50H
_T0CMPA         EQU     0FF52H
_T0CON          EQU     0FF56H
_T2CNT          EQU     0FF60H
_T2CMPA         EQU     0FF62H
_T2CON          EQU     0FF66H

; Others
ReloadID        EQU     0A5H
NAK             EQU     015H
ACK             EQU     06H
HWRESET         EQU     03FH
P82551          EQU     0F163H  ;trace module CONTROL PORT OF 8255
P82551A         EQU     0F160H  ;trace module PORT A OF 8255
P82551B         EQU     0F161H  ;trace module PORT B OF 8255
P82551C         EQU     0F162H  ;trace module PORT C OF 8255
P82552          EQU     0F143H  ;probe module CONTROL PORT OF 8255
P82552A         EQU     0F140H  ;probe module PORT A OF 8255
P82552B         EQU     0F141H  ;probe module PORT B OF 8255
P82552C         EQU     0F142H  ;probe module PORT C OF 8255
AOBOCI          EQU     089H
AIBOCI          EQU     099H
HSWR            EQU     01H
HSCLK           EQU     02H
HSLOAD          EQU     04H
PNPGD1          EQU     0F100H   ; Port for Re-configure EREC1
PNPGD2          EQU     0F108H   ; Port for Re-configure EREC2
PNPGDT          EQU     0F110H   ; Port for Re-configure TMAN
PNXCONFIG       EQU     0f128H
;PRGM_HSEG       EQU     02000H
PRGM_HSEG       EQU     04000H  ;Dec/4/1995, tsai, change f/w program size

; ----- UART REGISTER -------
RBR             EQU     0F000H  ;Receiver Buffer Register (read only)if DLAB=0
THR             EQU     0F000H  ;Transmitter Holding Reg (write only) if DLAB=0
IER             EQU     0F001H  ;Interrupt Enable Register if DLAB=0
DLL             EQU     0F000H  ;Divisor Latch (bit0~bit7) if DLAB=1
DLM             EQU     0F001H  ;Divisor Latch (bit8~bit15) if DLAB=1
AFR             EQU     0F002H  ;Alternate Function Register if DLAB=1
IIR             EQU     0F002H  ;Interrupt Ident. Register (read only)
FCR             EQU     0F002H  ;FIFO Control Register (write only)
LCR             EQU     0F003H  ;Line Control Register
MCR             EQU     0F004H  ;Modem Control Register
LSR             EQU     0F005H  ;Line Status Register
MSR             EQU     0F006H  ;Modem Status Registe
SCR             EQU     0F007H  ;Scratch Register
;
SS_SEG  segment at 08000h
SP_PTR          EQU     00H
;
inBuf           db      400h    dup(?)  ; Input command buffer of parallel port
outBuf          db      400h    dup(?)  ; Output buffer of parallel port I/F
BUF_SIZE        EQU     400H            ; Both the input and output buffer size is 0400h
FwHdrStr        db      10       dup(?)
; Parameters
inSize          dw      0
inWrPtr         dw      0
inRdPtr         dw      0
outSize         dw      0
outWrPtr        dw      0
outRdPtr        dw      0
paraData        db      0
timeout         db      0       ; to indicate communication error or 
				; watchdog timer elapsed
rs232err        dw      0
HeaderAddr      dw      0
BodyLength      dw      0
checksum        db      0
skpchksmfg      db      0
Packet          db      0
PrgmOffset      dw      0
FwBodyLength    dw      0
LdFwLgth        dd      0
RxErrFg         db      0
dwnldfwfg       db      0
fwchksum        dw      0
FwHdrChksm      db      0
timeoutcounter  dw      0
baudrate        dw      0
baudratestr     db      4h      dup(?)

SS_SEG  ends
;
OUTPB macro _ptr,_data
	mov     dx,_ptr
	mov     al,_data
	out     dx,al
endm

WRITE_PCB macro _ptr,_data
	mov     dx,_ptr
	mov     ax,_data
	out     dx,ax
endm

WRITE_PCBax macro _ptr
	mov     dx,_ptr
	out     dx,ax
endm

READ_PCB macro _ptr
	mov     dx,_ptr
	in      ax,dx
endm

;------------------------------------------------
; This segment is executed while cpu reset.
; The actual eprom is located starting at 0FFFF0.
;------------------------------------------------
_boot   SEGMENT at 0FFFFh

; Initial Chip select nUCS signal
	WRITE_PCB       _UMCS, 0f83dh   ; Insert 1-wait while accessing EPROM
;        WRITE_PCB       _UMCS, 0f83ch   ; Insert 0-wait while accessing EPROM
					; with 27512 - 15 EPROM.
	jmp     far ptr _initialCode
;
	org     0eh
;
	db      055h    ; RAM test patterns
	db      0aah

_boot   ENDS
;------------------------------------------
; initial cpu
;------------------------------------------
DGROUP  GROUP SS_SEG
_init SEGMENT at 0F800h
   ASSUME CS:_init, DS:DGROUP, SS:DGROUP
_initialCode:
	cli
	cld
; Initial Chip select controller
	WRITE_PCB       _LMCS, 03ffch
	WRITE_PCB       _MMCS, 081fch
	WRITE_PCB       _MPCS, 0c0bfh
	WRITE_PCB       _PACS, 0f3fh
	OUTPB           P82551, AOBOCI
	OUTPB           P82552, AOBOCI
;
;
;-------------------------------------------------------------------
; Reset Emulation porcessor, Xilinx, isolate EP buffers and
; disable MTAT VRAM output control (NSE) for power on protection
;-------------------------------------------------------------------
;
;      initial cp 8255 output port, to isolate EP bus
;
	OUTPB   P82552A,11101100b       ;isoep  nforcebrk nx0cs_tg x1cs_clk
					;  1        1         1        0
					;nproge  hsload     hsclk     hswr
					;  1        1         0        0
;
;      sel0 default 1 for XILINX (RD~) configuration
	OUTPB   P82552B,00100000b       ; ***      ***   nforcerdy ncricebrk
					;  0        0         1        0
					;hldc     sel2      sel1     sel0
					;  0        0         0        0

; Let All the Xilinx chips enter tri-state (reset) mode
	OUTPB   P82551B,00110011b       ; SEL2     SEL1      SEL0   ENPROGX 
					;  0        0         1        1
					; ***      ***      SELTR    RSTCNT
					;  0        0         1        1
;
; Note: The following operation is to generate a write pulse to 
;       pull-down the following dedicated signals; 
;       so, the AL register content (value) is meaningless.
;
	MOV     DX,PNPGD1               ; Force to Re-configure EREC1
	OUT     DX,AL
	MOV     DX,PNPGD2               ; Force to Re-configure EREC2
	OUT     DX,AL
	MOV     DX,PNPGDT               ; Force to Re-configure TMAN
	OUT     DX,AL

	OUTPB   PNXCONFIG,10111000b     ; NSE      ***      M0TRT    M0TR2
					;  1        0         1        1
					;M0TR1    NXRSTT    NXRST2   NXRST0
					;  1        0         0        0

;-------------------------------------------------------------------
;       Initialize TargetHeader serial to parallel output port data
;       to reset the Emulation Processor while system bootstrapping.
;
;       ***      ***     niceres resicebrk
;        0        0         0        1
;       bmsk8    pden      ent9     ent8
;        0        0         0        0
;       ent7     ent6      ent5     ent4
;        0        0         0        0
;       ent3     ent2      ent1     ent0
;        0        0         0        0
;-------------------------------------------------------------------
_PgmSo:
	mov     bx,1000h
	mov     cx,16           ;loop 16 times
	mov     dx,P82552A
INITSO:
	rcl     bx,1
	jc      so1
SO0:    in      al,dx
	and     al,not HSWR
	jmp     SFTSO
SO1:    in      al,dx
	or      al,HSWR
SFTSO:  and     al,not HSCLK
	out     dx,al           ;serial data is sent out
	or      al,HSCLK
	out     dx,al           ;serial clk rising edge is sent out
	loop    INITSO
LDSO:
	in      al,dx
	and     al,not HSLOAD
	out     dx,al
	or      al,HSLOAD
	out     dx,al                   ;serial load rising edge is sent out
;===========================================================================
; Chech the Program checksum first
chksumloop:
	xor     ax,ax
	mov     es,ax           ; Initial from data segment 0
	mov     bx,0400h        ; Program RAM starts from 0400h, 
	mov     cx,07e00H       ; so, 07e00=(010000h-0400h)/2
	test    word ptr es:[bx],0ffffh
	je      _testRam
	xor     ax,ax
_chksumLoop1:
	add     al,es:[bx]               
	add     bx,1
	add     al,es:[bx]               
	add     bx,1
	loop    _chksumLoop1    
	mov     cx,8000h        ; 64 KB per segment
	mov     dx,es
	add     dx,01000h
	mov     es,dx           ; Set the segment data for next loop
	cmp     dx,PRGM_HSEG    ; Check the max. segment value of Program RAM
	jb      _chksumLoop1    
	or      al,al
	jne     short _testRam
_chksumOk:
        mov     ax,08000h
        mov     ss,ax
        mov     ds,ax
        mov     es,ax
        ;---------------------- To initialze pointer variable
        ;---------------------- insert  -> start : by clement
	mov     sp,SP_PTR
	mov     dx,P82552B
	in      al,dx
	and     al,07fh         ;set WRPRAM=L to write protected Program RAM
	out     dx,al
        ; Parameters initialization
	xor     ax,ax
	mov     bx,ax
	mov     cx,040H            ;initial prarmeters and variables
ClrPtrLoop:
	mov     inSize[bx],ax
	inc     bx
	inc     bx
        loop    ClrPtrLoop
        ;-----------------------           end : by clement
        nop
        call    InitVectorTable
        mov     dx,MCR
        mov     al,01h
        out     dx,al
        mov     dx,LCR
        mov     al,03h
        out     dx,al
        call    AutoBaudRate
        mov     dx,MCR
        mov     al,0
        out     dx,al
        mov     al,HWRESET
        call    PutPortData
        jmp     far ptr _mainStart

; Initial DMA to test the "Program RAM"
_testRam:
	mov     dx,P82552B
	in      al,dx
	or      al,080h         ;set WRPRAM=H to write Program RAM
	out     dx,al
	nop
	WRITE_PCB       _DMA0SL, 0fffeh
	WRITE_PCB       _DMA0SH, 0fh
	WRITE_PCB       _DMA0DL, 0h
	WRITE_PCB       _DMA0DH, 0h
	WRITE_PCB       _DMA0TC, 01000h
	WRITE_PCB       _DX0CON, 0be26h
	xor     di,di
	mov     es,di
	mov     cx,1000h                ; Block size = 4 KB
	mov     al,055h
	repe    scasb
; If error, stop
	je      _ramOk1
	jmp     _programRamErr
;
_ramOk1:
	WRITE_PCB       _DMA0SL, 0ffffh
	WRITE_PCB       _DMA0SH, 0fh
	WRITE_PCB       _DMA0DL, 0h
	WRITE_PCB       _DMA0DH, 0h
	WRITE_PCB       _DMA0TC, 01000h
	WRITE_PCB       _DX0CON, 0be26h
	xor     di,di
	mov     es,di
	mov     cx,1000h        ; Block size = 4 KB
	mov     al,0aah
	repe    scasb
; If error, stop
	je      _ramOk2
	jmp     _programRamErr

; Test the "Data RAM" with normal operation
; Pattern = 055h
_ramOk2:        
	mov     ax,08000h
	mov     es,ax
	xor     di,di
	mov     cx,1000h        ; Block size = 4 KB
	mov     al,055h
	rep     stosb
	xor     di,di
	mov     cx,1000h        ; Block size = 4 KB
	repe    scasb
; If error, stop
	je      _ramOk3
	jmp     _dataRamErr

; Pattern = 0aah
_ramOk3:        
	mov     ax,08000h
	mov     es,ax
	xor     di,di
	mov     cx,1000h        ; Block size = 4 KB
	mov     al,0aah
	rep     stosb
	xor     di,di
	mov     cx,1000h        ; Block size = 4 KB
	repe    scasb
; If error, stop
	je      _systemSync
	jmp     _dataRamErr
_programRamErr:
_dataRamErr:
;        jmp     _dataRamErr
	hlt
	nop
;
VECTOR_TABLE:
	dw      00h             ; divide error
	dw      0ffffh
	dw      00h             ; unused  type 1
	dw      0ffffh
	dw      00h             ; unused  type 2
	dw      0ffffh
	dw      00h             ; unused  type 3
	dw      0ffffh
	dw      00h             ; unused  type 4
	dw      0ffffh
	dw      00h             ; unused  type 5
	dw      0ffffh
	dw      00h             ; illegal codes  type 6
	dw      0ffffh
	dw      00h             ; unused  type 7
	dw      0ffffh
	dw      offset (WatchDog)       ; type 8
	dw      seg (WatchDog)
	dw      00h             ; unused  type 9
	dw      0ffffh
	dw      00h             ; Unused  type 10
	dw      0ffffh
	dw      00h             ; Unused  type 11
	dw      0ffffh
	dw      00h             ; Unused  type 12
	dw      0ffffh
	dw      offset (GetPutUART) ; EXT INT1 type 13
	dw      seg (GetPutUART)
	dw      00h             ; unused   type 14
	dw      0ffffh
	dw      00h             ; unused   type 15
	dw      0ffffh
;
syncString      db      'MICE'
;-----------------------------------------------
; Ready to synchronize between Host and Michelle 
;-----------------------------------------------
_systemsync:
	mov     dx,P82552B
	in      al,dx
	and     al,07fh         ;set WRPRAM=L to write protected Program RAM
	out     dx,al
	nop
	cli
	mov     ax,DGROUP
	mov     ss,ax
	mov     ds,ax
	mov     es,ax
	mov     sp,SP_PTR

; Parameters initialization
	xor     ax,ax
	mov     bx,ax
	mov     cx,040H            ;initial prarmeters and variables
setptr:
	mov     inSize[bx],ax
	inc     bx
	inc     bx
	loop    setptr
;
;--------------------------------------------------------------
; Load the Interrupt handlers to Vector area via DMA controller
;--------------------------------------------------------------
; Initial Interrupt controller
_initPara:
        call    InitVectorTable
        WRITE_PCB       _IMASK, 0FFh    ; Disable interrupts
	call    InitUARTandINT
	call    AutoBaudRate
waitsync:
	mov     RxErrFg,00h
	mov     skpchksmfg,01h
	mov     dx,IER
	mov     al,01h
	out     dx,al
	test    inSize,0ffffh   ; wait for host sends char string
	je      waitsync
	call    GetPortData
	cmp     al,03ah         ; Check the leading code of command protocol
	jne     waitsync0
	jmp     skipcmd         ; skip command record to reload program code
waitsync0:
	call    TestRxStatus
	jnc     waitsync
	xor     bx,bx           ; if not a command record, then check
				; the another leading char "M" of "MICE"
	mov     ah,syncString[bx]
	inc     bx
	cmp     al,ah
	jne     waitsync
        mov     ax,7D0h         ; 2S to receive "ICE"
	call    SetWatchDog     ; Monitor the comm port enabled
waitsync1:
	test    timeout,0ffh    ; check flag to examine WatchDog timer
	jne     waitsync        ; restart the checking sequence
	mov     dx,MCR
	mov     al,01h
	out     dx,al
	test    inSize,0ffffh   ; wait for host sends char string "MICE"
	je      waitsync1
	call    GetPortData
	mov     ah,syncString[bx]
	call    TestRxStatus
	jnc     waitsync
	inc     bx
	cmp     al,ah           ; check the other chars "ICE"
	jne     waitsync        ; restart the checking sequence
	cmp     bx,4
	jb      waitsync1       ; until matching 4 characters
	mov     skpchksmfg,00h
	jmp     syncrdy
;
skipcmd:
	mov     cx,3            ; get another 3 characters =
				; length.low + length.high + checksum
        mov     ax,7D0h         ; wait 2S to get 3 ch to skip this cmd
	call    SetWatchDog     ; Monitor the comm port enabled
skipcmd1:
	test    timeout,0ffh    ; check flag to examine WatchDog timer
	jne     waitsync        ; restart the checking sequence
	test    inSize,0ffffh   ; wait for host sends the char string
	je      skipcmd1
	call    GetPortData     ; return the port data on the reg. al
	push    ax              ; save the input character on reg. al
	call    TestRxStatus
	jnc     resync
	loop    skipcmd1
	pop     ax              ; restore the checksum byte
	pop     bx              ; restore the length.high byte
	add     ax,bx           ; calculate the command checksum
	mov     ch,bl           ; the high byte of length word
	pop     bx              ; restore the length.low byte
	add     ax,bx           ; calculate the command checksum
	mov     cl,bl           ; the low byte of length word
	or      al,al           ; checksum = 0 ?
	jne     resync          ; command record error, restart it
skipcmd2:
	test    timeout,0ffh    ; check flag to examine WatchDog timer
	jne     resync          ; restart the checking sequence
	test    inSize,0ffffh   ; wait for host sends the char string
	je      skipcmd2
	call    GetPortData
	loop    skipcmd2        ; skip all data
	mov     al,ReloadID     ; Send the reloaded ID to inform Host
	call    PutPortData
resync:
	jmp     waitsync        ; restart to check "MICE" string

naknote:
	mov     dx,MSR
	in      al,dx
	and     al,020h
	je      naknote
	mov     al,NAK
	call    PutPortData
;WaitOutData4:
;        test    outSize,0ffffh
;        jne     WaitOutData4
;
waittxd0:
	test    outSize,0ffffh
	jne     waittxd0        ; hesitates after Host being informed
	jmp     _systemsync
;
syncrdy:
	mov     dx,MSR
	in      al,dx
	and     al,020h
	je      syncrdy
	mov     al,ACK          ; inform Host --- Sync OK! ---
	call    PutPortData
	mov     dx,MCR
	mov     al,00h
	out     dx,al
	mov     RxErrFg,00h
WaitDSRH2:
	mov     dx,MSR
	in      al,dx
	and     al,020h
	je      WaitDSRH2
	call    HltWatchDog     ; Halt the WatchDog timer
	mov     skpchksmfg,01h
;WaitOutData1:
;        test    outSize,0ffffh
;        jne     WaitOutData1
dwnldprgm:
	mov     dx,P82552B
	in      al,dx
	or      al,080h         ;set WRPRAM=H to write Program RAM
	out     dx,al
	nop
	call    HltWatchDog
	mov     dx,MCR
	mov     al,01h
	out     dx,al
dwnldprgm1:
	mov     dwnldfwfg,00h
	mov     fwchksum,00h
        call    GetFwHeader     ; get cmd HEADER
	call    HltWatchDog
	test    timeout,0ffh    ; communication error happened
	je      dwnldprgm2
;        cli                     ; test code
;        hlt
;        nop
;        sti
	mov     al,NAK
	call    putportdata
;WaitOutData8:
;        test    outSize,0ffffh
;        jne     WaitOutData8
	jmp     dwnldprgm1
dwnldprgm2:                     ;    if FwHdrStr[1:8] = 0 , download ready !
	mov     dx,MCR
	mov     al,00h          ;clear DTR to reject Host PC data
	out     dx,al
	mov     dwnldfwfg,01h
	mov     al,ACK          ;inform Host -- data received and setting OK!
	call    PutPortData
	test    word ptr FwHdrStr[1],0ffffh     ;check load ok?
	jne     dwnldprgm4
	test    word ptr FwHdrStr[3],0ffffh
	jne     dwnldprgm4
	test    word ptr FwHdrStr[5],0ffffh
	jne     dwnldprgm4
	test    word ptr FwHdrStr[7],0ffffh
	je      loadrdy
dwnldprgm4:
        mov     bx,word ptr FwHdrStr[3]         ; fw pgm segement
	mov     cl,0ch
	shl     bx,cl
	mov     es,bx
        mov     bx,word ptr FwHdrStr[1]         ; fw pgm offset
	mov     PrgmOffset,bx
        mov     bx,word ptr FwHdrStr[5]         ; fw body length
	mov     FwBodyLength,bx
	mov     dx,MCR
	mov     al,01h          ; set STR to reveice data
	out     dx,al
        mov     ax,7D0h        ; wait 2s to receive fw body
	call    SetWatchDog     ; Monitor the comm port to be enabled 
dwnld0:
	test    timeout,0ffh
	jne     txnak1
	nop
	test    FwBodyLength,0ffffh    ; if data transfer count = 0
	jne     dwnld0
	nop
	test    RxErrFg,0ffh
	je      txack1
	nop
txnak1:
;        cli
;        hlt
;        nop
;        sti
	mov     al,NAK
	jmp     response1
txack1:
	mov     bx,word ptr FwHdrStr[7] ;check body checksum
	cmp     fwchksum,bx
	jne     txnak1
	mov     ax,word ptr FwHdrStr[1]
	mov     word ptr [LdFwLgth],ax
	mov     ax,word ptr FwHdrStr[3]
	mov     word ptr [LdFwLgth+2],ax
	mov     ax,word ptr FwHdrStr[5]
	add     word ptr [LdFwLgth],ax
	adc     word ptr [LdFwLgth+2],0 ;current loaded length
	mov     al,ACK
response1:
	call    putportdata
	call    HltWatchDog
	mov     RxErrFg,00h
	jmp     dwnldprgm1
loadrdy:
	mov     dx,MCR
	mov     al,00h          ; clear STR to reject data
	out     dx,al
	mov     skpchksmfg,00h
	mov     dwnldfwfg,00h
	mov     dx,P82552B
	in      al,dx
	and     al,07fh         ;set WRPRAM=L to write protected Program RAM
	out     dx,al
	nop
;
downldOK:
	mov     dx,DGROUP       ; Initial the data 
	mov     es,dx
	jmp     far ptr _mainStart

; -----------------------------------------------------------
; Initial 16552
; function: Set baud rate, 8 bits data,1 stop bit and none parity.
; input :
; output:
; modified : bx
; --------------------------------------------------------------
InitUARTandINT  proc    near
initUART:
	push    dx
	push    ax
        xor     ax,ax
	mov     dx,FCR
	out     dx,al
	mov     dx,_IMASK
	in      ax,dx
	or      ax,020h
	out     dx,ax       ; mask INT1
	mov     dx,IER
	mov     al,01h      ;enable receiver buffer interrupt
	out     dx,al
	mov     dx,LSR      ;read LSR to clear any false
	in      al,dx       ;status interrupts
	mov     dx,MSR      ;read MSR to clear any false
	in      al,dx       ;Modem interrupts
	mov     dx,IIR      ;read IIR to clear any false
	in      al,dx       ;Transmitter Holding Register Empty
	mov     dx,RBR      ;read receiver buffer register
	in      al,dx
;
;=======================================================================
; Initial interrupt controller
	mov     dx,_TCUCON
        mov     ax,02h
	out     dx,ax
	READ_PCB       _I1CON
        and     ax,018h                 ;
        or      ax,010h                 ; Set INT1 with level trigger
	WRITE_PCBax     _I1CON
	mov     dx,_IMASK
	mov     al,0DCH
;        or      al,01H     ;mask timer
	out     dx,al
	mov     dx,_EOI
	mov     ax,08000h
	out     dx,ax
	sti

; Enable UART port communication.
; MicePack was reset from cold start (or warm start with wrong program code).
; check the Host's Sync-string "MICE" which may be followed a command protocol
; while Host system is still active for user's input.
;
	xor     bx,bx
	mov     dx,MCR
	mov     al,01h
	out     dx,al
	pop     ax
	pop     dx
	ret
InitUARTandINT  endp

; --------------------------------------------------------------------------
; Set automatically baud rate to link MICEpack. Baud Rate will be half while
; it be modified every time.
; function: AutoBaudRate
; input :
; output:
; modified : DLL register
; --------------------------------------------------------------------------
AutoBaudRate    proc     near
        push    di
        push    si
	push    dx
	push    cx
        push    bx
	push    ax
        push    ds
        mov     dx,_IMASK
        mov     al,0FCH
        out     dx,al
        sti
        mov     dx,LCR
        mov     al,080h
        out     dx,al       ;enable DLAB to set baud rate and AFR
        mov     baudrate,0A14h
        mov     ax,baudrate
        mov     dx,DLL
        out     dx,al
        mov     dx,DLM
        mov     al,00h
        out     dx,al
        mov     dx,AFR
        mov     al,00h
        out     dx,al
        mov     al,03H
        mov     dx,LCR
        out     dx,al       ;set DLAB=0,8 bits,1 stop,none parity.
AutoBaudRate3:
        mov     ax,7D0H     ; 2S, shorter than SLD timerout(1S)
	call    SetWatchDog
AutoBaudRate1:                  ;wait for receive 'M' char
	test    timeout,0ffh
	jne     AutoBaudRate4
	mov     dx,LSR
	in      al,dx
	and     al,01h
	je      AutoBaudRate1
	mov     dx,RBR
	in      al,dx
	cmp     al,'M'
	jne      AutoBaudRate1
        xor     bx,bx
        mov     baudratestr[bx],al
AutoBaudRate2:                         ; receive sync char
        test    timeout,0ffh
        jne     AutoBaudRate4
        mov     dx,LSR
        in      al,dx
        and     al,01h
        je      AutoBaudRate2
        inc     bx
        mov     dx,RBR
        in      al,dx
        mov     baudratestr[bx],al
        cmp     bx,3
        jb      AutoBaudRate2
AutoBaudRate6:                  ;compare syncString with buadratestr
        cld
        mov     cx,4
        mov     ax,seg(syncString)
        mov     ds,ax
        lea     si,syncString
        mov     ax,08000h
        mov     es,ax
        lea     di,baudratestr
        repe    cmpsb
        jcxz    AutoBaudRate5
AutoBaudRate4:                  ;switch baud rate between 57600 and 115200
	mov     dx,LCR
	mov     al,080h
	out     dx,al
	mov     ax,baudrate
	xchg    ah,al
	mov     baudrate,ax
	mov     dx,DLL
	out     dx,al
        mov     dx,LCR
        mov     al,03h
        out     dx,al
        jmp     AutoBaudRate3
AutoBaudRate5:                  ;buad rate ok
	call    HltWatchDog
	mov     dx,LCR          
	mov     al,03h
	out     dx,al
        mov     dx,_IMASK
        mov     al,0DCh
        out     dx,al
	sti
        pop     ax
        mov     ds,ax
	pop     ax
        pop     bx
	pop     cx
	pop     dx
        pop     si
        pop     di
	ret
AutoBaudRate    endp

;------------------------------------------------------------------------
; Function Routine of InitVectorTable
;        --- input parameter
;        --- modified : Program RAM 0h - 400h
;        --- output :
;
;-------------------------------------------------------------------------
InitVectorTable         proc    near
        mov     dx,P82552B
        in      al,dx
        or      al,080h                 ;set WRPRAM=H to write Program RAM
        out     dx,al
        nop
        mov     ax,SEG VECTOR_TABLE     ; Extract the segment of pointer
        mov     bx,010h         ; Calculate the physical address of pointer
        mul     bx
        add     ax,OFFSET VECTOR_TABLE
        adc     dx,0
        mov     bx,dx
        WRITE_PCBax     _DMA0SL ; Put the VECTOR_TABLE to source registers
        mov     ax,bx
        WRITE_PCBax     _DMA0SH
        WRITE_PCB       _DMA0DL, 0000h  ; Set the destination address registers
        WRITE_PCB       _DMA0DH, 0000h
        WRITE_PCB       _DMA0TC, 040h   ; Transfer count=40h (0h-3fh)
        WRITE_PCB       _DX0CON, 0b626h ; source and dest addr. are increased.
        mov     dx,P82552B
        in      al,dx
        and     al,07fh         ;set WRPRAM=L to write protected Program RAM
        out     dx,al
        ret
InitVectorTable         endp
;------------------------------------------------------------------------
; Function Routine of TestRxStatus ( check if error )
;        --- input parameter
;        --- register modified :FS
;        --- output : if OK, set CF
;                     else clear CF and reset RxErrFg
;-------------------------------------------------------------------------
TestRxStatus    proc    near
	test    RxErrFg,0ffffh
	jne     FoundRxErr
	stc
	ret
FoundRxErr:
	push    ax
	push    dx
	mov     al,NAK
	call    putportdata
	mov     RxErrFg,00h
	pop     dx
	pop     ax
	clc
	ret
TestRxStatus    endp


;---------------------------------------------------------------------
; Function Routine of SetWatchDog (monitor parallel port with Timer controller)
;       Note: --- After execution this routine, place all 
;            the setting data into control registers and enable its function
;        --- input parameter   : BX     [BX = 8000 ---> 2.8 sec.] (12Mhz)
;                                       [     4000 ---> 1.4 sec.]
;        --- register modified : AX, DX
;---------------------------------------------------------------------
SetWatchDog     proc    near
	nop                     ; "ret" to disable watchdog for debugging 
	push    ax              ; save the input parameter reg. ax
	mov     timeout,0       ; Reset the timer out flag
	WRITE_PCB       _T2CON, 00000h
	WRITE_PCB       _T0CON, 02008h          ; Halt Timers
	WRITE_PCB       _T0CNT,  00h
	pop     ax      ; restore the input parameter reg. ax
	WRITE_PCBax     _T0CMPA
	WRITE_PCB       _T2CNT,  00h
        WRITE_PCB       _T2CMPA, 0BB8h
	WRITE_PCB       _T0CON, 0E009h
	WRITE_PCB       _T2CON, 0C001h          ; Let timers go
	ret

SetWatchDog             endp
;
;---------------------------------------------------------------------
; Function Routine of HltWatchDog (halt WatchDog Timer controller)
;       Note: --- After execution this routine, place all 
;            the setting default data into control registers
;        --- register modified :
;---------------------------------------------------------------------
HltWatchDog     proc    near
;
	push    ax
	push    dx
;        WRITE_PCB       _T2CON, 04001h
;        WRITE_PCB       _T0CON, 06009h          ; Halt Timers
	WRITE_PCB       _T2CON, 04000h
	WRITE_PCB       _T0CON, 06008h          ; Halt Timers
	pop     dx
	pop     ax
	ret

HltWatchDog             endp
;
;---------------------------------------------------------------------
; Function Routine of GetFwHeader (Get FW header from buffer (inBuf)
;       Note: --- if ok, after execution this routine, place all 
;            the setting data on FwHdrStr[] array
;        --- if fail, set timeout
;---------------------------------------------------------------------
GetFwHeader1:
GetFwHeader      proc    near
;
	push    bx
;        mov     ax,0600h        ; wait for leading code to set timer=131ms
;        call    SetWatchDog     ; Monitor the comm port enabled
	xor     bx,bx
	mov     timeout,0
getfwheader0:
	sti
	call    GetData
	call    TestRxStatus
	jnc     getfwheader0
	cmp     al,03ah         ;receive leading code
	jne     getfwheader0
	mov     FwHdrStr[bx],al
	mov     FwHdrChksm,al
        mov     ax,7D0h         ;   waiting 2S to get header
	call    SetWatchDog
getfwheader7:
	inc     bx
	call    GetData
	jc      getfwheader5
	mov     FwHdrStr[bx],al
	add     FwHdrChksm,al
	cmp     bx,9
	jb      getfwheader7    ; the string count = 10 bytes
getfwheader2:
	call    HltWatchDog     ; Halt the WatchDog timer
	test    inSize,0ffffh
	jne      getfwheader6       ; make sure only 10 bytes to be received
	test    FwHdrChksm,0ffh
	je      getfwheader5
getfwheader6:
	mov     timeout,0ffh
getfwheader5:
	pop     bx
	ret     
GetFwHeader      endp
;-----------------------------------------------------------------------
; Function Routine of GetData (Get port data from inBuf)
;
; ---- register modified : AX
; ---- input parameter   : none
; ---- output parameter  : if ok ,return data to AX and clear CF of FS
;                          if timeout, set CF of FS
;------------------------------------------------------------------------
GetData         proc    near
GetData1:
	test    timeout,0ffh
	je      GetData2
	stc
	cli
;        hlt
	ret
GetData2:
	test    insize,0ffffh
	je      GetData1
	call    GetPortData
	clc
	ret
GetData         endp

;---------------------------------------------------------------------
; Function Routine of GetPortData (Get port data from buffer (inBuf)
;       Note: Before calling this routine, you must check the inSize >= 0
;        --- register modified : AX
;---------------------------------------------------------------------
GetPortData1:
GetPortData     proc    near
	push    dx
	push    bx
	mov     dx,IER
	mov     al,01h
	out     dx,al
	test    inSize,0ffffh
	je      EndGetPortData
	mov     bx,inRdPtr      ; load the index of input buffer (inBuf)
	mov     al,inBuf[bx]    ;load data
	push    ax
	inc     bx
	cmp     bx,BUF_SIZE
	jb      bgetp1
	xor     bx,bx
bgetp1:
	mov     inRdPtr,bx
	mov     bx,inWrPtr
	cmp     bx,inRdPtr
	jnc     FindInSize
	add     bx,0400h
FindInSize:
	sub     bx,inRdPtr
	mov     inSize,bx
	mov     dx,MCR
	mov     al,01h          ;set DTR
	out     dx,al
	pop     ax              ; restore port data
EndGetPortData:
	pop     bx
	pop     dx
	ret     
	
GetPortData     endp
;---------------------------------------------------------------------
; Function Routine of PutPortData (Put port data to buffer (outBuf)
;       Note: Before calling this routine, make sure the buffer full?
;       ---- input parameter : AL (put data)
;        --- register modified : AX
;---------------------------------------------------------------------
PutPortData1:
PutPortData     proc    near
	push    dx
	push    bx
	push    ax
WaitOutbufSpace:
	mov     bx,BUF_SIZE     ;if outbuf is full,wait space.
	sub     bx,02h
	cmp     bx,outSize
	jc      WaitOutbufSpace
putspace:
	pop     ax
	mov     bx,outWrPtr     ; load the index of output buffer (outBuf)
	mov     outBuf[bx],al   ; load data
	inc     bx
	cmp     bx,BUF_SIZE
	jb      bputp1
	xor     bx,bx
bputp1:
	mov     outWrPtr,bx
	cmp     bx,outRdPtr
	jnc     FindOutSize
	add     bx,0400h
FindOutSize:
	sub     bx,outRdPtr
	mov     outSize,bx
	mov     dx,IER
	mov     al,02h
	out     dx,al
	pop     bx
	pop     dx
	nop
	ret     
PutPortData     endp

;---------------------------------------------------------------------
; Interrupt Service Routine of GetPutUART
; Read UART data to put inBuf[] and AL, total length of data less than 1k
; Write UART data from outBuf[] and AL,total length of data less than 1k
;        --- initial parameter: "skpchksmfg" before receive
;        --- register modified :
;        --- output : RxErrFg
;---------------------------------------------------------------------
GetPutUART  label   far
;        sti
	push    ax              ; save the working register ax
	push    dx              ; save the working register dx
	push    bx              ; save the working register bx
GetPutUART1:
	mov     dx,LSR
	in      al,dx
	mov     dx,MSR
	in      al,dx
	mov     dx,IIR          ; read interrupt identification
	in      al,dx
	cmp     al,04h
	je      RxData
	cmp     al,02h
	je     TxDataPad
	cmp     al,06h
	je      GetLineStatus
	mov     dx,IER
	in      al,dx
	cmp     al,01h
	je      RxData
	cmp     al,02h
	je      TxDataPad
;----------------------------------------------------------------------
GetLineStatus:
	mov     ah,al
	mov     dx,LSR
	in      al,dx
	mov     rs232err,ax
	mov     RxErrFg,01h
	pop     bx
	WRITE_PCB       _EOI,EOI_DATA
	pop     dx
	pop     ax
	iret
;----------------------------------------------------------------------
TxDataPad:
	jno     TxData          ;

;----------------------------------------------------------------------
RxData:
	test    dwnldfwfg,0ffh
	je      RxData1
;        jmp     RxFwData
	jne     RxFwData
RxData1:
	mov     dx,RBR
	in      al,dx           ;get data from UART
;        add     _checksum,al
	mov     bx,inWrPtr      ;set bx as index of input buffer
	mov     inBuf[bx],al    ;save data at the inBuf[]
	inc     bx
	cmp     bx,BUF_SIZE
	jb      _RxunderTop1
	xor     bx,bx           ; reset the buffer index
_RxunderTop1:
	mov     inWrPtr,bx      ; restore the buffer index
	cmp     bx,inRdPtr
	jnc     FindInSize1
	add     bx,0400h
FindInSize1:
	sub     bx,inRdPtr
	mov     inSize,bx
	mov     bx,BUF_SIZE
	sub     bx,02h
	cmp     bx,inSize
	jnc     _RxunderTop2
	mov     dx,MCR
	mov     al,00h          ;if the inBuf is full,clear DTR to reject data.
	out     dx,al
_RxunderTop2:
	pop     bx
	WRITE_PCB       _EOI,EOI_DATA
	pop     dx
	pop     ax
	iret

;----------------------------------------------------------------------
RxFwData:
	mov     dx,RBR
	in      al,dx           ;get data from UART
	mov     bx,PrgmOffset
	mov     es:[bx],al      ;put data to Program RAM.
	inc     PrgmOffset
	dec     FwBodyLength
	mov     ah,00h
	add     fwchksum,ax
	pop     bx
	WRITE_PCB       _EOI,EOI_DATA
	pop     dx
	pop     ax
	iret
;----------------------------------------------------------------------
TxData:
;        mov     dx,MSR
;        in      al,dx
;        and     al,020h
;        jz      _TxunderTop3    ;if DSR=L,end I.S.R. ,else send data.
	mov     bx,outRdPtr             ;set bx as index of output buffer
	mov     dx,THR
	mov     al,outBuf[bx]
	out     dx,al
	inc     bx
	cmp     bx,BUF_SIZE
	jb      _TxunderTop4
	xor     bx,bx           ; reset the buffer index
_TxunderTop4:
	mov     outRdPtr,bx     ; restore the buffer index
	mov     bx,outWrPtr
	cmp     bx,outRdPtr
	jnc     FindOutSize1
	add     bx,0400h
FindOutSize1:
	sub     bx,outRdPtr
	mov     outSize,bx
	or      bx,bx
	jne     _TxunderTop3    ;if outSize is empty
	mov     dx,IER          ;disable Transmitter Holding Register Empty
				;interrupt,enable Received Data Available
				;interrupt
;        mov     al,00h
	mov     al,01h
	out     dx,al
_TxunderTop3:
	pop     bx
	WRITE_PCB       _EOI, EOI_DATA
	pop     dx
	pop     ax
	iret

;---------------------------------------------------------------------
; Interrupt Service Routine of WatchDog (Watch timer of serial port)
; Once this interrupt happened, communication error occurred while RxD
;        --- register modified : AX, DX
;---------------------------------------------------------------------
WatchDog        label   far
	push    ax              ; save the working register ax
	push    dx              ; save the working register dx
	call    HltWatchDog     ; Halt the WatchDog timer
WatchDog1:
	mov     timeout,0ffh    ; Set flag to indicate WatchDog captures error.
	inc     timeoutcounter
	WRITE_PCB       _EOI, EOI_DATA
	pop     dx
	pop     ax
	iret
_init   ENDS

;--------------------------------------------
; jump to the main routine of Control Program
;--------------------------------------------
_dispatch SEGMENT       at      040h
	assume  cs:_dispatch

_mainStart      label   far

_dispatch       ENDS

	END     _initialCode

;******************************* E O F ***********************************/
