/***************************************************************************
**
**    Name:    dasm.h
**
**    Description:
**        Routines for the disassembly, assembly and dequeue support.
**        The prefix used on these routines is Dad which stands for
**        Disassembler-Assembler-Dequeuer.
**
**    $Log:   S:/tbird/mt2_68k/dad32/dasm.h_v  $
** 
**    Rev 1.0   13 Feb 1997 08:45:08   gene
** Initial revision.
** 
**    Rev 1.1   12 Mar 1996 09:53:20   gene
** changed to large model
** 
**    Rev 1.0   07 Sep 1995 10:16:46   gene
** Initial revision.
**  
**     Rev 1.28   22 Jul 1993 10:42:32   ernie
**  Added support for backward dequeueing
**  
**     Rev 1.27   09 Jun 1993 08:29:30   ernie
**  Added new function DadDasmIsTransfer()
**  
**     Rev 1.26   25 May 1993 11:22:58   ernie
**  DasmSym is now a system-global rather than DESCRIPTOR-specific
**  
**     Rev 1.25   29 Mar 1993 09:55:16   ernie
**  Added ifdef'ed extern C stuff to support c++
**  
**     Rev 1.24   16 Dec 1992 15:36:44   mindy
**  moved dq to fw
**  
**     Rev 1.23   09 Dec 1992 10:36:08   mindy
**  changed forwardDqInst routine to include text len
**  
**     Rev 1.22   30 Oct 1992 18:16:58   john
**  fixed prototype for daddasminstbyline
**  
**     Rev 1.21   31 Aug 1992 09:41:04   marilyn
**  Changed the interface for DadForwardDqInst to use a HANDLE instead of
**  a TMalloc buffer.  The trace server requires a HANDLE.
**  
**     Rev 1.20   21 Jul 1992 15:50:20   marilyn
**  Added DadCallback.
**  
**     Rev 1.19   16 Jul 1992 17:22:56   marilyn
**  Changed interface for DadAsm.
**  
**     Rev 1.18   10 Jul 1992 17:23:40   marilyn
**  Added DadDasmInstByLine.
**  
**     Rev 1.17   09 Jul 1992 16:58:00   marilyn
**  Now using range descriptor for range routines and added
**  DadDasmGetRangeOffset.
**  
**     Rev 1.16   24 Jun 1992 08:45:48   marilyn
**  Clean up and new interfaces.
**
**     Rev 1.15   01 May 1992 13:32:42   brucea
**  Updated 3 headers to be more accurate and informative
**
**     Rev 1.14   27 Apr 1992 15:54:38   mindy
**  Changed interface to DasmOpen
**
**     Rev 1.13   03 Apr 1992 11:54:30   mindy
**  Added two new routines used by trace to get an address from a given frame.
**
**     Rev 1.12   01 Apr 1992 06:38:48   mindy
**  a) added number of instructions parameter to get previous dasm text
**  b) removed get previous dq instructions since we are not currently
**     supportly this.
**
**     Rev 1.11   13 Mar 1992 08:15:02   mindy
**  changed DadGetDqInst interface to get multiple inst
**
**     Rev 1.10   06 Mar 1992 05:55:40   mindy
**  added FAR to bufLen
**  
**     Rev 1.9   04 Mar 1992 13:44:12   tom
**  Added string length parameter to DadDasmInst, DadDasmPrevInst.
**  
**     Rev 1.8   03 Mar 1992 12:09:56   mindy
**  changed dasm interface for memory presenter
**
**     Rev 1.7   27 Feb 1992 12:41:52   mindy
**  a) added a couple routines
**  b) address now passed around as descriptors not ADDR pointers.
**  c) removed internal state structures - the world doesn't need to
**     know about these.
**  
**     Rev 1.6   22 Jan 1992 13:11:34   courtney
**  removed copyaddr prototype which should not be exported.
**  
**     Rev 1.5   15 Jan 1992 14:12:38   tom
**  Additional changes required by Hsinchu code.
**  
**     Rev 1.4   15 Jan 1992 13:36:36   tom
**  Added Hsinchu changes.
**  
**     Rev 1.3   20 Nov 1991 10:50:52   mindy
**  removed setting and getting of dq trace buffer and frame number.  These
**  values are stored in the trace server - so there's no reason to call the
**  dq server which would just turn around and call the trace server.
**  
**     Rev 1.2   07 Nov 1991 14:34:58   mindy
**  made changes needed to dequeuer routines to support trace presenter.
**  
**     Rev 1.1   30 Jul 1991 13:50:04   nghia
**  added changes for "basic system release"
** 
**    Rev 1.0   14 Jun 1991 13:34:24   jim
** Initial revision.
**
**    $Header:   S:/tbird/mt2_68k/dad32/dasm.h_v   1.0   13 Feb 1997 08:45:08   gene  $
*/
#ifndef DAD_SERVER
#define DAD_SERVER

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _BASETYPE_
#include "basetype.h"
#endif

#ifndef _EVTTMPLT_
#include "evttmplt.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* typedef UWORD DESCRIPTOR; */
typedef enum {DEFAULT} PROCESSOR_TYPE;
typedef enum {MEMORY,TRACEBUFFER} DATATYPE;
typedef enum {BYTESIZE, WORDSIZE, LONGSIZE} MEM_SIZE;
typedef enum {INST_NONE, INST_CALL, INST_RETURN} CALL_TYPE;
typedef enum {CALL_RET, CONTROL_FLOW, INST_ONLY, NO_FILTER, DQF_DUMMY=0x7fff}
   DQ_FILTER;

typedef struct {
   BOOLEAN transfer;
   BOOLEAN call;
   BOOLEAN destKnown;
} TRANSFER_TYPE;

/**************************************************************************
**
**  DadAsmOpen
**
** Description:
**   Allocates a memory block to contain information about conditions the
**   assembler needs to know about its environment.  This environment is
**   totally unique for each calling server or presenter.  Returns an
**   id to uniquely identify each caller.  The start address specifed
**   modifies the disassembly address for the caller.  The input address
**   descriptor is NOT destroyed, the caller is responsible for the address
**   descriptor.
**
** Parameters:
**   input:
**      startAddr:  address descriptor to start assembling at
**   output:
**      id:  id number assigned to calling routine.
**
** Errors:
**   system errors i.e. TMalloc, Address etc.
**
***************************************************************************/
RETCODE EXPORT DadAsmOpen(DESCRIPTOR FAR *id, DESCRIPTOR startAddr);


/**************************************************************************
**
**  DadAsm
**
** Description:
**   Takes the incoming string of text and attempts to assembles it into
**   a valid instruction.  After determining if the string contains a valid
**   instruction the opcode bytes are calculated and WRITTEN TO MEMORY.
**   The total number of bytes needed to code the instruction is returned
**   in 'numBytes'.  The assembler also returns a formatted version of the
**   valid instruction back to the calling routine in 'outInst'.  The
**   address kept in the state table is incremented by the number of
**   opcode bytes. Symbols are always "enabled".  The assembler parses
**   the incoming string and if a symbol it is passed to the symbol
**   table server.
**
** Parameters:
**   input:
**      id:       id number assigned to calling routine
**      inInst:   Text string of user input to assemble.  Buffer
**                is a NULL terminated text string that is
**                allocated by caller and can be any length.
**   output:
**      outInst:  Formatted version of instruction.  String is TMalloced
**                by server and must be TFreed by the caller.
**      numBytes: number of bytes consumed by instruction
**
** Errors:
**   system errors, invalid id.
**
****************************************************************************/
RETCODE EXPORT DadAsm(DESCRIPTOR id, LPSTR inInst,
                      LPSTR *outInst,LPWORD numBytes);


/***************************************************************************
**
** DadSetAsmAddr
**
** Description:
**   Sets the starting address to 'address' in the state table for the
**   given calling routine. The address descriptor is NOT destroyed.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      address:  new starting address descriptor
**   output:
**      none
**
** Errors:
**   system errors, invalid id.
**
***************************************************************************/
RETCODE EXPORT DadSetAsmAddr(DESCRIPTOR id, DESCRIPTOR address);


/**************************************************************************
**
** DadGetAsmAddr
**
** Description:
**   Returns a copy of the current descriptor for the starting address
**   found in the state for the identified caller.  The returned descriptor
**   must be destroyed by the caller.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      address: current starting address
**
** Errors:
**   system errors only
**
***************************************************************************/
RETCODE EXPORT DadGetAsmAddr(DESCRIPTOR id, DESCRIPTOR FAR *address);


/***************************************************************************
**
** DadAsmClose
**
** Description:
**   Destroys the id descriptor of the caller and removes the
**   descriptor's state information.
**
** Parameters:
**   input:
**      id:  id number of descriptor to close
**   output:
**      none
**
** Errors:
**   system errors, invalid id
**
***************************************************************************/
RETCODE EXPORT DadAsmClose(DESCRIPTOR id);


/***************************************************************************
**
** DadDasmOpen
**
** Description:
**   Allocates a memory block to contain information about conditions the
**   disassembler needs to know about its environment.
**   The starting address modifies the default address
**   to begin disassembly for the caller.  The address descriptor is NOT
**   destroyed.
**
** Parameters:
**   input:
**     startAddr:  address descriptor of address to start disassemble at
**   output:
**      id:  id number assigned to calling routine.
**
** Errors:
**   system errors only
**
***************************************************************************/
RETCODE EXPORT DadDasmOpen(DESCRIPTOR FAR *id, DESCRIPTOR startAddr);


/*************************************************************************
**
** DadSetDasmSymbol
**
** Description:
**   Enables or disables the displaying of symbols in the output of the
**   disassembler.  Dasmsym is a system-wide global affecting all disassembly
**   output (memory presenter, source presenter, shell, and trace presenter).
**
** Parameters:
**   input:
**      enable:  TRUE to enable symbol output.
**   output:
**      none
**
**************************************************************************/
RETCODE EXPORT DadSetDasmSymbol(BOOLEAN enable);


/**************************************************************************
**
** DadGetDasmSymbol
**
** Description:
**   Returns if symbols are enabled or disabled in the display output of the
**   disassembler.
**
** Parameters:
**   output:
**      enable:  TRUE if symbols are enabled, FALSE if symbols are disabled.
**
****************************************************************************/
RETCODE EXPORT DadGetDasmSymbol(BOOLEAN FAR *enable);


/*************************************************************************
**
** DadDasmInst
**
** Description:
**   Uses the address found in the state information for the given calling
**   routine as the address to start disassembling the next instruction at.
**   The disassembled instruction is returned in 'inst'.  The address is
**   included in the text returned.  The state table address is updated to
**   point to the next instruction.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      noInst:  number of instructions to return.
**   output:
**      buffer:  Text string that contains the address and instruction
**               disassembly.  This memory is malloc'd by this routine
**               and free'd by the caller. The buffer is NULL terminated
**               but may contain multiple lines. Mo maximum size of the
**               buffer except 64K for the TMalloc.
**      bufLen:  String length of buffer.
**
** Errors:
**   system errors, invalid id
***************************************************************************/
RETCODE EXPORT DadDasmInst(DESCRIPTOR id, U16 noInst, LPSTR *buffer,
                           U16 FAR *bufLen);
/*************************************************************************
**
** DadDasmInstByLine
**
** Description:
**   Uses the address found in the state information for the given calling
**   routine as the address to start disassembling the next instruction at.
**   The number of lines requested is disassembled but an instruction is
**   never broken up.  Therefor if 20 lines are requested, it is possible
**   that 20,21, or at most 22 lines would be returned.  There are 3 lines
**   maximum for any given instruction.
**   The state table address is updated to point to the next instruction.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      noInst:  number of lines required as a minumum.
**   output:
**      buffer:  Text string that contains the address and instruction
**               disassembly.  This memory is malloc'd by this routine
**               and free'd by the caller. The buffer is NULL terminated
**               but may contain multiple line instructions. Mo maximum
**               size of the buffer except 64K for the TMalloc.
**      bufLen:  String length of buffer.
**
** Errors:
**   system errors, invalid id
***************************************************************************/
RETCODE EXPORT DadDasmInstByLine(DESCRIPTOR id, U16 lines, LPSTR *buffer,
                           U16 FAR *bufLen);


/*************************************************************************
**
** DadDasmRangeOfInst
**
** Description:
**   Uses the start address provided as the address to start disassembling
**   instructions at.  Uses the end address to stop dissassembling.
**   The addresses of all the instructions are included in the text
**   returned.  The state table address is updated to point to the
**   next instruction available to disassemble.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      addrRange :  address descriptor containing the start and end
**                   addresses for disassembly.  This descriptor is
**                   allocated by the caller and freed by dasm even when
**                   an error is returned.  The  startAddress must be
**                   less than the ending address.
**   output:
**      buffer:  Text string that contains the address and instruction
**               disassembly.  This memory is malloc'd by this routine
**               and free'd by the caller.  The buffer is NULL terminated
**               and may contain multiple lines.  If buffer is NULL then
**               there is no valid data to display.
**      bufLen:  String length of buffer. If 0 then no valid data.
**
**  Errors:
**      system errors, invalid id, invalid address descriptor
**      If an error is returned and the buffer is NULL then no valid
**      data is available.  If an error is returned but the buffer is not
**      NULL, then valid data is returned in the buffer.  In this case
**      some kind of error occured during cleanup i.e. TFree or AdrDestroy.
**
***************************************************************************/
RETCODE EXPORT DadDasmRangeOfInst(DESCRIPTOR id, DESCRIPTOR addrRange,
                                  LPSTR *buffer, U16 FAR *bufLen);


/***************************************************************************
**
** DadDasmPrevInst
**
** Description:
**   Uses the address found in the state information for the given calling
**   routine as the address to start disassembling the previous instruction.
**   The disassembled instruction is returned in 'buffer". The address is
**   included in the text returned.  With symbols this routine could back
**   back up to the closest previous label and return the instruction
**   just prior to the starting address.  Without symbols the disassembler
**   would need to back up one byte, do a disassembly of the instruction
**   starting at that location then check to see if that instruction
**   ended at the start address.  If the instruction doesn't line up the
**   disassembler would need to back up another byte and try again.
**   It would need to repeat this sequence until the boundary condition
**   is met.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      noInst:  number of instructions to return.
**   output:
**      buffer:  Text string that contains the address and instruction
**               disassembly.  This memory is malloc'd by this routine and free'd
**               by the caller.  The buffer is NULL terminated and may
**               contain multiple lines.
**      bufLen:  String length of buffer.
**
** Errors:
**   system errors, at top of memory
**
**************************************************************************/
RETCODE EXPORT DadDasmPrevInst(DESCRIPTOR id, U16 noInst, LPSTR *buffer,
                               U16 FAR *bufLen);


/*************************************************************************
**
** DadDasmGetRangeInstCount
**
** Description:
**   Given an address range (start and end addresses) determine the
**   number of disassembled instruction within the range.  Also determine
**   the number of lines for the instructions within the range.
**   The state table address is NOT updated.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      addrRange: address descriptor of the start and end addresses for
**                     disassembly.  This descriptor is allocated by
**                     the caller and freed by dasm.
**   output:
**      instCount: the number of instructions within the range (inclusive).
**      lineCount: the number of lines of instructions within the range.
**
** Errors:
**   system errors, invalid id, invalid address descriptors, invalid
**   instruction address
**
***************************************************************************/
RETCODE EXPORT DadDasmGetRangeInstCount(DESCRIPTOR id,
                                        DESCRIPTOR addrRange,
                                        U16 FAR *instCount,
                                        U16 FAR *lineCount);


/*************************************************************************
**
** DadDasmGetRangeOffset
**
** Description:
**   Given a line number count and using the current address offset,
**   determine the next address offset and the actual number of lines
**   required for display of the range.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      lines : number of lines to use in determining the next offset.
**   output:
**      nextOffset: then offset following the actual number of lines
**         returned in an address descriptor allocated by Dasm and
**         destroyed by the caller.
**      lineCount: the actual number of lines required for the range    
**         between the current address offset and the next offset.
**
** Errors:
**   system errors, invalid id
**
***************************************************************************/
RETCODE EXPORT DadDasmGetRangeOffset(DESCRIPTOR id,
                                     U16 lines,
                                     DESCRIPTOR FAR *nextOffset,
                                     U16 FAR *lineCount);


/**************************************************************************
**
** DadSetDasmAddr
**
** Description:
**    Sets the starting address (stored in the internal state table),also
**    used by a number of other dasm functions. The address descriptor
**    is NOT destroyed.
**
** Input parameters:
**    id: identifier created by DadDasmOpen used to reference internal state
**        variables
**
**    address: current starting address descriptor to be set.
**             This address IS NOT consumed by this function.
**             It is the responsibility of the caller to TFree their own
**             local address passed in.
**
** Output parameters: none
**
** Errors:
**   system errors, invalid id
**
***************************************************************************/
RETCODE EXPORT DadSetDasmAddr(DESCRIPTOR id, DESCRIPTOR address);


/***************************************************************************
**
** DadGetDasmAddr
**
** Description:
**    Returns a copy of the current descriptor of the starting address
**    found in the state table for the identified caller.  Also used by
**    a number of other internal dasm functions.
**
** Input parameters:
**    id: identifier created by DadDasmOpen used to reference internal state
**        variables
**
** Output parameters:
**    address: current starting address.  This address structure is created
**             by this function.  It is the responsibility of the caller
**             to TFree it when finished with it.
**
** Errors:
**   system errors only
**
****************************************************************************/
RETCODE EXPORT DadGetDasmAddr(DESCRIPTOR id, DESCRIPTOR FAR *address);


/***************************************************************************
**
** DadDasmIsCallOrRTS
**
** Purpose: determines whether the present instruction is a call or return.
**    "Present instruction" is that pointed to by the internal <startAddr>
**    state.  To set this address, call DadSetDasmAddr first!
**
** Input parameters:
**    id: identifier created by DadDasmOpen used to reference internal state
**        variables
**
** Output parameters:
**    calledAddr: address descriptor previously created by caller.  This
**       function does not create any new addresses.  The <calledAddr>
**       address structure is filled out by DadDasmIsCallorRTS.  The value
**       is either:
**          1) the next instruction address if the looked up instruction
**             is NOT a call, or
**          2) the address of the destination of the call instruction
**
**    type: choices: INST_NONE -> not a call or return
**                   INST_CALL -> a call instruction found
**                   INST_RETURN -> a return instruction found
**
****************************************************************************/
RETCODE EXPORT DadDasmIsCallOrRTS(DESCRIPTOR id, DESCRIPTOR calledAddr,
                                  CALL_TYPE FAR *type);


/***************************************************************************
**
** DadDasmIsTransfer
**
** Purpose:
**    Returns information about whether the current instruction is a tranfer
**    (e.g. call, return, trap, branch), and information about the transfer.
**    "Present instruction" is that pointed to by the internal <startAddr>
**    state.  To set this address, call DadSetDasmAddr first!
**
** Input parameters:
**    id: identifier created by DadDasmOpen used to reference internal state
**        variables
**
** Output parameters:
**    transferAddr: address descriptor previously created by caller.  This
**       function does not create any new addresses.  The <transferAddr>
**       address structure is filled out by DadDasmIsTransfer.  The value
**       is either:
**          1) the address of the destination of the transfer instruction
**             if the destination is known, or
**          2) left alone.
**
**    type: information about transfer instruction.
**    id:   current dasm address is updated by this function.
**
****************************************************************************/
RETCODE EXPORT DadDasmIsTransfer(DESCRIPTOR id, DESCRIPTOR transferAddr,
                                 TRANSFER_TYPE FAR *type);


/***************************************************************************
**
** DadDasmClose
**
** Description:
**   Removes indicated descriptor's state information.
**
** Parameters:
**   input:
**      id:  id number of descriptor to close
**   output:
**      none
**
***************************************************************************/
RETCODE EXPORT DadDasmClose(DESCRIPTOR id);


/***************************************************************************
**
** DadDqOpen
**
** Description:
**   Allocates a memory block to contain information about conditions the
**   dequeuer needs to know about its environment. The traceId is the
**   descriptor to the trace server to use.  The trace server will have
**   the trace buffer and frame number information.  By default symbols
**   are OFF, filter is NO_FILTER and text search mode is OFF.
**
** Parameters:
**   input:
**      traceId:  trace server id.
**   output:
**      id:  id number assigned to calling routine.
**
** Errors:
**   system errors, invalid id
**
***************************************************************************/
RETCODE EXPORT DadDqOpen(DESCRIPTOR FAR *id, DESCRIPTOR traceId);


/*************************************************************************
**
** DadSetDqSymbol
**
** Description:
**   Enables or disables the displaying of symbols in the output of the
**   dequeuer.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      enable:  TRUE to enable symbol output, FALSE to disable.
**   output:
**      none
**
** Errors:
**   invalid id only
**
***************************************************************************/
RETCODE EXPORT DadSetDqSymbol(DESCRIPTOR id, BOOLEAN enable);


/**************************************************************************
**
** DadGetDqSymbol
**
** Description:
**   Returns if symbols are enabled or disabled in the display output
**   of the dequeuer.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      enable:  TRUE if symbols are enabled, FALSE if disabled
**
** Errors:
**   invalid id only
**
***************************************************************************/
RETCODE EXPORT DadGetDqSymbol(DESCRIPTOR id, BOOLEAN FAR *enable);


/*************************************************************************
**
** DadForwardDqInst
**
** Description:
**   Uses the trace buffer information found in the state information
**   for the given calling routine as the trace frame to start dequeuing
**   from for the next instruction. The ASCII text of the dequeued
**   trace frames is returned in 'trcInst'.  The ASCII text includes
**   the frame number.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      numInst:  number of instructions or data frames to get
**   output:
**      startFrame:  starting frame number
**      endFrame:   ending frame number
**      buffer:   Text string that contains the frame number and
**                instruction dequeuing information to display.
**                This text string can and will often be multiple
**                lines of text.  The memory is allocated by the
**                server using TMalloc and the calling routine is
**                responsible for deallocating.
**
** Errors:
**   system errors,invalid id
**
***************************************************************************/
RETCODE EXPORT DadForwardDqInst(DESCRIPTOR id,U32 numInst,S32 FAR *startFrame,
                        S32 FAR *endFrame, LPSTR *buffer, U16 FAR *textLen);

RETCODE EXPORT DadBackwardDqInst(DESCRIPTOR id,U32 numInst,S32 FAR *startFrame,
                        S32 FAR *endFrame, LPSTR *buffer, U16 FAR *textLen);

/***************************************************************************
**
** DadDqFlush
**
** Description:
**   Flushes the state of the dequeuer.  This causes the dequeuer to do a
**   fresh read from the trace buffer the next time a dq get instruction
**   call is made.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      none
**
***************************************************************************/
RETCODE EXPORT DadDqFlush(DESCRIPTOR id);



/**************************************************************************
**
** DadSetDqOffset
**
** Description:
**   Sets the byte offset within the starting trace buffer frame to
**   'offset' in the state table for the given calling routine.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      offset:  new starting trace buffer frame byte offset
**   output:
**      none
**
** Errors:
**   invalid id only
**
***************************************************************************/
RETCODE EXPORT DadSetDqOffset(DESCRIPTOR id, U16 offset);


/*************************************************************************
**
** DadGetDqOffset
**
** Description:
**   Returns the current value of the starting trace buffer frame byte
**   offset found in the state table for the given calling routine.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      frame:  current starting trace buffer frame byte offset
**
** Errors:
**   invalid id only
**
***************************************************************************/
RETCODE EXPORT DadGetDqOffset(DESCRIPTOR id, U16 FAR *offset);


/************************************************************************
**
** DadSetDqFilter
**
** Description:
**   Sets up the conditions for the filtering method to be used by the get
**   next and previous dequeued instruction.  CALL_RET - returns only
**   CALL and RETURN instructions. CONTROL_FLOW would return all control
**   flow instructions. INST_ONLY returns all instruction trace frames
**   but doesn't include the corresponding data frames. NO_FILTER applies
**   no filter.  Defaults to NO_FILTER.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      filter:  filtering method to use
**   output:
**      none
**
***************************************************************************/
RETCODE EXPORT DadSetDqFilter(DESCRIPTOR id, DQ_FILTER filter);


/************************************************************************
**
** DadGetDqFilter
**
** Description:
**   Returns the current conditions of the filtering method being used
**   by the get next and previous dequeued instruction.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      filter:  filtering method in use
**
****************************************************************************/
RETCODE EXPORT DadGetDqFilter(DESCRIPTOR id, DQ_FILTER FAR *filter);


/**************************************************************************
**
** DadSetDqSearchPattern
**
** Description:
**   Sets up the text string pattern used to find the next or previous
**   trace instruction if search mode is active.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      pattern:  pattern to search for
**   output:
**      none
**
**************************************************************************/
RETCODE EXPORT DadSetDqSearchPattern(DESCRIPTOR id, LPSTR pattern);


/*************************************************************************
**
** DadGetDqSearchPattern
**
** Description:
**   Returns the currently active text string pattern used when search mode
**   is active.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      pattern:  pattern being searched for
**
***************************************************************************/
RETCODE EXPORT DadGetDqSearchPattern(DESCRIPTOR id, LPSTR pattern);


/**************************************************************************
**
** DadSetDqSearch
**
** Description:
**   Activates or deactivates the searching mode.  Once activated
**   each subsequent call to DadGetDqInst or DadGetPrevDqInst
**   must match the search condition to return an instruction.  If
**   search is deactivated then the DadGetDqInst and DadGetPrevDqInst
**   routines only have to worry about the filtering conditions. Search
**   is disabled by default.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**      enable:  TRUE to activate search pattern
**   output:
**      none
**
**************************************************************************/
RETCODE EXPORT DadSetDqSearch(DESCRIPTOR id, BOOLEAN enable);


/**************************************************************************
**
** DadGetDqSearch
**
** Description:
**   Returns status of search.  TRUE if search is currently active.
**
** Parameters:
**   input:
**      id:  id number of calling routine.
**   output:
**      enable:  TRUE if search active
**
***************************************************************************/
RETCODE EXPORT DadGetDqSearch(DESCRIPTOR id, BOOLEAN FAR *enable);


/**************************************************************************
**
** DadDqClose
**
** Description:
**   Destroys the id descriptor and removes the indicated descriptor's
**   state information.
**
** Parameters:
**   input:
**      id:  id number of descriptor to close
**   output:
**      none
**
**************************************************************************/
RETCODE EXPORT DadDqClose(DESCRIPTOR id);

/****************************************************************************
**
**   DadCallback
**
**   Purpose:
**      Function to handle event notification callback.
**
**   Input Parameters:
**      eventNum:  the identifier of the event causing the callback
**
**   Output Parmeters:
**      none
**
***************************************************************************/
RETCODE EXPORT DadCallback(U32 eventNum);

#ifdef __cplusplus
}
#endif
#endif
/****************************** E O F ******************************/
