/****************************************************************************
**
** Name:  proc.c
**
** Currently supports: PPC603E, PPC860,
                       M68330,331,332,F333,334,340,349,360
**                     M68306,M68307,M68328
**                     M68040,EC040,LC040
**                     M68HC16 Z1/Z2/Y1
**                     I80186, XL/EA/EB/EC
**                     I80188, XL/EA/EB/EC
**                     I80386 CX/SX/EX/DX
**                     I80486 SX/DX
**
** Description:
**    Processor-specific calls to allow all other modules to be processor
**    independent
**
** $Log:   S:/tbird/arcppc/proc/proc.c_v  $
** 
**    Rev 1.3   08 Apr 1998 10:46:44   hera
** For PPC821
** 
**    Rev 1.2   18 Mar 1998 14:40:32   hera
** For PowerPC .cfg file.
** 
**    Rev 1.1   17 Jan 1997 14:27:22   kevin
** added PowerPC family
** 
**    Rev 1.0   17 Jan 1997 09:23:18   kevin
** Initial revision.
** 
**    Rev 1.1   17 Dec 1996 10:49:08   gene
** added 5204
** 
**    Rev 1.0   03 Jun 1996 11:36:14   gene
** Initial revision.
** 
**    Rev 1.1   06 Nov 1995 09:59:14   kevin
** added 68307 and 68328 cpu types
** 
**    Rev 1.0   07 Sep 1995 11:09:42   gene
** Initial revision.
   
      Rev 1.21   22 Mar 1994 10:26:32   marilyn
   Added ProcInsertSpecificCpu
   
      Rev 1.20   17 Mar 1994 10:45:04   tom
   Avoid using C++ reserved word "template".
   
      Rev 1.19   16 Mar 1994 14:37:32   ernie
   Added ProcInsertSystemName function
   
      Rev 1.18   03 Mar 1994 11:58:32   john
   Added CPU32+ handling
   
      Rev 1.17   28 Feb 1994 17:17:24   marilyn
   Moved temporary max address routines to the address server.
   
      Rev 1.16   14 Feb 1994 13:17:26   marilyn
   Updated for intel processors.
   
      Rev 1.15   15 Jun 1993 11:15:08   mindy
  Added default stack size routine.
   
      Rev 1.14   06 May 1993 07:17:08   doug
   add demo version support
   
      Rev 1.13   07 Apr 1993 17:44:12   mindy
   added two routines
   
      Rev 1.12   31 Mar 1993 13:31:04   ernie
   Added ProcInsertSpecificProcessor() and ProcInsertCpu() calls.
   These can be used to genericize components by eliminating the need
   to update a source file when a new processor is added.  For example,
   the event template dll uses this to create a filename to look for
   containing event bit definitions (ev330.cfg, ev331.cfg, etc).
   
      Rev 1.11   30 Mar 1993 08:56:12   ernie
   1. Heavy cleanup.
   2. Added ProcReturnCpu() function.
   3. Added cpu16 and 68360 to processor list.
   
      Rev 1.10   23 Mar 1993 09:43:12   doug
   merged John's generic 340 change
   
      Rev 1.8.1.1   01 Mar 1993 13:27:22   john
   Added proper definitions for 330/340
   
      Rev 1.8.1.0   12 Feb 1993 11:29:16   john
   No change.
   
      Rev 1.8   13 Jan 1993 15:19:56   john
   added steelhead definitions to procfamily call
   
      Rev 1.7   11 Jan 1993 18:08:06   john
   processor determination is now dynamic for some calls
   
      Rev 1.6   11 Dec 1992 15:53:20   john
   added call to return the system type connected
   
      Rev 1.5   12 Oct 1992 10:56:46   brucea
   Removed:ProcReturnMaxAddress
   
      Rev 1.4   17 Aug 1992 09:16:30   brucea
   Added: ProcReturnMaxInputAddrDigits, ProcReturnMaxOutputAddrDigits
   
      Rev 1.3   14 Aug 1992 17:52:22   brucea
   Added: ProcReturnMaxInputAddress
          ProcReturnMaxOutputAddress
          ProcReturnProcessorName
   
      Rev 1.2   12 Aug 1992 09:56:04   brucea
   Changed: maxAddress value from 0xFFFFFF to 0xFFFFFFL
   
      Rev 1.1   22 Jul 1992 11:09:22   brucea
   Revamped: now return RETCODE for every function, and return information
      is done via call-by-reference parameter
   Added: ProcReturnSpecificProcessor, ProcReturnLogicalEqualsPhysical
   
      Rev 1.0   13 Jul 1992 11:21:10   brucea
   Initial revision.
**
** $Header:   S:/tbird/arcppc/proc/proc.c_v   1.3   08 Apr 1998 10:46:44   hera  $
**
**  Copyright (C) 1991,92 Microtek International.  All rights reserved.
**
*****************************************************************************/


                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _PROC_
#include "proc.h"
#endif

#ifndef _PROCERR_
#include "procerr.h"
#endif

#ifndef _SDNUM_
#include "sdnum.h"
#endif

#ifndef _SDTEMPL_
#include "sdtempl.h"
#endif

#ifndef _SSHARED_
#include "sshared.h"
#endif

#ifndef __STRING_H
#include <string.h>
#endif


                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/
#define INI_FILE_NAME "micepack.ini"
#define INI_APP_NAME "Demonstration"
#define INI_PRODUCT_NAME "Product"
#define INI_NOT_THERE "Not there"  // default string if missing
#define PS "PS"

#define PP "PP"

#define MP "MP"
#define MP302 "MP302"
#define MP306 "MP306"
#define MP307 "MP307"
#define MP328 "MP328"
#define MP860 "MP860"
#define MP821 "MP821"
#define MP603E "MP603E"
#define MP5202 "MP5202"
#define MP5204 "MP5204"
#define MP186 "MP186"
#define MP186XL "MP186XL"
#define MP186EA "MP186EA"
#define MP186EB "MP186EB"
#define MP186EC "MP186EC"
#define MP188 "MP188"
#define MP188XL "MP188XL"
#define MP188EA "MP188EA"
#define MP188EB "MP188EB"
#define MP188EC "MP188EC"

static BOOLEAN isDemoSet = FALSE;    // false until pwrviews.ini checked
static BOOLEAN isDemoRunning;        // set after reading pwrivews.ini
static BOOLEAN isMemberSet = FALSE;  // false until set; shared data not
                                     // initialized when return demonstration
                                     // is called, so delay write until later
                                     // when it is loaded.
static PROBE_TYPE demoProc;          // probe set for demo

BOOLEAN     inTestMode = FALSE;   // testing mode switch for CLI
PROBE_TYPE  testCpu = PT_DUMMY;   // testing cpu used with test mode

                        /****************************
                         *                          *
                         *      PROTOTYPES          *
                         *                          *
                         ****************************/

                        /****************************
                         *                          *
                         *      EXECUTABLE CODE     *
                         *                          *
                         ****************************/


//--------------------------------------------------------------------------
// ProcReturnCpu
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnCpu(PROC_CPU FAR *cpu)  {
   RETCODE err;
   PROBE_TYPE specificProcessor;
   
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
      return(err);
   switch (specificProcessor) {
      case PPC603E_MP:
      case PPC860_MP:
      case PPC821_MP:
         *cpu = PROC_CPU_POWERPC;
         break;
      case M68306_MP:
      case M68307_MP:
      case M68328_MP:
      case MCF5202_MP:
      case MCF5204_MP:
         *cpu = PROC_CPU_68000;
         break;
      case M68040_TB: case M68040_SH:
      case M68EC040_TB: case M68EC040_SH:
      case M68LC040_TB: case M68LC040_SH:
         *cpu = PROC_CPU_68040;
         break;
      case I80C186_TB: case I80C186_SH: case I80C186_MP:
      case I80C186XL_TB: case I80C186XL_SH: case I80C186XL_MP:
      case I80C186EA_TB: case I80C186EA_SH: case I80C186EA_MP:
      case I80C186EB_TB: case I80C186EB_SH: case I80C186EB_MP:
      case I80C186EC_TB: case I80C186EC_SH: case I80C186EC_MP:
      case I80C188_TB: case I80C188_SH: case I80C188_MP:
      case I80C188XL_TB: case I80C188XL_SH: case I80C188XL_MP:
      case I80C188EA_TB: case I80C188EA_SH: case I80C188EA_MP:
      case I80C188EB_TB: case I80C188EB_SH: case I80C188EB_MP:
      case I80C188EC_TB: case I80C188EC_SH: case I80C188EC_MP:
         *cpu = PROC_CPU_80186;
         break;
      default:
         *cpu = PROC_CPU_DUMMY;
         break;

   }
   return GOOD;
}


//--------------------------------------------------------------------------
// ProcReturnDemonstrationVersion
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnDemonstrationVersion(BOOLEAN FAR *demoVersion) {
#define MAX_TYPE_STRING 32
   char typeString[MAX_TYPE_STRING];
   if(!isDemoSet) {  // first time through
      *demoVersion = isDemoRunning = FALSE;  // default in case of error
      isDemoSet = TRUE;
      GetPrivateProfileString((LPSTR)INI_APP_NAME, INI_PRODUCT_NAME,
         INI_NOT_THERE, (LPSTR)typeString, MAX_TYPE_STRING,
         (LPSTR)INI_FILE_NAME);
      if(!stricmp(typeString, (char *)INI_NOT_THERE)) return(GOOD);
      if (!strnicmp(typeString,(char *)MP,strlen(MP))) {
         if(!stricmp(typeString, (char *)MP306))   demoProc = M68306_MP;
         else if(!stricmp(typeString, (char *)MP603E))   demoProc = PPC603E_MP;
         else if(!stricmp(typeString, (char *)MP860))   demoProc = PPC860_MP;
         else if(!stricmp(typeString, (char *)MP821))   demoProc = PPC821_MP;
         else if(!stricmp(typeString, (char *)MP307))   demoProc = M68307_MP;
         else if(!stricmp(typeString, (char *)MP328))   demoProc = M68328_MP;
         else if(!stricmp(typeString, (char *)MP5202))   demoProc = MCF5202_MP;
         else if(!stricmp(typeString, (char *)MP5204))   demoProc = MCF5204_MP;
         else if(!stricmp(typeString, (char *)MP186))   demoProc = I80C186_MP;
         else if(!stricmp(typeString, (char *)MP186XL)) demoProc = I80C186XL_MP;
         else if(!stricmp(typeString, (char *)MP186EA)) demoProc = I80C186EA_MP;
         else if(!stricmp(typeString, (char *)MP186EB)) demoProc = I80C186EB_MP;
         else if(!stricmp(typeString, (char *)MP186EC)) demoProc = I80C186EC_MP;
         else if(!stricmp(typeString, (char *)MP188))   demoProc = I80C188_MP;
         else if(!stricmp(typeString, (char *)MP188XL)) demoProc = I80C188XL_MP;
         else if(!stricmp(typeString, (char *)MP188EA)) demoProc = I80C188EA_MP;
         else if(!stricmp(typeString, (char *)MP188EB)) demoProc = I80C188EB_MP;
         else if(!stricmp(typeString, (char *)MP188EC)) demoProc = I80C188EC_MP;
      }
      else return(ER_INVALID_PRODUCT_IN_DEMO);
      isDemoRunning = TRUE;
   }
   *demoVersion = isDemoRunning;
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcReturnByteOrder
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnByteOrder(ENDIAN_TYPE FAR *byteOrder)  {
   RETCODE err;
   PROCESSOR_FAMILY family;

   if ((err = ProcReturnProcFamily(&family)) != GOOD)
      return err;
   switch (family) {
      case FAMILY_UNKNOWN:
      case FAMILY_68K:
         *byteOrder = BIG_ENDIAN;
         break;
      case FAMILY_X86:
         *byteOrder = LITTLE_ENDIAN;
         break;
      default:
         *byteOrder = BIG_ENDIAN;
         break;
   }
   return GOOD;
}


//--------------------------------------------------------------------------
// ProcReturnBytesPerInstr
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnBytesPerInstr(U16 FAR *bytesPerInstr)  {
   RETCODE err;
   PROCESSOR_FAMILY family;

   if ((err = ProcReturnProcFamily(&family)) != GOOD)
      return err;
   switch (family) {
      case FAMILY_UNKNOWN:
      case FAMILY_68K:
         *bytesPerInstr = 2;
         break;
      case FAMILY_POWERPC:
         *bytesPerInstr = 4;
         break;
      case FAMILY_X86:
         *bytesPerInstr = 1;
         break;
      default:
         *bytesPerInstr = 2;
         break;
   }
   return GOOD;
}


//--------------------------------------------------------------------------
// ProcReturnFetchBoundary
//--------------------------------------------------------------------------
RETCODE EXPORT
ProcReturnFetchBoundary(PROC_INSTR_FETCH_SIZE FAR *fetchBoundary)  {
   RETCODE err;
   PROC_CPU cpu;
   
   if ((err = ProcReturnCpu(&cpu)) != GOOD)
      return(err);
   switch (cpu) {
      case PROC_CPU_CPU16:
      case PROC_CPU_CPU32:
         *fetchBoundary = PROC_WORD;
          break;
      case PROC_CPU_CPU32P:
         *fetchBoundary = PROC_DWORD;
          break;
      case PROC_CPU_68000:
      case PROC_CPU_68010:
      case PROC_CPU_68020:
      case PROC_CPU_68030:
      case PROC_CPU_68040:
         *fetchBoundary = PROC_LONG;
         break;
      case PROC_CPU_80186:
      case PROC_CPU_80386:
      case PROC_CPU_80486:
         *fetchBoundary = PROC_BYTE;
         break;
      default:
         *fetchBoundary = PROC_WORD;
         break;
   }
   return GOOD;
}


//--------------------------------------------------------------------------
// ProcReturnProcFamily
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnProcFamily(PROCESSOR_FAMILY FAR *procFamily)  {
   RETCODE err;
   PROC_CPU cpu;
   
   if ((err = ProcReturnCpu(&cpu)) != GOOD) return(err);
   switch (cpu) {
      case PROC_CPU_POWERPC:
         *procFamily = FAMILY_POWERPC;
         break;
      case PROC_CPU_CPU32:
      case PROC_CPU_CPU32P:
      case PROC_CPU_CPU16:
      case PROC_CPU_68000:
      case PROC_CPU_68010:
      case PROC_CPU_68020:
      case PROC_CPU_68030:
      case PROC_CPU_68040:
         *procFamily = FAMILY_68K;
         break;
      case PROC_CPU_80186:
      case PROC_CPU_80386:
      case PROC_CPU_80486:
         *procFamily = FAMILY_X86;
         break;
      default:
         *procFamily = FAMILY_UNKNOWN;
         break;
   }

   return GOOD;
}

//--------------------------------------------------------------------------
// ProcReturnSpecificProcessor
//--------------------------------------------------------------------------
RETCODE EXPORT
ProcReturnSpecificProcessor(PROBE_TYPE FAR *specificProcessor) {
   RETCODE err;
   if(isDemoRunning && !isMemberSet) {
      if((err = SdnWriteMember(SDN_PROBE_TYPE,(U8 FAR *)&demoProc, GOOD))
         !=GOOD) return(err);
      isMemberSet = TRUE;
   }
   if (!inTestMode)
      return(SdnReadMember(SDN_PROBE_TYPE,(U8 *) specificProcessor));
   else {
      *specificProcessor = testCpu;
      return(GOOD);
   }
}
//--------------------------------------------------------------------------
// ProcGetProcessorType
//--------------------------------------------------------------------------
RETCODE EXPORT ProcGetProcessorType(LPSTR processorName,
                                    PROC_SYSTEM_TYPE system,
                                    PROBE_TYPE *proc) {

   // searches by specified system type.  If a match isn't found
   // then processor isn't supported on the particular chassis.
   // In this case return PT_DUMMY. 
   *proc = PT_DUMMY;
   if ((system == PROC_POWERPACK) || (system == PROC_DUMMY))  {
      if(!stricmp(processorName,"MC68330")) *proc = M68330_TB;
      else if(!stricmp(processorName,"MC68331")) *proc = M68331_TB;
      else if(!stricmp(processorName,"MC68332")) *proc = M68332_TB;
      else if(!stricmp(processorName,"MC68F333")) *proc = M68333_TB;
      else if(!stricmp(processorName,"MC68334")) *proc = M68334_TB;
      else if(!stricmp(processorName,"MC68340")) *proc = M68340_TB;
      else if(!stricmp(processorName,"MC68360")) *proc = M68360_TB;
      else if(!stricmp(processorName,"MC68349")) *proc = M68349_TB;
      else if(!stricmp(processorName,"MC68040")) *proc = M68040_TB;
      else if(!stricmp(processorName,"MC68EC040")) *proc = M68EC040_TB;
      else if(!stricmp(processorName,"MC68LC040")) *proc = M68LC040_TB;
      else if(!stricmp(processorName,"MC68HC16Y1")) *proc = M68HC16Y1_TB;
      else if(!stricmp(processorName,"MC68HC16Z1")) *proc = M68HC16Z1_TB;
      else if(!stricmp(processorName,"MC68HC16Z2")) *proc = M68HC16Z2_TB;
      else if(!stricmp(processorName,"I80C186")) *proc = I80C186_TB;
      else if(!stricmp(processorName,"I80C186XL")) *proc = I80C186XL_TB;
      else if(!stricmp(processorName,"I80C186EA")) *proc = I80C186EA_TB;
      else if(!stricmp(processorName,"I80C186EB")) *proc = I80C186EB_TB;
      else if(!stricmp(processorName,"I80C186EC")) *proc = I80C186EC_TB;
      else if(!stricmp(processorName,"I80C188")) *proc = I80C188_TB;
      else if(!stricmp(processorName,"I80C188XL")) *proc = I80C188XL_TB;
      else if(!stricmp(processorName,"I80C188EA")) *proc = I80C188EA_TB;
      else if(!stricmp(processorName,"I80C188EB")) *proc = I80C188EB_TB;
      else if(!stricmp(processorName,"I80C188EC")) *proc = I80C188EC_TB;
      else if(!stricmp(processorName,"I80386CX")) *proc = I80386CX_TB;
      else if(!stricmp(processorName,"I80386SX")) *proc = I80386SX_TB;
      else if(!stricmp(processorName,"I80386EX")) *proc = I80386EX_TB;
      else if(!stricmp(processorName,"I80386DX")) *proc = I80386DX_TB;
      else if(!stricmp(processorName,"I80486SX")) *proc = I80486SX_TB;
      else if(!stricmp(processorName,"I80486DX")) *proc = I80486DX_TB;
   }
   else if ((system == PROC_POWERSCOPE) || (system == PROC_DUMMY)) {
      if(!stricmp(processorName,"MC68330")) *proc = M68330_SH;
      else if(!stricmp(processorName,"MC68331")) *proc = M68331_SH;
      else if(!stricmp(processorName,"MC68332")) *proc = M68332_SH;
      else if(!stricmp(processorName,"MC68F333")) *proc = M68333_SH;
      else if(!stricmp(processorName,"MC68334")) *proc = M68334_SH;
      else if(!stricmp(processorName,"MC68340")) *proc = M68340_SH;
      else if(!stricmp(processorName,"MC68360")) *proc = M68360_SH;
      else if(!stricmp(processorName,"MC68349")) *proc = M68349_SH;
      else if(!stricmp(processorName,"MC68040")) *proc = M68040_SH;
      else if(!stricmp(processorName,"MC68EC040")) *proc = M68EC040_SH;
      else if(!stricmp(processorName,"MC68LC040")) *proc = M68LC040_SH;
      else if(!stricmp(processorName,"MC68HC16Y1")) *proc = M68HC16Y1_SH;
      else if(!stricmp(processorName,"MC68HC16Z1")) *proc = M68HC16Z1_SH;
      else if(!stricmp(processorName,"MC68HC16Z2")) *proc = M68HC16Z2_SH;
      else if(!stricmp(processorName,"I80C186")) *proc = I80C186_SH;
      else if(!stricmp(processorName,"I80C186XL")) *proc = I80C186XL_SH;
      else if(!stricmp(processorName,"I80C186EA")) *proc = I80C186EA_SH;
      else if(!stricmp(processorName,"I80C186EB")) *proc = I80C186EB_SH;
      else if(!stricmp(processorName,"I80C186EC")) *proc = I80C186EC_SH;
      else if(!stricmp(processorName,"I80C188")) *proc = I80C188_SH;
      else if(!stricmp(processorName,"I80C188XL")) *proc = I80C188XL_SH;
      else if(!stricmp(processorName,"I80C188EA")) *proc = I80C188EA_SH;
      else if(!stricmp(processorName,"I80C188EB")) *proc = I80C188EB_SH;
      else if(!stricmp(processorName,"I80C188EC")) *proc = I80C188EC_SH;
      else if(!stricmp(processorName,"I80386CX")) *proc = I80386CX_SH;
      else if(!stricmp(processorName,"I80386SX")) *proc = I80386SX_SH;
      else if(!stricmp(processorName,"I80386EX")) *proc = I80386EX_SH;
      else if(!stricmp(processorName,"I80386DX")) *proc = I80386DX_SH;
      else if(!stricmp(processorName,"I80486SX")) *proc = I80486SX_SH;
      else if(!stricmp(processorName,"I80486DX")) *proc = I80486DX_SH;
   }
   else if ((system == PROC_MICEPACK) || (system == PROC_DUMMY)) {
      if(!stricmp(processorName,"MC68306")) *proc = M68306_MP;
      else if(!stricmp(processorName,"PPC603E")) *proc = PPC603E_MP;
      else if(!stricmp(processorName,"PPC860"))  *proc = PPC860_MP;
      else if(!stricmp(processorName,"PPC821"))  *proc = PPC821_MP;
      else if(!stricmp(processorName,"MC68307")) *proc = M68307_MP;
      else if(!stricmp(processorName,"MC68328")) *proc = M68328_MP;
      else if(!stricmp(processorName,"MCF5202")) *proc = MCF5202_MP;
      else if(!stricmp(processorName,"MCF5204")) *proc = MCF5204_MP;
      else if(!stricmp(processorName,"I80C186")) *proc = I80C186_MP;
      else if(!stricmp(processorName,"I80C186XL")) *proc = I80C186XL_MP;
      else if(!stricmp(processorName,"I80C186EA")) *proc = I80C186EA_MP;
      else if(!stricmp(processorName,"I80C186EB")) *proc = I80C186EB_MP;
      else if(!stricmp(processorName,"I80C186EC")) *proc = I80C186EC_MP;
      else if(!stricmp(processorName,"I80C188")) *proc = I80C188_MP;
      else if(!stricmp(processorName,"I80C188XL")) *proc = I80C188XL_MP;
      else if(!stricmp(processorName,"I80C188EA")) *proc = I80C188EA_MP;
      else if(!stricmp(processorName,"I80C188EB")) *proc = I80C188EB_MP;
      else if(!stricmp(processorName,"I80C188EC")) *proc = I80C188EC_MP;
   }
   return GOOD;
}

//--------------------------------------------------------------------------
// ProcGetProcessorName
//--------------------------------------------------------------------------
RETCODE EXPORT ProcGetProcessorName(PROBE_TYPE specificProcessor,
                                    LPSTR processorName) {

   processorName[0] = '\0';
   switch (specificProcessor) {
      case M68330_TB: case M68330_SH:
         strcat(processorName, "MC68330");
         break;
      case M68331_TB: case M68331_SH:
         strcat(processorName, "MC68331");
         break;
      case M68332_TB: case M68332_SH:
         strcat(processorName, "MC68332");
         break;
      case M68333_TB: case M68333_SH:
         strcat(processorName, "MC68F333");
         break;
      case M68334_TB: case M68334_SH:
         strcat(processorName, "MC68334");
         break;
      case M68340_TB: case M68340_SH:
         strcat(processorName, "MC68340");
         break;
      case M68360_TB: case M68360_SH:
         strcat(processorName, "MC68360");
         break;
      case M68349_TB: case M68349_SH:
         strcat(processorName, "MC68349");
         break;
      case M68040_TB: case M68040_SH:
         strcat(processorName, "MC68040");
         break;
      case M68EC040_TB: case M68EC040_SH:
         strcat(processorName, "MC68EC040");
         break;
      case M68LC040_TB: case M68LC040_SH:
         strcat(processorName, "MC68LC040");
         break;
      case PPC603E_MP:
         strcat(processorName, "PPC603E");
         break;
      case PPC860_MP:
         strcat(processorName, "PPC860");
         break;
      case PPC821_MP:
         strcat(processorName, "PPC821");
         break;
      case M68306_MP:
         strcat(processorName, "MC68306");
         break;
      case M68307_MP:
         strcat(processorName, "MC68307");
         break;
      case M68328_MP:
         strcat(processorName, "MC68328");
         break;
      case MCF5202_MP:
         strcat(processorName, "MCF5202");
         break;
      case MCF5204_MP:
         strcat(processorName, "MCF5204");
         break;
      case M68HC16Z1_TB: case M68HC16Z1_SH:
         strcat(processorName, "MC68HC16Z1");
         break;
      case M68HC16Z2_TB: case M68HC16Z2_SH:
         strcat(processorName, "MC68HC16Z2");
         break;
      case M68HC16Y1_TB: case M68HC16Y1_SH:
         strcat(processorName, "MC68HC16Y1");
         break;
      case I80C186_TB: case I80C186_SH: case I80C186_MP:
         strcat(processorName, "I80C186");
         break;
      case I80C186XL_TB: case I80C186XL_SH: case I80C186XL_MP:
         strcat(processorName, "I80C186XL");
         break;
      case I80C186EA_TB: case I80C186EA_SH: case I80C186EA_MP:
         strcat(processorName, "I80C186EA");
         break;
      case I80C186EB_TB: case I80C186EB_SH: case I80C186EB_MP:
         strcat(processorName, "I80C186EB");
         break;
      case I80C186EC_TB: case I80C186EC_SH: case I80C186EC_MP:
         strcat(processorName, "I80C186EC");
         break;
      case I80C188_TB: case I80C188_SH: case I80C188_MP:
         strcat(processorName, "I80C188");
         break;
      case I80C188XL_TB: case I80C188XL_SH: case I80C188XL_MP:
         strcat(processorName, "I80C188XL");
         break;
      case I80C188EA_TB: case I80C188EA_SH: case I80C188EA_MP:
         strcat(processorName, "I80C188EA");
         break;
      case I80C188EB_TB: case I80C188EB_SH: case I80C188EB_MP:
         strcat(processorName, "I80C188EB");
         break;
      case I80C188EC_TB: case I80C188EC_SH: case I80C188EC_MP:
         strcat(processorName, "I80C188EC");
         break;
      case I80386CX_TB: case I80386CX_SH:
         strcat(processorName, "I80386CX");
         break;
      case I80386SX_TB: case I80386SX_SH:
         strcat(processorName, "I80386SX");
         break;
      case I80386EX_TB: case I80386EX_SH:
         strcat(processorName, "I80386EX");
         break;
      case I80386DX_TB: case I80386DX_SH:
         strcat(processorName, "I80386DX");
         break;
      case I80486SX_TB: case I80486SX_SH:
         strcat(processorName, "I80486SX");
         break;
      case I80486DX_TB: case I80486DX_SH:
         strcat(processorName, "I80486DX");
         break;
      default:
         strcat(processorName, "UNKNOWN");
         break;
   }

   return GOOD;
}



//--------------------------------------------------------------------------
// ProcReturnProcessorName
//--------------------------------------------------------------------------
RETCODE EXPORT
ProcReturnProcessorName(LPSTR processorName) {
   RETCODE err;
   PROBE_TYPE specificProcessor;
   
   processorName[0] = '\0';
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
      return(err);
   return(ProcGetProcessorName(specificProcessor,processorName));

}


//--------------------------------------------------------------------------
// ProcReturnSystemType
//--------------------------------------------------------------------------
RETCODE EXPORT ProcReturnSystemType(PROC_SYSTEM_TYPE *systemType) {
   RETCODE err;
   PROBE_TYPE specificProcessor;
   BOOLEAN savedTestMode = inTestMode;

   /* system type is always determined by the actual processor not the
   ** test processor.  So save and restore test mode.
   */
   inTestMode = FALSE;
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD) {
      inTestMode = savedTestMode;
      return(err);
   }
   inTestMode = savedTestMode;
   switch (specificProcessor) {
      case M68330_TB:
      case M68331_TB:
      case M68332_TB:
      case M68333_TB:
      case M68334_TB:
      case M68340_TB:
      case M68360_TB:
      case M68349_TB:
      case M68040_TB:
      case M68EC040_TB:
      case M68LC040_TB:
      case M68HC16Z1_TB:
      case M68HC16Z2_TB:
      case M68HC16Y1_TB:
      case I80C186_TB:
      case I80C186XL_TB:
      case I80C186EA_TB:
      case I80C186EB_TB:
      case I80C186EC_TB:
      case I80C188_TB:
      case I80C188XL_TB:
      case I80C188EA_TB:
      case I80C188EB_TB:
      case I80C188EC_TB:
      case I80386CX_TB:
      case I80386SX_TB:
      case I80386EX_TB:
      case I80386DX_TB:
      case I80486SX_TB:
      case I80486DX_TB:
         *systemType = PROC_POWERPACK;
         break;
      case M68330_SH:
      case M68331_SH:
      case M68332_SH:
      case M68333_SH:
      case M68334_SH:
      case M68340_SH:
      case M68360_SH:
      case M68349_SH:
      case M68040_SH:
      case M68EC040_SH:
      case M68LC040_SH:
      case M68HC16Z1_SH:
      case M68HC16Z2_SH:
      case M68HC16Y1_SH:
      case I80C186_SH:
      case I80C186XL_SH:
      case I80C186EA_SH:
      case I80C186EB_SH:
      case I80C186EC_SH:
      case I80C188_SH:
      case I80C188XL_SH:
      case I80C188EA_SH:
      case I80C188EB_SH:
      case I80C188EC_SH:
      case I80386CX_SH:
      case I80386SX_SH:
      case I80386EX_SH:
      case I80386DX_SH:
      case I80486SX_SH:
      case I80486DX_SH:
         *systemType = PROC_POWERSCOPE;
         break;
      case PPC603E_MP:
      case PPC860_MP:
      case PPC821_MP:
      case M68306_MP:
      case M68307_MP:
      case M68328_MP:
      case MCF5202_MP:
      case MCF5204_MP:
      case I80C186_MP:
      case I80C186XL_MP:
      case I80C186EA_MP:
      case I80C186EB_MP:
      case I80C186EC_MP:
      case I80C188_MP:
      case I80C188XL_MP:
      case I80C188EA_MP:
      case I80C188EB_MP:
      case I80C188EC_MP:
         *systemType = PROC_MICEPACK;
         break;
      default:
         *systemType = PROC_DUMMY;
   }
   
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcInsertSpecificProcessor
//
// Note: Processor codes are limited to 5 characters.
//--------------------------------------------------------------------------
RETCODE EXPORT ProcInsertSpecificProcessor(LPSTR templateString, LPSTR result){
   RETCODE err;
   PROBE_TYPE specificProcessor;
   LPSTR proc;
   
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
      return(err);
   switch (specificProcessor) {
      case M68330_TB: case M68330_SH:
         proc = "330"; break;
      case M68331_TB: case M68331_SH:
         proc = "331"; break;
      case M68332_TB: case M68332_SH:
         proc = "332"; break;
      case M68333_TB: case M68333_SH:
         proc = "f333"; break;
      case M68334_TB: case M68334_SH:
         proc = "334"; break;
      case M68340_TB: case M68340_SH:
         proc = "340"; break;
      case M68360_TB: case M68360_SH:
         proc = "360"; break;
      case M68349_TB: case M68349_SH:
         proc = "349"; break;
      case M68040_TB: case M68040_SH:
         proc = "040"; break;
      case PPC603E_MP:
         proc = "603e"; break;
      case PPC860_MP:
         proc = "860"; break;
      case PPC821_MP:
         proc = "821"; break;
      case M68306_MP:
         proc = "306"; break;
      case M68307_MP:
         proc = "307"; break;
      case M68328_MP:
         proc = "328"; break;
      case MCF5202_MP:
         proc = "5202"; break;
      case MCF5204_MP:
         proc = "5204"; break;
      case M68EC040_TB: case M68EC040_SH:
         proc = "ec040"; break;
      case M68LC040_TB: case M68LC040_SH:
         proc = "lc040"; break;
      case M68HC16Z1_TB: case M68HC16Z1_SH:
         proc = "16z1"; break;
      case M68HC16Z2_TB: case M68HC16Z2_SH:
         proc = "16z2"; break;
      case M68HC16Y1_TB: case M68HC16Y1_SH:
         proc = "16y1"; break;
      case I80C186_TB: case I80C186_SH: case I80C186_MP:
         proc = "186"; break;
      case I80C186XL_TB: case I80C186XL_SH: case I80C186XL_MP:
         proc = "186xl"; break;
      case I80C186EA_TB: case I80C186EA_SH: case I80C186EA_MP:
         proc = "186ea"; break;
      case I80C186EB_TB: case I80C186EB_SH: case I80C186EB_MP:
         proc = "186eb"; break;
      case I80C186EC_TB: case I80C186EC_SH: case I80C186EC_MP:
         proc = "186ec"; break;
      case I80C188_TB: case I80C188_SH: case I80C188_MP:
         proc = "188"; break;
      case I80C188XL_TB: case I80C188XL_SH: case I80C188XL_MP:
         proc = "188xl"; break;
      case I80C188EA_TB: case I80C188EA_SH: case I80C188EA_MP:
         proc = "188ea"; break;
      case I80C188EB_TB: case I80C188EB_SH: case I80C188EB_MP:
         proc = "188eb"; break;
      case I80C188EC_TB: case I80C188EC_SH: case I80C188EC_MP:
         proc = "188ec"; break;
      case I80386CX_TB: case I80386CX_SH:
         proc = "386cx"; break;
      case I80386SX_TB: case I80386SX_SH:
         proc = "386sx"; break;
      case I80386EX_TB: case I80386EX_SH:
         proc = "386ex"; break;
      case I80386DX_TB: case I80386DX_SH:
         proc = "386dx"; break;
      case I80486SX_TB: case I80486SX_SH:
         proc = "486sx"; break;
      case I80486DX_TB: case I80486DX_SH:
         proc = "486dx"; break;
      default:
         proc = "none"; break;
   }
   wsprintf(result,templateString,proc);
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcInsertSystemName
//--------------------------------------------------------------------------
RETCODE EXPORT ProcInsertSystemName(LPSTR templateString, LPSTR result) {
   RETCODE err;
   S8 temp[32];
   PROC_SYSTEM_TYPE systemType;
   if ((err = ProcReturnSystemType(&systemType)) != GOOD) return(err);
   switch (systemType) {
      case PROC_POWERPACK:  wsprintf(temp,templateString,"pp%s"); break;
      case PROC_POWERSCOPE: wsprintf(temp,templateString,"ps%s"); break;
      case PROC_MICEPACK:   wsprintf(temp,templateString,"mp%s"); break;
      default:              wsprintf(temp,templateString,"%s"); break;
   };
   if ((err = ProcInsertSpecificProcessor(temp,result)) != GOOD) return(err);
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcInsertCpu
// Note: Processor type codes are limited to 5 characters.
//--------------------------------------------------------------------------
RETCODE EXPORT ProcInsertCpu(LPSTR templateString, LPSTR result) {
   RETCODE err;
   PROC_CPU cpu;
   LPSTR name;
   
   if ((err = ProcReturnCpu(&cpu)) != GOOD) return(err);
   switch (cpu) {
      case PROC_CPU_CPU32:
         name = "cpu32"; break;
      case PROC_CPU_CPU32P:
         name = "32p"; break;
      case PROC_CPU_CPU16:
         name = "cpu16"; break;
      case PROC_CPU_68000:
         name = "68000"; break;
      case PROC_CPU_68010:
         name = "68010"; break;
      case PROC_CPU_68020:
         name = "68020"; break;
      case PROC_CPU_68030:
         name = "68030"; break;
      case PROC_CPU_68040:
         name = "68040"; break;
      case PROC_CPU_80186:
         name = "80186"; break;
      case PROC_CPU_80386:
         name = "80386"; break;
      case PROC_CPU_80486:
	 name = "80486"; break;
      case PROC_CPU_POWERPC:
         name = "860";   break;
      default:
         name = "none"; break;
   }
   wsprintf(result,templateString,name);
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcInsertSpecificCpu
// Note: Processor type codes are limited to 5 characters.
//--------------------------------------------------------------------------
RETCODE EXPORT ProcInsertSpecificCpu(LPSTR templateString, LPSTR result) {
   RETCODE err;
   PROC_CPU cpu;
   PROBE_TYPE specificProcessor;
   LPSTR name;
   
   if ((err = ProcReturnCpu(&cpu)) != GOOD) return(err);
   switch (cpu) {
      case PROC_CPU_CPU32:
         name = "cpu32"; break;
      case PROC_CPU_CPU32P:
         name = "32p"; break;
      case PROC_CPU_CPU16:
         name = "cpu16"; break;
      case PROC_CPU_68000:
         name = "68000"; break;
      case PROC_CPU_68010:
         name = "68010"; break;
      case PROC_CPU_68020:
         name = "68020"; break;
      case PROC_CPU_68030:
         name = "68030"; break;
      case PROC_CPU_68040:
         name = "68040"; break;
      case PROC_CPU_80186:
         if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
            return(err);
         switch (specificProcessor) {
            case I80C186_TB: case I80C186_SH: case I80C186_MP:
            case I80C186XL_TB: case I80C186XL_SH: case I80C186XL_MP:
            case I80C186EA_TB: case I80C186EA_SH: case I80C186EA_MP:
            case I80C188_TB: case I80C188_SH: case I80C188_MP:
            case I80C188XL_TB: case I80C188XL_SH: case I80C188XL_MP:
            case I80C188EA_TB: case I80C188EA_SH: case I80C188EA_MP:
               name = "186"; break;
            case I80C186EB_TB: case I80C186EB_SH: case I80C186EB_MP:
            case I80C188EB_TB: case I80C188EB_SH: case I80C188EB_MP:
               name = "186eb"; break;
            case I80C186EC_TB: case I80C186EC_SH: case I80C186EC_MP:
            case I80C188EC_TB: case I80C188EC_SH: case I80C188EC_MP:
               name = "186ec"; break;
         }
      case PROC_CPU_80386:
         if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
            return(err);
         switch (specificProcessor) {
            case I80386CX_TB: case I80386CX_SH:
               name = "386cx"; break;
            case I80386EX_TB: case I80386EX_SH:
               name = "386ex"; break;
            case I80386DX_TB: case I80386DX_SH:
            case I80386SX_TB: case I80386SX_SH:
               name = "386"; break;
         }
      case PROC_CPU_80486:
         name = "80486"; break;
      default:
         name = "none"; break;
   }
   wsprintf(result,templateString,name);
   return(GOOD);
}


//--------------------------------------------------------------------------
// ProcReturnFunctionCodesMapable
//--------------------------------------------------------------------------
RETCODE EXPORT
ProcReturnFunctionCodesMapable(BOOLEAN FAR *mapable) {
   PROBE_TYPE specificProcessor;
   PROCESSOR_FAMILY family;
   RETCODE err;

   if ((err = ProcReturnProcFamily(&family)) != GOOD)
      return err;
   if (family != FAMILY_68K && family != FAMILY_POWERPC)
      return(ER_PRC_OPER_INCOMPAT);
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
      return(err);
   switch(specificProcessor) {
      case M68331_TB: case M68331_SH: 
      case M68332_TB: case M68332_SH: 
      case M68333_TB: case M68333_SH: 
      case M68HC16Z1_TB: case M68HC16Z1_SH:
      case M68HC16Z2_TB: case M68HC16Z2_SH:
      case M68HC16Y1_TB: case M68HC16Y1_SH:
         *mapable = TRUE;
         break;
      case M68330_TB: case M68330_SH:
      case M68340_TB: case M68340_SH:
         *mapable = FALSE;
         break;
      default:
         *mapable = FALSE;
         break;
   }
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcReturnBusControlMapable
//--------------------------------------------------------------------------
RETCODE EXPORT
ProcReturnBusControlMapable(BOOLEAN FAR *mapable) {
   PROBE_TYPE specificProcessor;
   PROCESSOR_FAMILY family;
   RETCODE err;

   if ((err = ProcReturnProcFamily(&family)) != GOOD)
      return err;
   if (family != FAMILY_68K && family != FAMILY_POWERPC)
      return(ER_PRC_OPER_INCOMPAT);
   if ((err = ProcReturnSpecificProcessor(&specificProcessor)) != GOOD)
      return(err);
   switch(specificProcessor) {
      case M68331_TB: case M68331_SH:
      case M68332_TB: case M68332_SH:
      case M68333_TB: case M68333_SH:
      case M68HC16Z1_TB: case M68HC16Z1_SH:
      case M68HC16Z2_TB: case M68HC16Z2_SH:
      case M68HC16Y1_TB: case M68HC16Y1_SH:
         *mapable = TRUE;
         break;
      case M68330_TB: case M68330_SH:
      case M68340_TB: case M68340_SH:
         *mapable = FALSE;
         break;
      default:
         *mapable = FALSE;
         break;
   }
   return(GOOD);
}

//--------------------------------------------------------------------------
// ProcDefaultStackSize
//--------------------------------------------------------------------------
RETCODE EXPORT ProcDefaultStackSize(U32 FAR *size) {
   RETCODE err;
   PROC_CPU cpu;
   PROCESSOR_FAMILY family;

   if ((err = ProcReturnProcFamily(&family)) != GOOD)
      return err;
   if (family != FAMILY_68K && family != FAMILY_POWERPC)
      return ER_PRC_OPER_INCOMPAT;
   if ((err = ProcReturnCpu(&cpu)) != GOOD) return(err);
   switch (cpu) {
      case PROC_CPU_CPU16:
         *size = 2L; break;
      case PROC_CPU_CPU32:
      case PROC_CPU_CPU32P:
      case PROC_CPU_68000:
      case PROC_CPU_68010:
      case PROC_CPU_68020:
      case PROC_CPU_68030:
      case PROC_CPU_68040:
      default:
         *size = 4L; break;
   }
   return(GOOD);
}

/******************************** E O F ***********************************/
