/*----------------------------------------------------------------------------
** Name: indextbl.h
**
** Title: Index Table
**
** Purpose:
**  Supports the creation of a Base Index Table and Type Index Table and
**  provides methods for adding an index and retrieving an offset for an index
**
** Status: PRELIMINARY | CODED
**
** $Log:   S:/tbird/arcppc/symbol/indextbl.h_v  $
** 
**    Rev 1.3   31 Mar 1998 12:00:16   Winky
** No change.
** 
**    Rev 1.2   12 Mar 1997 14:47:36   hera
** No change.
** 
**    Rev 1.1   03 Mar 1997 15:47:40   hera
** 
**    Rev 1.0   07 Sep 1995 11:16:34   gene
** Initial revision.
** 
**    Rev 1.5   12 Feb 1993 17:41:44   brucea
** Cleanup of headers (no code change)
** 
**    Rev 1.4   22 Oct 1992 18:23:26   brucea
** Placed: maxIndex into protected: region rather than private: so that the
**    inherited base index class can change it to its own restricted value
** 
**    Rev 1.3   09 Dec 1991 09:19:38   john
** Added member function to free the allocated memory for the 
** index table.
** 
**    Rev 1.2   18 Nov 1991 15:20:52   brucea
** Added #include symutil.h
** 
**    Rev 1.1   08 Nov 1991 13:52:32   john
** Added code to update type table index offset.  This was done
** because a type may grow beyond a 64k boundary limit.  When this
** occurs the header must be moved and the offset pointing to that
** header must be updated.
** 
**    Rev 1.0   09 Sep 1991 11:07:22   brucea
** Initial revision.
**
** $Header:   S:/tbird/arcppc/symbol/indextbl.h_v   1.3   31 Mar 1998 12:00:16   Winky  $
**
** Copyright (C) 1991 Microtek International.  All rights reserved.
**
**--------------------------------------------------------------------------*/

#define _INDEXTBL_

#ifndef _MEMMANAG_
#include "memmanag.h"
#endif

#ifndef _SYMUTIL_
#include "symutil.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

typedef U32 TABLE_INDEX;                     // may be large numbers of types


                  /*********************************************
                   *                                           *
                   *   MEMBER FUNCTION PROTOTYPES AND INLINES  *
                   *                                           *
                   *********************************************/

class IndexTable : public Table {

private:
   U32 entryCount;            // number of entries in the table

protected:
   U32 maxIndex;              // largest index allowed

public:
//---------------------------------------------------------------------------
// IndexTable
//
// Purpose:
//    Initialize index table variables to default values.
//    Does not create the table or fix up the size (SetIndexTableSize)
//      because a constructor cannot return a parameter, and there can
//      be an out-of-memory error resulting from creation of table.
//---------------------------------------------------------------------------
IndexTable();    // constructor


//---------------------------------------------------------------------------
// GetEntryCount
//
// Purpose:
//    FOR TESTING.
//    Return the value of the variable holding the index entries put into the
//    table so far.
//---------------------------------------------------------------------------
U32 GetEntryCount() { return entryCount; };


//---------------------------------------------------------------------------
// AddIndex
//
// Purpose:
//    Add an index into the index table with its offset to the item.
//
// Input parameters:
//    index: TABLE_OFFSET index into index table.
//    itemOffset: offset into the symbol table to be stored at table[index]
//
// Output parameters: None
//
// Error:
//    Reports ER_INDEX_ALREADY_EXISTS if the index already has a legitimate
//    value in the index location in the table.
//    Reports ER_INDEX_OUT_OF_BOUNDS if the requested index is larger than
//       the maximum table size
//------------------------------------------------------------------------
RETCODE AddIndex(TABLE_INDEX index, TABLE_OFFSET itemOffset);


//---------------------------------------------------------------------------
// GetOffset
//
// Purpose:
//    Retrieve offset of item in symbol table given the index to the index
//    table.
//
// Input parameters:
//    index: TABLE_OFFSET index into index table.
//
// Output parameters:
//    itemOffset:
//       offset into symbol table of item
//
// Error:
//    Reports ER_INDEX_NOT_IN_TABLE if the index field is NULL
//    Reports ER_INDEX_OUT_OF_BOUNDS if the requested index is larger than
//       the table size
//------------------------------------------------------------------------
RETCODE GetOffset(TABLE_INDEX index, TABLE_OFFSET& itemOffset);


//---------------------------------------------------------------------------
// ObliterateTable
//
// Purpose:
//    Unlocks, then frees the table to Windows memory.
//---------------------------------------------------------------------------
VOID ObliterateTable(VOID);


//---------------------------------------------------------------------------
// SetMaxIndex
//
// Purpose:
//    Set the maximum index value <maxIndex>
//------------------------------------------------------------------------
VOID SetMaxIndex(U32 index)  { maxIndex = index; };


//---------------------------------------------------------------------------
// UpdateOffset
//
// Purpose:
//    Updates the table offset of the type index specified.  This is to
//    be used by the type table manager when creating new types.  If a 
//    type extends past a 64k boundary the type offset needs to be modified.
//
// Input parameters:
//    index: TABLE_OFFSET index into index table.
//    newOffset: TABLE_OFFSET to be inserted in place of the existing value.
//
// Output parameters:
//    none:
//
// Error:
//    Reports ER_INDEX_NOT_IN_TABLE if the index field is NULL
//    Reports ER_INDEX_OUT_OF_BOUNDS if the requested index is larger than
//       the table size
//------------------------------------------------------------------------
RETCODE UpdateOffset(TABLE_INDEX index, TABLE_OFFSET newOffset);

};
/******************************** E O F *************************************/
