/*----------------------------------------------------------------------------
** Name: mempool.h
**
** Title: Memory Pool
**
** Purpose:
**  Define the interface for the Memory Pool routines; generic routines used
**  to create, expand, and put data into an expandable pool of memory.
**
** Status: PRELIMINARY | CODED
**
** $Log:   S:/tbird/arcppc/symbol/mempool.h_v  $
** 
**    Rev 1.3   31 Mar 1998 12:00:34   Winky
** No change.
** 
**    Rev 1.2   12 Mar 1997 14:47:46   hera
** No change.
** 
**    Rev 1.1   03 Mar 1997 15:48:12   hera
** 
**    Rev 1.0   07 Sep 1995 11:16:42   gene
** Initial revision.
** 
**    Rev 1.7   15 Jun 1992 09:08:22   brucea
** Changed: GetString second param from U8 FAR * to LPU8 (which is actually the
**    same)
** 
**    Rev 1.6   13 Dec 1991 14:20:06   brucea
** Added IsValidTableOffset member function
** 
**    Rev 1.5   06 Dec 1991 17:02:36   john
** Added member function to free the allocated for the symbol table.
** 
**    Rev 1.4   18 Nov 1991 15:20:22   brucea
** Added #include symutil.h
** 
**    Rev 1.3   13 Nov 1991 18:16:38   brucea
** Added #ifdef __cplusplus and #endif
** Changed far to FAR
** Enhanced comments for GetString
** 
**    Rev 1.2   12 Sep 1991 14:54:02   brucea
** Created AllocBytes to get N bytes of storage from the memory pool
** without putting data.
** 
**    Rev 1.1   09 Sep 1991 15:17:36   brucea
** No change.
** 
**    Rev 1.0   05 Sep 1991 11:05:32   brucea
** Initial revision.
**
** $Header:   S:/tbird/arcppc/symbol/mempool.h_v   1.3   31 Mar 1998 12:00:34   Winky  $
**
** Copyright (C) 1991 Microtek International.  All rights reserved.
**
**--------------------------------------------------------------------------*/

#define _MEMPOOL_

#ifndef _MEMMANAG_
#include "memmanag.h"
#endif

#ifndef _SYMUTIL_
#include "symutil.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/


                       /**********************************
                        *                                *
                        *   MEMBER FUNCTION PROTOTYPES   *
                        *                                *
                        **********************************/
#ifdef __cplusplus
class MemPool : public Table {

private:
   TABLE_OFFSET availMemOffset;   //acts as pointer to start of available mem
                                  //also equals the number of bytes used
   U16 bytesThrownAway;           //measurement of memory loss

public:
//---------------------------------------------------------------------------
// MemPool constructor
//
// Purpose:
//    Initialize specific table variables for memory management.
//    Insert one byte so that any pointers to table will be non-zero;
//    i.e. non-NULL
//---------------------------------------------------------------------------
MemPool();    // constructor


//------------------------------------------------------------------------
// AllocBytes
//
// Purpose:
//    Allocate 1 to 255 bytes into table.  Returns offset to start of data.
//
// Input parameters:
//    size: number of bytes
//
// Output parameters:
//    tableDataOffset: offset to start of stored bytes.
//
// Error: Reports ER_MEM_OUT_OF_MEMORY if there is not enough memory
//        to hold data
//------------------------------------------------------------------------
RETCODE AllocBytes(U8 size, TABLE_OFFSET& tableDataOffset);


//------------------------------------------------------------------------
// IsValidTableOffset
//
// Purpose:
//    Checks that the input table offset is not larger that the memory
//    allocated in the table thus far.
//
// Input parameters:
//    tableDataOffset: value to be compared with end of memory offset pointer
//
// Output parameters:  returns FALSE if not valid, TRUE if valid
//------------------------------------------------------------------------
BOOLEAN IsValidTableOffset(TABLE_OFFSET tableDataOffset)   {
   return (tableDataOffset >= availMemOffset) ? FALSE : TRUE;
   }


//---------------------------------------------------------------------------
// GetAvailMemOffset
//
// Purpose:
//    Return the value of the variable that points to the next available
//    location in the table to add data.
//---------------------------------------------------------------------------
TABLE_OFFSET GetAvailMemOffset() {return availMemOffset;};


//---------------------------------------------------------------------------
// GetBytesThrownAway
//
// Purpose:
//    FOR TESTING.
//    Return the value of the variable holding the number of bytes that were
//    "discarded" when it crossed the allocated boundary (which required
//    ReAlloc)
//---------------------------------------------------------------------------
U16 GetBytesThrownAway() { return bytesThrownAway; };


//---------------------------------------------------------------------------
// GetString
//
// Purpose:
//    Get a name from the symbol table and return it in a standard C string
//    The internal name format is the PASCAL string format:
//        length (U8) followed by that number of characters
//
// Input:
//    tableDataOffset: offset into symbol table
//
// Output:
//    outputString: far pointer to U8 holding the string; max length is 255
//                  256 with NULL termination
//---------------------------------------------------------------------------
VOID GetString(TABLE_OFFSET tableDataOffset, LPU8 outputString);


//---------------------------------------------------------------------------
// ObliterateTable
//
// Purpose:
//    Unlocks, then frees the table to Windows memory.
//---------------------------------------------------------------------------
VOID ObliterateTable(VOID);


//------------------------------------------------------------------------
// PutBytes
//
// Purpose:
//    Insert 1 to 255 bytes into table.  Returns offset to start of data.
//    This routine is used for putting structures into the table by passing
//    a pointer to the structure with size = sizeof(struct).
//
// Input parameters:
//    inputDataPtr: pointer to data.
//    size: number of bytes
//
// Output parameters:
//    tableDataOffset: offset to start of stored bytes.
//
// Error: Reports ER_MEM_OUT_OF_MEMORY if there is not enough memory
//        to hold data
//------------------------------------------------------------------------
RETCODE PutBytes(U8 *inputDataPtr, U8 size,
                    TABLE_OFFSET& tableDataOffset);


//------------------------------------------------------------------------
// PutString
//
// Purpose:
//    Insert string into table.  The maximum storable string is 255 chars.
//    Any string longer will be truncated with no error.
//    The representation of the string is: length, string
//    The stored string is NOT null terminated; rather, the length is the
//    first byte.  This makes string compares much faster (length checked
//    first).
//
// Input parameters:
//    lpString: long pointer to string.  Must be null terminated.
//
// Output parameters:
//    tableDataOffset: offset to start of stored string.  Points to length.
//
// Error: Reports STRING_LENGTH_0 if string is length 0
//        Reports ER_MEM_OUT_OF_MEMORY if there is not enough memory
//        to hold string
//------------------------------------------------------------------------
RETCODE PutString(LPSTR lpString, TABLE_OFFSET& tableDataOffset);


//------------------------------------------------------------------------
// PutU32
//
// Purpose:
//    install a U32 into the table
//------------------------------------------------------------------------
RETCODE PutU32(U32 data);


//---------------------------------------------------------------------------
// SetAvailableMem
//
// Purpose:
//    TESTING ONLY!!  Positions the unused memory pointer to a new location.
//---------------------------------------------------------------------------
VOID SetAvailableMem(U32 setValue) {
	availMemOffset = min(setValue, tableSize);
	}

};  // end of class MemPool
#endif

/******************************** E O F *************************************/