/* Class: FixedSizeQueue
    A FixedSizeQueue is a fixed size collection.  It will hold maximum 
   maxSize objects.  Insert an object will place the new object on top of 
   the collection.  Insert into a full collection will automatically remove the
   last object in the collection.   
   For Example,
       coll := new(FixedSizeQueue, 4);
   if coll = FixedSizeQueue("1","2","3","4") then
      add(coll, "5");
   will produce
      FixedSizeQueue("5", "1","2","3"); 
   And
      addToEnd(coll, "6");   
   will produce
      FixedSizeQueue("5", "1", "2", "6"); 
 */!!

inherit(OrderedCollection, #FixedSizeQueue, #(maxSize /* Maximum number of object in Q */), 2, 0)!!

now(class(FixedSizeQueue))!!

/* 5/20/1993 14:21 - PUBLIC
   Create a new FixedSizeQueue collection 
*/
Def new(self, size | coll)
{ 
  coll := new(self:ancestor, size);
  setSize(coll, size);
  ^coll;
}
!!

now(FixedSizeQueue)!!

/* 5/20/1993 17:46 - PUBLIC
  Return the maximum number of objects that the collection can hold.
*/
Def maxSize(self)
{ 
  ^maxSize;
}
!!

/* 5/20/1993 17:38 - PUBLIC
  Insert an object into the collection and maintain the maximum number 
  of objects in the collection.  If the collection is full, the last
  item will be dropped off.
*/
Def insert(self, obj, idx)
{ 
  if maxSize cand size(self) >= maxSize then
     removeLast(self);
  endif;
  insert(self:ancestor, obj, idx);
  ^firstElement; /*Hera 6/12/98*/
}
!!

/* 5/20/1993 14:30 - PUBLIC
  Add an object to the collection and maintain the maxSize of the 
  collection. If the collection is full, the new object will replace the
  last one in the collection.  Return the index location that the new obj added.
*/
Def addToEnd(self, obj)
{ 
  /* maintain maxSize invariant */
  if maxSize cand (size(self) >= maxSize) then
     removeLast(self);  /* Throw away the last one in the collection */
  endif;   
  /* Now add the new object to the end of the collection */
  add(self:ancestor, obj);
  ^(lastElement - 1);
}
!!

/* 5/20/1993 14:30 - PUBLIC
  Add an object to the collection and maintain the maxSize of the collection.
*/
Def add(self, obj)
{ 
  /* maintain maxSize invariant */
  if maxSize cand (size(self) >= maxSize) then
     removeLast(self);  /* Throw away the last one in the collection */
  endif;
  /* Now add the new object to the collection */
  insert(self:ancestor, obj, 0);
}
!!

/* 5/20/1993 14:27 - PRIVATE
  Set the maximum number of objects in a FixedSizeQueue
*/
Def setSize(self, size)
{ 
  maxSize := size;
}
!!