/* CLASS: SplitRegion
   Cursor over region changes to SPLIT cursor.  Left button down grabs
   the context and draws line across parent window.  When released, sends
   newSplit() message to parent with new Y value.  Draws self as black.
   Used by the SourcePresenter and CLI Presenter.
 */!!

inherit(Window, #SplitRegion, #(dragDC
inCapture
yPos
yAdjust  /* amount to add to yPos (offset) */
), 2, nil)!!

now(class(SplitRegion))!!

/* Create a new window class Struct. Assigns the 
 SPLIT cursor style to windows of this class
 and descendants. */
Def newWClass(self, className, iconName | wc)
{ wc := newWClass(self:ancestor, className, iconName);
  putWord(wc, Call LoadCursor( HInstance, asciiz("SPLIT") ), 14);
  ^wc;
}!!

/* no border -- draw as filled w black */
Def style(self)
{
  ^WS_CHILD bitOr WS_BORDER
}
!!

/* Return static string for this window class name ("EditWindow"). */
Def wndClass(self)
{ ^"SplitAreaWindow";
}!!

now(SplitRegion)!!

/* PRIVATE */
Def init(self)
{ 
  init( self:ancestor ) ;
  yAdjust := 0 ;
}
!!

/* PRIVATE 
   Draw self as black.
*/
Def paint(self, hdc | rect, newBrush, oldBrush)
{
  rect := clientRect( self );
  newBrush := Call CreateSolidBrush(Call GetSysColor(COLOR_ACTIVECAPTION)); 
  oldBrush := Call SelectObject(hdc, newBrush);
  fill( rect, newBrush, hdc );
  Call SelectObject(hdc, oldBrush);
  Call DeleteObject(newBrush);
  drawText(self, " ", asInt(width(rect)/2)-2, top(rect)-2);  
}!!
/* PUBLIC */
Def setYadjust(self, newYadjust)
{ yAdjust := newYadjust ; }
!!

/* WINDOWS (PUBLIC)
   Draw xor-line in parent.
*/
Def WM_LBUTTONDOWN(self, wp, lp | mousePt, parentRect)
{
  Call SetCapture( handle(self) ) ;
  dragDC  := getContext( parent ) ;
  mousePt := asPoint( lp ) ;
  inCapture := y( mousePt ) + yAdjust ; 
  parentRect := clientRect( parent ) ;
  setBottom( parentRect, inCapture+1 ) ;
  setTop(    parentRect, inCapture ) ;
  Call InvertRect( dragDC, parentRect ) ;
}!!

/* MS-Window's message for left button release. If
  this is the first time receiving this message after
  a button-down message, release mouse input and sends
  an endDrag message.  Convert lp data to a Point
  object. */
Def WM_LBUTTONUP(self, wp, lp | parentRect)
{
  if inCapture
  then 
    /* erase old line */
    yPos := inCapture ; /* new Y pos */
    parentRect := clientRect( parent ) ;
    setBottom( parentRect, inCapture+1 ) ;
    setTop(    parentRect, inCapture ) ;
    Call InvertRect( dragDC, parentRect ) ;
    
    /* cleanup */
    inCapture := nil ;
    releaseContext( self, dragDC ) ;
    Call ReleaseCapture();
    
    /* inform parent */
    newSplit( parent, yPos ) ;
  endif ;
  ^0
}!!

/* Respond to MS-Window's mouse move message.  Just track the xor-line... */
Def WM_MOUSEMOVE(self, wp, lp | mousePt, parentRect)
{ if inCapture
  then 
    /* erase old line */
    parentRect := clientRect( parent ) ;
    setBottom( parentRect, inCapture+1 ) ;
    setTop(    parentRect, inCapture   ) ;
    Call InvertRect( dragDC, parentRect ) ;
    
    /* draw new line */
    mousePt := asPoint( lp ) ; 
    inCapture := y( mousePt ) + yAdjust; 
    setBottom( parentRect, inCapture+1 ) ;
    setTop(    parentRect, inCapture   ) ;
    Call InvertRect( dragDC, parentRect ) ;
  endif;
}!!
