/*****************************************************************************
**
** Name: stkservr.h
**
** Description:
**   Typedef and function definitions to handle the stack server's
**   services.
**
**  Status:  PRELIMINARY
**
** $Log:   S:/tbird/arcppc/inc/stkservr.h_v  $
** 
**    Rev 1.0   15 Jan 1997 17:36:24   kevin
** Initial revision.
** 
**    Rev 1.0   03 Jun 1996 11:25:34   gene
** Initial revision.
** 
**    Rev 1.0   14 Sep 1995 14:22:28   kevin
** Initial revision.
** 
**    Rev 1.45   04 Aug 1993 11:23:28   marilyn
** Changed interface to StkVarIsActive to return the first encountered
** frame for a variable active on the stack.
** 
**    Rev 1.44   18 Jul 1993 19:16:56   nghia
** PIP for stack
** 
**    Rev 1.43   15 Jun 1993 11:24:48   mindy
** Changed StkGetFrameCount to just return a BOOLEAN determining if there
** are any frames on the stack (StkAnyFrames). 
** 
**    Rev 1.42   06 Jan 1993 16:56:24   marilyn
** Updated comments for StkSetStackBase.
** 
**    Rev 1.41   22 Oct 1992 11:29:52   marilyn
** Added new parameter percent to StkGetAlarmLimit.
** 
**    Rev 1.40   21 Oct 1992 17:22:58   marilyn
** Updated the interfaces to stkRead and stkWrite.
** 
**    Rev 1.39   08 Sep 1992 11:03:28   marilyn
** Removed stack event defines.  See events.h.
** 
**    Rev 1.38   18 Aug 1992 11:20:52   marilyn
** Modified interface in fill stack routine.  No longer requires a descriptor
** but only the base and size params.
** 
**    Rev 1.37   29 May 1992 09:09:12   marilyn
** Added StkVarIsActive.
** 
**    Rev 1.36   28 May 1992 11:22:24   marilyn
** Added parameter frameNum to StkRead and StkWrite.
** 
**    Rev 1.35   24 Apr 1992 15:41:20   marilyn
** Fine tuning on the stack events.
** 
**    Rev 1.34   22 Apr 1992 16:11:18   marilyn
** Changed all params for StkCalcStkOffset to be S32.
** 
**    Rev 1.33   20 Apr 1992 15:22:54   marilyn
** Added two new events, STK_ALARM_CHANGE and STK_HWM_CHANGE.
** 
**    Rev 1.32   17 Apr 1992 09:46:20   marilyn
** Changed U32 to S32 in StkCalcStkOffset.
** 
**    Rev 1.31   17 Apr 1992 09:38:02   marilyn
** Added StkCalcStkOffset and updated comments for StkGetStkDirection.
** 
**    Rev 1.30   26 Mar 1992 18:05:22   brucea
** Added: StkCalcSymbolicInfo (made it EXPORT)
** 
**    Rev 1.29   20 Mar 1992 08:52:20   marilyn
** Expanded the interface for StkGetFrameSymbol.
** 
**    Rev 1.28   12 Mar 1992 17:31:24   marilyn
** No change.
** 
**    Rev 1.27   09 Mar 1992 11:25:52   marilyn
** Added cplusplus ifdef. Added StkGetFrameCount and StkGetStackDirection.
** 
**    Rev 1.26   04 Mar 1992 10:29:44   marilyn
** Added StkGetFrameSymbol.
** 
**    Rev 1.25   03 Mar 1992 10:06:52   marilyn
** Added interface for InitServer and TerminateServer routines.
** 
**    Rev 1.24   02 Mar 1992 11:37:00   marilyn
** Updated interface of StkRead to use LPU8 *.
** 
**    Rev 1.23   28 Feb 1992 10:49:50   marilyn
** Removed references to varservr.h and stkerrs.h, stkerrs.h  is now
** included in the .c files.
** 
**    Rev 1.22   27 Feb 1992 16:40:02   marilyn
** Fixed weird comment.
** 
**    Rev 1.21   27 Feb 1992 16:07:34   marilyn
** No change.
** 
**    Rev 1.20   27 Feb 1992 15:47:54   marilyn
** Updated interfaces of StkRead and StkWrite to utilize the new memory
** server interface.
** 
**    Rev 1.19   17 Feb 1992 14:11:58   brucea
** Added: isEnabled parameter to StkGetAlarmLimit
** 
**    Rev 1.18   14 Feb 1992 17:05:16   marilyn
** No change.
** 
**    Rev 1.17   14 Feb 1992 11:45:44   brucea
** Added: event notification #defines
** Added: stkSessionId comment to all functions in which this parameter was added
**        as an input
** Added: *currentSymbolOffset return parameter to StkGetCurrentLowestLevel
** Removed: U32 pattern from StkSetStackToPattern
** 
**    Rev 1.16   11 Feb 1992 13:17:04   marilyn
** Added stack descriptor to StkRead type functions.
** 
**    Rev 1.15   07 Feb 1992 17:08:40   marilyn
** Added stkSessionId as a parameter to StkReadFrameLocals and StkRead
** FrameN.
** 
**    Rev 1.14   07 Feb 1992 13:17:08   brucea
** Added Enable and Disable Alarm Limit functions
** 
**    Rev 1.13   07 Feb 1992 10:43:24   marilyn
** No change.
** 
**    Rev 1.12   06 Feb 1992 16:07:40   marilyn
** Fixed bad comment.  Fixed StkStartSession parameters.
** 
**    Rev 1.11   06 Feb 1992 16:03:52   marilyn
** Added StkStartSession and StkEndSession.
** 
**    Rev 1.10   06 Feb 1992 12:36:42   brucea
** Added new function interfaces; added DESCRIPTOR stkSessionId to most
** 
**    Rev 1.9   04 Feb 1992 15:26:02   marilyn
** No change.
** 
**    Rev 1.8   31 Jan 1992 15:40:20   marilyn
** Fixed spelling of StkWrite.
** 
**    Rev 1.7   31 Jan 1992 14:49:26   marilyn
** Changed interface for StkReadFrameN.  Added interActive as additional
** parameter.
** 
**    Rev 1.6   31 Jan 1992 09:32:24   marilyn
** Clarified some comments.
** 
**    Rev 1.5   30 Jan 1992 17:25:06   marilyn
** Added STK_TMALLOC define.
** 
**    Rev 1.4   30 Jan 1992 14:07:52   marilyn
** Changed isValid to *isValid in StkGetStackBase.
** 
**    Rev 1.3   29 Jan 1992 11:36:10   marilyn
** Further changes to StkRead and StkWrite.
** 
**    Rev 1.2   28 Jan 1992 10:52:14   marilyn
** Made changes to StkOpenStack. Added StkRead and StkWrite.
**
**    Rev 1.1   28 Jan 1992 08:44:30   brucea
** One comment changed
**
**    Rev 1.0   24 Jan 1992 12:14:10   brucea
** Initial revision.
**
**    Rev 1.0   20 Jan 1992 18:48:36   brucea
** Initial revision.
**
** $Header:   S:/tbird/arcppc/inc/stkservr.h_v   1.0   15 Jan 1997 17:36:24   kevin  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/

#define _STKSERVR_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _SYMBLSVR_
#include "symblsvr.h"
#endif
                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/
#ifdef __cplusplus
extern "C" {
#endif


#define STK_MAX_FRAMES_PER_BUF 20
#define STK_TMALLOC 0 

typedef U8 *BUF_PTR;

typedef struct {
   WORD allocIndicator;       /* STK_TMALLOC or GlobalAlloc handle */
   U16 numOfFrames;
   U32 textLength;
   U8 text[VARIABLE_SIZED];
}STK_DISPLAY_TYPE;

typedef STK_DISPLAY_TYPE *STK_DISPLAY_PTR;


                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

/*------------------------------------------------------------------------
**  InitServer
**
**  Purpose: Initialize the stack server.  This routine must be called
**     before any other routines in stack server can be used.
**
**  Parameters:
**     Input: none
**     Output: none
**
*------------------------------------------------------------------------*/
RETCODE EXPORT InitServer(VOID);

/*------------------------------------------------------------------------
**  TerminateServer
**
**  Purpose: Cleanup and terminate the stack server.  After calling this
**     routine, no other stack server routines should be called.
**
**  Parameters:
**     Input: none
**     Output: none
**
*--------------------------------------------------------------------------*/
RETCODE EXPORT TerminateServer(VOID);

/*------------------------------------------------------------------------
** StkDisableAlarmLimit
**
** Purpose: Turns off the check for stack alarm limit
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters: none
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkDisableAlarmLimit(DESCRIPTOR stkSessionId);
 

/*------------------------------------------------------------------------
** StkDisableHighWaterMark
**
** Purpose: Turns off the measurement of the stack high water mark
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters: none
** 
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkDisableHighWaterMark(DESCRIPTOR stkSessionId);
 

/*------------------------------------------------------------------------
** StkEnableAlarmLimit
**
** Purpose: Turns on the check for stack alarm limit
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters: none
** 
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkEnableAlarmLimit(DESCRIPTOR stkSessionId);
 

/*------------------------------------------------------------------------
** StkEnableHighWaterMark
**
** Purpose: Turns on the measurement of the stack high water mark
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters: none
** 
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkEnableHighWaterMark(DESCRIPTOR stkSessionId);
 

/*------------------------------------------------------------------------
**   StkEndSession
**
**   Purpose:
**      To shut down a client's access to the stack server.
**
**   Input parameters:
**      stkSessionId:  client id
**
**   Output parameters:  none
**   
**   Error:
**      Returns ER_STK_CLIENT_DISCREP if the client is not registered 
**      (client discrepancy).
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkEndSession(DESCRIPTOR stkSessionId);


/*------------------------------------------------------------------------
** StkGetAlarmLimit
**
** Purpose:
**    Get the address from which a warning is generated if, when the emulator
**    is halted, the stack usage has gone beyond. Also returns limit as
**    percentage.
**
** Input parameter:
**    stkSessionId: stack selector
**
** Output parameter:
**    alarmLimit: 
**       Address where, if stack usage goes beyond, a warning is issued
**    percent: percentage equivalent to the alarm limit physical address
**    isValid: TRUE -> value successfully obtained
**             FALSE-> value not obtained or indeterminant
**    isEnabled: TRUE -> measurement is enabled
**               FALSE -> measurement not enabled
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetAlarmLimit(DESCRIPTOR stkSessionId,
                                DESCRIPTOR *alarmLimit,
                                U32        *percent,
                                BOOLEAN    *isValid,
                                BOOLEAN    *isEnabled);


/*------------------------------------------------------------------------
** StkGetAlarmState
**
** Purpose:
**    returns the current status of the alarm limit  
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    alarmLimitReached: TRUE if stack checked and has exceeded alarmLimit
** 
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetAlarmState(DESCRIPTOR stkSessionId,
                                BOOLEAN *localAlarmLimitReached);

/*------------------------------------------------------------------------
** StkCalcStkOffset
**
** Purpose:
**    Given a current stack offset and an additional byte offset
**    calculate the resultant offset based on stack direction.  The
**    resultant offset is returned in stack offset.
**
** Input parameter:
**    stackoffset : original stack offset
**    byteOffset : additional byte offset
**
** Output parameter:
**    Recalculated stack offset returned.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkCalcStkOffset(S32 *stackOffset,
                                S32 byteOffset);


/*------------------------------------------------------------------------
** StkCalcSymbolicInfo
**
** Purpose:
**    maps current PC to symbols and saves symbol descriptors for
**    any caller to access.
**    Resulting symbolic access is thru:
**       StkGetCurrentFunction
**       StkGetCurrentLinenum
**       StkGetCurrentLowestLevel
**       StkGetCurrentModule
**
** Input parameter: none
**
** Output parameter: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkCalcSymbolicInfo(VOID);


/*------------------------------------------------------------------------
** StkGetCurrentFunction
**
** Purpose:
**    Returns the SYM_DESCRIPTOR of the function that contains the current PC
**
** Input parameters:
**    none
**
** Output parameters:
**    currentFunction:
**       descriptor to matching function; NULL_SYMBOL if no match
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetCurrentFunction(SYM_DESCRIPTOR *currentFunction);


/*------------------------------------------------------------------------
** StkGetCurrentLinenum
**
** Purpose:
**    Returns the LINENUM_DESCRIPTOR of the line number in the symbol table
**    hierarchy that contains the current PC.
**
** Input parameters:
**    none
**
** Output parameters:
**    currentLinenum: closest linenum <= current PC
**    currentColumn:  column value assigned to linenum found
**    linenumIndex:
**       abstract pointer to linenum found; can be used to get more
**       information about line number with SymGetLinenumByIndex or
**       SymGetLinenumStmtByIndex
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT
StkGetCurrentLinenum(LINENUM_TYPE       *currentLinenum,
                     COLUMN_TYPE        *currentColumn,
                     LINENUM_DESCRIPTOR *linenumIndex);


/*------------------------------------------------------------------------
** StkGetCurrentLowestlevel
**
** Purpose:
**    Returns the SYM_DESCRIPTOR of the lowest level in the symbol table
**    hierarchy that contains the current PC.  This will be a function, block,
**    or label.  Also returns offset between current PC and lowest level found
**
** Input parameters:
**    none
**
** Output parameters:
**    currentLowestlevel:
**       descriptor to matching Lowestlevel; NULL_SYMBOL if no match
**    currentSymbolOffset:
**       distance between current PC and currentLowestLevel symbol address
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT
StkGetCurrentLowestLevel(SYM_DESCRIPTOR *currentLowestlevel,
                         U32            *currentSymbolOffset);


/*------------------------------------------------------------------------
** StkGetCurrentModule
**
** Purpose:
**    Returns the SYM_DESCRIPTOR of the module that contains the current PC
**
** Input parameters:
**    none
**
** Output parameters:
**    currentModule: descriptor to matching module; NULL_SYMBOL if no match
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetCurrentModule(SYM_DESCRIPTOR *currentModule);


/*------------------------------------------------------------------------
** StkAnyFrames
**
** Purpose:
**    Returns TRUE if there are any frames for the current stack.
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    anyFrames : TRUE if there are any frames; else FALSE
**
**-----------------------------------------------------------------------*/
RETCODE EXPORT StkAnyFrames(DESCRIPTOR stkSessionId, BOOLEAN FAR *anyFrames);

/*-----------------------------------------------------------------------
**  StkGetFrameSymbol
**
**  Purpose:
**      Given the frame number (0 to N-1 for N frames) return the
**      symbol descriptor for the function and module associated with the
**      frame.  Also return an address descriptor for the range of the
**      associated function.  The descriptor is created by the server and
**      destroyed by the caller.
**
**  Input:
**      stkSessionId: stack selector
**      frameNum: the frame number
**
**  Output:
**      funcDesc: the symbol descriptor for the function associated
**         with the requested frame.
**      modDesc: the symbol descriptor for the module associated with
**         the requested frame.
**      funcAddrRangeDesc: the address descriptor for the address range
**         of the associated function.
**
**  Error:  ER_STK_FRAME_INVALID
**          ER_STK_INVALID_DESCRIPTOR
**
*---------------------------------------------------------------------------*/
RETCODE EXPORT StkGetFrameSymbol(DESCRIPTOR stkSessionId,
                                 U8 frameNum,
                                 SYM_DESCRIPTOR *funcDesc,
                                 SYM_DESCRIPTOR *modDesc,
                                 DESCRIPTOR *frameAddrRangeDesc);

/*------------------------------------------------------------------------
** StkGetHighWaterMark
**
** Purpose:
**    returns the current stack high water mark or FALSE for <isValid>.
**    Does not calculate if the measurement is disabled. 
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    highWaterMark:
**       the address of the highest (or lowest) part of the dynamic stack
**       that has been accessed
**    isEnabled: TRUE -> value successfully obtained
**               FALSE-> measurement not enabled
** 
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetHighWaterMark(DESCRIPTOR stkSessionId,
                                   DESCRIPTOR *highWaterMark,
                                   BOOLEAN    *isEnabled);
 

/*------------------------------------------------------------------------
** StkGetStackBase
**
** Purpose:
**    Returns value of the stack base address.  The initial value is unknown.
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    baseAddr:
**       Base address of the stack of type address descriptor.  The address
**       is created locally, and its descriptor is returned in baseAddr.  The
**       caller is responsible for de-allocating the memory.
**       NOTE: baseAddr is invalid if isValid = FALSE and, in fact, is not
**             allocated (i.e. do not AdrDestroyAddress if this is the case)
**    isValid: indicates that the value in *baseAddr is good (TRUE) or
**             not good (FALSE)
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetStackBase(DESCRIPTOR stkSessionId,
                               DESCRIPTOR *baseAddr,
                               BOOLEAN    *isValid);


/*------------------------------------------------------------------------
** StkGetStackDirection
**
** Purpose:
**    Returns the growth direction of the stack.
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    direction: a boolean indicating direction, FALSE = high to low memory,
**       TRUE = low to high memory.
**
** Error:
**    none
**
**-----------------------------------------------------------------------*/
RETCODE EXPORT StkGetStackDirection(DESCRIPTOR stkSessionId,
                                    BOOLEAN *direction);

/*------------------------------------------------------------------------
** StkGetStackSize
**
** Purpose:
**    Returns size stack.  Initial value = 0;
**
** Input parameters:
**    stkSessionId: stack selector
**
** Output parameters:
**    stackSize: Size of the stack in bytes
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetStackSize(DESCRIPTOR stkSessionId,
                               U32 *stackSizeParam);



/*------------------------------------------------------------------------
** StkOpenStack
**
**   Purpose:
**      To open the stack or portions of the stack for perusal.  This
**      places the associated frame functions into a frame table for
**      future reference.
**
**   Input parameters:
**      frameCount : suggested number of frames (N) to open starting at 
**         stack top which is frame 0.  Frames are numbered 0 to N-1.
**         If frameCount is 0, the stack server will default to 4 frames.
**      stkSessionId: session descriptor to uniquely identify stack
**
**   Output parameters:
**      frameCount : the total number of displayable frames found
**
**   Error:
**      Returns ER_STK_NO_FRAMES if no frame symbols are found.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkOpenStack(U8 *frameCount, DESCRIPTOR stkSessionId);


/*------------------------------------------------------------------------
**   StkRead
**
**   Purpose:
**      To read variable and parameter values from the user stack based on
**      a stack offset and the current context.
**      NOTE: stack reads are always in the direction of stack growth,
**      that is high to low or low to high depending on processor.
**      The buffer is TMalloced by StkRead.
**
**   Input parameters:
**      stkOffset : the offset from stack frame for the value on the stack.
**      frameNum : the frame number associated with the variable
**      frameType : the frame type - this with CPU type determines if there's
**                  a stack frame or not.
**      length : the length of the read in bytes.
**
**   Output parameters:
**      buffer : the value(s) read are stored in buffer.  The buffer is
**         allocated by the caller based on the size required to contain
**         the requested read. The caller is responsible for TFreeing
**         the buffer.
**
**   Error:
**      ER_STK_STK_READ_FAILED if the operation failed for any reason.
*/
/*-----------------------------------------------------------------------*/
RETCODE EXPORT StkRead(S32 stkOffset,
                       U32 length,
                       U8 frameNum,
                       U8 frameType,
                       LPU8 *buffer);

/*------------------------------------------------------------------------
**   StkReadFrameLocals
**
**   Purpose:
**      Given the frame number (0 to n-1), return a pointer to a list of
**      descriptors for the locals and pararmeters of a frame.
**
**   Input parameters:
**      stkSessionId : stack identifier
**      frameNum : the frame number to read locals for
**
**   Output parameters:
**      numOfParams:
**         number of function parameters.  The presenter can use this value
**         to only display the parameters (ignoring the locals) if the user
**         has selected this display option.  The assumption is that the
**         parameters are always at the beginning of list of variables.
**
**      firstLocalPtr :
**         pointer to the head of the local variable list.  To access the
**         remaining variables on the list, use the SymGetSymbolSibling until
**         NULL_SYMBOL is returned.
**
**   Error:
**      Returns ER_STK_FRAMENUM_INVALID if the frame number is invalid
**      NOTE:  Returns FirstLocalPtr = NULL_SYMBOL when error is indicated.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkReadFrameLocals(DESCRIPTOR stkSessionId,
                                  U8 frameNum,
                                  U8 *numOfParams,
                                  SYM_DESCRIPTOR *firstLocalPtr);

/*------------------------------------------------------------------------
**   StkReadFrameN
**
**   Purpose:
**      This routine is intended for use after the StkOpenStack routine.
**      Given the frame number (0 to n-1 for n frames), return a display
**      buffer containing the formatted frame information.  This routine
**      is used to read any frame, or multiple frames.  A maximum of
**      twenty frames can be read at one time.  Frames are numbered 0 to
**      n-1, the most recent to the oldest.
**
**   Input parameters:
**      stkSessionId : stack identifier
**      frameStart : the starting frame to read
**      frameEnd : the last frame to read
**      interActive : user mode, interactive = TRUE and batch = FALSE
**      displayBufPtr : ptr to be set to the new display buffer
**
**   Output parameters:
**      displayBufPtr: pointer to the buffer containing formatted display
**      information of the requested frame or frames.  The display
**      buffer is allocated by the stack server and deallocated by the
**      calling routine (see STK_DISPLAY_TYPE).  An interactive user mode
**      produces an anotated display buffer.
**
**   Error:
**      Returns ER_STK_FRAMENUM_INVALID if the frame number is invalid
**      NOTE:  Returns displayBufPtr set to NULL when error is indicated.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkReadFrameN(DESCRIPTOR stkSessionId,
                             U8 frameStart,
                             U8 frameEnd,
                             BOOLEAN interActive,
                             STK_DISPLAY_PTR *displayBufPtr);

/*------------------------------------------------------------------------
** StkSetAlarmLimitPercent
**
** Purpose:
**    Set the alarm limit from which a warning is generated if, when
**    the emulator is halted, the stack usage has gone beyond
**
** Input parameter:
**    stkSessionId: stack selector
**    alarmPercent: percent location of the alarm setting, if stack
**                  usage goes beyond, a warning is issued
**
** Error:
**    ER_STK_INVALID_ALARMLIMIT if the value is not inside the stack base
**    address and size.  This value is dependent on the stack type (top-down
**    versus bottom-up).
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetAlarmLimitPercent(DESCRIPTOR stkSessionId,
                                       U32 alarmPercent);

/*------------------------------------------------------------------------
** StkSetAlarmLimit
**
** Purpose:
**    Set the address from which a warning is generated if, when the emulator
**    is halted, the stack usage has gone beyond
**
** Input parameter:
**    stkSessionId: stack selector
**    alarmLimit : Address where, if stack usage goes beyond, a warning is
**                 issued
**
** Error:
**    ER_STK_INVALID_ALARMLIMIT if the value is not inside the stack base
**    address and size.  This value is dependent on the stack type (top-down
**    versus bottom-up).
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetAlarmLimit(DESCRIPTOR stkSessionId,
                                DESCRIPTOR alarmLimitAddrParam);


/*------------------------------------------------------------------------
** StkSetStackBase
**
** Purpose:
**    Set up the stack base.  Automatically rounds to the minimum writable
**    size of the stack.
**
** Input parameters:
**    stkSessionId: stack selector
**    baseAddr:  Base address of the stack. The passed address is TMalloc'ed 
**               by the caller using CreateAddr routine and is consumed 
**               by the stack server.
** Output parameters:
**    none
**
** Error:  
**    Returns ER_STK_BASE_AND_SIZE_INVALID if any arithmetic combination of 
**       <baseAddr> and <stackSize> causes an overflow; i.e.
**       if STK_BOTTOM_UP and stackSize + baseAddr > 0xFFFFFFFF
**    or if STK_TOP_DOWN and stackSize - baseAddr < 0
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetStackBase(DESCRIPTOR stkSessionId,
                               DESCRIPTOR baseAddr);


/*------------------------------------------------------------------------
** StkSetStackSize
**
** Purpose:
**    Set up the stack size in bytes
**
** Input parameters:
**    stkSessionId: stack selector
**    stackSize: Size of the stack in bytes
**
** Output parameters:
**    none
**
** Error:
**    Returns ER_STK_BASE_AND_SIZE_INVALID if any arithmetic combination of
**       <baseAddr> and <stackSize> causes an overflow; i.e.
**       if STK_BOTTOM_UP and stackSize + baseAddr > 0xFFFFFFFF
**    or if STK_TOP_DOWN and stackSize - baseAddr < 0
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetStackSize(DESCRIPTOR stkSessionId,
                               U32 stackSizeParam);


/*------------------------------------------------------------------------
** StkSetStackToPattern
**
** Purpose:
**    Sets the target memory from the current stack pointer to the size
**    of the stack.  Initially, the stack pointer will equal the stack base,
**    thus, when called, the entire stack will be initialized.
**    NOTE: this function does not modify the state of measuring the
**    high water mark; it stays enabled or disabled.
**
** Input parameter:
**    stackBaseAddrDesc : an address descriptor indicating the stack base
**       address.  It is allocated by the caller and freed by this routine.
**    stackSize : number of bytes in the stack
**
** Error:
**    must have valid base address and size of stack
**    Returns ER_STK_INIT_STK_FAIL if the stack initialization
**    fails caused by an error calculating the fill zone.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetStackToPattern(DESCRIPTOR stackBaseAddrDesc,
                                    U32 stackSize);

/*------------------------------------------------------------------------
**   StkStartSession
**
**   Purpose:
**      To register a calling client with a unique id for access to the 
**      stack server.
**
**    Input parameters:
**      winHandle:  the calling window's id
**
**   Output parameters:
**      stkSessionId:  client id
**
**   Error:  
**      Returns ER_OUT_OF_MEMORY 
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkStartSession(HWND  winHandle,
                               DESCRIPTOR *stkSessionId);


/*------------------------------------------------------------------------
** StkVarIsActive
**
** Purpose:
**    Returns a boolean indicating if the given variable context
**    exists within the call stack.
**
** Input parameters:
**    varModule : variable's module descriptor
**    varFunc : variable's function descriptor
**    varLow : variable's lowest level descriptor
**
** Output parameters:
**    Returns TRUE if a match occurs else returns FALSE.
**    varFrame : returns the frame where the first occurance of the variable
**               is found.
**
** Error:
**
**-----------------------------------------------------------------------*/
RETCODE EXPORT StkVarIsActive(SYM_DESCRIPTOR varModule,
                              SYM_DESCRIPTOR varFunc,
                              SYM_DESCRIPTOR varLow,
                              U8 *varFrame);


/*------------------------------------------------------------------------
**   StkWrite
**
**   Purpose:
**      To write variable and parameter values to the user stack based on
**      stack offset from frame pointer and current context.
**      NOTE: stack writes are always in the direction of stack growth,
**      that is high to low or low to high depending on processor.
**      StkWrite Tfree's the write buffer.
**
**   Input parameters:
**      stkOffset : the offset for the value on the stack.  
**      length : the length of the write in bytes. 
**      buffer : the value(s) to write are read from buffer 
**         The buffer is TMalloced by the caller based on the size required 
**         to contain the requested write.
**         
**   Output parameters:
**      none
**
**   Error:
**      ER_STK_STK_WRITE_FAILED if the operation fails for any reason.
*/
/*-----------------------------------------------------------------------*/
RETCODE EXPORT StkWrite(S32 stkOffset,     
                        U32 length,
                        U8 frameNum,
                        U8 frameType,
                        LPU8 buffer);

/*------------------------------------------------------------------------
** StkSetStackBaseAndSize
**
** Purpose:
**    Set the current stack session with base and size. Check
**    for the current setting of the stack HWM, if it's enabled then fill the
**    stack else set flag for isStkFilled for later stack filling.
**    Initially, the stack pointer will equal the stack base, thus the
**    entire stack will be initialized.
**    if (sp == 0)
**       fill from base to top
**    else
**       fill from sp to top
**    NOTES: baseAddr will be used or destroyed by this function.
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkSetStackBaseAndSize(DESCRIPTOR stkSessionId,
                                      DESCRIPTOR baseAddr, U32 stkSize);

/*------------------------------------------------------------------------
** StkGetStackInfo
**
** Purpose:
**    Returns all values of the current stack session information.
**    The addresses are created locally, and their descriptors are returned
**    in parameters. The caller is responsible for de-allocating the memory.
**    IMPORTANT NOTE: XAddr is invalid if isXValid = FALSE or
**    isXEnabled = FALSE and, in fact, is not allocated (i.e. do not
**    call AdrDestroyAddress if this is the case)
**
** Parameters:
**    stkSessionId (in):    Stack selector
**    baseAddr (out):       Base address of the stack of type address descriptor.
**    isBaseValid (out):    Flag indicates that the value in *baseAddr is
**                            good (TRUE) or not good (FALSE)
**    stackSize (out):      Size of the stack in bytes.
**    alarmLimitAddr (out): Alarm address where, if stack usage goes beyond, a warning is issued
**    alarmPercent (out):   Percentage equivalent to the alarm limit physical address
**    isALarmValid (out):   TRUE | FALSE = validity state of alarm value.
**    isAlarmEnabled (out): TRUE | FALSE = Alarm enabled | disabled
**    highWaterMarkAddr (out): The address of the highest (or lowest) part of
**                               the dynamic stack that has been accessed.
**    isHWMEnabled (out):   TRUE | FALSE = HWM enabled | disabled
**    stkDirection (out):   FALSE = high to low memory, TRUE = low to high memory.
**
** Error: none
*/
/*------------------------------------------------------------------------*/
RETCODE EXPORT StkGetStackInfo(DESCRIPTOR stkSessionId,
                               DESCRIPTOR *baseAddr,
                               BOOLEAN    *isBaseValid,
                               U32        *stackSize,
                               DESCRIPTOR *alarmLimitAddr,
                               U32        *alarmPercent,
                               BOOLEAN    *isAlarmValid,
                               BOOLEAN    *isAlarmEnabled,
                               DESCRIPTOR *highWaterMarkAddr,
                               BOOLEAN    *isHWMEnabled,
                               BOOLEAN    *stkDirection);

#ifdef __cplusplus
}
#endif
/******************************** E O F ***********************************/
