/***************************************************************************
**
**  File name : ASMTYP86.C
**
**  Description : type processing routines of assembler
**
**  Status:  REVIEWED
**
**  $Log:   S:/tbird/arcm332/dasm/asmtyp86.c_v  $
** 
**    Rev 1.6   21 Oct 1994 16:33:12   steve
** Fixed a bug in Atype_Xs1_modregrm_data1().
** 
**    Rev 1.5   18 Oct 1994 14:57:34   steve
** using the upper 16 bits of Extra to indicate a 32-bit datum
** 
**    Rev 1.4   14 Oct 1994 17:51:04   steve
** let SetModrm() be able to deal with disp32[*][index]
** 
**    Rev 1.3   13 Oct 1994 14:36:56   steve
** Jcc instructions don't have short-jump type.
**
**  $Header:   S:/tbird/arcm332/dasm/asmtyp86.c_v   1.6   21 Oct 1994 16:33:12   steve  $
**
**  Copyright (C) 1994 Microtek International Inc.    All Rights Reserved
**
****************************************************************************/


                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _ASMDEF86_
#include "asmdef86.h"
#endif

                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/
#pragma warn -cln
#define FIRSTBYTE(x)  ((x) & 0x00FF)
#define SECONDBYTE(x) ((x) & 0xFF00) >> 8
#define THIRDBYTE(x)  ((x) & 0x00FF0000) >> 16
#define FOURTHBYTE(x) ((x) & 0xFF000000) >> 24
#define SIGNEXTEND(x) ((x) >= -0x80 && (x) <= 0x7F)

                        /****************************
                         *                          *
                         *     LOCAL PROTOTYPES     *
                         *                          *
                         ****************************/
PRIVATE U16 SetModrm (OPERANDSTRUCT Operand, U8 *Codes);
PRIVATE U16 SetWmodrm(OPERANDSTRUCT Operand, U8 *Codes);

                        /****************************
                         *                          *
                         *      EXECUTABLE CODE     *
                         *                          *
                         ****************************/

/***************************************************************************
**
**  Name: Atype_*(Operands,  Codes, Len)
**
**  Description:
**     generate object codes of instruction for its type
**
**  Input:  buffer of operands information
**
**  Output: object codes, instruction length
**
**  Return: error code
**
****************************************************************************/

/***************************************************************************
**  no operands
**  obj codes: | ** *** ***|
****************************************************************************/
#pragma argsused
RETCODE Atype_X(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Codes[1] == 0x0A && (Codes[0] == 0xD4 || Codes[0] == 0xD5))
      /* AAM, AAD */
      *Len = 2;
   else
      *Len = 1;
   return (GOOD);
}  /* end of Atype_X() */

/***************************************************************************
**  PUSH/POP seg
**  obj codes: | ** 0sg ***|
****************************************************************************/
RETCODE Atype_X0sgX(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].MemberID > 3) { // FS: GS:
      Codes[0] += 0x7A;
      Codes[0] |= Operands[0].MemberID << 3;   /* set seg field */
      Codes[1] = Codes[0];
      Codes[0] = 0x0F;
      *Len = 2;
   }
   else {
      Codes[0] |= Operands[0].MemberID << 3;   /* set seg field */
      *Len = 1;
   }
   return (GOOD);
}  /* end of Atype_X0sgX() */

/***************************************************************************
**  shift/rotate reg/mem,1/CL
**  obj codes: | ** *** *cw|mod *** r/m|
****************************************************************************/
RETCODE Atype_Xcw_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[1].OperandType == CL)
      Codes[0] |= 0x02;
   *Len = SetWmodrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_Xcw_modXrm() */

/***************************************************************************
**  MOV seg,reg/mem
**  MOV reg/mem,seg
**  obj codes: | ** *** *d0|mod 0sg r/m|
****************************************************************************/
RETCODE Atype_Xd0_mod0sgrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == SEG) {       /* MOV seg,reg/mem */
      Codes[0] |= 0x02;
      Codes[1] |= Operands[0].MemberID << 3;   /* set seg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {                                      /* MOV reg/mem,seg */
      Codes[1] |= Operands[1].MemberID << 3;   /* set seg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (GOOD);
}  /* end of Atype_Xd0_mod0sgrm() */

/***************************************************************************
**  OUT DX,acc
**  IN  acc,DX
**  obj codes: | ** *** *dw|
****************************************************************************/
RETCODE Atype_Xdw(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == DX) {   /* OUT DX,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
   }
   else                                   /* IN  acc,DX */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
   *Len = 1;
   return (GOOD);
}  /* end of Atype_Xdw() */

/***************************************************************************
**  MOV memdirect,acc
**  MOV acc,memdirect
**  obj codes: | ** *** *dw| addr_low  | addr_high |
****************************************************************************/
RETCODE Atype_Xdw_laddr_haddr(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[1].OperandType == AL || Operands[1].OperandType == AX) {
   /* MOV memdirect,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[0].Number);
      Codes[2] = SECONDBYTE(Operands[0].Number);
   }
   else {
   /* MOV acc,memdirect */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[1].Number);
      Codes[2] = SECONDBYTE(Operands[1].Number);
   }
   *Len = 3;
   return (GOOD);
}  /* end of Atype_Xdw_laddr_haddr() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/MOV
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** *dw|mod reg r/m|
****************************************************************************/
RETCODE Atype_Xdw_modregrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[1].OperandType != REG8 && Operands[1].OperandType != REG16) {
   /* reg,reg/mem */
      Codes[0] |= 0x02;
      if (Operands[0].OperandType==REG16 && Operands[1].OperandType!=MEMBYTE)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {
   /* reg/mem,reg */
      if (Operands[1].OperandType==REG16 && Operands[0].OperandType!=MEMBYTE)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[1].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (GOOD);
}  /* end of Atype_Xdw_modregrm() */

/***************************************************************************
**  OUT port,acc
**  IN  acc,port
**  obj codes: | ** *** *dw|   port    |
****************************************************************************/
RETCODE Atype_Xdw_port(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == DATA8) {   /* OUT port,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[0].Number);
   }
   else {                                    /* IN acc,port */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[1].Number);
   }
   *Len = 2;
   return (GOOD);
}  /* end of Atype_Xdw_port() */

/***************************************************************************
**  INC/DEC/PUSH/POP reg
**  XCHG reg,AX
**  obj codes: | ** *** reg|
****************************************************************************/
RETCODE Atype_Xreg(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[0] |= Operands[0].MemberID;
   *Len = 1;
   return (GOOD);
}  /* end of Atype_Xreg() */

/***************************************************************************
**  XCHG AX,reg
**  obj codes: | ** *** reg|
****************************************************************************/
RETCODE Atype_Xreg1(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[0] |= Operands[1].MemberID;
   *Len = 1;
   return (GOOD);
}  /* end of Atype_Xreg1() */

/***************************************************************************
**  PUSH data
**  obj codes: | ** *** *s0|   data    |data if s=0|
****************************************************************************/
RETCODE Atype_Xs0_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   if (SIGNEXTEND(Operands[0].Number)) {
      Codes[0] |= 0x02;
      *Len = 2;
   }
   else {
      switch(Operands[0].OperandType) {
         case DATA8: break;
         case MEMBYTE: break;
         case DATA: break;
         default: break;
      }
      Codes[2] = SECONDBYTE(Operands[0].Number);
      *Len = 3;
   }
   // check if DATA32
   if (Operands[0].Extra & 0xFFFF0000) {
      Codes[3] = THIRDBYTE(Operands[0].Number);
      Codes[4] = FOURTHBYTE(Operands[0].Number);
      *Len = 5;
   }
   return (GOOD);
}  /* end of Atype_Xs0_data() */

/***************************************************************************
**  JMP address
**  obj codes: | ** *** *s1|   disp    |
****************************************************************************/
RETCODE Atype_Xs1_disp(U32 PC, OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
S16 Displacement;

   Displacement = Operands[0].Number - (PC+2);  /* assume short jump */
   if (Codes[0] == 0xE9 && SIGNEXTEND(Displacement)) { /* JMP: short jump  */
      Codes[0] |= 0x02;
      Codes[1] = FIRSTBYTE(Displacement);
      *Len = 2;
   }
   else {                                       /* JMP and Jcc: near jump */
      Displacement--;
      Codes[1] = FIRSTBYTE(Displacement);
      Codes[2] = SECONDBYTE(Displacement);
      *Len = 3;
   }
   return (GOOD);
}  /* end of Atype_Xs1_disp() */

/***************************************************************************
**  IMUL reg,reg/mem,data
**  obj codes: | ** *** *s1|mod reg r/m|   data    |data if s=0|
****************************************************************************/
RETCODE Atype_Xs1_modregrm_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[1], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[2].Number);
   if (SIGNEXTEND(Operands[2].Number)) {
      Codes[0] |= 0x02;
      (*Len)++;
   }
   else {
      Codes[*Len + 1] = SECONDBYTE(Operands[2].Number);
      *Len += 2;
      if (Operands[2].Extra&0xFFFF0000) { // DWORD
         Codes[(*Len)++] = THIRDBYTE(Operands[2].Number);
         Codes[(*Len)++] = FOURTHBYTE(Operands[2].Number);
      }
   }
   return (GOOD);
}  /* end of Atype_Xs1_modregrm_data() */

/***************************************************************************
**  IMUL reg,data
**  obj codes: | ** *** *s1|mod reg r/m|   data    |data if s=0|
****************************************************************************/
RETCODE Atype_Xs1_modregrm_data1(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   if (SIGNEXTEND(Operands[1].Number)) {
      Codes[0] |= 0x02;
      (*Len)++;
   }
   else {
      Codes[*Len + 1] = SECONDBYTE(Operands[1].Number);
      *Len += 2;
      if (Operands[1].Extra&0xFFFF0000) { // DWORD
         Codes[(*Len)++] = THIRDBYTE(Operands[1].Number);
         Codes[(*Len)++] = FOURTHBYTE(Operands[1].Number);
      }
   }
   return (GOOD);
}  /* end of Atype_Xs1_modregrm_data1() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR reg/mem,data
**  obj codes: | ** *** *sw|mod *** r/m|   data    |data if sw=01|
****************************************************************************/
RETCODE Atype_Xsw_modXrm_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   if (Operands[0].OperandType == REG16 ||
       Operands[0].OperandType == MEMWORD ||
       Operands[0].OperandType == MEMDWORD ||
       Operands[0].OperandType == MEM)
      if (SIGNEXTEND(Operands[1].Number))
         Codes[0] |= 0x02;
      else {
         Codes[(*Len)++] = SECONDBYTE(Operands[1].Number);
         if (Operands[1].Extra & 0xFFFF0000) {
            Codes[(*Len)++] = THIRDBYTE(Operands[1].Number);
            Codes[(*Len)++] = FOURTHBYTE(Operands[1].Number);
         }
      }
   return (GOOD);
}  /* end of Atype_Xsw_modXrm_data() */

/***************************************************************************
**  INT data
**  obj codes: | ** *** **v|   data    |
****************************************************************************/
RETCODE Atype_Xv_type(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].Number != 3) {
      Codes[0] |= 0x01;
      Codes[1] = Operands[0].Number;
      *Len = 2;
   }
   else
      *Len = 1;
   return (GOOD);
}  /* end of Atype_Xv_type() */

/***************************************************************************
**  MOV reg,data
**  obj codes: | ** **w reg|   data    |data if w=1|
****************************************************************************/
RETCODE Atype_Xwreg_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[0] |= Operands[0].MemberID;
   Codes[1] = FIRSTBYTE(Operands[1].Number);
   if (Operands[0].OperandType == REG16) {
      Codes[0] |= 0x08;
      Codes[2] = SECONDBYTE(Operands[1].Number);
      *Len = 3;
      if (Operands[1].Extra&0xFFFF0000) { // double word
         Codes[3] = THIRDBYTE(Operands[1].Number);
         Codes[4] = FOURTHBYTE(Operands[1].Number);
         *Len = 5;
      }
   }
   else
      *Len = 2;
   return (GOOD);
}  /* end of Atype_Xwreg_data() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/TEST acc,data
**  obj codes: | ** *** **w|   data    |data if w=1|
****************************************************************************/
RETCODE Atype_Xw_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] = FIRSTBYTE(Operands[1].Number);
   if (Operands[0].OperandType == AX) {
      Codes[0] |= 0x01;
      Codes[2] = SECONDBYTE(Operands[1].Number);
      *Len = 3;
      if (Operands[1].Extra&0xFFFF0000) { // 4 bytes
         Codes[3] = THIRDBYTE(Operands[1].Number);
         Codes[4] = FOURTHBYTE(Operands[1].Number);
         *Len = 5;
      }
   }
   else
      *Len = 2;
   return (GOOD);
}  /* end of Atype_Xw_data() */

/***************************************************************************
**  TEST/XCHG
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** **w|mod reg r/m|
***************************************************************************/
RETCODE Atype_Xw_modregrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == REG8 || Operands[0].OperandType == REG16) {
   /* reg,reg/mem */
      if (Operands[0].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {
   /* reg/mem,reg */
      if (Operands[1].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[1].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (GOOD);
}  /* end of Atype_Xw_modregrm() */

/***************************************************************************
**  MOV
**     reg16,specReg  or  specReg,reg16
**  obj codes: | ** *** ***|11 sepcReg reg|
***************************************************************************/
RETCODE Atype_Xspec_modregrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == REG16) {
   /* reg16,specReg */
      switch (Operands[1].OperandType) {
         case CR:          //0x20
            break;
         case DR:
            Codes[0]++;    //0x21
            break;
         case TR:
            Codes[0] += 4; //0x24
            break;
      }
      Codes[1] |= Operands[1].MemberID << 3;
      *Len = SetModrm(Operands[0], Codes);
   }
   else {
   /* specReg,reg16 */
      switch (Operands[0].OperandType) {
         case CR:
            Codes[0] += 2;    //0x22
            break;
         case DR:
            Codes[0] += 3;    //0x23
            break;
         case TR:
            Codes[0] += 6; //0x26
            break;
      }
      Codes[1] |= Operands[0].MemberID << 3;
      *Len = SetModrm(Operands[1], Codes);
   }
   return (GOOD);
}  /* end of Atype_Xspec_modregrm() */

/***************************************************************************
**  NOT/NEG/MUL/IMUL/DIV/IDIV/INC/DEC reg/mem
**  obj codes: | ** *** **w|mod *** r/m|
****************************************************************************/
RETCODE Atype_Xw_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_Xw_modXrm() */

/***************************************************************************
**  DIV/IDIV AX/AL,reg/mem
**  obj codes: | ** *** **w|mod *** r/m|
****************************************************************************/
RETCODE Atype_Xw_modXrm1(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetWmodrm(Operands[1], Codes);
   return (GOOD);
}  /* end of Atype_Xw_modXrm1() */

/***************************************************************************
**  SHLD/SHRD reg/mem,reg,data8
**  obj codes: | ** *** *s1|mod reg r/m|   data    |data if s=0|
****************************************************************************/
RETCODE Atype_Xs_modregrm_data8(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] |= Operands[1].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[2].Number);
   if (SIGNEXTEND(Operands[2].Number)) {
      (*Len)++;
   }
   else {
      return(ER_DAD_ASM_ERROR);
   }
   return (GOOD);
}  /* end of Atype_Xs_modregrm_data8() */

/***************************************************************************
**  shift/rotate reg/mem,count
**  obj codes: | ** *** **w|mod *** r/m|   count   |
****************************************************************************/
RETCODE Atype_Xw_modXrm_count(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   return (GOOD);
}  /* end of Atype_Xw_modXrm_count() */

/***************************************************************************
**  MOV/TEST reg/mem,data
**  obj codes: | ** *** **w|mod *** r/m|   data    |data if w=1|
****************************************************************************/
RETCODE Atype_Xw_modXrm_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   if (Operands[1].OperandType == DATA8)
      return(GOOD);

   if (Operands[0].OperandType == REG16 ||
       Operands[0].OperandType == MEMWORD ||
       Operands[0].OperandType == MEMDWORD ||
       Operands[0].OperandType == MEM) {
      Codes[(*Len)++] = SECONDBYTE(Operands[1].Number);
   }
   if (Operands[0].OperandType == MEMDWORD) {
      Codes[(*Len)++] = THIRDBYTE(Operands[1].Number);
      Codes[(*Len)++] = FOURTHBYTE(Operands[1].Number);
   }
   return (GOOD);
}  /* end of Atype_Xw_modXrm_data() */

/***************************************************************************
**  conditional branch
**  obj codes: | ** *** ***|   disp    |
****************************************************************************/
RETCODE Atype_X_disp(U32 PC, OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
S16 Displacement;

   Displacement = Operands[0].Number - (PC+2);
   if (SIGNEXTEND(Displacement)) {
      Codes[1] = FIRSTBYTE(Displacement);
      *Len = 2;
   }
   else
      return (ER_DAD_ASM_ERROR);
   return (GOOD);
}  /* end of Atype_X_disp() */

/***************************************************************************
**  RET/RETF data
**  obj codes: | ** *** ***| data_low  | data_high |
****************************************************************************/
RETCODE Atype_X_ldata_hdata(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].Number & 0xFFFF0000)
      return (ER_DAD_ASM_ERROR);
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   *Len = 3;
   return (GOOD);
}  /* end of Atype_X_ldata_hdata() */

/***************************************************************************
**  ENTER data,data
**  obj codes: | ** *** ***| data_low  | data_high |   data    |
****************************************************************************/
RETCODE Atype_X_ldata_hdata_data(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   Codes[3] = FIRSTBYTE(Operands[1].Number);
   *Len = 4;
   return (GOOD);
}  /* end of Atype_X_ldata_hdata_data() */

/***************************************************************************
**  CALL address
**  obj codes: | ** *** ***| disp_low  | disp_high |
****************************************************************************/
RETCODE Atype_X_ldisp_hdisp(U32 PC, OPERANDSTRUCT *Operands, U8 *Codes,
                            U16 *Len)
{
S16 Displacement;

   Displacement = Operands[0].Number - (PC+3);
   Codes[1] = FIRSTBYTE(Displacement);
   Codes[2] = SECONDBYTE(Displacement);
   *Len = 3;
   return (GOOD);
}  /* end of Atype_X_ldisp_hdisp() */

/***************************************************************************
**  CALL/JMP segment:offset
**  obj codes: | ** *** ***|offset_low |offset_high|  seg_low  | seg_high  |
****************************************************************************/
RETCODE Atype_X_loff_hoff_lseg_hseg(OPERANDSTRUCT *Operands, U8 *Codes,
                                    U16 *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   Codes[3] = THIRDBYTE(Operands[0].Number);
   Codes[4] = FOURTHBYTE(Operands[0].Number);
   if (Operands[0].Extra&0xFFFF0000) {
      Codes[5] = FIRSTBYTE(Operands[0].Extra);
      Codes[6] = SECONDBYTE(Operands[0].Extra);
      *Len = 7;
   }
   else
      *Len = 5;
   return (GOOD);
}  /* end of Atype_X_loff_hoff_lseg_hseg() */

/***************************************************************************
**  BOUND/LEA/LES/LDS reg,mem
**  obj codes: | ** *** ***|mod reg r/m|
****************************************************************************/
RETCODE Atype_X_modregrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[1], Codes);
   return (GOOD);
}  /* end of Atype_X_modregrm() */

/***************************************************************************
**  PUSH/POP/CALL/JMP reg/mem
**  obj codes: | ** *** ***|mod *** r/m|
****************************************************************************/
RETCODE Atype_X_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetModrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_X_modXrm() */

/***************************************************************************
**  CALL/JMP FAR mem
**  obj codes: | ** *** ***|mod *** r/m|
****************************************************************************/
RETCODE Atype_X_modXrm_FAR(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = SetModrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_X_modXrm_FAR() */

/***************************************************************************
**  ESC data,reg/mem
**  obj codes: | 11 011 ***|mod *** r/m|
****************************************************************************/
RETCODE Atype_ESCxxx_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
long Number;

   Number = Operands[0].Number;
   if (Number < 0 || Number > 0x3F)
      return (ER_DAD_ASM_ERROR);
   Codes[0] |= (Number & 0x38) >> 3;      /* X  X  X  X  X b5 b4 b3 */
   Codes[1] |= (Number & 0x07) << 3;      /* X  X b2 b1 b0  X  X  X */
   *Len = SetModrm(Operands[1], Codes);
   return (GOOD);
}  /* end of Atype_ESCxxx_modXrm() */

/***************************************************************************
**  operands: ST,ST(i)  /  ST(i),ST
**  obj codes: | 11 011 d**| 11 *** r/m|
****************************************************************************/
RETCODE Atype_ESCdxx_11xxxrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= Operands[0].MemberID;
   }
   else
      Codes[1] |= Operands[1].MemberID;
   *Len = 2;
   return (GOOD);
}  /* end of Atype_ESCdxx_11xxxrm() */

/***************************************************************************
**  operands: ST,ST(i) / ST(i),ST
**  obj codes: | 11 011 d**| 11 **d r/m|
****************************************************************************/
RETCODE Atype_ESCdxx_11xxdrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= 0x08;
      Codes[1] |= Operands[0].MemberID;
   }
   else
      Codes[1] |= Operands[1].MemberID;
   *Len = 2;
   return (GOOD);
}  /* end of Atype_ESCdxx_11xxdrm() */

/***************************************************************************
**  operands: ST,ST(i) / ST(i),ST
**  obj codes: | 11 011 d**| 11 **s r/m|
****************************************************************************/
RETCODE Atype_ESCdxx_11xxsrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= Operands[0].MemberID;
   }
   else {
      Codes[1] |= 0x08;
      Codes[1] |= Operands[1].MemberID;
   }
   *Len = 2;
   return (GOOD);
}  /* end of Atype_ESCdxx_11xxsrm() */

/***************************************************************************
**  operands: QWORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 q**|mod *** r/m|
****************************************************************************/
RETCODE Atype_ESCqxx_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == MEMQWORD)
      Codes[0] |= 0x04;
   *Len = SetModrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_ESCqxx_modXrm() */

/***************************************************************************
**  operands: WORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 w**|mod *** r/m|
****************************************************************************/
RETCODE Atype_ESCwxx_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == MEMWORD)
      Codes[0] |= 0x04;
   *Len = SetModrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_ESCwxx_modXrm() */

/***************************************************************************
**  operands: mem / TBYTE PTR mem
**  obj codes: | 11 011 *t*|mod *** r/m|
****************************************************************************/
RETCODE Atype_ESCxtx_modXrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   if (Operands[0].OperandType == MEMTBYTE)
      Codes[0] |= 0x02;
   *Len = SetModrm(Operands[0], Codes);
   return (GOOD);
}  /* end of Atype_ESCxtx_modXrm() */

/***************************************************************************
**  operands: ST(i)
**  obj codes: | 11 011 ***| 11 0** r/m|
****************************************************************************/
RETCODE Atype_ESCxxx_110xxrm(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   Codes[1] |= Operands[0].MemberID;
   *Len = 2;
   return (GOOD);
}  /* end of Atype_ESCxxx_110xxrm() */

/***************************************************************************
**  FPinst with no operands
**  obj codes: | 11 011 ***| ** *** ***|
****************************************************************************/
#pragma argsused
RETCODE Atype_ESCxxx_X(OPERANDSTRUCT *Operands, U8 *Codes, U16 *Len)
{
   *Len = 2;
   return (GOOD);
}  /* end of Atype_ESCxxx_X() */


/***************************************************************************
**                Procedures used by type processing routines
****************************************************************************/

/***************************************************************************
**
**  Name: SetModrm(Operand,  Codes)
**
**  Description:
**     generate codes: xxxxxxxx mod_xxx_rm
**
**  Input:  operand information
**
**  Output: object codes
**
****************************************************************************/
PRIVATE U16 SetModrm(OPERANDSTRUCT Operand, U8 *Codes)
{
U8 mod;
U16 Len;
BOOLEAN fDisp32 = FALSE;

   Len = 2;
   Codes[1] |= Operand.MemberID;
/*** register mode ***/
   if (Operand.OperandType == REG8 || Operand.OperandType == REG16)
      Codes[1] |= 0xC0;
/*** memory mode ***/
   else {
      mod = (Operand.MemberID & 0xC0) >> 6;
      // here mod should be 0, 1, or 2, because mod==3 is a register case
      if (mod == 1) {
      /* 1 offset byte */
         Codes[2] = Operand.Number;
         Len = 3;
      }
      else {
         if (Operand.Number&0xFFFF0000) {
            fDisp32 = TRUE;
            Operand.Number &= 0x0000FFFF;
         }
         if ((mod == 2) ||     // has disp16 or disp32
             // 32-bit addressing forms with the ModR/M
             (mod == 0 &&  fDisp32 && (Operand.MemberID & 0x07) == 5) ||
             // disp32[*][index]
             (mod == 0 &&  fDisp32 && (Operand.MemberID & 0x07) == 4) ||
             // 16-bit addressing forms with the ModR/M
             (mod == 0 && !fDisp32 && (Operand.MemberID & 0x07) == 6 &&
                Operand.Number)) {
         /*  2/4 offset bytes  or  direct address  */
            Codes[2] = FIRSTBYTE(Operand.Number);
            Codes[3] = SECONDBYTE(Operand.Number);
            Len = 4;
            if (fDisp32) {
               Codes[4] = FIRSTBYTE(Operand.Extra);
               Codes[5] = SECONDBYTE(Operand.Extra);
               Len = 6;
            }
         }
      }
   }
   return (Len);
}  /* end of SetModrm() */

/***************************************************************************
**
** Name: SetWmodrm(Operand,  Codes)
**
** Description:
**    generate codes: xxxxxxxw mod_xxx_rm
**
** Input:  operand information
**
** Output: object codes
**
****************************************************************************/
PRIVATE U16 SetWmodrm(OPERANDSTRUCT Operand, U8 *Codes)
{
   if (Operand.OperandType == MEM || Operand.OperandType == MEMWORD ||
       Operand.OperandType == MEMDWORD || Operand.OperandType == REG16) {

      Codes[0] |= 0x01;   /* set word bit */
      if (Codes[0] == 0xB9)
         Codes[0] = 0xBA;
   }
   return (SetModrm(Operand, Codes));
}  /* end of SetWmodrm() */

/******************************** E O F ***********************************/
