/****************************************************************************
**
**  Name:  BASE.CPP
**
**  Description:
**      Routines for processing bases.
**
**  Status:  CODED
**
**  $Log:   S:/tbird/arcm332/omf86/base.cpv  $
** 
**    Rev 1.13   11 Oct 1994 09:53:50   joyce
** 1. Check for the exact string "STACK" for the stack base information.
** 2. Get base type from SymFindBaseIndexFromSegment(), so the symbol types
**    (code or data) will be consistant in the same base.
** 3. Abort from the program if the TFree() failed.
**
**    Rev 1.12   06 Sep 1994 09:51:42   joyce
** Change the size of base name buffer.
**
**    Rev 1.11   23 Aug 1994 14:23:42   joyce
** 1. Modify GetBaseName to set default to BASE_DATA.
** 2. Change the algorithm of handling base :
**    . return err if base is DATA and not found in SEGDEF table.
**    . Create base if base is Code and not yet created even if it's not in
**      SEGDEF table.
**    . Change the base type to BASE_CODE if the caller is ProcessLINNUM().
** 3. Add GetBaseTypeName to retrieve base type names from PWRVIEWS.INI.
**
**    Rev 1.10   03 Aug 1994 13:25:00   steve
** Joyce's changes for 386 build 11
**
**    Rev 1.9   25 Jul 1994 12:41:10   steve
** Joyce's omf changes for 386 build 9
**
**    Rev 1.8   14 Jul 1994 15:23:14   steve
** More changes for 386 build 8 from Joyce
**
**    Rev 1.4   17 Jun 1994 14:54:50   joyce
** Check the return code from LdrWarning(), if it's not GOOD then cleanup and
** return.
**
**    Rev 1.3   16 Jun 1994 13:29:58   joyce
** Use LdrWarning() to replace Warning() and remove err.h from this module.
**
**    Rev 1.2   02 Jun 1994 18:35:34   joyce
** Changes made as the result of code reviews.
**
**    Rev 1.1   25 May 1994 16:03:28   joyce
** Make DestroyName() public so it can be called in OMF86LoaderLoad();
** this is to stop memory leak.
**
**    Rev 1.0   24 May 1994 14:47:54   joyce
** Initial revision.
**
**  $Header:   S:/tbird/arcm332/omf86/base.cpv   1.13   11 Oct 1994 09:53:50   joyce  $
**
**  Copyright (C) 1994 Microtek International.  All rights reserved.
**
*****************************************************************************/

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#include <ctype.h>
#include <io.h>
#include <sys\timeb.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#ifndef _CLISRV_
#include "clisrv.h"
#endif

#ifndef _HEAP_
#include "heap.h"
#endif

#ifndef _INIUTIL_
#include "iniutil.h"
#endif

#ifndef __OMF86LDR__
#include "omf86ldr.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef __BASE__
#include "base.h"
#endif

                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/
SEG_INFO *SegInfo[MAX_SECTIONS];
U16 SegDefCnt;
STATIC U16 NameCnt;
STATIC CHAR *NamePtrBuf[MAX_NAMES];
STATIC U8 CacheBottom, NumInCache=0 ;
STATIC BASE_CACHE BaseCache[CACHE_SIZE];
STATIC U16 BaseCnt=0;
STATIC CHAR *NullName = NULL;
CHAR BaseTypeNames[MAX_BASE_TYPE_NAMES][BASE_NAME_LEN+1]; // 1st:code, 2nd:data


                       /****************************
                        *                          *
                        *    EXTERNAL VARIABLES    *
                        *                          *
                        ****************************/
extern BOOLEAN HasStackInfo;
extern LADDRESS StackTop;
extern U32     StackSize;
extern MODULE_BASE ModuleBase;

                       /****************************
                        *                          *
                        *     LOCAL PROTOTYPES     *
                        *                          *
                        ****************************/
PRIVATE RETCODE CreateSection(LPSTR, U16, BASE_ADDRESS, U16, U16, BASE_TYPE);
PRIVATE BASE_TYPE GetBaseType(LPSTR className);
PRIVATE SEG_INFO *BaseInSegDefTable(U16 segment);
                       /****************************
                        *                          *
                        *      EXECUTABLE CODE     *
                        *                          *
                        ****************************/
/******************************************************************************
**
**  ProcessLNAMES - only process during initial load
**
******************************************************************************/
RETCODE ProcessLNAMES(OMF86_FUNC_PARA *funcPara)
{
U8 length;
RETCODE err;
CHAR tmpName[NAME_LEN];

   if (funcPara->oFlag == MCREATE) {
      while (CurrentLocation() < funcPara->endLoc) {
         if ((err=GetName(funcPara->hFile, &length, tmpName)) != GOOD) {
            DestroyNames();
            return(err);
         }
         if (NameCnt >= MAX_NAMES)
            return(ER_MISSING_SECTION_NAME);
         if ((NamePtrBuf[NameCnt] = (CHAR *)TMalloc((U32)(length+1))) == NULL) {
            DestroyNames();
            return(ER_NO_MEMORY);
         }
         strncpy( NamePtrBuf[NameCnt++], tmpName, length+1 );
      }
   }
   return(GOOD);
}

/******************************************************************************
**
**  ProcessSEGDEF - only process during initial load
**
******************************************************************************/
RETCODE ProcessSEGDEF(OMF86_FUNC_PARA *funcPara)
{
ACBP aCBP;
SEG_DATA segData;
U16 segNameIndex, classNameIndex;
CHAR *segNamePtr, *classNamePtr;
RETCODE err;
CHAR segNameBuf[NAME_LEN];

   if (funcPara->oFlag == MCREATE) {
      segNamePtr = classNamePtr = NullName; // unnamed segments
      if ((err=OMFGetBytes(funcPara->hFile,
          (U8 *)&aCBP, sizeof(aCBP))) != GOOD) return(err);
      if (aCBP.alignment == 0 || aCBP.alignment == 5) {
         if ((err=OMFGetBytes(funcPara->hFile, (U8 *)&segData,
              sizeof(segData))) != GOOD)
            return(err);
         if (aCBP.alignment == 0) {
            if ((err=GetIndex(funcPara->hFile, &segNameIndex)) != GOOD)
               return(err);
            if ((err=GetIndex(funcPara->hFile, &classNameIndex)) != GOOD)
               return(err);
            if ((segNameIndex > 0 && segNameIndex <= NameCnt) &&
                (classNameIndex > 0 && classNameIndex <= NameCnt)) {
               segNamePtr = NamePtrBuf[segNameIndex-1];
               classNamePtr = NamePtrBuf[classNameIndex-1];
            }
         }
         if (!segNamePtr || *segNamePtr == NULL) {
            // create a name if it doesn't have one
            sprintf(segNameBuf,"_SEGDEF%d", SegDefCnt+1 );
            segNamePtr = segNameBuf;
         }
         if ((!HasStackInfo) && (strcmp(segNamePtr, "STACK") == NULL)) {
            U32 stackBaseOffset;

            HasStackInfo = TRUE;
            StackTop.segment = segData.segFrame;
            stackBaseOffset = segData.segOffset + segData.segLength;
            StackTop.offset  = stackBaseOffset > 0xFFFFL ?
                               0xFFFF :  (U16)stackBaseOffset;
            StackSize = segData.segLength;
         }
         if (SegDefCnt >= MAX_SECTIONS)
            return(ER_INDEX_OUT_OF_BOUNDS);
         if ((err = CreateSection(segNamePtr, SegDefCnt,
                                  segData.segFrame, segData.segOffset,
                                  segData.segLength,
                                  GetBaseType(classNamePtr))) != GOOD)
            return(err);
         SegDefCnt++;
      }
      else return(ER_REC_NOT_HANDLED);
   }
   return(GOOD);
}

/******************************************************************************
**
**  GetBaseIndex - search for the base of the input address, if not found
**                 create a new base.
**
******************************************************************************/
RETCODE GetBaseIndex(BASE_ADDRESS segment, SEG_ADDR_SPACE addrSpace,
                     BOOLEAN isCode, BASE_INDEX *baseIndex,
                     BASE_TYPE *baseType)
{
LOOP_VAR i;
CHAR baseName[NAME_LEN];
SEG_INFO *segPtr=NULL;
RETCODE err;

   if (ModuleBase.baseKnown && ModuleBase.baseAddr == (U16)segment) {
      *baseIndex = ModuleBase.baseIndex;
      *baseType = ModuleBase.baseType;
      return(GOOD);
   }
   /* First lookup the base cache */
   for (i=0; i < NumInCache; i++) {
      if (BaseCache[i].baseAddr == segment &&
          BaseCache[i].addrSpace == addrSpace) {
         *baseIndex = BaseCache[i].baseIndex;
         *baseType = BaseCache[i].baseType;
         return(GOOD);
      }
   }
   *baseType = BASE_DATA;
   /*  Find base index from Symbol Table */
   if (SymFindBaseIndexFromSegment(segment, addrSpace, baseIndex, baseType)
        != GOOD) {
      /* See if it's defined in SEGDEF, if yes, get the base name */
      if (SegDefCnt > 0)
         segPtr = BaseInSegDefTable((U16)segment);
      if (segPtr) {
         lstrcpy((LPSTR)baseName, (LPSTR)segPtr->name);
         if (isCode)
            segPtr->baseType = BASE_CODE;
         *baseType = segPtr->baseType;
      }
      else if (isCode) { //if a code base is not defined in SEGDEF, add it anyway
         *baseType = BASE_CODE;
         sprintf( baseName, "_seg%d", BaseCnt+1 );
      }
      else return(ERR_BAD_BASE);
      if ((err=SymAddBaseCreateFromSegment(baseName, BaseCnt, segment,
           *baseType, addrSpace)) != GOOD) return(err);
      *baseIndex = BaseCnt++;
   }
   BaseCache[CacheBottom].baseAddr = segment;
   BaseCache[CacheBottom].addrSpace = addrSpace;
   BaseCache[CacheBottom].baseType = *baseType;
   BaseCache[CacheBottom++].baseIndex = *baseIndex;
   CacheBottom %= CACHE_SIZE;
   if (NumInCache < CACHE_SIZE) NumInCache++;
   return( GOOD );
}  /* GetBaseIndex */

/******************************************************************************
**
**  GetBaseTypeNames
**
**  Description:
**     Extract the base type names used by tool chains from PWRVIEWS.INI file.
**     Save the base type name into toolUseName variable.
**
******************************************************************************/
RETCODE GetBaseTypeNames(VOID)
{
CHAR tmpStr[NAME_LEN];

   /* Retrieve base type names for OMF86 Loader */
   if (IniGetString((LPSTR) TOOL_CHAIN,  (LPSTR) OMF_BASE_TYPE,
                    PWRVIEWS_INI_DEFAULT, (LPSTR)tmpStr) != GOOD)
      return(ER_NO_SECTION_NAME_DEFINED);
   else {
      /* Parse the string into base type names */
      LOOP_VAR i=0;
      CHAR *loopPtr = tmpStr, *baseTypePtr = NULL;
      U16 nameLen;

      do {
         /* strtok return the pointer to the token found in tmpStr */
         if ((baseTypePtr = strtok(loopPtr, DELIMITER_SET)) != NULL) {
            nameLen = strlen(baseTypePtr) + 1; /* string terminator */
            if ((nameLen > BASE_NAME_LEN) ||
                (strncpy(BaseTypeNames[i], baseTypePtr, nameLen) == NULL)) {
               return(ER_INVALID_SECTION_NAME);
            }
         }
         else {
            if (i < MAX_BASE_TYPE_NAMES)
               return(ER_MISSING_SECTION_NAME);
            break;
         }
         ++i;
         loopPtr = NULL; /* Reset to walk the rest of the string tmpStr */
      } while (i < MAX_BASE_TYPE_NAMES);
   }
   return(GOOD);
}

/******************************************************************************
**
**  DestroyNames - destroy LNAMES table
**
******************************************************************************/
VOID DestroyNames(VOID)
{
LOOP_VAR i=0;

  for (i=0; i < NameCnt; i++) {
     if (NamePtrBuf[i]) {
       if (TFree(NamePtrBuf[i]) != GOOD) {
          if (LdrWarning(ER_BAD_FREE, NULL, NULL, NULL) != GOOD)
             return;
       }
     }
  }
  memset(NamePtrBuf, NULL, sizeof(NamePtrBuf));
  NameCnt =0;
}  /* DestroyNames */

/******************************************************************************
**
**  FreeSections - free memory associated with section info struct
**
******************************************************************************/
RETCODE FreeSections(VOID)
{
LOOP_VAR i;
RETCODE err;

   for (i = 0; i < SegDefCnt; i++) {
      if (TFree((LPSTR)SegInfo[i]) != GOOD) {
         err = LdrWarning(ER_BAD_FREE, NULL, NULL, NULL);
         return( err == GOOD ? ER_BAD_FREE : err);
      }
   }
   DestroyNames();
   SegDefCnt = 0;
   BaseCnt=0;
   NumInCache = CacheBottom = 0;
   memset(BaseCache, NULL, sizeof(BaseCache));
   memset(BaseTypeNames, NULL, sizeof(BaseTypeNames));
   return(GOOD);
}  /* FreeSections */

/******************************************************************************
**
**  CreateSection - create section for SEGDEF's
**
******************************************************************************/
PRIVATE RETCODE CreateSection(LPSTR secName, U16 secIndex,
                              BASE_ADDRESS baseAddr, U16 offset,
                              U16 baseSize, BASE_TYPE baseType)
{
SEG_INFO *psection;

   if ((psection = (SEG_INFO *)TMalloc((U32)sizeof(SEG_INFO))) == NULL) {
      return(ER_NO_MEMORY);
   }
   lstrcpy((LPSTR)psection->name, (LPSTR)secName);
   psection->baseIndex = secIndex;
   psection->size = baseSize;
   psection->frame = (U16)baseAddr;
   psection->offset = offset;
   psection->baseType = baseType;

   /* Linked to SegInfo table */
   SegInfo[secIndex] = psection;
   return(GOOD);
} /* CreateSection */

/******************************************************************************
**
**  GetName
**
******************************************************************************/
RETCODE GetName(HANDLE hFile, U8 *length, CHAR *name)
{
RETCODE err;

   if ((err=OMFGetBytes(hFile, length, sizeof(U8))) != GOOD)
      return(err);
   if (*length > 0) {
      if ((err=OMFGetBytes(hFile, (U8 *)name, (U16)*length)) != GOOD)
         return(err);
   }
   name[*length] = '\0';
   return(GOOD);
} /* GetName */

/******************************************************************************
**
**  GetBaseType - examine the class name to decide the base type
**
******************************************************************************/
PRIVATE BASE_TYPE GetBaseType(LPSTR className)
{
   if (className) {
      if (((*BaseTypeNames[0] != '\0') &&
          (strstr((CHAR *)className, BaseTypeNames[0]) != NULL)) ||
          (strstr((CHAR *)className, "CODE") != NULL))
         return(BASE_CODE);
   }
   return(BASE_DATA);
}  /* GetBaseType */

/******************************************************************************
**
**  BaseInSegDefTable - Find base in SEGDEF Table
**
******************************************************************************/
PRIVATE SEG_INFO *BaseInSegDefTable(U16 segment)
{
LOOP_VAR i;

   for (i=0; i < SegDefCnt; i++) {
      if (!(SegInfo[i]->frame-segment))
         return(SegInfo[i]);
   }
   return(NULL);
}  /* CompareWithSegDefTable */

/******************************** E O F ***********************************/
