/*----------------------------------------------------------------------------
** Name: ordtbl.h
**
** Title: Ordinal Table
**
** Purpose:
**  Class and methods for managing the ordinal table - list of offsets to
**  symbols in the order they are loaded.
**
** Status: PRELIMINARY | CODED
**
** $Log:   S:/tbird/arccore/symbol/ordtbl.h_v  $
** 
**    Rev 1.4   06 Dec 1991 17:04:30   john
** Added member function to free the allocated memory for the 
** ordinal table.
** 
**    Rev 1.3   06 Nov 1991 14:50:02   brucea
** Added AddSymOffsetToOT declaration
** 
**    Rev 1.2   13 Sep 1991 14:39:18   brucea
** Added GetOffset method
** 
**    Rev 1.1   13 Sep 1991 09:52:50   brucea
** No changes
** 
**    Rev 1.0   09 Sep 1991 15:13:04   brucea
** Initial revision.
** 
** $Header:   S:/tbird/arccore/symbol/ordtbl.h_v   1.4   06 Dec 1991 17:04:30   john  $
**
** Copyright (C) 1991 Microtek International.  All rights reserved.
**
**--------------------------------------------------------------------------*/

#define _ORDTBL_

#ifndef _MEMPOOL_
#include "mempool.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

typedef U32 ORDINAL_INDEX;

                       /**********************************
                        *                                *
                        *   MEMBER FUNCTION PROTOTYPES   *
                        *                                *
                        **********************************/

class OrdinalTable : public MemPool {

public:

//------------------------------------------------------------------------
// OrdinalTable() - contructor
//
// Purpose:
//    Initialize ordinal table variables for table creation and expansion.
//---------------------------------------------------------------------------
OrdinalTable();    // constructor


//------------------------------------------------------------------------
// AddSymOffsetToOT
//
// Purpose:
//    Put the offset to the symbol into the next free space in the ordinal
//    table.  Return the index number where loaded.
//
// Input parameters:
//    symOffset:
//       ORDINAL_INDEX index into ordinal table; the index is an even size
//       of a TABLE_OFFSET
//
// Output parameters:
//    ordinalIndex:
//       ORDINAL_INDEX index (into ordinal table) where symbol offset placed
//
// Error:
//    Returns calling error messages
//------------------------------------------------------------------------
RETCODE
AddSymOffsetToOT(TABLE_OFFSET symOffset, ORDINAL_INDEX& ordIndex);


//------------------------------------------------------------------------
// GetOffset
//
// Purpose:
//    Retrieve offset to symbol table item given the index to the ordinal
//    table.
//
// Input parameters:
//    ordIndex:
//       ORDINAL_INDEX index into ordinal table; the index is an even size
//       of a TABLE_OFFSET
//
// Output parameters:
//    itemOffset:
//       offset into symbol table of item
//
// Error:
//    Reports ER_INDEX_NOT_IN_TABLE if the index field is NULL or
//    Reports ER_INDEX_OUT_OF_BOUNDS if the requested index is larger than
//       last item in the table or larger than the table size
//------------------------------------------------------------------------
RETCODE GetOffset(ORDINAL_INDEX ordIndex,
                  TABLE_OFFSET& itemOffset);


//------------------------------------------------------------------------
// GetOrdinalCount
//
// Purpose: returns the number of symbol offsets placed into table
//-----------------------------------------------------------------------
U32 GetOrdinalCount(void) { return (GetAvailMemOffset()
                            / sizeof(TABLE_OFFSET)); };

//------------------------------------------------------------------------
// GetOrdinalFromPercent
//
// Purpose:
//    Given a percentage value (0 - 100), return the ordinal value
//    (location) closest to the percentage value requested.
//    e.g. 0  (%) = first location
//         100(%) = last location
//
// Input parameter:
//    percent: desired percentage of total entry count
//
// Return value:
//    ordinal index (0 to n) that is the closest to the percentage of the
//    total table size requested
//
// Errors: No errors; limits percent to 100
//------------------------------------------------------------------------
U32 GetOrdinalFromPercent(U8 percent);

//
// Purpose:
//    Unlocks, then frees the table to Windows memory.
//---------------------------------------------------------------------------
void ObliterateTable(void);

};  // end of class OrdinalTable

/******************************** E O F *************************************/