/*****************************************************************************
**
**  Name: ldrobj.h
**
**  Description:
**     Declaration of the Loader Server classes.
**
**  Status:  PRELIMINARY | CODED | REVIEWED | TESTED
**
**  $Log:   S:/tbird/arcm332/loader/ldrobj.h_v  $
** 
**    Rev 1.6   13 Oct 1994 17:42:00   nghia
** Added loadFormat data member as the format id for each loader.
** 
**    Rev 1.5   11 Oct 1994 10:28:40   nghia
** Added LOAD_FORMAT enumeration - order match LOADERS.INI file.
** Added new member function to support load ondemand.
** Extended the LdrLoadByDesc() to use load options.
** 
**    Rev 1.4   01 Sep 1994 18:00:26   nghia
** Added IniLoadRegister to class.
** 
**    Rev 1.3   20 Jul 1994 09:58:02   nghia
** Added inline member function to set load options and load space.
** 
**    Rev 1.2   14 Jul 1994 10:47:46   nghia
** Added load address space to interface functions.
** Added routine to save and restore load options.
** 
**    Rev 1.1   20 May 1994 10:02:54   nghia
** Revised from code review.
** 
**    Rev 1.0   18 May 1994 16:58:46   nghia
** Initial revision.
**
**  $Header:   S:/tbird/arcm332/loader/ldrobj.h_v   1.6   13 Oct 1994 17:42:00   nghia  $
**
**  Copyright (C) 1994 Microtek International.  All rights reserved.
**
*****************************************************************************/
#ifndef __LDROBJ__
#define __LDROBJ__

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _PROC_
#include "proc.h"
#endif

#ifndef _SYMBLSVR_
#include "symblsvr.h"
#endif

#ifndef _LOADER_
#include "loader.h"
#endif


                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/
// Specific Loader - External Interface Function prototypes
typedef RETCODE EXPORT (*FP_LOADPROC) (LPSTR, U32, HWND, ADDR_SPACE);
typedef RETCODE EXPORT (*FP_LOADBYDESCPROC) (LPSTR, LPSTR,
                                             SYM_DESCRIPTOR, ADDR_SPACE,
                                             U32);
typedef RETCODE EXPORT (*FP_GETPCPROC) (DESCRIPTOR FAR *);
typedef RETCODE EXPORT (*FP_GETSTKPROC) (DESCRIPTOR FAR *, U32 FAR *);

// Specific Loader External Interface Function Id
typedef enum {
   LDR_LOAD = 0,
   LDR_LOADBYDESC,
   LDR_GETPC,
   LDR_GETSTK,
   LDR_MAX_FUNC  // must be the last element in the enum list
} LDR_FUNC_TYPE;             

// Load formats supported by the loader launcher
// These are load format id matching the order of loader in LOADER.INI
typedef enum {
   IEEE695 = 0,
   SRECORD,
   OMF86,
   OMF386,
   INHEX,
   UNKNOWN = 0xff          
} LOAD_FORMAT;

//----------------------------------------------------------------------------
// Debug assertion macro
//
// Notes:  If DEBUG is defined then assert macro will be generated.
// The assert() macro popups a message box with the sorce file name and line
// number if the assertion is failed - the expression x is evaluated to FALSE.
//
#undef assert
#ifdef DEBUG
#  define assert(x) ((void) ((x) || (DoAssert(#x,__FILE__,__LINE__), 1)))
#else
#  define assert(ignore) ((void) 0)
#endif

//---------------------------------------------------------------------------
// Global DEBUG functions
extern VOID DoAssert(const LPSTR msg, const LPSTR file, U32 line);

#define MAX_FILE_NAME 15  
#define MAX_FUNC_NAME 40

#define MAX_SIGNATURES 5
#define DELIMITER_SET  ","
#ifdef __cplusplus

//----------------------------------------------------------------------------
// Classes Declarations
class LdrLauncher;
class Loader;

// Pointer to object
typedef LdrLauncher *PTLdrLauncher;
typedef Loader *PTLoader; 

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

//-----------------------------------------------------------------------------
//
// Class: Loader
//
// Description:
//    Declaration of the Loader class.  This is a class represents objects
//    that contain members to be used in connnecting and launching the
//    specific loader DLL.  This class is a private class of the LdrLauncher.
//
//-----------------------------------------------------------------------------
class Loader {
private:

                    //--------------------------//
                    // CONSTRUCTOR & DESTRUCTOR //
                    //--------------------------//
   Loader(const LPSTR dllName = NULL, U8 numSig = 0,
      U8* sigList = NULL, LOAD_FORMAT formatId);
   ~Loader();
                    //--------------------------//
                    //   INITIALIZED METHODS    //
                    //--------------------------//
   RETCODE Init(const LPSTR dllName, U8 numSig, U8* sigList,
                LOAD_FORMAT formatId);

                    //--------------------------//
                    //    ACCESSING METHODS     //
                    //--------------------------//
   RETCODE GetFunction(LDR_FUNC_TYPE fnId, FARPROC *lpFunc);
   RETCODE LastErr();
   VOID SaveLoadOptions(U32 flags);
   U32 LastLoadOptions();
   LOAD_FORMAT LoadFormat();
   
                    //--------------------------//
                    //    OPERATOR METHODS      //
                    //--------------------------//

                    //--------------------------//
                    //    OPERATION METHODS     //
                    //--------------------------//
   RETCODE LoadLib(VOID);
   RETCODE UnLoadLib(VOID);
   RETCODE IsLoadFormatMatched(const LPSTR loadFileName, BOOLEAN *found);
   PROCESSOR_FAMILY ProcessorFamilySupported(VOID);

private:
                    //--------------------------//
                    //       DATA MEMBERS       //
                    //--------------------------//
   // Loader Class Members - function names of all specific loader
   static CHAR *ExportFuncNames[LDR_MAX_FUNC];
   
   // Loader general information
   CHAR      libName[MAX_FILE_NAME];
   HANDLE    hLib;   
   RETCODE   lastErr;
   // table of Loader external interfaces
   FARPROC   funcTable[LDR_MAX_FUNC];
   
   // load format Id and signatures
   LOAD_FORMAT loadFormat;
   U8          numSignatures; // signatures that the Loader supported 
   CHAR        signatureList[MAX_SIGNATURES]; // list of numSignatures

   // Load options used for ondemand loading
   U32 loadOptions;

//---------------------------------------------------------------------------
// FRIEND
friend LdrLauncher;

};
//// End of Loader
                    
//-----------------------------------------------------------------------------
//
// Class: LdrLauncher
//
// Description:
//    Declaration of the LdrLauncher class.  This class represents the
//    generic loader server.  It provides function and data members
//    to dynamically launch the specific loader according to the format
//    of the input loadfile.
//
//-----------------------------------------------------------------------------
class LdrLauncher {
public:

                    //--------------------------//
                    // CONSTRUCTOR & DESTRUCTOR //
                    //--------------------------//
   LdrLauncher();
   ~LdrLauncher();
   
                    //--------------------------//
                    //   INITIALIZED METHODS    //
                    //--------------------------//
   RETCODE Init();
   
                    //--------------------------//
                    //    ACCESSING METHODS     //
                    //--------------------------//
   RETCODE Status() { return status; }
   BOOLEAN IsLoading() { return isLoading; }
   const TIMESTAMP_TYPE FAR *LastLoadFileTimeStamp() {
      return &lastLoadFileTimeStamp; }
   const LPSTR LastLoadFileName(VOID) { return (LPSTR) lastLoadFileName; }
   U32 LastLoadOptions(VOID) { return lastLoadOptions; }
   VOID LastLoadOptions(U32 options) { lastLoadOptions = options; }
   HWND LastParentHwnd(VOID) { return lastParentHwnd; }
   ADDR_SPACE LastLoadSpace(VOID) { return lastLoadAddrSpace; }
   VOID LastLoadSpace(ADDR_SPACE loadSpace) { lastLoadAddrSpace = loadSpace; }
   RETCODE EXPORT GetLoadFormat(const LPSTR fileName,
                                LOAD_FORMAT *loadFormat);   

                    //--------------------------//
                    //    OPERATOR METHODS      //
                    //--------------------------//

                    //--------------------------//
                    //    OPERATION METHODS     //
                    //--------------------------//
   RETCODE Load(const LPSTR loadFileName, U32 loadOptions,
                HWND parentHWnd, ADDR_SPACE loadAddrSpace);
   RETCODE LoadModuleByDesc(const LPSTR loadFileName, const LPSTR moduleName,
                            SYM_DESCRIPTOR moduleDesc,
                            ADDR_SPACE loadAddrSpace);
   RETCODE GetStartPC(DESCRIPTOR FAR* lpAddrDesc);
   RETCODE GetStackInfo(DESCRIPTOR FAR* lpAddrDesc, U32 FAR* stkSize);
   RETCODE GetLoadInfo(LPSTR lpFileName, U32 FAR* lpOptions,
                       ADDR_SPACE FAR *lpLoadAddrSpace);

                    //--------------------------//
                    //       CLASS METHODS      //
                    //--------------------------//

protected:

private:
                    //--------------------------//
                    //       LOCAL METHODS      //
                    //--------------------------//

   RETCODE GetLoader(const LPSTR loadFileName);
   RETCODE GetLoaderFunc(LDR_FUNC_TYPE funcId, FARPROC* lpLdrFunc);
   RETCODE GetLoadOptions(VOID);
   RETCODE SaveLoadOptions(VOID);

private:
                    //--------------------------//
                    //       DATA MEMBERS       //
                    //--------------------------//
   //------------------------------------------------------------------------
   // LdrLauncher class members
   static LPSTR IniFileName, IniSectionName, IniNumLoaders, IniLoaderItem;
   static LPSTR IniLoadOptions, IniLoadCode, IniLoadSymbol, IniLoadStatus,
                IniLoadWarning, IniLoadAsm, IniLoadOnDemand, IniLoadDemangle,
                IniLoadUpdateBase, IniLoadRegister, IniLoadIgnoreCase;
   static CHAR PwrViewsDir[MAX_PATH_SIZE];
   static BOOLEAN HasPwrViewsDir;
   
   // LdrLauncher private object data      
   PTLoader   *ldrObjTable;       // table of Loader Object pointers
   U8         numLoaders;         // number of loaders in LOADERS.INI
   PTLoader   lastActiveLoader;   // the last active loader

   // LdrLauncher public object data
public:
   RETCODE    status;              // current status of LdrLauncher object
   BOOLEAN    isLoading;           // semaphore to prevent re-entrancy

   //-----------------------------------------------------------------------
   // Global variables of the the last load operation.
   CHAR             lastLoadFileName[MAX_STRING_SIZE];        
   U32              lastLoadOptions;
   ADDR_SPACE       lastLoadAddrSpace;
   TIMESTAMP_TYPE   lastLoadFileTimeStamp;
   HWND             lastParentHwnd; 
   
//---------------------------------------------------------------------------
// FRIEND
friend RETCODE EXPORT LdrCliResetLoaders(LPSTR cmdString, U32 argc,
                                         U32 argv[]);

};
//// End of LdrLauncher

#endif /* __cplusplus */
#endif /* __LDROBJ__ */

/******************************** E O F *************************************/
