/****************************************************************************
**
**  Name:  trig.c
**
**  Description:
**     Provides shared data server handling routines for trigger.
**
**  Status:  CODED
**
**  $Log:   S:/tbird/arcm332/flash/trig.c_v  $
** 
**    Rev 1.7   05 Aug 1993 12:27:26   ernie
** Changed types of tc1,tc1 variables to match shared data sizes.  The
** size mismatch interfered with timer mode.
** 
**    Rev 1.6   28 Jul 1993 13:19:38   ernie
** Added GetYoungestFrame() prototype
** 
**    Rev 1.5   09 Jul 1993 16:44:24   ernie
** Fixed problem with event 6 triggering.  Xilinx def was wrong.
** 
**    Rev 1.4   01 Jun 1993 12:24:56   ernie
** Rearranged EvtrecTask() for efficiency
** 
**    Rev 1.3   26 May 1993 15:52:08   john
** moved function call to be called only when necessary
** 
**    Rev 1.2   19 May 1993 15:28:18   john
** Corrected bug in finding TRACE STORE FULL info
** 
**    Rev 1.1   10 Mar 1993 12:16:04   ernie
** Added support for show cycle triggering and tracing
** 
**    Rev 1.0   16 Dec 1992 15:26:34   mindy
** moved trace to fw
** 
**  $Header:   S:/tbird/arcm332/flash/trig.c_v   1.7   05 Aug 1993 12:27:26   ernie  $
**
**  Copyright (C) 1992 Microtek International.  All rights reserved.
**
*****************************************************************************/

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _STDIO_
#include "stdio.h"
#endif
#ifndef _CPMAP_
#include "cpmap.h"
#endif
#ifndef _DESCRIP_
#include "descrip.h"
#endif
#ifndef _ERROR_
#include "error.h"
#endif
#ifndef _INTERNAL_
#include "internal.h"
#endif
#ifndef _MEMORY_H
#include "memory.h"
#endif
#ifndef _REGCTRL_
#include "regctrl.h"
#endif
#ifndef _SDPROBE_
#include "sdprobe.h"
#endif
#ifndef _SSHARED_
#include "sshared.h"
#endif
#ifndef _TRACE_
#include "trace.h"
#endif
#ifndef _TRIG_
#include "trig.h"
#endif
#ifndef _VRTX_
#include "vrtx.h"
#endif
#ifndef _XILINX_
#include "xilinx.h"
#endif

/*!!!*/
#undef PRIVATE
#define PRIVATE  

                        /****************************
                         *                          *
                         *     LOCAL DEFINITIONS    *
                         *                          *
                         ****************************/
#define INIT_NUM_TRCBUFS    1
#define TOTAL_NUM_SUB_BUFFERS 1024
#define DMA_DISABLE  0x0C
#define DMA_ENABLE   0x08

#define CLB_EVTREC32_QUAL  0x110A   /* RK */
#define CLB_EVTREC32_FRAME 0x0200   /* CA */
#define CLB_EVTREC32_DELAY0 0x0C0A  /* MK */
#define CLB_EVTREC32_DELAY1 0x110B  /* RL */
#define CLB_EVTREC8_QUAL0  0x110B   /* RL */
#define CLB_EVTREC8_QUAL1  0x0F0B   /* PL */
#define CLB_EVTREC8_FRAME  0x0101   /* BB */

typedef struct {
   EVTREC_TYPE type;
   U16 func1;
   U16 func2;
   U8 frameQual;
   BOOLEAN useDelayed;
} EVTREC_CONFIG;
/*
**    word 0:  bit 15:  1=trace store full (has wrapped)
**             bit 14:  1=trace buffer full (has wrapped)
**             bit 13:  1=trace buffer was entered by trigger-continue
**             bit 12-10:  unused
**             bit 9-0:  physical sub buffer number where trigger occurred
**    word 1:  bit 15-8:  unused
**             bit 7-0:  frame number within sub buffer where trigger is
**                      (valid regardless of size of sub buffer 128 or 256)
*/
typedef struct {
   U16 subBuffer:10;
   U16 Unused1:3;
   U16 trigCont:1;
   U16 bufferFull:1;
   U16 storeFull:1;
   U8 frame;
   U8 Unused2;
} TRIG_POS;

static const U8 evtrec32RawBits[] = {
#include "evtrec32.h"
};

static const U8 evtrec8RawBits[] = {
#include "evtrec8.h"
};

static U8 *evtrec32Bits;

static DESCRIPTOR descQualMode;
static BOOLEAN triggerAbort = FALSE;
static U16 subBufferSize;

static EVTREC_CONFIG evtrecConfig[NUM_GROUPS] = {
   { EVTREC32, 0, 0, 1, FALSE },
   { EVTREC32, 0, 0, 1, FALSE },
   { EVTREC32, 0, 0, 1, FALSE },
   { EVTREC32, 0, 0, 1, FALSE }};

/* This table is the clb positions of the evtrec comparators in evtrec32 */
/* Note: 1st subscript is the row, 2nd is column in following table */
static const U16 evtLowClb[8][8] = {
   { 0x0000, 0x0002, 0x0004, 0x0006, 0x0008, 0x000A, 0x000C, 0x000E },
   { 0x0300, 0x0302, 0x0304, 0x0306, 0x0308, 0x030A, 0x030C, 0x020E },
   { 0x0500, 0x0502, 0x0504, 0x0506, 0x0508, 0x050A, 0x050C, 0x050E },
   { 0x0800, 0x0802, 0x0804, 0x0806, 0x0808, 0x080A, 0x080C, 0x070E },
   { 0x0A00, 0x0A02, 0x0A04, 0x0A06, 0x0A08, 0x0A0A, 0x0A0C, 0x0A0E },
   { 0x0D00, 0x0D02, 0x0D04, 0x0D06, 0x0D08, 0x0D0A, 0x0D0C, 0x0C0E },
   { 0x0F00, 0x0F02, 0x0F04, 0x0F06, 0x0F08, 0x0F0A, 0x0F0C, 0x0F0E },
   { 0x1200, 0x1202, 0x1204, 0x1206, 0x1208, 0x120A, 0x120C, 0x110E }};
static const U16 evtHighClb[8][8] = {
   { 0x0100, 0x0102, 0x0104, 0x0106, 0x0108, 0x010A, 0x010C, 0x010E },
   { 0x0400, 0x0402, 0x0404, 0x0406, 0x0408, 0x040A, 0x040C, 0x040E },
   { 0x0600, 0x0602, 0x0604, 0x0606, 0x0608, 0x060A, 0x060C, 0x060E },
   { 0x0900, 0x0902, 0x0904, 0x0906, 0x0908, 0x090A, 0x090C, 0x090E },
   { 0x0B00, 0x0B02, 0x0B04, 0x0B06, 0x0B08, 0x0B0A, 0x0B0C, 0x0B0E },
   { 0x0E00, 0x0E02, 0x0E04, 0x0E06, 0x0E08, 0x0E0A, 0x0E0C, 0x0E0E },
   { 0x1000, 0x1002, 0x1004, 0x1006, 0x1008, 0x100A, 0x100C, 0x100E },
   { 0x1300, 0x1302, 0x1304, 0x1306, 0x1308, 0x130A, 0x130C, 0x130E }};

                        /****************************
                         *                          *
                         *    EXTERNAL VARIABLES    *
                         *                          *
                         ****************************/
/*
** This array is the destination of DMA cycles initiated by the TMAN when
** a trigger occurs.  It must be word-aligned and totally contained within
** a single 64k physical memory block.
*/
extern TRIG_POS triggerPos[MAX_TRCBUFFERS];

                        /****************************
                         *                          *
                         *     LOCAL PROTOTYPES     *
                         *                          *
                         ****************************/
/*
** Shared Data Server Handler routines
*/
RETCODE PRIVATE TriggerProgramTask(DESCRIPTOR desc) ;
RETCODE PRIVATE TriggerProgram(DESCRIPTOR desc);
RETCODE PRIVATE EvtrecTask(DESCRIPTOR desc);
/*
** Other local routines
*/
RETCODE PRIVATE ProgramEvtrec32(MEMBER_INDEX index, XLX_CHIP chip);
RETCODE PRIVATE ProgramEvtrec8(MEMBER_INDEX index, XLX_CHIP chip);
RETCODE PRIVATE TriggerProgramAbort(DESCRIPTOR desc);

                        /****************************
                         *                          *
                         *      EXECUTABLE CODE     *
                         *                          *
                         ****************************/

RETCODE EXPORT SetEvtrecConfig(TRACE_GROUP group, EVTREC_TYPE type,
                                U16 func1, U16 func2, BOOLEAN useDelayed) {
   evtrecConfig[group].type = type;
   evtrecConfig[group].func1 = func1;
   evtrecConfig[group].func2 = func2;
   evtrecConfig[group].useDelayed = useDelayed;
   return(GOOD);
}

RETCODE EXPORT GetEvtrecConfig(TRACE_GROUP group, EVTREC_TYPE *type,
                                U16 *func1, U16 *func2, BOOLEAN *useDelayed) {
   *type = evtrecConfig[group].type;
   *func1 = evtrecConfig[group].func1;
   *func2 = evtrecConfig[group].func2;
   *useDelayed = evtrecConfig[group].useDelayed;
   return(GOOD);
}

RETCODE EXPORT SetEvtrecFramequal(TRACE_GROUP group, U8 frameQual) {
   evtrecConfig[group].frameQual = frameQual;
   return(GOOD);
}

RETCODE EXPORT GetEvtrecFramequal(TRACE_GROUP group, U8 *frameQual) {
   *frameQual = evtrecConfig[group].frameQual;
   return(GOOD);
}

RETCODE EXPORT TrigInit(VOID) {
   RETCODE err;
   DESCRIPTOR desc;
   MEMBER_NAME fullName[SD_MAX_NAME];
   LOOP_VAR i;
/*
** Register routines to be called when shared data is written.
** Write initial values to members as they are registered so we don't have
** to keep the descriptor.  Some of these need to actually fill data into
** hardware. This is done by re-registering with a null callback so that
** SdWriteMember() will call the callback.
*/
   if((err = SdnRegister(SDN_TRIGGER_INFO,TriggerProgramTask,&desc))!=GOOD)
      return(err);
   if((err=SdnRegister(SDN_TRIGGER_PROG_ABORT,TriggerProgramAbort,&desc))
      != GOOD) return(err);
   {
      FW_TRIGGER fw;
      U16 level,cond;
      if((err = SdnRegister(SDN_TRIGGER_INFO,NULLPTR,&desc))!=GOOD) 
         return(err);
      for (level=0; level<NUM_LEVELS; level++) {
         for (cond=0; cond<NUM_EVENT_COND; cond++) {
            fw.useEvt[cond][level] = fw.useExt[cond][level] = FALSE;
            fw.action[cond][level] = NULL_MASK;
         }
      }
      if((err = SdWriteMember(desc,&fw,GOOD)) != GOOD) return(err);
      if ((err=SdUnRegister(desc)) != GOOD) return(err);
   }
  
   {
      QUAL_MODE data = QUAL_BUS;
      if ((err = SdnWriteMember(SDN_QUAL_MODE,&data,GOOD))!=GOOD) return(err);
   }

/*
** Copy evtrec32 bitstream into RAM, clear out programming for comparators.
** This will then be used as the starting point for evtrec programming.
** This is done to save time in programming.
*/
   {
      DESCRIPTOR xlxDesc;
      U8 row, col;
      if ((err = XlxOpen(XC3090, evtrec32RawBits, &xlxDesc)) != GOOD)
          return(err);
      if ((err = XlxCopyBitstream(xlxDesc)) != GOOD) return(err);
      if ((err = XlxGetBitstream(xlxDesc, &evtrec32Bits)) != GOOD) return(err);
      for (row=0;row<8;row++) for (col=0;col<8;col++) {
         if ((err = XlxModifyClb(xlxDesc,evtLowClb[row][col],XLX_F,0,0,0))
            !=GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,evtLowClb[row][col],XLX_G,0,0,0))
            !=GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,evtHighClb[row][col],XLX_F,0,0,0))
            !=GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,evtHighClb[row][col],XLX_G,0,0,0))
            !=GOOD) return(err);
      }
   }

   for(i=0; i<NUM_GROUPS; i++) {
      MEMBER_SIZE evtrecSize;
      U32 *evtrecRam;
      LOOP_VAR j;
      sprintf(fullName, "%s %d", SD_EVTREC, i);
      if((err = SdnRegister(SDN_EVTREC+i,EvtrecTask,&desc))!=GOOD)
         return(err);
      if((err = SdnGetMemberSize(SDN_EVTREC+i,&evtrecSize)) != GOOD)
         return(err);
      if((evtrecRam=sc_gmem((U32)evtrecSize))==NULL) return(ER_NO_MEM);
      for (j=0; j<evtrecSize/sizeof(*evtrecRam); j++) evtrecRam[j] = 0;
      if((err = SdnWriteMember(SDN_EVTREC+i,evtrecRam,GOOD)) != GOOD) {
         sc_rmem(evtrecRam,(U32)evtrecSize);
         return(err);
      }
      if((err = sc_rmem(evtrecRam,(U32)evtrecSize))!=GOOD) return(err);
   }

   if((err = SdnRegister(SDN_QUAL_MODE,NULLPTR,&descQualMode))!=GOOD)
      return(err);

   subBufferSize = 128;   
   if( DoubleVRam() ) subBufferSize *= 2;

/*
** Set up DMA channels 6 (vram xfr) and 5 (trig pos) for TMAN
*/
   _outb(IO_BIU_DMA2_CSR, DMA_DISABLE);
   _outb(IO_BIU_DMA2_MODE,0xCA);              /* ch 6 cascade (TMAN master) */
   _outb(IO_BIU_DMA2_MASK,0x02);              /* enable channel 6 */
   _outb(IO_BIU_DMA2_MODE,0x15);              /* ch 5 demand mode autoinit */
   _outb(IO_BIU_DMA2_MASK,0x01);              /* enable channel 5 */
   _outb(IO_BIU_DMA_PAGE5,(U8)((OffsetOf(triggerPos)>>16) & 0xfe));
   _outb(IO_BIU_DMA2_ADDR1,(U8)((OffsetOf(triggerPos)>>1) & 0xff));
   _outb(IO_BIU_DMA2_ADDR1,(U8)((OffsetOf(triggerPos)>>9) & 0xff));
   _outb(IO_BIU_DMA2_COUNT1,lobyte(2*INIT_NUM_TRCBUFS-1));
   _outb(IO_BIU_DMA2_COUNT1,hibyte(2*INIT_NUM_TRCBUFS-1));
   _outb(IO_BIU_DMA2_CSR, DMA_ENABLE);

   return(GOOD);
}

RETCODE PRIVATE TriggerProgramAbort(DESCRIPTOR desc) {
   return(SdReadMember(desc,&triggerAbort));
}

/****************************************************************************
**
**  TriggerProgramTask
**
**  Description:
**     First gets the trigger programming information then creates the
**     entire sequencer ram setup. Next writes the entire sequencer ram.
**     Called when the host write the trigger information in the shared
**     data server.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor
**     output:
**        none
**
*****************************************************************************/
#define SIZE_SEQUENCER     16384
/*
** Sequencer data bits
*/
#define SEQ_EXTOFF_BIT   0
#define SEQ_EXTON_BIT    1
#define SEQ_BRK_BIT       2
#define SEQ_TRIG_CONT_BIT 3
#define SEQ_TRIG_HALT_BIT 4
#define SEQ_CLEAR1_BIT    5
#define SEQ_STOP1_BIT     6
#define SEQ_INC1_BIT      7
#define SEQ_RUN1_BIT      8
#define SEQ_CLEAR0_BIT    9
#define SEQ_STOP0_BIT     10
#define SEQ_INC0_BIT      11
#define SEQ_RUN0_BIT      12
#define SEQ_LEVEL_BIT     13
#define SEQ_NULL_VALUE    0
#define SEQ_LEVEL_MASK    0x9FFF
#define SEQ_SEQ_DUMMY_BIT     13
#define SEQ_RESET_DUMMY_BIT   14
#define SEQ_DUMMY_SEQ_RES_BITS ((1<<SEQ_SEQ_DUMMY_BIT)|(1<<SEQ_RESET_DUMMY_BIT))
/*
** Sequencer address bit
*/
#define SEQ_LEVEL_IN_BIT  11   /* Which sequencer address bit is seq level */
#define SEQ_LEVEL_IN_MASK 0x1800
#define SEQUENCER_WORDS   (SIZE_SEQUENCER / sizeof(U16))
#define SEQ_ACTIVE_LOW_MASK 0x1cff /* evt 0-7, exttrig, and sequence levels */

RETCODE PRIVATE TriggerProgramTask(DESCRIPTOR desc) {
   RETCODE err;
   err = TriggerProgram(desc);
   return(SdnWriteMember(SDN_TRIGGER_PROGRAM_ACK,&err,GOOD));
}
   
RETCODE PRIVATE TriggerProgram(DESCRIPTOR desc) {
   FW_TRIGGER fw;
   U16 *sequencer,index,level,cond;
   U16 tc0, tc1;
   PWORD *sequencerAlias = (PWORD*)&sequencer;
   RETCODE err;
   triggerAbort = FALSE;
   ClearTriggerPositions();   /* need to have this completely done before
                                 we program the sequencer - host sw is
                                 reading trigger info too soon. !!!!
                               */
   _outb(SEQENBWR, SEQ_PROG);  /* Enable writes to sequencer */
   sequencerAlias->selector = PL0_DATA_SEL;
   sequencerAlias->offset = MEM_SEQUENCER;
   if((err=SdReadMember(desc, &fw)) != GOOD ) return(err);
   if (triggerAbort) return(ER_TRIGGER_ABORTED);

   if((err=SdnReadMember(SDN_TERMINAL_COUNT_A,&tc0))!=GOOD) return(err);
   if((err=SdnReadMember(SDN_TERMINAL_COUNT_B,&tc1))!=GOOD) return(err);
   
   for (index=0; index<SEQUENCER_WORDS; index++) {
      sequencer[index] = SEQ_LEVEL_MASK | ((index & SEQ_LEVEL_IN_MASK) << 2);
   }
/* we can't leave sequencer in unknown state - so only check for abort
   when it's safe to leave.
*/      
   if (triggerAbort) return(ER_TRIGGER_ABORTED);
   for (level=0; level<NUM_LEVELS; level++) {
      for (cond=0; cond<NUM_EVENT_COND; cond++) {
         U16 addr=(level<<SEQ_LEVEL_IN_BIT);
         U16 mask=(3<<SEQ_LEVEL_IN_BIT);     /* only level bits significant */
         U16 data=SEQ_NULL_VALUE;
         if( !fw.useEvt[cond][level] ) continue;
         if( fw.counterMode == TIMER ) {
            /* ignore counter 1 specific actions in timer mode */
            if( cond == CNTR1_COND ) continue; 
            else if((cond == CNTR0_COND) && (tc0>0)){
               U16 tmpAddr = addr | TC0_COND;
               U16 tmpData = data | (1<<SEQ_INC1_BIT);
               U16 tmpMask = mask | TC0_COND;
               /* need to program sequencer to increment counter 1
                  when counter 0 reaches terminal count.
                */
               tmpAddr ^= SEQ_ACTIVE_LOW_MASK;
               for (index=0; index<SEQUENCER_WORDS; index++)
                  if ((index & tmpMask) == (tmpAddr & tmpMask))
                     sequencer[index]=~((~sequencer[index])|tmpData);
               /* need to do counter 0 actions when BOTH terminal
                  count 0 and 1 are present. 
                */
               if( tc1 > 0 ) {
                  addr |= (1<<CNTR1_COND);
                  mask |= (1<<CNTR1_COND);
               }
            }
         }
         addr |= (1<<cond);
         mask |= (1<<cond);
         if (fw.useExt[cond][level]) {
            addr |= EXT_COND;
            mask |= EXT_COND;
         }
         if (fw.action[cond][level] & SEQ_MASK)
            data |= (1<<SEQ_SEQ_DUMMY_BIT);
         if (fw.action[cond][level] & RESET_MASK)
            data |= (1<<SEQ_RESET_DUMMY_BIT);
         if (fw.action[cond][level] & TRIG_CONT_MASK)
            data |= (1<<SEQ_TRIG_CONT_BIT);
         if (fw.action[cond][level] & TRIG_HALT_MASK)
            /* trigh action requires activating both trigh and trigc bits */
            data |= (1<<SEQ_TRIG_HALT_BIT) | (1<<SEQ_TRIG_CONT_BIT);
         if (fw.action[cond][level] & EM_BRK_MASK) {
            data |= (1<<SEQ_BRK_BIT);
            /* force the TOFF bit set as well to get trigger at brkpt */
            data |= (1<<SEQ_TRIG_HALT_BIT) | (1<<SEQ_TRIG_CONT_BIT);
         }
         if (fw.action[cond][level] & EXT_TRIG_ON)
            data |= (1<<SEQ_EXTON_BIT);
         if (fw.action[cond][level] & EXT_TRIG_OFF)
            data |= (1<<SEQ_EXTOFF_BIT);
         if( fw.counterMode == COUNTERS ) {
            if(fw.action[cond][level] & INC0_MASK)
               data |= (1<<SEQ_INC0_BIT);
            if (fw.action[cond][level] & INC1_MASK)
               data |= (1<<SEQ_INC1_BIT);
            if (fw.action[cond][level] & RST0_MASK)
               data |= (1<<SEQ_CLEAR0_BIT);
            if (fw.action[cond][level] & RST1_MASK)
               data |= (1<<SEQ_CLEAR1_BIT);
         }
         else {
            if (fw.action[cond][level] & START_TMR_MASK)
               data |= (1<<SEQ_RUN0_BIT);
            if (fw.action[cond][level] & STOP_TMR_MASK)
               data |= (1<<SEQ_STOP0_BIT);
            if (fw.action[cond][level] & RST_TMR_MASK) {
               data |= (1<<SEQ_CLEAR0_BIT);
               data |= (1<<SEQ_CLEAR1_BIT);
            }
         }
         addr ^= SEQ_ACTIVE_LOW_MASK; /* invert active low address bits */
         /* Or data into sequencer (sequencer data bits are active low) */
         for (index=0; index<SEQUENCER_WORDS; index++) {
            if ((index & mask) == (addr & mask)) {
               if ((data & SEQ_DUMMY_SEQ_RES_BITS) == 0) {
                  /* Simple case--no sequencing actions specified */
                  sequencer[index]=~((~sequencer[index])|data);
               }
               else if (data & (1<<SEQ_RESET_DUMMY_BIT)) {
                  /* Clear out default sequence level, and set to zero */
                  sequencer[index]=~((~sequencer[index] & SEQ_LEVEL_MASK));
                  /* Or in action bits */
                  sequencer[index]=~(~sequencer[index]|(data&SEQ_LEVEL_MASK));
               }
               else { /* SEQ_SEQ_DUMMY_BIT */
                  /* Clear out default sequence level, and set to zero */
                  sequencer[index]=~((~sequencer[index] & SEQ_LEVEL_MASK));
                  /* Change sequence level to level+1 */
                  sequencer[index]=~(~sequencer[index]
                                    | ((level+1)<<SEQ_LEVEL_BIT));
                  /* Or in action bits */
                  sequencer[index]=~(~sequencer[index]|(data&SEQ_LEVEL_MASK));
               }
            }
         }
      }
   }
   return(GOOD);
}

/****************************************************************************
**
**  EvtrecTask
**
**  Description:
**     Programs one event recognizer.  This routine takes the generic event
**     recognizer xilinx config and modifies the range and mask portions
**     according to input data.  The resulting bitstream is programmed into
**     one xilinx (determined by calling SdGetMemberIndex).
**
**     The QualMode parameter is read when the evtrec sd item is written.
**     The host must follow each write of QualMode by a reconfig of each
**     event recognizer.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor
**     output:
**        none
**
*****************************************************************************/
RETCODE PRIVATE EvtrecTask(DESCRIPTOR desc){
   RETCODE err;
   MEMBER_INDEX index;
   if ((err = SdGetMemberIndex(desc,&index)) != GOOD) return(err);
   return(TrigProgramEvtrec(index));
}

RETCODE EXPORT TrigProgramEvtrec(MEMBER_INDEX index) {
   XLX_CHIP chip;
   switch (index) {
      case 0: chip = XLX_TMODR; break;
      case 1: chip = XLX_TMODS; break;
      case 2: chip = XLX_TMODT; break;
      case 3: chip = XLX_TMODU; break;
      default: return(ER_UNKNOWN_XILINX);
   }
   if (evtrecConfig[index].type == EVTREC32)
      return(ProgramEvtrec32(index,chip));
   else
      return(ProgramEvtrec8(index,chip));
}

static const U8 DABC[] =   /* Swizzle DABC -> BCAD */
   { 0x0,0x4,0x8,0xC,0x2,0x6,0xA,0xE,0x1,0x5,0x9,0xD,0x3,0x7,0xB,0xF };
static const U8 ADBC[] =   /* Swizzle ADBC -> BCAD */
   { 0x0,0x4,0x8,0xC,0x1,0x5,0x9,0xD,0x2,0x6,0xA,0xE,0x3,0x7,0xB,0xF };

static const U8 *swizzleLowTable[8][8] = {
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, ADBC },
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, DABC },
   { ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC },
   { ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC, ADBC },
   { ADBC, ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC },
   { DABC, ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC },
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, ADBC },
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, DABC }};
static const U8 *swizzleHighTable[8][8] = {
   { DABC, ADBC, DABC, DABC, ADBC, DABC, DABC, ADBC },
   { DABC, ADBC, DABC, DABC, ADBC, DABC, DABC, ADBC },
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, ADBC },
   { DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC, ADBC },
   { ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC },
   { ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC },
   { DABC, ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC },
   { DABC, DABC, DABC, DABC, ADBC, DABC, DABC, ADBC }};
static const U8 *swizzle8Table[8][8] = {
   { ADBC, DABC, ADBC, ADBC, DABC, ADBC, ADBC, DABC },
   { DABC, DABC, DABC, DABC, ADBC, DABC, DABC, ADBC },
   { DABC, DABC, ADBC, DABC, DABC, ADBC, DABC, ADBC },
   { ADBC, DABC, ADBC, ADBC, DABC, DABC, ADBC, DABC },
   { DABC, DABC, DABC, DABC, ADBC, DABC, ADBC, ADBC },
   { DABC, DABC, DABC, ADBC, ADBC, DABC, DABC, ADBC },
   { DABC, DABC, DABC, DABC, ADBC, DABC, ADBC, ADBC },
   { DABC, DABC, DABC, ADBC, ADBC, DABC, DABC, ADBC }};

#define SwizzleLowBits(row,col,addr) \
   ((swizzleLowTable[row][col])[(addr)&0xF])
#define SwizzleHighBits(row,col,addr) \
   ((swizzleHighTable[row][col])[(addr)&0xF])
#define Swizzle8Bits(row,col,addr) \
   ((swizzle8Table[row][col])[(addr)&0xF])

RETCODE PRIVATE ProgramEvtrec8(MEMBER_INDEX index, XLX_CHIP chip) {
   RETCODE err;
   DESCRIPTOR xlxDesc;
   U8 row,col;
   if ((err = XlxOpen(XC3090, evtrec8RawBits, &xlxDesc))!=GOOD) return(err);
   if ((err = XlxCopyBitstream(xlxDesc)) != GOOD) {
      XlxClose(xlxDesc);
      return(err);
   }
/*
** Configure recognizers
*/
   for (row=0; row<8; row++) {
      /* Note: 1st subscript is the row, 2nd is column in following table */
      static const U16 evt8Clb[8][8] = {
         { 0x0200, 0x0202, 0x0204, 0x0206, 0x0208, 0x020A, 0x020C, 0x020E },
         { 0x0400, 0x0402, 0x0404, 0x0406, 0x0408, 0x040A, 0x040C, 0x040E },
         { 0x0600, 0x0602, 0x0604, 0x0606, 0x0608, 0x060A, 0x060C, 0x060E },
         { 0x0800, 0x0802, 0x0804, 0x0806, 0x0808, 0x080A, 0x080C, 0x070E },
         { 0x0A00, 0x0A02, 0x0A04, 0x0A06, 0x0A08, 0x0A0A, 0x0A0C, 0x0A0E },
         { 0x0D00, 0x0D02, 0x0D04, 0x0D06, 0x0D08, 0x0D0A, 0x0D0C, 0x0C0E },
         { 0x1000, 0x1002, 0x1004, 0x1006, 0x1008, 0x100A, 0x100C, 0x100E },
         { 0x1300, 0x1302, 0x1304, 0x1306, 0x1308, 0x130A, 0x130C, 0x130E }};
      static const U16 evt8NotClb[] = { 0x030f, 0x070f, 0x0b0f, 0x110f };
      EVTREC_RANGE range;
      
      if ((err = SdnReadPartialMember(SDN_EVTREC+index,row*sizeof(range),
         sizeof(range),&range)) != GOOD) return(err);
      for (col=0; col<8; col++) {
         /* First clear out existing programming */
         if ((err = XlxModifyClb(xlxDesc,evt8Clb[row][col],XLX_F,0,0,0))
            !=GOOD) return(err);
         /* Program output */
         if ((err = XlxModifyClb(xlxDesc,evt8Clb[row][col],XLX_F,
            Swizzle8Bits(row,col,(range.low>>(col*4))&0xF),
            Swizzle8Bits(row,col,(range.mask>>(col*4))&0xF),1)) != GOOD)
            return(err);
      }
      if (range.not) {
         if ((row%2)==0) {   /* evt0,2,4,6 */
            if ((err=XlxModifyClb(xlxDesc,evt8NotClb[row/2],XLX_F,0,0,0))
               !=GOOD) return(err);
            if ((err=XlxModifyClb(xlxDesc,evt8NotClb[row/2],XLX_F,0x2,0x6,1))
               !=GOOD) return (err);                                 /*~C*A*/
         }
         else {              /* evt1,3,5,7 */
            if ((err=XlxModifyClb(xlxDesc,evt8NotClb[row/2],XLX_G,0,0,0))
               !=GOOD) return(err);
            if ((err=XlxModifyClb(xlxDesc,evt8NotClb[row/2],XLX_G,0x2,0xA,1))
               != GOOD) return(err);                                 /*~B*A*/
         }
      }
   }
/*
** Configure qualifiers
*/
   {
      QUAL_MODE qualMode;
      if ((err = SdReadMember(descQualMode, &qualMode)) != GOOD) return(err);
      if (qualMode == QUAL_CLOCK) {
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_G,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL1,XLX_F,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL1,XLX_G,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_F,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_G,0,0,1))
            != GOOD) return(err);
      } else {
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,0,0,0))
            != GOOD) return(err);  /* clear existing programming */
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_G,0,0,0))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL1,XLX_F,0,0,0))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL1,XLX_G,0,0,0))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,
            evtrecConfig[index].func1&0x0f,
            evtrecConfig[index].func2&0x0f, 1)) != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,
            (evtrecConfig[index].func1>>4)&0x0f,
            (evtrecConfig[index].func2>>4)&0x0f, 1)) != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,
            (evtrecConfig[index].func1>>8)&0x0f,
            (evtrecConfig[index].func2>>8)&0x0f, 1)) != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_QUAL0,XLX_F,
            (evtrecConfig[index].func1>>0xC)&0x0f,
            (evtrecConfig[index].func2>>0xC)&0x0f,1)) != GOOD) return(err);
         /* Frame qual polarity select */
         if (evtrecConfig[index].frameQual == 0) {
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_F,0,0,0))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_G,0,0,0))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_F,0,2,1))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC8_FRAME,XLX_G,0,2,1))
               != GOOD) return(err);
         }
      }
   }
/*
** Program xilinx with new bitstream
*/
   if ((err = XlxProgram(xlxDesc, chip)) != GOOD) {
      XlxClose(xlxDesc);
      return(err);
   }
   if ((err = XlxClose(xlxDesc)) != GOOD) return(err);
   return(GOOD);
}

RETCODE PRIVATE ProgramEvtrec32(MEMBER_INDEX index, XLX_CHIP chip) {
   RETCODE err;
   DESCRIPTOR xlxDesc;
   U8 row,col;
   if ((err = XlxOpen(XC3090, evtrec32Bits, &xlxDesc))!=GOOD) return(err);
   if ((err = XlxCopyBitstream(xlxDesc)) != GOOD) {
      XlxClose(xlxDesc);
      return(err);
   }
/*
** Read each range from shared data and modify xilinx bitstream for each.
*/
   for (row=0; row<8; row++) {
      /* These tables show which function generator (XLX_F or XLX_G) is
         used for the equal half of the comparator.  The opposite
         value is the greater/less comparator */
      static const XLX_FUNC eqLowFuncTable[8][8] = {
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_G, XLX_F, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_G, XLX_F, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_F, XLX_F, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G }};
      static const XLX_FUNC eqHighFuncTable[8][8] = {
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_F, XLX_F, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_F, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_F, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_F, XLX_F, XLX_F, XLX_F, XLX_F, XLX_F, XLX_F },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G },
         { XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G, XLX_G }};
      static const U16 evt32NotClb[] = { 0x020f, 0x070f, 0x0c0f, 0x110f };

      EVTREC_RANGE range;
      U8 addr;
      
      if ((err = SdnReadPartialMember(SDN_EVTREC+index,row*sizeof(range),
         sizeof(range),&range)) != GOOD) return(err);
      /*
      ** First check if event is used.  If programmed always OFF, leave
      ** programming as default.  OFF is denoted by mask=0 && not=1.
      */
      if ((range.mask != 0) || (range.not != 1)) for (col=0; col<8; col++) {
         U8 mask = (range.mask>>(col*4))&0xF;
         U8 maskedRangeLow = (range.low>>(col*4)) & mask;
         U8 maskedRangeHigh = (range.high>>(col*4)) & mask;
         /*
         ** Program one output with equality and other with less/greater
         */
         for (addr=0; addr<0x10; addr++) {
            if ((addr & mask) == maskedRangeLow) {
               if ((err = XlxModifyClb(xlxDesc,evtLowClb[row][col],
                  eqLowFuncTable[row][col],
                  SwizzleLowBits(row,col,addr),0xF,1)) != GOOD) return(err);
            } else if ((addr & mask) > maskedRangeLow) {
               if ((err = XlxModifyClb(xlxDesc,evtLowClb[row][col],
                  (eqLowFuncTable[row][col]==XLX_F)?XLX_G:XLX_F,
                  SwizzleLowBits(row,col,addr),0xF,1)) != GOOD) return(err);
            }
            if ((addr & mask) == maskedRangeHigh) {
               if ((err = XlxModifyClb(xlxDesc,evtHighClb[row][col],
                  eqHighFuncTable[row][col],
                  SwizzleHighBits(row,col,addr),0xF,1)) != GOOD) return(err);
            } else if ((addr & mask) < maskedRangeHigh) {
               if ((err = XlxModifyClb(xlxDesc,evtHighClb[row][col],
                  (eqHighFuncTable[row][col]==XLX_F)?XLX_G:XLX_F,
                  SwizzleHighBits(row,col,addr),0xF,1)) != GOOD) return(err);
            }
         }
         /*
         ** ls bits: program lg with equality (to make greater/less or equal)
         */
         if (col==0) {
            if ((err = XlxModifyClb(xlxDesc,evtLowClb[row][col],
               (eqLowFuncTable[row][col]==XLX_F)?XLX_G:XLX_F,
               SwizzleLowBits(row,col,range.low>>(col*4)),
               SwizzleLowBits(row,col,mask),1)) != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,evtHighClb[row][col],
               (eqHighFuncTable[row][col]==XLX_F)?XLX_G:XLX_F,
               SwizzleHighBits(row,col,range.high>>(col*4)),
               SwizzleHighBits(row,col,mask),1)) != GOOD) return(err);
         }
      }
      if (range.not) {
         if ((row%2)==0) {   /* evt0,2,4,6 */
            if ((err=XlxModifyClb(xlxDesc,evt32NotClb[row/2],XLX_F,0,0,0))
               !=GOOD) return(err);
            if ((err=XlxModifyClb(xlxDesc,evt32NotClb[row/2],XLX_F,0x2,0x6,1))
               !=GOOD) return (err);                                  /*~C*A*/
         }
         else {              /* evt1,3,5,7 */
            if ((err=XlxModifyClb(xlxDesc,evt32NotClb[row/2],XLX_G,0,0,0))
               !=GOOD) return(err);
            if ((err=XlxModifyClb(xlxDesc,evt32NotClb[row/2],XLX_G,0x2,0xA,1))
               !=GOOD) return(err);                                   /*~B*A*/
         }
      }
   }
/*
** Use QualMode, func1, and func2 to configure qualifier select CLB
*/
   {
      QUAL_MODE qualMode;
      if ((err = SdReadMember(descQualMode, &qualMode)) != GOOD) return(err);
      if (qualMode == QUAL_CLOCK) {
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_QUAL,XLX_F,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_F,0,0,1))
            != GOOD) return(err);
         if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_G,0,0,1))
            != GOOD) return(err);
      } else {
         U16 bit;
         for (bit=0; bit<16; bit++) {
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_QUAL,XLX_F,
               bit, 0x0f, (evtrecConfig[index].func1>>bit)&1)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_QUAL,XLX_G,
               bit, 0x0f, (evtrecConfig[index].func2>>bit)&1)) != GOOD)
               return(err);
         }
         if (!evtrecConfig[index].useDelayed) {
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_DELAY0,XLX_F,
               0, 0, 0)) != GOOD) return(err);    /* clear existing */
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_DELAY1,XLX_F,
               0, 0, 0)) != GOOD) return(err);    /* clear existing */
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_DELAY0,XLX_F,
               8, 8, 1)) != GOOD) return(err);    /* F=B */
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_DELAY1,XLX_F,
               8, 8, 1)) != GOOD) return(err);    /* F=B */
         }
         /* Frame qual polarity select */
         if (evtrecConfig[index].frameQual == 0) {
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_F,0,0,0))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_G,0,0,0))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_F,0,2,1))
               != GOOD) return(err);
            if ((err = XlxModifyClb(xlxDesc,CLB_EVTREC32_FRAME,XLX_G,0,2,1))
               != GOOD) return(err);
         }
      }
   }
/*
** Program xilinx with new bitstream
*/
   if ((err = XlxProgram(xlxDesc, chip)) != GOOD) {
      XlxClose(xlxDesc);
      return(err);
   }
   if ((err = XlxClose(xlxDesc)) != GOOD) return(err);
   return(GOOD);
}

VOID EXPORT ClearTriggerPositions(VOID) {
   LOOP_VAR i;
   U8 *ptr = (U8 *)triggerPos;
   for(i=0;i<sizeof(triggerPos);i++)
      *ptr++ = 0;
}

/****************************************************************************
**
**  ResetDMAPointer
**
**  Description:
**    set DMA chan 5 count to match number of trace buffers 
**    and reset DMA chan 5 address to point to trigpos for trcbfr 0 
**
**  Parameters:
**     none
**
*****************************************************************************/
VOID EXPORT ResetDMAPointer(VOID) {
   U16 numTrcbufs;
   DESCRIPTOR descNumTrcbufs;
   if(SdnRegister(SDN_NUM_TRCBUFS,NULLPTR,&descNumTrcbufs)) return;
   if(SdReadMember(descNumTrcbufs,&numTrcbufs)) return;
   SdUnRegister(descNumTrcbufs);
   _outb(IO_BIU_DMA2_CSR, DMA_DISABLE);
   _outb(IO_BIU_DMA2_ADDR1,(U8)((OffsetOf(triggerPos)>>1) &0xff));
   _outb(IO_BIU_DMA2_ADDR1,(U8)((OffsetOf(triggerPos)>>9) &0xff));
   _outb(IO_BIU_DMA2_COUNT1,lobyte(2*numTrcbufs-1));
   _outb(IO_BIU_DMA2_COUNT1,hibyte(2*numTrcbufs-1));
   _outb(IO_BIU_DMA2_CSR, DMA_ENABLE);
}

U16 GetTriggerSubBuffer(U16 buffer) {
   if( triggerPos[buffer].subBuffer & 0x1 )  /* QSF == 1 */
      return(IncSubBuffer(triggerPos[buffer].subBuffer,-1));
   return(IncSubBuffer(triggerPos[buffer].subBuffer,1));
}

U32 GetTriggerFrame(U16 buffer) {
    return(triggerPos[buffer].frame);
}

BOOLEAN EXPORT TraceStoreFull(U16 buffer) {
   if (buffer==0) return(triggerPos[(NumTraceBuffers()-1)].storeFull);
   return( triggerPos[buffer-1].storeFull );
}

BOOLEAN EXPORT TraceIsEmpty(U16 buffer) {
   BOOLEAN traceIsEmpty;
   U32 *trigInfo = (U32 *)&triggerPos[buffer];
   if( *trigInfo == 0 ) return(TRUE);
   if( SdnReadMember(SDN_TRACE_EMPTY,&traceIsEmpty) ) return(TRUE);
   return(traceIsEmpty);
}

BOOLEAN EXPORT TraceBufferFull(U16 buffer) {
   U32 numFrames;
   if( TraceIsEmpty(buffer) ) return(FALSE);
   if( triggerPos[buffer].bufferFull ) return(TRUE);
   /* since trigger might of happened before post fill was done which
      might of caused the trace buffer to wrap.  SO... we have to check
      the number of frames collected up to the trigger then add the
      post fill count and if the results is bigger than the number
      of frames for the trace buffer then we wrapped.
    */
      numFrames = ( ((U32)GetTriggerSubBuffer(buffer) 
                  - ((U32)buffer*(U32)NumSubBufsPerTraceBuffer())
                  + GetPostFillCount()) * (U32)subBufferSize)
                  + GetTriggerFrame(buffer);
      if( numFrames > ((U32)NumSubBufsPerTraceBuffer()*(U32)subBufferSize) )
         return(TRUE);
   return(FALSE);
}

RETCODE EXPORT GetPhysicalTriggerFrame(U16 buffer, U32 *trigger) {
   *trigger = (GetTriggerSubBuffer(buffer)*(U32)subBufferSize) 
               + GetTriggerFrame(buffer) - 8;
   return(GOOD);
}

U16 DecrementTraceBuffer(U16 buffer) {
   U16 numTraceBuffers=NumTraceBuffers();
   return( ((buffer - 1) & (numTraceBuffers-1))
         | (buffer & ~(numTraceBuffers-1)) );
}

U32 EXPORT MaxNumFramesPerTraceBuffer(VOID) {
   return((U32)NumSubBufsPerTraceBuffer() * (U32)subBufferSize);
}

/*
** SubBufferStartOffset
**    Determine if the given trace buffer starts in the first sub buffer
**    (returns offset 0).
*/
U32 EXPORT SubBufferStartOffset(U16 buffer) {
   if( (NumTraceBuffers() > 1) 
      && ((buffer!=0)||((buffer==0) && TraceStoreFull(buffer)))) {
      /* need to read the trigger information of previous trace buffer
         in order to determine which sub buffer we started filling
         this trace buffer with.  hw alternates between filling even
         and odd sub buffers.
       */
      if((GetTriggerSubBuffer(DecrementTraceBuffer(buffer)) & 0x1) != 0)
         /* previous buffer triggered in an odd sub buffer, therefore
            ended filling on an even sub buffer thus this trace buffer
            started filling in sub buffer 1.
          */     
         return((U32)subBufferSize);
   }
   return(0);
}

RETCODE EXPORT GetOldestFrame(U16 buffer, U32 *oldest) {
   U32 oldSubBufNum;
   if( TraceBufferFull(buffer) ) {  /* trace buffer has wrapped */
      /* if trace buffer has wrapped then the oldest frame would be
         the first frame in the sub buffer following the trigger and
         post fill sub buffers.
      */           
      oldSubBufNum = IncSubBuffer(GetTriggerSubBuffer(buffer),
                                       GetPostFillCount());
      oldSubBufNum = IncSubBuffer(oldSubBufNum,1);
      *oldest = (U32)(oldSubBufNum * (U32)subBufferSize);
   }
   else {
      /* trace hasn't wrapped so oldest frame is at the start of the trace
         buffer */
      *oldest = buffer * MaxNumFramesPerTraceBuffer()
                     + SubBufferStartOffset(buffer);
   }
   return(GOOD);
}

RETCODE EXPORT GetYoungestFrame(U16 buffer, U32 *youngest) {
   RETCODE err;
   U32 youngestSubBuf;
   youngestSubBuf = IncSubBuffer(GetTriggerSubBuffer(buffer),
                                 GetPostFillCount());
   *youngest = (youngestSubBuf*(U32)subBufferSize) + ((U32)subBufferSize - 1);
   return(GOOD);
}

U32 EXPORT GetNumFramesBeforeTrigger(U16 buffer) {
   U32 numFrames;
   if( TraceIsEmpty(buffer) ) return(0);
   if( TraceBufferFull(buffer) ) {
      numFrames = MaxNumFramesPerTraceBuffer()
               -  ((U32)subBufferSize - GetTriggerFrame(buffer))
               -  (GetPostFillCount() * (U32)subBufferSize)
               -  8;
   } else {
      numFrames = ( ((U32)GetTriggerSubBuffer(buffer) 
                   - ((U32)buffer*(U32)NumSubBufsPerTraceBuffer()))
                  * (U32)subBufferSize)
                  + GetTriggerFrame(buffer)
                  - 8                  
                  - SubBufferStartOffset(buffer);
   }
   return(numFrames);
}

U32 EXPORT GetNumFramesAfterTrigger(U16 buffer) {
   if( TraceIsEmpty(buffer) ) return(0);
   return( ((U32)subBufferSize - GetTriggerFrame(buffer))
          + (GetPostFillCount()*(U32)subBufferSize)
          + 8 - 1);
}

U32 EXPORT GetNumFramesInTraceBuffer(U16 buffer) {
   if( TraceIsEmpty(buffer) ) return(0);
   return( GetNumFramesBeforeTrigger(buffer)
         + GetNumFramesAfterTrigger(buffer)
         + 1 /* trigger position itself */);
}

U32 EXPORT GetPhysicalFrame(U16 buffer, S32 frame) {
   U32 trigger;
   if( GetPhysicalTriggerFrame(buffer, &trigger) ) return(0);
   return(IncPhysicalFrame(trigger,frame));
}
   

/******************************** E O F ***********************************/
