/***************************************************************************
**  Name:  emumem.c
**
**  Description:
**     Michelle memory-group routines.
**     include:
**        "Read N bytes memory from Michelle -- emuReadMemN()"
**        "Write N bytes memory to Michelle  -- emuSetMemN()"
**        "Memory data comparing             -- emuCompare()"
**        "Memory data coping                -- emuCopy()"
**        "Memory data filling               -- emuFill()"
**        "Memory data testing               -- emuTest()"
**        "Memory data searching             -- emuSearch()"
**        "Memory data checksum             -- emuChecksum()"
**
**  Status:  preliminary
**
**  $Log:   S:/tbird/arcmtat2/mp186/emumem.c_v  $
** 
**    Rev 1.0   03 Dec 1996 09:27:36   gene
** Initial revision.
** 
**    Rev 1.0   10 May 1996 10:06:24   jacky
** Get file from ATL. V1.3
** 
** 
**  $Header:   S:/tbird/arcmtat2/mp186/emumem.c_v   1.0   03 Dec 1996 09:27:36   gene  $
**
** Copyright (C) 1992 Microtek International, Inc.
**
****************************************************************************/

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _EMU_MEM_
#include "emumem.h"
#endif

#ifndef _EMU_BASE_TYPES_
#include "emutypes.h"
#endif

#ifndef _EMU_LIB_
#include "emulib.h"
#endif

#ifndef _LLFW_
#include "emullfw.h"
#endif

#ifndef _EMU_EXTERNAL_
#include "emuext.h"
#endif

#include <string.h>


// diff = (U8 *)&canTemp;
// pos = outputStreamLen;
// outputStreamLen = pos + 1;/* Output stream length */

//  #define OutStream(diff,pos,streamLen) \
//           outputStream[pos++] = streamLen;\
//           memcpy(&outputStream[pos],diff,streamLen);/* error address */\
//           pos += streamLen;\
//           outputStream[pos++] = 0x00;

                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/

                        /****************************
                         *                          *
                         *    EXTERNAL VARIABLES    *
                         *                          *
                         ****************************/

                        /****************************
                         *                          *
                         *     LOCAL PROTOTYPES     *
                         *                          *
                         ****************************/

STATUS ReadMem(U32 addr, U8 *buff, U16 len);
STATUS WriteMem(U32 addr, U16 dataLen, U8 *buff);

                        /****************************
			 *                          *
			 *      EXECUTABLE CODE     *
			 *                          *
			 ****************************/
/***************************************************************************
**
**  CopyTarget
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
STATUS CopyTargetMem(U32 addr1, U32 addrLen) {
STATUS status;
U32 addr,count,lp;
U8 memBuff[BUFF_LEN];

   status = GOOD;
   ClrViolFlg();
   count = addrLen/BUFF_LEN;
   addr = addr1;
   for (lp = 0; lp < count; lp++) {
      DisEmStop();
      status = ReadMem(addr,memBuff,(U16)BUFF_LEN);
      if (status) return(status);
      EnEmDisViol();
      status = WriteMem(addr,(U16)BUFF_LEN,memBuff);
      if (status) return(status);
      addr = AddSegAddr(addr,BUFF_LEN);
   }
   count = addrLen % (U32)BUFF_LEN;
   if ((count != 0) && (status == GOOD)) {
      DisEmStop();
      status = ReadMem(addr,memBuff,(U16)count);
      if (status) return(status);
      EnEmDisViol();
      status = WriteMem(addr,(U16)count,memBuff);
      if (status) return(status);
   }
   return(status);
}

VOID HiDumpByteMem(U32 addr, U32 len, U8 *buff,U16 loadFlag) {
U32 offset,posLen,preLen;

   offset = LowWord(addr);
   posLen = offset+ (U16)len;
   if (offset > posLen) {
      preLen = len - posLen;
      DumpByteMem(addr,preLen,buff,loadFlag);
      buff+= preLen;
      addr = AddSegAddr(addr,preLen);
      if (posLen) DumpByteMem(addr,posLen,buff,loadFlag);
   }
   else DumpByteMem(addr,len,buff,loadFlag);

};
/***************************************************************************
**
**  ReadMem
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
STATUS ReadMem(U32 addr, U8 *buff, U16 dataLen) {
U16 lp,count,modLen;
U16 size,loadCan;
STATUS status;

   if (modLen = (U16)CAN_BUFFER_NO - (U16)(addr % (U16)CAN_BUFFER_NO)) {
      if (dataLen < modLen) modLen = dataLen;
      //HiDumpByteMem(addr,modLen,buff,ON);
      DumpByteMem(addr,modLen,buff,ON);
      addr = AddSegAddr(addr,modLen);
   }
   dataLen -= modLen;
   status = GOOD;
   status = AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   count = dataLen / (U16)CAN_BUFFER_NO;
   loadCan = OFF;
   for (lp = 0; lp < count; lp++) {
      if (status != GOOD) return(status);
      if (lp == 0) loadCan = ON;
      DumpByteMem(addr,CAN_BUFFER_NO,&buff[lp*CAN_BUFFER_NO+modLen],loadCan);
      //loadCan = OFF;
      loadCan = ON;
      if (brOver == 0) {
         addr = AddSegAddr(addr,(U16)CAN_BUFFER_NO);
         loadCan = ON;
      }
      else return(brOver);
   }
   count = dataLen % (U16)CAN_BUFFER_NO;
   if ((count != 0) && (status == GOOD)) {
      DumpByteMem(addr,count,&buff[lp*CAN_BUFFER_NO+modLen],ON);
   }
   return(brOver);
}

/***************************************************************************
**
**  WriteMem
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
STATUS WriteMem(U32 addr, U16 dataLen, U8 *buff) {
U16 verifyFlag;
U16 lp;
STATUS status;
U16 count,loadCan,modLen;

   status = GOOD;
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   if (modLen = (U16)CAN_BUFFER_NO - (U16)(addr % (U16)CAN_BUFFER_NO)) {
      if (dataLen < modLen) modLen = dataLen;
      status = FillByteMem(addr,modLen,modLen,buff,verifyFlag,ON) & 0xff;
      if (status) status = ERROR_WRITE;
      else status = CheckViolation();
      addr = AddSegAddr(addr,modLen);
   }
   dataLen -= modLen;
   if (status) return(status);
   count = dataLen / (U16)CAN_BUFFER_NO;
   loadCan = OFF;
   for (lp = 0; lp < count; lp++) {
      if (status != GOOD) return(status);
      if(lp == 0) loadCan = ON;
      status = FillByteMem(addr,CAN_BUFFER_NO,(U16)CAN_BUFFER_NO,
                           &buff[lp*CAN_BUFFER_NO+modLen],verifyFlag,loadCan) & 0xff;
      if (status) status = ERROR_WRITE;
      else status = CheckViolation();
      if (status) return(status);
      //loadCan = OFF;
      loadCan = ON;
      if (brOver == 0)
         addr = AddSegAddr(addr,(U16)CAN_BUFFER_NO);
      else return(brOver);
   }
   count = dataLen % (U16)CAN_BUFFER_NO;
   if ((count != 0) && (status == GOOD)) {
      status = FillByteMem(addr,count,count,&buff[lp*CAN_BUFFER_NO+modLen],verifyFlag,ON) & 0xff;
      if (status) status = ERROR_WRITE;
      else status = CheckViolation();
   }
   if (status) return(status);
   return(brOver);

}

/***************************************************************************
**
**  EmuReadMemN
**
**  Description: Michelle memory-group routine, to read a chunk memory
**               from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
VOID EmuReadMemN(U32 addr, U32 addrLen) {
STATUS status;
U8 dataLen,memBuff[BUFF_LEN];
U16 i86Reg[MAX_CPU_REG_NO];

   status = GOOD;
   if (addrLen > BUFF_LEN) {
      OutputStatus(LEN_TOO_LONG,ON);
      return;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   status = ReadMem(addr,memBuff,(U16)addrLen);
   DisEmStop();

   if (status == GOOD) {
      OutputStatus(status,OFF);
      dataLen = (U8)addrLen;
      OutData(dataLen,memBuff,ON);
      OutEnd();
   }
   else {
      OutputBrOverError(OFF);
      if (status == 2 ) { //NOT_READY
         OutData(4,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}


/***************************************************************************
**
**  EmuSetMemN
**
**  Description: Michelle memory-group routine, to write a chunk data
**               from "buff" to "addr" location. The driver program can
**               use this function to implement to download code from a
**               file.
**               The length should be limitted up to 1024 bytes(1KB).
**
**  Parameters:
**     input:
**        addr  -- unsigned long,
**        *buff -- unsigned char,
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuSetMemN(U32 addr, U8 *buff, U16 len) {
STATUS status;
U16 i86Reg[MAX_CPU_REG_NO];

   status = GOOD;
   if (len > BUFF_LEN) {
      OutputStatus(LEN_TOO_LONG,ON);
      return;
   }
   else if (len == 0) {
      OutputStatus(LEN_EQUAL_ZERO,ON);
      return;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   status = WriteMem(addr,len,buff);
   DisEmStop();

   switch (status) {
      case GOOD :
      case MEM_PROTECT :
         OutputStatus(GOOD,ON);
         break;
      case MEM_GUARD :
         OutputStatus(status,ON);
         break;
      case ERROR_WRITE :
         OutputStatus(status,OFF);
         OutData(4,&canTemp,ON);
         OutEnd();
         break;
      default :
         OutputBrOverError(OFF);
         if (status == 2 ) OutData(4,&canTemp,ON);  //NOT_READY
         OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}


/***************************************************************************
**
** EmuCompare
**
**  Description: Michelle memory-group routine, to compare two memory
**               blocks.
**               Return the locations and its contents, if there has
**               difference.
**               The "addr1" and "addr2" is the addres range of the
**               first block.
**               The "addr3" is the start address of the second block.
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        addr3 -- unsigned long
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "location" + "contents"
**
****************************************************************************/
VOID EmuCompare(U32 addr1, U32 addrLen, U32 addr2) {
STATUS status;
U16 i86Reg[MAX_CPU_REG_NO],size;

   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   switch ((SIZE)size) {
      case S_WORD :
         ReduceEven(addr1);
         IncreaseEven(addrLen);
         ReduceEven(addr2);
         break;
      case S_LONG :
         ReduceQuad(addr1);
         IncreaseQuad(addrLen);
         ReduceQuad(addr2);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   status = CompareMem(addr1,addrLen,addr2,size) & 0xff;
   DisEmStop();
   //status = CheckViolation();
   if (brOver == GOOD) {
      if (status == GOOD) OutputStatus(NO_DIFF,ON);
      else {
         OutputStatus(GOOD,OFF);
         OutData(12,&canTemp,ON);
         OutEnd();
      }
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) {
         OutData(12,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}


/***************************************************************************
**
**  EmuCopy
**
**  Description: Michelle memory-group routine, to copy a memory block
**               data to another block's.
**               The "addr1" and "addr2" is the addres range of the
**               first block.
**               The "addr3" is the start address of the destination
**               block.
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        addr3 -- unsigned long
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuCopy(U32 addr1, U32 addrLen, U32 addr2) {
STATUS status;
U16 i86Reg[MAX_CPU_REG_NO], size,verifyFlag;
U32 phyAddr1,phyAddr2;

   status = GOOD;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   switch ((SIZE)size) {
      case S_WORD :
         //ReduceEven(addr1);      // del by Joan Tai. May 18,1994
         //IncreaseEven(addrLen);  // del by Joan Tai  May 18,1994
         //ReduceEven(addr2);      // del by Joan Tai  May 18,1994
         break;
      case S_LONG :
         ReduceQuad(addr1);
         IncreaseQuad(addrLen);
         ReduceEven(addr2);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   phyAddr1 = LinearToPhysical(addr1);
   phyAddr2 = LinearToPhysical(addr2);
   if (phyAddr1 == phyAddr2)
      status = CopyTargetMem(addr1,addrLen);
   else if (phyAddr1 > phyAddr2)
      status = PreCopyMem(addr1,addrLen,addr2,verifyFlag) & 0xff;
   else {
      addr1 = AddSegAddr(addr1,addrLen-1);
      addr2 = AddSegAddr(addr2,addrLen-1);
      status = PostCopyMem(addr1,addrLen,addr2,verifyFlag) & 0xff;
   }
   DisEmStop();

   //status = CheckViolation();
   if (brOver == GOOD) {
      if (status == GOOD) OutputStatus(GOOD,ON);
      else {
         OutputStatus(ERROR_WRITE,OFF);
         OutData(8,(U8 *)&canTemp+4,ON);
         OutEnd();
      }
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) {
         OutData(8,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}

/***************************************************************************
**
**  EmuFill
**
**  Description: Michelle memory-group routine, to fill a memory range
**               with a specified pattern.
**               Fill a memory range with a specified pattern.
**               Data pattern is specified in "buff" array.
**               The lengh of the data pattern is "len".
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        *buff -- unsigned char, Max 16 bytes;
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuFill(U32 addr, U32 addrLen, U8 *buff, U16 len) {
STATUS status;
U16 size, n;
U16 i86Reg[MAX_CPU_REG_NO],verifyFlag;

   status = GOOD;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   switch ((SIZE)size) {
      case S_BYTE :
         status = FillByteMem(addr,addrLen,len,buff,verifyFlag,ON) & 0xff;
         status |= CheckViolation();
         break;
      case S_WORD :
         //ReduceEven(addr);
         //IncreaseEven(addrLen); // del by Joan Tai.  May 18,1994
         //if (len%2) buff[len] = buff[0];

         if (addrLen%2) {                                   // add by Joan Tai.  May 18,1994
            n = (len == 1 ? 0 : addrLen%len ? (U16)(addrLen%len)-1 : len-1);// add by Joan Tai.  May 18,1994
            addrLen--;                                      // add by Joan Tai.  May 18,1994
            status = FillByteMem(addr+addrLen,1,1,&buff[n],verifyFlag,ON) & 0xff;// add by Joan Tai.  May 18,1994
         }                                                  // add by Joan Tai.  May 18,1994

         status = FillWordMem(addr,addrLen,len,buff,verifyFlag,ON) & 0xff;
         status |= CheckViolation();
         break;
      case S_LONG :
         //ReduceQuad(addr);
         //IncreaseQuad(addrLen);
         status = FillByteMem(addr,addrLen,len,buff,verifyFlag,ON) & 0xff;
         status |= CheckViolation();
         break;
      default :
         status = FillByteMem(addr,addrLen,len,buff,verifyFlag,ON) & 0xff;
         status |= CheckViolation();
         break;
   }
   DisEmStop();
//   status = CheckViolation();
   if (brOver == GOOD) {
      switch (status) {
         case GOOD :
         case MEM_PROTECT :
            OutputStatus(GOOD,ON);
            break;
         case MEM_GUARD :
            OutputStatus(status,ON);
            break;
         default :
            OutputStatus(ERROR_WRITE,OFF);
            OutData(4,&canTemp,ON);
            OutEnd();
      }
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) OutData(4,&canTemp,ON); // NOT READY
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}

/***************************************************************************
**
**  EmuTest
**
**  Description: Michelle memory-group routine, to test a spcified
**               address range working O.K. or not? Is RAM or not?
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**
**    *** *pattern[n] -- char ===> optional
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**  NOTE: I have three ways to do this testing, you can decide how
**        to implement it by one of these.
**        1. (Read data from specified address location) -->
**           (Write pre-read data back to the same memory location) -->
**           (Compare these result then check it to report the status)
**        2. Write the defined special data pattern to the specified
**           memory loaction, then check the virolation.
**        3. Like method 2, but driver can select the "data pattern"
**           as she like. In this case, to add a parameter to indicate
**           the data pattern is necessary.
**
****************************************************************************/
VOID EmuTest(U32 addr, U32 addrLen) {
STATUS status;
U16 i86Reg[MAX_CPU_REG_NO];
U16 size;

   status = GOOD;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   switch ((SIZE)size) {
      case S_WORD :
         ReduceEven(addr);
         //IncreaseQuad(addrLen);  // del by Joan Tai.   May 18,1994
         IncreaseEven(addrLen);    // add by Joan Tai.   May 18,1994
         break;
      case S_LONG :
         ReduceQuad(addr);
         IncreaseQuad(addrLen);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   status = TestMem(addr,addrLen,size) & 0xff;
   DisEmStop();
//   status = CheckViolation();
   if (brOver == GOOD) {
      if (status == GOOD) OutputStatus(GOOD,ON);
      else {
         OutputStatus(ERROR_WRITE,OFF);
         OutData(4,&canTemp,ON);
         OutEnd();
      }
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) {
         OutData(4,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}

/***************************************************************************
**
**  EmuSearch
**
**  Description: Michelle memory-group routine, to search data pattern
**               in the specified address range.
**               Data pattern is specified in "buff" array.
**               The lengh of the data pattern is "len".
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        *buff -- unsigned char,  Max 32 Bytes;
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuSearch(U32 addr, U32 addrLen, U8 *buff, U16 len) {
#define MATCH  0xff /* template data, it will redefine. */
STATUS status;
U16 i86Reg[MAX_CPU_REG_NO];

   status = GOOD;
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   status = SearchMem(addr,addrLen,len,buff) & 0xff;
   DisEmStop();

//   status = CheckViolation();
   if (brOver == GOOD) {
      if (status == GOOD) {
         OutputStatus(GOOD,OFF);
         OutData(6,&canTemp,ON);
         OutEnd();
      }
      else OutputStatus(NO_FOUND,ON);
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) {
         OutData(6,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);

}

/***************************************************************************
**
**  EmuChecksum
**
**  Description: Michelle memory-group routine, to caculate the data
**               sum value of the specified address range.
**               The "addr1" and "addr2" is the address range.
**               The "size" is the sum-width = 0 --> caculate in byte
**                                           = 1 --> caculate in word
**                                           = 2 --> caculate in long word
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        size  -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuChecksum(U32 addr, U32 addrLen) {
U16 i86Reg[MAX_CPU_REG_NO];
U16 size,checksumData;

   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,i86Reg);
   if (size == S_WORD) IncreaseEven(addrLen);
   Checksum(addr,addrLen,size);
   DisEmStop();
//   status = CheckViolation();
   if (brOver == GOOD) {
      OutputStatus(brOver,OFF);
      diff = (U8 *)&canTemp;
      switch (size) {
        case S_BYTE :
           checksumData = *(U8 *)diff;/* byte checksum */
           break;
        case S_WORD :
           checksumData = *(U16 *)diff;/* word checksum */
           break;
      }
      OutData(2,&checksumData,ON);
      OutEnd();
   }
   else {
      OutputBrOverError(OFF);
      if (brOver == 2) {
         OutData(4,&canTemp,ON);
      }
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,i86Reg);
}

/******************************** E O F ***********************************/

