/***************************************************************************
**
**  Name:  emusetup.c
**
**  Description:
**     Michelle setup-group routines.
**     include:
**     "Get the identity of the Michelle FW version, HW configuration
**      and HW model type",
**     "Set access size to be byte/word/dword-accessing",
**     "Set or read the memory-write verify-flag",
**     "Control ready using external/internal",
**     "Set or read the memory mapping",
**     "Read all setup information",
**     "Set control signals",
**    * "set or read the wait-state" --- This is a issue to be discussed.
**      We will decide this latter.
**
**  Status:  preliminary
**
**  $Log:   S:/tbird/arcmtat2/mp186/emusetup.c_v  $
** 
**    Rev 1.0   03 Dec 1996 09:27:36   gene
** Initial revision.
** 
**    Rev 1.0   10 May 1996 10:05:14   jacky
** Get file from ATL. V1.3
** 
** 
**  $Header:   S:/tbird/arcmtat2/mp186/emusetup.c_v   1.0   03 Dec 1996 09:27:36   gene  $
**
** Copyright (C) 1992 Microtek International, Inc.
**
****************************************************************************/

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _EMU_SETUP_
#include "emusetup.h"
#endif

#ifndef _EMU_LIB_
#include "emulib.h"
#endif

#ifndef _EMU_TRACE_
#include "emutrace.h"
#endif

#ifndef _EMU_LLFW_
#include "emullfw.h"
#endif

#ifndef _EMU_EXTERNAL_
#include "emuext.h"
#endif

#include <string.h>
                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/
struct {
   U32 addr1,addr2;
   U16 attr;
}mapData;

#define MAX_BANK          4
#define TOP_NULL          0xfffffffeul
#define BUTTOM_NULL       0xfffffffful
#define MAX_ADDR          0xffffful
#define BLOCK_SIZE        2048    // 2K

typedef struct {
U32  top,
     bottom;
U16  status,
     attr;
}BANK_INFO;

typedef struct {
U32  addr1,addr2;
U8   attr;
} MAP_INFO;

typedef struct {
U32  addr1,
     addr2;
} SLICE;


#define BLOCK_NO     512 // depend on emmSize

BANK_INFO bankInfo[MAX_BANK];
U16 bankTable[MAX_BANK][BLOCK_NO];
SLICE slice[MAX_BANK];

U8 bankInfoCount;
enum BANK_NO {BANK1, BANK2, BANK3, BANK4, NO_BANK = 0xff};

U32 bankSize;

                        /****************************
                         *                          *
                         *    EXTERNAL VARIABLES    *
                         *                          *
                         ****************************/

                        /****************************
                         *                          *
                         *     LOCAL PROTOTYPES     *
                         *                          *
                         ****************************/

                        /****************************
                         *                          *
                         *      EXECUTABLE CODE     *
                         *                          *
                         ****************************/


/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS InitMapTable(VOID) {
STATUS status;
U16 lp,lp2;

   status = GOOD;
   mapBuf[0] = MAP_END_RECORD; /* set map data to map buffer */
   if (InitMapCtr()) status = EMM_SET_ERROR;
   bankInfoCount = 0;
   for (lp = 0; lp < MAX_BANK; lp++) {
      bankInfo[lp].status = NO_BANK;   // bank no use
      bankInfo[lp].attr   = 0x0c00; // map all external
      bankInfo[lp].bottom = BUTTOM_NULL;
      bankInfo[lp].top    = TOP_NULL;
      slice[lp].addr1 = BUTTOM_NULL;
      slice[lp].addr2 = TOP_NULL;
      for (lp2 = 0; lp2 < BLOCK_NO; lp2++)
         bankTable[lp][lp2] = 0x0c00;  // external
   }
   return(status);
}
/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS SliceNewAddr(U32 startAddr, U32 endAddr) {
U8 lp,lp2;

   if ((endAddr - startAddr) > MAX_ADDR) return(ADDR_RANGE_OVERFLOW);
   else {
      lp = 0;
      slice[lp].addr1 = startAddr;
      slice[lp].addr2 = ((startAddr/bankSize)+1)*bankSize - 1;
      while (slice[lp].addr2 < endAddr) {
         if (lp >= 4) return(ADDR_RANGE_OVERFLOW);
         slice[lp+1].addr1 = slice[lp].addr2 + 1;
         slice[lp+1].addr2 =
            ((slice[lp+1].addr1/bankSize)+1)*bankSize - 1;
         lp++;
      }
      slice[lp].addr2 = endAddr;
      for (lp2 = (U8)(lp+1); lp2 < MAX_BANK; lp2++) {   /* CLEAR ANOTHER addr */
         slice[lp2].addr1 = BUTTOM_NULL;
         slice[lp2].addr2 = TOP_NULL;
      }
   }
   return(GOOD);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U8 SearchBank(U32 newAddr) {
U8  lp,bankNo;

   bankNo = 0xFF;
   if (bankInfoCount > 0) {
      for (lp = 0; lp <= (U8)(bankInfoCount - 1); lp++)
         if ((newAddr >= bankInfo[lp].bottom) && (newAddr <= bankInfo[lp].top))
            bankNo = lp;
   }
   return(bankNo);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
VOID SetMapTable(U32 bankAddr1, U32 bankAddr2, U16 attr,U8 bankNo) {
U16 startBlock,endBlock,lp;

   startBlock = (U16)((bankAddr1 - bankInfo[bankNo].bottom) / BLOCK_SIZE);
   endBlock = (U16)( ((bankAddr2 - bankInfo[bankNo].bottom) / BLOCK_SIZE) + 1);
   for (lp = startBlock; lp < endBlock ; lp++) {
      bankTable[bankNo][lp] = attr & 0xff;
   }
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U8 CreateBank(U32 bankAddr1, U32 bankAddr2, U16 attr) {
U8 lp,bank ;

   bank = 0xFF;
   lp = 0 ;
   for (lp = 0; lp < MAX_BANK; lp++) {
      if (bankInfo[lp].status == NO_BANK) {
         bank = lp;
         bankInfo[lp].status = lp;
         bankInfo[lp].attr   = attr;
         bankInfo[lp].bottom = (bankAddr1 / bankSize) * bankSize;
         bankInfo[lp].top = bankInfo[lp].bottom + bankSize - 1;
         bankInfoCount++ ;
         break;
      }
   }
   return (bank);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrSw2Fw(U16 attr) {
U16 tempAttr = 0;

   //Bank No.
   // tempAttr = (attr & 0x18) << 5; // copy bit 3,4 to bit 8,9
   // Allocation Type.
   if ((attr & 0x07) < 4) tempAttr += ((attr & 0x03) << 10);// copy bit 0,1 to 10,11
   else tempAttr += 0xf00;
   // Space.
   if (attr & 0x20) tempAttr += 0x10; // copy bit 5 to bit 4
   if (attr & 0x40) tempAttr += 0x60; // copy bit 6 to bit 5,6
   return(tempAttr);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrFw2Sw(U16 attr) {
U16 tempAttr = 0;

   // Bank No.
   // tempAttr = (attr & 0x300) >> 5; // copy bit8,9 to bit 3,4
   // Allocation Type.
   if ((attr & 0xf00) == 0x0f00) tempAttr += 0x04;// copy bit 10,11 to 0,1
   else tempAttr += (attr & 0x0c00) >> 10;
   // Space.
   if ((attr & 0x10) > 0) tempAttr += 0x20;
   if ((attr & 0x60) > 0) tempAttr += 0x40;
   return(tempAttr);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS SetMap(U32 startAddr, U32 endAddr, U16 attr) {
STATUS status;
int tempAttr;
U8 bankNo,lp;
U16 addrStart,addrEnd;

   if (bankSize == 0) return(NO_EMM);
   attr = AttrSw2Fw(attr);
   if ((startAddr == 0) && (endAddr == 0)) return(InitMapTable());
   else {
      if (status = SliceNewAddr(startAddr,endAddr))
         return(ADDR_RANGE_OVERFLOW);
      lp = 0;
      while ((slice[lp].addr1 != BUTTOM_NULL) &&
             (slice[lp].addr2 != TOP_NULL) && (lp < 4)) {
         bankNo = SearchBank(slice[lp].addr1);
         if (bankNo == 0xFF) {
            bankNo = CreateBank(slice[lp].addr1,slice[lp].addr2,attr);
            if (bankNo == 0xFF)
               return(BANK_FULL);
            else {
               tempAttr = attr;
               if (!(tempAttr & 0x0800)) tempAttr |= (bankNo << 8); // not external or guard.
               addrStart = (U16)(slice[lp].addr1 >> 11);
               addrEnd = (U16)(slice[lp].addr2 >> 11);
               MapSet(tempAttr,addrEnd,addrStart);
               // MapSet(slice[lp].addr1,slice[lp].addr2,tempAttr);
            }
         }
         else if (bankNo < MAX_BANK) {
                 tempAttr = attr;
                 if (!(tempAttr & 0x0800)) tempAttr |= (bankNo << 8);
                 addrStart = (U16)(slice[lp].addr1 >> 11);
                 addrEnd = (U16)(slice[lp].addr2 >> 11);
                 MapSet(tempAttr,addrEnd,addrStart);
                 //MapSet(slice[lp].addr1,slice[lp].addr2,tempAttr);
              }
         lp++;
      }
   }
   return(GOOD);
}

/***************************************************************************
**
**  EmuSetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**               "addr" is the starting address of this mapping memory.
**               It should be on the address boundary of 64KB.
**               "map" is the mapping of current 64K memory bank.
**               If memoryresolution is 4KB and map is 0xFFCF. The emulation
**               memory will be mapped as two segments. 0..0x3fff and
**               0x6000..0xffff.
**               The definition of "attr" as below:
**               BIT Meaning
**               --- --------------------------------------------
**                0  "0" = Internal Memory, "1" = External memory
**                1  "0" = Read/Write, "1" = Read-Only
**                2  "0" = Exist memory, "1" = Gard memory
**                3  Reserved
**                4  Segments(with bit 5) -- "00" = ES, "01" = CS
**                5  Segments(with bit 4) -- "10" = SS, "11" = DS
**                6  Reserved
**                7  Reserved
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetMap(U32 addr1, U32 addr2, U16 attrib) {
STATUS status;

   status = GOOD;
   StopRun();
   status = SetMap(addr1,addr2,attrib);
   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

//VOID EmuSetMap(U32 addr1, U32 addr2, U16 attrib) {
//STATUS status;
//U16 addrStart,addrEnd;
//
//   if ((addr1 == 0) && (addr2 == 0)) {
//      mapBuf[0] = MAP_END_RECORD; /* set map data to map buffer */
//      status = InitMapCtr();
//   }
//   else {
//      addrStart = (U16)(addr1 >> 11);
//      addrEnd = (U16)(addr2 >> 11);
//      status = MapSet(attrib,addrEnd,addrStart);
//   }
//
//   status = status & 0x00ff;
//   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
//}


/***************************************************************************
**
**  EmuGetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**               "addr" is the starting address of this mapping memory.
**               It should be on the address boundary of 64KB.
**               "map" is the mapping of current 64K memory bank.
**               If memoryresolution is 4KB and map is 0xFFCF. The emulation
**               memory will be mapped as two segments. 0..0x3fff and
**               0x6000..0xffff.
**               The definition of "attr" as below:
**               BIT Meaning
**               --- --------------------------------------------
**                0  "0" = Internal Memory, "1" = External memory
**                1  "0" = Read/Write, "1" = Read-Only
**                2  "0" = Exist memory, "1" = Gard memory
**                3  Reserved
**                4  Segments(with bit 5) -- "00" = ES, "01" = CS
**                5  Segments(with bit 4) -- "10" = SS, "11" = DS
**                6  Reserved
**                7  Reserved
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMap(VOID) {
#define MAP_UNIT   3
U16 lp,*mapPtr1,*mapPtr2,*tempPtr,count,tempEnd;

   StopRun();
   tempEnd = 0xaa55;
   SetMapTbl();
   count = 0;
   mapPtr1 = mapPtr2 = mapBuf;
   mapPtr2 += MAP_UNIT;
   if (*mapPtr1 != MAP_END_RECORD) {
      while ((*mapPtr2 != MAP_END_RECORD) && (count < 86)) {
         // same address range.
         count++;
         if ((*mapPtr1 == *mapPtr2) && (*(mapPtr1+1) == *(mapPtr2+1)) &&
                (HighByte(*(mapPtr1+2)) == HighByte(*(mapPtr2+2))))
            break;
         mapPtr2 += MAP_UNIT;
      }
      if (count >= 86) {
         OutputStatus(MAP_FAIL,ON);
         return;
      }
      OutputStatus(GOOD,OFF);
      for (lp = 0; lp < count; lp++) {
         tempPtr = mapPtr2;
         while (*tempPtr != MAP_END_RECORD) {
            if ((*mapPtr1 == *(tempPtr)) && (*(mapPtr1+1) == *(tempPtr+1)) &&
                (HighByte(*(mapPtr1+2)) == HighByte(*(tempPtr+2))))
               *(mapPtr1+2) = *(mapPtr1+2) | *(tempPtr+2); // combine space
            tempPtr += MAP_UNIT;
         }

         mapData.addr1 = (U32)(*mapPtr1) << 11;
         mapData.addr2 = ((U32)(*(mapPtr1+1)) << 11) + 0x7ff;
         mapData.attr = AttrFw2Sw(*(mapPtr1+2));
         mapPtr1 += MAP_UNIT;

         OutData(10,&mapData,OFF);
      }
   }
   else OutputStatus(GOOD,OFF);
   OutData(2,&tempEnd,OFF);
   OutEnd();


}

/***************************************************************************
**
**  EmuGetMapSize
**
**  Description:
**
**  Parameters:
**     input:
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMapSize(VOID) {
U16 emmSize;
U8 dataLen;

   OutputStatus(GOOD,OFF);
   AccessIceFlag(READ_ONE,EMM_SIZE,&emmSize);
   dataLen = 2;
   OutData(dataLen,&emmSize,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuSetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetControl(U16 signals) {

   AccessSignals(WRITE_ALL,noUse,&signals);
   OutputStatus(GOOD,ON);
}


/***************************************************************************
**
**  EmuGetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetControl(VOID) {
U16 signals;
U8 dataLen;

   AccessSignals(READ_ALL,noUse,&signals);
   OutputStatus(GOOD,OFF);
   dataLen = sizeof(signals);
   OutData(dataLen,&signals,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuGetStatus
**
**  Description: Michelle setup-group routine, to read the current setting
**               of the Michelle.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." be Normal return
**        "Emulator Fatal Error" for HW error
**        return data -- "length" + "events" + "qualify: + "trigger" +
**                       "MAP" + control signals" +
**                       "ready setting"
**
**                     * "ID" is a issue to be discussed later
**
****************************************************************************/
VOID EmuGetStatus(VOID) {
U16 flyFlag;

  AccessIceFlag(READ_ONE,FLYING,&flyFlag);
  if (flyFlag) OutputStatus(CPU_FLY,ON);
  else if (!CheckEpStop()) OutputStatus(CPU_RUN,ON);
  else OutputStatus(BROKE,ON);
}

/***************************************************************************
**
**  EmuGetID
**
**  Description: Michelle setup-group routine, to get the emulator's ID to
**               make sure that driver program has connected to a correct
**               emulator.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**        return data -- "length" + "ID"
**
****************************************************************************/
VOID EmuGetID(VOID) {
U8 dataLen;

   OutputStatus(GOOD,OFF);
   dataLen = MAX_ID;
   // get probe,trace,cp board info from llfw and save to idData
   OutData(dataLen,idData,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuGetMode()
**
**  Description: Michelle setup-group routine, to select the Mode
**               "src" = 0 --> command mode
**               "src" = 1 --> queue status mode
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetMode(VOID) {
U8 dataLen,modeFlag;
U16 intReg[8],phyCpu;

   AccessIceFlag(READ_ONE,PHY_CPU_TYPE,&phyCpu);
   switch (phyCpu) {
      case XL_EA :
         modeFlag = ModeStatus();
         break;
      case EB :
         modeFlag = 0;
         break;
      case EC :
         modeFlag = 0;
         break;
      default :
         modeFlag = ModeStatus();
   }
   if ((cpuFlag & 0x0f) == 1) { //cpu = 801xxXL
      IntRegister(0xfff0,0,01,intReg);
      if ((intReg[0] & 0x7ffc) != 0) modeFlag |= 0x02; // bit2 equal 1
   }
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = 1;
   OutData(dataLen,&modeFlag,ON);
   OutEnd();

}


/****************************************************************************
**
**  EmuSetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetReady(U16 readyFlag) {

   AccessIceFlag(WRITE_ONE,READY_FLAG,&readyFlag);
   if (readyFlag == 0) ReadyInternal();
   else ReadyExternal();
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetReady(VOID) {
STATUS status;
U16 readyFlag;
U8 dataLen;

   status = GOOD;
   AccessIceFlag(READ_ONE,READY_FLAG,&readyFlag);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(readyFlag);
   OutData(dataLen,&readyFlag,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetAccessSize(SIZE size) {
STATUS status;

   status = GOOD;
   status = AccessIceFlag(WRITE_ONE,SIZE_FLAG,&size);

   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetAccessSize(VOID) {
STATUS status;
SIZE size;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(size);
   OutData(dataLen,&size,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetWait(U16 count) {

   AccessIceFlag(WRITE_ONE,WAIT_FLAG,&count);

   WaitStates(count);

   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}


/****************************************************************************
**
**  EmuGetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetWait(VOID) {
U8 dataLen;
U16 count;

   AccessIceFlag(READ_ONE,WAIT_FLAG,&count);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(count);
   OutData(dataLen,&count,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetSync(U16 state) {

   AccessIceFlag(WRITE_ONE,SYNC_FLAG,&state);
   switch (state) {
      case 0:
         DisableSSI();
         AccessIceFlag(WRITE_ONE,SYNC_IN,&C_OFF);
        break;
      case 1:
         EnableSSI();
         AccessIceFlag(WRITE_ONE,SYNC_IN,&C_ON);
        break;
      case 2:
         DisableSSO();
        break;
      case 3:
         EnableSSO();
        break;
   }
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}

/****************************************************************************
**
**  EmuGetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetSync(VOID) {
U16 sync;
U8 dataLen;

   AccessIceFlag(READ_ONE,SYNC_FLAG,&sync);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(sync);
   OutData(dataLen,&sync,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuSetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetVerify(U16 verifyFlag) {

   AccessIceFlag(WRITE_ONE,VERIFY_FLAG,&verifyFlag);

   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetVerify(VOID) {
STATUS status;
U16 verifyFlag;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(verifyFlag);
   OutData(dataLen,&verifyFlag,ON);
   OutEnd();
}

/***************************************************************************
**
**  EmuEpTimer
**
**  Description:
**
**
**
**
**  Parameters:
**     input:
**        flag -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuEpTimer(U16 timerFlag) {
U16 baseReg[MAX_BASE_REG_NO],timerValue;

   AccessBaseReg(READ_ALL,noUse,baseReg);
   switch (timerFlag) {
      case 0 :
         AccessInternalReg(READ_ONE,T0CON,&timerValue);
         baseReg[R_TIMER0] = timerValue;
         break;
      case 1 :
         AccessInternalReg(READ_ONE,T1CON,&timerValue);
         baseReg[R_TIMER1] = timerValue;
         break;
      case 2 :
         AccessInternalReg(READ_ONE,T2CON,&timerValue);
         baseReg[R_TIMER2] = timerValue;
         break;
      default : // 0
         break;

   }
   AccessBaseReg(WRITE_ALL,noUse,baseReg);
   stpTimFlg = timerFlag;
   AccessIceFlag (WRITE_ONE,TIMER_FLAG,&timerFlag);
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}
/******************************** E O F ***********************************/

