/****************************************************************************
**
**  Name:  iniutil.c
**
**  Description:
**     This module contains utility routines to access and manipulate
**               the PWRVIEWS.INI and other INI files
**
**  Status:  PRELIMINARY | CODED | REVIEWED | TESTED
**
**  $Log:   S:/tbird/mt2_186/cliulib/iniutil.c_v  $
** 
**    Rev 1.0   16 Dec 1996 16:12:36   Judy
** Initial revision.
** 
**    Rev 1.1   01 Sep 1994 17:57:06   nghia
** Revised all ini service functions to report the section and item name.
** 
**    Rev 1.0   18 May 1994 17:29:06   nghia
** Initial revision.
** 
**  $Header:   S:/tbird/mt2_186/cliulib/iniutil.c_v   1.0   16 Dec 1996 16:12:36   Judy  $
**
**  Copyright (C) 1991, 1994 Microtek International.  All rights reserved.
**
*****************************************************************************/

		       /****************************
			*                          *
			*       INCLUDE FILES      *
			*                          *
			****************************/

#include <stdio.h>
#include <stdlib.h>

#ifndef __STRING_H
#include "string.h"
#endif

#ifndef _INIUITIL_
#include "iniutil.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif
		       /****************************
			*                          *
			*     LOCAL DEFINITIONS    *
			*                          *
			****************************/
//
// Application and Key names in WIN.INI for PowerViews 
#define WAPP_NAME "MICEpack/SLD"
#define WAPP_KEY  "PowerViewDir"

#define MAX_ADDR_BUFF_SIZE 20
#define MAX_NUM_STR_SIZE   20

// Global PWRVIEWS paths - MAX_STRING_SIZE defined in BASICTYP.H 
PRIVATE CHAR pwrViewsIniPath[MAX_STRING_SIZE] = "";
PRIVATE BOOLEAN pwrViewsIniPathAvailable = FALSE;

		       /****************************
			*                          *
			*    EXTERNAL VARIABLES    *
			*                          *
			****************************/

		       /****************************
			*                          *
			*     LOCAL PROTOTYPES     *
			*                          *
			****************************/

		       /****************************
			*                          *
			*      EXECUTABLE CODE     *
			*                          *
			****************************/

/******************************************************************************
**
**  IniGetPwrViewsIniPath
**
**  Description:
**     Locate the PWRVIEWS.INI file directory and save it for all
**     subsequent access to the PWRVIEWS.INI file.
**
**  Parameters:
**     input:
**        none.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT IniGetPwrViewsIniPath(VOID) {
   CHAR tmpStr[MAX_STRING_SIZE];
   FILE *tmpFile;

   if (!pwrViewsIniPathAvailable) {
      // Get "pwrviewsdir" from WIN.INI first, if failed
      // use the current WINDOWS directory for PWRVIEWS.INI file
      if (GetProfileString((LPSTR) WAPP_NAME, (LPSTR) WAPP_KEY,
	   (LPSTR) "", (LPSTR) tmpStr, MAX_STRING_SIZE) == 0) {
	 /* Find the Windows directory to look again */
	 if (GetWindowsDirectory(tmpStr, MAX_STRING_SIZE) == 0)
	    return(ER_CLIULIB_WDIR_NOT_FOUND);
      }
      // check to make sure the PWRVIEWS.INI exist 
      strcat(tmpStr, "\\mp186.ini");
      if ((tmpFile = fopen(tmpStr, "r")) == NULL)
	 return(ER_CLIULIB_PFILE_NOT_FOUND);
      // Save the file name to global variable for later use 
      strcpy(pwrViewsIniPath, tmpStr);
      pwrViewsIniPathAvailable = TRUE;
      fclose(tmpFile);
   }
   return(GOOD);
}

/*****************************************************************************
**
**  IniSetNumber
**
*****************************************************************************/
RETCODE EXPORT IniSetNumber(LPSTR sectionName, LPSTR itemName,
			    LPSTR iniFileName, U32 numParam) {
   CHAR tmpStr[MAX_NUM_STR_SIZE];
   RETCODE err, ret;
   S16 buttonID;
   
   // make sure that the PWRVIEWS.INI path is valid if use default
   if (iniFileName == PWRVIEWS_INI_DEFAULT) {
      // use the PWRVIEWS.INI as default
      if (!pwrViewsIniPathAvailable &&
	  ((err = IniGetPwrViewsIniPath()) != GOOD))
	 return err;
      iniFileName = (LPSTR) pwrViewsIniPath;
   }
   if (!iniFileName)
      return ER_CLIULIB_INIPATH_NOT_AVAIL;
   
   // convert numParam to text 
   wsprintf((LPSTR)tmpStr, "%lu\0", numParam);
   // write it to the INI file 
   if (!WritePrivateProfileString(sectionName, itemName,
				(LPSTR)tmpStr, (LPSTR) iniFileName)) {
      // 08/31/94 - Nghia
      // report the error in place
      err =  ER_CLIULIB_UNABLE_TO_WRITE_INIFILE;
      if ((ret = ErrDisplayFormattedError(err, CHECK_MODE,
				       sectionName, itemName, iniFileName,
				       MB_OK,(S16 FAR *) &buttonID)) != GOOD)
	 ErrDisplayError(ret, CHECK_MODE);

      return ERR_REPORTED(err);
   }
   return GOOD;
}

/*****************************************************************************
**
**  IniGetNumber
**
*****************************************************************************/
RETCODE EXPORT IniGetNumber(LPSTR sectionName, LPSTR itemName,
			    LPSTR iniFileName, U32 *numParam) {
   CHAR tmpStr[MAX_NUM_STR_SIZE];
   S8 *endPtr;
   RETCODE err, ret;
   S16 buttonID;
   
   // make sure that the PWRVIEWS.INI path is valid if use default
   if (iniFileName == PWRVIEWS_INI_DEFAULT) {
      // use the PWRVIEWS.INI as default
      if (!pwrViewsIniPathAvailable &&
	  ((err = IniGetPwrViewsIniPath()) != GOOD))
	 return err;
      iniFileName = (LPSTR) pwrViewsIniPath;
   }
   if (!iniFileName)
      return ER_CLIULIB_INIPATH_NOT_AVAIL;
   
   // get the number from the specified item 
   if (GetPrivateProfileString(sectionName, itemName, "",
	   (LPSTR) tmpStr, MAX_NUM_STR_SIZE, (LPSTR) iniFileName) == 0) {
      // 08/31/94 - Nghia
      // report the error in place
      err = ER_CLIULIB_UNABLE_TO_READ_INIFILE;
      if ((ret = ErrDisplayFormattedError(err, CHECK_MODE,
				       sectionName, itemName, iniFileName,
				       MB_OK,(S16 FAR *) &buttonID)) != GOOD)
	 ErrDisplayError(ret, CHECK_MODE);

      return ERR_REPORTED(err);
   }
   // convert tmpStr text to number
   // use radix 0 -> first letter determine the radix of the number
   *numParam = (U32) strtoul((LPSTR)tmpStr, &endPtr, 0);
   return GOOD;   
}

/*****************************************************************************
**
**  IniSetString
**
*****************************************************************************/
RETCODE EXPORT IniSetString(LPSTR sectionName, LPSTR itemName,
			    LPSTR iniFileName, LPSTR strParam) {
   RETCODE err, ret;
   S16 buttonID;
   
   // make sure that the PWRVIEWS.INI path is valid if use default
   if (iniFileName == PWRVIEWS_INI_DEFAULT) {
      // use the PWRVIEWS.INI as default
      if (!pwrViewsIniPathAvailable &&
	  ((err = IniGetPwrViewsIniPath()) != GOOD))
	 return err;
      iniFileName = (LPSTR) pwrViewsIniPath;
   }
   if (!iniFileName)
      return ER_CLIULIB_INIPATH_NOT_AVAIL;

   // write the information to INI file
   if (!WritePrivateProfileString(sectionName, itemName,
				 strParam, (LPSTR) iniFileName)) {
      // 08/31/94 - Nghia
      // report the error in place
      err =  ER_CLIULIB_UNABLE_TO_WRITE_INIFILE;
      if ((ret = ErrDisplayFormattedError(err, CHECK_MODE,
				       sectionName, itemName, iniFileName,
				       MB_OK,(S16 FAR *) &buttonID)) != GOOD)
	 ErrDisplayError(ret, CHECK_MODE);

      return ERR_REPORTED(err);
   }
   return GOOD;
}

/*****************************************************************************
**
**  IniGetString
**
*****************************************************************************/
RETCODE EXPORT IniGetString(LPSTR sectionName, LPSTR itemName,
			    LPSTR iniFileName, LPSTR strParam) {
   RETCODE err, ret;
   S16 buttonID;
   
   // make sure that the PWRVIEWS.INI path is valid if use default
   if (iniFileName == PWRVIEWS_INI_DEFAULT) {
      // use the PWRVIEWS.INI as default
      if (!pwrViewsIniPathAvailable &&
	  ((err = IniGetPwrViewsIniPath()) != GOOD))
	 return err;
      iniFileName = (LPSTR) pwrViewsIniPath;
   }
   if (!iniFileName)
      return ER_CLIULIB_INIPATH_NOT_AVAIL;
   
   // Read the string from the specified item 
   if (GetPrivateProfileString(sectionName, itemName,
	  "", strParam, MAX_STRING_SIZE, (LPSTR)iniFileName) == 0) {
      // 08/31/94 - Nghia
      // report the error in place
      err = ER_CLIULIB_UNABLE_TO_READ_INIFILE;
      if ((ret = ErrDisplayFormattedError(err, CHECK_MODE,
				       sectionName, itemName, iniFileName,
				       MB_OK,(S16 FAR *) &buttonID)) != GOOD)
	 ErrDisplayError(ret, CHECK_MODE);

      return ERR_REPORTED(err);
   }
   return GOOD;
}

/********************************** E O F ***********************************/
