/****************************************************************************
**
**  Name:  iniutil.c
**
**  Description:
**     This module contains utility routines to access and manipulate
**               the PWRVIEWS.INI
**
**  Status:  PRELIMINARY | CODED | REVIEWED | TESTED
**
**  $Log:   S:/tbird/mt2_186/stkservr/iniutil.c_v  $
** 
**    Rev 1.0   16 Dec 1996 14:44:56   Judy
** Initial revision.
** 
**    Rev 1.4   11 Oct 1994 09:45:42   matthew
** No change.
** 
**    Rev 1.3   06 Apr 1994 17:06:30   nghia
** - Replaced AdrCreateAddress() with CpuGetSP() to make sure address will
** be compatible.
** 
**    Rev 1.2   18 Jul 1993 19:02:04   steve
** Changed ER_PPATH_NOT_AVAILABLE to ..._NOT_AVAIL
** 
**    Rev 1.1   16 Jul 1993 17:12:36   nghia
** Removed error checking when iniGetNum() get a number == 0.
** 
**    Rev 1.0   16 Jul 1993 11:59:10   nghia
** Initial revision.
**
**  $Header:   S:/tbird/mt2_186/stkservr/iniutil.c_v   1.0   16 Dec 1996 14:44:56   Judy  $
**
**  Copyright (C) 1991, 1993 Microtek International.  All rights reserved.
**
*****************************************************************************/

		       /****************************
			*                          *
			*       INCLUDE FILES      *
			*                          *
			****************************/

#include <stdio.h>
#include <stdlib.h>

#ifndef __STRING_H
#include "string.h"
#endif

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _INIUITIL_
#include "iniutil.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _ERRCODEC_
#include "errcodec.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _CPU_
#include "cpu.h"
#endif

		       /****************************
			*                          *
			*     LOCAL DEFINITIONS    *
			*                          *
			****************************/
/* Application and Key names in WIN.INI for PowerViews */
#define WAPP_NAME "PowerViews"
#define WAPP_KEY  "PowerViewDir"

#define MAX_ADDR_BUFF_SIZE 20

/* Global PWRVIEWS.INI path - MAX_STR_SIZE defined in cliulib.h */
CHAR pwrViewsPath[MAX_STR_SIZE];
PRIVATE BOOLEAN iniPathAvailable = FALSE;


		       /****************************
			*                          *
			*    EXTERNAL VARIABLES    *
			*                          *
			****************************/

		       /****************************
			*                          *
			*     LOCAL PROTOTYPES     *
			*                          *
			****************************/

		       /****************************
			*                          *
			*      EXECUTABLE CODE     *
			*                          *
			****************************/

/******************************************************************************
**
**  IniGetPwrViewsIniPath
**
**  Description:
**     Locate the PWRVIEWS.INI file directory and store it in for INI access
**     functions.
**
**  Parameters:
**     input:
**        none.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT IniGetPwrViewsIniPath(VOID) {
   CHAR tmpStr[MAX_STR_SIZE];
   FILE *tmpFile;

   if (!iniPathAvailable) {
      /* Get "pwrviewsdir" from WIN.INI */
      if (GetProfileString((LPSTR) WAPP_NAME, (LPSTR) WAPP_KEY,
			  (LPSTR) "", (LPSTR) tmpStr, MAX_STR_SIZE) == 0) {
	 /* Find the Windows directory to look again */
	 if (GetWindowsDirectory(tmpStr, MAX_STR_SIZE) == 0)
	    return(ER_CLIULIB_WDIR_NOT_FOUND);
	 else {
	    /* Build path to locate PWRVIEWS.INI file in WINDOWS  dir */
	    strcat(tmpStr, "\\mp186.ini");
	    if ((tmpFile = fopen(tmpStr, "r")) == NULL)
	       return(ER_CLIULIB_PFILE_NOT_FOUND);
	 }
      }
      else {
	 /* Check to makesure the filename specified in WIN.INI exist */
	 strcat(tmpStr, "\\mp186.ini");
	 if ((tmpFile = fopen(tmpStr, "r")) == NULL)
	    return(ER_CLIULIB_PFILE_NOT_FOUND);
      }
      /* Save the file name to global variable for later use */
      strcpy(pwrViewsPath, tmpStr);
      iniPathAvailable = TRUE;
      fclose(tmpFile);
   }
   return(GOOD);
}

/*****************************************************************************
**
**  IniSetAddress
**
*****************************************************************************/
RETCODE EXPORT IniSetAddress(LPSTR sectionName, LPSTR itemName,
			      DESCRIPTOR addrParam) {
   RETCODE err;
   CHAR addrBuff[MAX_ADDR_BUFF_SIZE];
   LPSTR bufPtr;

   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);
   /* Convert the address descritor to text */
   memset(&addrBuff[0],'\0', sizeof(addrBuff));
   bufPtr = (LPSTR)&addrBuff[0];
   if ((addrParam != NULL) && /* prefix 0x and zero fill the address */
      ((err = AdrConvAddressToTextWithParams(addrParam, TRUE,
					   TRUE, bufPtr)) != GOOD)) {
      return(err);
   }
   if (!WritePrivateProfileString(sectionName, itemName,
			(LPSTR) bufPtr, (LPSTR) pwrViewsPath)) 
      return(ER_CLIULIB_UNABLE_TO_WRITE_PFILE);   
   return(AdrDestroyAddress(addrParam));   
}

/*****************************************************************************
**
**  IniGetAddress
**
*****************************************************************************/
RETCODE EXPORT IniGetAddress(LPSTR sectionName, LPSTR itemName,
			      DESCRIPTOR *addrParam) {
   RETCODE err;
   CHAR tmpStr[MAX_STR_SIZE];

   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);
   
   /* Get the address text at the sepcified itemName */
   if (GetPrivateProfileString(sectionName, itemName,"", (LPSTR)tmpStr,
			       MAX_STR_SIZE, (LPSTR) pwrViewsPath) == 0) {
      return(ER_CLIULIB_UNABLE_TO_READ_PFILE);
   }
   /* Allocate an address descriptor */
   if ((err = CpuGetSP((DESCRIPTOR FAR*)addrParam)) != GOOD)
      return(err);
   /* Convert address text to descriptor value */
   if ((err = AdrConvTextToAddress(*addrParam, tmpStr)) != GOOD) {
      AdrDestroyAddress(*addrParam);
      return(err);
   }
   return(GOOD); 
}

/*****************************************************************************
**
**  IniSetNum
**
*****************************************************************************/
RETCODE EXPORT IniSetNum(LPSTR sectionName, LPSTR itemName, U32 numParam) {
   CHAR tmpStr[MAX_STR_SIZE];

   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);
   
   /* Convert numParam to text */
   wsprintf((LPSTR)tmpStr, "%lu\0", numParam);
   /* Write it to the INI file */
   if(!WritePrivateProfileString(sectionName, itemName,
				(LPSTR)tmpStr, (LPSTR) pwrViewsPath)){
      return(ER_CLIULIB_UNABLE_TO_WRITE_PFILE);
   }
   return(GOOD);
}

/*****************************************************************************
**
**  IniGetNum
**
*****************************************************************************/
RETCODE EXPORT IniGetNum(LPSTR sectionName, LPSTR itemName, U32 *numParam) {
   CHAR tmpStr[MAX_STR_SIZE];
   S8 *endPtr;
   
   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);
   
   /* Get the number from the specified item */
   if (GetPrivateProfileString(sectionName, itemName, "",
	   (LPSTR) tmpStr, MAX_STR_SIZE, (LPSTR) pwrViewsPath) == 0) {
      return(ER_CLIULIB_UNABLE_TO_READ_PFILE);
   }
   /* Convert tmpStr text to number */
   *numParam = strtoul((LPSTR)tmpStr, &endPtr, 0);
   return(GOOD);   
}

/*****************************************************************************
**
**  IniSetString
**
*****************************************************************************/
RETCODE EXPORT IniSetString(LPSTR sectionName, LPSTR itemName,
					       LPSTR strParam) {
   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);

   if(!WritePrivateProfileString(sectionName, itemName,
				 strParam, (LPSTR) pwrViewsPath)){
      return(ER_CLIULIB_UNABLE_TO_WRITE_PFILE);
   }
   return(GOOD);
}

/*****************************************************************************
**
**  IniGetString
**
*****************************************************************************/
RETCODE EXPORT IniGetString(LPSTR sectionName, LPSTR itemName,
					       LPSTR strParam) {
   if (!iniPathAvailable)
      return(ER_CLIULIB_PPATH_NOT_AVAIL);

   /* Read the string from the specified item */
   if (GetPrivateProfileString(sectionName, itemName,
	  "", strParam, MAX_STR_SIZE, (LPSTR)pwrViewsPath) == 0) {
      return(ER_CLIULIB_UNABLE_TO_READ_PFILE);
   }
   return(GOOD);
}

/********************************** E O F ***********************************/
