/* Class: BkptIntelDialog
   This class handles setting breakpoint for the Intel processors family.
*/!!

inherit(BkptDialog, #BkptIntelDialog, #(addrSpace /* Address Space */), 2, nil)!!

now(class(BkptIntelDialog))!!

/* 3/15/1994 17:13 - PUBLIC */
Def new(self, tmpDict, space | theDlg)
{ 
  theDlg := new(self:Behavior);
  setAddrSpace(theDlg, space);
  ^init(theDlg, tmpDict);
}
!!

now(BkptIntelDialog)!!

/* 3/16/1994 10:06 - PRIVATE
  Set a breakpoint according to user input. 
  Return nil if successful or error message.
  NOTES: Override ancestor's method to handle address space and filter out physical
         address input.
*/
Def setBkptWithInput(self, bkptInput | addrDesc, addrType, spaceTxt, errTxt)
{ 
  /* Check breakpoint input - convert to an <addrDesc> */
  if not(addrDesc := convertTextToAddressNoError(AddressLibClass$Inst, bkptInput)) then
    ^getErrorText(ErrorTextLibClass$Inst, lastError(AddressLibClass$Inst));
  endif;
  if not(addrType := getAddressType(AddressLibClass$Inst, addrDesc)) then 
    errTxt := getErrorText(ErrorTextLibClass$Inst, lastError(AddressLibClass$Inst));
    destroyAddress(AddressLibClass$Inst, addrDesc);
    ^errTxt;
  endif;
  
  /* 06/15/94 - Intel processor does not support setting breakpoint with 
  ** physical address.
  */
  if (addrType = ADDR_PHYSICAL) then
    destroyAddress(AddressLibClass$Inst, addrDesc);
    ^getErrorText(ErrorTextLibClass$Inst, ER_ADR_TYPE_NOT_SUPPORTED);
  endif;
  
  /* retrieve the specified address space to set breakpoint */
  if not(spaceTxt := getCBText(self, DLG_COMBO_ADDR_SPACE)) then
    /* Force Address server to return error for the invalid spaceTxt */
    spaceTxt := "UNKNOWN";  
  endif;
  /* Translate address space text */
  if not(addrSpace := translateAddrSpace(AddressLibClass$Inst, spaceTxt)) then
    errTxt := getErrorText(ErrorTextLibClass$Inst, lastError(AddressLibClass$Inst));
    destroyAddress(AddressLibClass$Inst, addrDesc);
    ^errTxt;
  endif;
  
  if not(setAddrSpace(AddressLibClass$Inst, addrDesc, addrSpace)) then
    errTxt := getErrorText(ErrorTextLibClass$Inst, lastError(AddressLibClass$Inst));
    destroyAddress(AddressLibClass$Inst, addrDesc);
    ^errTxt;
  endif;  
  /* Set breakpoint using address descriptor - Server consumes descriptor */
  if not(prime_setAsmBkpt(HLBreakLibClass$Inst, bkptState, bkptLife, addrDesc)) then
    ^getErrorText(ErrorTextLibClass$Inst, lastError(HLBreakLibClass$Inst));       
  endif;  
  ^nil; /* Successful */
}


!!

/* 3/15/1994 17:19 - PRIVATE
  Initialize self before display.
*/
Def initDialog(self, wp, lp | spaceColl, addrTxt)
{ 
  initDialog(self:ancestor, wp, lp);
  clearCBText(self, DLG_COMBO_ADDR_SPACE);
  /* Retrieve the address Spaces list from the Address Server */
  if (spaceColl := getAddressSpaces(AddressLibClass$Inst)) cand
     (size(spaceColl) > 0) then
    /* remove io entry */
    spaceColl := extract(spaceColl, {using(space)  asLowerCase(space) <> "io"; });
    do (spaceColl,
      {using(txtElement)
        addCBText(self, DLG_COMBO_ADDR_SPACE, txtElement);
      }); 
    /* Set current selection for space */    
    if (addrTxt := getAddrSpaceText(AddressLibClass$Inst, addrSpace)) then
      sendDlgItemMessage(self, DLG_COMBO_ADDR_SPACE, CB_SELECTSTRING, 0, asciiz(addrTxt)); 
    else 
      sendDlgItemMessage(self, DLG_COMBO_ADDR_SPACE, CB_SETCURSEL, 0, 0L); 
    endif;  
  else
    /* No spaces return - disable the control */
    disableItem(self, DLG_COMBO_ADDR_SPACE);
  endif;
}
!!

/* 3/15/1994 17:17 - PRIVATE 
  Set the current address space use in setting breakpoint.
  NOTES:
    <space> is the actual address space enumuration of the address
    server.
*/
Def setAddrSpace(self, space)
{ 
  if not(space) then
    space := ADDR_SPACE_DEFAULT; /* Processor family independent */
  endif;  
  addrSpace := space;
}
!!