/* Class : BkptPresenter - BKPTPRES.CLS
   The Breakpoint Presenter class provide interface and operations to the
   user to manipulate breakpoints in a debug session. 
   Interface directly to the PowerTool Bar and the Source presenter
   REQUIRE: BKPTDIAL.CLS, BKPTINFO.CLS, BKPTLIST.CLS & PBKPT.H
 */!!

inherit(Window, #BkptPresenter, #(childDict /* child controls collection */
childActions /* action of childs */
eventDescSet /* set of events notify */
eventRespSet /* respond of events */
bkptLB /* Breakpoints Listbox */
currentSel /* Current Selection */
tmWidth /* Text metrics */
tmHeight
minButtonWidth /* Length of button text */), 2, nil)!!

now(class(BkptPresenter))!!

/* 6/12/1992 14:18 - Actor */
Def open(self | aPresenter)
{ 
  TheBreakpointPresenter := new(BkptPresenter, nil, nil, "Breakpoint", nil);
  show(TheBreakpointPresenter, SW_SHOW);
  ^TheBreakpointPresenter;
}
!!

/* start presenter with specific position */
Def openWithPosAndState(self, sizeRect, showVal | x, y)
{
  /* Provide a reasonable default size for the Breakpoint window */
  if not(sizeRect) then
    /* Set its location and size relative to the screen size */
    x := asInt((x(screenSize()) - 600)/4);
    y := asInt((y(screenSize()) - 200)/4);
    if (sizeRect := new(Rect))
      init(sizeRect, x, y, x+600, y+200);
    endif;  
  endif;      

  /* Open The Breakpoint presenter */
  TheBreakpointPresenter := newStyle(BkptPresenter, nil, nil, "Breakpoint",
                   sizeRect, nil, WS_OVERLAPPEDWINDOW);
  show(TheBreakpointPresenter, showVal);
 ^TheBreakpointPresenter;
}

!!


/* 6/12/1992 14:22 */
Def style(self)
{ 
  ^WS_OVERLAPPEDWINDOW;
}
!!

/* 6/12/1992 14:23 - Actor */
Def wndClass(self)
{ 
  ^"BkptPresenter";
}
!!

/* 6/12/1992 14:23 - Actor */
Def wndIcon(self)
{ 
  ^"Breakpoint";
}
!!

now(BkptPresenter)!!

/* 06/15/92 - PRIVATE
  Get the text metrics data for this window.  Load the font data 
  into textMetrics, get the text width and height of window, then
  return the tuple(tmWidth, tmheight)
*/
Def initTextMetrics(self | hdc, tm)
{ 
  tm := new(Struct, 32);
  Call GetTextMetrics(hdc := getContext(self), tm);
  Call SelectObject(hdc, Call GetStockObject(SYSTEM_FIXED_FONT));
  tmWidth  := asInt(wordAt(tm, 10));
  tmHeight := asInt(wordAt(tm, 8)) + asInt(wordAt(tm, 0));
  Call SelectObject(hdc, Call GetStockObject(SYSTEM_FONT));
  releaseContext(self, hdc);
}!!

/* 6/12/1992 16:22 - PRIVATE
  Clear the selected breakpoint.
*/
Def cmdEditClear(self, item | bkpt)
{ break("CCC");
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  /* Remove it from list and server */
  if not(bkpt := getSelBkpt(bkptLB)) then
     displayFormattedError(ErrorTextLibClass$Inst, 
        ER_BKPT_NONE_SELECTED, FORCE_POPUP, "to delete", nil, nil);
     ^nil;
  endif;     
  deleteBkpt(bkptLB, bkpt);
  /* Let Events switch the command. */
}
!!

/* 6/12/1992 16:22 - PRIVATE
  Run the Set breakpoint dialog.
*/
Def cmdEditSet(self, item | theDlg, modulesColl, resourceId, loadFile, msgDlg)
{ 
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  msgDlg := nil;
  if not(loadFile := currentLoadFile(TheProjectInfoObj)) then
    ^nil;
  endif;
  if (not(modulesDictUpdated(loadFile)) cand
     (numLoadedModules(loadFile) > 10)) then
    msgDlg := new(Dialog); 
    runModeless(msgDlg, DLG_WAIT_MSG, ThePort);  
  endif;    
  /* Get module Names dictionary from the load file ondemand */
  modulesColl := getModuleNamesDict(loadFile);

  /* Clean up : message dialog */
  if (msgDlg <> nil) then  end(msgDlg, 1); endif;

  /* Create a different BkptDialog object for each processor family */
  if (TheProcFamily = PROC_FAMILY_X86) then
    resourceId := INTEL_BKPT_DLG;      
    theDlg := new(BkptIntelDialog, modulesColl, ADDR_SPACE_DEFAULT);    
  else
    resourceId := BKPT_DIALOG_INDEX;
    theDlg := new(BkptDialog, modulesColl);
  endif;
  /* Run the dialog box */
  if not(theDlg) cor (runModal(theDlg, resourceId, ThePort) <> IDOK) then
    ^nil;
  endif;   
  /* Let Event Notification updates the new breskpoint setting */
  ^GOOD;                 
}
!!

/* 06/20/92 - PUBLIC */
Def close(self)
{
  closeEvents(EvNoteLibClass$Inst, eventDescSet);
  clear(bkptLB); /* Release bkptInfo objects & addrDesc */
  removeWindowMenu(self);
  TheBreakpointPresenter := nil ; /* kill global reference */
  close( self:ancestor ) ;
}
!!

/* 6/12/1992 16:22 - PRIVATE
*/
Def cmdEditDeleteAll(self, item)
{ 
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  if (errMessageBox(ErrorTextLibClass$Inst, "Warning", 
       "Do you really want to delete all breakpoints?", 
       MB_YESNO bitOr MB_ICONQUESTION, HE_DLGM_BKPTPRES_1) = IDNO) then
    ^nil;
  endif;
  break("CCC");
  deleteBkpts(bkptLB); 
}
!!

/* 6/12/1992 16:22 - PRIVATE
*/
Def cmdEditDisable(self, item | bkpt)
{ 
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  if not(bkpt := getSelBkpt(bkptLB)) then
     displayFormattedError(ErrorTextLibClass$Inst, 
        ER_BKPT_NONE_SELECTED, FORCE_POPUP, "to disable", nil, nil);
    ^nil;
  endif;     
  ^disableBkpt(HLBreakLibClass$Inst, bkptID(bkpt));  
}
!!

/* 6/12/1992 16:22 - PRIVATE
*/
Def cmdEditEnable(self, item | bkpt)
{ 
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  if not(bkpt := getSelBkpt(bkptLB)) then
    displayFormattedError(ErrorTextLibClass$Inst, 
       ER_BKPT_NONE_SELECTED, FORCE_POPUP, "to enable", nil, nil);
    ^nil;
  endif;     
  enableBkpt(HLBreakLibClass$Inst, bkptID(bkpt));  
  /* Event Notification will update the list */
  setFocus(bkptLB);
}
!!

/* 6/12/1992 16:22 - PRIVATE
*/
Def cmdEditGotoSource(self, item | bkpt, cloneDesc)
{ 
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;

  if bkpt := getSelBkpt(bkptLB) then
    if not(TheSourcePresenter) then
      TheSourcePresenter := open(SourcePresenter);
    endif;  
    /* Need to clone the address descriptor, showFileFromAddress() will consume it */
    if TheSourcePresenter cand (cloneDesc := bkptAddress(bkpt)) then
      /* 
      *** Call SourcePresenter to show source - Source will destroy the cloneDesc 
      */
      showWaitCurs();
      showDataObjFromAddress(TheSourcePresenter, cloneDesc, nil); /* nil - no TokenHilight */
      show(TheSourcePresenter, SW_NORMAL);
      bringToTop(TheSourcePresenter);
      showOldCurs();
      ^GOOD;
    endif;
    ^nil; /* Failed to perform the requested operation */
  endif;  
  displayFormattedError(ErrorTextLibClass$Inst, 
     ER_BKPT_NONE_SELECTED, FORCE_POPUP, "to goto source", nil, nil);
  ^nil;
}
!!

/* 6/12/1992 16:22 - PRIVATE
*/
Def cmdEditSetAll(self, item | bkpt)
{ break("CCC");
  /* If emulation is running then do not perform the requested function */
  if not(processorHalted?(HLBrkRootLibClass$Inst)) then
    ^nil;
  endif;
  showWaitCurs();
  if size(bkptCollection(bkptLB)) = 0 then
     displayFormattedError(ErrorTextLibClass$Inst, 
        ER_BKPT_NONE_AVAILABLE, FORCE_POPUP, nil, nil, nil);
     ^nil;
  endif;     
  if (item = BKPT_EDIT_ENABLE_ALL) cor (item = ENABLE_ALL_BUTTON) then
    enableAllBkpt(HLBreakLibClass$Inst);
  else  
    disableAllBkpt(HLBreakLibClass$Inst);
  endif;
  /* Events Notification will update */
  showOldCurs();
}
!!

/* 6/12/1992 16:37 */
Def cmdFileExit(self, item)
{ 
  close(self);
}
!!

/* 6/12/1992 15:56 - PRIVATE
  Provide Help dispatching for the Breakpoint Presenter.
*/
Def cmdHelp(self, item)
{ 
  select
    case item = INDEX_HELP
      is contents(TheHelpObject, hWnd(self));
    endCase
    case item = USING_HELP
      is help(TheHelpObject, hWnd(self));
    endCase
    case item = ABOUT_HELP
      is contextNumber(TheHelpObject, hWnd(self), HELP_ENTRY_BREAKPOINT);
    endCase    
  endSelect;
}

!!

/* 6/16/1992 10:22 - Actor 
  Respond to the menu events.
  The wp argument gives the selected menu ID.
  Get a message symbol from the menu object. 
  NOTES: Nghia - 10/12/93
  Clear ESC key before processing command.
*/
Def command(self, wp, lp | msg)
{ break("CCC");
  /* Perform the menu action */  
  if msg := action(menu, wp) then 
    /* Clear ESC key */
    if (TaskLibClass$Inst)
      checkAbort(TaskLibClass$Inst);
    endif;
    ^perform(self, wp, msg)
  endif;

  /* Process child control actions */
  if msg := at(childActions, wp) then 
    /* Clear ESC key */
    if (TaskLibClass$Inst)
      checkAbort(TaskLibClass$Inst);
    endif;
    if wp = BKPT_LB then
      ^perform(self, wp, lp, msg); /* BkptList requires lp parameter */
    endif;
    perform(self, wp, msg); /* Button's action */ 
    if wp <> INSPECT_BUTTON then
      ^setFocus(self);
    endif;
    ^self;  
  endif; 
  
  /* Process Windows menu commands */         
  if not(doAction(PreLauncher, wp)) then
    command(self:ancestor, lp, wp);
  endif;         
}
!!

/* 5/8/1992 10:47 - PRIVATE 
  Disable all specified menu items.
  EX: disableCommands(self, #(MENU_ID, ...));
*/
Def disableCommands(self, menuCmdSet)
{ break("BBB");
  if menuCmdSet then
    do(menuCmdSet,
      {using(item)
        grayMenuItem(menu, item);
      });
  endif; 
}
!!

/* 6/18/1992 8:50 - PRIVATE
  Draw Breakpoint list labels 
*/
Def drawBkptLabels(self | x, y)
{ 
  /* draw objects label on top of its location */
  x := left(cRect(bkptLB));
  y := top(cRect(bkptLB)) - tmHeight;
  drawText(self, "State", x, y);
  drawText(self, "Type", x+asInt(tmWidth*10), y);
  drawText(self, "Breakpoints", x+asInt(tmWidth*20), y);
}
!!

/* 5/8/1992 10:47 - PRIVATE 
  Enable all specified menu items and the corresponding child objects.
  EX: enableCommands(self, #(MENU_ID, ...));
*/
Def enableCommands(self, menuCmdSet)
{ break("AAA");
  if menuCmdSet then
    do(menuCmdSet,
      {using(item)
        enableMenuItem(menu, item);
      });
  endif; 
}
!!

/* 6/12/1992 15:14 - PRIVATE 
  Rebuild the breakpoint collection and enable/disable appropriate commands.
*/
Def eventBreakpointChanged(self | numBkpt)
{ 
  if (numBkpt := size(rebuildBkptCollection(bkptLB))) > 0 then
  
    enableCommands(self, tuple(BKPT_EDIT_DEL, 
                             BKPT_EDIT_INSPECT,
                             BKPT_EDIT_ENABLE,
                             BKPT_EDIT_DISABLE,
                             BKPT_EDIT_ENABLE_ALL,
                             BKPT_EDIT_DISABLE_ALL,
                             BKPT_EDIT_DEL_ALL)); 
    enable(childDict[DEL_BUTTON]);
    enable(childDict[INSPECT_BUTTON]);
    enable(childDict[ENABLE_BUTTON]);
    enable(childDict[DISABLE_BUTTON]);
    enable(childDict[ENABLE_ALL_BUTTON]);
    enable(childDict[DISABLE_ALL_BUTTON]);                         
  else
    disableCommands(self, tuple(BKPT_EDIT_DEL, 
                             BKPT_EDIT_INSPECT,
                             BKPT_EDIT_ENABLE,
                             BKPT_EDIT_DISABLE,
                             BKPT_EDIT_ENABLE_ALL,
                             BKPT_EDIT_DISABLE_ALL,
                             BKPT_EDIT_DEL_ALL));
    disable(childDict[DEL_BUTTON]);
    disable(childDict[INSPECT_BUTTON]);
    disable(childDict[ENABLE_BUTTON]);
    disable(childDict[DISABLE_BUTTON]);
    disable(childDict[ENABLE_ALL_BUTTON]);
    disable(childDict[DISABLE_ALL_BUTTON]);
  endif;                             
  /* Reset the currentSelection */  
  if (numBkpt <= currentSel) then
    currentSel := max(0, numBkpt-1);
  endif;
  setCurSel(bkptLB, currentSel);
  if hasFocus(self) then
    setFocus(bkptLB);
  endif;
}
!!

/* 6/12/1992 14:24 - PUBLIC
  initialize the presenter.
*/
Def init(self | ret)
{ break("CCC");
  showWaitCurs();
  init(self:ancestor);
  initTextMetrics(self);  
  initMainMenu(self);
  registerF1Help(CLIULibraryClass$Inst, HI_ENTRY_BREAKPOINT,
     getHWnd(self), HELP_ENTRY_BREAKPOINT);
  if not(initExecutionBar(self)) then
    displayFormattedError(ErrorTextLibClass$Inst, 
       ER_BKPT_NO_WINDOW, FORCE_POPUP, nil, nil, nil);
    ^nil;
  endif;  
  if not(initEvents(self)) then
    displayFormattedError(ErrorTextLibClass$Inst, 
       ER_BKPT_CANT_REGISTER, FORCE_POPUP, nil, nil, nil);
    ^nil;
  endif;
  currentSel := 0;
  if not(ret := rebuildBkptCollection(bkptLB)) then 
    ^nil; /* Error in BkptList - bail out */
  endif;
  /* Set up commands */
  if size(ret) = 0 then  
    /* There are no breakpoints present */
    disableCommands(self, tuple(BKPT_EDIT_DEL, 
                             BKPT_EDIT_INSPECT,
                             BKPT_EDIT_ENABLE,
                             BKPT_EDIT_DISABLE,
                             BKPT_EDIT_ENABLE_ALL,
                             BKPT_EDIT_DISABLE_ALL,
                             BKPT_EDIT_DEL_ALL)); 
    disable(childDict[DEL_BUTTON]);
    disable(childDict[INSPECT_BUTTON]);
    disable(childDict[ENABLE_BUTTON]);
    disable(childDict[DISABLE_BUTTON]);
    disable(childDict[ENABLE_ALL_BUTTON]);
    disable(childDict[DISABLE_ALL_BUTTON]);
  endif;          
  setCurSel(bkptLB, currentSel);                 
  setFocus(bkptLB);
  /* use a different help index than above */
  registerF1Help(CLIULibraryClass$Inst, HI_MESSAGEBOXEX,
     getHWnd(bkptLB), HELP_ENTRY_BREAKPOINT);
  initMainWindow(self);     
  showOldCurs();  
}
!!

/* 6/12/1992 14:28 - PRIVATE 
*/
Def initEvents(self | events)
{
  eventRespSet := new(Dictionary, 1);
    
  /* dynamic to pick up current values */
  events := tuple( EVENT_BKPT_EDIT );
  /* Setup events respond messages */
  add(eventRespSet, EVENT_BKPT_EDIT, #eventBreakpointChanged);
 
  /* register with Event Notification library */
  requireWithPath( EvNoteLib, "enlib.dll");                
  ^(eventDescSet := registerEvents( EvNoteLibClass$Inst, hWnd(self), events));
}
!!

/* 06/12/92 - PRIVATE
   Init Execution Buttons
*/
Def initExecutionBar(self | theBut)
{
  /* Create child controls for execution buttons */
  childDict    := new(Dictionary, 7);
  childActions := new(Dictionary, 7);

  /* 12 = number of chars in a longest button title - "Disable All" */
  minButtonWidth := (tmWidth * 12);  

  theBut := newPush(Button, ADD_BUTTON,         self, "&Set");
  add(childDict, ADD_BUTTON, theBut);
  theBut := newPush(Button, DEL_BUTTON,         self, "&Clear");
  add(childDict, DEL_BUTTON, theBut);
  theBut := newPush(Button, INSPECT_BUTTON,     self, "&Go To Source");
  add(childDict, INSPECT_BUTTON, theBut);
  theBut := newPush(Button, ENABLE_BUTTON,      self, "&Enable");
  add(childDict, ENABLE_BUTTON, theBut);
  theBut := newPush(Button, DISABLE_BUTTON,     self, "&Disable");
  add(childDict, DISABLE_BUTTON, theBut);
  theBut := newPush(Button, ENABLE_ALL_BUTTON,  self, "E&nable All");
  add(childDict, ENABLE_ALL_BUTTON, theBut);
  theBut := newPush(Button, DISABLE_ALL_BUTTON, self, "D&isable All");
  add(childDict, DISABLE_ALL_BUTTON, theBut);

  add(childActions, ADD_BUTTON,         #cmdEditSet);
  add(childActions, DEL_BUTTON,         #cmdEditClear);
  add(childActions, INSPECT_BUTTON,     #cmdEditGotoSource);
  add(childActions, ENABLE_BUTTON,      #cmdEditEnable);
  add(childActions, DISABLE_BUTTON,     #cmdEditDisable);
  add(childActions, ENABLE_ALL_BUTTON,  #cmdEditSetAll);
  add(childActions, DISABLE_ALL_BUTTON, #cmdEditSetAll);

  /* Create Breakpoint List */
  if not(bkptLB := new(BkptList, BKPT_LB, self)) then
    ^nil;
  endif;
  /* #seletBkpt is the events processor for bkptList object */
  add(childActions, BKPT_LB, #selectBkpt);
  resizeChilds(self);
}
!!

/* 6/12/1992 15:27 - PRIVATE */
Def initMainMenu(self | aPopup)
{break("CCC");
  menu := create(new(Menu), self);

  /* FILE menu */
  aPopup := newPopup(MenuItem, "&File");
  addItem(aPopup, new(MenuItem, "E&xit",       
      BKPT_FILE_EXIT, #cmdFileExit));
  addItem(menu, aPopup);

  /* BREAKPOINTS menu */
  aPopup := newPopup(MenuItem, "&Breakpoints");
  addItem(aPopup, new(MenuItem, "Set &Breakpoint...",       
      BKPT_EDIT_ADD, #cmdEditSet));
  addItem(aPopup, nil); 
  addItem(aPopup, new(MenuItem, "&Clear",
      BKPT_EDIT_DEL, #cmdEditClear)); 
  addItem(aPopup, new(MenuItem, "&Enable",
      BKPT_EDIT_ENABLE, #cmdEditEnable)); 
  addItem(aPopup, new(MenuItem, "&Disable",
      BKPT_EDIT_DISABLE, #cmdEditDisable));       
  addItem(aPopup, nil);
  addItem(aPopup, new(MenuItem, "Clear &All",
      BKPT_EDIT_DEL_ALL, #cmdEditDeleteAll));                  
  addItem(aPopup, new(MenuItem, "E&nable All",
      BKPT_EDIT_ENABLE_ALL, #cmdEditSetAll));
  addItem(aPopup, new(MenuItem, "D&isable All",
      BKPT_EDIT_DISABLE_ALL, #cmdEditSetAll));
  addItem(aPopup, nil);     
  addItem(aPopup, new(MenuItem, "&Go To Source",       
      BKPT_EDIT_INSPECT, #cmdEditGotoSource));
  addItem(menu, aPopup);
 
  /* Append Windows menu item */
  Call AppendMenu(handle(menu), MF_POPUP, windowsMenu(PreLauncher), asciiz("&Windows")); 
  addItem(menu, initMenuHelp(self));
  drawMenu(self);
}
!!

/* Initialize the main window of the Presenter within the screen */
Def initMainWindow(self | x, y, width)
{
  /* do nothing; window position either restored from pwrviews.ini,
     or set by openWithPosAndState. */
}!!

/* 6/3/1992 15:25 - PUBLIC */
Def initMenuHelp(self | aMenu)
{ 
  /* HELP menu */
  aMenu := newPopup(MenuItem, "&Help");
  addItem(aMenu, new(MenuItem, "Help &Index",          
    INDEX_HELP, #cmdHelp));
/*
  addItem(aMenu, new(MenuItem, "&Keyboard",          
    KEYBOARD_HELP, #cmdHelp));
  addItem(aMenu, new(MenuItem, "&Commands",          
    COMMANDS_HELP, #cmdHelp));
  addItem(aMenu, new(MenuItem, "&Procedures",          
    PROCEDURES_HELP, #cmdHelp));
  addItem(aMenu, new(MenuItem, "&Glossary",          
    GLOSSARY_HELP, #cmdHelp)); 
*/    
  addItem(aMenu, new(MenuItem, "&Help With Help",          
    USING_HELP, #cmdHelp));
  addItem(aMenu, nil);
  addItem(aMenu, new(MenuItem, "Help With &Breakpoint",          
    ABOUT_HELP, #cmdHelp));
  ^aMenu;
}
!!

/* 6/18/1992 8:55 - Actor
  Paint self.
*/
Def paint(self, hdc)
{ 
  paint(self:ancestor, hdc);
  drawBkptLabels(self);
}
!!

/* 6/15/1992 17:01 - Actor */
Def reSize(self, wp, lp)
{ 
  if not(childDict[ADD_BUTTON]) then ^nil; endif; 
  resizeChilds(self);
}
!!

/* 6/15/1992 16:31 - PRIVATE 
  Respond to resize message - recalculate child position.
*/
Def resizeChilds(self | bSize, bHeight, width, height, 
  maxRight, numButton, cr)
{ 
  cr       := clientRect(self);  
  maxRight := right(cr);
  width    := maxRight - left(cr);
  height   := height(cr);
  bSize    := max(asInt(width/7), minButtonWidth);
  bHeight  := tmHeight + 4;
 
  /* Set position of execution button - child objects */
  setCRect(at(childDict, ADD_BUTTON),
    rect(0, 0, bSize, bHeight));
  setCRect(at(childDict, DEL_BUTTON),
    rect(bSize+1,  0, bSize*2, bHeight));
  setCRect(at(childDict, INSPECT_BUTTON),
    rect(bSize*2+1, 0, bSize*3, bHeight));
  setCRect(at(childDict, ENABLE_BUTTON),
    rect(bSize*3+1,0,bSize*4, bHeight));
  setCRect(at(childDict, DISABLE_BUTTON),
    rect(bSize*4+1,0,bSize*5, bHeight));
  setCRect(at(childDict, ENABLE_ALL_BUTTON),
    rect(bSize*5+1,0,bSize*6, bHeight));
  setCRect(at(childDict, DISABLE_ALL_BUTTON),
    rect(bSize*6+1,0, bSize*7,  bHeight));
 
  /* Breakpoint List box */  
  setCRect(bkptLB,
    rect(10, bHeight+20, (width-10), (height-6)));
  /* Set tab in bkptList */
  setLBStringTabs(bkptLB, tmWidth);
  
  /* Move buttons to new positions accordingly - Hide buttons that does not fit */
  numButton := 1;
  maxRight := maxRight-4; /* offset the windows border */
  do (childDict,
     {using(childId | crButton) 
        moveWindow(childId);
        crButton := clientRect(childId); 
        if ((right(crButton)*numButton) >= maxRight) then
          show(childId, SW_HIDE); /* Do not show the one overlap the splitButton */
        else
          show(childId, SW_SHOW); /* Show the buttons that visible */
        endif;  
        numButton := numButton+1;
  });    
  
  moveWindow(bkptLB);
  invalidate(self);  
}
!!

/* 6/16/1992 9:45 - PRIVATE
  Actions of the BkptLB child.
*/
Def selectBkpt(self, wp, lp)
{ 
  /* Set the current breakpoint if there is a selection from the listbox */
  if (high(lp) = LBN_SELCHANGE) then 
    currentSel := getSelIdx(bkptLB);
    getSelBkpt(bkptLB);
  endif;        
}
!!

/* 6/15/1992 16:58 - Actor
  show all child.
*/
Def show(self, displayMode)
{ 
  show(self:ancestor, displayMode);
  do(childDict, 
    {using(childItem) 
      show(childItem, SW_NORMAL);
    });
  show(bkptLB, SW_NORMAL);  
}
!!

/* 8/11/1992 13:18 - PRIVATE
  Return the tmWidth of self.
*/
Def tmWidth(self)
{ 
  ^tmWidth; 
}
!!

/* 6/12/1992 15:16 - WINDOWS 
  Sent by event manager (evNoteLib) 
*/
Def WM_EVENT(self, ignored, event)
{
  if event cand at(eventRespSet, event) then    
    perform(self, at(eventRespSet, event)); 
  else  
    /* Windows went nuts */
    displayFormattedError(ErrorTextLibClass$Inst, 
       ER_UNKNOWN_EVENT, FORCE_POPUP, asciiz(asHex(event)), nil, nil);
  endif;
  invalidate(self);
  ^0
} 
!!

/* 7/9/1992 12:40 - WINDOWS
  Initialize Windows dynamic menu.
*/
Def WM_INITMENU(self, wp, lp)
{ 
  ^WM_INITMENU(PreLauncher, wp, lp);
}
!!
