/****************************************************************************
**
** Name: trctman.c
**
** Description:
**   Services for trigger and trace operation
**
**   physical frame             physical Trace Buffer ID
**  ================    _____  ==========================
**   frame        0 -> |_____|
**   frame        1 -> |_____|  Trace Buffer 0
**   frame        2 -> |_____|
**     :               |  :  |__________________
**     :               |  :  |
**     :               |  :  |  Trace Buffer 1
**     :               |  :  |
**     :               |  :  |__________________
**     :               |  :  |
**     :               |  :  |        :
**     :               |  :  |        :
**     :               |  :  |__________________
**     :               |  :  |
**     :               |  :  |  Trace Buffer J (depend on trace config)
**     :               |__:__|
**   frame 128*1024 -> |_____|
**
** Status: PRELIMINARY
**
** $Log:   S:/tbird/arcm302/mpfw_302/trctman.c_v  $
** 
**    Rev 1.0   10 Jun 1996 08:51:16   gene
** Initial revision.
** 
**    Rev 1.3   13 Feb 1996 08:43:20   gene
** fixed ListTraceBuffer(), trace window list wrong trace frame
** 
**    Rev 1.2   26 Jan 1996 15:32:52   gene
** fixed FilterEMON() for BUFFER_EMPTY
** 
**    Rev 1.1   22 Nov 1995 10:58:32   kevin
** fixed a bug while the end frame is on 128 boundary, it will lost 128 frames.
** 
**    Rev 1.0   07 Sep 1995 10:47:46   gene
** Initial revision.
**
** $Header:   S:/tbird/arcm302/mpfw_302/trctman.c_v   1.0   10 Jun 1996 08:51:16   gene  $
**
** Copyright (C) 1992 Microtek International. All rights reserved.
**
****************************************************************************/

                       /****************************
                        *                          *
                        *      INCLUDE FILES       *
                        *                          *
                        ****************************/

#include <string.h>
#include <conio.h>

#ifndef _EMU_ERROR_
#include "emuerror.h"
#endif

#ifndef _TRCERROR_
#include "trcerror.h"
#endif
//
//#ifndef _TRCDIAG_
//#include "trcdiag.h"
//#endif
//
#ifndef _TRCXLX_
#include "trcxlx.h"
#endif

#ifndef _TRCTMAN_
#include "trctman.h"
#endif

#ifndef _TRCDTST_
#include "trclib.h"
#endif

                       /****************************
                        *                          *
                        *    LOCAL DEFINITIONS     *
                        *                          *
                        ****************************/

#define MK_FP(seg, ofs) ((VOID _far *) ((((U32) seg) << 16) | ((U16) ofs)))

#define MAX_FRAME_WIDTH 16
#define FRAME_WIDTH 16
#define LIST_BUFFER_SIZE 1024
#define MAX_TRACE_BUFFER_NUMBER 256
#define FRAME_NUMBER_PER_SUB_BUFFER 128
#define TOTAL_SUB_BUFFER_NUMBER 1024
#define TOTAL_FRAME_NUMBER  128 * 1024
#define NUMBER_OF_SBA_BIT 10
#define NUMBER_OF_PFC_BIT 9
#define MAX_TERMINAL_CNT 1024

#define offDATA     0x00
#define offSTAT     0x04
#define offTMST     0x06
#define offSPAR     0x07
#define offADDR     0x08
#define offPORT     0x0C

enum QUALIFY_STATUS {QUALIFY_STATUS1,QUALIFY_STATUS2};

#define pSTATRD     0xF346
#define pPSEUWR     0xF34C
#define pPFCREAD    0xF348
#define pSTARTWR    0xF348
#define pCLRSB      0xF340
#define pSBAREAD    0xF340
#define pSLRESWR    0xF34A
#define pCLRTPIWR   0xF342
#define pTPIRD      0xF342
#define pCPUTRIGHWR 0xF34E
#define pCTR0RD     0xF34A
#define pCTR1RD     0xF34C

#define pSYSPB      0xF141
#define bitSEL0     0x01
#define bitSEL1     0x02
#define bitSEL2     0x04
#define bitCHGPHASE 0x40

#define bitNXRST1   0x01
#define bitNXRST2   0x02
#define bitNSE      0x80

#define TMP_OFFSET    0x06  /* TMP(time-stamp) port located at byte 6 */
#define TMP7_BIT      0x80
#define TBFILL_LOC    0x02  /* TBFILL located at TMP7 of frame2 */
#define FILL_LOC      0x03  /* FILL   located at TMP7 of frame3 */
#define TPI7_LOC      0x04  /* TPI7   located at TMP7 of frame4 */
#define TPI6_LOC      0x05  /* TPI6   located at TMP7 of frame5 */
#define TPI5_LOC      0x06  /* TPI5   located at TMP7 of frame6 */
#define TPI4_LOC      0x07  /* TPI4   located at TMP7 of frame7 */
#define TPI3_LOC      0x08  /* TPI3   located at TMP7 of frame8 */
#define TPI2_LOC      0x09  /* TPI2   located at TMP7 of frame9 */
#define TPI1_LOC      0x0A  /* TPI1   located at TMP7 of frameA */
#define TPI0_LOC      0x0B  /* TPI0   located at TMP7 of frameB */
#define TPIRUN_LOC    0x0D  /* TPIRUN located at TMP7 of frameD */
#define TBFILL_BIT    0x20
#define TBFILL_VAL    0x10  /* TBFILL valid at frame10 of each Sub-buffer */
#define TPIRUN_BIT    0x10
#define TPIRUN_VAL    0x7F  /* TPIRUN valid at frame10 of each Sub-buffer */

#define EMON_OFFSET 0x0F  /* EMON located at byte F */
#define EMON_BIT    0x08

#define EP_STATUS_SIZE 8

#define EP_STATUS_NUM 7   /* SP, UP, SR, SW, UR, UW, AK */

#if (EP_STATUS_SIZE <= 8)
   #define EP_STATUS_SIG U8
#elif (EP_STATUS_SIZE <= 16)
   #define EP_STATUS_SIG U16
#elif (EP_STATUS_SIZE <= 32)
   #define EP_STATUS_SIG U32
#endif

static TRACE_FORM currentTraceMode = CYCLE_TRACE;
static U16 traceBufferNumber = 1;
static U8 numberOfBitOfTBN = 0;
static TRIG_LOC triggerLocation = POST_TRIG;
static BOOLEAN breakOnFull = YES;

static BOOLEAN fILLAfterTraceHalt;
static BOOLEAN hALTAfterTraceHalt;
static U16 sBAAfterTraceHalt;
static U16 tBNAfterTraceHalt;

static U8 bankLo;
static U8 bankHi;
static U16 addrLo;
static U16 addrHi;

static U32 trigPhyFrameOfTrcBuf[MAX_TRACE_BUFFER_NUMBER];

static U8 numOfValidFrameInSubBuf[TOTAL_SUB_BUFFER_NUMBER];

static BOOLEAN isTrcBufContentValid;

                       /****************************
                        *                          *
                        *    EXTERNAL VARIABLES    *
                        *                          *
                        ****************************/

U16 counter0 = 0;
U16 counter1 = 0;

BOOLEAN gotTrigInfoOfTrcBuf[MAX_TRACE_BUFFER_NUMBER];
S32 startLogFrameOfTrcBuf[MAX_TRACE_BUFFER_NUMBER];
S32 endLogFrameOfTrcBuf[MAX_TRACE_BUFFER_NUMBER];

extern U8 bitStreamBuffer1[];
extern U8 bitStreamBuffer2[];
extern U8 _far EREC1XCByte[];
extern U8 _far EREC2XCByte[];
extern U8 _far TMANStaticXCByte[];
extern U8 _far TMANTraceXCByte[];
extern U8 mpNXCONFIG;

extern BOOLEAN mtatExist;

                       /****************************
                        *                          *
                        *     LOCAL PROTOTYPES     *
                        *                          *
                        ****************************/

U16 IsPreTBTrigAtOddSubBuf(U16 trcBufPhyID);
U8 GetTPI(U16 addr, U8 *shift);
VOID GetBankAddrOfTrcBuf(U16 trcBufPhyID, U8 *bankL, U8 *bankH, U16 *addrL,
                         U16 *addrH);
RETCODE GetTrcBufPhyID(U16 trcBufLogID, U16 *trcBufPhyID, BOOLEAN *isEndTB);
RETCODE FilterEMON(U16 trcBufLogID, U32 topPhyFrameOfTrcBuf,
                   U32 bottomPhyFrameOfTrcBuf,BOOLEAN trcBufFill);
BOOLEAN SearchTriggerPoint(U16 trcBufLogID, U16 addrStart, U16 addrEnd,
                           U8 bankEnd, U32 *topPhyFrameOfTrigSubBuf);
RETCODE GetInfoOfMidTB(U16 trcBufLogID, U16 trcBufPhyID);
RETCODE GetInfoOfEndTB(U16 trcBufLogID, U16 trcBufPhyID);
RETCODE SetCounters(VOID);
RETCODE SetNumberOfTraceBuffer(VOID);
RETCODE SetTriggerLocation(VOID);
RETCODE DumpDRAM(U8 bankS, U16 addrS, U16 length, VOID *buffer, U16 *frameNum,
                 S32 logFrame, S32 endLogFrameOfTB);
RETCODE QualifyDumpDRAM(U8 bankS, U16 addrS, U16 length, VOID *buffer,
                        U16 *frameNum, S32 logFrame, S32 endLogFrameOfTB,
                        QUALIFY_LIST *qualify);

                       /****************************
                        *                          *
                        *     EXECUTABLE CODE      *
                        *                          *
                        ****************************/

/******************************************************************************
**
**  ConfigTrace
**
*******************************************************************************/
RETCODE ConfigTrace(U16 traceBufNum, TRIG_LOC trigLoc, BOOLEAN breakFull) {
   U16 temp = traceBufNum;

   if (temp > (U16) MAX_TRACE_BUFFER_NUMBER) return(ERR_UNKNOW_TBN);
   numberOfBitOfTBN = 0;
   while (temp != 1) {
      if (temp % 2 != 0) return(ERR_UNKNOW_TBN);
      numberOfBitOfTBN++;
      temp /= 2;
   }
   traceBufferNumber = traceBufNum;
   triggerLocation = trigLoc;
   breakOnFull = breakFull;
   isTrcBufContentValid = FALSE;
   return(GOOD);
}

/******************************************************************************
**
**  GetTrigInfoOfTrcBuf
**
*******************************************************************************/
RETCODE GetTrigInfoOfTrcBuf(U16 trcBufLogID) {
   RETCODE err;
   BOOLEAN isEndTrcBuf;
   U16 trcBufPhyID;

   if ((err = GetTrcBufPhyID(trcBufLogID, &trcBufPhyID, &isEndTrcBuf)) != GOOD)
      return(err);
   if ((err = isEndTrcBuf ? GetInfoOfEndTB(trcBufLogID, trcBufPhyID)
              : GetInfoOfMidTB(trcBufLogID, trcBufPhyID)) != GOOD) return(err);
   return(GOOD);
}

/******************************************************************************
**
**  InitialTraceMachine
**
*******************************************************************************/
RETCODE InitialTraceMachine(EMU_MODE emuMode) {
   RETCODE err;
   U16 xlxExist, haltTB;
   U32 cnt;
   U8 tempNXCONFIG;

   isTrcBufContentValid = FALSE;
   currentTraceMode = CYCLE_TRACE;
   counter0 = 0;
   counter1 = 0;
   traceBufferNumber = 1;
   numberOfBitOfTBN = 0;
   triggerLocation = POST_TRIG;
   breakOnFull = YES;

   outp(pSYSPB, (inp(pSYSPB) & ~(bitSEL2 | bitCHGPHASE)) | bitSEL1 | bitSEL0 );
// outp(pSYSPB, (inp(pSYSPB) & ~(bitSEL2 | bitSEL1)) | bitCHGPHASE | bitSEL0 );
// outp(pSYSPB, (inp(pSYSPB) & ~(bitSEL2)) | bitSEL1 | bitCHGPHASE | bitSEL0 );
   CheckExist(&mtatExist, &xlxExist);
   if (mtatExist == NO) return(GOOD);
   tempNXCONFIG = mpNXCONFIG;
   if (xlxExist & 0x0100) {
      mpNXCONFIG &= ~(bitNXRST1 | bitNXRST2);
      if ((err = ProgramTMANStatic(&haltTB, &cnt)) != GOOD) return(err);
      Delay(4);  /* Wait for performing a minimum of 8 /CAS-before-/RAS and
                    8 SC dummy cycles to stabilize the VRAM internal ckt */
      outp(pPSEUWR, 0);
                 /* Perform a pseudo write cycle to set the SAM
                    port into input mode and TAP to 0 */
   }
   switch (xlxExist) {
      case 0x0111:
         fmemcpy(bitStreamBuffer1, EREC1XCByte, (U16) XC3090SIZE);
         mpNXCONFIG |= bitNXRST1;
         if ((err = XlxReprogram(XLX_EREC1)) != GOOD) return(err);
         if (emuMode == MODE_32BIT)
            fmemcpy(bitStreamBuffer2, EREC2XCByte, (U16) XC3090SIZE);
         else fmemcpy(bitStreamBuffer2, EREC2XCByte, (U16) XC3090SIZE);
         mpNXCONFIG = tempNXCONFIG;
         if ((err = XlxReprogram(XLX_EREC2)) != GOOD) return(err);
         break;
      case 0x0110:
         fmemcpy(bitStreamBuffer1, EREC1XCByte, (U16) XC3090SIZE);
         mpNXCONFIG |= bitNXRST1;
         if ((err = XlxReprogram(XLX_EREC1)) != GOOD) return(err);
         mpNXCONFIG = tempNXCONFIG;
         if (emuMode == MODE_32BIT) return(ERR_MODE_32BIT);
            /* not support 32 bit mode */
         break;
      case 0x0100: case 0x0101:
      case 0x0000: case 0x0001: case 0x0010: case 0x0011:
         fmemcpy(bitStreamBuffer1, EREC1XCByte, (U16) XC3090SIZE);
         fmemcpy(bitStreamBuffer2, EREC2XCByte, (U16) XC3090SIZE);
         mpNXCONFIG |= bitNXRST1;
         if (((xlxExist & 0x0010) != 0) &&
             ((err = XlxReprogram(XLX_EREC1)) != GOOD)) return(err);
         mpNXCONFIG = tempNXCONFIG;
         if (((xlxExist & 0x0001) != 0) &&
             ((err = XlxReprogram(XLX_EREC2)) != GOOD)) return(err);
         return(ERR_MTAT_HW);  /* MTAT not work */
   }
   return(GOOD);
}

/******************************************************************************
**
**  IsTMANBreak
**
*******************************************************************************/
BOOLEAN IsTMANBreak(VOID) {
   return((BOOLEAN) (inp(pSTATRD) >> 7));
}

/******************************************************************************
**
**  ListTraceBuffer
**
*******************************************************************************/
RETCODE ListTraceBuffer(TRACE_HEAD *log, QUALIFY_LIST *qualify,
                        VOID *frameContents) {
   RETCODE err;
   U32 topPhyFrameOfTrcBuf = (((U32) bankLo) << 12) | (((U32) addrLo) >> 4);
   U32 bottomPhyFrameOfTrcBuf = (((U32) bankHi) << 12) | (((U32) addrHi) >> 4);
   S32 frameCount = 0;
   U16 trcBufPhyID, subBufPtr, addr;
   U8 _far *data, bank;
   BOOLEAN isEndTrcBuf;

   outp(pSEQENBWR, bitACTDIS);
   if ((err = GetTrcBufPhyID(log->traceBufId, &trcBufPhyID, &isEndTrcBuf)) != GOOD)
      return(err);
   switch (currentTraceMode) {
      case CLOCK_TRACE:
         switch (log->traceform) {
            case CLOCK_LIST:
               return(err);  /* not support yet */
               break;
            case CYCLE_LIST:
               return(err);  /* not support yet */
               break;
         }
         break;
      case CYCLE_TRACE:
         switch (log->traceform) {
            case CLOCK_LIST:
               return(ERR_LIST_MODE);
               /* when the tracing clock is EP cycle, it is impossible to list
                  information in EP clock */
               break;
            case CYCLE_LIST:
               if ((gotTrigInfoOfTrcBuf[log->traceBufId] == NO) &&
                   ((err = isEndTrcBuf ?
                           GetInfoOfEndTB(log->traceBufId, trcBufPhyID) :
                           GetInfoOfMidTB(log->traceBufId, trcBufPhyID)) != GOOD))
                  return(err);
               if (log->startFrame == 0) {
                  bank = (U8) ((U32) trigPhyFrameOfTrcBuf[log->traceBufId] >> 12);
                  addr = (U16) (((trigPhyFrameOfTrcBuf[log->traceBufId] & 0x0FFF)
                                << 4)+ EMON_OFFSET);
               }
               else {
                  bank = (U8) (trigPhyFrameOfTrcBuf[log->traceBufId] >> 12);
                  addr = (U16) ((U16) (trigPhyFrameOfTrcBuf[log->traceBufId] << 4)
                               + EMON_OFFSET);
                  outp(pBANKWR, bank << 2);
                  switch (log->startFrame > 0) {
                     case YES:
                        do {
                           addr += MAX_FRAME_WIDTH;
                           data = (U8 _far *) MK_FP(TMAN_SEG, addr);
                           if (((*data) & EMON_BIT) != 0) frameCount++;
                        } while ((frameCount != log->startFrame) &&
                                ((addr & 0x07F0) != 0x07F0));
                        if (frameCount == log->startFrame) break;
                        subBufPtr = (U16) (trigPhyFrameOfTrcBuf[log->traceBufId]
                                           >> 7);
                        while (frameCount < log->startFrame) {
                           if (subBufPtr == (U16) (bottomPhyFrameOfTrcBuf >> 7))
                              subBufPtr = (U16) (topPhyFrameOfTrcBuf >> 7);
                           else subBufPtr++;
                           frameCount += numOfValidFrameInSubBuf[subBufPtr];
                           addr += MAX_FRAME_WIDTH * numOfValidFrameInSubBuf[subBufPtr];
                        }
                        if (frameCount > log->startFrame) {
                           bank = (U8) (subBufPtr >> 5);
                           outp(pBANKWR, bank << 2);
                           addr = (U16) (subBufPtr << 11) + (U16) EMON_OFFSET +
                                  (U16) ((FRAME_NUMBER_PER_SUB_BUFFER - 1) *
                                          MAX_FRAME_WIDTH);

                           while (frameCount != log->startFrame) {
                              data = (U8 _far *) MK_FP(TMAN_SEG, addr);
                              if (((*data) & EMON_BIT) != 0) frameCount--;
                              addr -= MAX_FRAME_WIDTH;
                           }
                        }
                        break;
                     case NO:
                        while ((frameCount != log->startFrame) &&
                                ((addr & 0x07F0) != 0)) {
                           addr -= MAX_FRAME_WIDTH;
                           data = (U8 _far *) MK_FP(TMAN_SEG, addr);
                           if (((*data) & EMON_BIT) != 0) frameCount--;
                        }
                        if (frameCount == log->startFrame) break;
                        subBufPtr = (U16) (trigPhyFrameOfTrcBuf[log->traceBufId]
                                           >> 7);
                        while (frameCount > log->startFrame) {
                           if (subBufPtr == (U16) (topPhyFrameOfTrcBuf >> 7))
                              subBufPtr = (U16) (bottomPhyFrameOfTrcBuf >> 7);
                           else subBufPtr--;
                           frameCount -= numOfValidFrameInSubBuf[subBufPtr];
                           addr -= MAX_FRAME_WIDTH * numOfValidFrameInSubBuf[subBufPtr];
                        }
                        if (frameCount < log->startFrame) {
                           bank = (U8) (subBufPtr >> 5);
                           outp(pBANKWR, bank << 2);
                           addr = (U16) (subBufPtr << 11) + (U16) EMON_OFFSET;

                           while (frameCount != log->startFrame) {
                              data = (U8 _far *) MK_FP(TMAN_SEG, addr);
                              if (((*data) & EMON_BIT) != 0) frameCount++;
                              addr += MAX_FRAME_WIDTH;
                           }
                        }
                        break;
                  }
               }
               switch (qualify->qFlag) {
                  case NO:
                     if ((err = DumpDRAM(bank, addr - EMON_OFFSET,
                        LIST_BUFFER_SIZE, frameContents, &(log->frameLen),
                        log->startFrame, endLogFrameOfTrcBuf[log->traceBufId]))
                        != GOOD) return(err);
                     break;
                  case YES:
                     if ((err = QualifyDumpDRAM(bank, addr - EMON_OFFSET,
                        LIST_BUFFER_SIZE, frameContents, &(log->frameLen),
                        log->startFrame, endLogFrameOfTrcBuf[log->traceBufId],
                        qualify)) != GOOD) return(err);
                     break;
               }
               break;
         }
         break;
   }
   return(GOOD);
}

/******************************************************************************
**
**  ProgramTMANStatic
**
*******************************************************************************/
RETCODE ProgramTMANStatic(U16 *endTrcBufLogID, U32 *cntAfterTraceHalt) {
   RETCODE err;
   U16 i, cTR0AfterTraceHalt, cTR1AfterTraceHalt;
   U8 pFCAfterTraceHalt, portData;

   outp(pCPUTRIGHWR, 0);

   fILLAfterTraceHalt = (BOOLEAN) ((inp(pSBAREAD + 1) >> 4) & 0x01);
   hALTAfterTraceHalt = (BOOLEAN) ((inp(pSTATRD) & 0x08) >> 3);
   sBAAfterTraceHalt = ((U16) inp(pSBAREAD)) |
                       (((U16) (inp(pSBAREAD + 1) >> 6)) << 8);
   tBNAfterTraceHalt = sBAAfterTraceHalt >> (10 - numberOfBitOfTBN);
   if (hALTAfterTraceHalt && (tBNAfterTraceHalt != 0)) tBNAfterTraceHalt--;
   *endTrcBufLogID = fILLAfterTraceHalt ?
                     (traceBufferNumber - 1) : tBNAfterTraceHalt;

   cTR0AfterTraceHalt = ((U16) inp(pCTR0RD)) |
                        (((U16) (inp(pCTR0RD + 1) >> 6)) << 8);
   cTR1AfterTraceHalt = ((U16) inp(pCTR1RD)) |
                        (((U16) (inp(pCTR1RD + 1) >> 6)) << 8);
   *cntAfterTraceHalt = (U32) (((U32) cTR1AfterTraceHalt) << 16) |
                              ((U32) cTR0AfterTraceHalt);

   pFCAfterTraceHalt = (U8) inp(pPFCREAD);  /* force DONE = 1 */
   portData = (U8) inp(pSYSPB);
   outp(pSYSPB, portData & ((U8) (~(bitSEL2 | bitSEL1 | bitSEL0))));
   for (i = 0; i < 128; i++); /* delay */
   outp(pSYSPB, portData);

   fmemcpy(bitStreamBuffer1, TMANStaticXCByte, (U16) XC3064SIZE);
   if ((err = XlxReprogram(XLX_TMAN)) != GOOD) return(err);
   for (i = 0; i < MAX_TRACE_BUFFER_NUMBER; i++) gotTrigInfoOfTrcBuf[i] = NO;
   for (i = 0; i < TOTAL_SUB_BUFFER_NUMBER; i++)
      numOfValidFrameInSubBuf[i] = 0xFF;
   outp(pNXCONFIG, mpNXCONFIG |= bitNSE);
   return(GOOD);
}

/******************************************************************************
**
**  ProgramTMANTrace
**
*******************************************************************************/
RETCODE ProgramTMANTrace(VOID) {
   RETCODE err;
   static const U16 _based(_segname("_CODE")) clbOfEMBRK = 0x0E01;

   fmemcpy(bitStreamBuffer1, TMANTraceXCByte, (U16) XC3064SIZE);
   if ((err = SetCounters()) != GOOD) return(err);
   if ((err = SetNumberOfTraceBuffer()) != GOOD) return(err);
   if ((err = SetTriggerLocation()) != GOOD) return(err);
   if (breakOnFull == NO) {
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfEMBRK, XLX_G, 0x0A, 0x0A, 0))
              != GOOD) return(err);  /* addr = (B,_,A,_) clear G = A*B */
   }
   if ((err = XlxReprogram(XLX_TMAN)) != GOOD) return(err);
   Delay(4);  /* Wait for performing a minimum of 8 /CAS-before-/RAS and
                 8 SC dummy cycles to stabilize the VRAM internal ckt */
   outp(pPSEUWR, 0);
              /* Perform a pseudo write cycle to set the SAM
                 port into input mode and TAP to 0 */
   outp(pSLRESWR, 0);    /* Clear Sequential RAM Address Port Latch */
   outp(pCLRTPIWR, 0);   /* Clear TPI counter */
   outp(pCLRSB, 0);      /* Clear SBA counter */
   outp(pSTARTWR, 0);    /* Set RUNEV and TPIRUN */
   outp(pNXCONFIG, mpNXCONFIG &= ~bitNSE);
   isTrcBufContentValid = TRUE;
   return(GOOD);
}

/******************************************************************************
**
**  WriteSeqRAMs
**
*******************************************************************************/
RETCODE WriteSeqRAMs(U8 levelID, U8 *data) {
   RETCODE err;
   U8 _far *seqRAM0 = MK_FP(TMAN_SEG, (U16) (((levelID ^ 0x03) << 11) |
                                             SEQ_RAM0_OFFSET));
   U8 _far *seqRAM1 = MK_FP(TMAN_SEG, (U16) (((levelID ^ 0x03) << 11) |
                                             SEQ_RAM1_OFFSET));

   outp(pSEQENBWR, bitSEQENB);
   fmemcpy(seqRAM0, data, SEQ_RAM_SIZE >> 2);
   if ((err = fmemcmp(seqRAM0, data, SEQ_RAM_SIZE >> 2)) != GOOD)
      return(ERR_SEQ_RAM_WRITE);
   data += SEQ_RAM_SIZE >> 2;
   fmemcpy(seqRAM1, data, SEQ_RAM_SIZE >> 2);
   if ((err = fmemcmp(seqRAM1, data, SEQ_RAM_SIZE >> 2)) != GOOD)
      return(ERR_SEQ_RAM_WRITE);
   return(GOOD);
}

/******************************************************************************
**
**  IsPreTBTrigAtOddSubBuf
**
**  Description:
**    Check "Is previous trace buffer trigger at odd sub-buffer ?"
**    If it is true  -> return 1   ; else -> return 0
**
**                EVEN(QSF=0)   ODD(QSF=1)
**                 ||            ||
**            _____\/____________\/______
**   trace   |_____________|_____________|   If the trace memory is configured
**   buffer  |_____________|_____________|   to 128 trace buffers, then there
**     N     |_____________|___trigger___|   are 8 sub-buffers in each trace
**           |_____________|_____________|   buffer.
**
**           Trace buffer N trigger at odd sub-buffer.
**
**  Parameters:
**    input:
**      trcBufPhyID: current trace buffer physical ID
**    output:
**      <none>
**
*******************************************************************************/
U16 IsPreTBTrigAtOddSubBuf(U16 trcBufPhyID) {
   const U16 SKIP = FRAME_NUMBER_PER_SUB_BUFFER * MAX_FRAME_WIDTH;
   U16 addr, preAddrLo, preAddrHi, preTrcBufPhyID;
   U8  bank, preBankLo, preBankHi, _far *data, tpi, tpiShift, frameID;
   BOOLEAN hiTPIRUN = FALSE, condition2Match;

   switch (fILLAfterTraceHalt) {
      case NO:
         switch (trcBufPhyID) {
            case 0:
               return(0);  break;
            default:
               preTrcBufPhyID = trcBufPhyID - 1;  break;
         }
         break;
      case YES:
         if (trcBufPhyID == (tBNAfterTraceHalt + 1) % traceBufferNumber)
             return(1);
         switch (trcBufPhyID) {
            case 0:
               preTrcBufPhyID = traceBufferNumber - 1;  break;
            default:
               preTrcBufPhyID = trcBufPhyID - 1;  break;
         }
         break;
   }
   GetBankAddrOfTrcBuf(preTrcBufPhyID, &preBankLo, &preBankHi, &preAddrLo,
                       &preAddrHi);
   bank = preBankLo;
   addr = preAddrLo;
   /* The trigger sub-buffer must match the following two conditions :
        (1) TPIRUN changed from 1 to 0 in the trigger sub-buffer
        (2) all frames of the sub-buffer next to it must be "TPIRUN = 0" */
   do {
      outp(pBANKWR, bank << 2);
      do {
         data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET);
         switch (((U8) (*data)) & TPIRUN_BIT) {
            case 0:
               if (hiTPIRUN == FALSE) break;
               for (frameID = 0, condition2Match = YES;
                    (condition2Match == YES) &&
                    (frameID < FRAME_NUMBER_PER_SUB_BUFFER);
                    frameID++) {
                  data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET +
                                           frameID * MAX_FRAME_WIDTH);
                  if (((U8) (*data)) & TPIRUN_BIT) condition2Match = NO;
               }
               if (condition2Match) {
                  if (addr == addrLo) {
                     addr = addrHi - SKIP + 1;
                     bank = (bank == bankLo) ? (U8) bankHi : (U8) (bank - 1);
                     outp(pBANKWR, bank << 2);
                  }
                  else addr -= SKIP;
                  if ((tpi = GetTPI(addr, &tpiShift)) < tpiShift) addr -= SKIP;
                  return((addr >> 11) & 0x0001);
               }
               break;
            default:
               if (hiTPIRUN == FALSE) hiTPIRUN = TRUE;
               break;
         }
         addr += SKIP;
      } while (addr != preAddrHi + 1);
      addr = preAddrLo;
      bank++;
   } while (bank <= preBankHi);
   if (hiTPIRUN == TRUE) return(1);
}

/******************************************************************************
**
**  GetTPI
**
**  Description:
**    1. Get TPI of trace buffer from trigger sub-buffer and return TPI value
**    2. Get the number of frame from the trigger frame to the frame with
**       TPIRUN changes from 1 to 0 (i.e. get *shift)
**
**    Example : in the following case, *shift = 3
**
**                        |_________|   TPIRUN
**                        |_________|     1
**     trigger frame ---> |___t_____|     1
**                        |___t+1___|     1
**                        |___t+2___|     1
**                        |___t+3___|     0  <--- frame with TPIRUN changes
**                        |_________|     0                  from 1 to 0
**                        |         |     0
**
**  Parameters:
**    input:
**      addr: the DRAM access address of the first frame of trigger sub-buffer
**    output:
**      *shift: the number of frame from the trigger frame to the frame with
**              TPIRUN changes from 1 to 0
**
*******************************************************************************/
U8 GetTPI(U16 addr, U8 *shift) {
   U8 _far *data;
   U8 tpi;

   do {
      data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET);
      addr += MAX_FRAME_WIDTH;
   } while ((((U8) (*data)) & TPIRUN_BIT) != 0);
   *shift = (U8) (((U8) (*data)) >> 6);
   tpi = (U8) (((addr - MAX_FRAME_WIDTH) >> 4) & 0x007F);
   return((U8) (tpi == 0 ? 0x80 : tpi));
}

/******************************************************************************
**
**  GetBankAddrOfTrcBuf
**
**  Description:
**    Get the <bank> and <addr> informations of trace buffer
**    so that the control processor can access the trace memory (DRAM)
**    according to these informations
**
**  Parameters:
**    input:
**      trcBufPhyID: current trace buffer physical ID
**    output:
**      *bankL: low bank of trace buffer
**      *bankH: high bank of trace buffer
**      *addrL: low addr of trace buffer
**      *addrH: high addr of trace buffer
**
*******************************************************************************/
VOID GetBankAddrOfTrcBuf(U16 trcBufPhyID, U8 *bankL, U8 *bankH, U16 *addrL,
                         U16 *addrH) {
   U8 bankR;
   U16 addrR;

   switch (traceBufferNumber) {
      case 1: case 2: case 4: case 8: case 16: case 32:
         bankR = (U8) (32 / traceBufferNumber);
         *bankL = bankR * (U8) trcBufPhyID;
         *bankH = *bankL + bankR - (U8) 1;
         *addrL = 0x0000;
         *addrH = 0xFFFF;
         break;
      case 64: case 128: case 256:
         *bankL = *bankH = (U8) trcBufPhyID / (U8) (traceBufferNumber / 32);
         addrR = (U16) ((U32) TOTAL_FRAME_NUMBER / (U32) traceBufferNumber)
                  * MAX_FRAME_WIDTH;
         *addrL = addrR * (U8) trcBufPhyID;
         *addrH = *addrL + addrR - 1;
	 break;
   }
}

/******************************************************************************
**
**  GetTrcBufPhyID
**
**  Description:
**    Get physical ID of trace buffer and check if it is the END trace buffer
**    (END trace buffer is the last trace buffer been used before trace halt)
**
**  Parameters:
**    input:
**      trcBufPhyID: current trace buffer physical ID
**    output:
**      *trcBufPhyID: physical ID of trace buffer trcBufLogID
**      *isEndTB: boolean flag to identify if trcae bbuffer trcBufLogID is
**                END trace buffer
**
*******************************************************************************/
RETCODE GetTrcBufPhyID(U16 trcBufLogID, U16 *trcBufPhyID, BOOLEAN *isEndTB) {

   if (trcBufLogID >= traceBufferNumber) return(ERR_UNKNOW_TB);
   if (!isTrcBufContentValid) return(BUFFER_EMPTY);
   if (traceBufferNumber == 1) *trcBufPhyID = trcBufLogID;
   else {
      outp(pBANKWR, 0);
      switch (fILLAfterTraceHalt) {
         case NO:  /* trace memory is not filled */
            *trcBufPhyID = trcBufLogID;
            if (*trcBufPhyID > tBNAfterTraceHalt)
               return(BUFFER_EMPTY);  /* this trace buffer is empty */
            break;
         case YES:  /* trace memory is filled */
            *trcBufPhyID = (trcBufLogID + tBNAfterTraceHalt + 1)
                           % traceBufferNumber;
            break;
      }
   }
   *isEndTB = (BOOLEAN) (hALTAfterTraceHalt ? NO :
                        (((*trcBufPhyID) == tBNAfterTraceHalt) ? YES : NO));
   GetBankAddrOfTrcBuf(*trcBufPhyID, &bankLo, &bankHi, &addrLo, &addrHi);
   return(GOOD);
}

/******************************************************************************
**
**  FilterEMON
**
**  Description:
**    Modify global variables "startLogFrameOfTrcBuf[trcBufLogID]" and
**    "endLogFrameOfTrcBuf[trcBufLogID]" according to EMON traced in each frames
**    Only those frames with EMON=1 make sense to the user and ignore the other
**    frames
**
**  Parameters:
**    input:
**      trcBufLogID: current trace buffer logical ID
**      topPhyFrameOfTrcBuf: current trace buffer top physical frame
**      bottomPhyFrameOfTrcBuf: current trace buffer bottom physical frame
**    output:
**      <none>
**
*******************************************************************************/
RETCODE FilterEMON(U16 trcBufLogID, U32 topPhyFrameOfTrcBuf,
                   U32 bottomPhyFrameOfTrcBuf,BOOLEAN trcBufFill) {
   U8 bankTrig = (U8) (trigPhyFrameOfTrcBuf[trcBufLogID] >> 12);
   U16 addrTrig = (U16) (trigPhyFrameOfTrcBuf[trcBufLogID] << 4);
   U8 bank, _far *data, validFrameInSubBuf;
   U16 addr, frameShift = 0;
   S32 frame, startLogFrame = 0, endLogFrame = 0;
   U32 phyFramePointer;

   outp(pBANKWR, (bank = bankTrig) << 2);
   data = (U8 _far *) MK_FP(TMAN_SEG, addrTrig + EMON_OFFSET);
   if (((*data) & EMON_BIT) == 0) {   /* Trigger Point EMON == 0 */
      phyFramePointer = trigPhyFrameOfTrcBuf[trcBufLogID];
      addr = addrTrig + EMON_OFFSET;
      do {
         frameShift++;
         if (phyFramePointer == topPhyFrameOfTrcBuf) {
            if (trcBufFill) {
               addr = ((addrHi - MAX_FRAME_WIDTH) + 1) + EMON_OFFSET;
               bank = bankHi;
               outp(pBANKWR, bank << 2);
               phyFramePointer = bottomPhyFrameOfTrcBuf;
             } else
               return(BUFFER_EMPTY);
         }
         else {
            phyFramePointer--;
            if (addr == 0x0000 + EMON_OFFSET) outp(pBANKWR, (--bank) << 2);
            addr -= MAX_FRAME_WIDTH;
         }
         data = (U8 _far *) MK_FP(TMAN_SEG, addr);
      } while ((*data & EMON_BIT) == 0);
      startLogFrameOfTrcBuf[trcBufLogID] += frameShift;
      endLogFrameOfTrcBuf[trcBufLogID] += frameShift;
      bankTrig = bank;
      addrTrig = addr & 0xFFF0;
      trigPhyFrameOfTrcBuf[trcBufLogID] = (((U32) bank) << 12) |
                                          (((U32) addr) >> 4);
   }
   phyFramePointer = trigPhyFrameOfTrcBuf[trcBufLogID];
   frame = 0;
   addr = addrTrig + EMON_OFFSET;
   while (((addr & 0x07F0) != 0) &&
         (frame != startLogFrameOfTrcBuf[trcBufLogID])) {
      addr -= MAX_FRAME_WIDTH;
      data = (U8 _far *) MK_FP(TMAN_SEG, addr);
      if (*data & EMON_BIT) startLogFrame--;
      phyFramePointer--;
      frame--;
   }
   validFrameInSubBuf = 0;
   while ((frame--) != startLogFrameOfTrcBuf[trcBufLogID]) {
      if (phyFramePointer == topPhyFrameOfTrcBuf) {
         if (trcBufFill) {
            addr = ((addrHi - MAX_FRAME_WIDTH) + 1) + EMON_OFFSET;
            bank = bankHi;
            outp(pBANKWR, bank << 2);
            phyFramePointer = bottomPhyFrameOfTrcBuf;
         } else
            break;
      }
      else {
         phyFramePointer--;
         if (addr == 0x0000 + EMON_OFFSET) outp(pBANKWR, (--bank) << 2);
         addr -= MAX_FRAME_WIDTH;
      }
      data = (U8 _far *) MK_FP(TMAN_SEG, addr);
      if (*data & EMON_BIT) {
         startLogFrame--;
         validFrameInSubBuf++;
      }
      if ((addr & 0x07F0) == 0) {
         numOfValidFrameInSubBuf[(U16) ((bank << 5) | (addr >> 11))]
            = validFrameInSubBuf;
         validFrameInSubBuf = 0;
      }
   }
   outp(pBANKWR, (bank = bankTrig) << 2);
   phyFramePointer = trigPhyFrameOfTrcBuf[trcBufLogID];
   frame = 0;
   addr = addrTrig + EMON_OFFSET;
   while (((addr & 0x07F0) != 0x07F0) &&
         (frame != endLogFrameOfTrcBuf[trcBufLogID])) {
      addr += MAX_FRAME_WIDTH;
      data = (U8 _far *) MK_FP(TMAN_SEG, addr);
      if (*data & EMON_BIT) endLogFrame++;
      phyFramePointer++;
      frame++;
   }
   validFrameInSubBuf = 0;
   while (frame++ != endLogFrameOfTrcBuf[trcBufLogID]) {
      if (phyFramePointer == bottomPhyFrameOfTrcBuf) {
         if (trcBufFill) {
            addr = addrLo | 0x000F;
            bank = bankLo;
            outp(pBANKWR, bank << 2);
            phyFramePointer = topPhyFrameOfTrcBuf;
         } else
            break;
      }
      else {
         phyFramePointer++;
         if (addr == 0xFFFF) outp(pBANKWR, (++bank) << 2);
         addr += MAX_FRAME_WIDTH;
      }
      data = (U8 _far *) MK_FP(TMAN_SEG, addr);
      if (*data & EMON_BIT) {
         endLogFrame++;
         validFrameInSubBuf++;
      }
      if ((addr & 0x07F0) == 0x07F0) {
         numOfValidFrameInSubBuf[(U16) ((bank << 5) | (addr >> 11))]
            = validFrameInSubBuf;
         validFrameInSubBuf = 0;
      }
   }
   startLogFrameOfTrcBuf[trcBufLogID] = startLogFrame;
   endLogFrameOfTrcBuf[trcBufLogID] = endLogFrame;
   return(GOOD);
}

/******************************************************************************
**
**  SearchTriggerPoint
**
**  Description:
**    Search the trigger frame of trace buffer
**    When the trigger condition matched, TPIRUN will change from 1 to 0
**    But there is a 10 EPCLK delay of TPIRUN so tpiShift exist
**    input parameters addrStart/addrEnd/bankEnd identify the search area
**
**  Parameters:
**    input:
**      trcBufLogID: current trace buffer logical ID
**      addrStart: the start addr of the low bank
**      addrEnd: the end addr of the bankEnd
**      bankEnd: the last bank to search
**    output:
**      *topPhyFrameOfTrigSubBuf: top physical frame of the sub-buffer with
**                                trigger frame in it
**
*******************************************************************************/
BOOLEAN SearchTriggerPoint(U16 trcBufLogID, U16 addrStart, U16 addrEnd,
                           U8 bankEnd, U32 *topPhyFrameOfTrigSubBuf) {
   const U16 SKIP = FRAME_NUMBER_PER_SUB_BUFFER * MAX_FRAME_WIDTH;
   U8 bank, bankTrig, _far *data, tpi, tpiShift, frameID;
   U16 addr, addrTrig, addrStop;
   BOOLEAN hiTPIRUN = FALSE, foundTrigSubBuf = FALSE, condition2Match;

   bank = bankLo;
   addr = addrLo;
   addrStop = addrHi;
   /* The trigger sub-buffer must match one of the following two conditions :
        (1) TPIRUN changed from 1 to 0 in the trigger sub-buffer
        (2) all frames of the trigger sub-buffer are "TPIRUN = 1", but
            all frames of the sub-buffer next to it must be "TPIRUN = 0"
      The purpose of the following "Do loop" is going to find "bankTrig" and
      "addrTrig" which point to the first frame of trigger sub-buffer */
   do {
      outp(pBANKWR, bank << 2);
      if (bank == bankEnd) addrStop = addrEnd;
      do {
         data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET);
         switch (((U8) (*data)) & TPIRUN_BIT) {
            case 0:
               if (hiTPIRUN == NO) break;
               for (frameID = 0, condition2Match = YES;
                    (condition2Match == YES) &&
                    (frameID < FRAME_NUMBER_PER_SUB_BUFFER);
                    frameID++) {
                  data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET +
                                           frameID * MAX_FRAME_WIDTH);
                  if (((U8) (*data)) & TPIRUN_BIT) condition2Match = NO;
               }
               if (condition2Match) {
                  if (addr == addrLo) {
                     addr = addrHi - SKIP + 1;
                     bank = (bank == bankLo) ? (U8) bankHi : (U8) (bank - 1);
                     outp(pBANKWR, bank << 2);
                  }
                  else addr -= SKIP;
                  if ((tpi = GetTPI(addr, &tpiShift)) < tpiShift) {
                     if (addr == addrLo) {
                        addr = addrHi - SKIP + 1;
                        bank = (bank == bankLo) ? (U8) bankHi : (U8) (bank - 1);
                     }
                     else addr -= SKIP;
                     tpi += FRAME_NUMBER_PER_SUB_BUFFER;
                  }
                  foundTrigSubBuf = YES;
                  bankTrig = bank;
                  addrTrig = addr;
               }
               break;
            default:
               if (hiTPIRUN == FALSE) hiTPIRUN = TRUE;
               data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET
                                                  + (SKIP - MAX_FRAME_WIDTH));
               if ((((U8) (*data)) & TPIRUN_BIT) == 0) { /* condition1Match */
                  if ((tpi = GetTPI(addr, &tpiShift)) < tpiShift) {
                     if (addr == addrLo) {
                        addr = addrHi - SKIP + 1;
                        bank = (bank == bankLo) ? (U8) bankHi : (U8) (bank - 1);
                     }
                     else addr -= SKIP;
                     tpi += FRAME_NUMBER_PER_SUB_BUFFER;
                  }
                  data = (U8 _far *) MK_FP(TMAN_SEG, addr + EMON_OFFSET +
                                           (tpi - tpiShift) * MAX_FRAME_WIDTH);
                  if ((((U8) (*data)) & EMON_BIT) != 0) {
                     foundTrigSubBuf = YES;
                     bankTrig = bank;
                     addrTrig = addr;
                  }
               }
               break;
         }
         addr += SKIP;
      } while ((addr != addrStop + 1) && (foundTrigSubBuf == NO));
      addr = addrLo;
      bank++;
   } while ((bank <= bankEnd) && (foundTrigSubBuf == NO));
   if ((foundTrigSubBuf == FALSE) && (hiTPIRUN == TRUE)) {
      bank = bankLo;
      addr = addrLo;
      outp(pBANKWR, bank << 2);
      data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET);
      for (frameID = 0, condition2Match = YES;
           (condition2Match == YES) &&
           (frameID < FRAME_NUMBER_PER_SUB_BUFFER);
           frameID++) {
         data = (U8 _far *) MK_FP(TMAN_SEG, addr + TMP_OFFSET +
                                  frameID * MAX_FRAME_WIDTH);
         if (((U8) (*data)) & TPIRUN_BIT) condition2Match = NO;
      }
      if (condition2Match) {
         addr = addrHi - SKIP + 1;
         bank = bankHi;
         outp(pBANKWR, bank << 2);
         if ((tpi = GetTPI(addr, &tpiShift)) < tpiShift) {
            addr -= SKIP;
            tpi += FRAME_NUMBER_PER_SUB_BUFFER;
         }
         data = (U8 _far *) MK_FP(TMAN_SEG, addr + EMON_OFFSET +
                                  (tpi - tpiShift) * MAX_FRAME_WIDTH);
         if ((((U8) (*data)) & EMON_BIT) != 0) {
            foundTrigSubBuf = YES;
            bankTrig = bank;
            addrTrig = addr;
         }
      }
   }
   if (foundTrigSubBuf) {
      *topPhyFrameOfTrigSubBuf =
         (((U32) bankTrig) << 12) | (((U32) addrTrig) >> 4);
      trigPhyFrameOfTrcBuf[trcBufLogID] =
         (U32) (*topPhyFrameOfTrigSubBuf) + (U32) (tpi - tpiShift);
   }
   return(foundTrigSubBuf);
}

/******************************************************************************
**
**  GetInfoOfMidTB
**
**  Description:
**    Get informations of trace buffers other than the last trace buffer
**    Get trigPhyFrameOfTrcBuf[trcBufLogID]
**        startLogFrameOfTrcBuf[trcBufLogID]
**        endLogFrameOfTrcBuf[trcBufLogID]
**    then set gotTrigInfoOfTrcBuf[trcBufLogID] = TRUE
**
**  Parameters:
**    input:
**      trcBufLogID: current trace buffer logical ID
**      trcBufPhyID: current trace buffer physical ID
**    output:
**      <none>
**
*******************************************************************************/
RETCODE GetInfoOfMidTB(U16 trcBufLogID, U16 trcBufPhyID) {
   RETCODE err;
   const U16 SKIP = FRAME_NUMBER_PER_SUB_BUFFER * MAX_FRAME_WIDTH;
   U16 addrEnd, addrStart, preTBTrigAtOddSubBuf;
   U8  bankEnd, _far *data;
   BOOLEAN trcBufFill = TRUE;
   U32 topPhyFrameOfTrcBuf, bottomPhyFrameOfTrcBuf;
   U32 NumSubBufFromTrigToEnd, topPhyFrameOfTrigSubBuf, topPhyFrameOfEndSubBuf;

   topPhyFrameOfTrcBuf = (((U32) bankLo) << 12) | (((U32) ((addrLo)) >> 4));
   bottomPhyFrameOfTrcBuf = (((U32) bankHi) << 12) | (((U32) ((addrHi)) >> 4));
   preTBTrigAtOddSubBuf = IsPreTBTrigAtOddSubBuf(trcBufPhyID);
   addrStart = addrLo + SKIP * preTBTrigAtOddSubBuf;
   if (SearchTriggerPoint(trcBufLogID, addrStart, addrHi, bankHi,
          &topPhyFrameOfTrigSubBuf) == FALSE) return(ERR_TB_WITHOUT_TRIG);
      /* there must be a trigger point in the middle trace buffer */
   switch (triggerLocation) {
      case PRE_TRIG:
         NumSubBufFromTrigToEnd =
            (TOTAL_SUB_BUFFER_NUMBER / traceBufferNumber) - 1;  break;
      case CENTER_TRIG:
         NumSubBufFromTrigToEnd =
            ((TOTAL_SUB_BUFFER_NUMBER / traceBufferNumber) / 2) - 1;  break;
      case POST_TRIG:
         NumSubBufFromTrigToEnd = 1;  break;
   }
   topPhyFrameOfEndSubBuf = topPhyFrameOfTrigSubBuf
                            + (NumSubBufFromTrigToEnd * 128);
   if (topPhyFrameOfEndSubBuf > bottomPhyFrameOfTrcBuf)
      topPhyFrameOfEndSubBuf = (topPhyFrameOfEndSubBuf - bottomPhyFrameOfTrcBuf)
                               + topPhyFrameOfTrcBuf - 1;
   bankEnd = (U8) (topPhyFrameOfEndSubBuf >> 12);
   addrEnd = (U16) (topPhyFrameOfEndSubBuf << 4);
   outp(pBANKWR, bankEnd << 2);
   data = (U8 _far *) MK_FP(TMAN_SEG, addrEnd + TMP_OFFSET +
                            TBFILL_VAL * MAX_FRAME_WIDTH);
   if (((*data) & TBFILL_BIT) == 0) trcBufFill = FALSE;
   endLogFrameOfTrcBuf[trcBufLogID] =
      127 - (trigPhyFrameOfTrcBuf[trcBufLogID] - topPhyFrameOfTrigSubBuf)
      + (NumSubBufFromTrigToEnd * 128);
   switch (trcBufFill) {
      case FALSE:
         startLogFrameOfTrcBuf[trcBufLogID] =
            topPhyFrameOfTrcBuf - trigPhyFrameOfTrcBuf[trcBufLogID]
            + FRAME_NUMBER_PER_SUB_BUFFER * preTBTrigAtOddSubBuf;
         break;
      case TRUE:
         startLogFrameOfTrcBuf[trcBufLogID] = (endLogFrameOfTrcBuf[trcBufLogID]
            + 1) - ((U32) TOTAL_FRAME_NUMBER / (U32) traceBufferNumber);
         break;
   }
   if ((err = FilterEMON(trcBufLogID, topPhyFrameOfTrcBuf,
                         bottomPhyFrameOfTrcBuf,trcBufFill)) != GOOD) return(err);
   gotTrigInfoOfTrcBuf[trcBufLogID] = TRUE;
   return(GOOD);
}

/******************************************************************************
**
**  GetInfoOfEndTB
**
**  Description:
**    Get informations of the last trace buffer where trace halt at
**    Get trigPhyFrameOfTrcBuf[trcBufLogID]
**        startLogFrameOfTrcBuf[trcBufLogID]
**        endLogFrameOfTrcBuf[trcBufLogID]
**    then set gotTrigInfoOfTrcBuf[trcBufLogID] = TRUE
**
**  Parameters:
**    input:
**      trcBufLogID: current trace buffer logical ID
**      trcBufPhyID: current trace buffer physical ID
**    output:
**      <none>
**
*******************************************************************************/
RETCODE GetInfoOfEndTB(U16 trcBufLogID, U16 trcBufPhyID) {
   RETCODE err;
   const U16 SKIP = FRAME_NUMBER_PER_SUB_BUFFER * MAX_FRAME_WIDTH;
   U16 addrStart, addrEnd, preTBTrigAtOddSubBuf;
   U8  bankEnd, _far *data;
   BOOLEAN trcBufFill;
   U32 topPhyFrameOfTrcBuf, bottomPhyFrameOfTrcBuf, topPhyFrameOfTrigSubBuf;
   U32 endPhyFrameOfTrcBuf;

   topPhyFrameOfTrcBuf = (((U32) bankLo) << 12) | (((U32) ((addrLo)) >> 4));
   bottomPhyFrameOfTrcBuf = (((U32) bankHi) << 12) | (((U32) ((addrHi)) >> 4));
   bankEnd = (U8) (sBAAfterTraceHalt >> 5);
   addrEnd = ((U16) sBAAfterTraceHalt) << 11;
   outp(pBANKWR, bankEnd << 2);
   data = (U8 _far *) MK_FP(TMAN_SEG, addrEnd + TMP_OFFSET +
                            TBFILL_VAL * MAX_FRAME_WIDTH);
   if ((*data) & TBFILL_BIT) {  /* TBFILL == 1 */
      trcBufFill = TRUE;
      addrStart = addrLo;
      addrEnd = addrHi;
   }
   else {  /* TBFILL == 0 */
      trcBufFill = FALSE;
      preTBTrigAtOddSubBuf = IsPreTBTrigAtOddSubBuf(trcBufPhyID);
      addrStart = addrLo + SKIP * preTBTrigAtOddSubBuf;
      addrEnd += (SKIP - 1);
   }
   endPhyFrameOfTrcBuf = (U32) (((U32) sBAAfterTraceHalt << 7)
                         + (FRAME_NUMBER_PER_SUB_BUFFER - 1));
   if (SearchTriggerPoint(trcBufLogID, addrStart, addrEnd, bankEnd,
                          &topPhyFrameOfTrigSubBuf) == FALSE) {
      endLogFrameOfTrcBuf[trcBufLogID] = 0;
      trigPhyFrameOfTrcBuf[trcBufLogID] = endPhyFrameOfTrcBuf;
      switch (trcBufFill) {
         case TRUE:
            startLogFrameOfTrcBuf[trcBufLogID] =
               (S32) (topPhyFrameOfTrcBuf - bottomPhyFrameOfTrcBuf);
            break;
         case FALSE:
            startLogFrameOfTrcBuf[trcBufLogID] =
               (U32) ((topPhyFrameOfTrcBuf
                       + FRAME_NUMBER_PER_SUB_BUFFER * preTBTrigAtOddSubBuf)
                       - trigPhyFrameOfTrcBuf[trcBufLogID]);
            break;
      }
   }
   else {
      if (trigPhyFrameOfTrcBuf[trcBufLogID] <= endPhyFrameOfTrcBuf) {
         endLogFrameOfTrcBuf[trcBufLogID] =
            (S32) (endPhyFrameOfTrcBuf - trigPhyFrameOfTrcBuf[trcBufLogID]);
         switch (trcBufFill) {
            case TRUE:
               startLogFrameOfTrcBuf[trcBufLogID] =
                  (S32) (topPhyFrameOfTrcBuf - trigPhyFrameOfTrcBuf[trcBufLogID]
                         + endPhyFrameOfTrcBuf - bottomPhyFrameOfTrcBuf);
               break;
            case FALSE:
               startLogFrameOfTrcBuf[trcBufLogID] =
                  (S32) ((topPhyFrameOfTrcBuf
                          + FRAME_NUMBER_PER_SUB_BUFFER * preTBTrigAtOddSubBuf)
                          - trigPhyFrameOfTrcBuf[trcBufLogID]);
               break;
         }
      }
      else {
         endLogFrameOfTrcBuf[trcBufLogID] =
            (S32) (bottomPhyFrameOfTrcBuf - trigPhyFrameOfTrcBuf[trcBufLogID]
                  + endPhyFrameOfTrcBuf - topPhyFrameOfTrcBuf + 1);
         startLogFrameOfTrcBuf[trcBufLogID] =
            (S32) (endPhyFrameOfTrcBuf - trigPhyFrameOfTrcBuf[trcBufLogID] + 1);
      }
   }
   if ((err = FilterEMON(trcBufLogID, topPhyFrameOfTrcBuf,
                         bottomPhyFrameOfTrcBuf,trcBufFill)) != GOOD) return(err);
   gotTrigInfoOfTrcBuf[trcBufLogID] = TRUE;
   return(GOOD);
}

/******************************************************************************
**
**  SetCounters
**
**  Description:
**    Copy the TMANTraceXCByte to bitStreamBuffer1 first then call this founction
**    According to the global variables "counter0" / "counter1", modify the
**    Terminal Count 0 and Terminal Count 1 of "bitStreamBuffer1"
**
**  Parameters:
**    input:
**      <none>
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetCounters(VOID) {
   RETCODE err;
   static const U16 _based(_segname("_CODE")) clbOfTerminalCNT0[] = {
      0x0808,0x0808,0x0708,0x0708,0x0A08,0x0A08,0x0B08,0x0B08,0x0C08,0x0C08 };
   static const U16 _based(_segname("_CODE")) clbOfTerminalCNT1[] = {
      0x080C,0x080C,0x070C,0x070C,0x0A0C,0x0A0C,0x0B0C,0x0B0C,0x0D0C,0x0D0C };
   U16 test;
   U8 bit, addrBitEven, addrBitOdd, addr;

   for (bit = 0; (test = 0x0001 << bit) < MAX_TERMINAL_CNT; bit += 2) {
      addrBitEven = (U8) (((test & counter0) == 0) ? 0x00 : 0x08);
      addrBitOdd  = (U8) ((((test << 1) & counter0) == 0) ? 0x00 : 0x04);
      addr = addrBitEven | addrBitOdd;
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_F, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_F, addr,
              0x0C, 1)) != GOOD) return(err);
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_G, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_G, addr,
              0x0C, 1)) != GOOD) return(err);
   }
   {
      bit = 6;
      test = 0x0001 << bit;
      addrBitEven = (U8) (((test & counter0) == 0) ? 0x00 : 0x08);
      addrBitOdd  = (U8) ((((test << 1) & counter0) == 0) ? 0x00 : 0x02);
      addr = addrBitEven | addrBitOdd;
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_F, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,_,A,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_F, addr,
              0x0A, 1)) != GOOD) return(err);
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_G, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,_,A,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT0[bit], XLX_G, addr,
              0x0A, 1)) != GOOD) return(err);
   }
   for (bit = 0; (test = 0x0001 << bit) < MAX_TERMINAL_CNT; bit += 2) {
      addrBitEven = (U8) (((test & counter1) == 0) ? 0x00 : 0x08);
      addrBitOdd  = (U8) ((((test << 1) & counter1) == 0) ? 0x00 : 0x04);
      addr = addrBitEven | addrBitOdd;
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT1[bit], XLX_F, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT1[bit], XLX_F, addr,
              0x0C, 1)) != GOOD) return(err);
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT1[bit], XLX_G, 0x00,
              0x00, 0)) != GOOD) return(err);  /* addr = (B,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfTerminalCNT1[bit], XLX_G, addr,
              0x0C, 1)) != GOOD) return(err);
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetNumberOfTraceBuffer
**
**  Description:
**    Copy the TMANTraceXCByte to bitStreamBuffer1 first then call this founction
**    According to the global variable "numberOfBitOfTBN", modify the
**    SBA counter of "bitStreamBuffer1"
**
**  Parameters:
**    input:
**      <none>
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetNumberOfTraceBuffer(VOID) {
   RETCODE err;
   static const U16 _based(_segname("_CODE")) clbOfSBA[] = {
      0x0000,0x0401,0x0501,0x0601,0x0701,0x0801,0x0901,0x0A01,0x0B01,0x0C01 };
   U8 bit = NUMBER_OF_SBA_BIT - 1;
   U16 temp = numberOfBitOfTBN;

   while (temp-- > 1) {  /* Set TBN bits other then LSB */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (B,QX,A,E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x04, 0x05, 1))
              != GOOD) return(err);  /* F = (QX*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x0B, 0x0F, 1))
              != GOOD) return(err);  /*    +(B*~QX*A*E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x0D, 0x0F, 1))
              != GOOD) return(err);  /*    +(B*QX*~A*E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QX,C,A,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x0E, 0x0E, 1))
              != GOOD) return(err);  /* G = (QX*C*A) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit--], XLX_G, 0x02, 0x06, 1))
              != GOOD) return(err);  /*    +(~C*A) */
   }
   if (temp == 0) {  /* Set TBN LSB */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (B,QX,_,E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x04, 0x05, 1))
              != GOOD) return(err);  /* F = (QX*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x09, 0x0D, 1))
              != GOOD) return(err);  /*    +(B*~QX*E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QX,C,A,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x02, 0x06, 1))
              != GOOD) return(err);  /* G = (~C*A) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit--], XLX_G, 0x0C, 0x0C, 1))
              != GOOD) return(err);  /*    +(QX*C) */
   }
   while (bit > 1) {  /* Set SBN bits other then LSB */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (B,QX,A,E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x0A, 0x0F, 1))
              != GOOD) return(err);  /* F = (B*~QX*A*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x0C, 0x0F, 1))
              != GOOD) return(err);  /*    +(B*QX*~A*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x04, 0x0D, 1))
              != GOOD) return(err);  /*    +(~B*QX*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QX,_,A,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit--], XLX_G, 0x0A, 0x0A, 1))
              != GOOD) return(err);  /* G = (QX*A) */
   }
   if (bit == 1) {  /* Set SBN LSB */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (B,QX,A,E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x08, 0x0D, 1))
              != GOOD) return(err);  /* F = (B*~QX*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_F, 0x04, 0x0D, 1))
              != GOOD) return(err);  /*    +(~B*QX*~E) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QX,_,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfSBA[bit], XLX_G, 0x08, 0x08, 1))
              != GOOD) return(err);  /* G = (QX) */
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetTriggerLocation
**
**  Description:
**    Copy the TMANTraceXCByte to bitStreamBuffer1 first then call this founction
**    According to the global variable "numberOfBitOfTBN" modify the PFC counter
**    of "bitStreamBuffer1"
**    According to the global variable "triggerLocation" modify the trigger
**    location logic (CLB DONE) of "bitStreamBuffer1"
**
**  Parameters:
**    input:
**      <none>
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetTriggerLocation(VOID) {
   RETCODE err;
   static const U16 _based(_segname("_CODE")) clbOfPFC[] = {
      0x020A,0x040A,0x050A,0x060A,0x070A,0x080A,0x090A,0x0B0A,0x0C0A };
   static const U16 _based(_segname("_CODE")) clbOfTrigLoc = 0x0D0A;
   U8 bit = NUMBER_OF_PFC_BIT - 1;
   U16 temp = numberOfBitOfTBN;

   while (temp-- != 0) {
      if (bit != 6) {
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* F = 0 */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (_,_,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit--], XLX_G, 0x02, 0x02, 1))
                 != GOOD) return(err);  /* G = A */
      }
      else {
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* G = 0 */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (_,_,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit--], XLX_F, 0x02, 0x02, 1))
                 != GOOD) return(err);  /* F = A */
      }
   }
   while (bit > 0) {
      if (bit != 6) {
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (QX,C,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x08, 0x0C, 1))
                 != GOOD) return(err);  /* F = (QX*~C) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x06, 0x0E, 1))
                 != GOOD) return(err);  /*   + (~QX*C*A) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x0C, 0x0E, 1))
                 != GOOD) return(err);  /*   + (QX*C*~A) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (QX,_,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit--], XLX_G, 0x0A, 0x0A, 1))
                 != GOOD) return(err);  /* G = (QX*A) */
      }
      else {
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (QX,C,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x08, 0x0C, 1))
                 != GOOD) return(err);  /* G = (QX*~C) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x06, 0x0E, 1))
                 != GOOD) return(err);  /*   + (~QX*C*A) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_G, 0x0C, 0x0E, 1))
                 != GOOD) return(err);  /*   + (QX*C*~A) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (QX,_,A,_) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfPFC[bit--], XLX_F, 0x0A, 0x0A, 1))
                 != GOOD) return(err);  /* F = (QX*A) */
      }
   }
   if (bit == 0) {
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QY,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfPFC[bit], XLX_F, 0x00, 0x08, 1))
              != GOOD) return(err);  /* F = ~QY */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfPFC[bit], XLX_G, 0x00, 0x00, 0))
              != GOOD) return(err);  /* addr = (QY,C,_,_) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfPFC[bit], XLX_G, 0x08, 0x0C, 1))
              != GOOD) return(err);  /* G = (QY*~C) */
      if ((err = XlxModifyClb(bitStreamBuffer1,
                              XC3064, clbOfPFC[bit--], XLX_G, 0x04, 0x0C, 1))
              != GOOD) return(err);  /*   + (~QY*C) */
   }
   switch (triggerLocation) {
      case POST_TRIG:
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_F, 0x00, 0x00, 1))
                 != GOOD) return(err);  /* F = 1 */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (_,QY,_,E) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x04, 0x04, 1))
                 != GOOD) return(err);  /* G = QY */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x01, 0x01, 1))
                 != GOOD) return(err);  /*    +E */
         break;
      case CENTER_TRIG:
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_F, 0x00, 0x00, 1))
                 != GOOD) return(err);  /* F = 1 */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x00, 0x00, 0))
                 != GOOD) return(err);  /* addr = (_,QY,A,E) */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x04, 0x04, 1))
                 != GOOD) return(err);  /* G = QY */
         if ((err = XlxModifyClb(bitStreamBuffer1,
                                 XC3064, clbOfTrigLoc, XLX_G, 0x03, 0x03, 1))
                 != GOOD) return(err);  /*    +(A*E) */
         break;
      case PRE_TRIG: /* default */
         break;
   }
   return(GOOD);
}

/******************************************************************************
**
**  DumpDRAM
**
**  Description:
**    Dump DRAM frames with EMON=1 to the pointer "buffer"
**
**  Parameters:
**    input:
**      bankS: dump start bank
**      addrS: dump start addr
**      length: dump length
**      buffer: pointer to the dumped data
**    output:
**      <none>
**
*******************************************************************************/
RETCODE DumpDRAM(U8 bankS, U16 addrS, U16 length, VOID *buffer, U16 *frameNum,
                 S32 logFrame, S32 endLogFrameOfTB) {
   RETCODE err;
   U32 topPhyFrameOfTrcBuf = (((U32) bankLo) << 12) | (((U32) addrLo) >> 4);
   U32 bottomPhyFrameOfTrcBuf = (((U32) bankHi) << 12) | (((U32) addrHi) >> 4);
   U32 phyFramePointer = (U32) (((U32) bankS) << 12) | ((U32) (addrS >> 4));
   U8 _far *addr = MK_FP(TMAN_SEG, addrS);
   U8 bank;
   U16 len;

   *frameNum = 0;
   bank = bankS;
   outp(pBANKWR, bank << 2);
   while (length > 0) {
      len = (length / FRAME_WIDTH == 0) ?
            (length % FRAME_WIDTH) : FRAME_WIDTH;
      if (*(addr + EMON_OFFSET) & EMON_BIT) {
         fmemcpy(buffer, addr, len);
         if ((err = fmemcmp(buffer, addr, len)) != GOOD) return(ERR_DRAM_DUMP);
         (*frameNum)++;
         (U8 *) buffer += len;
         length -= len;
      }
      if (phyFramePointer == bottomPhyFrameOfTrcBuf) {
         addr = MK_FP(TMAN_SEG, addrLo);
         bank = bankLo;
         outp(pBANKWR, bank << 2);
         phyFramePointer = topPhyFrameOfTrcBuf;
      }
      else {
         phyFramePointer++;
         if ((phyFramePointer & 0x0FFF) == 0x0) outp(pBANKWR, (++bank) << 2);
         addr += MAX_FRAME_WIDTH;
      }
      if ((logFrame++) == endLogFrameOfTB) break;
   } 
   return(GOOD);
}

/******************************************************************************
**
**  QualifyDumpDRAM
**
**  Description:
**    Dump DRAM frames which match the following two conditions to the pointer
**    "buffer"
**       1. EMON=1
**       2. match the qualifications Address/Status/Trace-Bit
**
**   # Address qualification is defined by qualify->startAddr and qualify->endAddr
**   # Status qualification is defined by qualify->status
**   # Trace-Bit qualification is defined by qualify->traceBitData and
**     qualify->traceBitWild
**
**    There are 8 statuses(S, R, W, I, O, A, H, D) for 80C186/188.
**    Each bit of qualify->status represented one status.
**       bit -> 151413121110 9 8 7 6  5  4  3  2  1  0
**       rep ->  . . . . . . . . . AK UW UR SW SR UP SP   0 = Don't care
**    The traced status signals is shown as follows
**
**           stats nibble 3                   status nibble 2
**        =======================          ========================
**           ES15=> NCS7                      ES11=> NCS3
**           ES14=> NCS6                      ES10=> NCS2
**           ES13=> NCS5                      ES9 => NCS1
**           ES12=> NCS4                      ES8 => NCS0
**
**           stats nibble 1                   status nibble 0
**        =======================          ========================
**           ES7 => FC2                       ES3 => NDRAM
**           ES6 => FC1                       ES2 => spare
**           ES5 => FC0                       ES1 => UDS-
**           ES4 => RW-                       ES0 => LDS-
**
**        status  |  ES7 ES6 ES5 ES4 ES3 ES2 ES1 ES0  |  pattern  |  mask
**      ----------+-----------------------------------+-----------+---------
**          SP    |   1   1   0   X   X   X   X   X   |   0x0C0   |  0x0E0
**          UP    |   0   1   0   X   X   X   X   X   |   0x040   |  0x0E0
**          SR    |   1   0   1   1   X   X   X   X   |   0x0B0   |  0x0F0
**          SW    |   1   0   1   0   X   X   X   X   |   0x0A0   |  0x0F0
**          UR    |   0   0   1   1   X   X   X   X   |   0x030   |  0x0F0
**          UW    |   0   0   1   0   X   X   X   X   |   0x020   |  0x0F0
**          AK    |   1   1   1   X   X   X   X   X   |   0x0E0   |  0x0E0
**
**        Note:  `0' means "Low", `1' means "High", `X' means "Don't Care".
**
**  Parameters:
**    input:
**      bankS: dump start bank
**      addrS: dump start addr
**      length: dump length
**      buffer: pointer to the dumped data
**      qualify: structure of qualifications Addr/Status/Trace-Bit
**    output:
**      <none>
**
*******************************************************************************/
RETCODE QualifyDumpDRAM(U8 bankS, U16 addrS, U16 length, VOID *buffer,
                        U16 *frameNum, S32 logFrame, S32 endLogFrameOfTB,
                        QUALIFY_LIST *qualify) {
   RETCODE err;
   static const EP_STATUS_SIG _based(_segname("_CODE")) pattern[] = {
      0x0C0 /* SP */,
      0x040 /* UP */,
      0x0B0 /* SR (SD-READ) */,
      0x0A0 /* SW (SD-WRITE) */,
      0x030 /* UR (UD-READ) */,
      0x020 /* UW (UD-WRITE) */,
      0x0E0 /* AK (interrupt acknowledge) */ };
   static const EP_STATUS_SIG _based(_segname("_CODE")) mask[] = {
      0x0E0 /* SP */,
      0x0E0 /* UP */,
      0x0F0 /* SR (SD-READ) */,
      0x0F0 /* SW (SD-WRITE) */,
      0x0F0 /* UR (UD-READ) */,
      0x0F0 /* UW (UD-WRITE) */,
      0x0E0 /* AK (interrupt acknowledge) */ };
   U32 topPhyFrameOfTrcBuf = (((U32) bankLo) << 12) | (((U32) addrLo) >> 4);
   U32 bottomPhyFrameOfTrcBuf = (((U32) bankHi) << 12) | (((U32) addrHi) >> 4);
   U32 phyFramePointer = (U32) (((U32) bankS) << 12) | ((U32) (addrS >> 4));
   U8 _far *addr = MK_FP(TMAN_SEG, addrS);
   U8 bank;
   U16 len;
   U8 bit;
   U32 address;
   U16 status;
   U8 traceBit;
   BOOLEAN statusQualifyMatch, thisFrameQualified;
   BOOLEAN qualifyAddress, qualifyStatus, qualifyTraceBit;

   *frameNum = 0;
   bank = bankS;
   outp(pBANKWR, bank << 2);
   qualifyAddress = (BOOLEAN) (((qualify->startAddr == 0) &&
                                (qualify->endAddr == 0xFFFFFFFF)) ? NO : YES);
   qualifyStatus = (BOOLEAN) ((qualify->status.statusFc == 0) ? NO : YES);
   qualifyTraceBit = (BOOLEAN) ((qualify->traceBitWild == 0) ? NO : YES);
   while (length > 0) {
      thisFrameQualified = YES;
      len = (length / FRAME_WIDTH == 0) ?
            (length % FRAME_WIDTH) : FRAME_WIDTH;
      if (*(addr + EMON_OFFSET) & EMON_BIT) {
         if (qualifyAddress) {
            address = ( ((U32) ((U8) *(addr + offADDR + 3))) << 24) |
                      ( ((U32) ((U8) *(addr + offADDR + 2))) << 16) |
                      ( ((U32) ((U8) *(addr + offADDR + 1))) << 8 ) |
                        ((U32) ((U8) *(addr + offADDR + 0)));
            if ((address < qualify->startAddr) || (address > qualify->endAddr))
               thisFrameQualified = NO;
         }
         if (qualify->statusFlag == QUALIFY_STATUS1) {
            if (qualifyStatus && thisFrameQualified) {
               status = (U8) *(addr + offSTAT);
               bit = 0;
               statusQualifyMatch = NO;
               while ((statusQualifyMatch == NO) && (bit < EP_STATUS_NUM)) {
                  if ((qualify->status.statusFc & (1 << bit)) &&
                     ((status & mask[bit]) == pattern[bit]))
                     statusQualifyMatch = YES;
                  bit++;
               }
               if (statusQualifyMatch == NO) thisFrameQualified = NO;
            }
            if ((qualify->status.statusDsHigh == 0xff) && thisFrameQualified) {
               if (qualify->status.statusDsLow != (U8)(*(addr + offSTAT) & 0x3))
                  thisFrameQualified = NO;
            }
            if ((qualify->status.statusCsHigh == 0xffff) && thisFrameQualified) {
               status = ((U8) *(addr + offSTAT + 1)) | ((((U8) *(addr + offSTAT)) & 8) << 5);
               if (qualify->status.statusCsLow != (status & 0x1ff))
                  thisFrameQualified = NO;
            }
         } else {
            status = ((U8) *(addr + offSTAT)) | (((U8) *(addr + offSTAT + 1)) << 8);
            if ((qualify->status2.status2Low != (((U32)(status)) & qualify->status2.status2High)) && thisFrameQualified) {
                  thisFrameQualified = NO;
            }
         }
         if (qualifyTraceBit && thisFrameQualified) {
            traceBit = (U8) *(addr + offSPAR);
            if ((traceBit & qualify->traceBitWild) != qualify->traceBitData)
               thisFrameQualified = NO;
         }
         if (thisFrameQualified) {
            fmemcpy(buffer, (U8 far *) &logFrame, sizeof(U32));
            (U8 *) buffer += sizeof(U32);
            fmemcpy(buffer, addr, len);
            if ((err = fmemcmp(buffer, addr, len)) != GOOD) return(ERR_DRAM_DUMP);
            (U8 *) buffer += len;
            length -= len;
            (*frameNum)++;
         }
         logFrame++;
      }
      if (phyFramePointer == bottomPhyFrameOfTrcBuf) {
         addr = MK_FP(TMAN_SEG, addrLo);
         bank = bankLo;
         outp(pBANKWR, bank << 2);
         phyFramePointer = topPhyFrameOfTrcBuf;
      }
      else {
         phyFramePointer++;
         if ((phyFramePointer & 0x0FFF) == 0x0) outp(pBANKWR, (++bank) << 2);
         addr += MAX_FRAME_WIDTH;
      }
      if (logFrame > endLogFrameOfTB) break;
   } 
   return(GOOD);
}

/********************************* EOF *************************************/
