/** gauge.c
 *
 *  DESCRIPTION: 
 *      Yet another 'Gas Gauge Custom Control.'  This control gives you
 *      a 'progress bar' class (named Gauge) for use in your applications.
 *      You can set the range, position, font, color, orientation, and 3d
 *      effect of the gauge by sending messages to the control.
 *
 *      Before you can use this control, you MUST first export the window
 *      procedure for the control (or define it with the _export keyword):
 *
 *          EXPORTS     gaugeWndProc
 *
 *      You then need initialize the class before you use it:
 *
 *          if (!gaugeInit(hInstance))
 *              die a horrible death
 *          else
 *              you are good to go
 *
 *      The colors used by the control default to black and white if you
 *      are running on a mono-display.  They default to blue and white
 *      if you are on a color display.  You enable the 3D effect by setting
 *      the GS_3D style flag in the styles field of the control (like
 *      any other control).
 *
 *      To select your own colors, you can send the G_SETFGCOLOR and
 *      G_SETBKCOLOR messages to set the foreground (percent done) and
 *      background (percent not done) colors.  The lParam is the RGB()
 *      value--wParam is ignored.
 *
 *      In all of the following G_??? messages, the arguments are
 *      WORDS.  If you are setting parameters, the value is sent as
 *      the wParam (lParam is ignored).  If you are getting parameters,
 *      the value is returned as a LONG and should be cast to a *signed*
 *      integer.
 *
 *      To set the depth of the 3D effect (if enabled), you can send the
 *      G_SETBEZELFACE and G_SETWIDTH3D messages.  The bezel face
 *      is the flat top on the 3D border--its color will be that of the
 *      button-face.  The 3D width is the width of the bezel itself; inside
 *      and outside.  The light color is white, the dark color is gray.
 *      Both widths *can* be zero--both default to 2 which looks to me.
 *
 *      The range of the control can be set by sending the G_SETRANGE
 *      message to the control.  It can be any integer from 1 to 32767.
 *      What this specifies is the number of pieces that create a whole.
 *      The default is 100.  You can get the current range setting by
 *      sending the G_GETRANGE message to the control.
 *
 *      The position (number of pieces out of the whole have been used) is
 *      set with the G_SETPOSITION message.  It can be any integer from
 *      0 to the current range setting of the control--it will be clipped
 *      if the position is out of bounds.  The default position is 0.  You
 *      can get the current position at any time with the G_GETPOSITION
 *      message.
 *
 *      You can also set the range using a delta from the current range.
 *      This is done by sending the G_SETDELTAPOS message with wParam
 *      set to a _signed_ integer value within the range of the control.
 *
 *      The font used for the percentage text can be set using the standard
 *      WM_SETFONT message.  You can get the current font at any time with
 *      the WM_GETFONT message.
 *
 *      The orientation can be left to right, right to left, bottom to top,
 *      or top to bottom.  Whatever suits your needs.  You set this by
 *      sending the G_ORIENTATION message to the control with one of
 *      the following values (default is G_ORIENT_LEFTTORIGHT):
 *
 *          G_ORIENT_LEFTTORIGHT (0)
 *          G_ORIENT_RIGHTTOLEFT (1)
 *          G_ORIENT_BOTTOMTOTOP (2)
 *          G_ORIENT_TOPTOBOTTOM (3)
 *
 *  HISTORY:
 *      3/12/91     cjp     put in this comment
 *      6/19/92     cjp     touched it a bit
 *
 ** cjp */
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//


/* get the includes we need */
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _VERSIONS_
#include "versions.h"
#endif

#ifndef _HLPENTRY_
#include "hlpentry.h"
#endif
#include <windows.h>
#include "g3d.h"
#include "gauge.h"


/* static global variables */
static char gszGaugeClass[] = "Gauge";
    

/* window word position definitions */
#define G_WW_PGAUGE   0
#define G_WW_EXTRABYTES  8


/* control block structure typedef */
typedef struct tGAUGE
{
    WORD    wRange;
    WORD    wPosition;
    WORD    wOrientation;
    WORD    wWidth3D;
    WORD    wWidthBezelFace;
    HFONT   hFont;
    DWORD   rgbTextColor;
    DWORD   rgbBkColor;

} GAUGE, *PGAUGE, FAR *LPGAUGE;

/* some default values for the control */
#define G_DEF_RANGE          100
#define G_DEF_POSITION       0
#define G_DEF_ORIENTATION    G_ORIENT_LEFTTORIGHT
#define G_DEF_WIDTH3D        2
#define G_DEF_BEZELFACE      2



/* the default settings for drawing colors--display dependent */
static DWORD    rgbDefTextColor;
static DWORD    rgbDefBkColor;
static BOOL     fSupport3D;


/* internal function prototypes */
static void NEAR PASCAL gaugePaint(HANDLE, HDC);
long FAR PASCAL _export gaugeWndProc(HANDLE, WORD, WORD, LONG);



/** BOOL FAR PASCAL gaugeInit(HANDLE hInstance)
 *
 *  DESCRIPTION: 
 *      Registers the window class for the Gauge control.  Performs
 *      other initialization for the Gauge text control.  This must
 *      be done before the Gauge control is used--or it will fail
 *      and your dialog box will not open!
 *
 *  ARGUMENTS:
 *      HANDLE hInstance :   Instance handle to register class with.
 *
 *  RETURN (BOOL FAR):
 *      The return value is TRUE if the Gauge class was successfully
 *      registered.  It is FALSE if the initialization fails.
 *
 *  NOTES:
 *
 ** cjp */

//#pragma alloc_text(init, gaugeInit)

BOOL FAR PASCAL gaugeInit(HANDLE hInstance)
{
    static BOOL fRegistered = FALSE;
    WNDCLASS    wc;
    HDC         hdc;
    
    /* assume already registered if not first instance */
    if (fRegistered)
        return (TRUE);

    /* fill in the class structure for the gauge control */
    wc.hCursor          = LoadCursor(NULL, IDC_ARROW);
    wc.hIcon            = NULL;
    wc.lpszMenuName     = NULL;
    wc.lpszClassName    = gszGaugeClass;
    wc.hbrBackground    = (HBRUSH)(COLOR_WINDOW + 1);
    wc.hInstance        = hInstance;

#ifdef GAUGE_DLL
    wc.style            = CS_GLOBALCLASS | CS_HREDRAW | CS_VREDRAW;
#else
    wc.style            = CS_HREDRAW | CS_VREDRAW;
#endif

    wc.lpfnWndProc      = gaugeWndProc;
    wc.cbClsExtra       = 0;
    wc.cbWndExtra       = G_WW_EXTRABYTES;

    /* attempt to register it--return FALSE if fail */
    if (!RegisterClass(&wc))
        return (FALSE);

    /*  Get a DC to determine whether device is mono or not, and set
     *  default foreground/background colors as appropriate.
     */
    if ((hdc = CreateIC("DISPLAY", NULL, NULL, NULL)) != 0)
    {
        /* check for mono-display */
        if ((GetDeviceCaps(hdc, BITSPIXEL) == 1) &&
             (GetDeviceCaps(hdc, PLANES) == 1))
        {
            /* using a mono DC--white foreground, black background */
            rgbDefTextColor = RGB(255, 255, 255);
            rgbDefBkColor = RGB(0, 0, 0);
        }

        /* good! we have color: blue foreground, white background */
        else 
        {
            rgbDefTextColor = RGB(0, 0, 255);
            rgbDefBkColor = RGB(255, 255, 255);
        }

        /* need at _least_ 8 for two shades of gray (>=VGA) */
        fSupport3D = (GetDeviceCaps(hdc, NUMCOLORS) >= 8) ? TRUE : FALSE;

        /* get rid of the DC (IC) */
        DeleteDC(hdc);
    }

    /* uh-oh... can't get DC (IC)... fail */
    else
    {
        /* unregister the class */
        UnregisterClass(gszGaugeClass, hInstance);
        return (FALSE);
    }

    /* return success */
    return (fRegistered = TRUE);
} /* gaugeInit() */


/** static void NEAR PASCAL gaugePaint(HANDLE hwnd, HDC hdc)
 *
 *  DESCRIPTION: 
 *      This function is responsible for painting the Gauge control.
 *
 *  ARGUMENTS:
 *      HANDLE hwnd   :   The window handle for the gauge.
 *
 *      HDC hdc     :   The DC for the gauge's window.
 *
 *  RETURN (void):
 *      The control will have been painted.
 *
 *  NOTES:
 *
 ** cjp */

static void NEAR PASCAL gaugePaint(HANDLE hwnd, HDC hdc)
{
    LPGAUGE  lpgauge;
    WORD        iRange, iPos;
    WORD        wOffset;
    DWORD       dwExtent;
    RECT        rc1, rc2;
    HFONT       hFont;
    char        ach[ 6 ];
    WORD        dx, dy, wGomerX, wGomerY;

    /* get pointer to the control's control block */
    lpgauge = (LPGAUGE)GetWindowLong(hwnd, G_WW_PGAUGE);

    /* set the colors into for the gauge into the control */
    SetTextColor(hdc, lpgauge->rgbTextColor);
    SetBkColor(hdc, lpgauge->rgbBkColor);

    /* draw black rectangle for gauge */
    GetClientRect(hwnd, &rc1);

    /* draw a black border on the _outside_ */
    FrameRect(hdc, &rc1, GetStockObject(BLACK_BRUSH));

    /* we want to draw _just inside_ the black border */
    InflateRect(&rc1, -1, -1);

    /* one line thick so far... */
    wOffset = 1;

    /* for 3D stuff, we need to have at least two shades of gray */
    if ((GetWindowLong(hwnd, GWL_STYLE) & GS_3D) && fSupport3D)
    {
        Draw3DRect(hdc, &rc1, lpgauge->wWidth3D, DRAW3D_OUT);
	InflateRect(&rc1, ~(lpgauge->wWidth3D), ~(lpgauge->wWidth3D));

        Draw3DFaceFrame(hdc, &rc1, lpgauge->wWidthBezelFace);
	InflateRect(&rc1, ~(lpgauge->wWidthBezelFace), ~(lpgauge->wWidthBezelFace));

        Draw3DRect(hdc, &rc1, lpgauge->wWidth3D, DRAW3D_IN);
	InflateRect(&rc1, ~(lpgauge->wWidth3D), ~(lpgauge->wWidth3D));

        /* draw a black border on the _inside_ */
        FrameRect(hdc, &rc1, GetStockObject(BLACK_BRUSH));

        /* we want to draw _just inside_ the black border */
        InflateRect(&rc1, -1, -1);

        /* add all the other pixels into the border width */
        wOffset += (2 * lpgauge->wWidth3D) + lpgauge->wWidthBezelFace + 1;
    }
   
    /* dup--one rc for 'how much filled', one rc for 'how much empty' */
    rc2 = rc1;

    /* get the range--make sure it's a valid range */
    if ((iRange = lpgauge->wRange) <= 0)
        iRange = 1;

    /* get the position--greater than 100% would be bad */
    if ((iPos = lpgauge->wPosition) > iRange)
        iPos = iRange;

    /* compute the actual size of the gauge */
    dx = rc1.right - rc1.left;
    dy = rc1.bottom - rc1.top;
    wGomerX = (WORD)((DWORD)iPos * dx / iRange);
    wGomerY = (WORD)((DWORD)iPos * dy / iRange);

    /* get the orientation and munge rects accordingly */
    switch (lpgauge->wOrientation)
    {
        case G_ORIENT_RIGHTTOLEFT:
            rc1.left = rc2.right = rc1.right - wGomerX;
            break;

        case G_ORIENT_BOTTOMTOTOP:
            rc1.top = rc2.bottom = rc1.bottom - wGomerY;
            break;

        case G_ORIENT_TOPTOBOTTOM:
            rc1.bottom = rc2.top += wGomerY;
            break;

        default:
            rc1.right = rc2.left += wGomerX;
            break;
    } /* switch () */

    /* select the correct font */
    hFont = SelectObject(hdc, lpgauge->hFont);

    /* build up a string to blit out--ie the meaning of life: "42%" */
    wsprintf(ach, "%3d%%", (WORD)((DWORD)iPos * 100 / iRange));
    dwExtent = GetTextExtent(hdc, ach, wGomerX = lstrlen(ach));


    /*  Draw the finished (ie the percent done) side of box.  If
     *  G_WW_POSITION is 42, (in range of 0 to 100) this ExtTextOut
     *  draws the meaning of life (42%) bar.
     */
    ExtTextOut(hdc, (dx - LOWORD(dwExtent)) / 2 + wOffset,
                    (dy - HIWORD(dwExtent)) / 2 + wOffset,
                    ETO_OPAQUE | ETO_CLIPPED, &rc2, ach, wGomerX, NULL);

    /*  Reverse fore and back colors for drawing the undone (ie the non-
     *  finished) side of the box.
     */
    SetBkColor(hdc, lpgauge->rgbTextColor);
    SetTextColor(hdc, lpgauge->rgbBkColor);

    ExtTextOut(hdc, (dx - LOWORD(dwExtent)) / 2 + wOffset,
                    (dy - HIWORD(dwExtent)) / 2 + wOffset,
                    ETO_OPAQUE | ETO_CLIPPED, &rc1, ach, wGomerX, NULL);

    /* unselect the font */
    SelectObject(hdc, hFont);
} /* gaugePaint() */
  

/** long FAR PASCAL _export gaugeWndProc(HANDLE hwnd, WORD uMsg, WORD wParam, LONG lParam)
 *
 *  DESCRIPTION: 
 *      This is the control's window procedure.  Its purpose is to handle
 *      special messages for this custom control.
 *
 *      The special control messages for the gauge control are:
 *
 *          G_SETRANGE       :   Sets the range of the gauge.  In other
 *                                  words, the number of parts that make a
 *                                  whole.
 *
 *          G_GETRANGE       :   Returns the current range of the gauge.
 *
 *          G_SETORIENTATION :   Sets the orientation of the gauge.  This
 *                                  can be one of the G_ORIENT_?? msgs.
 *
 *          G_GETORIENTATION :   Gets the current orientation of the 
 *                                  gauge.
 *
 *          G_SETPOSITION    :   Sets the current position of the gauge.
 *                                  In other words, how many pieces of the
 *                                  whole have been used.
 *
 *          G_GETPOSITION    :   Gets the current position of the gauge.
 *
 *          G_SETDELTAPOS    :   Sets the position of the gauge +/- the
 *                                  specified amount.
 *
 *          G_SETFGCOLOR     :   Sets the foreground (percent done) color.
 *
 *          G_GETFGCOLOR     :   Gets the foreground (percent done) color.
 *
 *          G_SETBKCOLOR     :   Sets the background (percent not done)
 *                                  color.
 *
 *          G_GETBKCOLOR     :   Gets the background (percent not done)
 *                                  color.
 *
 *          WM_SETFONT          :   Sets the font to use for the percentage
 *                                  text of the gauge.
 *
 *          WM_GETFONT          :   Gets the current font in use by the
 *                                  gauge.
 *
 *  NOTES:
 *
 ** cjp */

long FAR PASCAL _export gaugeWndProc(HANDLE hwnd, WORD uMsg, WORD wParam, LONG lParam)
{
    HFONT       hFont;
    PAINTSTRUCT ps;
    static HANDLE hgauge = 0;
    LPGAUGE  lpgauge;
    RECT        rc;

    lpgauge = (LPGAUGE)GetWindowLong(hwnd, G_WW_PGAUGE);

    /* break to get DefWindowProc() */
    switch (uMsg)
    {
        case WM_CREATE:
            /* need to allocate a control block */
            /* Fixed, since this is temporary anyway */
            hgauge = GlobalAlloc(GMEM_FIXED, sizeof(GAUGE));
            if (!hgauge)
                return (0L);
            lpgauge = (LPGAUGE)GlobalLock(hgauge);
            if (!lpgauge)
                return (0L);

            /* hang on to this control block */
            SetWindowLong(hwnd, G_WW_PGAUGE, (DWORD)lpgauge);

            /* fill control block with defaults */
            lpgauge->wRange           = G_DEF_RANGE;
            lpgauge->wPosition        = G_DEF_POSITION;
            lpgauge->wOrientation     = G_DEF_ORIENTATION;
            lpgauge->wWidth3D         = G_DEF_WIDTH3D;
            lpgauge->wWidthBezelFace  = G_DEF_BEZELFACE;
            lpgauge->rgbTextColor     = rgbDefTextColor;
            lpgauge->rgbBkColor       = rgbDefBkColor;

            /* use system font */
            SendMessage(hwnd, WM_SETFONT, NULL, 0L);

            /* go to DefWindowProc() to finish the job */
            break;

        case WM_DESTROY:
            /* get rid of the control's memory */
            if (lpgauge) {
                GlobalUnlock((HANDLE)hgauge);
                GlobalFree((HANDLE)hgauge);
                hgauge = 0;
                lpgauge = NULL;
                SetWindowLong(hwnd, G_WW_PGAUGE, (DWORD)lpgauge);
            }
            break;

        case G_GETPOSITION:
            return (lpgauge->wPosition);

        case G_GETRANGE:
            return (lpgauge->wRange);

        case G_GETORIENTATION:
            return (lpgauge->wOrientation);

        case G_GETWIDTH3D:
            return (lpgauge->wWidth3D);

        case G_GETBEZELFACE:
            return (lpgauge->wWidthBezelFace);

        case G_GETBKCOLOR:
            return (lpgauge->rgbTextColor);

        case G_GETFGCOLOR:
            return (lpgauge->rgbBkColor);

        case G_SETBKCOLOR:
            lpgauge->rgbBkColor = lParam;
            return (0L);

        case G_SETFGCOLOR:
            lpgauge->rgbTextColor = lParam;
            return (0L);


        case G_SETPOSITION:
            lpgauge->wPosition = wParam;

gForceRepaint:
            GetClientRect(hwnd, &rc);
            if ((GetWindowLong(hwnd, GWL_STYLE) & GS_3D) && fSupport3D)
            {
                wParam = (2 * lpgauge->wWidth3D) +
                            lpgauge->wWidthBezelFace + 2;
            }

            else
                wParam = 1;

	    InflateRect(&rc, ~(wParam), ~(wParam));
            InvalidateRect(hwnd, &rc, FALSE);
            UpdateWindow(hwnd);
            return (0L);

        case G_SETRANGE:
            lpgauge->wRange = wParam;
            goto gForceRepaint;

        case G_SETORIENTATION:
            lpgauge->wOrientation = wParam;
            goto gForceRepaint;

        case G_SETWIDTH3D:
            lpgauge->wWidth3D = wParam;

gForceRepaint3D:
            InvalidateRect(hwnd, NULL, FALSE);
            UpdateWindow(hwnd);
            return (0L);

        case G_SETBEZELFACE:
            lpgauge->wWidthBezelFace = wParam;
            goto gForceRepaint3D;

        case G_SETDELTAPOS:
            (signed)lpgauge->wPosition += (signed)wParam;
            goto gForceRepaint;

        case WM_PAINT:
            BeginPaint(hwnd, &ps);
            gaugePaint(hwnd, ps.hdc);
            EndPaint(hwnd, &ps);
            return (0L);

        case WM_GETFONT:
            hFont = lpgauge->hFont;

            /* if system font, then return NULL handle */
            return ((hFont == GetStockObject(SYSTEM_FONT)) ? NULL : hFont);

        case WM_SETFONT:
            /* if NULL hFont, use system font */
            if (!(hFont = (HFONT)wParam))
                hFont = GetStockObject(SYSTEM_FONT);

            lpgauge->hFont = hFont;

            /* redraw if indicated in message */
            if ((BOOL)lParam)
            {
                InvalidateRect(hwnd, NULL, TRUE);
                UpdateWindow(hwnd);
            }
            return (0L);
    } /* switch () */

    /* let the dialog mangler take care of this message */
    return (DefWindowProc(hwnd, uMsg, wParam, lParam));
} /* gaugeWndProc() */


/** EOF: gauge.c **/
