/*****************************************************************************
**
**    Name:    event.h
** 
**    Description: 
**       Routines to set up and manage event specifications.
**
**    Status: PRELIMINARY
**
**    $Log:   S:/tbird/arcm306/inc/event.h_v  $
** 
**    Rev 1.0   07 Sep 1995 10:25:56   gene
** Initial revision.
**  
**     Rev 1.7   21 Oct 1992 06:24:54   mindy
**  changed U8 FAR * to LPSTR to remove warnings
**  
**     Rev 1.6   03 Aug 1992 07:02:12   mindy
**  added end addr function routines
**  
**     Rev 1.5   25 Jun 1992 12:46:04   mindy
**  removed two init routines not being used
**  
**     Rev 1.4   16 Jun 1992 08:43:10   mindy
**  added Mask and StartAndMask get and set routines
**  
**     Rev 1.3   07 Jan 1992 10:01:04   ernie
**  changed to use BOOLEAN not BOOL
**  
**     Rev 1.2   01 Nov 1991 09:23:36   mindy
**  changes to match what is implemented and needed
**  
**     Rev 1.1   30 Jul 1991 13:52:04   nghia
**  added changes for "basic system release"
 * 
 *    Rev 1.0   14 Jun 1991 13:36:38   jim
 * Initial revision.
**
**    $Header:   S:/tbird/arcm306/inc/event.h_v   1.0   07 Sep 1995 10:25:56   gene  $
**
*****************************************************************************/

#define _EVENT_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _BASETYPE_
#include "basetype.h"
#endif

#ifndef _EVTTMPLT_
#include "evttmplt.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

typedef enum {END_ADDR_FUNC, END_LEN_FUNC} END_FUNC;

typedef struct {
   U32 lo;
   U32 hi;
   U32 mask;
   U8  not;
} EVENT_GROUP_INFO;

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

/*****************************************************************************
** 
**    EvtOpenEvent
** 
**    Description:
**       Returns a descriptor for a "copy" of the event server.
**    templateName indicates which template to use to verify
**    valid event field names.
** 
**    Parameters:
**       input:
**          templateName:     template to use 
**       output:
**          id:               id number assigned to calling routine.
**
*****************************************************************************/
RETCODE EXPORT EvtOpenEvent(DESCRIPTOR FAR *id, LPSTR templateName);


/*****************************************************************************
** 
**    EvtCloseEvent
** 
**    Description:
**       Informs event server is not longer is use.
** 
**    Parameters:
**       input:
**          id:               id number of descriptor to close
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtCloseEvent(DESCRIPTOR id);

/*****************************************************************************
** 
**    EvtRestoreEvents
** 
**    Description:
**       Reads in all the event specifications from fileName.  Events
**       are combined with existing events.  If an event already exists
**       the old one will be overwritten with the event from the file.
** 
**    Parameters:
**       input:
**          fileName:         name of file
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtRestoreEvents(LPSTR fileName);

/*****************************************************************************
** 
**    EvtSaveAsEvents
** 
**    Description:
**       Saves all the event specifications currently in memory.
** 
**    Parameters:
**       input:
**          fileName:         name of file
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSaveAsEvents(LPSTR fileName);

/* !!! In later versions will need to set and get template name */

/*****************************************************************************
** 
**    EvtSetActiveEvent
** 
**    Description:
**       Sets the active event.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          eventName:        name of event to become "active" event
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetActiveEvent(DESCRIPTOR id, LPSTR eventName);

/*****************************************************************************
** 
**    EvtGetActiveEvent
** 
**    Description:
**       Gets the currently "active" event's name.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for event name.
**       output:
**          eventName:        name of "active" event
**
*****************************************************************************/
RETCODE EXPORT EvtGetActiveEvent(DESCRIPTOR id,LPSTR eventName,U16 maxLen);

/*****************************************************************************
** 
**    EvtDoesEventExist
** 
**    Description:
**       Searches the defined events to see if the incoming event
**     is defined (exists).
** 
**    Parameters:
**       input:
**          eventName:        name of event
**       output:
**          exists:           returns TRUE if event is defined.
**
*****************************************************************************/
RETCODE EXPORT EvtDoesEventExist(LPSTR eventName, BOOLEAN FAR *exists);

/*****************************************************************************
** 
**    EvtClearEvent
** 
**    Description:
**       Clears the all of the conditions for the currently active
**       event.  If is handled by making all of the template field
**       conditions don't care (i.e.: always true).
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtClearEvent(DESCRIPTOR id);

/*****************************************************************************
** 
**    EvtDeleteEvent
** 
**    Description:
**       Removes the currently active event from memory.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtDeleteEvent(DESCRIPTOR id);

/*****************************************************************************
** 
**    EvtDeleteAllEvents
** 
**    Description:
**       Removes all events from system.
** 
**    Parameters:
**       input:
**          none
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtDeleteAllEvents(VOID);

/*****************************************************************************
** 
**    EvtSetActiveField
** 
**    Description:
**       Sets the active template field name.  All event routines will now
**       reference this template field.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          fieldName:        template field name
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetActiveField(DESCRIPTOR id, LPSTR fieldName);

/*****************************************************************************
** 
**    EvtGetActiveField
** 
**    Description:
**       Returns the name of the currently active template field.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for field name.
**       output:
**          fieldName:        template field name
**
*****************************************************************************/
RETCODE EXPORT EvtGetActiveField(DESCRIPTOR id, LPSTR fieldName, 
                                 U16 maxLen);

/*****************************************************************************
** 
**    EvtSetNotCondition
** 
**    Description:
**       Sets the event condition for template field.  Either event 
**       is programmed to match the event condition (not==TRUE) or to 
**       match the negated event condition.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          not:              match event or not
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetNotCondition(DESCRIPTOR id, BOOLEAN not);

/*****************************************************************************
** 
**    EvtGetNotCondition
** 
**    Description:
**       Returns the event condition for the currently active template field.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          not:              match event or not
**
*****************************************************************************/
RETCODE EXPORT EvtGetNotCondition(DESCRIPTOR id, BOOLEAN FAR *not);

/*****************************************************************************
** 
**    EvtSetEndFunction
** 
**    Description:
**       Sets the function of the template field's ending value.
**       Either the ending value acts as the ending address or as 
**       an offset that is added to the starting value.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          func:             end's function
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetEndFunction(DESCRIPTOR id, END_FUNC func);

/*****************************************************************************
** 
**    EvtGetEndFunction
** 
**    Description:
**       Returns the function of the template field's ending value.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          func:             ending value's function
**
*****************************************************************************/
RETCODE EXPORT EvtGetEndFunction(DESCRIPTOR id, END_FUNC FAR *func);

/*****************************************************************************
** 
**    EvtSetEnd
** 
**    Description:
**       Sets the event's ending condition for the template field.
**    All "addresses" are sent in as strings for the server to parse.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          end:              event's ending condition
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetEnd(DESCRIPTOR id, LPSTR end);

/*****************************************************************************
** 
**    EvtGetEnd
** 
**    Description:
**       Returns the event's ending condition for the template field.
**    Value is passed back as a string which would includes any don't
**    cares as "x"s.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for end value.
**       output:
**          end:              event's ending condition
**
*****************************************************************************/
RETCODE EXPORT EvtGetEnd(DESCRIPTOR id, LPSTR end, U16 maxLen);

/*****************************************************************************
** 
**    EvtGetMask
** 
**    Description:
**       Returns the event's mask for the template field. Value is passed
**    back as a string.  There are NO X (don't care).
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for mask value.
**       output:
**          mask:             event's mask
**
*****************************************************************************/
RETCODE EXPORT EvtGetMask(DESCRIPTOR id, LPSTR mask, U16 maxLen);

/*****************************************************************************
** 
**    EvtSetMask
** 
**    Description:
**       Sets the event's mask for the template field. All "addresses" are
**    sent in as strings for the server to parse.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          mask:             event's mask
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetMask(DESCRIPTOR id, LPSTR mask);

/*****************************************************************************
** 
**    EvtGetStart
** 
**    Description:
**       Returns the event's starting condition for the template field.
**    Value is passed back as a string.  There are NO X (don't care).
**    This routine works on RANGE fields.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for start value.
**       output:
**          start:            event's starting condition
**
*****************************************************************************/
RETCODE EXPORT EvtGetStart(DESCRIPTOR id, LPSTR start, U16 maxLen);

/*****************************************************************************
** 
**    EvtSetStart
** 
**    Description:
**       Sets the event's starting condition for the template field.
**    All "addresses" are sent in as strings for the server to parse.
**    This routine works on RANGE fields.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          start:            event's starting condition
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetStart(DESCRIPTOR id, LPSTR start);

/*****************************************************************************
** 
**    EvtGetStartAndMask
** 
**    Description:
**       Returns the event's starting condition for the template field.
**    Value is passed back as a string which would includes any don't
**    cares as "x"s.  This routine works on VALUE fields.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          maxLen:           max length of caller's storage for start value.
**       output:
**          start:            event's starting condition
**
*****************************************************************************/
RETCODE EXPORT EvtGetStartAndMask(DESCRIPTOR id, LPSTR start, U16 maxLen);

/*****************************************************************************
** 
**    EvtSetStartAndMask
** 
**    Description:
**       Sets the event's starting condition for the template field.
**    All "addresses" are sent in as strings for the server to parse.
**    This routine works on VALUE fields.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          start:            event's starting condition
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT EvtSetStartAndMask(DESCRIPTOR id, LPSTR start);

/*****************************************************************************
** 
**    EvtGetEventGroupInfo
** 
**    Description:
**       Returns the event's start, end, mask and not condition for the 
**    requested xilinx group.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**          xilinx:           which xilinx group of event.
**       output:
**          info:             xilinx data bits for an event
**
*****************************************************************************/
RETCODE EXPORT EvtGetEventGroupInfo(DESCRIPTOR id, XILINX_NUMBER xilinx,
                                    EVENT_GROUP_INFO FAR *info);

/*****************************************************************************
** 
**    EvtNumDefinedEvents
** 
**    Description:
**       Returns the number of events defined in the specified event server.
**    When this routine is called it resets the pointer for the 
**    EvtGetNextEvent calls.  Thus to get a list of all the events in the
**    server the call firsts calls EvtNumDefinedEvents and then calls
**    EvtGetNextEvent for that many times or until EvtGetNextEvent returns
**    a NULL event name.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          num:              number of defined events in server
**
*****************************************************************************/
RETCODE EXPORT EvtNumDefinedEvents(DESCRIPTOR id, U16 FAR *num);

/*****************************************************************************
** 
**    EvtGetNextEvent
** 
**    Description:
**       Returns the next defined event's name. See EvtNumDefinedEvents
**    for details.
** 
**    Parameters:
**       input:
**          id:               id number assigned to calling routine.
**       output:
**          num:              number of defined events in server
**
*****************************************************************************/
RETCODE EXPORT EvtGetNextEvent(DESCRIPTOR id, LPSTR name, U16 maxLen);

/*********************************** E O F **********************************/
