/*****************************************************************************
** Name:  varservr.h
**
** Description:
**    Routines to access the variable server for variable information 
**    (type, name, value and address).
**
** Status:   PRELIMINARY
**
** $Log:   S:/tbird/arcm306/inc/varservr.h_v  $
** 
**    Rev 1.0   14 Sep 1995 14:22:42   kevin
** Initial revision.
** 
**    Rev 1.13   12 May 1993 16:40:08   ron
** pass start col for edit text (prototype change)
** 
**    Rev 1.12   08 Sep 1992 11:03:58   marilyn
** Removed variable event.  See events.h.
** 
**    Rev 1.11   28 May 1992 11:23:44   marilyn
** Added VSReadStackVar
** 
**    Rev 1.10   08 May 1992 16:31:42   marilyn
** Added new interfaces, VSGetBufferSize and VSGetLinesForVar.
** 
**    Rev 1.9   13 Mar 1992 09:56:00   marilyn
** Added events defines.
** 
**    Rev 1.8   13 Mar 1992 09:33:22   marilyn
** Added InitServer and TerminateServer.
** 
**    Rev 1.7   28 Feb 1992 10:28:22   marilyn
** Removed all references to context.h
** 
**    Rev 1.6   18 Feb 1992 09:59:22   marilyn
** Changed definition of continue to be a word sized.
** 
**    Rev 1.5   17 Jan 1992 07:59:32   marilyn
** integration version
** 
**    Rev 1.4   12 Dec 1991 13:25:44   marilyn
** Implemented additional parameter for VSNewValue.
** 
**    Rev 1.3   19 Nov 1991 09:01:18   marilyn
** Revision of interface.  Added VSOpenCompositeVar, VSOpenComponentVar,
** VSInterpretMemAsType, VSInterpretVarAsType, and changed VSOpenVar.
**
**    Rev 1.2   31 Oct 1991 13:03:16   marilyn
** Defined values for enums.
** 
**    Rev 1.1   29 Oct 1991 12:02:08   marilyn
** Added additional typedefs for variable selection information.
** 
**    Rev 1.0   16 Oct 1991 10:06:02   marilyn
** Initial revision.
** 
** $Header:   S:/tbird/arcm306/inc/varservr.h_v   1.0   14 Sep 1995 14:22:42   kevin  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/

#define _VARSERVR_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_ 
#include "basewind.h"
#endif

#ifndef _SYMBLSVR_
#include "symblsvr.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define VS_ANONYMOUS   0
#define VS_MAX_BUFFER_LINES   20
#define VS_MAX_LINE_CHARS   132
#define VS_TMALLOC 0


typedef struct  {
   WORD allocIndicator;       /* VS_TMALLOC or GlobalAlloc handle */
   WORD continued;
   DESCRIPTOR bufnum;
   U32 textLength;
   U8 text[VARIABLE_SIZED];
}  VS_DISPLAY_BUFFER_TYPE;


typedef VS_DISPLAY_BUFFER_TYPE   FAR *VS_DISPLAY_PTR;

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

/***************************************************************************
**  InitServer
**
**  Purpose: Initialize the variable server.  This routine must be called
**     before any other routines in variable server can be used.
**
**  Parameters:
**     Input: none
**     Output: none
**
******************************************************************************/
RETCODE EXPORT InitServer(VOID);

/*****************************************************************************
**  TerminateServer
**
**  Purpose: Cleanup and terminate the variable server.  After calling this
**     routine, no other variable server routines should be called.
**
**  Parameters:
**     Input: none
**     Output: none
**
*****************************************************************************/
RETCODE EXPORT TerminateServer(VOID);


/******************************************************************************
**
**   VSStartSession
**
**   Purpose:
**      To register a calling client with a unique id for access to the 
**      variable server.
**
**    Input parameters:
**      winHandle:  the calling window's id
**
**   Output parameters:
**      VSSessionId:  client id
**      radixMenuBufPtr:  a pointer to the buffer of type strings allocated 
**      by the Variable Server for the radix menu
**
**   Error:  
**      Returns GOOD if no error.
**      Returns ER_VS_CLIENT_DISCREP if a client is already registered 
**      (client discrepancy).
**      Returns ER_OUT_OF_MEMORY if there is no more space in the client 
**      table and the table can not be made larger.
**
******************************************************************************/
RETCODE EXPORT  VSStartSession(HWND  winHandle,
                               DESCRIPTOR *VSSessionId,
                               LPSTR *radixMenuBufPtr);


/****************************************************************************
**
**   VSEndSession
**
**   Purpose:
**      To shut down a client's access to the variable server.
**
**   Input parameters:
**      VSSessionId:  client id
**
**   Output parameters:  none
**   
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_CLIENT_DISCREP if the client is not registered 
**      (client discrepancy).
**
*******************************************************************************/
RETCODE EXPORT  VSEndSession(DESCRIPTOR VSSessionId);


/***************************************************************************
**
**   VSOpenVar
**
**   Purpose:
**      To place the selected variable into the variable table for tracking 
**      and to identify it for future reference.  Intended for use with
**      simple type variables.
**
**   Input parameters:
**      VSSessionId:  client id
**      symbol :  Symbol descriptor of selected variable
**      interActive:  flag to indicate session type or preference
**
**   Output parameters:
**      VSVarId:  variable id
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found.
**      Returns ER_VS_MULTI_VAR if multiple targets are found and interActive 
**      is FALSE.
**      Returns ER_OUT_OF_MEMORY if there is no more space available in the 
**      variable table and it can not be make larger.
**
**      NOTE:  The Variable Server will post a dialog window for the user if 
**      multiple variables are found and interActive is TRUE.
**   
**********************************************************************/
RETCODE EXPORT  VSOpenVar(DESCRIPTOR VSSessionId,
                          SYM_DESCRIPTOR symbol,
                          BOOLEAN interActive,
                          DESCRIPTOR *VSVarId);


/***************************************************************************
**
**   VSOpenCompositeVar
**
**   Purpose:
**      To place the selected variable into the variable table for tracking 
**      and to identify it for future reference.  Intended for use with
**      composite type variables (e.g. a.b.c->x->y.z).
**
**   Input parameters:
**      VSSessionId:  client id
**      symbol :  Symbol descriptor of the root variable (e.g. a)
**      varString : string containing entire name (e.g. "a.b.c->x->y.z")
**      selectedVarStartOffset : offset of selected portion of variable 
**         (e.g. x->y.z)
**      interActive:  flag to indicate session type or preference
**
**   Output parameters:
**      VSVarId:  variable id
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found.
**      Returns ER_VS_MULTI_VAR if multiple targets are found and interActive 
**      is FALSE.
**      Returns ER_OUT_OF_MEMORY if there is no more space available in the 
**      variable table and it can not be make larger.
**
**      NOTE:  The Variable Server will post a dialog window for the user if 
**      multiple variables are found and interActive is TRUE.
**   
**********************************************************************/
RETCODE EXPORT  VSOpenCompositeVar(DESCRIPTOR VSSessionId,
                                   SYM_DESCRIPTOR symbol,
                                   LPSTR varString,
                                   U16 selectedVarStartOffset,
                                   BOOLEAN interActive,
                                   DESCRIPTOR *VSVarId);


/***************************************************************************
**
**   VSOpenComponentVar
**
**   Purpose:
**      To place the selected variable into the variable table for tracking 
**      and to identify it for future reference.  Intended for use with
**      the components of previously opened variables.
**
**   Input parameters:
**      VSSessionId:  client id
**      rootVarId : id of the root variable
**      bufferNum : buffer number containing the component selected
**      lineNum : line number of component selected
**      varString : string containing selected component including 
**         encoding markers
**      interActive:  flag to indicate session type or preference
**
**   Output parameters:
**      VSVarId:  variable id
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found.
**      Returns ER_VS_MULTI_VAR if multiple targets are found and interActive 
**      is FALSE.
**      Returns ER_OUT_OF_MEMORY if there is no more space available in the 
**      variable table and it can not be make larger.
**
**      NOTE:  The Variable Server will post a dialog window for the user if 
**      multiple variables are found and interActive is TRUE.
**   
**********************************************************************/
RETCODE EXPORT  VSOpenComponentVar(DESCRIPTOR VSSessionId,
                                   DESCRIPTOR rootVarId,
                                   U16 bufferNum,
                                   U16 lineNum,
                                   LPSTR varString,
                                   BOOLEAN interActive,
                                   DESCRIPTOR *VSVarId);


/***************************************************************************
**
**   VSInterpretVarAsType
**
**   Purpose:
**      To place the selected variable into the variable table 
**      to identify it for future reference and to display it in the
**      requested type format.
**
**   Input parameters:
**      VSSessionId:  client id
**      symbol :  Symbol descriptor of selected variable
**      typeIndex : type index to use for display format 
**      interActive:  flag to indicate session type or preference
**
**   Output parameters:
**      VSVarId:  variable id
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found.
**      Returns ER_VS_MULTI_VAR if multiple targets are found and interActive 
**      is FALSE.
**      Returns ER_OUT_OF_MEMORY if there is no more space available in the 
**      variable table and it can not be make larger.
**
**      NOTE:  The Variable Server will post a dialog window for the user if 
**      multiple variables are found and interActive is TRUE.
**   
**********************************************************************/
RETCODE EXPORT  VSInterpretVarAsType(DESCRIPTOR VSSessionId,
                                     SYM_DESCRIPTOR symbol,
                                     TYPE_INDEX typeIndex,
                                     BOOLEAN interActive,
                                     DESCRIPTOR *VSVarId);


/***************************************************************************
**
**   VSInterpretMemAsType
**
**   Purpose:
**      To identify the selected memory as a variable of the selected
**      type and save it for future reference and to display it in the
**      requested type format.
**
**   Input parameters:
**      VSSessionId:  client id
**      memaddr : address in memory to identify as a variable
**      typeIndex : type index to use for display format 
**      interActive:  flag to indicate session type or preference
**
**   Output parameters:
**      VSVarId:  variable id
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found.
**      Returns ER_VS_MULTI_VAR if multiple targets are found and interActive 
**      is FALSE.
**      Returns ER_OUT_OF_MEMORY if there is no more space available in the 
**      variable table and it can not be make larger.
**
**      NOTE:  The Variable Server will post a dialog window for the user if 
**      multiple variables are found and interActive is TRUE.
**   
**********************************************************************/
RETCODE EXPORT  VSInterpretMemAsType(DESCRIPTOR VSSessionId,
                                     DESCRIPTOR memaddr,
                                     TYPE_INDEX typeIndex,
                                     BOOLEAN interActive,
                                     DESCRIPTOR *VSVarId);


/**********************************************************************
**
**   VSCloseVar
**
**   Purpose:
**   To remove the selected variable from the variable table.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**
**   Output parameters:  none
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client is denied access to the variable.
**
******************************************************************************/
RETCODE EXPORT  VSCloseVar(DESCRIPTOR VSSessionId,
                           DESCRIPTOR VSVarId);

/*******************************************************************************
**
**   VSReadStackVar
**
**   Purpose:
**      For stack variables only.
**      Given the variable id and frame number, return a pointer to the
**      buffer containing the formatted display of the variable.  Type
**      information is also encoded in the buffer.  If the variable
**      requires more than one buffer to display all the associated
**      text then the continuation flag in the display buffer is set to
**      TRUE.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**      frameNum: the stack frame number associated with the variable.
**      bufferNum: the buffer number of the variable
**
**   Output parameters:
**      displayBufPtr:  pointer to buffer containing formatted display of
**      variable, buffer number and continuation indicator
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client is denied access to variable.
**      Returns ER_VS_VAR_INACTIVE if variable is inactive.
**      NOTE:  Returns displayBufPtr set to NULL when error is indicated.
**
******************************************************************************/
RETCODE EXPORT  VSReadStackVar(DESCRIPTOR VSSessionId,
                               DESCRIPTOR VSVarId,
                               DESCRIPTOR bufferNum,
                               U8 frameNum,
                               VS_DISPLAY_PTR FAR *displayBufPtr);


/*******************************************************************************
**
**   VSReadVar
**
**   Purpose:
**      Given the variable id, return a pointer to the buffer containing
**      the formatted display of the variable.  Type information is also
**      encoded in the buffer.  If the variable requires more than one buffer
**      to display all the associated text then the continuation flag in the
**      display buffer is set to TRUE.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**
**   Output parameters:
**      displayBufPtr:  pointer to buffer containing formatted display of
**      variable, buffer
**      number and continuation indicator
**      NOTE:  buffer number is always 0 for VSReadVar().
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client is denied access to variable.
**      Returns ER_VS_VAR_INACTIVE if variable is inactive.
**      NOTE:  Returns displayBufPtr set to NULL when error is indicated.
**
******************************************************************************/
RETCODE EXPORT  VSReadVar(DESCRIPTOR VSSessionId,
                          DESCRIPTOR VSVarId,
                          VS_DISPLAY_PTR FAR *displayBufPtr);


/*****************************************************************************
**
**   VSReadVarAtNBuf
**
**   Purpose:
**      Given the variable id and the buffer number, return a display buffer
**      containing the formatted variable information starting at N buffer
**      into the variable. This routine is used to read variables that
**      require more than one buffer to display all the information text.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**      bufferNum:  the buffer number where formatting is to start
**
**   Output parameters:
**      displayBufPtr:  pointer to the buffer containing formatted display
**      information of the requested variable starting at N buffer, buffer
**      number, and continuation indicator.
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client is denied access to the variable.
**      Returns ER_VS_VAR_INACTIVE if variable is inactive.
**      Returns ER_VS_BUFNUM_INVALID if buffer number is invalid for the
**      variable.
**      NOTE:  Returns displayBufPtr set to NULL when error is indicated.
**
*****************************************************************************/
RETCODE EXPORT  VSReadVarAtNBuf(DESCRIPTOR VSSessionId,
                                DESCRIPTOR VSVarId,
                                DESCRIPTOR buffernum,
                                VS_DISPLAY_PTR FAR *displayBufPtr);


/****************************************************************************
**
**   VSEditValue
**
**   Purpose:
**      To edit the value field of the selected variable.
**
**   Input parameters:
**      VSSessionId:  client id
**      VsVarId:  variable id
**      bufferNum:  buffer number where value field is located
**      lineNum:  line number in the buffer where value field is located
**      fieldWidth: set to the length of the string
**      charSetBufPtr: a string of the selected field including the markers
**      startCol: the starting column for the selected field (inc. markers)
**      
**   Output parameters:
**      fieldWidth:  number of characters allowed in value field
**      charSetBufPtr:  pointer to buffer containing the legal character 
**      set for value field
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable 
**      table.
**      Returns ER_VS_ACCESS_DENIED if client does not have access to the 
**      variable.
**      Returns ER_VS_BUFNUM_INVALID if buffer number is invalid for variable.
**      Returns ER_VS_LINENUM_INVALID if line number is invalid for the variable.
**      NOTE:  If an error is returned charSetBufPtr contains replacement
**      text for the value.
**
******************************************************************************/
RETCODE EXPORT  VSEditValue(DESCRIPTOR VSSessionId,
                            DESCRIPTOR VSVarId,
                            DESCRIPTOR bufferNum,
                            U16 lineNum,
                            U16 *fieldWidth,
                            LPSTR *charSetBufPtr,
                            U16 startCol);


/***************************************************************************
**
**   VSNewValue
**
**   Purpose:
**      To save the new edited value for the selected variable.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**      bufferNum:  buffer number where value field is located
**      lineNum:  line number in the buffer where value field is located
**      editFieldCharOffset: character offset for first character of editted
**         value in newValueText
**      newValueText:  string containing the text of the new value including
**         encoding markers
**
**   Output parameters:
**      displayBufPtr: pointer to buffer containing new formatted display
**      information, buffer number and continuation indicator
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client does not have access to the
**      variable.
**      Returns ER_VS_BUFNUM_INVALID if buffer number is invalid for variable.
**      Returns ER_VS_LINENUM_INVALID if line number is invalid for the variable.
**      Returns ER_VS_VALU_EDIT_FAIL if new value can not be saved and a
**      replacement display buffer.
**
****************************************************************************/
RETCODE EXPORT  VSNewValue(DESCRIPTOR VSSessionId,
                           DESCRIPTOR VSVarId,
                           DESCRIPTOR bufferNum,
                           U16 lineNum,
                           U16 editFieldCharOffset,
                           LPSTR newValueText,
                           VS_DISPLAY_PTR FAR *displayBufPtr);


/**************************************************************************
**
**   VSVarSymbol
**
**   Purpose:
**      To retrieve symbol information for the selected variable.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId:  variable id
**
**   Output parameters:
**      varSymbol:  descriptor of the variable from the symbol table
**
**   Error:
**      Returns GOOD if no error.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client does not have access to the
**      variable.
**      Returns ER_VS_VAR_INACTIVE if variable is inactive.
**
****************************************************************************/
RETCODE EXPORT  VSVarSymbol(DESCRIPTOR VSSessionID,
                            DESCRIPTOR VSVarId,
                            SYM_DESCRIPTOR *varSymbol);


/****************************************************************************
**
**   VSChangeRadix
**
**   Purpose:
**      To change the radix for a given type.
**
**   Input parameters:
**      VSSessionId:  client id
**      radixIndex:  index of type string selected for radix change
**
**   Output parameters:
**      NOTE:  no parameters are returned but this function assumes call by a
**      window client and provides radix selection via a pop up window.
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_CLIENT_DISCREP if client is not registered.
**      Returns ER_VS_RADIX_CHNG_FAIL if change was unsuccessful.
**
*****************************************************************************/
RETCODE EXPORT  VSChangeRadix(DESCRIPTOR VSSessionId,
                              U16 radixIndex);

/****************************************************************************
**
**   VSCallback
**
**   Purpose:
**      Function to handle event notification callback.
**
**   Input Parameters:
**      eventNum:  the identifier of the event causing the callback
**
**   Output Parmeters:
**      none
**
***************************************************************************/
RETCODE EXPORT VSCallback(U32 eventNum);

/***************************************************************************
**
**   VSGetBufferSize
**
**   Purpose:
**      To retrieve the size of variable server display buffers.
**
**   Input parameters:
**      VSSessionId:  client id
**
**   Output parameters:
**      bufferSize : number of lines in a display buffer.
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_CLIENT_DISCREP if client is not registered.
**
***************************************************************************/
RETCODE EXPORT  VSGetBufferSize(DESCRIPTOR VSSessionId,
                                U8 *bufferSize);

/***************************************************************************
**
**   VSGetLinesForVar
**
**   Purpose:
**      To retrieve the number of lines required to display a single var.
**
**   Input parameters:
**      VSSessionId:  client id
**      VSVarId : variable id
**
**   Output parameters:
**      numLines : number of lines required to display the var.
**
**   Error:
**      Returns GOOD if no errors.
**      Returns ER_VS_CLIENT_DISCREP if client is not registered.
**      Returns ER_VS_VAR_NOT_FOUND if variable is not found in the variable
**      table.
**      Returns ER_VS_ACCESS_DENIED if client does not have access to the
**      variable.
**
***************************************************************************/
RETCODE EXPORT  VSGetLinesForVar(DESCRIPTOR VSSessionId,
                                 DESCRIPTOR VSVarId,
                                 U32 *numLines);

/******************************** E O F ***********************************/
