/***************************************************************************
**  Name:  emumem.c
**
**  Description:
**     Michelle memory-group routines.
**     include:
**        "Read N bytes memory from Michelle -- emuReadMemN()"
**        "Write N bytes memory to Michelle  -- emuSetMemN()"
**        "Memory data comparing             -- emuCompare()"
**        "Memory data coping                -- emuCopy()"
**        "Memory data filling               -- emuFill()"
**        "Memory data testing               -- emuTest()"
**        "Memory data searching             -- emuSearch()"
**        "Memory data checksum             -- emuChecksum()"
**
**  Status:  preliminary
**
**  $Log:   S:/tbird/arcm302/mpfw_302/emumem.c_v  $
** 
**    Rev 1.1   10 Jun 1996 09:57:06   gene
** 
**    Rev 1.0   10 Jun 1996 08:50:40   gene
** Initial revision.
** 
**    Rev 1.9   29 Jan 1996 13:16:40   gene
** change emuTest(), return value of error
** 
**    Rev 1.8   11 Jan 1996 10:37:00   gene
** 
**    Rev 1.7   11 Jan 1996 09:51:32   kevin
** modified prototype of processHeadPos() and output the RET_ADDR_LEN of
** canTemp no matter what exceptBreak is.
** 
**    Rev 1.3   12 Dec 1995 09:24:44   gene
** fixed emuCopy/copyTargetMem bug
** 
**    Rev 1.4   30 Nov 1995 14:42:02   kevin
** in setmemn(), check exceptBreak first rather than checking status
** 
**    Rev 1.3   21 Nov 1995 15:50:12   kevin
** checked internal register correctly and turned verify off in internal
** register area
** 
**    Rev 1.2   26 Sep 1995 08:57:04   gene
** change emuCompare()
** 
**    Rev 1.1   22 Sep 1995 09:12:40   kevin
** change MemCompare() status case
** 
**    Rev 1.0   07 Sep 1995 10:47:26   gene
** Initial revision.
** 
** 
**  $Header:   S:/tbird/arcm302/mpfw_302/emumem.c_v   1.1   10 Jun 1996 09:57:06   gene  $
**
** Copyright (C) 1992 Microtek International, Inc.
**
****************************************************************************/

                   /****************************
                   *                           *
                   *       INCLUDE FILES       *
                   *                           *
                   ****************************/

#ifndef _EMU_MEM_
#include "emumem.h"
#endif

#ifndef _EMU_BASE_TYPES_
#include "emutypes.h"
#endif

#ifndef _EMU_LIB_
#include "emulib.h"
#endif

#ifndef _LLFW_
#include "emullfw.h"
#endif

#ifndef _EMU_EXTERNAL_
#include "emuext.h"
#endif

#include <string.h>


// diff = (U8 *)&canTemp;
// pos = outputStreamLen;
// outputStreamLen = pos + 1;/* Output stream length */

//  #define OutStream(diff,pos,streamLen) \
//           outputStream[pos++] = streamLen;\
//
//memcpy(&outputStream[pos],diff,streamLen);/* error address */\
//           pos += streamLen;\
//           outputStream[pos++] = 0x00;

                   /****************************
                   *                           *
                   *     LOCAL DEFINITIONS     *
                   *                           *
                   ****************************/
#define RET_ADDR_LEN     8
                   /****************************
                   *                           *
                   *    EXTERNAL VARIABLES     *
                   *                           *
                   ****************************/

                   /****************************
                   *                           *
                   *     LOCAL PROTOTYPES     *
                   *                           *
                   ****************************/
STATUS processHeadPos(ADDR *addr, U16 *dataLen, U8 **buff,U16 verifyFlag,
                      BOOLEAN *processTail);
STATUS ReadMem(ADDR *addr, U8 *buff, U16 len);
STATUS WriteMem(ADDR *addr, U16 dataLen, U8 *buff);
STATUS FillMem(ADDR *addr, U32 addrLen, U8 *buff, U16 len, U16 size, U16 verifyFlag);
                   /***************************
                   *                          *
                   *      EXECUTABLE CODE     *
                   *                          *
                   ****************************/
STATUS processHeadPos(ADDR *addr, U16 *dataLen, U8 **buff,U16 verifyFlag,
          BOOLEAN *processTail) {
   STATUS status = GOOD;
   U8     tmpBuff[2];

   if (addr->pos % 2) { // process head
      DumpByteMem(addr->pos-1,addr->space,1,tmpBuff,ON);
      tmpBuff[1] = (*buff)[0];
      status = FillWordMem(addr->pos-1,addr->space,2,2,tmpBuff,verifyFlag,ON) & 0xff;
      addr->pos++;
      (*dataLen)--;
      (*buff)++;
   }
   if (*dataLen % 2) { // process tail
      *processTail = TRUE;
      (*dataLen)--;
   }
   return(status);
}

STATUS processTailPos(ADDR *addr,U8 buff,U16 verifyFlag) {
   STATUS status = GOOD;
   U8     tmpBuff[2];

   DumpByteMem(addr->pos+1,addr->space,1,&tmpBuff[1],ON);
   tmpBuff[0] = buff;
   status = FillWordMem(addr->pos,addr->space,2,2,tmpBuff,verifyFlag,ON) & 0xff;
   return(status);
}

/***************************************************************************
**
**  ChkReduceAddrLen
**
**  Description: Check the address range. If it's partial in the internal register area
**               then reduce the length..
**         Note: The internal register address --0xfffff000.
**
**  Parameters:
**     input:
**        addrpos -- unsigned long
**        addrLen -- unsigned long
**
**     output:
**        return InterFlag == NO ----> the range is all in the area.
**                         == YES ---> the range is partial in the area.
**
****************************************************************************/
//STATUS ChkReduceAddrLen(U32 addrpos, U16 addrSpace, U32 *addrLen) {
//U16 InterFlag;
//
//   InterFlag = NO;
//   if (addrSpace != SPACE_SD) {
//      if (addrpos >= INTER_REG_ADDR)
//         InterFlag = YES;
//      else if ((addrpos + *addrLen) > INTER_REG_ADDR)
//         *addrLen = INTER_REG_ADDR - addrpos;
//   }
//   return(InterFlag);
//}
/***************************************************************************
**
**  CopyTarget
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
STATUS CopyTargetMem(ADDR *addr1, U32 addrLen) {
STATUS status;
ADDR tmpAddr1, tmpAddr2;
U32 count,lp;
U8 memBuff[BUFF_LEN];

   status = GOOD;
   ClrViolFlg();
   count = addrLen/BUFF_LEN;
   tmpAddr1 = tmpAddr2 = *addr1;
   for (lp = 0; lp < count; lp++) {
      DisEmStop();
      status = ReadMem(&tmpAddr1,memBuff,(U16)BUFF_LEN);
      if (status) return(status);
      EnEmDisViol();
      status = WriteMem(&tmpAddr2,(U16)BUFF_LEN,memBuff);
      if (status) return(status);
      tmpAddr2.pos = tmpAddr1.pos;
   }
   count = addrLen % (U32)BUFF_LEN;
   if ((count != 0) && (status == GOOD)) {
      DisEmStop();
      status = ReadMem(&tmpAddr1,memBuff,(U16)count);
      if (status) return(status);
      EnEmDisViol();
      status = WriteMem(&tmpAddr2,(U16)count,memBuff);
      if (status) return(status);
   }
   return(status);
}

/***************************************************************************
**
**  ReadMem
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
STATUS ReadMem(ADDR *addr, U8 *buff, U16 dataLen) {
U16 lp,count;
U16 size,loadCan;
STATUS status;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   count = dataLen / (U16)CAN_BUFFER_NO;
   loadCan = ON;
   for (lp = 0; lp < count; lp++) {
      if (status != GOOD) return(status);
      DumpByteMem(addr->pos,addr->space,CAN_BUFFER_NO,&buff[lp*CAN_BUFFER_NO],loadCan);
      loadCan = OFF;
      if (exceptBreak == 0) addr->pos += (U32)CAN_BUFFER_NO;
      else return(exceptBreak);
   }
   count = dataLen % (U16)CAN_BUFFER_NO;
   if ((count != 0) && (status == GOOD)) {
      DumpByteMem(addr->pos,addr->space,count,&buff[lp*CAN_BUFFER_NO],ON);
   }
   return(exceptBreak);
}

/***************************************************************************
**
**  WriteMem
**
**  Description: Read a chunk memory from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
**  NOTE: if interFlag is on, it will not check verifyFlag at all.
****************************************************************************/
STATUS WriteMem(ADDR *addr, U16 dataLen, U8 *buff) {
   U16 verifyFlag, size;
   U16 lp;
   STATUS status;
   U16 count,loadCan; //,interFlag;
   U32 length;
   BOOLEAN processTail = FALSE;

   status = GOOD;
   length = (U32)dataLen;
//   interFlag = ChkReduceAddrLen(addr->pos,addr->space,&length);
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
//   if (interFlag) verifyFlag = 0;

   if (size != S_BYTE)
      status = processHeadPos(addr,&dataLen,&buff,verifyFlag,&processTail);
   count = dataLen / (U16)CAN_BUFFER_NO;
   loadCan = ON;

   for (lp = 0; lp < count; lp++) {
      if (status != GOOD) return(status);
      if (size == S_BYTE)
         status = FillByteMem(addr->pos,addr->space,CAN_BUFFER_NO,(U16)CAN_BUFFER_NO,
                &buff[lp*CAN_BUFFER_NO],verifyFlag,loadCan) & 0xff;
      else {
         status = FillWordMem(addr->pos,addr->space,CAN_BUFFER_NO,(U16)CAN_BUFFER_NO,
                &buff[lp*CAN_BUFFER_NO],verifyFlag,loadCan) & 0xff;
      }
      if (status) status = ERROR_WRITE;
      //else status = CheckViolation();
      if (status) return(status);
      loadCan = OFF;
      if (exceptBreak == 0) addr->pos += (U32)CAN_BUFFER_NO;
      else return(exceptBreak);
   }
   count = dataLen % (U16)CAN_BUFFER_NO;
   if ((count != 0) && (status == GOOD)) {
      if (size == S_BYTE)
         status = FillByteMem(addr->pos,addr->space,count,count,&buff[lp*CAN_BUFFER_NO],verifyFlag,ON) & 0xff;
      else {
         status = FillWordMem(addr->pos,addr->space,count,count,&buff[lp*CAN_BUFFER_NO],verifyFlag,ON) & 0xff;
      }
      if (status) status = ERROR_WRITE;
         addr->pos += count;
      //else status = CheckViolation();
   }
   if (processTail)
      status = processTailPos(addr,buff[dataLen],verifyFlag);
   if (status) status = ERROR_WRITE;
   return(status);
}

/***************************************************************************
**
**  EmuReadMemN
**
**  Description: Michelle memory-group routine, to read a chunk memory
**               from "addr" location.
**         Note: The data length maximum - 0x100
**
**  Parameters:
**     input:
**        addr -- unsigned long
**        len  --
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "N bytes data string"
**
****************************************************************************/
VOID EmuReadMemN(ADDR *addr, U32 addrLen) {
STATUS status;
U8 dataLen,memBuff[BUFF_LEN];
U32 m68Reg[MAX_CPU_REG_NO];
//U16 interFlag;

   status = GOOD;
   if (addrLen > BUFF_LEN) {
      OutputStatus(LEN_TOO_LONG,ON);
      return;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);
//   interFlag = ChkReduceAddrLen(addr->pos,addr->space,&addrLen);
//   if (!interFlag) { // all of the range isn't in the internal register area.
      ModifyRegisterCS();
      status = ReadMem(addr,memBuff,(U16)addrLen);
      RestoreRegisterCS();
//   }
//   else
//      status = ReadMem(addr,memBuff,(U16)addrLen);
   DisEmStop();

   if (status == GOOD) {
      OutputStatus(status,OFF);
      dataLen = (U8)addrLen;
      OutData(dataLen,memBuff,ON);
      OutEnd();

   }
   else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN,&canTemp,ON);
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}


/***************************************************************************
**
**  EmuSetMemN
**
**  Description: Michelle memory-group routine, to write a chunk data
**               from "buff" to "addr" location. The driver program can
**               use this function to implement to download code from a
**               file.
**               The length should be limitted up to 1024 bytes(1KB).
**         Note: if the set area is not in the internal register area
**               then do: ModifyRegister();
**                          set memory
**                        RestoreRegister();
**               else set memory only.
**
**  Parameters:
**     input:
**        addr  -- unsigned long,
**        *buff -- unsigned char,
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuSetMemN(ADDR *addr, U8 *buff, U16 len) {
STATUS status;
U32 m68Reg[MAX_CPU_REG_NO];
ADDR addr2;
U16 Len1, Len2;

   status = GOOD;
   if (len > BUFF_LEN) {
      OutputStatus(LEN_TOO_LONG,ON);
      return;
   }
   else if (len == 0) {
      OutputStatus(LEN_EQUAL_ZERO,ON);
      return;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);

   if (addr->space==SPACE_SD && (addr->pos + len) > INTER_REG_ADDR) //check internal register area
      if (addr->pos < INTER_REG_ADDR){ // partial of the range in internal register area
         Len1 = (U16)(INTER_REG_ADDR - addr->pos);
         addr2.space = addr->space;
         addr2.pos = INTER_REG_ADDR;
         Len2 = len - Len1;
         ModifyRegisterCS();
         status = WriteMem(addr,Len1,buff);
         RestoreRegisterCS();
         status = WriteMem(&addr2,Len2,&buff[Len1]);
      }
      else // all in internal register area
      status = WriteMem(addr,len,buff);
   else { // all not in internal register area
      ModifyRegisterCS();
      status = WriteMem(addr,len,buff);
      RestoreRegisterCS();
   }
   DisEmStop();

   if (exceptBreak) {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN*2,&canTemp,ON);
      OutEnd();
   }
   else {
      switch (status) {
      case GOOD :
         OutputStatus(GOOD,ON);
         break;
      case MEM_PROTECT :
      case MEM_GUARD :
         OutputStatus(status,ON);
         break;
      case ERROR_WRITE :
         OutputStatus(status,OFF);
         OutData(RET_ADDR_LEN*2,&canTemp,ON);
         OutEnd();
         break;
      default :
         OutputExceptReason(OFF);
         OutData(RET_ADDR_LEN*2,&canTemp,ON);  //NOT_READY
         OutEnd();
      }
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}


/***************************************************************************
**
** EmuCompare
**
**  Description: Michelle memory-group routine, to compare two memory
**               blocks.
**               Return the locations and its contents, if there has
**               difference.
**               The "addr1" and "addr2" is the addres range of the
**               first block.
**               The "addr3" is the start address of the second block.
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        addr3 -- unsigned long
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
**        return data -- "length" + "location" + "contents"
**
****************************************************************************/
VOID EmuCompare(ADDR *addr1, U32 addrLen, ADDR *addr2) {
STATUS status;
U32 m68Reg[MAX_CPU_REG_NO];
U16 size;

   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   switch ((SIZE)size) {
      case S_WORD :
         ReduceEven(addr1->pos);
         IncreaseEven(addrLen);
         ReduceEven(addr2->pos);
         break;
      case S_LONG :
         ReduceQuad(addr1->pos);
         IncreaseQuad(addrLen);
         ReduceQuad(addr2->pos);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);
   ModifyRegisterCS();
   status = CompareMem(addr1->pos,addr1->space,addrLen,
           addr2->pos,addr2->space,size) & 0xff;
   RestoreRegisterCS();
   DisEmStop();
   //status = CheckViolation();
   switch (status) {
      case 0 :
         OutputStatus(NO_DIFF,ON);
         break;
      case 1 :
         OutputStatus(GOOD,OFF);
         OutData(RET_ADDR_LEN*2,&canTemp,ON);
         OutEnd();
         break;
      case 2 :
      case 3 :
         OutputExceptReason(OFF);
         if (status == 2)  OutData(RET_ADDR_LEN,&canTemp,ON);
         else OutData(RET_ADDR_LEN,&canTemp + 4,ON);
         OutEnd();
         break;
      case 4 :
         OutputExceptReason(ON);
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}


/***************************************************************************
**
**  EmuCopy
**
**  Description: Michelle memory-group routine, to copy a memory block
**               data to another block's.
**               The "addr1" and "addr2" is the addres range of the
**               first block.
**               The "addr3" is the start address of the destination
**               block.
**
**         Note: If the range is partial in the internal register area
**               then reduce the length and do:
**                     ModifyRegisterCS()
**                     copy command
**                     RestoreRegisterCS()
**               If one of the source area or destination area is all in the
**               internal register area then don't do copy.
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        addr3 -- unsigned long
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuCopy(ADDR *addr1, U32 addrLen, ADDR *addr2) {
   STATUS status;
   U32 m68Reg[MAX_CPU_REG_NO];
   U16 size,verifyFlag; //,interFlag ;
   
   status = GOOD;
//   interFlag = NO;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   switch ((SIZE)size) {
      case S_WORD :
         ReduceEven(addr1->pos);
         IncreaseEven(addrLen);
         ReduceEven(addr2->pos);
         break;
      case S_LONG :
         ReduceQuad(addr1->pos);
         IncreaseQuad(addrLen);
         ReduceEven(addr2->pos);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);

//   interFlag = ChkReduceAddrLen(addr1->pos,addr1->space,&addrLen);
//   if (!interFlag)
//      interFlag = ChkReduceAddrLen(addr2->pos,addr2->space,&addrLen);

//   if (!interFlag) {//not all of the range is in the internal register area.
      ModifyRegisterCS();
      if (addr1->pos == addr2->pos && addr1->space == addr2->space)
         status = CopyTargetMem(addr1,addrLen);
      else if (addr1->pos >= addr2->pos)
         status = PreCopyMem(addr1->pos,addr1->space,addrLen,
            addr2->pos,addr2->space,verifyFlag) & 0xff;
      else {
         status = PostCopyMem(addr1->pos,addr1->space,addrLen,
         addr2->pos,addr2->space,verifyFlag) & 0xff;
      }
      RestoreRegisterCS();
//   }
   DisEmStop();

   //status = CheckViolation();
   if (exceptBreak == GOOD) {
      if (status == GOOD)
         OutputStatus(GOOD,ON);
      else {
         switch(status) {
         case 1 :
            status = ERROR_WRITE; break;
         case 2 :
         case 3 :
            status = NOT_READY; break;
         case 4 :
            status = ERROR_WRITE; break;
         }
         OutputStatus(status,OFF);
         OutData(RET_ADDR_LEN,(U8 *)&canTemp,ON);
         OutEnd();
      }
   } else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN,&canTemp,ON);
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
   return;
}

/***************************************************************************
**
**  FillMem
**
**  Description: Fill a memory range with a specified pattern.
**               Data pattern is specified in "buff" array.
**               The lengh of the data pattern is "len".
**
**  Parameters:
**     input:
**        addr -- unsigned long(address), int (space)
**        addrLen -- unsigned long,
**        *buff -- unsigned char, Max 16 bytes;
**        len   -- int
**        size --int
**        verifyFlag -- int
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
STATUS FillMem(ADDR *addr, U32 addrLen, U8 *buff, U16 len, U16 size, U16 verifyFlag) {
STATUS status;
U16 pos;
   switch ((SIZE)size) {
      case S_BYTE :
         status = FillByteMem(addr->pos,addr->space,addrLen,len,buff,verifyFlag,ON) & 0xff;
         //status |= CheckViolation();
         break;
      case S_WORD :
      case S_LONG :
         //ReduceEven(addr);
         //IncreaseEven(addrLen);

         //If the length is odd and the size is word then fill the last byte first
         if (addrLen%2) {
            pos = (len == 1 ? 0 : addrLen%len ? (U16)(addrLen%len)-1 : len-1);
            addrLen--;
            status = FillByteMem(addr->pos+addrLen,addr->space,1,1,&buff[pos],verifyFlag,ON) & 0xff;
            //status |= CheckViolation();
         }

         status = FillWordMem(addr->pos,addr->space,addrLen,len,buff,verifyFlag,ON) & 0xff;
         //status |= CheckViolation();
         break;
//         //ReduceQuad(addr);
//         //IncreaseQuad(addrLen);
//         status = FillByteMem(addr->pos,addr->space,addrLen,len,buff,verifyFlag,ON) & 0xff;
//         //status |= CheckViolation();
//         break;
      default :
         status = FillByteMem(addr->pos,addr->space,addrLen,len,buff,verifyFlag,ON) & 0xff;
         //status |= CheckViolation();
         break;
   }
   return(status);
}
/***************************************************************************
**
**  EmuFill
**
**  Description: Michelle memory-group routine, to fill a memory range
**               with a specified pattern.
**               Fill a memory range with a specified pattern.
**               Data pattern is specified in "buff" array.
**               The lengh of the data pattern is "len".
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        *buff -- unsigned char, Max 16 bytes;
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
** 
****************************************************************************/
VOID EmuFill(ADDR *addr, U32 addrLen, U8 *buff, U16 len) {
STATUS status;
U16 size,pos,pos2;
U32 addrLen1,addrLen2;
U32 m68Reg[MAX_CPU_REG_NO];
U8 buff2[BUFF_LEN];
U16 verifyFlag;
ADDR addr2;

   status = GOOD;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);

   if (addr->space==SPACE_SD && (addr->pos + addrLen) > INTER_REG_ADDR) 
      if (addr->pos < INTER_REG_ADDR){ // partial of the range in internal register area
         addrLen1 = INTER_REG_ADDR - addr->pos;
         addr2.space = addr->space;
         addr2.pos = INTER_REG_ADDR;
         addrLen2 = addrLen - addrLen1;
         pos = (len == 1 ? 0 : (addrLen1%len) ? (U16)(addrLen1%len) : 0);
         pos2 = len - pos;
         memcpy(&buff2[0], &buff[pos], pos2);
         memcpy(&buff2[pos2], &buff[0], pos);
         ModifyRegisterCS();
         status = FillMem(addr,addrLen1,buff,len,size,verifyFlag);
         RestoreRegisterCS();
         status = FillMem(&addr2,addrLen2,buff2,len,size,0);
      }
      else // all in internal register area
         status = FillMem(addr,addrLen,buff,len,size,0);
   else { // all not internal register area
      ModifyRegisterCS();
      status = FillMem(addr,addrLen,buff,len,size,verifyFlag);
      RestoreRegisterCS();
   }
   DisEmStop();
   if (exceptBreak == GOOD) {
      switch (status) {
      case GOOD :
      case MEM_PROTECT :
         OutputStatus(GOOD,ON);
         break;
      case MEM_GUARD :
         OutputStatus(status,ON);
         break;
      default :
         OutputStatus(ERROR_WRITE,OFF);
         OutData(RET_ADDR_LEN*2,&canTemp,ON);
         OutEnd();
      }
   }
   else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN*2,&canTemp,ON); // NOT READY
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}

/***************************************************************************
**
**  EmuTest
**
**  Description: Michelle memory-group routine, to test a spcified
**               address range working O.K. or not? Is RAM or not?
**         Note: If the range is partial in the internal register area
**               then reduce the length and do:
**                     ModifyRegisterCS()
**                     copy command
**                     RestoreRegisterCS()
**               If one of the source area or destination area is all in the
**               internal register area then don't do copy (InterFlag=YES).
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**
**    *** *pattern[n] -- char ===> optional
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**  NOTE: I have three ways to do this testing, you can decide how
**        to implement it by one of these.
**        1. (Read data from specified address location) -->
**           (Write pre-read data back to the same memory location) -->
**           (Compare these result then check it to report the status)
**        2. Write the defined special data pattern to the specified
**           memory loaction, then check the virolation.
**        3. Like method 2, but driver can select the "data pattern"
**           as she like. In this case, to add a parameter to indicate
**           the data pattern is necessary.
**
****************************************************************************/
VOID EmuTest(ADDR *addr, U32 addrLen) {
STATUS status;
U32 m68Reg[MAX_CPU_REG_NO];
U16 size; //,interFlag;

   status = GOOD;
//   interFlag = NO;
   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   switch ((SIZE)size) {
      case S_WORD :
         ReduceEven(addr->pos);
         IncreaseEven(addrLen);
         break;
      case S_LONG :
         ReduceQuad(addr->pos);
         IncreaseQuad(addrLen);
         break;
   }
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);
//   interFlag = ChkReduceAddrLen(addr->pos,addr->space,&addrLen);
//   if (!interFlag) { // all of the range is not in the internal register area.
      ModifyRegisterCS();
      status = TestMem(addr->pos,addr->space,addrLen,size) & 0xff;
      RestoreRegisterCS();
//   }
   DisEmStop();
//   status = CheckViolation();
   if (exceptBreak == GOOD) {
      if (status == GOOD) OutputStatus(GOOD,ON);
      else {
         OutputStatus(TEST_FAIL,OFF);
         OutData(RET_ADDR_LEN,&canTemp,ON);
         OutEnd();
      }
   }
   else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN,&canTemp,ON);
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}

/***************************************************************************
**
**  EmuSearch
**
**  Description: Michelle memory-group routine, to search data pattern
**               in the specified address range.
**               Data pattern is specified in "buff" array.
**               The lengh of the data pattern is "len".
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        *buff -- unsigned char,  Max 32 Bytes;
**        len   -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuSearch(ADDR *addr, U32 addrLen, U8 *buff, U16 len) {
#define MATCH  0xff /* template data, it will redefine. */
STATUS status;
U32 m68Reg[MAX_CPU_REG_NO];

   status = GOOD;
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);
   ModifyRegisterCS();
   status = SearchMem(addr->pos,addr->space,addrLen,len,buff) & 0xff;
   RestoreRegisterCS();
   DisEmStop();

//   status = CheckViolation();
   if (exceptBreak == GOOD) {
      if (status == GOOD) {
         OutputStatus(GOOD,OFF);
         OutData(RET_ADDR_LEN,&canTemp,ON);
         OutEnd();
      }
      else OutputStatus(NO_FOUND,ON);
   }
   else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN,&canTemp,ON);
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);

}

/***************************************************************************
**
**  EmuChecksum
**
**  Description: Michelle memory-group routine, to caculate the data
**               sum value of the specified address range.
**               The "addr1" and "addr2" is the address range.
**               The "size" is the sum-width = 0 --> caculate in byte
**                                           = 1 --> caculate in word
**                                           = 2 --> caculate in long word
**
**  Parameters:
**     input:
**        addr1 -- unsigned long,
**        addr2 -- unsigned long,
**        size  -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on Emulator" -- Fatal HW error; check status-code
**
**
****************************************************************************/
VOID EmuChecksum(ADDR *addr, U32 addrLen) {
U32 m68Reg[MAX_CPU_REG_NO];
U16 size,checksumData;

   AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   StopRun();
   ClrViolFlg();
   EnEmDisViol();
   AccessCpuReg(READ_ALL,noUse,m68Reg);
   ModifyRegisterCS();
   if (size == S_WORD) IncreaseEven(addrLen);
   Checksum(addr->pos,addr->space,addrLen,size);
   RestoreRegisterCS();
   DisEmStop();
   if (exceptBreak == GOOD) {
      OutputStatus(exceptBreak,OFF);
      diff = (U8 *)&canTemp;
      switch (size) {
         case S_BYTE :
            checksumData = *(U8 *)diff;/* byte checksum */
            break;
         case S_WORD :
            checksumData = *(U16 *)diff;/* word checksum */
            break;
      }
      OutData(2,&checksumData,ON);
      OutEnd();
   }
   else {
      OutputExceptReason(OFF);
      OutData(RET_ADDR_LEN,&canTemp,ON);
      OutEnd();
   }
   AccessCpuReg(WRITE_ALL,noUse,m68Reg);
}

/******************************** E O F ***********************************/

