
/****************************************************************************
**
** Name: trcevent.c
**
** Description:
**   Bus event setting services
**
** Status: PRELIMINARY
**
** $Log:   S:/tbird/hfw328/arcm/trcevent.c_v  $
** 
**    Rev 1.0   11 Jan 1996 10:48:22   gene
** Initial revision.
** 
**    Rev 1.0   03 Jan 1996 13:50:54   gene
** Initial revision.
** 
**    Rev 1.0   07 Sep 1995 10:47:42   gene
** Initial revision.
**
** $Header:   S:/tbird/hfw328/arcm/trcevent.c_v   1.0   11 Jan 1996 10:48:22   gene  $
**
** Copyright (C) 1992 Microtek International. All rights reserved.
**
****************************************************************************/

                       /****************************
                        *                          *
                        *      INCLUDE FILES       *
                        *                          *
                        ****************************/

#include <string.h>
#include <conio.h>

#ifndef _TRCERROR_
#include "trcerror.h"
#endif
#ifndef _TRCXLX_
#include "trcxlx.h"
#endif
#ifndef _TRCEVENT_
#include "trcevent.h"
#endif
#ifndef _TRCDTST_
#include "trclib.h"
#endif

                       /****************************
                        *                          *
                        *    LOCAL DEFINITIONS     *
                        *                          *
                        ****************************/

typedef enum { BASE_F, BASE_FG, BASE_FGM } XLX_BASE;
typedef enum { LOW_BOUNDARY, HIGH_BOUNDARY } EVENT_BOUNDARY;
typedef enum { ADDR_FIELD, DATA_FIELD, STATUS_FIELD, SPARE_FIELD,
               STATUSFC_FIELD, STATUSDS_FIELD, STATUSCS_FIELD,
               } EVENT_FIELD;

typedef struct {
   U16 clb;
   XLX_BASE base;
   U16 logicInputLocation;
   XLX_FUNC greatOrLess;
   XLX_FUNC equal;
} EREC_CLB_MAP;

typedef struct {
   EREC_CLB_MAP data0;
   EREC_CLB_MAP data1;
   EREC_CLB_MAP data2;
   EREC_CLB_MAP data3;
   EREC_CLB_MAP data4;
   EREC_CLB_MAP data5;
   EREC_CLB_MAP data6;
   EREC_CLB_MAP data7;
   EREC_CLB_MAP status0;
   EREC_CLB_MAP status1;
   EREC_CLB_MAP status2;
   EREC_CLB_MAP status3;
} EREC1_XLX_MAP;

typedef struct {
   EREC_CLB_MAP addr0;
   EREC_CLB_MAP addr1;
   EREC_CLB_MAP addr2;
   EREC_CLB_MAP addr3;
   EREC_CLB_MAP addr4;
   EREC_CLB_MAP addr5;
   EREC_CLB_MAP addr6;
   EREC_CLB_MAP addr7;
   EREC_CLB_MAP spare0;
   EREC_CLB_MAP spare1;
} EREC2_XLX_MAP;

#define EP_ADDR_SIZE 32   /* charles 12/15/93 */
#define EP_DATA_SIZE 32    /* charles 1/24/94 */
#define EP_STATUS_SIZE 16   /* for SetAlwaysMatch */
#define EP_SPARE_SIZE 8   /* charles 01/24/94 */
#define EP_STATUSFC_SIZE 4 /* for SetSingleMask & SetMultiStatus */
#define EP_STATUSDS_SIZE 4 /* for SetSingleMask */
#define EP_STATUSCS_SIZE 8 /* for SetSingleMask */

#define EP_STATUSFC_NUM 7   /* SP, UP, SR, SW, UR, UW, AK */

#if (EP_STATUS_SIZE <= 8)
   #define EP_STATUS_SIG U8
#elif (EP_STATUS_SIZE <= 16)
   #define EP_STATUS_SIG U16
#elif (EP_STATUS_SIZE <= 32)
   #define EP_STATUS_SIG U32
#endif

#define DEFAULT_LOGIC_INPUT_LOCATION 0xBCAD
#define EREC1_EVENT_QUANTITY 8

#define pEPCONTROL 0xF140
#define bitISOEP   0x40  // New machine, Allen 9/14/94

#define INVALID_PHY_EV 0xFF

#define pTM8255B  0xF161

static EREC1_XLX_MAP _based(_segname("_CODE")) erec1XlxMap[8] = {
#include "trcerec1.map"
};
static EREC2_XLX_MAP _based(_segname("_CODE")) erec2XlxMap[8] = {
#include "trcerec2.map"
};

                       /****************************
                        *                          *
                        *    EXTERNAL VARIABLES    *
                        *                          *
                        ****************************/

extern U8 bitStreamBuffer1[];
extern U8 bitStreamBuffer2[];
extern U8 _far EREC1XCByte[];
extern U8 _far EREC2XCByte[];
extern U8 phyEv[];

                       /****************************
                        *                          *
                        *     LOCAL PROTOTYPES     *
                        *                          *
                        ****************************/

U8 SwapLocation(U8 addr, U16 info);
RETCODE SetRangeAddr(EVENT_BOUNDARY boundaryType,
                             EREC1_XLX_MAP _far *xlx1Map,
                             EREC2_XLX_MAP _far *xlx2Map,
                             EP_ADDR boundaryAddr);
RETCODE SetSingleMask(EVENT_FIELD field,
                              EREC1_XLX_MAP _far *xlx1Map,
                              EREC2_XLX_MAP _far *xlx2Map,
                              U32 value, U32 mask);
RETCODE SetAlwaysMatch(EVENT_FIELD field,
                       EREC1_XLX_MAP _far *xlx1Map,
                       EREC2_XLX_MAP _far *xlx2Map
                       );
RETCODE SetMultiStatus(EREC1_XLX_MAP _far *xlx1Map,
                       EREC2_XLX_MAP _far *xlx2Map,
		       U16 status);
RETCODE SetOneEvent(BUS_EVENT *eventStruc,
                    EREC1_XLX_MAP _far *xlx1Map,
                    EREC2_XLX_MAP _far *xlx2Map,
                    U8 eventPhysicalID);
RETCODE SetOneEventP(BUS_EVENT *eventStruc,
                    EREC1_XLX_MAP _far *xlx1Map,
                    EREC2_XLX_MAP _far *xlx2Map,
                    U8 eventPhysicalID);
RETCODE ClearAllEvent(EREC1_XLX_MAP _far *xlx1Map, EREC2_XLX_MAP _far *xlx2Map);

                       /****************************
                        *                          *
                        *     EXECUTABLE CODE      *
                        *                          *
                        ****************************/

/******************************************************************************
**
**  SelectExternalTraceBit
**
*******************************************************************************/
VOID SelectExternalTraceBit(U8 selection) {
   outp(pTM8255B, (((U8) inp(pTM8255B)) & 0x1F) | ((selection & 0x07) << 5));
}

/******************************************************************************
**
**  SetAllBusEvent
**
*******************************************************************************/
RETCODE SetAllBusEvent(BUS_EVENT *busEvent) {
   RETCODE err;
   U8 temp, eventID, eventDmy, eventUsed = 0;

   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if (busEvent[eventID].enable == YES) {
         switch (busEvent[eventID].addrSpec) {
            case UNDEFINE_ADDR: eventUsed++; break;
            case SINGLE_ADDR: eventUsed++; break;
            case RANGE_ADDR: eventUsed +=2; break;
            case MASK_ADDR: eventUsed++; break;
         }
      }
   }
   if (eventUsed > EREC1_EVENT_QUANTITY) return(ERR_OUT_OF_EREC_RESOURCE);
   fmemcpy(bitStreamBuffer1, EREC1XCByte, (U16) XC3090SIZE);
   fmemcpy(bitStreamBuffer2, EREC2XCByte, (U16) XC3090SIZE);
   if ((err = ClearAllEvent(erec1XlxMap, erec2XlxMap)) != GOOD) return(err);
   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++)
      phyEv[eventID] = (U8) INVALID_PHY_EV;
   /* set events with range address setting */
   for (eventID = eventUsed = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if ((busEvent[eventID].enable == YES) &&
          (busEvent[eventID].addrSpec == RANGE_ADDR)) {
         if ((err = SetOneEvent(&busEvent[eventID],
                                erec1XlxMap,
                                erec2XlxMap,
                                eventUsed))
             != GOOD) return(err);
         phyEv[eventID] = eventUsed;
         eventUsed +=2;
      }
   }
   /* set events without range address setting */
   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if ((busEvent[eventID].enable == YES) &&
          (busEvent[eventID].addrSpec != RANGE_ADDR)) {
         if ((err = SetOneEvent(&busEvent[eventID],
                                erec1XlxMap,
                                erec2XlxMap,
                                eventUsed))
             != GOOD) return(err);
         phyEv[eventID] = eventUsed;
         eventUsed++;
      }
   }
   for (eventID = 0, eventDmy = -1; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if (phyEv[eventID] == (U8) INVALID_PHY_EV)
         phyEv[eventID] = (eventUsed < EREC1_EVENT_QUANTITY) ?
                          (eventUsed++) : (eventDmy += 2);
   }

   /* set qualify clb of EREC1 */

   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_F, 0, 0, 0)) != GOOD)
     return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_F, 0, 1, 1)) != GOOD)
      return(err);  /* F = ~D */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_G, 0, 1, 1)) != GOOD)
      return(err);  /* G = ~D */

   /* set qualify clb of EREC2 */

   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_F, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_F, 0, 4, 1)) != GOOD)
      return(err);  /* F = ~C */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_G, 0, 4, 1)) != GOOD)
      return(err);  /* G = ~C */

   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_F, 0, 0, 0)) != GOOD)
     return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_F, 0, 4, 1)) != GOOD)
      return(err);  /* F = ~C */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_G, 0, 4, 1)) != GOOD)
      return(err);  /* G = ~C */

   outp(pEPCONTROL, (temp = (U8) inp(pEPCONTROL)) | bitISOEP);
   if ((err = XlxReprogram(XLX_EREC1)) != GOOD) return(err);
   if ((err = XlxReprogram(XLX_EREC2)) != GOOD) return(err);
   outp(pEPCONTROL, temp);
   return(GOOD);
}

/******************************************************************************
**
**  SetAllBusEventP
**
**  Description: set all bus event with PowerPack status protocal
**
*******************************************************************************/
RETCODE SetAllBusEventP(BUS_EVENT *busEvent) {
   RETCODE err;
   U8 temp, eventID, eventDmy, eventUsed = 0;

   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if (busEvent[eventID].enable == YES) {
         switch (busEvent[eventID].addrSpec) {
            case UNDEFINE_ADDR: eventUsed++; break;
            case SINGLE_ADDR: eventUsed++; break;
            case RANGE_ADDR: eventUsed +=2; break;
            case MASK_ADDR: eventUsed++; break;
         }
      }
   }
   if (eventUsed > EREC1_EVENT_QUANTITY) return(ERR_OUT_OF_EREC_RESOURCE);
   fmemcpy(bitStreamBuffer1, EREC1XCByte, (U16) XC3090SIZE);
   fmemcpy(bitStreamBuffer2, EREC2XCByte, (U16) XC3090SIZE);
   if ((err = ClearAllEvent(erec1XlxMap, erec2XlxMap)) != GOOD) return(err);
   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++)
      phyEv[eventID] = (U8) INVALID_PHY_EV;
   /* set events with range address setting */
   for (eventID = eventUsed = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if ((busEvent[eventID].enable == YES) &&
          (busEvent[eventID].addrSpec == RANGE_ADDR)) {
         if ((err = SetOneEventP(&busEvent[eventID],
                                erec1XlxMap,
                                erec2XlxMap,
                                eventUsed))
             != GOOD) return(err);
         phyEv[eventID] = eventUsed;
         eventUsed +=2;
      }
   }
   /* set events without range address setting */
   for (eventID = 0; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if ((busEvent[eventID].enable == YES) &&
          (busEvent[eventID].addrSpec != RANGE_ADDR)) {
         if ((err = SetOneEventP(&busEvent[eventID],
                                erec1XlxMap,
                                erec2XlxMap,
                                eventUsed))
             != GOOD) return(err);
         phyEv[eventID] = eventUsed;
         eventUsed++;
      }
   }
   for (eventID = 0, eventDmy = -1; eventID < EREC1_EVENT_QUANTITY; eventID++) {
      if (phyEv[eventID] == (U8) INVALID_PHY_EV)
         phyEv[eventID] = (eventUsed < EREC1_EVENT_QUANTITY) ?
                          (eventUsed++) : (eventDmy += 2);
   }

   /* set qualify clb of EREC1 */

   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_F, 0, 0, 0)) != GOOD)
     return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_F, 0, 1, 1)) != GOOD)
      return(err);  /* F = ~D */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer1,
                           XC3090, 0x1000, XLX_G, 0, 1, 1)) != GOOD)
      return(err);  /* G = ~D */

   /* set qualify clb of EREC2 */

   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_F, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_F, 0, 4, 1)) != GOOD)
      return(err);  /* F = ~C */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0E, XLX_G, 0, 4, 1)) != GOOD)
      return(err);  /* G = ~C */

   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_F, 0, 0, 0)) != GOOD)
     return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_F, 0, 4, 1)) != GOOD)
      return(err);  /* F = ~C */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_G, 0, 0, 0)) != GOOD)
      return(err);  /* addr = (B, C, A, D) */
   if ((err = XlxModifyClb(bitStreamBuffer2,
                           XC3090, 0x0C0D, XLX_G, 0, 4, 1)) != GOOD)
      return(err);  /* G = ~C */

   outp(pEPCONTROL, (temp = (U8) inp(pEPCONTROL)) | bitISOEP);
   if ((err = XlxReprogram(XLX_EREC1)) != GOOD) return(err);
   if ((err = XlxReprogram(XLX_EREC2)) != GOOD) return(err);
   outp(pEPCONTROL, temp);
   return(GOOD);
}

/******************************************************************************
**
**  SwapLocation
**
**  Description:
**    A Xilinx XC3000 CLB compare 4 continuous signals of bus X.
**    (bus X can be addr, data, or status bus)
**    Each signal connected to a logic input A, B, C or D.
**    For example these 4 signals are X[n+3], X[n+2], X[n+1], and X[n].
**    The default locations are            ______________
**        X[n+3] --> B                    |              |
**        X[n+2] --> C          X[n+1] ---|A            X|---
**        X[n+1] --> A          X[n+3] ---|B    CLB      |
**        X[n]   --> D          X[n+2] ---|C             |
**                              X[n]   ---|D            Y|---
**                                        |______________|
**
**    But because of the Xilinx Chip internal layout purpose, the locations
**    may be changed.
**    If the locations were swapped, the bus signals should be swapped too.
**    This function return the bus signals value which are swapped according
**    to their locations.
**    If the locations "addrLogicInputOrder" are the same to the default(0xBCAD)
**    then the original value "addr" will be returned without swapping.
**
**  Parameters:
**    input:
**      addr: the 4 continuous bus signals value
**      addrLogicInputOrder: the location of these 4 bus signals
**    output:
**      <none>
**
*******************************************************************************/
U8 SwapLocation(U8 addr, U16 addrLogicInputOrder) {
   U8 doThisBit, setThisBit, returnValue = 0;
   U16 compareWithThisLogicInput, compareData;

   for (doThisBit = 0x08; doThisBit != 0x00; doThisBit >>= 1) {
      if ((addr & doThisBit) != 0) {
         setThisBit = 0x08;
         compareWithThisLogicInput = 0xF000;
         compareData = addrLogicInputOrder & compareWithThisLogicInput;
         while ((DEFAULT_LOGIC_INPUT_LOCATION & compareWithThisLogicInput) !=
                 compareData) {
            setThisBit >>= 1;
            compareWithThisLogicInput >>= 4;
            compareData >>= 4;
         }
         returnValue |= setThisBit;
      }
      addrLogicInputOrder <<= 4;
   }
   return(returnValue);
}

/******************************************************************************
**
**  SetRangeAddr
**
**  Description:
**    Set upper or lower bound of range addr of one event
**
**  Parameters:
**    input:
**      boundaryType: LOW_BOUNDARY or HIGH_BOUNDARY
**      *xlxMap: Xilinx CLB map for this field of the event
**      boundaryAddr: boundary address of the range of the event
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetRangeAddr(EVENT_BOUNDARY boundaryType,
                             EREC1_XLX_MAP _far *xlx1Map,
                             EREC2_XLX_MAP _far *xlx2Map,
                             EP_ADDR boundaryAddr) {
   RETCODE err;
   U8 nibble, nibbleAddr, doThisBit, addr, mask;
   U8 zero = 0, *modifyWhenEqualThis;
   U16 _far *clb, _far *swapInfo;
   XLX_BASE _far *base;
   XLX_FUNC _far *func;

   clb = &(xlx2Map->addr0.clb);
   base = &(xlx2Map->addr0.base);
   func = &(xlx2Map->addr0.greatOrLess);
   swapInfo = &(xlx2Map->addr0.logicInputLocation);
   switch (boundaryType) {
      case LOW_BOUNDARY:  modifyWhenEqualThis = &zero; break;
      case HIGH_BOUNDARY: modifyWhenEqualThis = &doThisBit; break;
   }
   for (nibble = 0; nibble < (EP_ADDR_SIZE / 4); nibble++) {
      nibbleAddr = (U8) ((boundaryAddr >> (nibble * 4)) & 0x0F);
      if ((err = SetSingleMask(ADDR_FIELD, xlx1Map,
                               xlx2Map,
                               boundaryAddr,
           0xFFFFFFFF)) != GOOD) return(err);
      for (doThisBit = 0x08, mask = 0x00; doThisBit != 0; doThisBit >>= 1) {
         mask |= doThisBit;
         if ((nibbleAddr & doThisBit) == *modifyWhenEqualThis) {
            addr = nibbleAddr ^ doThisBit;
            switch (*base) {
               case BASE_F:
                  if ((err = XlxModifyClb(bitStreamBuffer2,
                                          XC3090, *clb, XLX_F,
                                SwapLocation(addr,*swapInfo),
                                SwapLocation(mask, *swapInfo), 1)) != GOOD)
                     return(err);
                  if ((err = XlxModifyClb(bitStreamBuffer2,
                                          XC3090, *clb, XLX_G,
                                SwapLocation(addr,*swapInfo),
                                SwapLocation(mask, *swapInfo), 1)) != GOOD)
                     return(err);
                  break;
               default:
                  if ((err = XlxModifyClb(bitStreamBuffer2,
                                          XC3090, *clb, *func,
                                SwapLocation(addr,*swapInfo),
                                SwapLocation(mask, *swapInfo), 1)) != GOOD)
                     return(err);
                  break;
            }
         }
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) func += sizeof(EREC_CLB_MAP);
      (U8 _far *) swapInfo += sizeof(EREC_CLB_MAP);
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetSingleMask
**
**  Description:
**    Set addr, data or status or spare field of one event to
**    a single value or
**    a value with mask.
**    For a single value, set the input parameter "mask" to be 0xFFFFFFFF.
**
**  Parameters:
**    input:
**      field: ADDR_FIELD, DATA_FIELD or STATUS_FIELD or SPARE_FIELD or
**             STATUSFC_FIELD or STATUSDS_FIELD or STATUSCS_FIELD
**      *xlxMap: Xilinx CLB map for this field of the event
**      value: setting value
**      mask: setting mask
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetSingleMask(EVENT_FIELD field, EREC1_XLX_MAP _far *xlx1Map,
                              EREC2_XLX_MAP _far *xlx2Map,
                              U32 value, U32 mask) {
   RETCODE err;
   U8 nibble, nibbleValue, nibbleMask, fieldSize;
   U16 _far *clb, _far *swapInfo;
   XLX_BASE _far *base;
   XLX_FUNC _far *func;
   U8 *bitStreamBuffer;

   switch (field) {
      case ADDR_FIELD:
         clb = &(xlx2Map->addr0.clb);
         base = &(xlx2Map->addr0.base);
         func = &(xlx2Map->addr0.equal);
         swapInfo = &(xlx2Map->addr0.logicInputLocation);
         fieldSize = EP_ADDR_SIZE;
         bitStreamBuffer=bitStreamBuffer2;
         break;
      case DATA_FIELD:
         clb = &(xlx1Map->data0.clb);
         base = &(xlx1Map->data0.base);
         func = &(xlx1Map->data0.equal);
         swapInfo = &(xlx1Map->data0.logicInputLocation);
         fieldSize = EP_DATA_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case STATUS_FIELD:
         clb = &(xlx1Map->status0.clb);
         base = &(xlx1Map->status0.base);
         func = &(xlx1Map->status0.equal);
         swapInfo = &(xlx1Map->status0.logicInputLocation);
         fieldSize = EP_STATUS_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case SPARE_FIELD:
         clb = &(xlx2Map->spare0.clb);
         base = &(xlx2Map->spare0.base);
         func = &(xlx2Map->spare0.equal);
         swapInfo = &(xlx2Map->spare0.logicInputLocation);
         fieldSize = EP_SPARE_SIZE;
         bitStreamBuffer=bitStreamBuffer2;
         break;
      case STATUSFC_FIELD:
         clb = &(xlx1Map->status1.clb);
         base = &(xlx1Map->status1.base);
         func = &(xlx1Map->status1.equal);
         swapInfo = &(xlx1Map->status1.logicInputLocation);
         fieldSize = EP_STATUSFC_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case STATUSDS_FIELD:
         clb = &(xlx1Map->status0.clb);
         base = &(xlx1Map->status0.base);
         func = &(xlx1Map->status0.equal);
         swapInfo = &(xlx1Map->status0.logicInputLocation);
         fieldSize = EP_STATUSDS_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case STATUSCS_FIELD:
         clb = &(xlx1Map->status2.clb);
         base = &(xlx1Map->status2.base);
         func = &(xlx1Map->status2.equal);
         swapInfo = &(xlx1Map->status2.logicInputLocation);
         fieldSize = EP_STATUSCS_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
   }
   for (nibble = 0; nibble < (U8) (fieldSize / 4); nibble++) {
      nibbleValue = (U8) ((value >> (nibble * 4)) & 0x0F);
      nibbleMask = (U8) ((mask >> (nibble * 4)) & 0x0F);
      switch (*base) {
         case BASE_F:
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, XLX_F,
                          SwapLocation(nibbleValue, *swapInfo),
                          SwapLocation(nibbleMask, *swapInfo), 1)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, XLX_G,
                          SwapLocation(nibbleValue, *swapInfo),
                          SwapLocation(nibbleMask, *swapInfo), 1)) != GOOD)
               return(err);
            break;
         default:
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, *func,
                          SwapLocation(nibbleValue, *swapInfo),
                          SwapLocation(nibbleMask, *swapInfo), 1)) != GOOD)
               return(err);
            break;
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) func += sizeof(EREC_CLB_MAP);
      (U8 _far *) swapInfo += sizeof(EREC_CLB_MAP);
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetAlwaysMatch
**
**  Description:
**    Set addr, data or status or spare field of one event always match.
**
**  Parameters:
**    input:
**      field: ADDR_FIELD, DATA_FIELD or STATUS_FIELD or SPARE_FIELD
**      *xlxMap: Xilinx CLB map for this field of the event
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetAlwaysMatch(EVENT_FIELD field, EREC1_XLX_MAP _far *xlx1Map,
                                          EREC2_XLX_MAP _far *xlx2Map) {
   RETCODE err;
   U8 nibble, fieldSize;
   U16 _far *clb;
   XLX_BASE _far *base;
   XLX_FUNC _far *func;
   U8 *bitStreamBuffer;

   switch (field) {
      case ADDR_FIELD:
         clb = &(xlx2Map->addr0.clb);
         base = &(xlx2Map->addr0.base);
         func = &(xlx2Map->addr0.equal);
         fieldSize = EP_ADDR_SIZE;
         bitStreamBuffer=bitStreamBuffer2;
         break;
      case DATA_FIELD:
         clb = &(xlx1Map->data0.clb);
         base = &(xlx1Map->data0.base);
         func = &(xlx1Map->data0.equal);
         fieldSize = EP_DATA_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case STATUS_FIELD:
         clb = &(xlx1Map->status0.clb);
         base = &(xlx1Map->status0.base);
         func = &(xlx1Map->status0.equal);
         fieldSize = EP_STATUS_SIZE;
         bitStreamBuffer=bitStreamBuffer1;
         break;
      case SPARE_FIELD:
         clb = &(xlx2Map->spare0.clb);
         base = &(xlx2Map->spare0.base);
         func = &(xlx2Map->spare0.equal);
         fieldSize = EP_SPARE_SIZE;
         bitStreamBuffer=bitStreamBuffer2;
         break;
   }
   for (nibble = 0; nibble < (U8) (fieldSize / 4); nibble++) {
      switch (*base) {
         case BASE_F:
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, XLX_F, 0, 0, 1)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, XLX_G, 0, 0, 1)) != GOOD)
               return(err);
            break;
         default:
            if ((err = XlxModifyClb(bitStreamBuffer,
                                    XC3090, *clb, *func, 0, 0, 1)) != GOOD)
               return(err);
            break;
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) func += sizeof(EREC_CLB_MAP);
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetMultiStatus
**
**  Description:
**    Set status field of one event to multiple status.
**    For example:
**                 set EV1 to break EP at status SP or UP
**                     >Event 1 XXXXX XX SP UP
**                 we call the status field setting of EV1 "SP or UP"
**
**    There are 7 statuses(SP,UP,SR,SW,UR,UW,AK) for 68306 and  any
**    arbitrary combinations of these 7 statuses can be "or" together.
**    For 68306, the status are defined by 4 signals(ES[7..4]).
**    So there are 1 status nibbles(ES[7..4]).
**
**           status nibble 1
**        =======================
**           ES7 => FC2
**           ES6 => FC1
**           ES5 => FC0
**           ES4 => RW\
**
**    Two status X and Y can be set to "or" together if and only if status
**    X and Y are both consisting of the status signals residing in the same
**    status nibble.
**
**    For 68306, since there's only one status nibble, so every status could
**    be "or" together.
**
**        status  |  ES7 ES6 ES5 ES4  |  pattern  |  mask
**      ----------+-------------------+-----------+---------
**           SP   |   1   1   0   X   |   0x0C    |  0x0E
**           UP   |   0   1   0   X   |   0x04    |  0x0E
**           SR   |   1   0   1   1   |   0x0B    |  0x0F
**           SW   |   1   0   1   0   |   0x0A    |  0x0F
**           UR   |   0   0   1   1   |   0x03    |  0x0F
**           UW   |   0   0   1   0   |   0x02    |  0x0F
**           AK   |   1   1   1   X   |   0x0E    |  0x0E
**
**    But for 80C186/188, For example:
**       Only S, R, W, I, O, A, H can "or" with each other.
**       D can not "or" with S, R, W, I, O, A, H.
**       So there are two status groups of 80C186/188
**       (i.e. group {S,R,W,I,O,A,H} and group {D})
**
**        status  |  ES7 ES6 ES5 ES4 ES3 ES2 ES1 ES0  |  pattern  |  mask
**      ----------+-----------------------------------+-----------+---------
**           S    |   X   X   X   X   1   1   0   0   |   0x0C    |  0x0F
**           R    |   X   X   X   X   1   1   0   1   |   0x0D    |  0x0F
**           W    |   X   X   X   X   1   1   1   0   |   0x0E    |  0x0F
**           I    |   X   X   X   X   1   0   0   1   |   0x09    |  0x0F
**           O    |   X   X   X   X   1   0   1   0   |   0x0A    |  0x0F
**           A    |   X   X   X   X   1   0   0   0   |   0x08    |  0x0F
**           H    |   X   X   X   X   1   0   1   1   |   0x0B    |  0x0F
**           D    |   X   X   1   X   X   X   X   X   |   0x20    |  0x20
**
**        Note:  `0' means "Low", `1' means "High", `X' means "Don't Care".
**
**  Parameters:
**    input:
**      *xlxMap: Xilinx CLB map for this field of the event
**      status: this is a 16-bit data, each bit of it represent one status
**
**      bit -> 151413121110 9 8 7 6 5 4 3 2 1 0
**      rep ->  . . . . . . . . .AKUWURSWSRUPSP
**
**              Example: if the setting is "UR or SW" then status = 0x0018
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetMultiStatus(EREC1_XLX_MAP _far *xlx1Map,
                       EREC2_XLX_MAP _far *xlx2Map,
		       U16 status) {
   static const EP_STATUS_SIG _based(_segname("_CODE")) pattern[] = {
      0x0C /* SP */,
      0x04 /* UP */,
      0x0B /* SR */,
      0x0A /* SW */,
      0x03 /* UR */,
      0x02 /* UW */,
      0x0E /* AK */
      };
   static const EP_STATUS_SIG _based(_segname("_CODE")) mask[] = {
      0x0E /* SP */,
      0x0E /* UP */,
      0x0F /* SR */,
      0x0F /* SW */,
      0x0F /* UR */,
      0x0F /* UW */,
      0x0E /* AK */
      };

   RETCODE err;
   EP_STATUS_SIG bitDifferent = 0;
   U8 nibble, setNibble, s0, s1, ptn, msk;
   U16 _far *clb, _far *swapInfo;
   XLX_BASE _far *base;
   XLX_FUNC _far *func;
   BOOLEAN setAllStatus = YES, foundSetNibble = NO;

   clb = &(xlx1Map->status1.clb);
   base = &(xlx1Map->status1.base);
   func = &(xlx1Map->status1.equal);
   swapInfo = &(xlx1Map->status1.logicInputLocation);

   for (s0 = 0; s0 < EP_STATUSFC_NUM - 1; s0++) {
      if (status & (1 << s0)) {
         for (s1 = (U8) (s0 + 1); s1 < EP_STATUSFC_NUM; s1++) {
            if (status & (1 << s1)) {
               (EP_STATUS_SIG) bitDifferent |= (mask[s0] ^ mask[s1]);
               (EP_STATUS_SIG) bitDifferent |= (pattern[s0] & mask[s0]) ^
                                               (pattern[s1] & mask[s1]);
            }
         }
      }
      else setAllStatus = NO;
   }
   if (setAllStatus || (status == 0)) {
      for (nibble = 0; nibble < (EP_STATUSFC_SIZE / 4); nibble++) {
         switch (*base) {
            case BASE_F:
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, XLX_F, 0, 0, 1)) != GOOD)
                  return(err);
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, XLX_G, 0, 0, 1)) != GOOD)
                  return(err);
               break;
            default:
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, *func, 0, 0, 1)) != GOOD)
                  return(err);
               break;
         }
         (U8 _far *) clb += sizeof(EREC_CLB_MAP);
         (U8 _far *) base += sizeof(EREC_CLB_MAP);
         (U8 _far *) func += sizeof(EREC_CLB_MAP);
         (U8 _far *) swapInfo += sizeof(EREC_CLB_MAP);
      }
      return(GOOD);
   }
   for (nibble = 0; nibble < (EP_STATUSFC_SIZE / 4); nibble++) {
      switch (foundSetNibble) {
         case NO:
            if ((bitDifferent >> (nibble * 4)) & 0x000F) {
               setNibble = nibble;
               foundSetNibble = YES;
            }
            break;
         case YES:
            if ((bitDifferent >> (nibble * 4)) & 0x000F)
               return(ERR_STATUS_CANNOT_OR);
            break;
      }
   }
   for (nibble = 0; nibble < (EP_STATUSFC_SIZE / 4); nibble++) {
      switch (*base) {
         case BASE_F:
            if ((err = XlxModifyClb(bitStreamBuffer1,
                                    XC3090, *clb, XLX_F, 0, 0, 0)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(bitStreamBuffer1,
                                    XC3090, *clb, XLX_G, 0, 0, 0)) != GOOD)
               return(err);
            break;
         default:
            if ((err = XlxModifyClb(bitStreamBuffer1,
                                    XC3090, *clb, *func, 0, 0, 0)) != GOOD)
               return(err);
            break;
      }
      if (nibble == setNibble) {
         for (s0 = 0; s0 < EP_STATUSFC_NUM - 1; s0++) {
            if (status & (1 << s0)) {
               ptn = (U8) ((pattern[s0] >> (nibble * 4)) & 0x0F);
               msk = (U8) ((mask[s0] >> (nibble * 4)) & 0x0F);
               switch (*base) {
                  case BASE_F:
                     if ((err = XlxModifyClb(bitStreamBuffer1,
                                             XC3090, *clb, XLX_F,
                                   SwapLocation(ptn, *swapInfo),
                                   SwapLocation(msk, *swapInfo), 1)) != GOOD)
                        return(err);
                     if ((err = XlxModifyClb(bitStreamBuffer1,
                                             XC3090, *clb, XLX_G,
                                   SwapLocation(ptn, *swapInfo),
                                   SwapLocation(msk, *swapInfo), 1)) != GOOD)
                        return(err);
                     break;
                  default:
                     if ((err = XlxModifyClb(bitStreamBuffer1,
                                             XC3090, *clb, *func,
                                   SwapLocation(ptn, *swapInfo),
                                   SwapLocation(msk, *swapInfo), 1)) != GOOD)
                        return(err);
                     break;
               }
            }
         }
      }
      else {   /* for single status (no "or" case) */

         for (s0 = 0; (status & (1 << s0)) == 0; s0++);   /* skip those 0's */

         ptn = (U8) ((pattern[s0] >> (nibble * 4)) & 0x0F);
         msk = (U8) ((mask[s0] >> (nibble * 4)) & 0x0F);
         switch (*base) {
            case BASE_F:
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, XLX_F,
                             SwapLocation(ptn, *swapInfo),
                             SwapLocation(msk, *swapInfo), 1)) != GOOD)
                  return(err);
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, XLX_G,
                             SwapLocation(ptn, *swapInfo),
                             SwapLocation(msk, *swapInfo), 1)) != GOOD)
                  return(err);
               break;
            default:
               if ((err = XlxModifyClb(bitStreamBuffer1,
                                       XC3090, *clb, *func,
                             SwapLocation(ptn, *swapInfo),
                             SwapLocation(msk, *swapInfo), 1)) != GOOD)
                  return(err);
               break;
         }
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) func += sizeof(EREC_CLB_MAP);
      (U8 _far *) swapInfo += sizeof(EREC_CLB_MAP);
   }
   return(GOOD);
}

/******************************************************************************
**
**  SetOneEvent
**
**  Description:
**    Set one busevent
**
**  Parameters:
**    input:
**      *eventStruct: the event setting
**      *xlx1Map: 1st Xilinx CLB map for this event eventPhysicalID
**      *xlx2Map: 2nd Xilinx CLB map for this event eventPhysicalID
**      eventPhysicalID: the physical location of the event
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetOneEvent(BUS_EVENT *eventStruct,
                    EREC1_XLX_MAP _far *xlx1Map,
                    EREC2_XLX_MAP _far *xlx2Map,
                    U8 eventPhysicalID) {
   RETCODE err;

   if (eventStruct->enable == NO) return(ERR_SET_A_DISABLE_EVENT);
   switch (eventStruct->addrSpec) {
      case UNDEFINE_ADDR:
	 if ((err = SetAlwaysMatch(ADDR_FIELD,
				   &xlx1Map[eventPhysicalID],
				   &xlx2Map[eventPhysicalID]))
             != GOOD) return(err);
         break;
      case RANGE_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, 0xFFFFFFFF)) != GOOD) return(err);
	 if ((err = SetRangeAddr(LOW_BOUNDARY,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow)) != GOOD) return(err);
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID+1],
	     &xlx2Map[eventPhysicalID+1],
	      eventStruct->addrHigh, 0xFFFFFFFF)) != GOOD) return(err);
	 if ((err = SetRangeAddr(HIGH_BOUNDARY,
	     &xlx1Map[eventPhysicalID+1],
	     &xlx2Map[eventPhysicalID+1],
	      eventStruct->addrHigh)) != GOOD) return(err);
         break;
      case SINGLE_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, 0xFFFFFFFF)) != GOOD) return(err);
         break;
      case MASK_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, eventStruct->addrHigh)) != GOOD)
            return(err);
         break;
   }
   if ((eventStruct->addrSpec != RANGE_ADDR) &&
       ((eventPhysicalID & 1) == 0) &&
       ((err = XlxModifyClb(bitStreamBuffer2,
                            XC3090, xlx2Map[eventPhysicalID+1].addr7.clb,
                            xlx2Map[eventPhysicalID+1].addr7.greatOrLess,
                            0, 0, 1))
         != GOOD))
      return(err);
   switch (eventStruct->dataSpec) {
      case UNDEFINE_DATA:
	 if ((err = SetAlwaysMatch(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID]))
             != GOOD) return(err);
         break;
      case SINGLE_DATA:
	 if ((err = SetSingleMask(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              eventStruct->dataLow, 0xFFFFFFFF)) != GOOD) return(err);
         break;
      case MASK_DATA:
	 if ((err = SetSingleMask(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              eventStruct->dataLow, eventStruct->dataHigh)) != GOOD)
            return(err);
         break;
   }
   if ((err = SetMultiStatus(&xlx1Map[eventPhysicalID],
			     &xlx2Map[eventPhysicalID],
			    eventStruct->status.statusFc))
           != GOOD) return(err);
/* set status Ds group and DRAM bit */
	 if ((err = SetSingleMask(STATUSDS_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              ((eventStruct->status.statusDsLow & 0x03) |
               ((eventStruct->status.statusCsLow & 0x0100) >> 0x05) ),
              ((eventStruct->status.statusDsHigh & 0x03) |
               ((eventStruct->status.statusCsHigh & 0x0100) >> 0x05) )
              )) != GOOD) return(err);
/* set status Cs group except DRAM bit */
	 if ((err = SetSingleMask(STATUSCS_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              (eventStruct->status.statusCsLow & 0xFF ),
              (eventStruct->status.statusCsHigh & 0xFF)
              )) != GOOD) return(err);
/* set spare */
	 if ((err = SetSingleMask(SPARE_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              eventStruct->spareLow,
              eventStruct->spareHigh)) != GOOD) return(err);
   return(GOOD);
}

/******************************************************************************
**
**  SetOneEventP
**
**  Description:
**    Set one busevent with PowerPack status protocal
**
**  Parameters:
**    input:
**      *eventStruct: the event setting
**      *xlx1Map: 1st Xilinx CLB map for this event eventPhysicalID
**      *xlx2Map: 2nd Xilinx CLB map for this event eventPhysicalID
**      eventPhysicalID: the physical location of the event
**    output:
**      <none>
**
*******************************************************************************/
RETCODE SetOneEventP(BUS_EVENT *eventStruct,
                    EREC1_XLX_MAP _far *xlx1Map,
                    EREC2_XLX_MAP _far *xlx2Map,
                    U8 eventPhysicalID) {
   RETCODE err;

   if (eventStruct->enable == NO) return(ERR_SET_A_DISABLE_EVENT);
   switch (eventStruct->addrSpec) {
      case UNDEFINE_ADDR:
	 if ((err = SetAlwaysMatch(ADDR_FIELD,
				   &xlx1Map[eventPhysicalID],
				   &xlx2Map[eventPhysicalID]))
             != GOOD) return(err);
         break;
      case RANGE_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, 0xFFFFFFFF)) != GOOD) return(err);
	 if ((err = SetRangeAddr(LOW_BOUNDARY,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow)) != GOOD) return(err);
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID+1],
	     &xlx2Map[eventPhysicalID+1],
	      eventStruct->addrHigh, 0xFFFFFFFF)) != GOOD) return(err);
	 if ((err = SetRangeAddr(HIGH_BOUNDARY,
	     &xlx1Map[eventPhysicalID+1],
	     &xlx2Map[eventPhysicalID+1],
	      eventStruct->addrHigh)) != GOOD) return(err);
         break;
      case SINGLE_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, 0xFFFFFFFF)) != GOOD) return(err);
         break;
      case MASK_ADDR:
	 if ((err = SetSingleMask(ADDR_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
	      eventStruct->addrLow, eventStruct->addrHigh)) != GOOD)
            return(err);
         break;
   }
   if ((eventStruct->addrSpec != RANGE_ADDR) &&
       ((eventPhysicalID & 1) == 0) &&
       ((err = XlxModifyClb(bitStreamBuffer2,
                            XC3090, xlx2Map[eventPhysicalID+1].addr7.clb,
                            xlx2Map[eventPhysicalID+1].addr7.greatOrLess,
                            0, 0, 1))
         != GOOD))
      return(err);
   switch (eventStruct->dataSpec) {
      case UNDEFINE_DATA:
	 if ((err = SetAlwaysMatch(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID]))
             != GOOD) return(err);
         break;
      case SINGLE_DATA:
	 if ((err = SetSingleMask(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              eventStruct->dataLow, 0xFFFFFFFF)) != GOOD) return(err);
         break;
      case MASK_DATA:
	 if ((err = SetSingleMask(DATA_FIELD,
	     &xlx1Map[eventPhysicalID],
	     &xlx2Map[eventPhysicalID],
              eventStruct->dataLow, eventStruct->dataHigh)) != GOOD)
            return(err);
         break;
   }
/* set status */
   if ((err = SetSingleMask(STATUS_FIELD,
       &xlx1Map[eventPhysicalID],
       &xlx2Map[eventPhysicalID],
        (eventStruct->status2.status2Low),
        (eventStruct->status2.status2High)
        )) != GOOD) return(err);
/* set spare */
   if ((err = SetSingleMask(SPARE_FIELD,
       &xlx1Map[eventPhysicalID],
       &xlx2Map[eventPhysicalID],
        eventStruct->spareLow,
        eventStruct->spareHigh)) != GOOD) return(err);
/* */
   return(GOOD);
}

/******************************************************************************
**
**  ClearAllEvent
**
**  Description:
**    Clear all busevent
**
**  Parameters:
**    input:
**      *xlx1Map: 1st Xilinx CLB map for this
**                event with "eventPhysicalID = 0"
**      *xlx2Map: 2nd Xilinx CLB map for this
**                event with "eventPhysicalID = 0"
**    output:
**      <none>
**
*******************************************************************************/
RETCODE ClearAllEvent(EREC1_XLX_MAP _far *xlx1Map,
                      EREC2_XLX_MAP _far *xlx2Map) {
   RETCODE err;
   U8 nibble, nibbleQuantity;
   U16 _far *clb;
   XLX_BASE _far *base;
   XLX_FUNC _far *funcGL, _far *funcEQ;

/* 1st xilinx clb map */

   clb = &(xlx1Map[0].data0.clb);
   base = &(xlx1Map[0].data0.base);
   funcGL = &(xlx1Map[0].data0.greatOrLess);
   funcEQ = &(xlx1Map[0].data0.equal);
   nibbleQuantity = EREC1_EVENT_QUANTITY * (EP_DATA_SIZE +
                    EP_STATUS_SIZE) / 4;
   for (nibble = 0; nibble < nibbleQuantity; nibble++) {
      switch (*base) {
         case BASE_F:
            if ((err = XlxModifyClb(bitStreamBuffer1, XC3090, *clb,
                                    XLX_F, 0, 0, 0)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(bitStreamBuffer1, XC3090, *clb,
                                    XLX_G, 0, 0, 0)) != GOOD)
               return(err);
            break;
         default:
            if ((*funcGL != XLX_X) &&
                ((err = XlxModifyClb(bitStreamBuffer1, XC3090, *clb,
                                     *funcGL, 0, 0, 0)) != GOOD))
               return(err);
            if ((*funcEQ != XLX_X) &&
                ((err = XlxModifyClb(bitStreamBuffer1, XC3090, *clb,
                                     *funcEQ, 0, 0, 0)) != GOOD))
               return(err);
            break;
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) funcGL += sizeof(EREC_CLB_MAP);
      (U8 _far *) funcEQ += sizeof(EREC_CLB_MAP);
   }

/* 2nd xilinx clb map */

   clb = &(xlx2Map[0].addr0.clb);
   base = &(xlx2Map[0].addr0.base);
   funcGL = &(xlx2Map[0].addr0.greatOrLess);
   funcEQ = &(xlx2Map[0].addr0.equal);
   nibbleQuantity = EREC1_EVENT_QUANTITY * (EP_ADDR_SIZE +
                    EP_SPARE_SIZE) / 4;
   for (nibble = 0; nibble < nibbleQuantity; nibble++) {
      switch (*base) {
         case BASE_F:
            if ((err = XlxModifyClb(bitStreamBuffer2, XC3090, *clb,
                                    XLX_F, 0, 0, 0)) != GOOD)
               return(err);
            if ((err = XlxModifyClb(bitStreamBuffer2, XC3090, *clb,
                                    XLX_G, 0, 0, 0)) != GOOD)
               return(err);
            break;
         default:
            if ((*funcGL != XLX_X) &&
                ((err = XlxModifyClb(bitStreamBuffer2, XC3090, *clb,
                                     *funcGL, 0, 0, 0)) != GOOD))
               return(err);
            if ((*funcEQ != XLX_X) &&
                ((err = XlxModifyClb(bitStreamBuffer2, XC3090, *clb,
                                     *funcEQ, 0, 0, 0)) != GOOD))
               return(err);
            break;
      }
      (U8 _far *) clb += sizeof(EREC_CLB_MAP);
      (U8 _far *) base += sizeof(EREC_CLB_MAP);
      (U8 _far *) funcGL += sizeof(EREC_CLB_MAP);
      (U8 _far *) funcEQ += sizeof(EREC_CLB_MAP);
   }

   return(GOOD);
}

/********************************* EOF *************************************/
