;*******************************************************************
;      MICEpack 68306 CAN Program 
;   author : Yi-tsang Tsai in MICROTEK PDL on DEC/01/1993
;*******************************************************************
;---------------------------------------------------------------------
;        GET_IF & SAVE_PC
;---------------------------------------------------------------------
;
;   EA9  EA8  EA7 |  TRIGGER  |  OFFSET
;  ---------------+-----------+---------------
;    0    0    X  | NOTHING   |   XXXX
;    0    1    1  | NEMFC0    |   0180H    (SOURCE,read)
;    1    0    1  | NEMFC1    |   0280H    (DESTINATION,write)
;    1    1    1  | CANEND    |   0380H
;
;  note : EA7 = HIGH LEVEL, CAN RAM IS DISABLED
;         CAN END location at xxxxxx6AH address.
;--------------------------------------------------------------------
_canEndTrig:    equ  10380h
_spBuf:         equ  1006ah-4*1
_baseAdd:       equ  1006ah-4*2
canBuf1:        equ  06Ah
canBuf2:        equ  070h
souTrigOs:      equ  0180h
desTrigOs:      equ  0280h
notReadyFlg:    equ  081h

;-----------------------------------------------------------------------
;       Set CAN RAM's bace address
;      ----------------------------
;   modify source code as below:
;  1. _canSavePc+2.3.4.5      bit31..bit10   bit9..bit0
;  2. _canSavePc+8.9.A.B            |            |
;  3. _canGetIf+2.3.4.5             |            |____ keep original code
;  4. _canGetIf+8.9.A.B             |_________________ replaced them
;                                                      with base address
;                                                      of CAN by User setting
;                                                      or default [0] value.
;                                                             
;-----------------------------------------------------------------------
                ORG  00H
canStart_P:
_canSavePc:
                movea.l _spBuf,a7               ;save ssp
                move.b  d0,_canEndTrig          ;trigger Can end
                rte
                nop
_canGetIf:      
                move.l  a7,_spBuf               ;get ssp value
                movea.l _baseAdd,a7             ;set Can's base address
                bra     _canEnnd                ;Let EPSTOP active.
                ORG  06AH
_canEnnd:             
                nop
                bra  _canSavePc                 ; &&end

                org   100000h
_canVector:
                .long  _canGetIf                ; &&end

;-----------------------------------------------------------------------
;               GET   REGISTER
;-----------------------------------------------------------------------
rgBuf:          equ     06ah-4*1
regBufA5:       equ     06ah-4*2
regBufSR:       equ     028h
                org     0100h
_getRegst:
                move    sr,regBufSR(a7)         ;get SR
                move.l  a6,rgBuf(a7)            ;get a6 register
                movea.l a7,a6                   ;set a6 is a pointer of
                adda.l  #rgBuf,a6               ;can buffer.

                movem.l a0-a5/d0-d7,-(a6)       ;get a5-a0/d7-d0 register.
                move    usp,a5
                move.l  a5,-(a6)                ;get USP 

                movea.l rgBuf(a7),a6            ;restore A6
                movea.l regBufA5(a7),a5         ;restore A5
                move    regBufSR(a7),sr         ;restore SR
                bra.b   _canEnnd1               ;Let EP stop.

                ORG     016AH
_canEnnd1:            
                nop
                bra.b _getRegst                  ; &&end

;-----------------------------------------------------------------------
;               save register
;-----------------------------------------------------------------------
rgBufSave:      equ     06ah-16*4
saveCntWord:    equ     070h
;     ----------------------------------------------------
;     save register control word : (byte)
;     bit7  bit6  bit5  bit4  bit3  bit2  bit1  bit0
;       X     X     X     X     X     X     X    |___     USP
;
;             [1] -- need to save.
;             [0] -- does not need to save.
;              X  -- unused
;     ----------------------------------------------------
                org     0200h
_saveRegst:
                movea.l a7,a6
                adda.l  #rgBufSave,a6           ;get parameter location in CAN
                btst.b  #00h,saveCntWord(a7)    ;check Save USP flag ?
                beq.b   SAVER1                  ;jump if it need not save USP.
                move.l  (a6),a5
                move    a5,usp                  ;save USP
SAVER1:
                adda.l  #04h,a6                 ;adjust pointer.
                movem.l (a6)+,a0-a5/d0-d7       ;SAVE D0-D7/A0-A5
                movea.l rgBufSave+15*4(a7),a6   ;save a6 register.
                bra.b   _canEnnd2

                org     026AH
_canEnnd2:
                nop
                bra.b   _saveRegst              ; &&end
;-----------------------------------------------------------------------
;     START OF REGISTER BUFFER     <--- 28H
;                                SR    (size : word)  
;                                  <--- 2AH            
;                                USP                   
;                                  <--- 2EH            
;                                D0                    
;                                  <--- 32H            
;                                D1                    
;                                  <--- 36H            
;                                D2                    
;                                  <--- 3AH            
;                                D3                    
;                                  <--- 3EH            
;                                D4                    
;                                  <--- 42H            
;                                D5                    
;                                  <--- 46H            
;                                D6                    
;                                  <--- 4AH            
;                                D7                    
;                                  <--- 4EH            
;                                A0                    
;                                  <--- 52H            
;                                A1                    
;                                  <--- 56H            
;                                A2                    
;                                  <--- 5AH            
;                                A3                    
;                                  <--- 5EH            
;                                A4                    
;                                  <--- 62H            
;                                A5                    
;                                  <--- 66H            
;                                A6                    
;                                  <--- 6AH            
;-----------------------------------------------------------------------
;  Command address1 LENGTH address2
;
;  A0 -- Source starting address (address1)
;        A0 --> low word : data pattern length if Fill command
;  D0 -- length (ending address - starting address, start point is 0.)
;        (length = LENGTH - 1   , size byte).
;        (length = LENGTH/2 - 1 , size word).
;  A1 -- Destination address (address2).
;  A7 -- Can RAM base address.
;  A6 -- Source Trigger.
;  A5 -- Destination Trigger.
;  A4 -- the pointer of CAN data buffer.
;  D1 -- Control word 
;        bit15 bit14 bit13 bit12 bit11 bit10  bit9  bit8
;
;
;
;        bit7  bit6  bit5  bit4  bit3  bit2   bit1  bit0
;         |                                          |____ size word[1]/byte[0]
;         |                                             
;         |_______________________________________________  Memory verify flag
;                                                           OFF[0] / ON[1]
;   A2 -- working register.                         
;   A3 -- working register.
;-----------------------------------------------------------------------
; set parameter
                org  0300h
_setParaSt:
                movea.l a7,a2
                adda.l  #canBuf1,a2
                move    -(a2),d1        ;set control word.
                movea.l -(a2),a0        ;set source starting address
                move.l  -(a2),d0        ;get LENGTH.
                btst    #00h,d1         ;check size word/byte ?
                beq     setParByte
                lsr.l   #01h,d0         ;LENGTH/2
setParByte:
                subq.l  #01h,d0         ;set length.
                movea.l -(a2),a1        ;set destination  address
                movea.l a7,a6           
                movea.l a7,a5
                movea.l a7,a4
                adda.l  #souTrigOs,a6   ;set source trigger point.
                adda.l  #desTrigOs,a5   ;set destination trigger point.
                adda.l  #canBuf1,a4    ;set Can data buffer endding point.
                bra     _setParaEd

                org  036Ah
_setParaEd:   
                nop
                bra     _setParaSt      ; &&end

;-----------------------------------------------------------------------
;read byte  from target/EMM
;    pattern length maximum = 44h bytes
                org  0400h
_disByteSt:     
                movea.l a4,a2
                move.l  d0,d6           ;load length, maximum 44h bytes
                                        ;d6 : loop counter
disByte1:     
                move.b  d0,(a6)         ;source trigger
                move.b  (a0)+,d5
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   notRdy1
                move.b  d5,-(a2)
                dbf     d6,disByte1
                bra.b   _disByteEd
notRdy1:
                subq.l  #01h,a0          ;compute not ready address.
                move.l  a0,canBuf2+2(a7) ;save not ready address.
                bra.b   _disByteEd

                org 046ah
_disByteEd:
                nop
                bra.b   _disByteSt      ; &&end

;-----------------------------------------------------------------------
;read word  from target/EMM
;    pattern length maximum = 44h bytes
                org  0500h
_disWordSt:     
                movea.l a4,a2           ;get the start point of Can data Buffer 
                move.l  d0,d6           ;counter max value 22h words
                                        ; d6: loop counter.
disWord1:     
                move.b  d0,(a6)         ;source trigger
                move    (a0)+,d5
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   notRdy2
                ror.w   #08h,d5          ;swap bit7-0, bit15-8
                move    d5,-(a2)
                dbf     d6,disWord1
                bra.b   _disWordEd
notRdy2:
                subq.l  #02h,a0          ;compute not ready address.
                move.l  a0,canBuf2+2(a7) ;save not ready address.
                bra.b   _disWordEd

                org 056ah
_disWordEd:
                nop
                bra.b   _disWordSt       ; &&end
;-----------------------------------------------------------------------
;write data to target/emm with size-byte.
;    pattern length maximum = 20h  (old : 22h, 11/03/1994)
;   d3 : byte,verify error data.               11/03/1994
;   d5 : byte,expectant data.                  11/03/1994
; On 11/03/1994, to support data information if verify failure.
;-----------------------------------------------------------------------
                org     0600h
_wrByteSt:                              
                movea.l a4,a2           ;get the start point of Can data Buffer 
                move.l  d0,d6           ;d6: loop counter, temp
                clr     d7              ;clear d7, pattern length counter
wrByte1:     
                move.b  -(a2),d5        ;get fill data from can buffer.
                move.b  d0,(a5)         ;trigger of destination.
                move.b  d5,(a1)         ;fill data to Target/Emm
                btst    #07h,d1         ;verify flag on ?
                beq.b   wrByte2         ;  off[0],on[1]
                move.b  d0,(a5)         ;trigger of destination.
                                        ;modified by Tsai,11/03/1994,begin.
                move.b  (a1),d3         ;verify data (read data from emm/target)
                cmp.b   d3,d5           ;verify data
                                        ;Tsai,11/03/1994,End
                bne.b   wrError
wrByte2:
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   wrNotRdy
                adda.l  #01h,a1         ;next address
                add     #01h,d7         ;pattern index increase
                cmp     a0,d7           ;check pattern pointer end ?
                bne.b   wrByte3         ;no,jump
                clr     d7              ;yes, clear index
                movea.l a4,a2           ;     reload data buffer pointer to
                                        ;     start location.
wrByte3:
                sub.l   #01h,d6         ;decrease loop counter
                cmp.l   #-1,d6          ;loop end ?
                bne.b   wrByte1         ; no,jump to loop start.
                bra.b   _wrByteEd       ; yes, go to end (for stop EP).
wrError:
                move.b  #01h,canBuf2+1(a7) ;set error flag
wrNotRdy:
                move.l  a1,canBuf2+2(a7)   ;save [error]/[not ready] address
                bra.b   _wrByteEd

                org     066ah
_wrByteEd:
                nop
                bra.b   _wrByteSt
                                           ; &&end

;-----------------------------------------------------------------------
;write data to target/emm with size-word.
;    pattern length maximum = 22h
;    pattern length maximum = 20h  (old : 22h, 11/03/1994)
;   d3 : word,verify error data.               11/03/1994
;   d5 : word,expectant data.                  11/03/1994
; On 11/03/1994, to support data information if verify failure.
;-----------------------------------------------------------------------
;
                org     0700h
_wrWordSt:                              
                movea.l a4,a2           ;get the start point of Can data Buffer 
                move.l  d0,d6           ;d6 : loop counter, temp
                clr     d7              ; pattern index (counter)
                move    d7,d4           ;d4 : even flag,even[0000]/odd[ffffh]
wrWord1:     
                lsl.w   #08h,d5         ;d5 : data buffer,temp
                                        ;     shift low byte to high byte.
                move.b  -(a2),d5        ;get pattern from can buffer.
                bra.b   _wrWord3        ;jump to adjust pattern index
wrWord1_A:
                not     d4              ;change,even flag.
                bne.b   wrWord1         ;jump if even byte.
;
                move.b  d0,(a5)         ;trigger of destination.
                move    d5,(a1)         ;fill data to Target/Emm
                btst    #07h,d1         ;verify flag on ?
                beq.b   wrWord2         ;  off[0],on[1]
                                        ;Tsai,11/03/1994,begin
                move.b  d0,(a5)         ;trigger of destination.
                move    (a1),d3         ;verify data (read data from emm/target)
                cmp     d3,d5           ;verify data
                                        ;Tsai,11/03/1994,end
                bne.b   wrError_w
wrWord2:
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   wrNotRdy_w
                adda.l  #02h,a1         ;next address
                sub.l   #01h,d6         ;decrease d6 loop counter
                cmp.l   #-1h,d6         ;loop end ?
                bne.b   wrWord1
                bra.b   _wrWordEd_w     ;go to end(for stop ep)
wrError_w:
                move.b  #01h,canBuf2+1(a7) ;set error flag
wrNotRdy_w:
                move.l  a1,canBuf2+2(a7)   ;save [error]/[not ready] address
                bra.b   _wrWordEd_w

                org     076ah
_wrWordEd_w:
                nop
                bra.b   _wrWordSt       ; go to begin of program

                org     0774h
_wrWord3:
                add     #01h,d7         ;increase pattern index (counter)
                cmp     a0,d7           ;check pattern index =  pattern length ?
                bne.b   wrWord1_A       ;no, just return
                clr     d7              ;yes, clear pattern index &
                move.l  a4,a2           ;     get pattern start location
                                        ;     for pattern accessing.
                bra.b   wrWord1_A       ; &&end

;-----------------------------------------------------------------------
; test target/emm with size byte (fill test pattern only)
                org     0800h
_tstByte1St:
                move.l  d0,d2           ;get byte length (d2:working length)
                movea.l a0,a2           ;get start address(a2:working add)
tstByte1:
                move    a2,d4           ;compute test pattern
                move    a2,d5
                lsr     #08h,d4
                eor.b   d4,d5           ;high byte/low XOR.
                move.b  d0,(a6)         ;trigger with source FC[0..2]
                move.b  d5,(a2)+        ;write test pattern to Target/emm
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   _tstNotRdy1
                sub.l   #01h,d2         ;DEC length.
                cmp.l   #-01h,d2        ;check end of loop ?
                bne.b   tstByte1
                bra.b   _tstByte1Ed
_tstNotRdy1:
                subq.l  #01h,a2
                move.l  a2,canBuf2+2(a7)   ;save [not ready] address
                bra.b   _tstByte1Ed

                org     086ah
_tstByte1Ed:
                nop
                bra.b   _tstByte1St     ; &&end

;-----------------------------------------------------------------------
;  test target/emm with size byte (verify test pattern)
;
                org     0900h
_tstByte2St:
                move    a0,d4
                move    a0,d5
                lsr     #08h,d4
                eor.b   d4,d5           ;high byte/low XOR.
                move.b  d0,(a6)         ;trigger with source FC[0..2]
                cmp.b   (a0),d5         ;compare Target/emm data with test patt
                bne.b   tstByteErr
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   tstByteErr
                not.b   d5

                move.b  d0,(a6)         ;trigger with source FC[0..2]
                move.b  d5,(a0)         ;write Complement pattern
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   tstByteErr

                move.b  d0,(a6)         ;trigger with source FC[0..2]
                cmp.b   (a0)+,d5        ;compare Target/emm data with test 
                bne.b   tstByteErr2
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   tstByteErr2
                sub.l   #01h,d0         ;DEC length.
                cmp.l   #-01h,d0        ;check end of loop ?
                bne.b   _tstByte2St
                clr.b   canBuf2+1(a7)   ;clear error flage (Pass)
                bra.b   _tstByte2Ed

tstByteErr2:
                subq.l  #01h,a0         ;adjust Error/notRdy address
tstByteErr:  
                move.l  a0,canBuf2+2(a7)   ;save [Error]/[not ready] address
                bra.b   _tstByte2Ed

                org     096ah
_tstByte2Ed:        
                nop
                bra.b   _tstByte2St      ; &&end
;-----------------------------------------------------------------------
; test target/emm with word byte (fill test pattern only)
                org     0A00h
_tstWord1St:
                move.l  d0,d2           ;get byte length (d2:working length)
                movea.l a0,a2           ;get start address(a2:working add)
                move.l  a0,d3
tstWord1:
                move    d3,d4           ;compute test pattern
                move    d3,d5
                lsr     #08h,d4
                eor.b   d4,d5           ;even byte test pattern
                lsl     #08h,d5
                andi    #ff00h,d5
                addq.l  #01h,d3         ;next byte address.

                move    d3,d4
                move    d3,d6
                lsr     #08h,d4
                eor.b   d4,d6           ;odd byte test pattern
                andi    #00ffh,d6
                or      d5,d6           ;combine even/odd test pattern.
                addq.l  #01h,d3         ;next byte address

                move.b  d0,(a6)         ;trigger with source FC[0..2]
                move    d6,(a2)+        ;write test pattern to Target/emm
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   tstNotRdy_Wr1
                sub.l   #01h,d2         ;DEC length.
                cmp.l   #-01h,d2        ;check end of loop ?
                bne.b   tstWord1
                bra.b   _tstWord1Ed
tstNotRdy_Wr1:
                subq.l  #02h,a2
                move.l  a2,canBuf2+2(a7)   ;save [not ready] address
                bra.b   _tstWord1Ed

                org     0A6ah
_tstWord1Ed:
                nop
                bra.b   _tstWord1St     ; &&end

;-----------------------------------------------------------------------
;  test target/emm with size byte (verify test pattern)
;
                org     0b00h
_tstWord2St:
                move.l  a0,d3

tstWord2_L:     move    d3,d4           ;compute test pattern
                move    d3,d5
                lsr     #08h,d4
                eor.b   d4,d5           ;even byte test pattern
                lsl     #08h,d5
                andi    #ff00h,d5
                addq.l  #01h,d3         ;next byte address.

                move    d3,d4
                move    d3,d6
                lsr     #08h,d4
                eor.b   d4,d6           ;odd byte test pattern
                andi    #00ffh,d6
                or      d5,d6           ;combine even/odd test pattern.
                addq.l  #01h,d3         ;next byte address

                move.b  d0,(a6)         ;trigger with source FC[0..2]
                cmp     (a0),d6         ;compare Target/emm data with test patt
                bne.b   _tstWordErr
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   _tstWordErr

                not     d6
                move.b  d0,(a6)         ;trigger with source FC[0..2]
                move    d6,(a0)         ;write Complement pattern

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   _tstWordErr

                move.b  d0,(a6)         ;trigger with source FC[0..2]
                cmp     (a0)+,d6        ;compare Target/emm data with test 
                bne.b   _tstWordErr

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   _tstWordErr
                sub.l   #01h,d0         ;DEC length.
                cmp.l   #-01h,d0        ;check end of loop ?
                bne.b   tstWord2_L
                clr.b   canBuf2+1(a7)   ;clear error flage (Pass)
                bra.b   _tstWord2Ed


                org     0b6ah
_tstWord2Ed:
                nop
                bra.b   _tstWord2St

                org     0b76h
_tstWordErr:
                sub.l   #02h,d3         ;adjust Error/notRdy address
                move.l  d3,canBuf2+2(a7)   ;save [Error]/[not ready] address
                bra.b   _tstWord2Ed     ;&&end

;-----------------------------------------------------------------------
;      READ/WRITE/TEST COMMAND  CAN PROGRAM REPLICATION
; 
;   address offset |  message       |    note
;   ---------------+----------------+-----------------------
;        70h       |   Ready Flag   |  [01h]: not_Ready
;                  |                |  [00h]: ready
;                  |                |         set it by Cp
;                  |                |         for Ep checking.
;   ---------------+----------------+-----------------------
;        71h       |  test result   |  [00h]:Pass Flag
;                  |                |  default 01h by Cp.
;                  |                |  cleared by Ep if
;                  |                |  test OK.
;   ---------------+----------------+-----------------------
;        72h       |   error        |   EA[31..24]
;        73        |   or           |   EA[23..16]
;        74        |   not ready    |   EA[15..8]
;        75        |   address      |   EA[7..0]
;                                    
;------------------------------------------------------------------------
;========================================================================
; Reset Command of MICEpack 68306
;
;  If CPU 68306 NMI can not use auto vector,
;     the H/W has to inster 01h code to
;     ED[8..15](D[0..7] of 68306,low byte)
;     during IACK cycle after reset.
;-----------------------------------------------------------------------
        org 0000h
_resSt:
        .LONG   00000060H       ;stack pointer.
        .LONG   epStart         ;PC start address
epStart:
        nop
        nop
        bra     _resEd
        nop
        nop

        org     epStart*4       ;program start if NMI is not Autovector
_nmiResSt:
        .LONG   epStart         ;PC start address


         org 006AH
_resEd:
         nop
         bra    _resSt

         org  007ch
_nmiVec: .LONG  epStart         ; &&end  NMI Auto VECTOR
;-----------------------------------------------------------------------
;       Set CAN RAM's bace address
;      ----------------------------
;   modify source code as below:
;  1. _setBaseSt+2.3.4.5      bit31..bit10   bit9..bit0
;  2. _setBaseSt+8.9.A.B            |            |
;                                   |            |____ keep original code
;                                   |_________________ replaced them
;                                                      with base address
;                                                      of CAN by User setting
;                                                      or default [0] value.
;                                                             
;-----------------------------------------------------------------------
         org  0c00h
_setBaseSt:
         move.l a7,_spBuf       ;save current A7 register to 066h
         movea.l _baseAdd,a7    ;replace new A7 register by 062h
         bra     _setBaseEd     ; to jump here location.

        org   0c6ah
_setBaseEd:
        nop
        bra     _setBaseSt      ; &&end
;-----------------------------------------------------------------------
;       Search  memory command
;     ===========================
; D0 : memory length - 1.
; A0 : search pattern length. (maximum = 28h)
; A1 : start address of memory.
; A4 : start location of search pattern,(buffer 1 pointer)
; A5 : destination trigger DFC[0..2]
; A7 : can base address.
; ................................................................
; working register
; ----------------
; D2 : memory counter.
; D3 : search pattern counter.
;
        org     0d00h
_searchSt:
;        compute address loop counter D2
;        end of searching if D2 = -2.
                move.l  d0,d2
                sub.l   a0,d2

;       compute pattern length loop counter D3
                subq    #1,a0           ;length -1 = loop counter of pattern
CANSEAR0:                               ; end at -1 for DBF instruction.
                move.l  a4,a2            ;point to canBuf1 location. (A2)
                move    a0,d3

                movea.l a1,a3   ;pointer of target/emm a3
CANSEAR1:
                move.b  d0,(a5)         ;trigger with destination FC[0..2]
                move.b  (a3)+,d5        ;read target/emm data

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CANSENOTRDY

                cmp.b   -(a2),d5        ;compare pattern(can buffer1).
                bne.b   CANSEAR2        ;target/emm data = can buffer1 ?
                                        ; No,jump to adjust next memory
                                        ; address.
                dbf     d3,CANSEAR1     ;check pattern end ?
                bra.b   CANSEAROK       ;jump if match.
CANSEAR2:
                adda.l  #01h,a1         ;increase address pointer.
                sub.l   #01,d2          ;decrease address loop counter.
                cmp.l   #-2,d2          ;check memory end ?
                bne.b   CANSEAR0        ;jump if it does not end ?
                bra.b   _searchEd
CANSEAROK:
                clr.b   canBuf2+1(a7)   ;clear search flag (found).
CANSENOTRDY:
                move.l  a1,canBuf2+2(a7) ;save match/NotReady address .
                bra.b   _searchEd

                org     0d6ah
_searchEd:
                nop
                bra.b   _searchSt       ;&&end
;-----------------------------------------------------------------------
;               Compare memory with size-byte
;               =============================
;  A0 -- block1 start address (source)
;  A1 -- block2 start address (destination)
;  D0 -- memory length.
;  A6 -- source trigger.
;  A5 -- destination trigger
;  .........................................
;  A2 -- reply flag pointer.
;........................................................................
                org     0e00h
_compByteSt:
                movea.l a7,a2
                adda.l  #canBuf2+1,a2   ; A2 point to reply flag.
CMPByte1:
                move.b  d0,(a6)         ;trigeer of source
                move.b  (a0)+,d5        ;read data from source area

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CMPNOTRDY_B1     ;Jump if not ready.

                move.b  d0,(a5)         ;trigger of destination
                move.b  (a1)+,d6        ;read data from destination (Block 2)

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CMPNOTRDY_B2     ;jump if not ready.

                cmp.b   d6,d5           ;compare block-1 & block-2 data.
                bne.b   COMNONEQU_B

                subq.l  #1,d0           ;descrease loop counter, D0
                cmp.l   #-1,d0          ;check,loop End ?
                bne.b   CMPByte1        ;No,jump to start of program.

                clr.b  (a2)             ;clear, reply flag [match]
                bra.b  _compByteEd      ;let EP stop.
CMPNOTRDY_B1:
                move.b  #02h,(a2)       ;set reply flag,[02]: block 1
                bra.b   COMNONEQU_B     ; Not ready.

CMPNOTRDY_B2:
                move.b  #03h,(a2)       ;set reply flag,[03]: block 2
                                        ; Not ready.
COMNONEQU_B:
;....................................................................
;               Save NotReady/NotMatch address & data
;....................................................................
                subq.l  #1,a0           ;adjust block 1 address.
                subq.l  #1,a1           ;adjust block 2 address.
                addq.l  #1,a2           ;A2 point to replay buffer
                move.l  a0,(a2)+        ;save block 1 address.
                move.b  #00,(a2)+       ;save block 1 data (dummy byte).
                move.b  d5,(a2)+        ;save block 1 data.
                move.l  a1,(a2)+        ;save Block 2 address
                move.b  #00,(a2)+       ;save Block 2 data (dummy byte).
                move.b  d6,(a2)         ;save Block 2 data.
                bra.b  _compByteEd      ;let EP stop.

                org     e6ah
_compByteEd:
                nop
                bra.b   _compByteSt     ;&&end
;-----------------------------------------------------------------------
;               Compare memory with size-Word
;               =============================
;  A0 -- block1 start address (source)
;  A1 -- block2 start address (destination)
;  D0 -- memory length.
;  A6 -- source trigger.
;  A5 -- destination trigger
;  .........................................
;  A2 -- reply flag pointer.
;........................................................................
                org     0f00h
_compWordSt:
                movea.l a7,a2
                adda.l  #canBuf2+1,a2   ; A2 point to reply flag.
CMPWord1:
                move.b  d0,(a6)         ;trigeer of source
                move    (a0)+,d5        ;read data from source area

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CMPNOTRDY_W1     ;Jump if not ready.

                move.b  d0,(a5)         ;trigger of destination
                move    (a1)+,d6        ;read data from destination (Block 2)

                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CMPNOTRDY_W2     ;jump if not ready.

                cmp     d6,d5           ;compare block-1 & block-2 data.
                bne.b   COMNONEQU_W

                subq.l  #1,d0           ;descrease loop counter, D0
                cmp.l   #-1,d0          ;check,loop End ?
                bne.b   CMPWord1        ;No,jump to start of program.

                clr.b  (a2)             ;clear, reply flag [match]
                bra.b  _compWordEd      ;let EP stop.
CMPNOTRDY_W1:
                move.b  #02h,(a2)       ;set reply flag,[02]: block 1
                bra.b   COMNONEQU_W     ; Not ready.

CMPNOTRDY_W2:
                move.b  #03h,(a2)       ;set reply flag,[03]: block 2
                                        ; Not ready.
COMNONEQU_W:
;....................................................................
;               Save NotReady/NotMatch address & data
;....................................................................
                subq.l  #2,a0           ;adjust block 1 address.
                subq.l  #2,a1           ;adjust block 2 address.
                addq.l  #1,a2           ;A2 point to replay buffer
                move.l  a0,(a2)+        ;save block 1 address.
                move    d5,(a2)+        ;save block 1 data.
                move.l  a1,(a2)+        ;save Block 2 address
                move    d6,(a2)         ;save Block 2 data.
                bra.b  _compWordEd      ;let EP stop.

                org    0f6ah
_compWordEd:
                nop
                bra.b   _compWordSt     ;&&end
;-----------------------------------------------------------------------
;   Adjust Pc to Can area with can base address.
;   note: A7 -- is can base address .
;........................................................................
                org     1000h
_pcToBaseSt:
                jmp     4(a7)           ;jump to base address
                nop
                nop
                bra.b   _pcToBaseEd     ;let EP stop

                org     106ah
_pcToBaseEd:
                nop
                bra.b   _pcToBaseSt     ; &&end

;               move.l  a7,(a7)         ;set  the location what PC will
;
;   Tsai, 1/2/1993, Can't use the "addi" instruction because
;                   it may be change the SR register value.
;               addi.l  #04h,(a7)
;               move.b  #04h,3(a7)      ; jump address.
;               jmp     (a7)            ;

;-----------------------------------------------------------------------
;       check Sum with size byte
;       ========================
;  A0 -- start address (source)
;  D0 -- memory length.
;  A6 -- source trigger.
;......................................................................
;  D6 -- check Sum (word), U32 but the high word is dummy (0000h).
;  D5 -- working register
;.......................................................................
                org     1100h
_chkSByteSt:
                clr.l   d4              ;set D4 to zero,for "addx" instruction.
                clr.l   d5              ;clear data buffer.
                clr.l   d6              ;clear check
CHKSByte1:
                move.b  d0,(a6)         ;trigger of source function code.
                move.b  (a0)+,d5        ;read one byte data from tar/emm.
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CHKSBNotRDy     ;jump if not ready.

                add     d5,d6           ;add this byte to check sum register.
                addx    d4,d6           ;add the extended bit.
                subq.l  #01h,d0
                cmp.l   #-1,d0          ;check,loop end ?
                bne.b   CHKSByte1

                move.l  d6,canBuf2+2(a7) ;save check Sum to can buffer 2.
                bra.b   _chkSByteEd
CHKSBNotRDy:
                subq.l  #1,a0
                move.l  a0,canBuf2+2(a7) ;save NotReady address .
                bra.b   _chkSByteEd

                org     116ah
_chkSByteEd:
                nop
                bra.b   _chkSByteSt     ;&&end
;-----------------------------------------------------------------------
;       check Sum with word byte
;       ========================
;  A0 -- start address (source)
;  D0 -- memory length.
;  A6 -- source trigger.
;......................................................................
;  D6 -- check Sum (word), U32 but the high word is dummy (0000h).
;  D5 -- working register
;.......................................................................
                org     1200h
_chkSWordSt:
                clr.l   d4              ;set D4 to zero,for "addx" instruction.
                clr.l   d6              ;clear check
CHKSWord1:
                move.b  d0,(a6)         ;trigger of source function code.
                move    (a0)+,d5        ;read one byte data from tar/emm.
                btst.b  #00h,notReadyFlg(a7) ;check Ep Not Ready ?
                bne.b   CHKSWNotRDy     ;jump if not ready.

                add     d5,d6           ;add this byte to check sum register.
                addx    d4,d6           ;add the extended bit.
                subq.l  #01h,d0
                cmp.l   #-1,d0          ;check,loop end ?
                bne.b   CHKSWord1

                move.l  d6,canBuf2+2(a7) ;save check Sum to can buffer 2.
                bra.b   _chkSWordEd
CHKSWNotRDy:
                subq.l  #2,a0
                move.l  a0,canBuf2+2(a7) ;save NotReady address .
                bra.b   _chkSWordEd

                org     126ah
_chkSWordEd:
                nop
                bra.b   _chkSWordSt     ;&&end
;-----------------------------------------------------------------------
;        PreCopyMem
;       ========================
;  A0 -- start address (source)
;  D0 -- memory length.
;  A1 -- destination address
;  A6 -- source trigger.
;  A5 -- destination trigger.
;  D1 -- Control word (bit7 -- verify flag, Off[0]/On[1]
;.......................................................................
;  A2 -- address of ready flag.
;  D6 -- working register
;  D5 -- working register
;.......................................................................
                org     1300h
_preCopySt:
                movea.l a7,a2
                adda.l  #notReadyFlg,a2
PRECop1:
                move.b  d0,(a6)         ; source trigger
                move.b  (a0),d6         ; read data from source
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   PRECNotRdy1     ;jump if not ready.

                move.b  d0,(a5)         ; destination trigger
                move.b  d6,(a1)         ; write data to destination
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   PRECNotRdy2     ;jump if not ready.

                btst    #07h,d1         ;verify flag on ?
                beq.b   PRECop2         ;  off[0],on[1]

                move.b  d0,(a5)         ; destination trigger
                move.b  (a1),d5         ; read Destination data
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   PRECNotRdy2     ;jump if not ready.

                cmp.b   d6,d5           ;verify data
                bne.b   PRECVriErr
PRECop2:
                addq.l  #1,a0           ;next source address
                addq.l  #1,a1           ;next destination address
                subq.l  #01h,d0         ;decrease loop counter
                cmp.l   #-1,d0          ;check, Loop End ?
                bne.b   PRECop1         ;No,jump.

                clr.b   canBuf2+1(a7)   ;clear echo flag. (OK)
                bra.b   _preCopyEd
PRECVriErr:
                move.l  a1,canBuf2+2(a7) ;save verify error address.
                bra.b   _preCopyEd

PRECNotRdy1:
                move.b  #02,canBuf2+1(a7) ; set echo flag, (source Not Ready)
                move.l  a0,canBuf2+2(a7)  ;save Not Ready address of source.
                bra.b   _preCopyEd
PRECNotRdy2:
                move.b  #03,canBuf2+1(a7) ; set echo flag,(destination
                move.l  a1,canBuf2+2(a7)  ; Not Ready), and save address.
                bra.b   _preCopyEd

                org     136ah
_preCopyEd:
                nop
                bra.b   _preCopySt      ; &&end
;-----------------------------------------------------------------------
;        PostCopyMem
;       ========================
;  A0 -- last address (source)
;  D0 -- memory length.
;  A1 -- destination last address
;  A6 -- source trigger.
;  A5 -- destination trigger.
;  D1 -- Control word (bit7 -- verify flag, Off[0]/On[1]
;.......................................................................
;  A2 -- address of ready flag.
;  D6 -- working register
;  D5 -- working register
;.......................................................................
                org     1400h
_posCopySt:
                movea.l a7,a2           ;compute a address of Ready Flag.
                adda.l  #notReadyFlg,a2
POSCop1:
                subq.l  #1,a0           ;Adjust source address.
                subq.l  #1,a1           ;Adjust destination address.
                move.b  d0,(a6)         ; source trigger
                move.b  (a0),d6         ; read data from source
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   POSCNotRdy1     ;jump if not ready.

                move.b  d0,(a5)         ; destination trigger
                move.b  d6,(a1)         ; write data to destination
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   POSCNotRdy2     ;jump if not ready.

                btst    #07h,d1         ;verify flag on ?
                beq.b   POSCop2         ;  off[0],on[1]

                move.b  d0,(a5)         ; destination trigger
                move.b  (a1),d5         ; read Destination data
                btst.b  #00h,(a2)       ;check, ready ?
                bne.b   POSCNotRdy2     ;jump if not ready.

                cmp.b   d6,d5           ;verify data
                bne.b   POSCVriErr
POSCop2:
                subq.l  #01h,d0         ;decrease loop counter
                cmp.l   #-1,d0          ;check, Loop End ?
                bne.b   POSCop1         ;No,jump.

                clr.b   canBuf2+1(a7)   ;clear echo flag. (OK)
                bra.b   _posCopyEd
POSCVriErr:
                move.l  a1,canBuf2+2(a7) ;save verify error address.
                bra.b   _posCopyEd

POSCNotRdy1:
                move.b  #02,canBuf2+1(a7) ; set echo flag, (source Not Ready)
                move.l  a0,canBuf2+2(a7)  ;save Not Ready address of source.
                bra.b   _posCopyEd
POSCNotRdy2:
                move.b  #03,canBuf2+1(a7) ; set echo flag,(destination
                move.l  a1,canBuf2+2(a7)  ; Not Ready), and save address.
                bra.b   _posCopyEd

                org     146ah
_posCopyEd:
                nop
                bra.b   _posCopySt      ; &&end
;-----------------------------------------------------------------------
;        Read and Set the CSFC5 to high (enable) for
;        Chip Select Configuration Register Low word
;       ================================================
;Register    |  location
;------------+------------------
;   CSRL0    : ffffffc2h (word)
;   CSRL1    : ffffffc6h   "
;   CSRL2    : ffffffcah   "
;   CSRL3    : ffffffceh   "
;   CSRL4    : ffffffd2h   "
;   CSRL5    : ffffffd6h   "
;   CSRL6    : ffffffdah   "
;   CSRL7    : ffffffdeh   "
; ............................
;   DR0      : ffffffe2h
;   DR1      : ffffffe6h
;
;
;   A7 : CAN base address.
;   A0 : Pointer of CS Configuration Register Low word
;   A6 : Source Trigger.
;   D0 : data buffer.
;   D1 : loop counter. (low word)
;   D2 : FC5 buffer of chip select register
;        bit15  bit14  bit13  bit12  bit11  bit10  bit9   bit 8
;         1      1      1      1      1      1     DRAM1  DRAM0
;
;        bit7   bit6   bit5   bit4   bit3   bit2   bit1  bit 0
;        CS7    CS6    CS5    CS4    CS3    CS2    CS1   CS0
;
;        return this data to offset 68h(high byte) & offset 69h(low byte).
;
;  Note :  After execution the can program , the can program will
;          restore all registers that are used. (keep original value
;          for all registers that are used.)
;----------------------------------------------------------------------------
;               Working Register/buffer  location.
;
;                       D1 buffer       <----   06eh
;                     bit[15..8],bit[7..0]
;   d1 register          (6eh)      (6fh)
;   (low word)
;                       A0 buffer       <----   070h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   A0 register             (70h)        (71h)       (72h)      (73h)
;                       A6 buffer       <----   074h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   A6 register             (74h)        (75h)       (76h)      (77h)
;                       D0 buffer       <----   078h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   D0 register             (78h)        (79h)       (7Ah)      (7Bh)
;                       D2 buffer       <----   07Ch
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   D2 register             (7Ch)        (7Dh)       (7Eh)      (7Fh)
;
;----------------------------------------------------------------------------
D1BUF:          equ     06eh
A0BUF:          equ     canBuf2
A6BUF:          equ     canBuf2+4
D0BUF:          equ     canBuf2+8
D2BUF:          equ     canBuf2+12
CSRLADD:        equ     0ffffffe6h
CSFC5BUF:       equ     canBuf1-2
                org     1500h
_enSdCsrlSt:
                move.l  a0,A0BUF(a7)    ;save registers of the working.
                move.l  a6,A6BUF(a7)
                move.l  d0,D0BUF(a7)
                move.l  d2,D2BUF(a7)
                move    d1,D1BUF(a7)    ;loop counter (low word)

                move.l  a7,a6           ;set source trigger.
                adda.l  #souTrigOs,a6
                clr     d2              ;initial Fc5 buffer.
                not     d2

                move.l  #CSRLADD,a0     ;set pointer of CSCR.
                move    #9h,d1          ;set loop counter (10 times)
enSdCscrLoop:
                move.b  d1,(a6)         ;trigger EMUSPACE to high with source
                                        ;trigger.
                move    (a0),d0         ;read CSCR low word.
                btst.l  #13,d0          ;check CSFC5 = high ?
                bne.b   enSdCsr1        ;Yes,jump.
                bclr.l  d1,d2           ;set the FC5 buffer to low.
;               ..... Enable SD space .... [Begin]
                bset.l  #13,d0          ;Set CSFC5 bit to high.
                move.b  d1,(a6)         ;trigger with source function code.
                move    d0,(a0)         ;write New value to internal register
;               ..... Enable SD space .... [End]
enSdCsr1:
                subq.l  #04h,a0         ;decrease pointer of CSCR.
                dbf     d1,enSdCscrLoop
                move    d2,CSFC5BUF(a7) ;save FC5 data to can buffer 1.

                move.l  A0BUF(a7),a0    ;restore registers of the working.
                move.l  A6BUF(a7),a6
                move.l  D0BUF(a7),d0
                move.l  D2BUF(a7),d2
                move    D1BUF(a7),d1
                bra.b   _enSdCsrlEd     ;Let EP Stop.

                org     156ah
_enSdCsrlEd:
                nop
                bra.b   _enSdCsrlSt     ; &&end

;-----------------------------------------------------------------------
;       Restore CSFC5 of Chip Select Configuration Register
;       ======================================================
;Register    |  location
;------------+------------------
;   CSRL0    : ffffffc2h (word)
;   CSRL1    : ffffffc6h   "
;   CSRL2    : ffffffcah   "
;   CSRL3    : ffffffceh   "
;   CSRL4    : ffffffd2h   "
;   CSRL5    : ffffffd6h   "
;   CSRL6    : ffffffdah   "
;   CSRL7    : ffffffdeh   "
; ............................
;   DR0      : ffffffe2h
;   DR1      : ffffffe6h
;
;
;   A7 : CAN base address.
;   A0 : Pointer of CS Configuration Register Low word
;   A6 : Source Trigger.
;   D0 : data buffer.
;   D1 : loop counter. (low word)
;   D2 : FC5 buffer of chip select register
;        bit15  bit14  bit13  bit12  bit11  bit10  bit9   bit 8
;         1      1      1      1      1      1     DRAM1  DRAM0
;
;        bit7   bit6   bit5   bit4   bit3   bit2   bit1  bit 0
;        CS7    CS6    CS5    CS4    CS3    CS2    CS1   CS0
;
;        return this data from offset 68h(high byte) & offset 69h(low byte).
;
;  Note :  After execution the can program , the can program will
;          restore all registers that are used. (keep original value
;          for all registers that are used.)
;----------------------------------------------------------------------------
;               Working Register/buffer  location.
;
;                       D1 buffer       <----   06eh
;                     bit[15..8],bit[7..0]
;   d1 register          (6eh)      (6fh)
;   (low word)
;                       A0 buffer       <----   070h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   A0 register             (70h)        (71h)       (72h)      (73h)
;                       A6 buffer       <----   074h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   A6 register             (74h)        (75h)       (76h)      (77h)
;                       D0 buffer       <----   078h
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   D0 register             (78h)        (79h)       (7Ah)      (7Bh)
;                       D2 buffer       <----   07Ch
;                          bit[32..24], bit[23..16],bit[15..8],bit[7..0]
;   D2 register             (7Ch)        (7Dh)       (7Eh)      (7Fh)
;
;----------------------------------------------------------------------------
                org     1600h
_rsCsrlSt:
                move.l  a0,A0BUF(a7)    ;save registers of the working.
                move.l  a6,A6BUF(a7)
                move.l  d0,D0BUF(a7)
                move.l  d2,D2BUF(a7)
                move    d1,D1BUF(a7)

                move.l  a7,a6           ;set source trigger.
                adda.l  #souTrigOs,a6
                move    CSFC5BUF(a7),d2 ;get the original FC5 data.

                move.l  #CSRLADD,a0     ;set pointer of CSCR.
                move    #9h,d1          ;set loop counter (10 times)
RsCscrLoop:
                btst.l  d1,d2
                bne.b   resStrCsr1
                move.b  d1,(a6)         ;source trigger
                move    (a0),d0         ;read internal CS register.
                bclr.l  #13,d0          ;set CSFC5 = low.

                move.b  d1,(a6)         ;trigger EMUSPACE to high with source
                                        ;trigger.
                move    d0,(a0)         ;write CSCR low word.
resStrCsr1:
                subq.l  #04h,a0         ;increase pointer of CSCR.
                dbf     d1,RsCscrLoop

                move.l  A0BUF(a7),a0    ;restore registers of the working.
                move.l  A6BUF(a7),a6
                move.l  D0BUF(a7),d0
                move.l  D2BUF(a7),d2
                move    D1BUF(a7),d1
                bra.b   _rsCsrlEd       ;Let EP Stop.

                org     166ah
_rsCsrlEd:
                nop
                bra.b   _rsCsrlSt       ; &&end


;-----------------------------------------------------------------------
; Get data information if FillByteMem command verify error
; Tsai,11/03/1994
;-----------------------------------------------------------------------
                org     1700h
_getByteInData:
                andi   #00ffh,d5        ;mask high byte,(expectant data)
                andi   #00ffh,d3        ;mask high byte,(error data)
                move   d5,canBuf2(a7)   ;save expectant data to canBuf2
                move   d3,canBuf2+2(a7) ;save error data to canBuf2+2
                bra.b  _getBIDEND

                org     176ah
_getBIDEND:
                nop
                bra.b   _getByteInData  ; &&end
;-----------------------------------------------------------------------
; Get data information if FillWordMem command verify error
; Tsai,11/03/1994.
;-----------------------------------------------------------------------
                org     1800h
_getWordInData:
                move   d5,canBuf2(a7)   ;save expectant data to canBuf2
                move   d3,canBuf2+2(a7) ;save error data to canBuf2+2
                bra.b  _getWIDEND

                org     186ah
_getWIDEND:
                nop
                bra.b   _getWordInData  ; &&end
;-----------------------------------------------------------------------
               .end
