/*****************************************************************************
**
**    Name:    trace.h
** 
**    Description: 
**       Routines to maintain information on available traces
**
**    Status: PRELIMINARY
**
**    $Log:   S:/tbird/arcm306/dad32/trace.h_v  $
** 
**    Rev 1.0   07 Sep 1995 10:17:00   gene
** Initial revision.
**  
**     Rev 1.27   08 Nov 1993 20:44:08   mindy
**  TrcMaxValidTraceBuffers call added
**  
**     Rev 1.26   22 Jul 1993 15:37:54   mindy
**  added timestamp stuff
**  
**     Rev 1.25   12 Mar 1993 09:46:42   mindy
**  added cplusplus ifdef support
**  
**     Rev 1.24   16 Dec 1992 15:36:26   mindy
**  moved dq to fw
**  
**     Rev 1.23   02 Dec 1992 13:07:32   mindy
**  change trace read routine's interfaces
**  
**     Rev 1.22   21 Oct 1992 06:25:06   mindy
**  changed U8 FAR * to LPSTR to remove warnings
**  
**     Rev 1.21   08 Sep 1992 09:44:40   john
**  removed EVENT_ definitions
**  
**     Rev 1.20   27 Aug 1992 06:34:32   mindy
**  a) changed tracing event to be two events one when tracing goes on and
**     one when tracing goes off.
**  b) change RAW_INFO frame type to S32 for dequeuer.
**  c) removed temp. prototypes for go and halt.
**  
**     Rev 1.19   19 Aug 1992 07:51:44   mindy
**  a) removed Force status update routine - now fw will only update member
**     that have changed.  And when tracing changed the trace server is
**     registered to do a force status update automatically.
**  b) Added two events: one happens when tracing changes and the
**     other happens when the sub buffer number changes.
**  
**     Rev 1.18   11 Aug 1992 10:05:50   mindy
**  open no longer requires processor type
**  
**     Rev 1.17   25 Jun 1992 12:45:28   mindy
**  removed swat type routines
**  
**     Rev 1.16   22 Jun 1992 07:56:06   mindy
**  added get sub buffer size routine
**  
**     Rev 1.15   28 Apr 1992 08:10:22   mindy
**  removed switch buffer routine and added getcliid routine
**  
**     Rev 1.14   03 Apr 1992 11:56:04   mindy
**  added cursor link routine
**  
**     Rev 1.13   01 Apr 1992 06:42:08   mindy
**  a) added typedef for RAW_... - this is the structure type returned
**     be read raw trace however the structure type isn't referenced
**     explicitly.
**  b) added TrcNumStatusToggles to count number of instruction boundaries
**     occur between the starting (current - frame number) refill and the
**     next refill.
**  
**     Rev 1.12   13 Mar 1992 08:15:42   mindy
**  changed interface to read raw trace frames
**  
**     Rev 1.11   18 Feb 1992 14:17:14   mindy
**  now have a backward and forward read trace routine
**  
**     Rev 1.10   14 Jan 1992 11:41:44   mindy
**  temp. have go and halt in here
**  
**     Rev 1.9   26 Nov 1991 11:38:02   mindy
**  a) added timestamp units routines
**  b) added setting of tracing.
**  
**     Rev 1.8   20 Nov 1991 11:14:08   mindy
**  added timestamp routines
**  
**     Rev 1.7   15 Nov 1991 15:02:52   mindy
**  removed routines not being supported
**  
**     Rev 1.6   07 Nov 1991 14:30:24   mindy
**  Added get header routine.
**  Also removed all the compare trace prototypes since we are not supporting
**  for first release and they will probably change when they are implemented.
**  
**     Rev 1.5   01 Nov 1991 09:19:14   mindy
**  added stuff needed to support final version of trace presenter
**  
**     Rev 1.4   06 Sep 1991 09:14:16   mindy
**  share processor type definition in dasm.h and add frame display mode back in
**  
**     Rev 1.3   31 Jul 1991 14:43:22   john
**  Updated to version used in basic system release.
**
**     Rev 1.2   30 Jul 1991 13:47:58   nghia
**  added changes for basic system release
 * 
 *    Rev 1.0   14 Jun 1991 13:34:52   jim
 * Initial revision.
**
**    $Header:   S:/tbird/arcm306/dad32/trace.h_v   1.0   07 Sep 1995 10:17:00   gene  $
**
*****************************************************************************/

#define _TRACE_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASETYPE_
#include "basetype.h"
#endif

#ifndef _DASM_
#include "dasm.h"
#endif

#ifndef _EVTTMPLT_
#include "evttmplt.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _SDPROBE_
#include "sdprobe.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define  NUM_HDR_LINES  3
#define  MAX_HDR_LINE   128

typedef struct {
   S32 frameNum;
   XILINX_GROUP frameInfo;
} RAW_FRAME_INFO;

typedef enum {START_ALIGN, TRIGGER_ALIGN, NO_ALIGN, FDM_DUMMY=0x7fff 
} FRAME_DISPLAY_MODE;

typedef enum { RELATIVETOFRAME, DELTA, TSM_DUMMY=0x7fff
} TIMESTAMP_MODE;

#ifdef __cplusplus
extern "C" {
#endif
                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

/*****************************************************************************
** 
**    TrcInitTraceServer
** 
**    Description:
**       Does initialization for trace server.
** 
**    Parameters:
**       input:
**          none
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcInitTraceServer(VOID);
      
/*****************************************************************************
** 
**    TrcInitTraceServerCli
** 
**    Description:
**       Registers the trace server's routines with the CLI server.
** 
**    Parameters:
**       input:
**          cliHandle:  registration target for servers.
**          dllHandle:  handle to use in messages to tag this DLL.
**
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcInitTraceServerCli(HANDLE cliHandle, HANDLE dllHandle);
      
/*****************************************************************************
** 
**    TrcFrameDisplayModeGet
** 
**    Description:
**       Returns current frame numbering mode used within each trace buffer.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          mode:          how to number frames within a trace buffer.
**
*****************************************************************************/
RETCODE EXPORT TrcFrameDisplayModeGet(DESCRIPTOR id, 
                                      FRAME_DISPLAY_MODE FAR *mode);
      
/*****************************************************************************
** 
**    TrcFrameDisplayModeSet
** 
**    Description:
**       This routine basically sets up the frame numbering scheme using
**       within each trace buffer.  START_ALIGN means the first frame in each
**       trace buffer is numbered frame zero(0) and each following frame is
**       sequentially numbered.  TRIGGER_ALIGN means the trigger frame is 
**       numbered frame zero(0) and each frame found before the trigger 
**       has a negative number and each frame following the trigger is a
**       positive number. And finally NO_ALIGN means the frame numbers
**       correspond directly to the hardware frame number no adjustment 
**       is made this is an internal selection (to be used for diagnostics
**       purposes).
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          mode:          how to number frames within a trace buffer.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcFrameDisplayModeSet(DESCRIPTOR id, FRAME_DISPLAY_MODE mode);
      
/*****************************************************************************
** 
**    TrcFrameNumberAreaSizeGet
** 
**    Description:
**       Returns the number of characters in the frame number portion of
**       each trace line.
** 
**    Parameters:
**       input:
**          none
**       output:
**          size:          number of positions in the frame number area.
**
*****************************************************************************/
RETCODE EXPORT TrcFrameNumberAreaSizeGet(U16 FAR *size);
      
/*****************************************************************************
** 
**    TrcSubBufferSizeGet
** 
**    Description:
**       Returns the sub buffer size.  The trace store is made up of 1024
**       sub buffers.  To determine the trace buffer size: 
**                (1024/# of buffers) * sub buffer size
** 
**    Parameters:
**       input:
**          none
**       output:
**          size:          sub buffer size
**
*****************************************************************************/
RETCODE EXPORT TrcSubBufferSizeGet(U16 FAR *size);
      
/*****************************************************************************
** 
**    TrcGetHeader
** 
**    Description:
**       Returns a text string that contains three lines of text holding
**       the header information for the bus and clock cycles.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          maxLen:        max length of storage allocated for the header
**       output:
**          header:        storage where to copy header
**
*****************************************************************************/
RETCODE EXPORT TrcGetHeader(DESCRIPTOR id, LPSTR header, U16 maxLen);
      
/*****************************************************************************
** 
**    TrcReadBufferGet
** 
**    Description:
**       Returns the current value of the read trace buffer found in the state
**       table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          buffer:        current starting trace buffer
**
*****************************************************************************/
RETCODE EXPORT TrcReadBufferGet(DESCRIPTOR id, U16 FAR *buffer);
                                         
/*****************************************************************************
** 
**    TrcReadBufferSet
** 
**    Description:
**       Sets the read trace buffer to 'buffer' in the state table for the
**       given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          buffer:        new starting trace buffer
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcReadBufferSet(DESCRIPTOR id, U16 buffer);
      
/*****************************************************************************
** 
**    TrcReadFrameGet
** 
**    Description:
**       Returns the current value of the read trace buffer frame found in the
**       state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frame:         current starting trace buffer frame
**
*****************************************************************************/
RETCODE EXPORT TrcReadFrameGet(DESCRIPTOR id, S32 FAR *frame);
      
/*****************************************************************************
** 
**    TrcReadFrameSet
** 
**    Description:
**       Sets the read trace buffer frame to 'frame' in the state table 
**       for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frame:         new starting trace buffer frame
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcReadFrameSet(DESCRIPTOR id, S32 frame);
      
/*****************************************************************************
** 
**    TrcNumFramesGet
** 
**    Description:
**       Returns the number of trace frames in the active trace buffer. This
**       will return the number of trace collected not the physical number
**       of frames per buffer.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frames:        number of frames
**
*****************************************************************************/
RETCODE EXPORT TrcNumFramesGet(DESCRIPTOR id, U32 FAR *frames);
      
/*****************************************************************************
** 
**    TrcSearchBufferGet
** 
**    Description:
**       Returns the current value of the search trace buffer found in 
**       the state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          buffer:        current starting trace buffer
**
*****************************************************************************/
RETCODE EXPORT TrcSearchBufferGet(DESCRIPTOR id, U16 FAR *buffer);
      
/*****************************************************************************
** 
**    TrcSearchBufferSet
** 
**    Description:
**       Sets the search trace buffer to 'buffer' in the state table for the
**       given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          buffer:        new starting trace buffer
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcSearchBufferSet(DESCRIPTOR id, U16 buffer);
      
/*****************************************************************************
** 
**    TrcSearchEventGet
** 
**    Description:
**       Returns the event name of event conditions being used in search
**       commands.
** 
**    Parameters:
**       input:
**          id:            id number assigned to calling routine
**          maxLen:        size of string storage
**       output:
**          eventId:       descriptor of event being used in search
**
*****************************************************************************/
RETCODE EXPORT TrcSearchEventGet(DESCRIPTOR id, LPSTR event, U16 maxLen);
      
/*****************************************************************************
** 
**    TrcSearchEventSet
** 
**    Description:
**       Sets up the conditions for the search.  Thus any sequential search
**       calls made will use these event conditions to perform the search.
** 
**    Parameters:
**       input:
**          id:            id number assigned to calling routine
**          event:         name of event to use in search
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcSearchEventSet(DESCRIPTOR id, LPSTR event);
      
/*****************************************************************************
** 
**    TrcSearchFrameGet
** 
**    Description:
**       Returns the current value of the search trace buffer frame found 
**       in the state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frame:         current starting trace buffer frame
**
*****************************************************************************/
RETCODE EXPORT TrcSearchFrameGet(DESCRIPTOR id, S32 FAR *frame);
      
/*****************************************************************************
** 
**    TrcSearchFrameSet
** 
**    Description:
**       Sets the search trace buffer frame to 'frame' in the state table 
**       for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frame:         new starting trace buffer frame
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcSearchFrameSet(DESCRIPTOR id, S32 frame);
      
/*****************************************************************************
** 
**    TrcTraceBufferClear
** 
**    Description:
**       Clears all trace buffers.  Now no trace frames having been
**       collected.  This function would only be performed if the id
**       indicated the caller was accessing the hardware trace buffer.
** 
**    Parameters:
**       input:
**          none
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTraceBufferClear(VOID);
      
/*****************************************************************************
** 
**    TrcTraceBufferInfoGet
** 
**    Description:
**       Returns the starting, ending and trigger frame number of the current
**       read trace buffer.  This is used by the presenter to display for 
**       the user.  The frame display mode affects the frame numbers.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          oldest:        frame number of first frame within trace buffer
**          youngest:      frame number of last frame within trace buffer
**          trigger:       frame number of trigger frame within trace buffer
**
*****************************************************************************/
RETCODE EXPORT TrcTraceBufferInfoGet(DESCRIPTOR id, S32 FAR *oldest,
                                     S32 FAR *youngest, S32 FAR *trigger);
      
/*****************************************************************************
** 
**    TrcTraceClose
** 
**    Description:
**       Removes indicated descriptor's state information.
** 
**    Parameters:
**       input:
**          id:            id number of descriptor to close
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTraceClose(DESCRIPTOR id);

/*****************************************************************************
** 
**    TrcTraceCycleGet
** 
**    Description:
**       Returns the type of filtering currently active.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          type:          type of trace cycles to return.
**
*****************************************************************************/
RETCODE EXPORT TrcTraceCycleGet(DESCRIPTOR id, TRACE_CYCLES FAR *type);
      
/*****************************************************************************
** 
**    TrcTraceCycleSet
** 
**    Description:
**       This routine basically sets up a filtering of cycles of the user
**       specifies bus cycles.  The clock cycles would be combined to form
**       a entire bus cycle if BUS_CYCLE mode is selected.  Otherwise the
**       raw clock cycles are returned (i.e.: with no filtering).  The last
**       type BINARY_CYCLE is more of an internal option that is used by the
**       save and possibly the compare commands to know what form the trace 
**       frames are in.  BUS_CYCLE and CLOCK_CYCLE are returned in ASCII text
**       while BINARY_CYCLE returns the actual binary bits.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          type:          type of trace cycles to return.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTraceCycleSet(DESCRIPTOR id, TRACE_CYCLES type);
      
/*****************************************************************************
** 
**    TrcTraceOpen
** 
**    Description:
**       Opens the incoming file (either host file or hardware acquisition 
**       memory) as the active trace source.  Hardware acquisition memory 
**       is referred to by using filename "hwtrc".  traceBuffer and frame
**       are initially set to zero for the next trace command.  cycleMode
**       tells which type of cycles are returned when reading trace cycles.
**       frameMode determines the frame numbering scheme.  The default
**       search event and filter event are initialized to nil. The template
**       in use is the default template for the given processor type.
**       Other variables initialed automatically are the compare and search
**       trace buffer and frame number, the file the opened trace file
**       is compared to and the starting trace buffer and frame numbers.
**       Returned is a unique id number for calling routine to use when 
**       invoking any of the following trace routines.
** 
**    Parameters:
**       input:
**          fileName:      File name of file to open
**          cycleMode:     trace cycles to return (i.e: bus, clock)
**          frameMode:     numbering scheme for trace frames
**       output:
**          id:            id number assigned to calling routine.
**
*****************************************************************************/
RETCODE EXPORT TrcTraceOpen(DESCRIPTOR FAR *id, LPSTR fileName,
                            TRACE_CYCLES cycleMode, 
                            FRAME_DISPLAY_MODE frameMode);
      
/*****************************************************************************
** 
**    TrcTraceReadRaw
** 
**    Description:
**       Reads numFrames frames of raw qualified bus cycles from the trace 
**    file.  numFrames can be negative which means caller wants previous
**    frames.  For example numFrames=-2 will return the two frames prior
**    to the starting trace frame.  The frame data sent back is in the
**    following format and array of:
**          S32 frameNumber;
**          XILINX_GROUP frame;
** 
**    Parameters:
**       input:
**          id:            id number assigned to calling routine
**          numFrames:     number of trace frames to read
**       output:
**          numGotten:     number of frames read.
**          buffer:        buffer to write trace frames to (raw binary)
**                         Return pointer to TMALLOC'd memory.
**
*****************************************************************************/
RETCODE EXPORT TrcTraceReadRaw(DESCRIPTOR id, S32 numFrames,
                               U32 FAR *numGotten, LPSTR *buffer);
      
/*****************************************************************************
** 
**    TrcForwardRead
** 
**    Description:
**       Reads numFrames frames from the trace file starting in the
**       traceBuffer buffer at the frame number.  The number 
**       of frames requested is returned unless the end of trace buffer
**       is encountered first.  If we are not able to return any frames
**       an "end of file" error is returned.  This routine also returns
**       the actual starting and ending frame number.    
** 
**    Parameters:
**       input:
**          id:            id number assigned to calling routine
**          numFrames:     number of trace frames to read
**       output:
**          startFrame:    numeric value of first frame in returned text
**          endFrame:      numeric value of last frame in returned text
**          buffer:        buffer to write trace frames to (ASCII text)
**                         Return pointer to TMALLOC'd memory of text.
**
*****************************************************************************/
RETCODE EXPORT TrcForwardRead(DESCRIPTOR id, U32 numFrames,
  S32 FAR *startFrame, S32 FAR *endFrame, LPSTR *buffer, U16 FAR *textLen);
      
/*****************************************************************************
** 
**    TrcBackwardRead
** 
**    Description:
**       Reads 'numFrames' frames from the trace buffer prior to the
**       starting frame number.  The number of frames requested is returned
**       unless the beginning of trace buffer is encountered first.
**       If we are not able to return any frames an "end of file" error is 
**       returned.  This routine also returns the actual starting and ending
**       frame number.   
** 
**    Parameters:
**       input:
**          id:            id number assigned to calling routine
**          numFrames:     number of trace frames to read
**       output:
**          startFrame:    numeric value of first frame in returned text
**          endFrame:      numeric value of last frame in returned text
**          buffer:        buffer to write trace frames to (ASCII text)
**                         Return pointer to TMALLOC'd memory of text.
**
*****************************************************************************/
RETCODE EXPORT TrcBackwardRead(DESCRIPTOR id, U32 numFrames,
   S32 FAR *startFrame, S32 FAR *endFrame, LPSTR *buffer, U16 FAR *textLen);
      
/*****************************************************************************
**
**    TrcTraceSearch
** 
**    Description:
**       Searches the trace file for the conditions found in the event
**       currently active in the trace buffer starting at frame found in
**       state table and searching until the end of the trace buffer.
**       'found' indicates if a match is found.  If 'found' is than
**       user calls TrcSearchFrameGet to know which frame matched the
**       conditions.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          found:         TRUE if a match is found
**
*****************************************************************************/
RETCODE EXPORT TrcTraceSearch(DESCRIPTOR id, BOOLEAN FAR *found);
      
/*****************************************************************************
** 
**    TrcTraceTemplateGet
** 
**    Description:
**       Returns the template name of the currently active template.  The 
**       calling routine would then call the generic template routines 
**       using the template id to get the specific template information. 
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          maxLen:        length of storage
**       output:
**          name:          name of active trace template 
**
*****************************************************************************/
RETCODE EXPORT TrcTraceTemplateGet(DESCRIPTOR id, LPSTR name, U16 maxLen);
      
/*****************************************************************************
** 
**    TrcTraceTemplateSet
** 
**    Description:
**       Changes the active template that trace uses to display bus and
**       clock cycles.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          name:          template name
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTraceTemplateSet(DESCRIPTOR id, LPSTR name);
      
/*****************************************************************************
** 
**    TrcTracingGet
** 
**    Description:
**       Returns current trace management hardware tracing status.
** 
**    Parameters:
**       input:
**          none
**       output:
**          value:         Value of tracing 
**
*****************************************************************************/
RETCODE EXPORT TrcTracingGet(BOOLEAN FAR *value);

/*****************************************************************************
** 
**    TrcTracingSet
** 
**    Description:
**       Changes the current trace management hardware tracing status.
** 
**    Parameters:
**       input:
**          value:         Value of tracing 
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTracingSet(BOOLEAN value);

/*****************************************************************************
** 
**    TrcMaxValidTraceBuffesr
** 
**    Description:
**       Returns the number of valid trace buffers.  TrigTraceConfigureGet
**    returns the number of trace buffer the trace store is divided into.
**    This routine just returns the number of trace buffers that contain
**    valid trace data.
** 
**    Parameters:
**       input:
**          none
**       output:
**          numBuffers:         number of valid/active trace buffers
**
*****************************************************************************/
RETCODE EXPORT TrcMaxValidTraceBuffers(U16 FAR *numBuffers);

/*****************************************************************************
** 
**    TrcDebugEmonGet
** 
**    Description:
**       Returns the current status of the debug emon flag.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          value:         Value of debug emon
**
*****************************************************************************/
RETCODE EXPORT TrcDebugEmonGet(DESCRIPTOR id, BOOLEAN FAR *value);

/*****************************************************************************
** 
**    TrcDebugEmonSet
** 
**    Description:
**       Changes the current status of the debug emon flag.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          value:         Value of debug emon flag
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcDebugEmonSet(DESCRIPTOR id, BOOLEAN value);

/*****************************************************************************
** 
**    TrcUpdateTrace
** 
**    Description:
**       Sets up the incoming descriptor to point to the last collected
**    trace buffer.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcUpdateTrace(DESCRIPTOR id);

/*****************************************************************************
** 
**    TrcCollectingBufferGet
** 
**    Description:
**       Returns the trace buffer that is currently being filled by the
**       trace hardware.
** 
**    Parameters:
**       input:
**          none
**       output:
**          value:         Value of tracing 
**
*****************************************************************************/
RETCODE EXPORT TrcCollectingBufferGet(U16 FAR *traceBuffer);

/*****************************************************************************
** 
**    TrcGetCliId
** 
**    Description:
**       Returns the descriptor id from the trace cli commands.  If trace
**    hasn't been opened yet this call will "open" a trace.
** 
**    Parameters:
**       input:
**          pType:         Which processor to setup trace for.  Used only
**                         if trace isn't currently opened.
**       output:
**          trcCliId:      trace command id
**
*****************************************************************************/
RETCODE EXPORT TrcGetCliId(DESCRIPTOR FAR *trcCliId);

/*****************************************************************************
** 
**    TrcGetFrameInstAddress
** 
**    Description:
**       This routine finds the address associated with the incoming frame
**    number. We also return a BOOLEAN which tells the caller if the given
**    frame number is a code read (i.e.: instruction address).  Although
**    the descriptor is passed in - this routine only uses the frame 
**    alignment, trace buffer, etc information.  All state is preserved.
**    Since several frames make up one bus cycle and we need bus cycles
**    to get a valid address - several consecutive frames will yield the
**    same results.  If the incoming frame is not an instruction type frame
**    then we will fill in the address descriptor (we provide the memory
**    for the address descriptor) with the last seen code address.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frame:         frame to find address for.
**       output:
**          addr:          addr desc created by routine; filled with address
**                         of frame or last previous code read address.
**          frameIsInst:   TRUE if address returned is from the given frame.
**
*****************************************************************************/
RETCODE EXPORT TrcGetFrameInstAddress(DESCRIPTOR id, S32 frame, 
                              DESCRIPTOR FAR *addr, BOOLEAN FAR *frameIsInst);

RETCODE EXPORT TrcNumStatusToggles(DESCRIPTOR id, S32 frame, 
                                   U16 FAR *instCount);
/* !!! very processor specific routine - will need to put in the correct
   place if this solves the dequeuer's problems.
 */      

/*****************************************************************************
** 
**    TrcTimestampGet
** 
**    Description:
**       Returns TRUE if timestamp is enabled.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          value:         Value of timestampOn flag
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampGet(DESCRIPTOR id, BOOLEAN FAR *value);

/*****************************************************************************
** 
**    TrcTimestampSet
** 
**    Description:
**       Changes the status of the timestampOn flag.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          value:         Value of timestampOn flag
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampSet(DESCRIPTOR id, BOOLEAN value);

/*****************************************************************************
** 
**    TrcTimestampModeGet
** 
**    Description:
**       Returns RELATIVETOFRAME if absolute timestamp mode is enabled.
**    Timestamp is either printed as an absolute timestamp with a specific
**    frame number as the starting point or DELTA mode when the timestamp
**    field contains the delta time between the current frame and the
**    previous frame.
**
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          value:         Value of timestamp mode
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampModeGet(DESCRIPTOR id, TIMESTAMP_MODE FAR *value);

/*****************************************************************************
** 
**    TrcTimestampModeSet
** 
**    Description:
**       Changes the status of the timestamp mode. See
**    TrcTimestampModeGet for more Info.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          value:         Value of timestamp mode
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampModeSet(DESCRIPTOR id, TIMESTAMP_MODE value);

/*****************************************************************************
** 
**    TrcTimestampZeroFrameGet
** 
**    Description:
**       Returns the frame number of the frame currently used as the zero
**    reference point for the timestamp calculation.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frame:         zero frame for timestamp
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampZeroFrameGet(DESCRIPTOR id, S32 FAR *frame);
      
/*****************************************************************************
** 
**    TrcTimestampZeroFrameSet
** 
**    Description:
**       Sets the frame number to use as the zero reference point for the
**    timestamp calculation.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frame:         new starting trace buffer frame
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcTimestampZeroFrameSet(DESCRIPTOR id, S32 frame);
      
/*****************************************************************************
** 
**    TrcSystemCrystalGet
** 
**    Description:
**       Returns the current value trace has for the system crystal frequency.
**    The trace server uses this value with the current multilier found in
**    the Clock Synthesizer Control Register $YFFA04 to determine the
**    timestamp field.
**    NOTE: Am using a string to interface with Actor.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          rate:          system crystal frequency
**
*****************************************************************************/
RETCODE EXPORT TrcSystemCrystalGet(LPSTR freqText);
      
/*****************************************************************************
** 
**    TrcSystemCrystalSet
** 
**    Description:
**       Sets the trace server's system crystal frequency value. See
**    TrcSystemCrystalGet.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          rate:          value for system crystal.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcSystemCrystalSet(LPSTR freqText);

/*****************************************************************************
** 
**    TrcGetTimestampText
** 
**    Description:
**       Gets the text for the timestamp for the current frame. !!!
**    eventually we will need to pass the previous frame number to
**    support the relative timestamp mode.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frameNumber:   current frame we need timestamp text for
**          prevFrameNum:  frame number of previous frame used for relative TS
**          buffer:        caller allocate pointer for text
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TrcGetTimestampText(DESCRIPTOR id, S32 frameNumber,
                                   S32 prevFrameNum, LPSTR buffer);
      
#ifdef __cplusplus
}
#endif
/*********************************** E O F **********************************/
