/****************************************************************************
**
**  Name:  cpu.h
**
**  Description:
**     Routines to access CPU registers and control signals.
**
**  Status:  TESTED
**
**  $Log:   S:/tbird/arcm306/inc/cpu.h_v  $
** 
**    Rev 1.2   02 Jul 1996 15:54:00   kevin
** added CpuGetCoreType()
** 
**    Rev 1.1   01 Dec 1995 09:57:40   kevin
** joyce's modification
** 
**    Rev 1.0   07 Sep 1995 10:25:50   gene
** Initial revision.
** 
**    Rev 1.16   03 Mar 1994 15:48:44   john
** Increased number of chars available in a signal name.
** Added GetSignalName and GetNumberOfSignals function prototypes.
** 
**    Rev 1.15   15 Jun 1993 11:23:56   mindy
** Added two new routines. One routine registers register pairs. The other
** allows the loader to tell the CPU server which register to use as the
** frame pointer since in the HC16 world different compilers use different
** registers.
** 
**    Rev 1.14   30 Mar 1993 08:00:22   ernie
** 1. Changed interface to CpuFindRegisterName() to have caller
**    allocate memory.
** 2. Removed non-public information (such as error codes) from
**    this file and moved to local cpu server include file.
** 3. Added CpuGetRegisterInfo() entry point.
** 
**    Rev 1.13   09 Oct 1992 14:41:20   doug
** add command to reset only the cpu and not generate any events
** 
**    Rev 1.12   17 Sep 1992 11:02:36   doug
** added error message when certain registers set to odd values
** 
**    Rev 1.11   08 Sep 1992 08:15:32   doug
** Events are now centralized in events.h
** 
**    Rev 1.10   19 Aug 1992 07:16:40   doug
** added new events
** 
**    Rev 1.9   31 Jul 1992 13:45:02   doug
** init server is now done only by libmain to support on-demand loading
** 
**    Rev 1.8   21 Jul 1992 12:43:00   doug
** a) added error messages
** b) added event when frame pointer changes
** 
**    Rev 1.7   20 Jul 1992 15:49:26   doug
** added floating point not supported error message
** 
**    Rev 1.6   14 Jul 1992 12:29:38   courtney
** Added error id.
** 
**    Rev 1.5   13 May 1992 09:10:08   nghia
** Revised error codes and routines status.
** 
**    Rev 1.4   22 Apr 1992 09:56:40   nghia
** Added EVENT_CPU_REGISTERS_CHANGED_HALTED for CPU presenter.
** 
**    Rev 1.3   16 Apr 1992 18:00:48   nghia
** Fixed bug CpuFindRegisterName() - pointer to pointer parameter.
** 
**    Rev 1.2   16 Apr 1992 16:06:52   brucea
** Fixed: error in first #ifdef __cplusplus to make it
**    extern "C" {
** 
**    Rev 1.1   02 Apr 1992 16:56:10   nghia
** Added events for PC and SP register changed.
** 
**    Rev 1.0   27 Feb 1992 17:20:40   nghia
** Initial revision.
**
**  $Header:   S:/tbird/arcm306/inc/cpu.h_v   1.2   02 Jul 1996 15:54:00   kevin  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/
#define _CPU_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif
#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif


                       /****************************
                        *                          *
                        *       DEFINITIONS        *
                        *                          *
                        ****************************/

                       /**************************
                        *                        *
                        *      DEFINITIONS       *
                        *                        *
                        **************************/
typedef U16 REG_ID;

typedef U16 SIG_ID;

/*  Floating Point Registers Value -- 80bits wide */
typedef U8 U80[10];
typedef U80 FPREG_VAL;

#define MAX_REGNAME     10  /* max number of characters in register name */
#define MAX_SIGNAME     15  /* max number of characters in signal name */

/* define space attributes */
typedef enum { SUPERVISOR_MODE, USER_MODE
} SPACE_MODE;

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/
#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************************
**
**  TerminateServer
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Cleanup and terminate the CPU server current state. After calling
**     this routine, no other CPU Server routine should be called .
**
**  Parameters:
**     input:
**        none.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT TerminateServer(VOID);

/****************************************************************************
**
**  CpuGetRegister
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Retrieve the register content for the specified register id.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**  Parameters:
**     input:
**        regId      Register ID.
**
**     output:
**        regValue   Current value of the register.
**
*****************************************************************************/
RETCODE EXPORT CpuGetRegister(REG_ID regId, U32 FAR *regValue);

/****************************************************************************
**
**  CpuSetRegister
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Set a value for the specified register id.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**  Parameters:
**     input:
**        regId      Register ID
**        regValue   Value of Register
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuSetRegister(REG_ID regId, U32 regValue);

/****************************************************************************
**
**  CpuGetFpRegister
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Retrieve the contents of the specified floating point register id.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**  Parameters:
**     input:
**        regId      Floating point Register ID.
**
**     output:
**        regValue   Current value of the register.
**
*****************************************************************************/
RETCODE EXPORT CpuGetFpRegister(REG_ID regId, FPREG_VAL FAR *regValue);

/****************************************************************************
**
**  CpuSetFpRegister
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Set a value of the specified floating point register.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**  Parameters:
**     input:
**        regId      Register ID
**        regValue   Value of Register
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuSetFpRegister(REG_ID regId, FPREG_VAL regValue);

/****************************************************************************
**
**  CpuGetPC
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Get the content of the current PC register.
**     IMPORTANT NOTES:
**     The CPU Server creates a address descriptor to hold address information
**     of the current PC.  The Caller routine is responsible for deallocating
**     the descriptor by calling AdrDestroyAddress(), when done.
**
**  Parameters:
**     input:
**        none.
**     output:
**        pcDesc    Descriptor contains the current address value of the
**                  PC register.
**
*****************************************************************************/
RETCODE EXPORT CpuGetPC(DESCRIPTOR FAR *pcDesc);

/****************************************************************************
**
**  CpuSetPC
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Set the current PC register to the specified value.
**     Notify EVENT_CPU_PC_CHANGED event to Event Library.
**
**     IMPORTANT NOTES:
**     The CPU Server destroy the descriptor after it's done using it.
**     If Caller routine wants to have another copy of the descriptor, then
**     it needs to call AdrDuplicateAddress() to make another copy berfore
**     passing in the descriptor.
**
**  Parameters:
**     input:
**        pcValue    Descriptor contains address value of the PC to be set.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuSetPC(DESCRIPTOR pcDesc);

/****************************************************************************
**
**  CpuGetSP
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Get the content of the current SP register.
**
**     IMPORTANT NOTES:
**     The CPU Server creates a descriptor to hold address information of
**     the current PC.  The Caller routine is responsible for deallocating
**     the descriptor by calling AdrDestroyAddress(), when done.
**
**  Parameters:
**     input:
**        none.
**     output:
**        spDesc    Descriptor contains address of the current SP value.
**
*****************************************************************************/
RETCODE EXPORT CpuGetSP(DESCRIPTOR FAR *spDesc);

/****************************************************************************
**
**  CpuSetSP
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Set the current SP register to the specified address value of the
**     descriptor.
**     Notify EVENT_CPU_SP_CHANGED event.
**
**     IMPORTANT NOTES:
**     The CPU Server destroy the descriptor after it's done using it.
**     If Caller routine wants to have another copy of the descriptor, then
**     it needs to call AdrDuplicateAddress() to make another copy, berfore
**     passing in the descriptor.
**
**  Parameters:
**     input:
**        spDesc    Descriptor contains address value of the SP to be set.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuSetSP(DESCRIPTOR spDesc);

/****************************************************************************
**
**  CpuGetFramePtr
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Get the current Stack Frame pointer.
**
**     IMPORTANT NOTES:
**     The CPU Server creates a descriptor to hold address information of
**     the current FP.  The Caller routine is responsible for deallocating
**     the descriptor by calling AdrDestroyAddress(), when done.
**
**  Parameters:
**     input:
**        none.
**     output:
**        framePtrDesc   Descriptor contains address of the current SF pointer.
**
*****************************************************************************/
RETCODE EXPORT CpuGetFramePtr(DESCRIPTOR FAR *framePtrDesc);

/****************************************************************************
**
**  CpuResetCpu.
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Assert reset signal to the CPU. Reset all registers to reset values.
**     Post EVENT_CPU_REGISTERS_CHANGED event to Event Notification Server.
**
**  Parameters:
**     input:
**        none.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuResetCPU(VOID);

/****************************************************************************
**
**  CpuResetCpuOnly.
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Assert reset signal to the CPU only; do nothing else.  This is
**     useful if the events generated with CpuResetCpu cause the CPU to
**     hang.
**
**  Parameters:
**     input:
**        none.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuResetCPUOnly(VOID);

/****************************************************************************
**
**  CpuGetSignal
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Get the current state of the specified signal.
**
**  Parameters:
**     input:
**        signalId      Id of Control Signal.
**
**     output:
**        signalState   Current state of the control signal.
**
*****************************************************************************/
RETCODE EXPORT CpuGetSignal(SIG_ID signalId, U8 FAR *signalState);

/****************************************************************************
**
**  CpuSetSignal
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Set the specified signal with the input signal state.
**
**  Parameters:
**     input:
**        sigId         Id of control signal.
**        signalState   State to be set.
**     output:
**        none.
**
*****************************************************************************/
RETCODE EXPORT CpuSetSignal(SIG_ID signalId, U8 signalState);

/****************************************************************************
**
**  CpuGetRegisterInfo
**
**  Description:
**     Return information about the register.
**
**  Parameters:
**     input:
**        regId      Id of the register.
**     output:
**        regName    Pointer to buffer to receive name of the register.
**        sortIndex  Presenter displays registers in ascending sortIndex order
**        type       0 if numeric, non-0 if flag-type register
**        width      number of bits in this register
**
*****************************************************************************/
RETCODE EXPORT CpuGetRegisterInfo(REG_ID regId, LPSTR regName,
      U16 FAR *sortIndex, U16 FAR *type, U16 FAR *width);

/*****************************************************************************
**
**    CpuGetRegisterValueText
**
**  Description:
**     Return a string representing the value of the register.
**
**  Parameters:
**     input:
**        regId      Id of the register.
**        type       0 if numeric, non-0 if flag-type display
**     output:
**        valueText  Textual representation of register value.
**
*****************************************************************************/
RETCODE EXPORT CpuGetRegisterValueText(REG_ID regId,U16 type,LPSTR valueText);

/****************************************************************************
**
**  CpuFindRegisterName
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Find register name of the regId. Return the regName in uppercase
**     characters, null-terminated string.
**     Notes: Caller provides buffer which is filled in by this routine.
**
**  Parameters:
**     input:
**        regId      Id of the register.
**     output:
**        regName    Pointer to buffer to receive name of the register.
**
*****************************************************************************/
RETCODE EXPORT CpuFindRegisterName(REG_ID regId, LPSTR regName);

/****************************************************************************
**
**  CpuFindRegisterId
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Find register id with its name. Return the id of the regName.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**     IMPORTANT NOTES:
**       refRegName must be a null-terminated string and contains only
**       uppercase characters.
**
**  Parameters:
**     input:
**        refRegName    Name of the register to find id.
**     output:
**        regId         returned Id of the register name.
**
*****************************************************************************/
RETCODE EXPORT CpuFindRegisterId(U8 FAR *regName, REG_ID FAR *regId);

/****************************************************************************
**
**  CpuGetRegisterWidth
**
**  Status:  REVIEWED & TESTED
**
**  Description:
**     Return the width (in number of bits) of the specified register id.
**     See specific CPU header file for registers id. (Ex: 68030.h)
**
**  Parameters:
**     input:
**        regId      id of register.
**     output:
**        regWidth   returned the width of register.
**
*****************************************************************************/
RETCODE EXPORT CpuGetRegisterWidth(REG_ID regId, U8 FAR *regWidth);

/****************************************************************************
**
**  CpuRegisterMultipleRegisters
**
**  Description:
**     Introducing the concept of register pairs. Sometime in a processor with
**     16 bit registers the compiler will group two 16 bit registers together
**     and use this group of registers as a "new" 32 bit register.  To isolate
**     support for this concept the CPU server will assign a regular looking
**     REG_ID to use when refering to this "new" register.  
**
**  Parameters:
**     input:
**        regList    REG_ID for each CPU register that makes up this register
**                   pair.  regList[0] is the MS register.
**     output:
**        assignedRegNum   register pair's own shiny brand new REG_ID.
**
*****************************************************************************/
RETCODE EXPORT CpuRegisterMultipleRegisters(REG_ID regList[2],
                                            REG_ID FAR *assignedRegNum);

/****************************************************************************
**
**  CpuSetFramePtrRegister
**
**  Description:
**     Different compilers can use different registers as the frame pointer
**     register.  This allows the loader to inform the CPU server which
**     register to use as the frame pointer.
**
**  Parameters:
**     input:
**        regId      REG_ID for which CPU register to use as frame pointer
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT CpuSetFramePtrRegister(REG_ID regId);

/****************************************************************************
**
**  CpuGetNumberOfSignals
**
**  Description:
**     Get the number of signals defined by this pod.
**
**  Parameters:
**     numSignals (out): number of signals defined by the pod.
**
*****************************************************************************/
RETCODE EXPORT CpuGetNumberOfSignals(U16 FAR *numSignals);

/****************************************************************************
**
**  CpuGetSpaceMode
**
**  Description:
**     Get the space mode (Supervisor or User) from regitser SR
**
*****************************************************************************/
RETCODE EXPORT CpuGetSpaceMode(SPACE_MODE FAR *spaceMode);

/****************************************************************************
**
**  CpuGetCoreType
**
**  Description:
**     Get the core type
**
*****************************************************************************/
RETCODE EXPORT CpuGetCoreType(U16 FAR *coreType);

#ifdef __cplusplus
}
#endif

/******************************** E O F ***********************************/
