/*****************************************************************************
**
**    Name:    evttmplt.h
** 
**    Description: 
**       Routines to manage the trigger and trace event templates.
**
**    Status: PRELIMINARY
**
**    $Log:   S:/tbird/arcm306/inc/evttmplt.h_v  $
** 
**    Rev 1.0   07 Sep 1995 10:25:58   gene
** Initial revision.
**  
**     Rev 1.11   05 May 1994 15:57:16   dennis
**  Added TmpltGetMaskValueText definition. Part of fix for ppr 9110 and 8867.
**  event.c, evttmplt.c, evttmplt.def and event.def also changed.
**  
**     Rev 1.10   27 Jul 1993 14:39:30   john
**  Modified interface to TmpltGetValueText
**  
**     Rev 1.9   07 Apr 1993 17:38:56   mindy
**  a) pass XILINX_GROUP with pointers.
**  b) added routine to output formatted trace.
**  c)added two routines to support actor in the event presenter.
**  
**     Rev 1.8   21 Oct 1992 06:25:32   mindy
**  changed U8 FAR * to LPSTR to remove warnings
**  
**     Rev 1.7   06 Aug 1992 07:36:26   mindy
**  added new display type DISP_MOTO_ADDR
**  
**     Rev 1.6   03 Aug 1992 07:03:32   mindy
**  added routine to determine what's pin functions
**  
**     Rev 1.5   21 Jul 1992 06:55:14   mindy
**  added visible type typedef and prototype
**  
**     Rev 1.4   28 Apr 1992 08:09:22   mindy
**  added comment
**  
**     Rev 1.3   07 Nov 1991 14:36:52   mindy
**  a) added a routine that returns the display length for the active template 
**     field.
**  b) removed maxLen from get title since a title is a fixed string length
**  
**     Rev 1.2   01 Nov 1991 09:23:50   mindy
**  changes to match what is implemented and needed
**  
**     Rev 1.1   30 Jul 1991 13:43:26   nghia
**  changes needed for "basic system release"
 * 
 *    Rev 1.0   14 Jun 1991 13:36:44   jim
 * Initial revision.
**
**    $Header:   S:/tbird/arcm306/inc/evttmplt.h_v   1.0   07 Sep 1995 10:25:58   gene  $
**
*****************************************************************************/

#define _EVTTMPLT_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASETYPE_
#include "basetype.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/
#define MAX_TEMPLATE_NAME 16 /* 15 characters and a NULL */
#define MAX_FIELD_NAME 16 /* 15 characters and a NULL */

#define CHARS_IN_TITLE 3 /* number of characters in title, use trailing spaces
                            if less characters desired */
#define NUM_GROUPS   4

typedef enum { DISP_HEX, DISP_DEC, DISP_OCT, DISP_BIN, 
               ASSERT_DEF, ASSERT_FIRST,
               DISP_MOTO_ADDR,
               DF_DUMMY=0x7fff
} DISPLAY_FORMAT;

typedef enum { ACTIVE_LOW, ACTIVE_HIGH, ACTIVE_NA, AS_DUMMY=0x7fff 
} ACTIVE_STATE;
               /* ACTIVE_NA means this entry is not applicable */

typedef enum { VALUE_FIELD, RANGE_FIELD, FT_DUMMY=0x7fff 
} FIELD_TYPE;

typedef enum { R_GROUP, S_GROUP, T_GROUP, U_GROUP, XN_DUMMY=0x7fff
} XILINX_NUMBER;

typedef struct {
   U32 xilinxGroup[NUM_GROUPS];
} XILINX_GROUP;

typedef enum {IO_FUNC,SELF_FUNC,CS8_FUNC,CS16_FUNC,PF_DUMMY=0x7ff} PIN_FUNC;
/* !!! making enumeration values match register encoding (p. 4-35 of 332) */
      
                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

/*
**
** Open/Close Template
**
*/

/*****************************************************************************
** 
**    TmpltOpenTemplate
** 
**    Description:
**       Opens an event template, which is a list of entries containing
**       information about how to reference the Xilinx (trace) bits.  Each
**       entry has the following information:
**          name (used for identification and for long titles)
**          title (a three character abbreviation for creating trace titles)
**          display format (how trace should display information)
**          active state (for bits, whether active low or high)
**          trailingBlanks (how many blanks to put after the display)
**          entry type (rules about how the entry can be manipulated)
**          number of Xilinx bits (width of entry in terms of trace bits)
**          Xilinx start bit (least significant bit 0..127, where R=0..31,
**                            S=32..63, etc.)
**      
**    Parameters:
**       input:
**          none
**       output:
**          id:            id of template
**
*****************************************************************************/
RETCODE EXPORT TmpltOpenTemplate(DESCRIPTOR FAR *id);

/*****************************************************************************
** 
**    TmpltTemplateClose
** 
**    Description:
**       Removes from memory the state table associated with this template.
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TmpltTemplateClose(DESCRIPTOR id);

/*
**
** Set and Get active template
**
*/

/*****************************************************************************
** 
**    TmpltSetTemplateName
** 
**    Description:
**       Sets the incoming name as the currently active template. 
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          name:          name of active template
**
*****************************************************************************/
RETCODE EXPORT TmpltSetTemplateName(DESCRIPTOR id, LPSTR name);

/*****************************************************************************
** 
**    TmpltGetTemplateName
** 
**    Description:
**       Returns the name of the currently active template.
**
**    Parameters:
**       input:
**          id:            id of template
**          maxLen:        max length of storage for active template name
**       output:
**          name:          name of active template
**
*****************************************************************************/
RETCODE EXPORT TmpltGetTemplateName(DESCRIPTOR id, LPSTR name, U16 maxLen);

/*
**
** Get general template information
**
*/

/*****************************************************************************
** 
**    TmpltGetNumOfEntries
** 
**    Description:
**       Returns the number of template entries found in template.
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          numOfEntries:  number of template entries
**
*****************************************************************************/
RETCODE EXPORT TmpltGetNumOfEntries(DESCRIPTOR id, U16 FAR *numOfEntries);

/*****************************************************************************
** 
**    TmpltGetNameGiveFieldNumber
** 
**    Description:
**       Returns the name associated with  a particular template field number.
**       Also causes the active field to be set to the field number given if
**       field number is a valid field number. 
**
**    Parameters:
**       input:
**          id:            id of template
**          fieldNum:      number of field ( first field == 1 not 0 )
**          maxLen:        max length of storage
**       output:
**          name:          name of template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetNameGivenFieldNumber(DESCRIPTOR id, U16 fieldNum,
                                            LPSTR name, U16 maxLen);

/*
**
** Get information for a particular template field.
**
*/

/*****************************************************************************
** 
**    TmpltSetFieldName
** 
**    Description:
**       Sets the incoming field name as the currently active template 
**       field.
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          fieldName:     name of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltSetFieldName(DESCRIPTOR id, LPSTR fieldName);

/*****************************************************************************
** 
**    TmpltGetFieldName
** 
**    Description:
**       Returns the name of the currently active template field (e.g.
**       address, data, nmi, etc.).
**
**    Parameters:
**       input:
**          id:            id of template
**          maxLen:        max length of storage for active template field
**       output:
**          fieldName:     name of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetFieldName(DESCRIPTOR id,LPSTR fieldName,U16 maxLen);

/*****************************************************************************
** 
**    TmpltGetTitle
** 
**    Description:
**       Returns the "title" field of the currently active template field.
**
**    Parameters:
**       input:
**          id:        id of template
**       output:
**          title:     title of active template field (storage must be
**                     CHARS_IN_TITLE + 1.
**
*****************************************************************************/
RETCODE EXPORT TmpltGetTitle(DESCRIPTOR id, LPSTR title);

/*****************************************************************************
** 
**    TmpltGetDisplayFormat
** 
**    Description:
**       Returns the "display format" field of the currently active template
**       field.
**
**    Parameters:
**       input:
**          id:                id of template
**       output:
**          displayFormat:     display format of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetDisplayFormat(DESCRIPTOR id,
                                     DISPLAY_FORMAT FAR *displayFormat);

/*****************************************************************************
** 
**    TmpltGetDisplayLength
** 
**    Description:
**       Returns the number of character positions the current template field
**       needs to display itself.
**
**    Parameters:
**       input:
**          id:               id of template
**       output:
**          length:           number of character positions field needs 
**
*****************************************************************************/
RETCODE EXPORT TmpltGetDisplayLength(DESCRIPTOR id, U8 FAR *length);

/*****************************************************************************
** 
**    TmpltGetActiveState
** 
**    Description:
**       Returns the "active state" field of the currently active template
**       field.
**
**    Parameters:
**       input:
**          id:              id of template
**       output:
**          activeState:     active state of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetActiveState(DESCRIPTOR id,
                                   ACTIVE_STATE FAR *activeState);

/*****************************************************************************
** 
**    TmpltGetTrailingBlanks
** 
**    Description:
**       Returns the "trailing blanks" field of the currently active
**       template field.
**
**    Parameters:
**       input:
**          id:                 id of template
**       output:
**          trailingBlanks:     trailing blanks of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetTrailingBlanks(DESCRIPTOR id, U16 FAR *trailingBlanks);

/*****************************************************************************
** 
**    TmpltGetFieldType
** 
**    Description:
**       Returns the "entry type" of the currently active template
**       field.
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          fieldType:     type of active template field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetFieldType(DESCRIPTOR id, FIELD_TYPE FAR *fieldType);

/*****************************************************************************
** 
**    TmpltGetVisible
** 
**    Description:
**       The returns TRUE indicates that the active field should be displayed
**       in the trace and/or event presenters. 
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          visible:       visible display status of field
**
*****************************************************************************/
RETCODE EXPORT TmpltGetVisible(DESCRIPTOR id, BOOLEAN FAR *visible);

/*****************************************************************************
** 
**    TmpltGetValueText
** 
**    Description:
**       Returns the value field of the currently active template field
**       as an ASCII text string.  This elimates the calling routine
**       from having to get the number of xilinx bits, starting position
**       etc. and having to do the shifting and masking to get a single
**       template field's value.  This routine will return text in a 
**       suitible form depending upon the display type and number of bits,
**       etc.  This routine will also zero fill; thus the returned text
**       will not have any blanks.
**
**    Parameters:
**       input:
**          id:            id of template
**          bits:          xilinx bit data to use to calculate field value
**          maxLen:        max length of storage
**          prefix:        set TRUE if the value should have a radix prefix
**       output:
**          valueText:     ASCII text of active template field's value
**
*****************************************************************************/
RETCODE EXPORT TmpltGetValueText(DESCRIPTOR id, XILINX_GROUP FAR *bits,
                                 LPSTR valueText, U16 maxLen, BOOLEAN prefix);

/*****************************************************************************
** 
**    TmpltGetMaskValueText
** 
**    Description:
**       Returns the value field of the currently active mask field
**       as an ASCII text string.  This elimates the calling routine
**       from having to get the number of xilinx bits, starting position
**       etc. and having to do the shifting and masking to get a single
**       template field's value.  This routine will return text in a 
**       suitible form depending upon the display type and number of bits,
**       etc.  This routine will also zero fill; thus the returned text
**       will not have any blanks.
**
**    Parameters:
**       input:
**          id:            id of template
**          bits:          xilinx bit data to use to calculate field value
**          maxLen:        max length of storage
**       output:
**          valueText:     ASCII text of active template field's value
**
*****************************************************************************/
RETCODE EXPORT TmpltGetMaskValueText(DESCRIPTOR id, XILINX_GROUP FAR *bits,
                                 LPSTR valueText, U16 maxLen);

/*****************************************************************************
** 
**    TmpltGetValue
** 
**    Description:
**       Returns the value of the currently active template field.
**
**    Parameters:
**       input:
**          id:            id of template
**          bits:          xilinx bit data to use to calculate field value
**       output:
**          value:         active template field's value
**
*****************************************************************************/
RETCODE EXPORT TmpltGetValue(DESCRIPTOR id, XILINX_GROUP FAR *bits,
                                  U32 FAR *value);

/*****************************************************************************
** 
**    TmpltSetValue
** 
**    Description:
**       Sets the currently active template field value to incoming value.
**
**    Parameters:
**       input:
**          id:            id of template
**          value:         active template field's value
**          bits:          xilinx bit data of unaffected bits
**       output:
**          bits:          xilinx bit data with active field value changed.
**
*****************************************************************************/
RETCODE EXPORT TmpltSetValue(DESCRIPTOR id, XILINX_GROUP FAR *bits,
                                  U32 value);

/*****************************************************************************
** 
**    TmpltGetGroup
** 
**    Description:
**       Returns the xilinx group which contains the currently active 
**       template field.
**
**    Parameters:
**       input:
**          id:            id of template
**       output:
**          group:         active template field's xilinx group
**
*****************************************************************************/
RETCODE EXPORT TmpltGetGroup(DESCRIPTOR id, XILINX_NUMBER FAR *group);

/*****************************************************************************
** 
**    TmpltWhatsPinsFunction
** 
**    Description:
**       Returns the current pin function for the requested chip select.
**
**    Parameters:
**       input:
**          cs:            which chip select we want to know about.
**       output:
**          func:          current function for requested pin
**
*****************************************************************************/
RETCODE EXPORT TmpltWhatsPinsFunction(U16 cs, PIN_FUNC FAR *func);

/*
**
**  Template Utilites
**
*/

/*****************************************************************************
** 
**    TmpltRestoreTemplates
** 
**    Description:
**       Retrieves templates from the named file.
**      
**    Parameters:
**          fileName:      name of file that contains the templates.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TmpltRestoreTemplates(LPSTR fileName);

/*****************************************************************************
** 
**    TmpltSaveTemplates
** 
**    Description:
**       Saves all templates to the named file.
**      
**    Parameters:
**          fileName:      name of file to contain templates.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TmpltSaveTemplates(LPSTR fileName);

/*****************************************************************************
** 
**    TmpltInitEventTemplate
** 
**    Description:
**       Initializes the event template server
** 
**    Parameters:
**       input:
**          none
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TmpltInitEventTemplate(VOID);

/*****************************************************************************
** 
**    TmpltInitEventTemplateCli
** 
**    Description:
**       Registers the event template server's routines with the CLI
**       server, and performs other event template server initialization.
** 
**    Parameters:
**       input:
**          cliHandle:  registration target for servers.
**          dllHandle:  handle to use in messages to tag this DLL.
**       output:
**          none
**
*****************************************************************************/
RETCODE EXPORT TmpltInitEventTemplateCli(HANDLE cliHandle, HANDLE dllHandle);

/*
   Trace formatting routine(s)
 */
     
/*****************************************************************************
** 
**    TmpltGetFormattedTrace
** 
**    Description:
**       Goes through the entire active template and returns a formatted
**       buffer of all the visible template fields.  Caller supplies the
**       memory for the buffer.
**
**    Parameters:
**       input:
**          id:            id of template
**          buffer:        pointer to storage for text string
**          bits:          xilinx data to use in getting value.
**       output:
**          buffer:        text string filled in with template values.
**
*****************************************************************************/
RETCODE EXPORT TmpltGetFormattedTrace(DESCRIPTOR id,
                                      LPSTR buffer,
                                      XILINX_GROUP FAR *bits);

/*****************************************************************************
** 
**    TmpltDisplayAddressSpace
** 
**    Description:
**       Returns TRUE if the function code pins (fc2-0) are all being
**       used as function codes and thus the address space value formed
**       by the fc2-0 pins is a valid address space. 
**
**    Parameters:
**       input:
**          none
**       output:
**          status:        TRUE if address space should be displayed.
**
*****************************************************************************/
RETCODE EXPORT TmpltDisplayAddressSpace(BOOLEAN FAR *status);


/*****************************************************************************
** 
**    TmpltDisplayBusControlSignals
** 
**    Description:
**       Returns TRUE if the bus control signals should be displayed.
**       This routine checks the mapping of the BR, BG, and BGACK
**       signals to determine if they should be displayed.  If processors
**       that don't allow these signals to be mapped always return TRUE.
**
**    Parameters:
**       input:
**          none
**       output:
**          status:        TRUE if bus signals should be displayed.
**
*****************************************************************************/
RETCODE EXPORT TmpltDisplayBusControlSignals(BOOLEAN FAR *status);

/*********************************** E O F **********************************/
   