/***************************************************************************

**
**  Name:  emusetup.c
**
**  Description:
**     Michelle setup-group routines.
**     include:
**     "Get the identity of the Michelle FW version, HW configuration
**      and HW model type",
**     "Set access size to be byte/word/dword-accessing",
**     "Set or read the memory-write verify-flag",
**     "Control ready using external/internal",
**     "Set or read the memory mapping",
**     "Read all setup information",
**     "Set control signals",
**    * "set or read the wait-state" --- This is a issue to be discussed.
**      We will decide this latter.
**
**  Status:  preliminary
**
**  $Log:   S:/tbird/arcmtat2/m307/emusetup.c_v  $
** 
**    Rev 1.0   20 Mar 1997 13:20:28   gene
** Initial revision.
** 
**    Rev 1.0   15 Jan 1997 15:30:20   gene
** Initial revision.
** 
**    Rev 1.5   18 Sep 1996 16:00:22   gene
** added setcanbaseaddr
** 
**    Rev 1.4   18 Sep 1996 15:42:06   gene
** 
**    Rev 1.3   23 Jul 1996 14:04:44   gene
** 
**    Rev 1.3   15 Apr 1996 12:44:04   gene
** nothing
** 
**    Rev 1.2   25 Jan 1996 16:20:38   gene
** fixed MapEorER() for target rombrk set/delete
** 
**    Rev 1.1   11 Jan 1996 11:29:54   gene
** removed internal register limitation
** 
**    Rev 1.1   11 Jan 1996 11:28:04   gene
** removed internal register limitation
** 
**    Rev 1.0   11 Jan 1996 10:48:26   gene
** Initial revision.
** 
**    Rev 1.0   03 Jan 1996 13:50:50   gene
** Initial revision.
** 
**    Rev 1.8   11 Dec 1995 11:49:16   kevin
** gene's modification: ssi/sso
** 
**    Rev 1.7   17 Nov 1995 11:06:46   kevin
** added check set same map block
** 
**    Rev 1.6   17 Nov 1995 09:26:22   kevin
** rewrote emuGetMap() and fixed a bug in attrFw2Sw
** 
**    Rev 1.5   15 Nov 1995 16:22:32   kevin
** added getMapTable()
** 
**    Rev 1.5   09 Nov 1995 12:42:22   gene
** add EmuGetMapTable()
** 
**    Rev 1.4   19 Oct 1995 10:28:00   gene
** modify MapEorER() match one 128k block case
** 
**    Rev 1.2   18 Oct 1995 18:30:12   gene
** modify MapGuard and DeleteMap
** 
**    Rev 1.1   18 Oct 1995 08:08:42   gene
** new emusetup.c for SetMapAction
** 
**    Rev 1.0   07 Sep 1995 10:47:28   gene
** Initial revision.
** 
** 
**  $Header:   S:/tbird/arcmtat2/m307/emusetup.c_v   1.0   20 Mar 1997 13:20:28   gene  $
**
** Copyright (C) 1992 Microtek International, Inc.
**
****************************************************************************/

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _EMU_SETUP_
#include "emusetup.h"
#endif

#ifndef _EMU_LIB_
#include "emulib.h"
#endif

#ifndef _EMU_TRACE_
#include "emutrace.h"
#endif

#ifndef _EMU_LLFW_
#include "emullfw.h"
#endif

#ifndef _EMU_EXTERNAL_
#include "emuext.h"
#endif

#include <string.h>
                       /****************************
                        *                          *
                        *     LOCAL DEFINITIONS    *
                        *                          *
                        ****************************/
struct {
   U32 addr1,addr2;
   U16 attr;
} mapData;

#define MAX_BANK          4
#define MAX_MAP           16
#define TOP_NULL          0xfffffffeul
#define BUTTOM_NULL       0xfffffffful
#define MAX_ADDR          0xffffful
#define BLOCK_SIZE        2048L // 2K
#define BLOCK_NO          1024 // EMM Size max 8M, each bank 2M, 2M/2K=1024
#define ER_BOUND_SIZE     0x20000ul


typedef struct {
U32  addr1,
     addr2;
}SLICE;

BANK_INFO bankInfo[MAX_BANK+1];
MAP_INFO mapSetInfo[MAX_BANK*2-1];
MAP_TABLE_INFO mapTable[MAX_MAP+1];
SLICE slice[MAX_BANK];
U8 bankInfoCount,mapSetCount;
U32 bankSize;
enum BANK_NO {BANK1, BANK2, BANK3, BANK4, ER_BANK, NO_BANK = 0xff};

                        /****************************
                         *                          *
                         *    EXTERNAL VARIABLES    *
                         *                          *
                         ****************************/

                        /****************************
                         *                          *
                         *     LOCAL PROTOTYPES     *
                         *                          *
                         ****************************/

VOID PRIVATE RealMapSet(U16 attr, U32 mapEnd, U32 mapStart, U16 action);
VOID PRIVATE VirMapSet(U16 attr, U32 mapEnd, U32 mapStart, U16 action);
VOID PRIVATE L_FwMapSet(VOID);
VOID PRIVATE clearMapEntry(U8 entryPtr);
VOID PRIVATE setTableInfoBankAddr(MAP_TABLE_INFO *mapTableInfo, SLICE *slice,U8 bankNo);
STATUS PRIVATE addMapTable(MAP_TABLE_INFO *mapInfo);
STATUS PRIVATE FindEqualMap(U32 startAddr,U32 endAddr,U16 attr,U8 mapTablePtr);
STATUS PRIVATE DeleteMap(MAP_TABLE_INFO *mapInfo);
BOOLEAN PRIVATE checkMapEqu(U32 startAddr, U32 endAddr, U16 attr);

                        /****************************
                         *                          *
                         *      EXECUTABLE CODE     *
                         *                          *
                         ****************************/


/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS InitMapTable(VOID) {
STATUS status;
U16 lp,lp2;

   status = GOOD;
   mapBuf[0] = MAP_END_RECORD; /* set map data to map buffer */
   if (InitMapCtr()) status = EMM_SET_ERROR;
   bankInfoCount = 0;
   for (lp = 0; lp <=  MAX_BANK; lp++) {
      bankInfo[lp].status = NO_BANK;   // bank no use
      bankInfo[lp].bottom = BUTTOM_NULL;
      bankInfo[lp].top    = TOP_NULL;
      if (lp < MAX_BANK) { //** gene
         slice[lp].addr1 = BUTTOM_NULL;
         slice[lp].addr2 = TOP_NULL;
         for (lp2 = 0; lp2 < BLOCK_NO; lp2++)
            bankInfo[lp].table[lp2] = 0x0cff;  // external
      }
   }
   for (lp = 0 ; lp <= MAX_MAP ; lp++) {
      mapTable[lp].startAddr = BUTTOM_NULL;
      mapTable[lp].endAddr = TOP_NULL;
      mapTable[lp].attr = 0;
      for (lp2=0 ; lp2 < MAX_BANK+1 ; lp2++) {
         mapTable[lp].bank[lp2] = 0;
         mapTable[lp].bankSA[lp2] = BUTTOM_NULL;
         mapTable[lp].bankEA[lp2] = TOP_NULL;
      }
   }
   SetMapAction(0,0,0,0x4200);
   return(status);
}
/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS SliceNewAddr(U32 startAddr, U32 endAddr) {
U8 lp,lp2;

   if ((endAddr - startAddr) > (bankSize*4 - 1)) return(ADDR_RANGE_OVERFLOW);
   else {
      lp = 0;
      slice[lp].addr1 = startAddr;
      slice[lp].addr2 = ((startAddr/bankSize)+1)*bankSize - 1;
      while (slice[lp].addr2 < endAddr) {
         if (lp >= 3) return(ADDR_RANGE_OVERFLOW);
         slice[lp+1].addr1 = slice[lp].addr2 + 1;
         slice[lp+1].addr2 =
            ((slice[lp+1].addr1/bankSize)+1)*bankSize - 1;
         lp++;
      }
      slice[lp].addr2 = endAddr;
      for (lp2 = (U8)(lp+1); lp2 < MAX_BANK; lp2++) {   /* CLEAR ANOTHER addr */
         slice[lp2].addr1 = BUTTOM_NULL;
         slice[lp2].addr2 = TOP_NULL;
      }
   }
   return(GOOD);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/

U8 SearchBank(U32 newAddr,U8 *bankNo) {
U8  lp,findBank;

   findBank = 0;
   if (bankInfoCount > 0) {
      for (lp = 0; lp < MAX_BANK; lp++)
         if ((newAddr >= bankInfo[lp].bottom) && (newAddr <= bankInfo[lp].top)) {
            bankNo[findBank] = lp;
            findBank++;
         }
   }
   return(findBank);
}
/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
VOID VirMapSet(U16 attr, U32 addrEnd, U32 addrStart, U16 action) {
   mapSetInfo[mapSetCount].startAddr = addrStart;
   mapSetInfo[mapSetCount].endAddr = addrEnd;
   mapSetInfo[mapSetCount].attr = attr;
   mapSetInfo[mapSetCount].action = action;
   mapSetCount++;
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
VOID SetMapTable(U32 bankAddr1, U32 bankAddr2, U16 attr,U8 bankNo) {
   U16 startBlock, endBlock;
   U16 lp;
   U16 tempAttr, action;

   if (bankNo != ER_BANK) {
      action = bankNo ;
      tempAttr = attr ;
      if ( tempAttr >= 0x0c00 ) tempAttr = 0xcff;
      startBlock = (U16)((bankAddr1 - bankInfo[bankNo].bottom) / BLOCK_SIZE);
      endBlock = (U16)(((bankAddr2 - bankInfo[bankNo].bottom) / BLOCK_SIZE) + 1);
      for (lp = startBlock; lp < endBlock ; lp++) {
         bankInfo[bankNo].table[lp] = tempAttr;
      }
      if (tempAttr >= 0xc00) {                    // gene
         action |= 0x1300 ;
      } else {                                      //
         action |= (attr & 0x0c00) >> 2 ;
         action |= 0x1000 ;
      }
   } else {
      if ( attr >= 0x0c00 )
         action = 0x2800;
      else
         action = 0x2900;
   }
   VirMapSet(attr,bankAddr2,bankAddr1,action);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID RealMapSet(U16 attr, U32 mapEnd, U32 mapStart, U16 action) {
//U32 addrEnd,addrStart;
//   addrEnd = (mapEnd >> 11);
//   addrStart = (mapStart >> 11);
   SetMapAction(attr,mapEnd,mapStart,action);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U8 CreateBank(U32 bankAddr1, U32 bankAddr2, U16 attr) {
U8 lp;
S16 lp2;

   lp = 0 ;
   for (lp = 0; lp < MAX_BANK; lp++) {
      if (bankInfo[lp].status == NO_BANK) {
         bankInfo[lp].status = lp;
         bankInfo[lp].bottom = (bankAddr1 / bankSize) * bankSize;
         bankInfo[lp].top = bankInfo[lp].bottom + bankSize - 1;
         for (lp2 = 0; lp2 < BLOCK_NO; lp2++)
            bankInfo[lp].table[lp2] = 0x0cff;  // external
         SetMapTable(bankAddr1, bankAddr2, attr,lp);
         bankInfoCount++ ;
         break;
      }
   }

   return ((U8) bankInfo[lp].status);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS DeleteBank(U8 bankNo) {
S16 lp;

   if (bankNo > MAX_BANK) return(!GOOD);
   else {
      bankInfo[bankNo].status = NO_BANK;
      bankInfo[bankNo].bottom = BUTTOM_NULL;
      bankInfo[bankNo].top = TOP_NULL;
      for (lp = 0; lp < BLOCK_NO; lp++)
         bankInfo[bankNo].table[lp] = 0x0cff;  // external
      bankInfoCount--;
   }
   return(GOOD);
}

VOID ClearEmptyBank(U8 lp) {
   S16 lp2,blockNo;
   U16 action;

   blockNo = (S16) (bankSize/BLOCK_SIZE);
   for (lp2 = 0; lp2 < blockNo; lp2++)
      if ((bankInfo[lp].table[lp2] & 0x0cff) != 0x0cff)
         break; // not external

   if (lp2 >= blockNo) {
      action = 0x4000;
      action |= lp;
      VirMapSet(0xcff,bankInfo[lp].top,bankInfo[lp].bottom,action);
      DeleteBank(lp); // All blocks of bank are external or ER.
   }
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrSw2Fw(U16 attr) {
U16 tempAttr = 0;

   // Allocation Type.
   tempAttr += ((attr & 0x07) << 10);// copy bit 0,1,2 to 10,11,12 // I,IR,E,ER,G
   // Space.
   if (tempAttr & 0x1000) tempAttr=0x0f00; // G
   if (attr & 0x400) tempAttr |= 0x02; // copy bit 10 to bit 1   //UD
   if (attr & 0x800) tempAttr |= 0x04; // copy bit 11 to bit 2   //UP
   if (attr & 0x100) tempAttr |= 0x20; // copy bit  8 to bit 5   //SD
   if (attr & 0x200) tempAttr |= 0x40; // copy bit  9 to bit 6   //SP
   return(tempAttr);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrFw2Sw(U16 attr) {
U16 tempAttr = 0;

   // Allocation Type.
   tempAttr += (attr & 0x1c00) >> 10;
   // Space.
   if (attr & 0x02) tempAttr |= 0x400; // copy bit 1 to bit 10  //UD
   if (attr & 0x04) tempAttr |= 0x800; // copy bit 2 to bit 11  //UP
   if (attr & 0x20) tempAttr |= 0x100; // copy bit 5 to bit  8  //SD
   if (attr & 0x40) tempAttr |= 0x200; // copy bit 6 to bit  9  //SP
   return(tempAttr);
}

STATUS CheckValidBank(MAP_INFO mapInfo,U8 bankNo) {
U16 startBlock,endBlock,lp;

  startBlock = (U16)((mapInfo.startAddr - bankInfo[bankNo].bottom) / BLOCK_SIZE);
  endBlock = (U16) (((mapInfo.endAddr - bankInfo[bankNo].bottom) / BLOCK_SIZE) + 1);
  for (lp =startBlock; lp < endBlock; lp++)
     if ((bankInfo[bankNo].table[lp] & 0xcff) != 0xcff)
        return(FALSE);
  return(TRUE);
}

STATUS PRIVATE FindEqualMap(U32 startAddr,U32 endAddr,U16 attr,U8 mapTablePtr) {
   U8 spaceType;

   spaceType = (U8) (attr & 0xff);
   if ( (mapTable[mapTablePtr].startAddr == startAddr) &&
        (mapTable[mapTablePtr].endAddr == endAddr) &&
        ( (U8)(mapTable[mapTablePtr].attr & 0xff) == spaceType) )
      return(TRUE);
   return(FALSE);
}

VOID setTableInfoBankAddr(MAP_TABLE_INFO *mapTableInfo, SLICE *slice,U8 bankNo) {

    mapTableInfo->bank[bankNo] = 1;
    mapTableInfo->bankSA[bankNo] = slice->addr1;
    mapTableInfo->bankEA[bankNo] = slice->addr2;
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS MapGuard(MAP_TABLE_INFO *mapInfo) {
   if (((mapInfo->endAddr+1)%ER_BOUND_SIZE) || (mapInfo->startAddr % ER_BOUND_SIZE))
      return(ERROR_BOUN);
   else {
      mapInfo->bank[ER_BANK] = 1;
      mapInfo->bankSA[ER_BANK] = mapInfo->startAddr;
      mapInfo->bankEA[ER_BANK] = mapInfo->endAddr;
      RealMapSet(mapInfo->attr,mapInfo->endAddr,mapInfo->startAddr,0x2a00);
      addMapTable(mapInfo);
      return(GOOD);
   }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID CopyBankInfo(BANK_INFO *desBank, BANK_INFO *sourceBank) {
U8 lp;
U16 lp2;
   for (lp = 0; lp < MAX_BANK; lp++) {
      desBank[lp].top = sourceBank[lp].top;
      desBank[lp].bottom = sourceBank[lp].bottom;
      desBank[lp].status = sourceBank[lp].status;
      for ( lp2 = 0; lp2 < BLOCK_NO; lp2++)
         desBank[lp].table[lp2] = sourceBank[lp].table[lp2];
   }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS addMapTable(MAP_TABLE_INFO *mapInfo) {
   U8 lp,lp2;

   for ( lp=0 ; lp < MAX_MAP ; lp++)
      if (mapTable[lp].attr == 0) break;
   mapTable[lp].startAddr = mapInfo->startAddr;
   mapTable[lp].endAddr = mapInfo->endAddr;
   mapTable[lp].attr = mapInfo->attr;
   for ( lp2=0 ; lp2 < MAX_BANK+1 ; lp2++) {
      mapTable[lp].bank[lp2] = mapInfo->bank[lp2];
      if (mapInfo->bank[lp2] != 0) {
         mapTable[lp].bankSA[lp2] = mapInfo->bankSA[lp2];
         mapTable[lp].bankEA[lp2] = mapInfo->bankEA[lp2];
      }
   }
   return GOOD;
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID clearMapEntry(U8 entryPtr) {
   U8 lp;

   mapTable[entryPtr].startAddr = BUTTOM_NULL;
   mapTable[entryPtr].endAddr = TOP_NULL;
   mapTable[entryPtr].attr = 0;
   for (lp=0 ; lp < MAX_BANK+1 ; lp++)
      if ( mapTable[entryPtr].bank[lp] != 0) {
         mapTable[entryPtr].bank[lp] = 0;
         mapTable[entryPtr].bankSA[lp] = BUTTOM_NULL;
         mapTable[entryPtr].bankEA[lp] = TOP_NULL;
      }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
BOOLEAN checkMapEqu(U32 startAddr, U32 endAddr, U16 attr) {
   U8 lp;

   for (lp=0 ; lp < MAX_MAP ; lp++)
      if ( mapTable[lp].startAddr == startAddr &&
           mapTable[lp].endAddr == endAddr &&
           mapTable[lp].attr == attr)
         return (TRUE);
   return (FALSE);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS DeleteMap(MAP_TABLE_INFO *mapInfo) {
   U8 mapPtr, bankPtr;

   for (mapPtr = 0; mapPtr < MAX_MAP ; mapPtr++) {
      if ( FindEqualMap(mapInfo->startAddr,mapInfo->endAddr,mapInfo->attr,mapPtr) )
         break;
   }
   if (mapPtr >= MAX_MAP) return(BANK_FULL);
   for ( bankPtr=0 ; bankPtr < MAX_BANK+1 ; bankPtr++) {
      if (mapTable[mapPtr].bank[bankPtr] != 0) {
         SetMapTable(mapTable[mapPtr].bankSA[bankPtr],mapTable[mapPtr].bankEA[bankPtr],
                     mapInfo->attr,bankPtr);
         if (bankPtr < MAX_BANK)
            ClearEmptyBank(bankPtr);
      }
   }
   clearMapEntry(mapPtr);
   L_FwMapSet();
   return(GOOD);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID L_FwMapSet(VOID) {
U8 lp;
   for (lp = 0; lp < mapSetCount; lp++)
      RealMapSet(mapSetInfo[lp].attr,mapSetInfo[lp].endAddr,
                 mapSetInfo[lp].startAddr,mapSetInfo[lp].action);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS MapEorER(MAP_TABLE_INFO *mapTableInfo) {
   U8 NumOfBank,lp,lp2;
   U8 bankNo[MAX_BANK],oneBankNo;
   SLICE sliceER[2];
   BANK_INFO bankCopy[MAX_BANK];
   MAP_INFO mapInfo;

   if ( (mapTableInfo->attr & 0x0066) != 0x0066 )
      return (FALSE);

   CopyBankInfo(bankCopy, bankInfo);

   for (lp = 0; lp < 2; lp++) {
      sliceER[lp].addr1 = BUTTOM_NULL;
      sliceER[lp].addr2 = TOP_NULL;
   }

   //slice address to sliceER[0],mapStart--mapEnd,sliceER[1]
   //sliceER[0] and sliceER[1] are not full bank boundary
   // the range from mapStart to mapEnd is full bank boundary

   sliceER[0].addr1 = mapTableInfo->startAddr;
   sliceER[0].addr2 = ((mapTableInfo->startAddr/ER_BOUND_SIZE)+1)*ER_BOUND_SIZE - 1;
   if (mapTableInfo->endAddr < sliceER[0].addr2)
      return (FALSE);

   if ((sliceER[0].addr1 % ER_BOUND_SIZE) == 0) {
      while (mapTableInfo->endAddr >= (sliceER[0].addr2+ER_BOUND_SIZE)) {
         sliceER[0].addr2 += ER_BOUND_SIZE;
      }
      sliceER[0].addr2 = mapTableInfo->endAddr;
   }

   if (mapTableInfo->endAddr == sliceER[0].addr2) {
      if ((sliceER[0].addr1 % ER_BOUND_SIZE) != 0)
         return (FALSE);
   } else {
     sliceER[1].addr1 = sliceER[0].addr2 + 1;
     sliceER[1].addr2 = mapTableInfo->endAddr ;
   }

   if ((sliceER[0].addr1 % ER_BOUND_SIZE) == 0) {
      SetMapTable(mapTableInfo->startAddr,mapTableInfo->endAddr,mapTableInfo->attr,ER_BANK);
      setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),ER_BANK);
   }
   else if ( (sliceER[1].addr2-sliceER[1].addr1+1) >= ER_BOUND_SIZE) {
      mapInfo.startAddr = slice[lp].addr1;
      mapInfo.endAddr = slice[lp].addr2;
      mapInfo.attr = mapTableInfo->attr;
      NumOfBank = SearchBank(sliceER[0].addr1,bankNo);
      if (NumOfBank == 0) { // not found
         oneBankNo = CreateBank(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr);
         if (oneBankNo == 0xFF) {
            if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
            return(BANK_FULL);
         }
         setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),oneBankNo);
      } else {
         for (lp2 = 0; lp2 < NumOfBank; lp2++) {
            if (CheckValidBank(mapInfo,bankNo[lp2])) {
               SetMapTable(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr,bankNo[lp2]);
               setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),bankNo[lp2]);
               break;
            }
         }
         if (lp2 >= NumOfBank) {
            oneBankNo = CreateBank(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr);
            if (oneBankNo == 0xFF) {
               if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
               return(BANK_FULL);
            }
            setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),oneBankNo);
         }
      }
      SetMapTable(sliceER[1].addr1,sliceER[1].addr2,mapTableInfo->attr,ER_BANK);
      setTableInfoBankAddr(mapTableInfo,&(sliceER[1]),ER_BANK);
   } else return (FALSE);

   addMapTable(mapTableInfo);
   L_FwMapSet();
   return(TRUE);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS SetMap(U32 startAddr, U32 endAddr, U16 attr) {
   STATUS status;
   U8 NumOfBank,lp,lp2,allocType;
   U8 bankNo[MAX_BANK],oneBankNo;
   MAP_INFO mapInfo;
   MAP_TABLE_INFO mapTableInfo;
   BANK_INFO bankCopy[MAX_BANK];

   if (bankSize == 0) return(NO_EMM);
   attr = AttrSw2Fw(attr);
   
   if (checkMapEqu(startAddr,endAddr,attr)) return(GOOD);
   
   allocType = (U8)((attr & 0xff00) >> 8); // attribute
   mapTableInfo.startAddr = startAddr;
   mapTableInfo.endAddr = endAddr;
   mapTableInfo.attr = attr;
   for ( lp=0 ; lp < MAX_BANK+1 ; lp++) {
      mapTableInfo.bank[lp] = 0;
      mapTableInfo.bankSA[lp] = BUTTOM_NULL;
      mapTableInfo.bankEA[lp] = TOP_NULL;
   }
   mapSetCount = 0;
   if ((startAddr == 0) && (endAddr == 0)) return(InitMapTable());
   else {
//      if (((startAddr >= INTER_REG_ADDR) || (endAddr >= INTER_REG_ADDR)) && (attr & 0x0020))
//          return(MICE_LIMIT); //the address is in internal register(0xfffff000,0xffffffff),and attribute is SD.

      switch(allocType) {
         case EXTERNAL_MAP : return(DeleteMap(&mapTableInfo));
         case GUARD_MAP : return(MapGuard(&mapTableInfo));
         case EXTERNAL_READ1 :
         case EXTERNAL_READ2 :
         case EXTERNAL_READ3 :
         case EXTERNAL_READ4 : if (!(status=MapEorER(&mapTableInfo))) break;
                               if (status == BANK_FULL) return(status);
                               else return(GOOD);
         default : break;
      }

      if (status = SliceNewAddr(startAddr,endAddr))
            return(ADDR_RANGE_OVERFLOW);
      CopyBankInfo(bankCopy, bankInfo);
      lp = 0;
      while ((slice[lp].addr1 != BUTTOM_NULL) &&
             (slice[lp].addr2 != TOP_NULL) && (lp < 4)) {
         mapInfo.startAddr = slice[lp].addr1;
         mapInfo.endAddr = slice[lp].addr2;
         NumOfBank = SearchBank(slice[lp].addr1,bankNo);
         if (NumOfBank == 0) { // not found
            oneBankNo = CreateBank(slice[lp].addr1,slice[lp].addr2,attr);
            if (oneBankNo == 0xFF) {
               if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
               return(BANK_FULL);
            }
            setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),oneBankNo);
         } else {
            for (lp2 = 0; lp2 < NumOfBank; lp2++) {
               if (CheckValidBank(mapInfo,bankNo[lp2])) {
                  SetMapTable(slice[lp].addr1,slice[lp].addr2,attr,bankNo[lp2]);
                  setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),bankNo[lp2]);
                  break;
               }
            }
            if (lp2 >= NumOfBank) {
               oneBankNo = CreateBank(slice[lp].addr1,slice[lp].addr2,attr);
               if (oneBankNo == 0xFF) {
                  if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
                  return(BANK_FULL);
               }
               setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),oneBankNo);
            }
         }
         lp++;
      }
      addMapTable(&mapTableInfo);
      L_FwMapSet();
   }
   return(GOOD);
}

/***************************************************************************
**
**  EmuSetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**               "addr" is the starting address of this mapping memory.
**               It should be on the address boundary of 64KB.
**               "map" is the mapping of current 64K memory bank.
**               If memoryresolution is 4KB and map is 0xFFCF. The emulation
**               memory will be mapped as two segments. 0..0x3fff and
**               0x6000..0xffff.
**               The definition of "attr" as below:
**               BIT Meaning
**               --- --------------------------------------------
**                0  "0" = Internal Memory, "1" = External memory
**                1  "0" = Read/Write, "1" = Read-Only
**                2  "0" = Exist memory, "1" = Gard memory
**                3  Reserved
**                4  Segments(with bit 5) -- "00" = ES, "01" = CS
**                5  Segments(with bit 4) -- "10" = SS, "11" = DS
**                6  Reserved
**                7  Reserved
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetMap(U32 addr1, U32 addr2, U16 attrib) {
STATUS status;

   status = GOOD;
   StopRun();
   status = SetMap(addr1,addr2,attrib);
   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

U32 ChangeLowHigh(U32 longData){
U16 highWord,lowWord;

    highWord = HighWord(longData);
    lowWord = LowWord(longData);
    return(((U32)lowWord << 16) + (U32)highWord);

}
/***************************************************************************
**
**  EmuGetMapTable
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**
**  Parameters:
**     input:
**        "retMap" -- point to MAP_INFO struct, return H_FW MapTable.
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMapTable(MAP_INFO *retMap) {
   U16 count;

   StopRun();
//   tempEnd = 0xaa55;
//   SetMapTbl();
   count = 0;
   while (count < MAX_MAP) {
      retMap->startAddr = mapTable[count].startAddr;
      retMap->endAddr = mapTable[count].endAddr;
      retMap->attr = AttrFw2Sw(mapTable[count].attr);
      count++;
      retMap++;
   }
   OutputStatus(GOOD,ON);
}

/***************************************************************************
**
**  EmuGetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMap(VOID) {
U16 count, tempEnd;

   StopRun();
   OutputStatus(GOOD,OFF);
   for (count=0; count<MAX_MAP; count++) {
      mapData.addr1 = mapTable[count].startAddr;
      mapData.addr2 = mapTable[count].endAddr;
      mapData.attr  = AttrFw2Sw(mapTable[count].attr);
      OutData(10,&mapData,OFF);
   }
   tempEnd = 0xaa55;
   OutData(2,&tempEnd,OFF);
   OutEnd();
}

//VOID EmuGetMap(VOID) {
//U16 lp,count,tempEnd;
//MAP_INFO *mapPtr1,*mapPtr2,*tempPtr;
//
//   StopRun();
//   tempEnd = 0xaa55;
//   SetMapTbl();
//   count = 0;
//   mapPtr1 = mapPtr2 = (MAP_INFO *)mapBuf;
//   mapPtr2++;
//   if ((U16)(mapPtr1->startAddr) != MAP_END_RECORD) {
//      while (((U16)(mapPtr2->startAddr) != MAP_END_RECORD) && (count < 86)) {
//         // same address range.
//         count++;
//         if ((mapPtr1->startAddr == mapPtr2->startAddr) &&
//             (mapPtr1->endAddr == mapPtr2->endAddr) &&
//             (HighByte(mapPtr1->attr) == HighByte(mapPtr2->attr)))
//            break;
//         mapPtr2++;
//      }
//      if (count >= 86) {
//         OutputStatus(MAP_FAIL,ON);
//         return;
//      }
//      OutputStatus(GOOD,OFF);
//      for (lp = 0; lp < count; lp++) {
//         tempPtr = mapPtr2;
//         while ((U16)(tempPtr->startAddr) != MAP_END_RECORD) {
//            if ((mapPtr1->startAddr == tempPtr->startAddr) &&
//                (mapPtr1->endAddr == tempPtr->endAddr) &&
//                (HighByte(mapPtr1->attr) == HighByte(tempPtr->attr)))
//                mapPtr1->attr |= tempPtr->attr; // combine space
//            tempPtr++;
//         }
//         mapData.addr1 = mapPtr1->startAddr << 11;
//         mapData.addr2 = (mapPtr1->endAddr << 11) + 0x7ff;
//         mapData.attr = AttrFw2Sw(mapPtr1->attr);
//         mapPtr1++;
//
//         OutData(10,&mapData,OFF);
//      }
//   }
//   else OutputStatus(GOOD,OFF);
//   OutData(2,&tempEnd,OFF);
//   OutEnd();
//}

/***************************************************************************
**
**  EmuGetMapSize
**
**  Description:
**
**  Parameters:
**     input:
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMapSize(VOID) {
U32 phyEmmSize;
U8 dataLen;

   OutputStatus(GOOD,OFF);
   phyEmmSize = bankSize * 4;
   dataLen = sizeof(phyEmmSize);
   OutData(dataLen,&phyEmmSize,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuSetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetControl(U16 signals) {

   AccessSignals(WRITE_ALL,noUse,&signals);
   OutputStatus(GOOD,ON);
}


/***************************************************************************
**
**  EmuGetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetControl(VOID) {
U16 signals;
U8 dataLen;

   AccessSignals(READ_ALL,noUse,&signals);
   OutputStatus(GOOD,OFF);
   dataLen = sizeof(signals);
   OutData(dataLen,&signals,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuGetStatus
**
**  Description: Michelle setup-group routine, to read the current setting
**               of the Michelle.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." be Normal return
**        "Emulator Fatal Error" for HW error
**        return data -- "length" + "events" + "qualify: + "trigger" +
**                       "MAP" + control signals" +
**                       "ready setting"
**
**                     * "ID" is a issue to be discussed later
**
****************************************************************************/
VOID EmuGetStatus(VOID) {
U16 flyFlag;

  AccessIceFlag(READ_ONE,FLYING,&flyFlag);
  if (flyFlag) OutputStatus(CPU_FLY,ON);
  else if (!CheckEpStop()) OutputStatus(CPU_RUN,ON);
  else OutputStatus(BROKE,ON);
}

/***************************************************************************
**
**  EmuGetID
**
**  Description: Michelle setup-group routine, to get the emulator's ID to
**               make sure that driver program has connected to a correct
**               emulator.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**        return data -- "length" + "ID"
**
****************************************************************************/
VOID EmuGetID(VOID) {
U8 dataLen;

   OutputStatus(GOOD,OFF);
   dataLen = MAX_ID;
   // get probe,trace,cp board info from llfw and save to idData
   OutData(dataLen,idData,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuSetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetReady(U16 readyFlag) {

   AccessIceFlag(WRITE_ONE,READY_FLAG,&readyFlag);
   if (readyFlag == 0) ReadyInternal();
   else ReadyExternal();
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetReady(VOID) {
STATUS status;
U16 readyFlag;
U8 dataLen;

   status = GOOD;
   AccessIceFlag(READ_ONE,READY_FLAG,&readyFlag);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(readyFlag);
   OutData(dataLen,&readyFlag,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetAccessSize(SIZE size) {
STATUS status;

   status = GOOD;
   status = AccessIceFlag(WRITE_ONE,SIZE_FLAG,&size);

   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetAccessSize(VOID) {
STATUS status;
SIZE size;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(size);
   OutData(dataLen,&size,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetWait(U16 count) {

// remove for 328, 307  AccessIceFlag(WRITE_ONE,WAIT_FLAG,&count);

// remove for 328, 307  WaitStates(count);

   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}


/****************************************************************************
**
**  EmuGetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetWait(VOID) {
U8 dataLen;
U16 count;

   AccessIceFlag(READ_ONE,WAIT_FLAG,&count);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(count);
   OutData(dataLen,&count,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetSync(U16 state) {

   AccessIceFlag(WRITE_ONE,SYNC_FLAG,&state);
   switch (state) {
      case 0:
         DisableSSI();
         DisableSSO();
         AccessIceFlag(WRITE_ONE,SYNC_IN,&C_OFF);
        break;
      case 1:
         EnableSSI();
         DisableSSO();
         AccessIceFlag(WRITE_ONE,SYNC_IN,&C_ON);
        break;
      case 2:
         DisableSSI();
         EnableSSO();
        break;
      case 3:
         EnableSSI();
         EnableSSO();
        break;
   }
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}

/****************************************************************************
**
**  EmuGetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetSync(VOID) {
U16 sync;
U8 dataLen;

   AccessIceFlag(READ_ONE,SYNC_FLAG,&sync);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(sync);
   OutData(dataLen,&sync,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuSetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetVerify(U16 verifyFlag) {

   AccessIceFlag(WRITE_ONE,VERIFY_FLAG,&verifyFlag);

   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetVerify(VOID) {
STATUS status;
U16 verifyFlag;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(verifyFlag);
   OutData(dataLen,&verifyFlag,ON);
   OutEnd();
}

VOID EmuEnSwBkpt(VOID) {

   SwbpEnable();
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

VOID EmuDisSwBkpt(VOID) {

   SwbpDisable();
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

VOID EmuSetBkptCode(BKPT_OP_CODE bkptOpCode) {

   SelectBreakCode((U16)bkptOpCode);
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

VOID EmuSetCanBase(U32 baseAddr) {

   CanBaseAddr = baseAddr;
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}
/******************************** E O F ***********************************/

