/*****************************************************************************
**
** Name: addr.h
**
** Description:   provide the address conversion routines to other servers
**
** Copyright (C) 1991,1994 Microtek International.  All rights reserved.
**
** $Log:   S:/tbird/mt2_68k/inc/addr.h_v  $
** 
**    Rev 1.0   13 Feb 1997 08:49:34   gene
** Initial revision.
** 
**    Rev 1.1   14 Dec 1995 09:12:38   kevin
** added AdrConvTextToAddressForReg()
** 
**    Rev 1.0   07 Sep 1995 10:25:40   gene
** Initial revision.
**  
**     Rev 1.36   11 Apr 1994 09:58:26   marilyn
**  Added some comments
**  
**     Rev 1.35   08 Apr 1994 14:54:48   nghia
**  Added AdrDoPhysicalRangesOverlap()
**  
**     Rev 1.34   06 Apr 1994 16:26:16   nghia
**  Added new interface: AdrSetCurrentAddrSpace() and AdrMaskAddrMSB().
**  
**     Rev 1.33   18 Mar 1994 17:35:58   marilyn
**  Added another interface.
**  
**     Rev 1.32   17 Mar 1994 15:25:34   marilyn
**  Completed updated interfaces for x86 support.
**  
**     Rev 1.31   28 Feb 1994 16:33:00   marilyn
**  Updated interface to handle Intel processors.  Added address specific 
**  commands from the proc server.
**  
**     Rev 1.30   28 Jun 1993 07:22:04   doug
**  centralizing error messages in errtext/mkerrors.h
**  
**     Rev 1.29   26 Mar 1993 13:29:06   doug
**  merge John's 340 changes
**  
**     Rev 1.28.1.1   01 Mar 1993 13:08:12   john
**  added Subtract routine that doesn't error on underflow.
**  
**     Rev 1.28.1.0   25 Feb 1993 16:21:56   john
**  No change.
**  
**     Rev 1.28   12 Jan 1993 10:28:20   john
**  made addr_space enums match in addr.h and sdtempl.h
**  
**     Rev 1.27   03 Sep 1992 12:45:10   brucea
**  Added: AdrCliDebug
**  
**     Rev 1.26   31 Aug 1992 09:21:18   brucea
**  Added: AdrGetAllocCounts for debug purposes
**  
**     Rev 1.25   25 Aug 1992 10:50:58   brucea
**  Added: AdrIsValidCodeAddress
**  
**     Rev 1.24   18 Aug 1992 10:09:16   brucea
**  Added: AdrConvAddressToTextWithParams, AdrDoRangesOverlap
**  
**  
**     Rev 1.23   14 Aug 1992 15:12:46   doug
**  added syntax errro
**  
**     Rev 1.22   13 Aug 1992 11:23:44   brucea
**  Added: error 0x13 INVALID_DESCRIPTOR
**  
**     Rev 1.21   24 Jul 1992 12:53:14   brucea
**  Added: SPACE_DONT_CARE in enum ADDR_SPACE
**  Added: ER_ADR_SPACE_FOR_MOTO_ONLY error message
**  
**     Rev 1.20   23 Jul 1992 09:37:18   brucea
**  Added: errors ER_ADR_ADDRESS_TOO_LARGE and ER_ADR_RESULT_UNDERFLOW
**  
**     Rev 1.19   20 Jul 1992 15:50:28   doug
**  address space is duplicated in sdtempl.h, so use ifdef to share
**  
**     Rev 1.18   14 Jul 1992 14:38:26   brucea
**  Added: calls to PROC.H functions; removed processor-specific variables,
**     typedefs, and functions
**  #ifdef'ed around all logical/physical functions that deal with MMU
**  
**  
**     Rev 1.17   10 Jul 1992 18:46:04   brucea
**  Comment changes
**  
**     Rev 1.16   15 May 1992 14:36:42   brucea
**  Added: all error #defines (removed addrerrs.h)
**  
**     Rev 1.15   11 May 1992 11:25:00   brucea
**  Added: AdrCopyAddress to export interface
**  
**     Rev 1.14   19 Mar 1992 16:54:14   brucea
**  Added: AdrIsAddrInRange function declaration
**  
**     Rev 1.13   27 Feb 1992 17:42:18   brucea
**  Added: #define ADDR_BUFF_SZ 20
**  Removed: processor parameter from AdrInitAddressServer
**  Changed: symbol usage initialization to FALSE
**  Reversed: order of parameters for AdrCreateAddressWithType
**  Added: AdrGetAddressToTextLength, AdrConvAddressToTextNoFill,
**         AdrConvertToPhysical, AdrGetEndianType, AdrGetMaxAddress
**  Added: FAR to all return parameters that didn't have it
**  
**  
**     Rev 1.12   30 Jan 1992 19:27:56   brucea
**  Added enum ENDIAN_TYPE, header for AdrConvertPointerToAddress and
**     AdrGetLogicalEqualsPhysicalFlag
**  
**     Rev 1.11   25 Jan 1992 16:26:48   brucea
**  Added AdrCreateAddressWithType
**  Renamed AdrSetAddrType to AdrConvertSameAddress
**  
**     Rev 1.10   14 Jan 1992 17:04:56   tom
**  Added changes on Allen's behalf.
**  
**     Rev 1.9   05 Dec 1991 14:16:48   doug
**  added routines needed by variable watch server
**  
**     Rev 1.8   27 Nov 1991 15:19:42   brucea
**  Added #ifdef __cplusplus for compiling with C++ mode enabled
**  
**     Rev 1.7   20 Nov 1991 07:51:16   doug
**  added range of address support
**  
**     Rev 1.6   15 Nov 1991 11:05:18   doug
**  This is the new version of the address server that hides the address
**  structure so it is more generic
**  
**     Rev 1.5   23 Aug 1991 13:16:38   john
**  Changed the explicit path to basetype.h to implicit.
**  
**     Rev 1.4   13 Aug 1991 13:04:50   brucea
**  changed ADDR to ADDR_STRUCT
**  
**     Rev 1.3   16 Jul 1991 10:08:28   brucea
**  Changed ADDR_TYPE to {PHYSICAL, SELECTOR_OFFSET}
**  Changed MODIFIER to {POS, NEG}
**  Changed typedef ADDR by removing ldt component
**  Added typedef SELECTOR, OFFSET
**  Changed ADDR_PAIR to ADDR_FOR_WORD_RECOGNIZER and changed its structure
 * 
 *    Rev 1.2   26 Jun 1991 12:11:58   brucea
 * No change.
 * 
 *    Rev 1.1   26 Jun 1991 11:44:12   brucea
 * Changed the include path to s:\tbird\core\inc
 * 
 *    Rev 1.0   14 Jun 1991 13:35:16   jim
 * Initial revision.
**
** $Header:   S:/tbird/mt2_68k/inc/addr.h_v   1.0   13 Feb 1997 08:49:34   gene  $
**
*/

#define _ADDR_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _PROC_
#include "proc.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define ADDR_BUFF_SZ 45    /* used for allocating strings to be passed back */

/* used for allocating string for AdrConvAddressToTextWithParams when output
   is symbolic */
#define ADDR_BUFF_SZ_SYMBOLIC 256

typedef enum { /* types of addresses that can be referred to */
   ADDR_VIRTUAL, /* Intel's segment/offset (real/v86 mode) or selector/offset
                    (protected modes) */
   ADDR_LINEAR, /* Motorola logical space */
                /* Intel linear space (real/v86 is segment shifted 4 bits and
                   added to offset); protected uses GDT entry base + offset */
   ADDR_PHYSICAL /* Intel/Motorola output on address pins (after page tables,
                    if used or same as linear/logical if no page tables) */
} ADDR_TYPE;

typedef enum { /* different processor modes that affect address calculation */
   MODE_NONE, /* no mode, this (or current) is used for non-x86 processors */
   MODE_CURRENT,    /* use the current processor state */
   MODE_REAL,       /* x86 real mode */
   MODE_PROT16,     /* x86 16 bit protected mode */
   MODE_PROT32,     /* x86 32 bit protected mode */
   MODE_V86,        /* x86 virtual 86 mode */
   MODE_SMM         /* x86 smm mode */
} ADDR_MODE;

#ifndef _ADDR_SPACE_
typedef enum { /* Motorola address spaces */
   SPACE_UNDEF0, SPACE_UD, SPACE_UP, SPACE_UNDEF3, SPACE_UNDEF4,
   SPACE_SD, SPACE_SP, SPACE_CPU, SPACE_DONT_CARE, 
/* Intel address spaces */
   SPACE_USER, SPACE_SMM, SPACE_IO, SPACE_DEFAULT = 0x7fff
} ADDR_SPACE;
#define _ADDR_SPACE_
#endif

typedef enum { /* results from the address compare routine */
   FIRST_ADDR_GREATER, SECOND_ADDR_GREATER, EQUAL_ADDRS
} ADDR_COMPARE;

typedef enum { /* how the segment selector should be interpreted */
   ADDR_USE_VALUE, /* use the value supplied by the caller */
   ADDR_USE_CS,    /* use the current code segment value whenever referenced */
   ADDR_USE_DS,    /* current data segment */
   ADDR_USE_SS     /* current stack segment */
} ADDR_SEGSEL_TYPE;

typedef enum { /* adjust range by moving or expanding direction */
   ADDR_RANGE_LOW,  /* move the range to lower memory by n bytes */
   ADDR_RANGE_HIGH, /* move the range to higher memory by n bytes */
   ADDR_RANGE_EXPAND_LOW, /* expand range towards lower memory by n bytes */
   ADDR_RANGE_EXPAND_HIGH,/* expand range towards higher memory by n bytes */
   ADDR_RANGE_EXPAND_BOTH /* expand range towards high and low memory
                              by n bytes in both directions */
} ADDR_ADJ_RANGE_TYPE;

typedef enum { /* address range components */
   RANGE_START,
   RANGE_END,
   RANGE_BOTH
} ADDR_RANGE_COMP;

typedef enum { /* operand and address size */
   ADDR_USE_16,  /* 16 bit */
   ADDR_USE_32,  /* 32 bit */
   ADDR_USE_AUTO
} ADDR_OP_SIZE;


                       /****************************
                        *                          *
                        *          ERRORS          *
                        *                          *
                        ****************************/

/* errors are in errtext/mkerrors.h */

#ifdef __cplusplus
extern "C" {
#endif
                        /****************************
                         *                          *
                         *        PROTOTYPES        *
                         *                          *
                         ****************************/

/*
**  Initialization of Server
*/

/****************************************************************************
**
**  AdrInitAddressServer
**
**  Description:
**     This routine must be called before any other routines are called
**
**  Parameters:
**     input:
**        none
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrInitAddressServer(VOID);

/*
**  Create/Destroy Address Descriptor
*/

/****************************************************************************
**
**  AdrCreateAddress
**
**  Description:
**     This routine creates an opaque address descriptor.  After receiving
**     the descriptor, the other routines that follow are called to set and
**     manipulate the descriptor.  When done with descriptor,
**     AddrDestroyAddress eliminates the descriptor.
**
**     Sets the address to the default state.  The defaults are:
**
**     FAMILY_UNKNOWN (assumes straight 32 bit linear space):
**        address type:       ADDR_LINEAR
**        address mode:       MODE_NONE
**        address space:      unused
**        ldt selector:       unused
**        symbol usage:       FALSE
**        segment/selector:   unused
**
**     FAMILY_68K:
**        address type:       ADDR_LINEAR
**        address mode:       MODE_NONE
**        address space:      SPACE_SD
**        ldt selector:       unused
**        symbol usage:       FALSE
**        segment/selector:   unused
**
**     FAMILY_X86:
**        address type:       ADDR_VIRTUAL
**        address mode:       MODE_CURRENT
**        address space:      SPACE_USER
**        ldt selector:       0
**        symbol usage:       FALSE
**        segment/selector:   type=ADDR_USE_DS, value=0
**
**  Parameters:
**     input:
**        none
**     output:
**        desc:  opaque address descriptor
**
*****************************************************************************/
RETCODE EXPORT AdrCreateAddress(DESCRIPTOR FAR *desc);

/****************************************************************************
**
**  AdrCreateAddressWithType
**
**  Description:
**     This routine is the same as AdrCreateAddress with the addition that
**     it sets the address type to <type>.
**
**  Parameters:
**     input:
**        type: the type to set the address descriptor to after it is created
**     output:
**        desc: descriptor to opaque address structure
**
*****************************************************************************/
RETCODE EXPORT
AdrCreateAddressWithType(ADDR_TYPE type, DESCRIPTOR FAR *desc);

/****************************************************************************
**
**  AdrCreateAddressFromText
**
**  Description:
**     This routine is the same as AdrCreateAddress with the addition that
**     it creates the appropriate address type and initializes the
**     address from a text string.
**     Supported formats are as follows:
**
**     INTEL formats:
**          <offset>L			         Linear address
**          <offset>P			         Physical address
**          (#module)#symbol	         Symbolic segment:offset interpreted
**                                        as Virtual address
**          <ldt>:<segment>:<offset>	Virtual address with specific LDT
**          <segment>:<offset>	      Virtual address using current LDT
**          <offset>			            Virtual address assuming current
**                                        LDT and DS
**
**     MOTOROLA formats:
**          (#module)#symbol	Symbolic offset interpreted as Linear address
**          <offset>			   Linear address assuming default SD space
**
**
**  Parameters:
**     input:
**        string : address in textual format
**        space : space designator in textual format
**     output:
**        addrDesc: new address descriptor to opaque address structure
**                  *** CALLER MUST DESTROY ****
**
*****************************************************************************/
RETCODE EXPORT
AdrCreateAddressFromText(LPSTR string, LPSTR space, DESCRIPTOR FAR *desc);

/****************************************************************************
**
** AdrGetAllocCounts
**
** Description:
**    For debugging, returns the number of addresses created (allocated)
**    and destroyed (deallocated).
**
** Output parameters:
**    lpAddrsCreated: number of addrs created since addr.dll initialized
**    lpAddrsDestroyed: number of addrs destroyed since addr.dll initialized
**
*****************************************************************************/
RETCODE EXPORT
AdrGetAllocCounts(U32 FAR *lpAddrsCreated, U32 FAR *lpAddrsDestroyed);

/****************************************************************************
**
**  AdrDestroyAddress
**
**  Description:
**     This routine removes an address descriptor from use.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrDestroyAddress(DESCRIPTOR desc);

/**************************************************************************
**
**  AdrReturnSpaces
**
**  Description:
**      Returns a pointer to the space designator strings
**      separated by null terminators and the number of strings.
**
**  Parameters:
**      spaces (out) : pointer to first null terminated space string
**      spaceCount (out) : number of space strings present
**      spaceLen (out) : length of space string
**
***************************************************************************/
RETCODE EXPORT AdrReturnSpaces(LPSTR FAR *spaces, U16 FAR *spaceCount,
                               U16 FAR *spaceLen);

/*
**  Setting/Getting Address Descriptor Elements
*/

/****************************************************************************
**
**  AdrSetAddrType
**
**  Description:
**     This routine sets the address type.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        type:  type to set address to
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrType(DESCRIPTOR desc, ADDR_TYPE type);

/****************************************************************************
**
**  AdrGetAddrType
**
**  Description:
**     This routine gets the address type.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        type:  current setting of address type
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrType(DESCRIPTOR desc, ADDR_TYPE FAR *type);

/****************************************************************************
**
**  AdrSetAddrMode
**
**  Description:
**     This routine sets the address mode.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        mode:  mode to set address to
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrMode(DESCRIPTOR desc, ADDR_MODE mode);

/****************************************************************************
**
**  AdrGetAddrMode
**
**  Description:
**     This routine gets the address mode.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        mode:  current setting of address mode
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrMode(DESCRIPTOR desc, ADDR_MODE FAR *mode);

/****************************************************************************
**
**  AdrSetAddrSpace
**
**  Description:
**     This routine sets the address space.  Accepts DEFAULT_SPACE as a
**     valid space input, and uses the current default space.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        space:  space to set address to
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrSpace(DESCRIPTOR desc, ADDR_SPACE space);

/****************************************************************************
**
**  AdrGetAddrSpace
**
**  Description:
**     This routine gets the address space.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        space:  current setting of address space
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrSpace(DESCRIPTOR desc, ADDR_SPACE FAR *space);

/****************************************************************************
**
**  AdrSetCurrentAddrSpace
**
**  Description:
**     This routine sets the current processor space for the input address
**     descriptor.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetCurrentAddrSpace(DESCRIPTOR desc);

/****************************************************************************
**
**  AdrSetAddrSymbolUsage
**
**  Description:
**     This routine sets a flag indicating symbols should be searched when
**     returning text from the conversion routines.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        useSymbols:  true to convert using symbols, false for straight text
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrSymbolUsage(DESCRIPTOR desc, BOOLEAN useSymbols);

/****************************************************************************
**
**  AdrGetAddrSymbolUsage
**
**  Description:
**     This routine returns the current setting of whether symbols should
**     be displayed in conversion routines.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        useSymbols:  true to convert using symbols, false for straight text
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrSymbolUsage(DESCRIPTOR desc,
                                     BOOLEAN FAR *useSymbols);

/*--------------------------------------**
**   Address Text Conversion Routines   **
**--------------------------------------**/

/****************************************************************************
**
**  AdrConvAddressToText
**
**  Description:
**     This routine converts an opaque address descriptor into an ASCII
**     text string.  The text string will be in a standardized form which
**     depends on the processor family, the type, mode, etc.
**
**     To simplify the routine, the caller provides a buffer of size
**     ADDR_BUFF_SZ, which is larger than any possible address text.  This
**     allows the caller to allocate/free memory (or use the stack).
**
**     The symbolUsage flag determines if the address will be translated
**     to a symbolic form or not.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        text:  buffer supplied by caller
**     output:
**        text:  ASCII text of the address
**
*****************************************************************************/
RETCODE EXPORT AdrConvAddressToText(DESCRIPTOR desc, LPSTR text);

/****************************************************************************
**
** AdrConvAddressToTextWithParams
**    - supercedes AdrConvAddressToText and AdrConvAddressToTextNoFill
**
** Description:
**    This routine converts an opaque address descriptor into an ASCII
**    text string.  The text string will be in a standardized form which
**    depends on the processor family, the type, mode, etc.
**
**    If symbolUsage == OFF, the address will be converted to hex.
**    Use ADDR_BUFF_SZ for allocating a string for the return text.
**
**    If symbolUsage == ON, the address will be converted to its symbol
**    match.  Use ADDR_BUFF_SZ_SYMBOLIC (much larger) for return string
**    allocation.
**
** Input parameters:
**    desc:  opaque address descriptor
**    with0X: if TRUE, 0x prepended to hex output
**    leadZeroFill: if TRUE, leading 0's will be used to make the hex output
**       a fixed width equal to the return parameter from proc.dll
**       ProcReturnMaxOutputAddrDigits call.
**    
**    output:
**       str: ASCII text of the address in hex or symbolic form
**
*****************************************************************************/
RETCODE EXPORT
AdrConvAddressToTextWithParams(DESCRIPTOR desc,
                               BOOLEAN    with0X,
                               BOOLEAN    leadZeroFill,
                               LPSTR      str);

/****************************************************************************
**
** AdrConvAddrRangeToText
**
** Description:
**    This routine converts an opaque address range descriptor into an ASCII
**    text string.  The text string will be in a standardized form which
**    depends on the processor family, the type, mode, etc.
**    Address range is in the format "<startaddress>..<endaddress>".
**    Start address, end address, or both addresses can be converted to
**    text.  If the descriptor is not a range descriptor only the
**    start address can be formatted.  To format the end address the
**    descriptor must be a range.
**
**    If symbolUsage == OFF, the address will be converted to hex.
**    Use ADDR_BUFF_SZ for allocating a string for the return text.
**
**    If symbolUsage == ON, the address will be converted to its symbol
**    match.  Use ADDR_BUFF_SZ_SYMBOLIC (much larger) for return string
**    allocation.
**
** Input parameters:
**    desc:  opaque address descriptor
**    with0X: if TRUE, 0x prepended to hex output
**    leadZeroFill: if TRUE, leading 0's will be used to make the hex output
**       a fixed width equal to the return parameter from proc.dll
**       ProcReturnMaxOutputAddrDigits call.
**    whichPart: parts of range address to format, start, end, or both
**    
**    output:
**       str: ASCII text of the address in hex or symbolic form
**
*****************************************************************************/
RETCODE EXPORT
AdrConvAddrRangeToText(DESCRIPTOR rangeDesc,
                       BOOLEAN    with0X,
                       BOOLEAN    leadZeroFill,
                       ADDR_RANGE_COMP  whichPart,
                       LPSTR      str);


/****************************************************************************
**
**  AdrGetAddressToTextLength
**
**  Description:
**     Returns the length of the text string of the converted address
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        length: string length of address converted to ASCII
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddressToTextLength(DESCRIPTOR desc, U8 FAR *length);


/****************************************************************************
**
**  AdrConvAddressToTextNoFill
**
**  Description:
**     This routine converts an opaque address descriptor into an ASCII
**     text string.  The text string will be in a standardized form which
**     depends on the processor family, the type, mode, etc.
**
**     To simplify the routine, the caller provides a buffer of size
**     ADDR_BUFF_SZ, which is larger than any possible address text.  This
**     allows the caller to allocate/free memory (or use the stack).
**
**     The symbolUsage flag determines if the address will be translated
**     to a symbolic form or not.  [SYMBOLIC NOT IMPLEMENTED]
**
**     This routine differs from AdrConvAddressToText in that it does not
**     leading zero fill the ascii output.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        text:  buffer supplied by caller
**     output:
**        text:  ASCII text of the address
**
*****************************************************************************/
RETCODE EXPORT AdrConvAddressToTextNoFill(DESCRIPTOR desc, LPSTR text);

/****************************************************************************
**
**  AdrConvTextToAddress
**
**  Description:
**     This routine converts an ASCII text string to an opaque address
**     descriptor.  This routine requires a previously created address
**     descriptor.  This function fills in the address structure
**     pointed to by the passed-in descriptor.  User must destroy the
**     address descriptor. The descriptor passed in will be modified
**     to match the attibutes of the supplied text for virtual, linear,
**     and physical formats.  When the symbolic format is determined, the
**     address type is preserved from the input desc by translating the
**     symbolic address appropriately.
**
**     Leading white space (using isspace()) is skipped over.
**     The symbolic input must start with #.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        text:  ASCII text of the address
**     output:
**        desc:  the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrConvTextToAddress(DESCRIPTOR desc, LPSTR text);
RETCODE EXPORT AdrConvTextToAddressForReg(DESCRIPTOR desc, LPSTR text);


/***************************************************************************
**
**  AdrTranslateSpace
**
**  Description:
**      This routine accepts textual space designators, validates the space
**      according to processor and returns the space enumeration.
**
**      CPU32 space designators: ud, up, sd(default), sp, cpu
**      CPU16 space designators: data(default), program
**      Intel space designators: user(default), smm, io
**
**  Parameters:
**      inputText (in) : textual space string
**      space (out) : validated space enumeration
**
************************************************************************/
RETCODE EXPORT AdrTranslateSpace(LPSTR inputText, ADDR_SPACE FAR *space);


/**************************************************************************
**
** AdrGetSpaceStr
**
** Description:
**   This routine accepts a space enumeration and returns the appropriate
**   space string.
**
** Parameters:
**      space (in) : space enumeration from ADDR_SPACE
**      spaceStr (out) : space string, user must allocate string space
**
***************************************************************************/
RETCODE EXPORT AdrGetSpaceStr(ADDR_SPACE space, LPSTR spaceString);

/*
**  Address Manipulation Routines
*/

/****************************************************************************
**
**  AdrAddToAddress
**
**  Description:
**     This routine adds an unsigned long number to an address.  This will
**     be interpreted differently depending on the family processor, address
**     type, mode, etc.  However, this routine takes care of all these messy
**     details, so you don't have to worry about them!
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        value:  value to add to address
**     output:
**        desc:  the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrAddToAddress(DESCRIPTOR desc, U32 value);


/****************************************************************************
**
**  AdrAddAddressTogether
**
**  Description:
**     This routine adds the offsets of the two addresses together and
**     returns the results in the first address.   The first descriptor
**     determines the maximum offset allowed.
**
**  Parameters:
**     input:
**       desc1 (in) : opaque address descriptor, returned modified
**       desc2 (in) : opaque address descriptor, returned unmodified
**     output:
**       desc1 (out) : the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrAddAddressTogether(DESCRIPTOR desc1, DESCRIPTOR desc2);


/****************************************************************************
**
**  AdrSubtractFromAddress
**
**  Description:
**     This routine subtracts an unsigned long number from an address.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        value:  value to subtract from address
**     output:
**        desc:  the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrSubtractFromAddress(DESCRIPTOR desc, U32 value);


/****************************************************************************
**
**  AdrSubtractAddresses
**
**  Description:
**     This routine performs subtraction between two address descriptors
**     such that desc1 -= desc2
**
**  Parameters:
**     input:
**        desc1:  opaque address descriptor
**        desc2:  opaque address descriptor
**     output:
**        desc1:  the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrSubtractAddresses(DESCRIPTOR desc1, DESCRIPTOR desc2);


/****************************************************************************
**
**  AdrSubtractFromAddressUnderflow
**
**  Description:
**     This routine subtracts an unsigned long number from an address.
**     It does not attempt to prevent underflow errors, it just blindly
**     subtracts the two offsets.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        value:  value to subtract from address
**     output:
**        desc:  the contents of the address descriptor are modified
**
*****************************************************************************/
RETCODE EXPORT AdrSubtractFromAddressUnderflow(DESCRIPTOR desc, U32 value);


/****************************************************************************
**
**  AdrCompareAddresses
**
**  Description:
**     This routine compares the offsets of two addresses and reports
**     which is higher than the other (or they are equal).
**     Address must be of the same space, type, and if VIRTUAL type,
**     the same segment selector to be compared.
**
**  Parameters:
**     input:
**        desc1:   opaque descriptor of first address
**        desc2:   opaque descriptor of second address
**     output:
**        result:  result of comparison
**
*****************************************************************************/
RETCODE EXPORT AdrCompareAddresses(DESCRIPTOR desc1, DESCRIPTOR desc2,
                                   ADDR_COMPARE FAR *result);

/****************************************************************************
**
**  AdrComparePhysicalAddresses
**
**  Description:
**     This routine translates two addresses to physical types and
**     compares the physical addresses and reports
**     which is higher than the other (or they are equal). The input
**     descriptors are returned unmodified.
**
**  Parameters:
**     input:
**        desc1:   opaque descriptor of first address
**        desc2:   opaque descriptor of second address
**     output:
**        result:  result of comparison
**
*****************************************************************************/
RETCODE EXPORT AdrComparePhysicalAddresses(DESCRIPTOR desc1,
                                           DESCRIPTOR desc2,
                                           ADDR_COMPARE FAR *result);


/************************************************************************
**
**  AdrAreAddrsIdentical
**
**  Description:
**      Compares all aspects of the address descriptors to see if they are
**      identical.
**
**  Parameters:
**      addr1 (in): first input descriptor
**      addr2 (in): second input descriptor
**      identical (out): TRUE or FALSE
**
**************************************************************************/
RETCODE EXPORT AdrAreAddrsIdentical(DESCRIPTOR addr1, DESCRIPTOR addr2,
                                    BOOLEAN FAR *identical);


/****************************************************************************
**
**  AdrRemainingBytes
**
**  Description:
**     This routine returns the number of bytes from the input address to
**     the maximum input address offset according to processor, type and
**     mode.  This could be used to see how much room (bytes)
**     is left for display and calculations.
**
**  NOTE: This routine ignores a range designation.  For maximum offset of
**     a range use: AdrGetEndAddrOffset.  For remaining bytes in a range,
**     given an address descriptor somewhere in the range, use:
**     AdrRemainingBytesInRange.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        numBytes:  number of bytes left in section
**
*****************************************************************************/
RETCODE EXPORT AdrRemainingBytes(DESCRIPTOR desc, U32 FAR *numBytes);



/****************************************************************************
**
** AdrIsValidCodeAddress
**
** Description: checks that the input address meets the requirement of being
**    on a boundary that matches the target processor.
**
** Input parameter:
**    desc: descriptor to valid address
**
** Output parameter:
**    isValid: TRUE if the input address is one a valid code address boundary
**             and it does not exceed the max input address
**
** Return value:
**    returns an error if address exceeds max input address
**    returns GOOD otherwise, even if isValid is FALSE.  The reason is that
**       the caller may want to create a specific error message when
**       the address is not on a valid execution boundary.
*****************************************************************************/
RETCODE EXPORT
AdrIsValidCodeAddress(DESCRIPTOR desc, BOOLEAN FAR *isValid);


/****************************************************************************
**
**  AdrSetAddrOffset
**
**  Description:
**     This routine sets the address offset.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        offset:  offset to set address to
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrOffset(DESCRIPTOR desc, U32 offset);

/****************************************************************************
**
**  AdrGetAddrOffset
**
**  Description:
**     This routine gets the address offset.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        offset:  current setting of address offset
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrOffset(DESCRIPTOR desc, U32 FAR *offset);


/**************************************************************************
**
** AdrGetMaxInputAddrOffset
**
** Purpose: returns largest valid input offset address
**
** Input parameter:
**    desc: address descriptor
** Output parameter:
**    maxInputAddress: largest valid offset address
**
***************************************************************************/
RETCODE EXPORT AdrGetMaxInputAddrOffset(DESCRIPTOR desc,
                                        U32 FAR *maxInputAddress);


/***********************************************************************
**
** AdrGetMaxOutputAddrOffset
**
** Purpose: returns largest valid output offset address
**
** Input paramter:
**    desc: address descriptor
** Output parameter:
**    maxOutputAddress: largest valid offset address
**
*************************************************************************/
RETCODE EXPORT AdrGetMaxOutputAddrOffset(DESCRIPTOR desc,
                                         U32 FAR *maxOutputAddress);

/**************************************************************************
**
** AdrGetMaxOutputAddress
**
** Purpose: returns largest valid output address as a descriptor
**    Input descriptor is unmodified.
**
** Input parameter:
**    desc: address descriptor
** Output parameter:
**    maxDesc: largest valid address as a desciptor
**
***************************************************************************/
RETCODE EXPORT AdrGetMaxOutputAddress(DESCRIPTOR desc,
                                      DESCRIPTOR FAR *maxDesc);

/*************************************************************************
**
** AdrGetMaxInputAddrDigits
**
** Purpose: returns max number of hex characters representing largest
**          input address
**
** Input parameter:
**    desc: address descriptor
** Output parameter:
**    maxInputAddrDigits: number of digits of largest input address
**
**************************************************************************/
RETCODE EXPORT AdrGetMaxInputAddrDigits(DESCRIPTOR desc,
                                        U16 FAR *maxInputAddrDigits);


/*************************************************************************
**
** AdrGetMaxOutputAddrDigits
**
** Purpose: returns max number of hex characters representing largest
**          output address
**
** Input parameter:
**    desc: address descriptor
** Output parameter:
**    maxOutputAddrDigits: number of digits of largest output address
**
*************************************************************************/
RETCODE EXPORT AdrGetMaxOutputAddrDigits(DESCRIPTOR desc,
                                         U16 FAR *maxOutputAddrDigits);

/*************************************************************************
**
** AdrMaskAddrMSB
**
** Purpose:
**    Set the input address descriptor to a valid input offset.  If the
**    input descriptor is an address range then the range is modified.
**
** Input parameter:
**    desc: address descriptor
** Output parameter:
**    none
**
*************************************************************************/
RETCODE EXPORT AdrMaskAddrMSB(DESCRIPTOR desc);

/****************************************************************************
**
**  AdrGetLdtSelector
**
**  Description:
**     This routine gets the LDT selector to use when addressing local
**     descriptor table addresses.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        ldt:  current setting of LDT to use
**
*****************************************************************************/
RETCODE EXPORT AdrGetLdtSelector(DESCRIPTOR desc, U32 FAR *ldt);


/****************************************************************************
**
**  AdrSetLdtSelector
**
**  Description:
**     This routine sets the selector that should be when using the LDT.
**     Setting to 0 means to use the current LDT register value.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        ldt:  ldt to use in addressing
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetLdtSelector(DESCRIPTOR desc, U32 ldt);



/****************************************************************************
**
**  AdrSetAddrSegmentSelector
**
**  Description:
**     This routine sets the address segment/selector (depends on mode).
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        type:  what type of default (register or value)
**        segsel:  segment/selector to set address to
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrSegmentSelector(DESCRIPTOR desc,
                                         ADDR_SEGSEL_TYPE type, U32 segsel);

/****************************************************************************
**
**  AdrGetAddrSegmentSelector
**
**  Description:
**     This routine gets the address segment/selector.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        type:  what type of default (register or value)
**        segsel:  current setting of address segment/selector
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrSegmentSelector(DESCRIPTOR desc,
                                 ADDR_SEGSEL_TYPE FAR *type,
                                 U32 FAR *segsel);


/*
**  Address Conversion Routines
*/

/****************************************************************************
**
** AdrConvertAddress
**
** Description:
**    This routine takes an address descriptor and using its mode and
**    the given type, calculates the address to be of that type.  Use the
**    address create routines to get address descriptors and the
**    set routines to set up type, mode, etc.
**
** Parameters:
**    input:
**       desc:  opaque descriptor of address to convert
**       type:  type to convert to
**    output:
**       desc:  modified address descriptor
**       rangeCovered: TRUE if input range is completely covered by
**                     by output range or FALSE it isn't
**
**
** Restrictions:
**    Converting a single address from logical, linear, or virtual to
**    physical is generally not a problem.  But an address range in logical,
**    linear, or virtual form may map onto multiple non-contiguous physical
**    addresses.
**    When converting an address range to a physical range, the code will
**    attempt the conversion.  If the size of the range maps to
**    non-contiguous physical ranges, the error ER_ADR_NONCONTIGUOUS_RANGE
**    will be generated.  If it can convert, no error will result.
*****************************************************************************/
RETCODE EXPORT AdrConvertAddress(DESCRIPTOR desc, ADDR_TYPE type,
                                 BOOLEAN FAR *rangeCovered);


/****************************************************************************
**
**  AdrCopyAddress
**
**  Description:
**    This routine copies the contents of the first address into the second.
**    No create or destroy is done on either descriptor.
*****************************************************************************/
RETCODE EXPORT AdrCopyAddress(DESCRIPTOR sourceDesc,
                              DESCRIPTOR FAR destinationDesc);


/****************************************************************************
**
**  AdrDuplicateAddress
**
**  Description:
**     This routine takes one address and makes an exact copy of it.  This
**     can be used if an original copy of an address needs to be saved.
**
**  Parameters:
**     input:
**        desc1:  opaque descriptor of source address (the one to duplicate)
**     output:
**        desc2:  opaque descriptor of destination address
**
*****************************************************************************/
RETCODE EXPORT AdrDuplicateAddress(DESCRIPTOR desc1, DESCRIPTOR FAR *desc2);


/****************************************************************************
**
**  AdrConvertPointerToAddress
**
**  Description:
**     This routine uses the input address as a memory pointer (indirection);
**     the pointer value is read from memory, interpreted as an address, and
**     placed into the descriptor structure; i.e. this is a destructive
**     write to the passed-in descriptor.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor of address (contains the address to
**               the pointer)
**        pointerSize: pointer value in bytes; determines how many bytes
**                     to read to build the new address
**     output:
**        desc:  opaque descriptor has been modified to address of pointer
**
**  Assumptions:
**     Motorola pointers are assumed to be of the LINEAR address type.
**     Intel pointers are assumed to be of the VIRTUAL address type.
**     Supports 3 pointer sizes: 2 byte, 4 byte, and 6 byte
**     On a 2 byte pointer, the 16 bits are the lower bytes of the offset.
**     On a 4 byte pointer, the 32 bits are the offset.
**     On a 6 byte pointer, the upper 16 bits are assumed to contain the
**     segment descriptor, while the lower 32 bits contain the offset.
**
*****************************************************************************/
RETCODE EXPORT AdrConvertPointerToAddress(DESCRIPTOR desc, U8 pointerSize);

/*
**  Address Range Support
*/


/****************************************************************************
**
**  AdrRangeOfAddresses
**
**  Description:
**     This routine compares two addresses and returns the number of bytes
**     the range encompasses.  If one is 1000 and the other is 1000, the
**     range is 1 byte (addr2-addr1+1).  It returns the absolute value of
**     the range (order of addresses is insignificant, use the compare
**     address routine to find which is greater, if needed).  If the range is
**     a full 32 bits (0 and 0xffffffff), the value returned is 0 since this
**     overflows (0xffffffff-0+1=0x100000000).  Callers need to check for
**     this as a special case.
**
**     Requirement:
**       Both addresses must have the same type and space.
**
**     This routine is useful for getting the number of bytes in an address
**     range to do looping.
**
**  Parameters:
**     input:
**        desc1:  opaque descriptor of first address
**        desc2:  opaque descriptor of second address
**     output:
**        numBytes:  number of bytes in the address range
**
*****************************************************************************/
RETCODE EXPORT AdrRangeOfAddresses(DESCRIPTOR desc1, DESCRIPTOR desc2,
                                   U32 FAR *numBytes);


/****************************************************************************
**
**  AdrIsAddrInRange
**
**  Description:
**     This routine compares one address to see if it is included within the
**     range of the second input address.  "Included" means:
**       inputAddr >= rangeAddr start address  AND
**       inputAddr <= rangeAddr end address
**
**  Parameters:
**     input:
**        inputAddr: descriptor of single address to be compared with range
**        rangeAddr: descriptor of address range
**     output:
**        isInRange: TRUE if inputAddr in range of rangeAddr; FALSE if not
**
**  NOTE: error is returned if rangeAddr is not set as a range address
*****************************************************************************/
RETCODE EXPORT AdrIsAddrInRange(DESCRIPTOR inputAddr, DESCRIPTOR rangeAddr,
                                BOOLEAN FAR *isInRange);


/****************************************************************************
**
**  AdrIsPhysicalAddrInRange
**
**  Description:
**     This routine compares one address to see if it is included within the
**     range of the second input address.  "Included" means:
**       inputAddr >= rangeAddr start address  AND
**       inputAddr <= rangeAddr end address
**     Address are internally converted to physcial and then compared.
**     The address descriptors are returned unmodified.
**
**  NOTE:  Remember that multiple linear addresses can map to a single
**         physical address.
**
**  Parameters:
**     input:
**        inputAddr: descriptor of single address to be compared with range
**        rangeAddr: descriptor of address range
**     output:
**        isInRange: TRUE if inputAddr in range of rangeAddr; FALSE if not
**
**  NOTE: error is returned if rangeAddr is not set as a range address
*****************************************************************************/
RETCODE EXPORT AdrIsPhysicalAddrInRange(DESCRIPTOR inputAddr,
                                        DESCRIPTOR rangeAddr,
                                        BOOLEAN FAR *isInRange);


/****************************************************************************
**
** AdrIsAddrRangeEqual
**
** Description: Compares two address ranges for equality; i.e. the address
**    offsets must be the equal and the range lengths must be equal.
**    Also requires that both addresses have <rangeActive> TRUE and that
**    their <space> fields match.
**
** Parameters:
**    rangeAddr1, rangeAddrs: address ranges
**
**     output:
**        isEqual: TRUE if rangeAddr1 == rangeAddr2; FALSE if not
**
*****************************************************************************/
RETCODE EXPORT
AdrIsAddrRangeEqual(DESCRIPTOR rangeAddr1,
                    DESCRIPTOR rangeAddr2,
                    BOOLEAN FAR *isEqual);


/**************************************************************************
**
** AdrAdjustRange
**
** Description: Adjusts the range of the address as specified by the
**   direction parameter by the number of bytes supplied.  Ranges can
**   be moved or expanded in either directions.  Limit and
**   boundary checking is done for the user.  If the requested range
**   adjustment is invalid or beyong the allowed maximum and minumum
**   limits then an error is returned.  This routine accepts
**   range off descriptors and will automatically turn on the range and
**   set the start and end offsets. Refer to ADDR_ADJ_RANGE_TYPE for
**   movements and expansions allowed
**
** Parameters:
**   desc (in/out) : address descriptor/adjusted address range descriptor
**   numBytes (in) : number of bytes for the range adjustment
**   rangeLength (in) : number of bytes for adjusted range length
**   direction (in) : enum specifying adjustment type (move or expand)
**                    and direction
**
**************************************************************************/
RETCODE EXPORT
AdrAdjustRange(DESCRIPTOR desc, U32 numBytes,
               U32 rangeLength, ADDR_ADJ_RANGE_TYPE direction);


/****************************************************************************
**
** AdrDoRangesOverlap
**
** Description:
**    Compares two ranges to see if any address in one is inside the address
**    of the other.
**
** Parameters:
**    range1: first range in comparison 
**    range2: second range in comparison
**
**    output:
**       overlap: boolean set TRUE if any address in one range is included
**                in the range of the other; set FALSE otherwise
**                If either address is NOT a range, FALSE is returned
**
*****************************************************************************/
RETCODE EXPORT
AdrDoRangesOverlap(DESCRIPTOR range1,
                   DESCRIPTOR range2,
                   BOOLEAN FAR *overlap);

/****************************************************************************
**
** AdrDoPhysicalRangesOverlap
**
** Description:
**    Compares two ranges to see if any physical address in one
**    is inside the address of the other.
**
** Parameters:
**    range1: first range in comparison 
**    range2: second range in comparison
**
**    output:
**       overlap: boolean set TRUE if any physical address in one range
**                is included in the range of the other; set FALSE otherwise
**                If either address is NOT a range, FALSE is returned
**
*****************************************************************************/
RETCODE EXPORT
AdrDoPhysicalRangesOverlap(DESCRIPTOR range1,
                           DESCRIPTOR range2,
                           BOOLEAN FAR *overlap);


/****************************************************************************
**
**  AdrRemainingBytesInRange
**
**  Description:
**     This routine returns the number of bytes from the input address to
**     the maximum address offset of the given range.  If the input
**     address is not within the range, the maximum bytes of the range
**     is returned.
**
**  Parameters:
**     input:
**        rangeDesc:  opaque address descriptor of the range
**        inputDesc:  opaque address descriptor of address somewhere
**                    within the range.
**     output:
**        numBytes:  number of bytes left in the range using inputDesc
**                   as an offset into the range.
**
*****************************************************************************/
RETCODE EXPORT AdrRemainingBytesInRange(DESCRIPTOR rangeDesc,
                                        DESCRIPTOR inputDesc,
                                        U32 FAR *numBytes);


/****************************************************************************
**
**  AdrSetRangeActive
**
**  Description:
**     This routine sets or clears the active range flag.  The default is
**     FALSE.  It is normal set TRUE by callig one of the range routines.
**     It can be explicitly set TRUE or FALSE with this call.  The default
**     value for the range is 1 byte (start address and end address equal).
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        active:  TRUE to set range active, FALSE to turn off
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetRangeActive(DESCRIPTOR desc, BOOLEAN active);

/****************************************************************************
**
**  AdrGetRangeActive
**
**  Description:
**     This routine tells whether the range is active.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        active:  TRUE if active, FALSE if not active
**
*****************************************************************************/
RETCODE EXPORT AdrGetRangeActive(DESCRIPTOR desc, BOOLEAN FAR *active);


/****************************************************************************
**
**  AdrSetEndAddrOffset
**
**  Description:
**     This routine sets the offset of the end address and sets the range
**     active flag to TRUE.  Generates error if the end address is greater
**     than the max address allowed for the processor.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        offset:  offset of the end address
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetEndAddrOffset(DESCRIPTOR desc, U32 offset);


/****************************************************************************
**
**  AdrGetEndAddrOffset
**
**  Description:
**     This routine gets the offset of the end point of a range.  Use the
**     "is range active" to see if the end address is being used.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        offset:  offset of the end address
**
*****************************************************************************/
RETCODE EXPORT AdrGetEndAddrOffset(DESCRIPTOR desc, U32 FAR *offset);


/****************************************************************************
**
**  AdrSetAddrRangeLength
**
**  Description:
**     This routine sets the offset of the end address, using a length in
**     number of bytes, and sets the range active flag to TRUE.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        numBytes:  length of the range in bytes
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT AdrSetAddrRangeLength(DESCRIPTOR desc, U32 numBytes);


/****************************************************************************
**
**  AdrGetAddrRangeLength
**
**  Description:
**     This routine gets the length, in number of bytes, of the range.  Will
**     be 0 if range is an entire 32-bits (0 to 0xffffffff).
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        numBytes:  length of the range in bytes
**
*****************************************************************************/
RETCODE EXPORT AdrGetAddrRangeLength(DESCRIPTOR desc, U32 FAR *numBytes);


/*************************************************************************
**
**  AdrCallback :
**
**  Return function for event callback.  Used to update Pmode after
**  EVENT_BKPT_HALTED.
**
****************************************************************************/
RETCODE EXPORT AdrCallback(U32 eventNum);


/****************************************************************************
**
** AdrGetPmode :
**
**  Description:
**     This routine returns the current processor mode.
**     Motorola: NONE
**     Intel: PMODE_REAL, PMODE_V86, PMODE_P16, PMODE_P32, PMODE_SMM
**
**  Parameters:
**     output:
**        pmode (out): processor mode after last BKPT_HALTED
**        pmodeStr (out): processor mode string: REAL, V86, PROT16,
**                         PROT32, SMM, NONE
**
***************************************************************************/
RETCODE EXPORT AdrGetPmode(PMODE FAR *pmode, LPSTR pmodeStr);


/*************************************************************************
**
** AdrGetDefaultSize :
**
**  Description:
**     This routine returns the operand and address size of the given
**     address descriptor based on address type and pmode.  
**
**  Parameters:
**     desc (in): input address descriptor
**     size (out): processor mode after last BKPT_HALTED
**
***************************************************************************/
RETCODE EXPORT AdrGetDefaultSize(DESCRIPTOR desc,ADDR_OP_SIZE *size);


/****************************************************************************
** AdrCliDebug
*****************************************************************************/
RETCODE EXPORT AdrCliDebug(LPSTR cmdString, U32 argc, U32 argv[]);


/**************************************************************************
** AdrCliXlt
**************************************************************************/
RETCODE EXPORT AdrCliXlt(LPSTR cmdString, U32 argc, U32 argv[]);


/**************************************************************************
** AdrCliPmode
**************************************************************************/
RETCODE EXPORT AdrCliPmode(LPSTR cmdString, U32 argc, U32 argv[]);


#ifdef __cplusplus
}
#endif
/******************************** E O F ***********************************/
