/***************************************************************************
**
**  File name : ASM.C
**
**  Description : main module and parser of assembler
**
**  Date : 11/16/1992  by C. H. Lin
**    1. Initial version (include symbol processing)
**
**  Date : 11/17/1992  by C. H. Lin
**    1. Change global variables, asmBuffer and asmMnemonic,
**       to local variables, LookAhead and Mnemonic
**
**  Date : 12/10/1992  by C. H. Lin
**    1. In GetOperands(),
**       move checking xxx PTR to before parsing [...];
**       change isxdigit to isdigit, add AreXdigits(Token) in checking data
**
**  Date : 12/22/1992  by C. H. Lin
**    1. In Assemble(), add prefix processing
**          REPxx instruction, ES/CS/SS/DS: instruction,
**    2. In GetOperands(), add processing of segment override of mem operand
**          ES/CS/SS/DS:[xxx]
**
**  Date : 12/23/1992  by C. H. Lin
**    1. In macro definition, add checking ';' into ENDOFSTR(x)
**
**  Date : 2/5/1993  by C. H. Lin
**    1. Rename asmBuffer to LookAhead
**    2. In GetOperands(), handle $ operand
**    3. In Assemble(), add pseudo instruction DS (define storage)
**    4. In PseudoDBDW(), add pseudo instruction DUP
**
**  Date : 2/9/1993  by C. H. Lin
**    1. Rename asmMnemonic to Mnemonic
**    2. Add local variable Token; change GetToken(STR Token) to GetToken()
**    3. In GetToken(), limit length of Token
**
**  Date : 2/23/1993  by C. H. Lin
**    1. In Sym2Addr(), check sym_flag
**
**  Date : 3/3/1993  by C. H. Lin
**    1. Change Sym2Addr() to be global
**       Rename Sym2Addr() to asmSym2Addr()
**
**  Date : 5/6/1993  by C. H. Lin
**    1. In ParseMemory(), consider the case [...][...][...]
**    2. Consider CALL/JMP NEAR mem
**
**
**  Copyright (C) 1993 Microtek International Inc.
**  All Rights Reserved
**
****************************************************************************/


/****************************** Include File *******************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "usym1.h"
#include "asmdef.h"

/***************************** Macro Definition ****************************/

#define COMMENT_CHAR ';'
#define ENDOFSTR(x) ((x) == '\0' || (x) == '\r' || (x) == COMMENT_CHAR)

/*********************** External Function Prototype ***********************/

/* defined in ASMTYPE.C */
EXTERN RETCODE Atype_Xs1_disp     (PCTYPE, OPERANDSTRUCT*,  U8*, int*);
EXTERN RETCODE Atype_X_disp       (PCTYPE, OPERANDSTRUCT*,  U8*, int*);
EXTERN RETCODE Atype_X_ldisp_hdisp(PCTYPE, OPERANDSTRUCT*,  U8*, int*);

/* defined in WUSYM1.C */
EXTERN SymToAddr();

/********************** External Variable Declaration **********************/

/* defined in ASMTBL.C */
EXTERN ASMSTRUCT AsmTable[];
EXTERN int AsmTableLen;

/* defined in OLDGBL.H */
EXTERN int sym_flag;

/* defined in USYM2.H */
EXTERN struct common_symbol comsym;

/************************ Local Function Prototype *************************/

PRIVATE int     IdentifyMnemonic(STR Mnemonic);
PRIVATE RETCODE IdentifyFormat(int MnemonicID, OPERANDSTRUCT *Operands,
                               TYPEROUTINE *Type);
PRIVATE RETCODE GetType(int MnemonicID, U8 *Format,  TYPEROUTINE *Type);
PRIVATE RETCODE GetCode(int MnemonicID, TYPEROUTINE Type,  U8 *Codes);

PRIVATE         GetToken();
PRIVATE RETCODE PseudoDS(int *Length);
PRIVATE RETCODE PseudoDBDW(int Size,  U8 *Codes, int *Length);
PRIVATE RETCODE GetOperands(OPERANDSTRUCT *OperandBuffer);
PRIVATE RETCODE ParseST(OPERANDSTRUCT *Operand);
PRIVATE RETCODE ParseData(STR Prefix,  OPERANDSTRUCT *Operand);
PRIVATE RETCODE ParseNumber(STR Prefix,  long *Number);
PRIVATE RETCODE ParseMemory(OPERANDSTRUCT *Operand);
PRIVATE int     SearchTable(STR Table[], int TableLen, STR Key);
PRIVATE BOOLEAN AreXdigits(STR s);

/************************ Local Variable Definition ************************/

LOCAL char MsgDefTooLarge[50];   /* filled in Assemble() */
LOCAL STR ErrorMsgTable[] = {
   "(OK)",
   "Mnemonic error!",
   "Syntax error!",
   "Format error!",
   "Type error!",
   "Undefined symbol!",
   "Invalid number!",
   "Number is out of range!",
   "Jump out of range!",
   MsgDefTooLarge,
   "(DS)"
};

LOCAL STR SegTable[4] = {
    "ES", "CS", "SS", "DS"
};
LOCAL STR Reg8Table[8] = {
    "AL", "CL", "DL", "BL", "AH", "CH", "DH", "BH"
};
LOCAL STR Reg16Table[8] = {
    "AX", "CX", "DX", "BX", "SP", "BP", "SI", "DI"
};
/* The order in PtrTable must match order in enum OPERAND_TYPE (ASMDEF.H) */
LOCAL STR PtrTable[] = {
    "BYTE", "WORD", "DWORD", "QWORD", "TBYTE", "NEAR", "FAR"
};
LOCAL STR BaseIndexRegTable[4] = {
    "DI", "SI", "BP", "BX"
};
/* use BX-BP-SI-DI 4 bits as index to get r/m bits */
LOCAL U8 BaseIndexTable[16] = {
    6, 5, 4, -1,   6, 3, 2, -1,   7, 1, 0, -1,   -1, -1, -1, -1
};

LOCAL PCTYPE asmIP;
LOCAL U16    MaxDefSize;   /* max size when processing DB and DW */
LOCAL STR    LookAhead;
LOCAL char   Mnemonic[80], Token[80];
LOCAL U8     ExtraCodes[5], ExtraLen;


                           /*********************
                            *  Executable Code  *
                            *********************/


/***************************************************************************
**
**  Name: Assemble(CSIP, Instruction,  Codes, Length)
**
**  Description:
**     assemble one line of instruction into object codes
**
**  Input:  address, instruction
**
**  Output: code data, code length
**
**  Return: error code defined in ASM.H
**
****************************************************************************/
PUBLIC RETCODE Assemble(PCTYPE CSIP, STR Instruction, U8 *Codes, int *Length)
{
int           MnemonicID, SegID;
OPERANDSTRUCT Operands[MAXOPERAND];
TYPEROUTINE   Type;
RETCODE       Result;

   if ( (MaxDefSize = *Length) == 0 )  MaxDefSize = 0xFFFF;
   sprintf(MsgDefTooLarge,"Max definition size is %04X bytes!", MaxDefSize);
   asmIP = CSIP & 0x0000FFFFL;
   *Length = 0;

   LookAhead = Instruction;
   GetToken();
   strcpy(Mnemonic, Token);

/*** Process pseudo instructions ***/
   if (stricmp(Mnemonic,"DS") == 0 && *LookAhead != ':')
      return (PseudoDS(Length));
   if (stricmp(Mnemonic,"DB") == 0)
      return (PseudoDBDW(1,  Codes, Length));
   if (stricmp(Mnemonic,"DW") == 0)
      return (PseudoDBDW(2,  Codes, Length));

/*** Process prefix in instruction ***/
   ExtraLen = 0;
   if (stricmp(Mnemonic,"REP") == 0 || stricmp(Mnemonic,"REPE") == 0 ||
       stricmp(Mnemonic,"REPZ") == 0)
      ExtraCodes[ExtraLen++] = 0xF3;
   else if (stricmp(Mnemonic,"REPNE") == 0 || stricmp(Mnemonic,"REPNZ") == 0)
      ExtraCodes[ExtraLen++] = 0xF2;
   else if ((SegID=SearchTable(SegTable, COUNT(SegTable), Mnemonic)) != -1) {
      GetToken();          /* read ':' */
      if (*Token != ':')   /* ES: CS: SS: DS: */
         return (ASM_SYNTAXERR);
      ExtraCodes[ExtraLen++] = 0x26 + (SegID << 3);
   }
   if (ExtraLen > 0) {                       /* has prefix */
      memcpy(Codes, ExtraCodes, ExtraLen);   /* preset Codes and Length to */
      *Length = ExtraLen;                    /*   avoid having prefix only */
      GetToken();
      strcpy(Mnemonic, Token);
   }

/*** Check instruction format and obtain op-codes ***/
   if (*Mnemonic == '\0')   /* null instruction */
      return (ASM_OK);
   if ((MnemonicID = IdentifyMnemonic(Mnemonic)) == -1)
      return (ASM_MNEMONICERR);
   if ((Result = GetOperands(Operands)) != ASM_OK)
      return (Result);
   if ((Result = IdentifyFormat(MnemonicID, Operands,  &Type)) != ASM_OK)
      return (Result);
   if ((Result = GetCode(MnemonicID, Type,  Codes)) != ASM_OK)
      return (Result);

/*** Call type processing routine to generate object codes ***/
   if (Type == Atype_Xs1_disp || Type == Atype_X_disp ||
       Type == Atype_X_ldisp_hdisp)
      Result = (*Type)(asmIP, Operands,  Codes, Length);
   else
      Result = (*Type)(Operands,  Codes, Length);

   if (ExtraLen > 0) {
      memmove(Codes+ExtraLen, Codes, *Length);
      memcpy(Codes, ExtraCodes, ExtraLen);
      *Length += ExtraLen;
   }
   return (Result);
}  /* end of Assemble() */

/***************************************************************************
**
**  Name: IdentifyMnemonic(Mnemonic)
**
**  Description:
**     identify if mnemonic symbol is correct
**
**  Input:  mnemonic symbol
**
**  Return: index of matched record in assemble table if success,-1 otherwise
**
****************************************************************************/
PRIVATE int IdentifyMnemonic(STR Mnemonic)
{
int MnemonicCmp(STR, ASMSTRUCT*);
int (*fcmp)();
STR ptr;

   fcmp = MnemonicCmp;
   ptr = bsearch(Mnemonic, AsmTable, AsmTableLen, sizeof(AsmTable[0]), fcmp);
   if (ptr == NULL)  return (-1);
   return ( ((int)ptr-(int)AsmTable) / sizeof(AsmTable[0]) );
}  /* end of IdentifyMnemonic() */

/***************************************************************************/
PRIVATE int MnemonicCmp(STR Mnemonic, ASMSTRUCT *AsmRecord)
{
   return (stricmp(Mnemonic, AsmRecord->Mnemonic));
}

/***************************************************************************
**
**  Name: IdentifyFormat(MnemonicID, Operands, Type)
**
**  Description:
**     identify if there exists suitable format of operands
**
**  Input:  ID of mnemonic symbol, buffer of operands information
**
**  Output: code-type routine pointer
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE IdentifyFormat(int MnemonicID, OPERANDSTRUCT *Operands,
                               TYPEROUTINE *Type)
{
U8 AllFormats[MAXOPERAND][4];
int FormatCounts[MAXOPERAND];
U8 OperandType, MemberID, Format[MAXOPERAND];
int i, j, k;

/*** Find possible operand types of operand ***/
   for (i = 0; i < MAXOPERAND; i++) {
      FormatCounts[i] = 0;
      OperandType = Operands[i].OperandType;
      MemberID    = Operands[i].MemberID;
   /* check special operand */
      if (OperandType == REG8) {
         if (MemberID == 0)
            AllFormats[i][FormatCounts[i]++] = AL;
         else if (MemberID == 1)
            AllFormats[i][FormatCounts[i]++] = CL;
      }
      else if (OperandType == REG16) {
         if (MemberID == 0)
            AllFormats[i][FormatCounts[i]++] = AX;
         else if (MemberID == 2)
            AllFormats[i][FormatCounts[i]++] = DX;
      }
      else if (MemberID == 6) {   /* direct address */
         if (OperandType == MEM)
            AllFormats[i][FormatCounts[i]++] = MEMDIRECT;
         else if (OperandType == MEMBYTE)
            AllFormats[i][FormatCounts[i]++] = MEMDIRECTBYTE;
         else if (OperandType == MEMWORD)
            AllFormats[i][FormatCounts[i]++] = MEMDIRECTWORD;
      }
      else if (OperandType == DATA) {
         if (Operands[i].Number == 1)
            AllFormats[i][FormatCounts[i]++] = DATAONE;
         if (Operands[i].Number > -0x100L && Operands[i].Number < 0x100L)
            AllFormats[i][FormatCounts[i]++] = DATA8;
      }
      AllFormats[i][FormatCounts[i]++] = OperandType;
   }  /* end of for */

/*** try all possible combinations of operand formats ***/
   for (i = 0; i < FormatCounts[0]; i++) {
      Format[0] = AllFormats[0][i];
      for (j = 0; j < FormatCounts[1]; j++) {
         Format[1] = AllFormats[1][j];
         for (k = 0; k < FormatCounts[2]; k++) {
            Format[2] = AllFormats[2][k];
            if (GetType(MnemonicID, Format,  Type) == ASM_OK) {
               Operands[0].OperandType = Format[0];
               Operands[1].OperandType = Format[1];
               Operands[2].OperandType = Format[2];
               return (ASM_OK);
            }
         }
      }
   }
   return (ASM_FORMATERR);
}  /* end of IdentifyFormat() */

/***************************************************************************
**
**  Name: GetType(MnemonicID, Format,  Type)
**
**  Description:
**     search matched format in format table to obtain code type
**
**  Input:  ID of mnemonic symbol, format of operand
**
**  Output: code-type routine
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE GetType(int MnemonicID, U8 *Format,  TYPEROUTINE *Type)
{
FORMATSTRUCT *FormatTable;
int TotalFormat, i;

   FormatTable = AsmTable[MnemonicID].FormatTable;
   TotalFormat = AsmTable[MnemonicID].TotalFormat;
   for (i = 0; i < TotalFormat; i++)
      if (memcmp(Format, FormatTable[i].Format, MAXOPERAND) == 0) {
         *Type = FormatTable[i].Type;
         return (ASM_OK);
      }
   return (ASM_FORMATERR);
}  /* end of GetType() */

/***************************************************************************
**
**  Name: GetCode(MnemonicID, Type,  Codes)
**
**  Description:
**     search matched type in code table to obtain op-codes
**
**  Input:  ID of mnemonic symbol, code-type routine
**
**  Output: op-codes
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE GetCode(int MnemonicID, TYPEROUTINE Type,  U8 *Codes)
{
CODESTRUCT *CodeTable;
int TotalCode, i;

   CodeTable = AsmTable[MnemonicID].CodeTable;
   TotalCode = AsmTable[MnemonicID].TotalCode;
   for (i = 0; i < TotalCode; i++)
      if (Type == CodeTable[i].Type) {
         memcpy(Codes, CodeTable[i].Codes, MAXOPCODE);
         return (ASM_OK);
      }
   return (ASM_TYPEERR);
}  /* end of GetCode() */

/***************************************************************************
**
**  Name: asmErrorMsg(ErrorCode)
**
**  Description:
**     error message of assembler
**
**  Input:  error code
**
**  Return: error string
**
****************************************************************************/
PUBLIC STR asmErrorMsg(int ErrorCode)
{
   return (ErrorMsgTable[ErrorCode]);
}  /* end of asmErrorMsg() */


/***************************************************************************
**                       Procedures of parsing operands
****************************************************************************/

/***************************************************************************
**
**  Name: GetToken()
**
**  Description:
**     get an token from string variable LookAhead
**
**  Notes:  LookAhead always points to next available token
**
****************************************************************************/
PRIVATE GetToken()
{
STR TokenPtr;

   TokenPtr = Token;

/*** Skip leading white spaces ***/
   while (isspace(*LookAhead))
      LookAhead++;

/*** Extract token ***/
   if (isalnum(*LookAhead) || *LookAhead == '_')
      while (isalnum(*LookAhead) || *LookAhead == '_')
         if (TokenPtr-Token >= sizeof(Token)-1)  LookAhead++;
         else  *(TokenPtr++) = *(LookAhead++);
   else if (!ENDOFSTR(*LookAhead))
      *(TokenPtr++) = *(LookAhead++);
   *TokenPtr = '\0';

/*** Skip tailing white spaces ***/
   while (!ENDOFSTR(*LookAhead) && isspace(*LookAhead))  // skip to next char
      LookAhead++;
}  /* end of GetToken() */

/***************************************************************************
**
**  Name: PseudoDS(Length)
**
**  Description:
**     process the pseudo instruction: DS (define storage)
**
**  Output: storage size
**
**  Return: error code
**
*****************************************************************************/
PRIVATE RETCODE PseudoDS(int *Length)
{
long Number;
RETCODE Result;

   if (ENDOFSTR(*LookAhead))  return (ASM_OK);
   GetToken();
   if (*Token != '+' && !isxdigit(*Token))  return (ASM_SYNTAXERR);
   if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
      return (Result);
   *Length = Number;
   return (ASM_DEFSTORAGE);
}  /* end of PseudoDS() */

/***************************************************************************
**
**  Name: PseudoDBDW(Size,  Codes, Length)
**
**  Description:
**     process the pseudo instruction: DB, DW
**
**  Input:  size = 1 for DB, 2 for DW
**
**  Output: codes, code length
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE PseudoDBDW(int Size,  U8 *Codes, int *Length)
{
char Quote;
long Number;
int  SaveLen, DupSize, i;
RETCODE Result;
BOOLEAN HasData = FALSE;

   while (TRUE) {
      if (ENDOFSTR(*LookAhead)) break;
      if (*LookAhead == ')')   /* end of DUP ? */
         if (HasData)  break;
         else  return (ASM_SYNTAXERR);

   /*** Define string ***/
      if (*LookAhead == '"' || *LookAhead == '\'') {
         Quote = *(LookAhead++);
         if (*LookAhead == Quote)  return (ASM_SYNTAXERR);
         while (*LookAhead != Quote) {
            if (*LookAhead != COMMENT_CHAR && ENDOFSTR(*LookAhead))
               return (ASM_SYNTAXERR);
            if ((U16)(*Length) >= MaxDefSize)  return (ASM_DEFTOOLARGE);
            Codes[(*Length)++] = *(LookAhead++);
         }
         GetToken();   /* read ' or " */
      }

   /*** Define data ***/
      else {
         GetToken();
         if (*Token != '+' && *Token != '-' && !isxdigit(*Token))
            return (ASM_SYNTAXERR);
         if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
            return (Result);

         if (strnicmp(LookAhead, "DUP", 3) == 0) {
            GetToken();
            if (stricmp(Token, "DUP") != 0)  return (ASM_SYNTAXERR);
            if (Number <= 0)  return (ASM_SYNTAXERR);
            GetToken();
            if (*Token != '(')  return (ASM_SYNTAXERR);
            SaveLen = *Length;
            if ((Result=PseudoDBDW(Size, Codes, Length)) != ASM_OK)
               return (Result);
            GetToken();
            if (*Token != ')')  return (ASM_SYNTAXERR);
            DupSize = *Length - SaveLen;
            /* check definition size */
            if (((U32)(*Length) + (Number-1) * (U32)DupSize) > MaxDefSize)
               return (ASM_DEFTOOLARGE);
            for (i = 1; i < Number; i++)
               memcpy(&Codes[SaveLen + DupSize * i],
                      &Codes[SaveLen], DupSize);
            *Length += (Number - 1) * DupSize;
         }
         else if (Size == 2) {   /* DW */
            if ((U16)(*Length) >= MaxDefSize-1)  return (ASM_DEFTOOLARGE);
            Codes[(*Length)++] = Number & 0x00FF;
            Codes[(*Length)++] = (Number & 0xFF00) >> 8;
         }
         else if (Size == 1) {   /* DB */
            if (Number <= -0x100 || Number >= 0x100)  return(ASM_NUMOUTRANGE);
            if ((U16)(*Length) >= MaxDefSize)  return (ASM_DEFTOOLARGE);
            Codes[(*Length)++] = Number & 0x00FF;
         }
      }
      HasData = TRUE;
      if (*LookAhead == ',')  GetToken();
   }  /* end of while */
   return (ASM_OK);
}  /* end of PseudoDBDW() */

/***************************************************************************
**
**  Name: GetOperands(OperandBuffer)
**
**  Description:
**     parse the operands
**
**  Output: operand information
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE GetOperands(OPERANDSTRUCT *OperandBuffer)
{
OPERANDSTRUCT *Operand;
RETCODE Result;
long Number;
int i, ID;

   for (i = 0; i < MAXOPERAND; i++)
      OperandBuffer[i].OperandType = NONE;
   if (ENDOFSTR(*LookAhead))  return (ASM_OK);

   for (i = 0; i < MAXOPERAND; i++) {
      Operand = &OperandBuffer[i];
      Operand->Number = 0;
      GetToken();

   /*** Check segment operand ***/
      if (*LookAhead != ':' && (Operand->MemberID =
           SearchTable(SegTable, COUNT(SegTable), Token)) != -1)
         Operand->OperandType = SEG;

   /*** Check register operand ***/
      else if ((Operand->MemberID =
                SearchTable(Reg8Table, COUNT(Reg8Table), Token)) != -1)
         Operand->OperandType = REG8;
      else if ((Operand->MemberID =
                SearchTable(Reg16Table, COUNT(Reg16Table), Token)) != -1)
         Operand->OperandType = REG16;

   /*** Check ST or ST(i) operand ***/
      else if (stricmp(Token, "ST") == 0) {
         if ((Result = ParseST(Operand)) != ASM_OK)
            return (Result);
      }

   /*** Check data operand ***/
      else if (*Token == '+' || *Token == '-' ||
               *Token == '%' || isdigit(*Token) || AreXdigits(Token)) {
         if ((Result = ParseData(Token,  Operand)) != ASM_OK)
            return (Result);
      }

   /*** Check character operand ***/
      else if (*Token == '\'') {
         if (*LookAhead != COMMENT_CHAR && ENDOFSTR(*LookAhead))
            return (ASM_SYNTAXERR);
         Operand->Number = (long) ((unsigned char) *(LookAhead++));
         Operand->OperandType = DATA;
         if (*LookAhead != '\'')  return (ASM_SYNTAXERR);
         GetToken();   /* read ' */
      }

   /*** Check $ operand ***/
      else if (*Token == '$') {
         Operand->Number = (long)(int)asmIP;
         Operand->OperandType = DATA;
         if (!ENDOFSTR(*LookAhead)) {
            GetToken();
            if (*Token != '+' && *Token != '-')  return (ASM_SYNTAXERR);
            if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
               return (Result);
            Operand->Number = (long)(int)((Operand->Number + Number)
                                          & 0xFFFFL);
         }
      }

   /*** Check memory operand ***/
      else {
         /* check xxx PTR */
         if ((ID = SearchTable(PtrTable, COUNT(PtrTable), Token)) != -1) {
            Operand->OperandType = MEMBYTE + ID;
            GetToken();
            if (stricmp(Token, "PTR") == 0)  GetToken();
            if (ENDOFSTR(*Token))  return (ASM_SYNTAXERR);
         }
         /* check segment override */
         if ((ID = SearchTable(SegTable, COUNT(SegTable), Token)) != -1) {
            GetToken();                            /* read ':' */
            if (*Token != ':')                     /* ES: CS: SS: DS: */
               return (ASM_SYNTAXERR);
            ExtraCodes[ExtraLen++] = 0x26 + (ID << 3);
            GetToken();
         }
         if (*Token != '[')  return (ASM_SYNTAXERR);
         if ((Result = ParseMemory(Operand)) != ASM_OK)   /* [...] */
            return (Result);
      }

      if (ENDOFSTR(*LookAhead))  break;
      if (*LookAhead == ',')  GetToken();
      else  return (ASM_SYNTAXERR);
   }  /* end of for */
   return (ASM_OK);
}  /* end of GetOperands() */

/***************************************************************************
**
**  Name: ParseST(Operand)
**
**  Description:
**     parse ST or ST(i)
**
**  Output: binary number
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE ParseST(OPERANDSTRUCT *Operand)
{
RETCODE Result;
long Number;

   Operand->OperandType = ST;
   if (*LookAhead == '(') {
      GetToken();   /* read '(' */
      GetToken();
      if (ENDOFSTR(*Token))  return (ASM_SYNTAXERR);
      if (*Token != '+' && *Token != '-' && !isxdigit(*Token))
         return (ASM_SYNTAXERR);
      if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
         return (Result);
      /* Only ST(0) ... ST(7) are allowed */
      if (Number < 0 || Number > 7)  return (ASM_NUMOUTRANGE);
      Operand->OperandType = STI;
      Operand->MemberID = (U8)Number;
      GetToken();
      if (*Token != ')')  return (ASM_SYNTAXERR);
   }
   return (ASM_OK);
}  /* end of ParseST() */

/***************************************************************************
**
**  Name: ParseData(Prefix,  Operand)
**
**  Description:
**     parse data: xxx, +xxx, -xxx, xxxx:xxxx, %symbol, +%symbol, -%symbol
**
**  Input: string -- "+", "-", "%", or hex-number
**
**  Output: operand information
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE ParseData(STR Prefix,  OPERANDSTRUCT *Operand)
{
RETCODE Result;
BOOLEAN IsSymbol;
long Number;

   IsSymbol = (*Prefix == '%' || *LookAhead == '%');
   if ((Result = ParseNumber(Prefix,  &Number)) != ASM_OK)
      return (Result);

/*** Check offset[...] or %symbol[...] ***/
   if (*LookAhead == '[') {
      GetToken();   /* read '[' */
      Operand->Number = (long)(int)(Number & 0x0000FFFF);
      return (ParseMemory(Operand));
   }
/*** Check segment:offset ***/
   else if (*LookAhead == ':') {
      GetToken();   /* read ':' */
      Operand->Number = Number & 0x0000FFFF;
      Operand->OperandType = SEGOFFSET;
      GetToken();
      if (ENDOFSTR(*Token))  return (ASM_SYNTAXERR);
      if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
         return (Result);
      Operand->Number = ((Operand->Number)<<16) + (Number & 0x0000FFFF);
   }
/*** Check JMP/CALL %symbol ***/
   else if (IsSymbol &&
            (stricmp(Mnemonic, "JMP") == 0 ||
             stricmp(Mnemonic, "CALL") == 0)) {
      Operand->OperandType = SEGOFFSET;
      Operand->Number = Number;
   }
/*** Until now, it is actually immediate data operand ***/
   else {
      Operand->OperandType = DATA;
      Operand->Number = (long)(int)(Number & 0x0000FFFF);
   }
   return (ASM_OK);
}  /* end of ParseData() */

/***************************************************************************
**
**  Name: ParseNumber(Prefix,  Number)
**
**  Description:
**     parse number: xxx, +xxx, -xxx, %symbol, +%symbol, -%symbol
**
**  Input: string -- "+", "-", "%", or hex-num
**
**  Output: binary number
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE ParseNumber(STR Prefix,  long *Number)
{
char *Ptr;
char Sign, Xdigit;
long Value;

   Sign = *Prefix;
   if (Sign == '+' || Sign == '-')  GetToken();
   else  strcpy(Token, Prefix);
   *Number = 0;
   Ptr = Token;

   if (*Ptr == '%') {   /* symbol */
      GetToken();
      if (ENDOFSTR(*Token))  return (ASM_SYNTAXERR);
      if (asmSym2Addr(Token,  Number) == FALSE)  return (ASM_UNDEFSYMBOL);
   }
   else                 /* hex number */
      if (!ENDOFSTR(*Ptr))
         while (*Ptr) {
            Xdigit = toupper(*Ptr);
            Ptr++;
            if (!isxdigit(Xdigit))  return (ASM_INVALIDNUM);
            if (isdigit(Xdigit))
               Value = Xdigit - '0';
            else
               Value = 10 + Xdigit - 'A';
            *Number = (*Number << 4) + Value;
            if (*Number >= 0x10000L)  return (ASM_NUMOUTRANGE);
         }
   else  return (ASM_SYNTAXERR);
   if (Sign == '-')  *Number = -(*Number);
   return (ASM_OK);
}  /* end of ParseNumber() */

/***************************************************************************
**
**  Name: ParseMemory(Operand)
**
**  Description:
**     parse memory operand: [BX+SI],[BX+DI], ...
**
**  Output: operand information
**
**  Return: error code
**
****************************************************************************/
PRIVATE RETCODE ParseMemory(OPERANDSTRUCT *Operand)
{
char Sign;
U8 BaseIndex, mod;  /* BaseIndex -- bit 3:BX, bit 2:BP, bit 1:SI, bit 0:DI */
long Number;
RETCODE Result;
int ID;

   if (Operand->OperandType == NONE)   /* not set xxx PTR yet */
      Operand->OperandType = MEM;
   BaseIndex = 0;
   GetToken();

/*** Parsing [...][...]... ***/
   while (1) {
      if (AreXdigits(Token) || *Token == '%' ||
          (*Token == '+' || *Token == '-') &&
          (AreXdigits(LookAhead) || *LookAhead == '%')) {
      /* offset, +offset, -offset, %symbol, +%symbol, -%symbol */
         if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
            return (Result);
         Operand->Number = (long)(int)((Operand->Number + Number) & 0xFFFFL);
      }
      else {
      /* BX/BP/SI/DI */
         if (*Token == '+')  GetToken();
         if ((ID = SearchTable(BaseIndexRegTable,
                               COUNT(BaseIndexRegTable), Token)) != -1)
            if (BaseIndex & (1 << ID))   /* already set ? */
               return (ASM_SYNTAXERR);
            else
               BaseIndex |= (1 << ID);
         else
            return (ASM_SYNTAXERR);
      }
      GetToken();
      if (*Token == ']')
         if (*LookAhead == '[') {
            GetToken();   /* read '[' */
            GetToken();
         }
         else break;
   }  /* end of while */

/*** Check whether offset is after ']' ***/
   if (!ENDOFSTR(*LookAhead) && *LookAhead != ',') {
      GetToken();
      if ((Result = ParseNumber(Token,  &Number)) != ASM_OK)
         return (Result);
      Operand->Number += (long)(int)(Number & 0x0000FFFFL);
   }

/*** Determine mod and r/m bits, which are stored in Operand->MemberID ***/
   if ((Operand->MemberID = BaseIndexTable[BaseIndex]) == -1)  /* r/m bits */
      return (ASM_SYNTAXERR);
   if (BaseIndex == 0)  mod = 0;           /* direct address */
   else
      if (Operand->Number == 0L)           /* really no offset ? */
         if (BaseIndex == 0x04)  mod = 1;  /* is [BP]; set one offset byte */
         else  mod = 0;                    /* no offset */
      else
         if (Operand->Number >= -0x80 && Operand->Number <= 0x7F)
            mod = 1;                       /* 1 offset byte, sign extended */
         else
            mod = 2;                       /* 2 offset bytes */
   Operand->MemberID |= (mod << 6);        /* set mod bits */
   return (ASM_OK);
}  /* end of ParseMemory() */

/***************************************************************************
**
**  Name: SearchTable(Table, TableLen, Key)
**
**  Description:
**     search a string in a table
**
**  Input:  table, table length, key string
**
**  Return: index in table if success, -1 otherwise
**
****************************************************************************/
PRIVATE int SearchTable(STR Table[], int TableLen, STR Key)
{
int i;

   for (i = 0; i < TableLen; i++)
      if (stricmp(Table[i], Key) == 0)  return (i);
   return (-1);
}  /* end of SearchTable() */

/***************************************************************************
**
**  Name: AreXdigits(s)
**
**  Description:
**     check if characters in a string are all hex-digits
**
**  Input:  string
**
**  Return: TRUE or FALSE
**
****************************************************************************/
PRIVATE BOOLEAN AreXdigits(STR s)
{
   if ( !isxdigit(*(s++)) )  return (FALSE);
   while (isalnum(*s) || *s == '_')
      if ( !isxdigit(*(s++)) )  return (FALSE);
   return (TRUE);
}  /* end of AreXdigits() */

/****************************** End of File ********************************/
