/***************************************************************************
**
**  File name : ASMTYPE.C
**
**  Description : type processing routines of assembler
**
**  Date : 11/13/1992  by C. H. Lin
**
**
**  Copyright (C) 1993 Microtek International Inc.
**  All Rights Reserved
**
****************************************************************************/


/****************************** Include File *******************************/

#include "asmdef.h"

/**************************** Macro Definition *****************************/

#define FIRSTBYTE(x)  ((x) & 0x00FF)
#define SECONDBYTE(x) ((x) & 0xFF00) >> 8
#define SIGNEXTEND(x) ((x) >= -0x80 && (x) <= 0x7F)

/************************ Local Function Prototype *************************/

PRIVATE int SetModrm (OPERANDSTRUCT Operand,  U8 *Codes);
PRIVATE int SetWmodrm(OPERANDSTRUCT Operand,  U8 *Codes);


                           /*********************
                            *  Executable Code  *
                            *********************/


/***************************************************************************
**
**  Name: Atype_*(Operands,  Codes, Len)
**
**  Description:
**     generate object codes of instruction for its type
**
**  Input:  buffer of operands information
**
**  Output: object codes, instruction length
**
**  Return: error code
**
****************************************************************************/

/***************************************************************************
**  no operands
**  obj codes: | ** *** ***|
****************************************************************************/
PUBLIC RETCODE Atype_X(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   if (Codes[1] == 0x0A && (Codes[0] == 0xD4 || Codes[0] == 0xD5))
      /* AAM, AAD */
      *Len = 2;
   else
      *Len = 1;
   return (ASM_OK);
}  /* end of Atype_X() */

/***************************************************************************
**  PUSH/POP seg
**  obj codes: | ** 0sg ***|
****************************************************************************/
PUBLIC RETCODE Atype_X0sgX(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   Codes[0] |= Operands[0].MemberID << 3;   /* set seg field */
   *Len = 1;
   return (ASM_OK);
}  /* end of Atype_X0sgX() */

/***************************************************************************
**  shift/rotate reg/mem,1/CL
**  obj codes: | ** *** *cw|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_Xcw_modXrm(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   if (Operands[1].OperandType == CL)
      Codes[0] |= 0x02;
   *Len = SetWmodrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_Xcw_modXrm() */

/***************************************************************************
**  MOV seg,reg/mem
**  MOV reg/mem,seg
**  obj codes: | ** *** *d0|mod 0sg r/m|
****************************************************************************/
PUBLIC RETCODE Atype_Xd0_mod0sgrm(OPERANDSTRUCT *Operands,U8 *Codes,int *Len)
{
   if (Operands[0].OperandType == SEG) {       /* MOV seg,reg/mem */
      Codes[0] |= 0x02;
      Codes[1] |= Operands[0].MemberID << 3;   /* set seg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {                                      /* MOV reg/mem,seg */
      Codes[1] |= Operands[1].MemberID << 3;   /* set seg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (ASM_OK);
}  /* end of Atype_Xd0_mod0sgrm() */

/***************************************************************************
**  OUT DX,acc
**  IN  acc,DX
**  obj codes: | ** *** *dw|
****************************************************************************/
PUBLIC RETCODE Atype_Xdw(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == DX) {   /* OUT DX,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
   }
   else                                   /* IN  acc,DX */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
   *Len = 1;
   return (ASM_OK);
}  /* end of Atype_Xdw() */

/***************************************************************************
**  MOV memdirect,acc
**  MOV acc,memdirect
**  obj codes: | ** *** *dw| addr_low  | addr_high |
****************************************************************************/
PUBLIC RETCODE Atype_Xdw_laddr_haddr(OPERANDSTRUCT *Operands,
                                     U8 *Codes, int *Len)
{
   if (Operands[1].OperandType == AL || Operands[1].OperandType == AX) {
   /* MOV memdirect,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[0].Number);
      Codes[2] = SECONDBYTE(Operands[0].Number);
   }
   else {
   /* MOV acc,memdirect */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[1].Number);
      Codes[2] = SECONDBYTE(Operands[1].Number);
   }
   *Len = 3;
   return (ASM_OK);
}  /* end of Atype_Xdw_laddr_haddr() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/MOV
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** *dw|mod reg r/m|
****************************************************************************/
PUBLIC RETCODE Atype_Xdw_modregrm(OPERANDSTRUCT *Operands,U8 *Codes,int *Len)
{
   if (Operands[1].OperandType != REG8 && Operands[1].OperandType != REG16) {
   /* reg,reg/mem */
      Codes[0] |= 0x02;
      if (Operands[0].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {
   /* reg/mem,reg */
      if (Operands[1].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[1].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (ASM_OK);
}  /* end of Atype_Xdw_modregrm() */

/***************************************************************************
**  OUT port,acc
**  IN  acc,port
**  obj codes: | ** *** *dw|   port    |
****************************************************************************/
PUBLIC RETCODE Atype_Xdw_port(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == DATA8) {   /* OUT port,acc */
      Codes[0] |= 0x02;
      if (Operands[1].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[0].Number);
   }
   else {                                    /* IN acc,port */
      if (Operands[0].OperandType == AX)
         Codes[0] |= 0x01;
      Codes[1] = FIRSTBYTE(Operands[1].Number);
   }
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_Xdw_port() */

/***************************************************************************
**  INC/DEC/PUSH/POP reg
**  XCHG reg,AX
**  obj codes: | ** *** reg|
****************************************************************************/
PUBLIC RETCODE Atype_Xreg(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   Codes[0] |= Operands[0].MemberID;
   *Len = 1;
   return (ASM_OK);
}  /* end of Atype_Xreg() */

/***************************************************************************
**  PUSH data
**  obj codes: | ** *** *s0|   data    |data if s=0|
****************************************************************************/
PUBLIC RETCODE Atype_Xs0_data(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   if (SIGNEXTEND(Operands[0].Number)) {
      Codes[0] |= 0x02;
      *Len = 2;
   }
   else {
      Codes[2] = SECONDBYTE(Operands[0].Number);
      *Len = 3;
   }
   return (ASM_OK);
}  /* end of Atype_Xs0_data() */

/***************************************************************************
**  JMP address
**  obj codes: | ** *** *s1|   disp    |
****************************************************************************/
PUBLIC RETCODE Atype_Xs1_disp(PCTYPE PC, OPERANDSTRUCT *Operands,
                       U8 *Codes, int *Len)
{
int Displacement;

   Displacement = Operands[0].Number - (PC+2);   /* assume short jump */
   if (SIGNEXTEND(Displacement)) {               /* short jump */
      Codes[0] |= 0x02;
      Codes[1] = FIRSTBYTE(Displacement);
      *Len = 2;
   }
   else {                                        /* near jump */
      Displacement--;
      Codes[1] = FIRSTBYTE(Displacement);
      Codes[2] = SECONDBYTE(Displacement);
      *Len = 3;
   }
   return (ASM_OK);
}  /* end of Atype_Xs1_disp() */

/***************************************************************************
**  IMUL reg,reg/mem,data
**  obj codes: | ** *** *s1|mod reg r/m|   data    |data if s=0|
****************************************************************************/
PUBLIC RETCODE Atype_Xs1_modregrm_data(OPERANDSTRUCT *Operands,
                                       U8 *Codes, int *Len)
{
   Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[1], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[2].Number);
   if (SIGNEXTEND(Operands[2].Number)) {
      Codes[0] |= 0x02;
      (*Len)++;
   }
   else {
      Codes[*Len + 1] = SECONDBYTE(Operands[2].Number);
      *Len += 2;
   }
   return (ASM_OK);
}  /* end of Atype_Xs1_modregrm_data() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR reg/mem,data
**  obj codes: | ** *** *sw|mod *** r/m|   data    |data if sw=01|
****************************************************************************/
PUBLIC RETCODE Atype_Xsw_modXrm_data(OPERANDSTRUCT *Operands,
                                     U8 *Codes, int *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   if (Operands[0].OperandType == REG16 ||
       Operands[0].OperandType == MEMWORD || Operands[0].OperandType == MEM)
      if (SIGNEXTEND(Operands[1].Number))
         Codes[0] |= 0x02;
      else {
         Codes[*Len] = SECONDBYTE(Operands[1].Number);
         (*Len)++;
      }
   return (ASM_OK);
}  /* end of Atype_Xsw_modXrm_data() */

/***************************************************************************
**  INT data
**  obj codes: | ** *** **v|   data    |
****************************************************************************/
PUBLIC RETCODE Atype_Xv_type(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   if (Operands[0].Number != 3) {
      Codes[0] |= 0x01;
      Codes[1] = Operands[0].Number;
      *Len = 2;
   }
   else
      *Len = 1;
   return (ASM_OK);
}  /* end of Atype_Xv_type() */

/***************************************************************************
**  MOV reg,data
**  obj codes: | ** **w reg|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Atype_Xwreg_data(OPERANDSTRUCT *Operands, U8 *Codes, int *Len)
{
   Codes[0] |= Operands[0].MemberID;
   Codes[1] = FIRSTBYTE(Operands[1].Number);
   if (Operands[0].OperandType == REG16) {
      Codes[0] |= 0x08;
      Codes[2] = SECONDBYTE(Operands[1].Number);
      *Len = 3;
   }
   else
      *Len = 2;
   return (ASM_OK);
}  /* end of Atype_Xwreg_data() */

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/TEST acc,data
**  obj codes: | ** *** **w|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Atype_Xw_data(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   Codes[1] = FIRSTBYTE(Operands[1].Number);
   if (Operands[0].OperandType == AX) {
      Codes[0] |= 0x01;
      Codes[2] = SECONDBYTE(Operands[1].Number);
      *Len = 3;
   }
   else
      *Len = 2;
   return (ASM_OK);
}  /* end of Atype_Xw_data() */

/***************************************************************************
**  TEST/XCHG
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** **w|mod reg r/m|
***************************************************************************/
PUBLIC RETCODE Atype_Xw_modregrm(OPERANDSTRUCT *Operands, U8 *Codes,int *Len)
{
   if (Operands[0].OperandType == REG8 || Operands[0].OperandType == REG16) {
   /* reg,reg/mem */
      if (Operands[0].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[1], Codes);
   }
   else {
   /* reg/mem,reg */
      if (Operands[1].OperandType == REG16)
         Codes[0] |= 0x01;
      Codes[1] |= Operands[1].MemberID << 3;   /* set reg field */
      *Len = SetModrm(Operands[0], Codes);
   }
   return (ASM_OK);
}  /* end of Atype_Xw_modregrm() */

/***************************************************************************
**  NOT/NEG/MUL/IMUL/DIV/IDIV/INC/DEC reg/mem
**  obj codes: | ** *** **w|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_Xw_modXrm(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_Xw_modXrm() */

/***************************************************************************
**  shift/rotate reg/mem,count
**  obj codes: | ** *** **w|mod *** r/m|   count   |
****************************************************************************/
PUBLIC RETCODE Atype_Xw_modXrm_count(OPERANDSTRUCT *Operands,
                                     U8 *Codes, int *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   return (ASM_OK);
}  /* end of Atype_Xw_modXrm_count() */

/***************************************************************************
**  MOV/TEST reg/mem,data
**  obj codes: | ** *** **w|mod *** r/m|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Atype_Xw_modXrm_data(OPERANDSTRUCT *Operands,
                                    U8 *Codes, int *Len)
{
   *Len = SetWmodrm(Operands[0], Codes);
   Codes[*Len] = FIRSTBYTE(Operands[1].Number);
   (*Len)++;
   if (Operands[0].OperandType == REG16 ||
       Operands[0].OperandType == MEMWORD || Operands[0].OperandType== MEM) {
      Codes[*Len] = SECONDBYTE(Operands[1].Number);
      (*Len)++;
   }
   return (ASM_OK);
}  /* end of Atype_Xw_modXrm_data() */

/***************************************************************************
**  conditional branch
**  obj codes: | ** *** ***|   disp    |
****************************************************************************/
PUBLIC RETCODE Atype_X_disp(PCTYPE PC, OPERANDSTRUCT *Operands,
                            U8 *Codes, int *Len)
{
int Displacement;

   Displacement = Operands[0].Number - (PC+2);
   if (SIGNEXTEND(Displacement)) {
      Codes[1] = FIRSTBYTE(Displacement);
      *Len = 2;
   }
   else
      return (ASM_JMPOUTRANGE);
   return (ASM_OK);
}  /* end of Atype_X_disp() */

/***************************************************************************
**  RET/RETF data
**  obj codes: | ** *** ***| data_low  | data_high |
****************************************************************************/
PUBLIC RETCODE Atype_X_ldata_hdata(OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   *Len = 3;
   return (ASM_OK);
}  /* end of Atype_X_ldata_hdata() */

/***************************************************************************
**  ENTER data,data
**  obj codes: | ** *** ***| data_low  | data_high |   data    |
****************************************************************************/
PUBLIC RETCODE Atype_X_ldata_hdata_data(OPERANDSTRUCT *Operands,
                                        U8 *Codes, int *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   Codes[3] = FIRSTBYTE(Operands[1].Number);
   *Len = 4;
   return (ASM_OK);
}  /* end of Atype_X_ldata_hdata_data() */

/***************************************************************************
**  CALL address
**  obj codes: | ** *** ***| disp_low  | disp_high |
****************************************************************************/
PUBLIC RETCODE Atype_X_ldisp_hdisp(PCTYPE PC, OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
int Displacement;

   Displacement = Operands[0].Number - (PC+3);
   Codes[1] = FIRSTBYTE(Displacement);
   Codes[2] = SECONDBYTE(Displacement);
   *Len = 3;
   return (ASM_OK);
}  /* end of Atype_X_ldisp_hdisp() */

/***************************************************************************
**  CALL/JMP segment:offset
**  obj codes: | ** *** ***|offset_low |offset_high|  seg_low  | seg_high  |
****************************************************************************/
PUBLIC RETCODE Atype_X_loff_hoff_lseg_hseg(OPERANDSTRUCT *Operands,
                                           U8 *Codes, int *Len)
{
   Codes[1] = FIRSTBYTE(Operands[0].Number);
   Codes[2] = SECONDBYTE(Operands[0].Number);
   Codes[3] = (Operands[0].Number & 0x00FF0000) >> 16;
   Codes[4] = (Operands[0].Number & 0xFF000000) >> 24;
   *Len = 5;
   return (ASM_OK);
}  /* end of Atype_X_loff_hoff_lseg_hseg() */

/***************************************************************************
**  BOUND/LEA/LES/LDS reg,mem
**  obj codes: | ** *** ***|mod reg r/m|
****************************************************************************/
PUBLIC RETCODE Atype_X_modregrm(OPERANDSTRUCT *Operands, U8 *Codes, int *Len)
{
   Codes[1] |= Operands[0].MemberID << 3;   /* set reg field */
   *Len = SetModrm(Operands[1], Codes);
   return (ASM_OK);
}  /* end of Atype_X_modregrm() */

/***************************************************************************
**  PUSH/POP/CALL/JMP reg/mem
**  obj codes: | ** *** ***|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_X_modXrm(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   *Len = SetModrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_X_modXrm() */

/***************************************************************************
**  CALL/JMP FAR mem
**  obj codes: | ** *** ***|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_X_modXrm_FAR(OPERANDSTRUCT *Operands,U8 *Codes,int *Len)
{
   *Len = SetModrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_X_modXrm_FAR() */

/***************************************************************************
**  ESC data,reg/mem
**  obj codes: | 11 011 ***|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCxxx_modXrm(OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
long Number;

   Number = Operands[0].Number;
   if (Number < 0 || Number > 0x3F)
      return (ASM_NUMOUTRANGE);
   Codes[0] |= (Number & 0x38) >> 3;      /* X  X  X  X  X b5 b4 b3 */
   Codes[1] |= (Number & 0x07) << 3;      /* X  X b2 b1 b0  X  X  X */
   *Len = SetModrm(Operands[1], Codes);
   return (ASM_OK);
}  /* end of Atype_ESCxxx_modXrm() */

/***************************************************************************
**  operands: ST,ST(i)  /  ST(i),ST
**  obj codes: | 11 011 d**| 11 *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCdxx_11xxxrm(OPERANDSTRUCT *Operands,
                                    U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= Operands[0].MemberID;
   }
   else
      Codes[1] |= Operands[1].MemberID;
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_ESCdxx_11xxxrm() */

/***************************************************************************
**  operands: ST,ST(i) / ST(i),ST
**  obj codes: | 11 011 d**| 11 **d r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCdxx_11xxdrm(OPERANDSTRUCT *Operands,
                                    U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= 0x08;
      Codes[1] |= Operands[0].MemberID;
   }
   else
      Codes[1] |= Operands[1].MemberID;
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_ESCdxx_11xxdrm() */

/***************************************************************************
**  operands: ST,ST(i) / ST(i),ST
**  obj codes: | 11 011 d**| 11 **s r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCdxx_11xxsrm(OPERANDSTRUCT *Operands,
                                    U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == STI) {
      Codes[0] |= 0x04;
      Codes[1] |= Operands[0].MemberID;
   }
   else {
      Codes[1] |= 0x08;
      Codes[1] |= Operands[1].MemberID;
   }
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_ESCdxx_11xxsrm() */

/***************************************************************************
**  operands: QWORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 q**|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCqxx_modXrm(OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == MEMQWORD)
      Codes[0] |= 0x04;
   *Len = SetModrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_ESCqxx_modXrm() */

/***************************************************************************
**  operands: WORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 w**|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCwxx_modXrm(OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == MEMWORD)
      Codes[0] |= 0x04;
   *Len = SetModrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_ESCwxx_modXrm() */

/***************************************************************************
**  operands: mem / TBYTE PTR mem
**  obj codes: | 11 011 *t*|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCxtx_modXrm(OPERANDSTRUCT *Operands,
                                   U8 *Codes, int *Len)
{
   if (Operands[0].OperandType == MEMTBYTE)
      Codes[0] |= 0x02;
   *Len = SetModrm(Operands[0], Codes);
   return (ASM_OK);
}  /* end of Atype_ESCxtx_modXrm() */

/***************************************************************************
**  operands: ST(i)
**  obj codes: | 11 011 ***| 11 0** r/m|
****************************************************************************/
PUBLIC RETCODE Atype_ESCxxx_110xxrm(OPERANDSTRUCT *Operands,
                                    U8 *Codes, int *Len)
{
   Codes[1] |= Operands[0].MemberID;
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_ESCxxx_110xxrm() */

/***************************************************************************
**  FPinst with no operands
**  obj codes: | 11 011 ***| ** *** ***|
****************************************************************************/
PUBLIC RETCODE Atype_ESCxxx_X(OPERANDSTRUCT *Operands,  U8 *Codes, int *Len)
{
   *Len = 2;
   return (ASM_OK);
}  /* end of Atype_ESCxxx_X() */


/***************************************************************************
**                Procedures used by type processing routines
****************************************************************************/

/***************************************************************************
**
**  Name: SetModrm(Operand,  Codes)
**
**  Description:
**     generate codes: xxxxxxxx mod_xxx_rm
**
**  Input:  operand information
**
**  Output: object codes
**
****************************************************************************/
PRIVATE int SetModrm(OPERANDSTRUCT Operand,  U8 *Codes)
{
U8 mod;
int Len;

   Len = 2;
   Codes[1] |= Operand.MemberID;
/*** register mode ***/
   if (Operand.OperandType == REG8 || Operand.OperandType == REG16)
      Codes[1] |= 0xC0;
/*** memory mode ***/
   else {
      mod = (Operand.MemberID & 0xC0) >> 6;
      if (mod == 2 || (mod == 0 && (Operand.MemberID & 0x07) == 6)) {
      /*  2 offset bytes  or  direct address  */
         Codes[2] = Operand.Number;
         Codes[3] = SECONDBYTE(Operand.Number);
         Len = 4;
      }
      else if (mod == 1) {
      /* 1 offset byte */
         Codes[2] = Operand.Number;
         Len = 3;
      }
   }
   return (Len);
}  /* end of SetModrm() */

/***************************************************************************
**
** Name: SetWmodrm(Operand,  Codes)
**
** Description:
**    generate codes: xxxxxxxw mod_xxx_rm
**
** Input:  operand information
**
** Output: object codes
**
****************************************************************************/
PRIVATE int SetWmodrm(OPERANDSTRUCT Operand,  U8 *Codes)
{
   if (Operand.OperandType == MEM || Operand.OperandType == MEMWORD ||
       Operand.OperandType == REG16)
      Codes[0] |= 0x01;   /* set word bit */
   return (SetModrm(Operand, Codes));
}  /* end of SetWmodrm() */

/****************************** End of File ********************************/
