/***************************************************************************
**
**  File name : DEQ.C
**
**  Description : module of dequeue
**
**  Date : 11/13/1992  by C. H. Lin
**    1. Initial version (include symbol function)
**
**  Date : 11/26/1992  by C. H. Lin
**    1. In Print(), remove checking SPACE pressed
**
**  Date : 11/30/1992  by C. H. Lin
**    1. In deqListFrame(), fill remaining values in FlagMask
**
**  Date : 12/28/1992  by C. H. Lin
**    1. In DisplayFrame(), add displaying trace bits
**    2. Add ByteToBitstr()
**    3. Modify CheckFrame(), set default frame if *StartFrame == -1
**
**  Date : 1/4/1993  by C. H. Lin
**    1. In CheckFlush(), add checking flush bit
**
**  Date : 1/5/1993  by C. H. Lin
**    1. Add deqSetOutbuf(char *Buffer)
**    2. Modify Print() to allow output to be placed in buffer
**    3. Rename Print() to Puts()
**
**  Date : 2/2/1993  by C. H. Lin
**    1. Add bufID into parameter list of function
**    2. In DisplayFrame(), GetInstruction(), and CheckFlush(),
**       consider case of 80x88
**
**  Date : 2/5/1993  by C. H. Lin
**    1. In CheckFlush(), check if branch address is matched
**       when address isn't continuous or flush bit = 1
**
**  Date : 2/10/1993  by C. H. Lin
**    1. Rename deqSetOutbuf() to SetPutsBuf()
**    2. Rename Puts() to PutsMore(); change PutsMore() to be global
**    3. Add "EXPORT" to function
**
**  Date : 2/11/1993  by C. H. Lin
**    1. Change SetPutsBuf(char *Buffer, int BufRows)
**       to     SetPutsBuf(char *Buffer, int BufRows, int InitialCount)
**    2. In PutsMore(), handle [SPACE] key
**
**  Date : 3/16/1993  by C. H. Lin
**    1. In PutsMore(), handle Ctrl-S
**    2. PutsMore() returns code of pressed key
**
**  Date : 3/22/1993  by C. H. Lin
**    1. Modify CheckFlush() to do much more correct judgement
**
**  Date : 4/27/1993  by C. H. Lin
**    1. Adjust List Source output format: instruction first, then frame data
**
**
**  Copyright (C) 1993 Microtek International Inc.
**  All Rights Reserved
**
****************************************************************************/


/****************************** Include File *******************************/

#include <stdio.h>
//#ifndef _SD_ABI_
//#include "sdabi.h"
//#endif

#ifndef _ABI_FUNC_
#include "abifunc.h"
#endif

#ifndef _SD_ABI_DEF_
#include "sdabidef.h"
#endif

#include "sdabiext.h"

#include "deqdef.h"

/*********************** External Function Prototype ***********************/

/* defined in DISPLAY.C */
EXTERN DisplayStr(unsigned char *);

/* defined in MOUSEDRV.C */
EXTERN void ChkMouseLBAndEsc(int *keyin);

/********************** External Variable Declaration **********************/

/* defined in DSM.C */
EXTERN BOOLEAN dsmAddrGiven;
EXTERN long    dsmAddr;

/* defined in LIB2.C */
EXTERN int cmdfile_end_flag;

/* define in OLDGBL.H */
EXTERN char cmdfile_flag;

/* defined in GBL.H */
EXTERN char CmdFileBuf[];

/* defined in SYNTAX.C */
EXTERN struct ev_attr ev_def;

/* defined in COMPLEX.C */
EXTERN QUALIFY_LIST quaList;

EXTERN int lastBuff;

/************************ Local Function Prototype *************************/

PRIVATE BOOLEAN CheckFrame  (int bufID, S32 StartFrame);
PRIVATE BOOLEAN SkipFrame   (int bufID, S32 *EndFrame);
PRIVATE int     DisplayFrame(int bufID, S32 Frame);
PRIVATE int     PutOneFrame (S32 Frame, U32 Addr, U16 Data, U8 Stat, U8 Tbits);
PRIVATE char   *FrameNoStr  (S32 Frame);
PRIVATE         ByteToBitstr(U8 data, char *buffer);
PRIVATE         GetDeqInfo();
PRIVATE BOOLEAN GetInstruction(int bufID, S32 StartFrame,  U8 *RemainByte,
                               S32 *EndFrame, char *Instruction, U8 *Code);
PRIVATE int     CheckFlush (int bufID,S32 *EndFrame, U8 *Code,U8 *RemainByte);
PRIVATE         GetCodeKind(U8 *Code,  int *CodeKind);
PRIVATE BOOLEAN SearchCode (U8 *Table, int TableLen, U8 Code);
PRIVATE BOOLEAN ESCpressed();

/************************ Local Variable Definition ************************/

/* Codes must be sorted */
LOCAL U8 CondBranchCode[] = {
   0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
   0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F,
   0xCE, 0xE0, 0xE1, 0xE2, 0xE3
};

/* Codes must be sorted */
LOCAL U8 UncondBranchCode[] = {
   0x9A, 0xC2, 0xC3, 0xCA, 0xCB, 0xCC, 0xCD, 0xCF,
   0xE8, 0xE9, 0xEA, 0xEB
};

LOCAL char StatusCode[8] = {
   'A', 'I', 'O', 'H', 'S', 'R', 'W', 'D'
};

LOCAL char *OutputPtr = NULL;
LOCAL int   OutputRows = 16, RowCount = 0;
LOCAL int   AccessSize, DeqSize;
LOCAL S32   LastFrame;


                           /*********************
                            *  Executable Code  *
                            *********************/


/***************************************************************************
**
**  Name: deqListNumber(bufID)
**
**  Input: buffer ID
**
**  Description:
**    display information of trace buffers
**
****************************************************************************/
PUBLIC deqListNumber(int bufID)
{
char Temp[80];
int i;

#ifdef MICEPACK
   sprintf(Temp," %X trace buffers are used\r\n", lastBuff);
   DisplayStr(Temp);
   sprintf(Temp," Current trace buffer ID = %X\r\n", curBuf);
   DisplayStr(Temp);
#endif
   RowCount = 0;

   for (i = 1; i <= lastBuff; i++)
      if (bufID <= 0 || i == bufID) {   /* all buffers or specified */
         if (emuGetTraceDepth(i-1,&traceDepth[i-1]) == BUFFER_EMPTY) {
            sprintf(Temp," Trace buffer %X is empty", i);
            if (PutsMore(Temp) == ESC)  return;
         }
         else {
            sprintf(Temp," Trace buffer %X: start frame = %s, ",
                    i, FrameNoStr(traceDepth[i-1].startFrame));
            strcat(Temp,"end frame = ");
            strcat(Temp, FrameNoStr(traceDepth[i-1].endFrame));
            if (PutsMore(Temp) == ESC)  return;
         }
         if (bufID > 0)  break;   /* specified buffer */
      }
}  /* end of deqListNumber() */

/***************************************************************************
**
**  Name: deqListFrame(bufID, StartFrame, Addr1, Addr2, Status)
**
**  Description:
**     display trace frames satisfying restriction of
**     address range, status, and trace bits
**
**  Input: buffer ID, frame#, starting address, end address, status flag
**
**  Notes:
**     Static variable FlagMask must be set correctly
**     according to the procedure IsStatus() in SYNTAX.C
**
****************************************************************************/
PUBLIC deqListFrame(int bufID,S32 StartFrame, U32 Addr1,U32 Addr2, U8 Status) {
// static FlagMask[8] = {0x20, 0x08, 0x10, 0x40, 0x01, 0x02, 0x04, 0x80};
TRACE_INFO TraceBuf[64];
S32 i;
U8 FrameLen=64;
S16 len;
//TRACE_INFO *TraceBuf;

   RowCount = 0;
   if (CheckFrame(bufID, StartFrame) == FALSE)  return;
   GetDeqInfo();
   if (OutputPtr == NULL)
      DisplayStr(" FRAME  ADDRESS   DATA  STATUS  DMA FLUSH  TRACE-BITS    TIMER\r\n");
   RowCount = 0;

   if (quaList.qFlag) {
     //len=64*sizeof(TRACE_INFO);
     //if ((TraceBuf = (TRACE_INFO *)malloc(len)) == NULL) {
     //   prn_ferr(45);
     //   return;
     //}
      while (1) {
         emuListTraceBuff( bufID-1, StartFrame, &quaList, &FrameLen,TraceBuf);
         for (i=0; i < FrameLen; i++) {
            if ( PutOneFrame( TraceBuf[i].frameNo, TraceBuf[i].addr,
                              TraceBuf[i].data, TraceBuf[i].stat,
                              TraceBuf[i].tbit ) == ESC ) {
               free( TraceBuf );
               return;
            }
         }
         if (FrameLen < 64 ) break;
         else StartFrame = TraceBuf[FrameLen-1].frameNo+1;
      }
      free( TraceBuf );
   }
   else {
      for (i = StartFrame; i <= LastFrame; i++) {
         if (ESCpressed())  return;
//       emuGetOneFrame(bufID-1, i);
//       /* Check address range and status */
//       if ( trace_frame.addr >= Addr1 && trace_frame.addr <= Addr2 &&
//           (FlagMask[CYCLE(trace_frame.stat)] & Status) )
//          /* Check trace bits */
//          if ( ev_def.dataSpec == 0 ||
//              (trace_frame.tbit & (U8)ev_def.dataHi) == (U8)ev_def.dataLo )
               if (DisplayFrame(bufID, i) == ESC)  return;
      }
   }
}  /* end of deqListFrame() */

/***************************************************************************
**
**  Name: deqListTrace(Command, bufID, StartFrame)
**
**  Description:
**     display trace results with instruction (Command = 1)
**     or source (Command = 2)
**
**  Input:  command, buffer ID, frame#
**
****************************************************************************/
PUBLIC deqListTrace(int bufID, S32 StartFrame, int Command)
{
U8 RemainByte = 0;
S32 Frame, EndFrame, i;
U8 Code[10];
char Instruction[80], Buffer[100], Prefix[10];
BOOLEAN NewFrame = FALSE, ToDispFrame = FALSE, Ok, Flush;

   if (Command != LISTINST && Command != LISTSOURCE)  return;
   RowCount = 0;
   if (CheckFrame(bufID, StartFrame) == FALSE)  return;
   GetDeqInfo();
   if (Command == LISTINST)
      DisplayStr(" FRAME  LOC   OBJ CODE       SOURCE CODE\r\n");
   else
      DisplayStr(" FRAME  ADDRESS   DATA  STATUS  DMA FLUSH  TRACE-BITS    TIMER\r\n");

   Frame = StartFrame;
   while (Frame <= LastFrame) {
      if (Command == LISTINST) {
         if (RemainByte == 0 || NewFrame)
            strcpy(Prefix, FrameNoStr(Frame));
      }
      else
         if (RemainByte == 0)  ToDispFrame = TRUE;

   /*** Get an instruction ***/
      Ok = GetInstruction(bufID, Frame,  &RemainByte,  &EndFrame,
                          Instruction, Code);
      while (TRUE) {
      /* Check flush */
         if ( Ok && (RemainByte > 0 || SkipFrame(bufID, &EndFrame)) )
            if (CheckFlush(bufID, &EndFrame, Code, &RemainByte) == -1)
               strcat(Instruction, "   (??)");   /* cannot judge */
      /* Display instruction */
         if (Instruction[0] != '\0') {
            if (Command == LISTINST) {
               sprintf(Buffer, "%6s  %s", Prefix, Instruction);
               Prefix[0] = '\0';
            }
            else
               sprintf(Buffer, "%12s%s", "", Instruction);
            if (PutsMore(Buffer) == ESC) {
               EndFrame = LastFrame + 1;   /* force to stop */
               break;
            }
         }
      /* Check whether remaining bytes construct a complete instruction */
         if (Command == LISTSOURCE && RemainByte > 0) {
            emuGetOneFrame(bufID-1, EndFrame);
            Code[0] = MSB(trace_frame.data);
            if (DeqDisasm(trace_frame.addr + RemainByte, Code, sizeof(Code),
                          Instruction) <= RemainByte)
               RemainByte = 0;
            else  break;
         }
         else  break;
      }  /* end of while (TRUE) */
      if (EndFrame > LastFrame)  break;

   /*** Print checked frames ***/
      if (ToDispFrame) {
         ToDispFrame = FALSE;
         if (DisplayFrame(bufID, Frame) == ESC)  break;
      }
      if (Command == LISTSOURCE)
         for (i = Frame+1; i <= EndFrame; i++)
            if (DisplayFrame(bufID, i) == ESC) {
               EndFrame = LastFrame + 1;   /* force to stop */
               break;
            }

   /*** Update current frame# ***/
      NewFrame = (EndFrame != Frame);
      Frame = (RemainByte == 0) ? EndFrame + 1 : EndFrame;
   }  /* end of while (Frame <= LastFrame) */

}  /* end of deqListTrace() */

/***************************************************************************
**
**  Name: CheckFrame(bufID, StartFrame)
**
**  Description:
**     check if trace buffer is empty or frame No. is in range
**
**  Input:  buffer ID, frame#
**
**  Return: TRUE if yes, FALSE otherwise
**
****************************************************************************/
PRIVATE BOOLEAN CheckFrame(int bufID, S32 StartFrame)
{
char Temp[80];

   if (emuGetTraceDepth(bufID-1,&traceDepth[bufID-1]) == BUFFER_EMPTY) {
      sprintf(Temp," Trace buffer %X is empty", bufID);
      if (OutputPtr == NULL)  PutsMore(Temp);
      return (FALSE);
   }
   LastFrame = traceDepth[bufID-1].endFrame;
   /* Check range of frame# */
   if (StartFrame < traceDepth[bufID-1].startFrame ||
       StartFrame > LastFrame) {
      sprintf(Temp," Trace buffer %X: start frame = %s, ",
              bufID, FrameNoStr(traceDepth[bufID-1].startFrame));
      strcat(Temp,"end frame = ");
      strcat(Temp, FrameNoStr(LastFrame));
      if (OutputPtr == NULL)  PutsMore(Temp);
      return (FALSE);
   }
   return (TRUE);
}  /* end of CheckFrame() */

/***************************************************************************
**
**  Name: SkipFrame(bufID, EndFrame)
**
**  Description:
**     skip frames so that end frame No. = next S-frame No. - 1
**
**  Input:  buffer ID, end frame#
**
**  Output: end frame#
**
**  Return: TRUE if success, FALSE otherwise
**
****************************************************************************/
PRIVATE BOOLEAN SkipFrame(int bufID, S32 *EndFrame)
{
   for ((*EndFrame)++; *EndFrame <= LastFrame; (*EndFrame)++) {
      emuGetOneFrame(bufID-1, *EndFrame);
      if (CYCLE(trace_frame.stat) == FETCH_INST) {
         (*EndFrame)--;   /* pointing back */
         return (TRUE);
      }
   };
   /* beyond last frame */
   *EndFrame = LastFrame;
   return (FALSE);
}  /* end of SkipFrame() */

/***************************************************************************
**
**  Name: DisplayFrame(bufID, Frame)
**
**  Description:
**     display values of fields of a frame
**
**  Input:  buffer ID, frame#
**
**  Return: code of pressed key if key is pressed
**
****************************************************************************/
PRIVATE int DisplayFrame(int bufID, S32 Frame)
{
   emuGetOneFrame(bufID-1, Frame);
   return ( PutOneFrame( Frame, trace_frame.addr, trace_frame.data,
                         trace_frame.stat, trace_frame.tbit ) );
}

/***************************************************************************
**
**  Name: PutOneFrame( Frame, Addr, Data, Stat, Tbits )
**
**  Description:
**     display values of fields of a frame
**
**  Input:  buffer ID, frame#
**
**  Return: code of pressed key if key is pressed
**
****************************************************************************/
PRIVATE int PutOneFrame( S32 Frame, U32 Addr, U16 Data , U8 Stat, U8 Tbits )
{
char *Format, Buffer[80], Bitstr[10];
U8 DMA, Flush;
U16 ByteData;

   ByteData = Data;
   if (AccessSize == ACCESSSIZE86 && !(Addr & 1) && !IS_BHE(Stat))
      /* 80x86, even address, word transfer */
      Format = "%6s   %05lX    %04X    %c      %d    %d     %s";
   else {
      Format = "%6s   %05lX      %02X    %c      %d    %d     %s";
      if (AccessSize == ACCESSSIZE86 && (Addr & 1))   /* 80x86, odd address */
         ByteData = MSB(Data);
      else   /* 80x86, byte transfer on even address, or 80x88 */
         ByteData = LSB(Data);
   }
   Flush = ISFLUSH(Stat);
   DMA   = ISDMA(Stat);
   ByteToBitstr(Tbits, Bitstr);
   sprintf(Buffer,Format, FrameNoStr(Frame), Addr, ByteData,
           StatusCode[CYCLE(Stat)], DMA, Flush, Bitstr);
   return (PutsMore(Buffer));

}  /* end of PutOneFrame() */

/***************************************************************************
**
**  Name: FrameNoStr(Frame)
**
**  Description:
**     convert frame# to signed-number string
**
**  Input : frame#
**
**  Return: pointer to converted string
**
****************************************************************************/
PRIVATE char *FrameNoStr(S32 Frame)
{
static char Temp[12];

   if (Frame >= 0)  sprintf(Temp,"+%04lX", Frame);
   else             sprintf(Temp,"-%04lX", -Frame);
   return (Temp);
}  /* end of FrameNoStr() */

/***************************************************************************
**
**  Name: ByteToBitstr(data, buffer)
**
**  Description:
**     convert byte data to bit string
**
**  Input:  byte data
**
**  Output: in buffer
**
****************************************************************************/
PRIVATE ByteToBitstr(U8 data, char *buffer)
{
int i;

   for (i = 0; i < 8; i++) {
      *(buffer++) = (data & 0x80) ? '1' : '0';
      data <<= 1;
   }
   *buffer = '\0';
}  /* end of ByteToBitstr() */

/***************************************************************************
**
**  Name: GetDeqInfo()
**
**  Description:
**     get access size and deq size
**
**  Return:
**
****************************************************************************/
PRIVATE GetDeqInfo()
{
   if (MICE >= 1 && MICE <= 4) {
      AccessSize = ACCESSSIZE88;
      DeqSize = DEQSIZE88;
   }
   else {
      AccessSize = ACCESSSIZE86;
      DeqSize = DEQSIZE86;
   }
}  /* end of GetDeqInfo() */

/***************************************************************************
**
**  Name: GetInstruction(bufID, StartFrame,  RemainByte, EndFrame,
**                       Instruction, Code)
**
**  Description:
**     get an dequeue instruction
**
**  Input:  frame No. and remain byte
**
**  Output: number of remaining bytes, end frame No., instruction, codes
**
****************************************************************************/
PRIVATE BOOLEAN GetInstruction(int bufID, S32 StartFrame,  U8 *RemainByte,
                               S32 *EndFrame, char *Instruction, U8 *Code)
{
U32 PC, Addr;
int CodeCount, InstLen;

   *EndFrame = StartFrame;
   *Instruction = '\0';
   emuGetOneFrame(bufID-1, StartFrame);
   if (CYCLE(trace_frame.stat) != FETCH_INST) {   /* not fetch cycle */
      SkipFrame(bufID, EndFrame);                 /* advance to S-frame */
      *RemainByte = 0;
      return (FALSE);
   }

/*** collect initial codes ***/
   PC = Addr = trace_frame.addr;
   CodeCount = 1;
   if (AccessSize == ACCESSSIZE88)   /* 80x88 */
      Code[0] = LSB(trace_frame.data);
   else                              /* 80x86 */
      if (PC & 1) {                  /* odd address */
         Code[0] = MSB(trace_frame.data);
         Addr &= 0xFFFFFFFE;
      }
      else                           /* even address */
         if (*RemainByte == 1) {
            Code[0] = MSB(trace_frame.data);
            PC ++;
         }
         else {                      /* two bytes effective */
            Code[0] = LSB(trace_frame.data);
            Code[1] = MSB(trace_frame.data);
            CodeCount = 2;
         }
   *RemainByte = 0;

/*** disassemble ***/
   while ((InstLen = DeqDisasm(PC, Code, CodeCount, Instruction)) == 0) {
      *Instruction = '\0';
      if (!SkipFrame(bufID, EndFrame)) {   /* no more S-frames */
         *RemainByte = 0;
         return (FALSE);
      }
      (*EndFrame)++;
      emuGetOneFrame(bufID-1, *EndFrame);
      if (trace_frame.addr - Addr != AccessSize) {   /* not continuous */
         sprintf(Instruction, "%05lX (not enough codes)", PC);
         (*EndFrame)--;
         *RemainByte = 0;
         return (FALSE);
      }
      Addr = trace_frame.addr;
      Code[CodeCount++] = LSB(trace_frame.data);
      if (AccessSize == ACCESSSIZE86)   /* 80x86 */
         Code[CodeCount++] = MSB(trace_frame.data);
   };
   *RemainByte = CodeCount - InstLen;
   return (TRUE);
}  /* end of GetInstruction() */

/***************************************************************************
**
**  Name: CheckFlush(bufID, EndFrame, Code, RemainByte)
**
**  Description:
**     check if flush occurs, and set correct end frame No. if necessary
**
**  Input:  current end frame No., codes, and No. of remain bytes
**
**  Output: number of remaining bytes, end frame No.
**
**  Return: 1 if flush, 0 if no flush, -1 if cannot judge
**
****************************************************************************/
PRIVATE int CheckFlush(int bufID, S32 *EndFrame, U8 *Code, U8 *RemainByte)
{
S32 FrameSave, FlushFrame;
int CodeKind, ByteCount;
U32 addr;
BOOLEAN Flush = FALSE, AddrFound = 0;

   GetCodeKind(Code,  &CodeKind);
   if (CodeKind == NOBRANCH)  return (0);
   FrameSave = *EndFrame;
   if (*RemainByte == 0) {    /* base is next S-frame */
      (*EndFrame)++;
      ByteCount = AccessSize;
   }
   else                       /* base is current S-frame */
      ByteCount = *RemainByte;
   emuGetOneFrame(bufID-1, *EndFrame);
   addr = trace_frame.addr;   /* reference address */

/*** search flush frame ***/
   while (1) {
      if (!SkipFrame(bufID, EndFrame)) {   /* no more S-frames */
         *EndFrame = FrameSave;
         return (CodeKind == CONDBRANCH ? -1 : 1);
      }
      (*EndFrame)++;
      emuGetOneFrame(bufID-1, *EndFrame);
      ByteCount += AccessSize;

      /* address is not continuous  or  flush bit = 1 */
      if (trace_frame.addr - addr != AccessSize || ISFLUSH(trace_frame.stat))
         if (!dsmAddrGiven) {      /* explicit address is not given */
            Flush = TRUE;
            FlushFrame = *EndFrame;
            break;
         }
         else                      /* check whether address is matched */
            if ((trace_frame.addr & 0x0000FFFFL) == (dsmAddr & 0x0000FFFFL))
               if (trace_frame.addr - addr != AccessSize) {
                  Flush = TRUE;
                  FlushFrame = *EndFrame;
                  break;
               }
               else {
                  AddrFound = 1;   /* with flush bit = 1 */
                  FlushFrame = *EndFrame;
                  /* no break; prevent incorrect flush bit */
               }

      /* address is continuous  and  explicit branch address is matched */
      if ( !AddrFound && ByteCount >= AccessSize*2 && dsmAddrGiven &&
          (trace_frame.addr & 0x0000FFFFL) == (dsmAddr & 0x0000FFFFL))
         if (ByteCount <= DeqSize || CodeKind == UNCONDBRANCH) {
            AddrFound = 2;   /* with flush bit = 0 */
            FlushFrame = *EndFrame;
            /* no break; it's possible that next S-frame is flush-frame */
         }

      if ( !dsmAddrGiven && ( ByteCount == (DeqSize & 0xFE) ||
           (ByteCount == DeqSize && *RemainByte == 1) )  )
         FlushFrame = *EndFrame;    /* save last frame for indirect branch */

      addr = trace_frame.addr;
      /* end of checking ? */
      if (ByteCount >= DeqSize) {
         if (!dsmAddrGiven)
            Flush = TRUE;   /* indirect branch, assume flush */
         break;
      }
   }  /* end of while */

   /* explicit address found with  flush bit = 1  or unconditional branch) */
   if ( AddrFound == 1 || (AddrFound == 2 && CodeKind == UNCONDBRANCH) )
      Flush = TRUE;
   if (Flush) {
      *EndFrame = FlushFrame - 1;
      *RemainByte = 0;
   }
   else
      *EndFrame = FrameSave;
   return (Flush);
}  /* end of CheckFlush() */

/***************************************************************************
**
**  Name: GetCodeKind(Code,  CodeKind)
**
**  Description:
**     determine the kind of codes:
**     no branch, conditional branch, or unconditional branch
**
**  Input:  codes
**
**  Output: code kind
**
****************************************************************************/
PRIVATE GetCodeKind(U8 *Code,  int *CodeKind)
{
U8 ID;

   ID = (Code[1] & 0x38) >> 3;
   if (Code[0] == 0xFF && ID >= 2 && ID <= 5)   /* JMP/CALL reg/mem */
      *CodeKind = UNCONDBRANCH;   /* unconditional branch */
   else if (SearchCode(UncondBranchCode,
            sizeof(UncondBranchCode)/sizeof(U8), Code[0]))
      *CodeKind = UNCONDBRANCH;   /* unconditional branch */
   else if (SearchCode(CondBranchCode,
            sizeof(CondBranchCode)/sizeof(U8), Code[0]))
      *CodeKind = CONDBRANCH;     /* conditional branch */
   else
      *CodeKind = NOBRANCH;       /* no branch */
}  /* end of GetCodeKind() */

/***************************************************************************
**
**  Name: SearchCode(Table, TableLen, Code)
**
**  Description:
**     search code in code table
**
**  Input:  code table, table length, code
**
**  Return: TRUE if found, FALSE otherwise
**
****************************************************************************/
PRIVATE BOOLEAN SearchCode(U8 *Table, int TableLen, U8 Code)
{
int Low, High, Mid;

   Low = 0;
   High = TableLen - 1;
   while (Low <= High) {
      Mid = (Low + High) / 2;
      if (Code < Table[Mid])
         High = Mid - 1;
      else if (Code > Table[Mid])
         Low = Mid + 1;
      else
         return (TRUE);
   }
   return (FALSE);
}  /* end of SearchCode() */

/***************************************************************************
**
**  Name: ESCpressed()
**
**  Description:
**     If ESC key or right mouse button is pressed, display \
**
**  Return: TRUE or FALSE
**
****************************************************************************/
PRIVATE BOOLEAN ESCpressed()
{
int key = 0;

   if (ki_chk() == ESC)  key = ki();
   else  ChkMouseLBAndEsc(&key);
   if (key == ESC) {
      DisplayStr("\\\r\n");
      if (cmdfile_flag)  cmdfile_end_flag = 1;
      return (TRUE);
   }
   else  return (FALSE);
}  /* end of ESCpressed() */

/****************************** End of File ********************************/
