/***************************************************************************
**
**  File name : DSM.C
**
**  Description : main module of disassembler
**
**  Date : 11/13/1992  by C. H. Lin
**    1. Initial version (include symbol function)
**
**  Date : 11/25/1992  by C. H. Lin
**    1. Add IsStepCall()
**
**  Date : 11/26/1992  by C. H. Lin
**    1. Change IsStepCall(Addr, NextAddr)
**       to     IsStepCall(Addr, Buffer,  NextAddr)
**
**  Date : 12/22/1992  by C. H. Lin
**    1. Add DisasmOneline(PC, Code,  Instruction),
**       called by Disassemble() and DeqDisasm()
**    2. Add IsStrOpCode(Code)
**    3. In DisasmOneline() add repeat prefix and segment override
**
**  Date : 12/28/1992  by C. H. Lin
**    1. In IsStepCall(), add checking segment override
**
**  Date : 1/14/1993  by C. H. Lin
**    1. In IsStepCall(), return different step ID for CALL, RET ...
**
**  Date : 1/28/1993  by C. H. Lin
**    1. In Disassemble(), DeqDisasm(), and DisasmOneline(),
**       put nothing into output buffer if it's NULL
**
**  Date : 2/10/1993  by C. H. Lin
**    1. In IsStepCall(), add checking  REPZ/REPNZ str_manipulation
**
**  Date : 2/15/1993  by C. H. Lin
**    1. In DisasmOneline(), regard ES:,CS:,SS:,DS:,REPZ,and REPNZ as prefix
**
**
**  Copyright (C) 1993 Microtek International Inc.
**  All Rights Reserved
**
****************************************************************************/


/****************************** Include File *******************************/

#include <stdio.h>
#include <string.h>
#include "dsmdef.h"

/*********************** External Function Prototype ***********************/

/* defined in DSMTYPE.C */
EXTERN RETCODE Dtype_Xs1_disp     (PCTYPE PC, U8 *Code,  STR Buf, int *Len);
EXTERN RETCODE Dtype_X_disp       (PCTYPE PC, U8 *Code,  STR Buf, int *Len);
EXTERN RETCODE Dtype_X_ldisp_hdisp(PCTYPE PC, U8 *Code,  STR Buf, int *Len);

/********************** External Variable Declaration **********************/

/* defined in DSMTBL.C */
EXTERN DISASMSTRUCT DisasmTable[];

/************************ Local Function Prototype *************************/

PRIVATE int DisasmOneline(PCTYPE PC, U8 *Code, int CodeLen, STR Instruction);
PRIVATE GetMnemonicType(U8 *Code,  STR Mnemonic, TYPEROUTINE *Type);
PRIVATE BOOLEAN IsStrOpCode(U8 Code);

/************* Global Variable Definition for Related Modules **************/

GLOBAL U16     dsmCS;                      /* used for  address --> symbol */
GLOBAL BOOLEAN dsmIsAbsAddr, dsmAddrGiven; /* used for                     */
GLOBAL long    dsmAddr;                    /*    dequeue                   */
GLOBAL U8      dsmSegOverride;             /* used for                     */
GLOBAL BOOLEAN dsmSegUsed;                 /*    segment overide           */


                           /*********************
                            *  Executable Code  *
                            *********************/


/***************************************************************************
**
**  Name: Disassemble(CSIP, Code, CodeLen,  Buffer)
**
**  Description:
**     disassemble codes to get one line of instruction
**
**  Input:  address CS:IP, code data, code length
**
**  Output: formatted instruction
**
**  Return: instruction length if codes are enough, 0 otherwise
**
****************************************************************************/
PUBLIC int Disassemble(PCTYPE CSIP, U8 *Code, int CodeLen,  STR Buffer)
{
char Instruction[70];
int Length;

   dsmIsAbsAddr = FALSE;
   dsmCS = (CSIP & 0xFFFF0000) >> 16;
   Length = DisasmOneline(CSIP & 0x0000FFFF, Code, CodeLen,
                          Buffer == NULL ? NULL : Instruction);
   if (Buffer != NULL)
      sprintf(Buffer, "%04lX:%04lX %s",
             (CSIP & 0xFFFF0000) >> 16, CSIP & 0x0000FFFF, Instruction);
   return (Length);
}  /* end of Disassemble() */

/***************************************************************************
**
**  Name: DeqDisasm(PC, Code, CodeLen,  Buffer)
**
**  Description:
**     disassemble codes to get one line of instruction (used by dequeue)
**
**  Input:  absolute address, code data, code length
**
**  Output: formatted instruction
**
**  Return: instruction length if codes are enough, 0 otherwise
**
****************************************************************************/
PUBLIC int DeqDisasm(PCTYPE PC, U8 *Code, int CodeLen,  STR Buffer)
{
char Instruction[70];
int Length;

   dsmIsAbsAddr = TRUE;
   dsmCS = 0;
   /* if INTO, set dsmAddrGiven = TRUE (assume conditional branch in deq) */
   dsmAddrGiven = (Code[0] == 0xCE);
   dsmAddr = 0xFFFFFFFF;

   /* if branch address in instruction is known,
      dsmAddrGiven = TRUE after calling DisasmOneline() */
   Length = DisasmOneline(PC, Code, CodeLen,
                          Buffer == NULL ? NULL : Instruction);
   if (Buffer != NULL)
      sprintf(Buffer, "%05lX %s", PC, Instruction);
   return (Length);
}  /* end of DeqDisasm() */

/***************************************************************************
**
**  Name: DisasmOneline(PC, Code, CodeLen,  Instruction)
**
**  Description:
**     disassemble codes to get one line of instruction
**
**  Input:  PC, code data, code length
**
**  Output: instruction following code data
**
**  Return: instruction length if codes are enough, 0 otherwise
**
****************************************************************************/
PRIVATE int DisasmOneline(PCTYPE PC, U8 *Code, int CodeLen,  STR Instruction)
{
char Prefix[10], Mnemonic[20];
char Operands[50], CodeStr[20], Temp[60];
TYPEROUTINE Type;
int Length, ExtraLen = 0, i;

/*** Process prefix in instruction ***/
   dsmSegOverride = dsmSegUsed = 0;
   if (Code[0] == 0xF2 || Code[0] == 0xF3) {   /* REPNZ or REPZ */
      GetMnemonicType(Code,  Prefix, &Type);
      ExtraLen = 1;
   }
   else if ((Code[0] & 0xE7) == 0x26) {        /* ES:, CS:, SS:, DS: */
      GetMnemonicType(Code,  Prefix, &Type);
      ExtraLen = 1;
      dsmSegOverride = Code[0];
   }

/*** Get mnemonic symbol and type processing routine pointer ***/
   GetMnemonicType(Code+ExtraLen,  Mnemonic, &Type);

/*** Call type processing routine to derive operands ***/
   if (Type == Dtype_Xs1_disp || Type == Dtype_X_disp ||
       Type == Dtype_X_ldisp_hdisp)
      (*Type)(PC, Code+ExtraLen,  Operands, &Length);
   else
      (*Type)(Code+ExtraLen,  Operands, &Length);
   Length += ExtraLen;

/*** Put together codes, mnemonic symbol, and operands ***/
   if (Instruction != NULL) {
      for (i = 0; i < Length; i++)
         sprintf(&CodeStr[2*i],"%02X", Code[i]);
      if (*Operands == '\0')
         strcpy(Temp, Mnemonic);
      else
         sprintf(Temp,"%-8s%s", Mnemonic, Operands);
      if (ExtraLen > 0 && !dsmSegUsed)   /* add prefix before mnemonic */
         sprintf(Instruction,"%-15s%-8s%s", CodeStr, Prefix, Temp);
      else
         sprintf(Instruction,"%-15s%s", CodeStr, Temp);
   }
   return (Length <= CodeLen ? Length : 0);
}  /* end of DisasmOneline() */

/***************************************************************************
**
**  Name: GetMnemonicType(Code,  Mnemonic, Type)
**
**  Description:
**     get mnemonic symbol and code type routine of codes
**
**  Input:  code data
**
**  Output: mnemonic symbol, code type routine
**
****************************************************************************/
PRIVATE GetMnemonicType(U8 *Code,  STR Mnemonic, TYPEROUTINE *Type)
{
STR MnemonicPtr;

/*** Identify mnemonic symbol by checking first byte, then second byte ***/
   MnemonicPtr = DisasmTable[Code[0]].Mnemonic;
   *Type = DisasmTable[Code[0]].Type;
   if (MnemonicPtr != NULL)   /* unique mnemonic symbol */
      strcpy(Mnemonic, MnemonicPtr);
   else                       /* call further routine to check second byte */
      (**Type)(Code,  Mnemonic, Type);
}  /* end of GetMnemonicType() */

/***************************************************************************
**
**  Name: IsStepCall(Addr, Code,  NextAddr)
**
**  Description:
**     determine step ID of instruction
**
**  Input:  current address
**
**  Output: address of next executed code
**
**  Return: 1: CALL, 2: RET/RETF/IRET, 3: INT, 4: LOOPxx,
**          5: REPZ/REPNZ str_manipulation, 0: otherwise
**
****************************************************************************/
PUBLIC int IsStepCall(U32 Addr, U8 *Code,  U32 *NextAddr)
{
#define CALL 1
#define RET  2
#define INT  3
#define LOOP 4
#define REP  5
static U8 CodeTable[] = {
   0xE8, 0x9A, 0xC3, 0xCA, 0xCB, 0xCF, 0xCD, 0xCC, 0xCE, 0xE0, 0xE1, 0xE2
};
static int StepIDTable[] = {
   CALL, CALL, RET,  RET, RET,  RET,  INT,  INT,  INT,  LOOP, LOOP, LOOP
};
static int LengthTable[] = {
   3,    5,    1,    3,    1,    1,    2,    1,    1,    2,    2,    2
};
U8  Code0, Code1, ID;
int StepID = 0, Len = 0, i;
int SegFlag = 0;

   Code0 = Code[0];
   Code1 = Code[1];
   if ((Code0 & 0xE7) == 0x26) {    /* ES:, CS:, SS:, DS: */
      Code0 = Code[1];
      Code1 = Code[2];
	  SegFlag = 1;
   }
   for (i = 0; i < COUNT(CodeTable); i++)
      if (Code0 == CodeTable[i]) {
         StepID = StepIDTable[i];
         Len = LengthTable[i];
         break;
      }

   if (StepID == 0)   /* check further */
      if ((Code0 == 0xF2 || Code0 == 0xF3) && IsStrOpCode(Code1)) {
         StepID = REP;   // REPNZ/REPNZ str_manipulation
         Len = 2;
      }
      else if (Code0 == 0xFF)
         if ((ID = Code1 & 0x38) == 0x10 || ID == 0x18) {   // CALL (FAR) r/m
            StepID = CALL;
            switch ( (Code1 & 0xC0) >> 6 ) {   // mod bits
            case 0:  /* no offset byte */
			   Len = (Code1 & 0x07) == 6 ? 4 : 2;	// direct mem address ?
			   break;
            case 1:  /* 1 offset byte */
               Len = 3;
               break;
            case 2:  /* 2 offset bytes */
               Len = 4;
               break;
            case 3:  /* register */
               Len = 2;
               break;
            }
         }

   if (SegFlag == 1) Len++;
   *NextAddr = (Addr & 0xFFFF0000) + ((Addr + Len) & 0x0000FFFF);
   return (StepID);
}  /* end of IsStepCall() */


/***************************************************************************
**
**  Name: IsStepCall2(Addr, Code,  NextAddr, CodeLen)
**
**  Description:
**     determine step ID of instruction
**
**  Input:  current address
**
**  Output: end address of executed code
**
**  Return: 1: CALL, 2: RET/RETF/IRET, 3: INT, 4: LOOPxx,
**			5: REPZ/REPNZ str_manipulation, 6: JMP out
**			7: first Jb, the last is JMP addr, addr = (Jb)
**			8: first JMP adx1, but addr1 < adx1 < addr2, the last is
**			   Jb adx2, adx2 < (Jb)
**			0: otherwise
**	Notes:
**
****************************************************************************/
PUBLIC int IsStepCall2(U32 Addr, U8 *Code, U32 *prevJMPAddr, U32 *NextAddr,
					   U32 CodeLen, int readDirection)
{
#define CALL 1
#define RET  2
#define INT  3
#define LOOP 4
#define REP  5
#define JMP  6
#define Jb	 7

static U8 CodeTable[] = {
   0xE8, 0x9A, 0xC3, 0xCA, 0xCB, 0xCF, 0xCD, 0xCC, 0xCE, 0xE0, 0xE1, 0xE2,
   0xE9, 0XEA, 0XEB
};
static int StepIDTable[] = {
   CALL, CALL, RET,  RET, RET,  RET,  INT,  INT,  INT,  LOOP, LOOP, LOOP,
   JMP,  JMP , JMP
};
static int LengthTable[] = {
   3,    5,    1,    3,    1,    1,    2,    1,    1,    2,    2,    2,
   3,    5,    2
};

U8	Code0, Code1, ID;
U8 buf[80], *ptr;
U16 tempOff = 0;
int StepID, Len, i;
int JbFlag ;
int JumpFlag ;
int SegFlag;
int IntFlag;
U16 loop = 0;
U16 oneInstrucLen ;
U16 JbToOffset ;
U32 jumpToAddr;
U32 tempPrevAddr;
U32 tempAddr = Addr;
U32 tempLen = CodeLen;

   StepID = Len = 0;
   JbFlag = JumpFlag = 0;
   SegFlag = IntFlag = 0;
   oneInstrucLen = 0;

   ptr = Code;
   while ( 0 <= tempLen ){
	   if ( (Code[loop] & 0xE7) == 0x26) {	  /* ES:, CS:, SS:, DS: */
		  Code0 = Code[loop+1];
		  Code1 = Code[loop+2];
		  SegFlag = 1;
	   }
	   else {
		   Code0 = Code[loop];
		   Code1 = Code[loop+1];
		   SegFlag = 0;
	   }
       for (i = 0; i < COUNT(CodeTable); i++){
            if ( Code0 == CodeTable[i] ) {
				StepID = StepIDTable[i];
				Len = LengthTable[i];
				if (StepID == JMP) JumpFlag = 1;
				if (StepID == INT) IntFlag = 1;
				if (JbFlag == 1 && Code0 == 0xEB && IntFlag == 0){
				   tempOff = (Addr + loop+ Len +Code1) & 0x000000FF;
				   jumpToAddr = ((Addr+loop) & 0xFFFFFF00) + tempOff;
				   if ( Addr == jumpToAddr ) { //CASE : 7
						StepID = Jb;
						*NextAddr = (Addr & 0xFFFF0000)+ JbToOffset;
						return(Jb);
				   }
				   else goto breakOut;
				}
				else if (IntFlag == 1 && JbFlag == 0 && JumpFlag == 0 )
					break;
				else goto breakOut;
			}
       }
	   if (Code0 >= 0x70 && Code0 <= 0x7F){
		   if (JbFlag == 1) {
			  *prevJMPAddr = tempPrevAddr;
			  return(JMP);
		   }
		   JbFlag = 1;
		   tempOff = (Addr + loop + 2 +Code1) & 0x000000FF;
		   JbToOffset = (U16)(Addr & 0xFFFFFF00) + tempOff;
		   tempPrevAddr = (Addr & 0xFFFF0000) + ((Addr + loop) & 0x0000FFFF);
	   }
	   else if (StepID == 0){	/* check further */
           if ((Code0 == 0xF2 || Code0 == 0xF3) && IsStrOpCode(Code1)) {
                StepID = REP;   // REPNZ/REPNZ str_manipulation
                Len = 2;
				goto breakOut;
           }
		   else if (Code0 == 0xFF){
              if ((ID = Code1 & 0x38) == 0x10 || ID == 0x18) {   // CALL (FAR) r/m
				  StepID = CALL;
                  switch ( (Code1 & 0xC0) >> 6 ) {   // mod bits
                     case 0:  /* no offset byte */
						Len = (Code1 & 0x07) == 6 ? 4 : 2;	// direct mem address ?
                        break;
                     case 1:  /* 1 offset byte */
                        Len = 3;
                        break;
                     case 2:  /* 2 offset bytes */
                        Len = 4;
                        break;
                     case 3:  /* register */
                        Len = 2;
                        break;
                  }
				  goto breakOut;
			  } //end of "CALL (FAR)"
			}
          } //end of "StepID == 0"

		  tempAddr += oneInstrucLen;
          memset(buf, '\0', 80);
		  oneInstrucLen = DisasmOneline(tempAddr & 0x0000FFFF, ptr, tempLen,
                                buf);
          if (oneInstrucLen == 0) break;
          loop += oneInstrucLen;
          ptr += oneInstrucLen;
		  tempLen -= oneInstrucLen;
	  } // end of while
breakOut:
	 if (SegFlag == 1) Len++;
     *NextAddr = (Addr & 0xFFFF0000) + ((Addr + Len + loop) & 0x0000FFFF);
	 if (StepID == JMP){
		  *prevJMPAddr = *NextAddr - Len;
	 }
	 if (JbFlag == 1) {
		  *prevJMPAddr = tempPrevAddr;
		  return(JMP);
	 }
     return (StepID);
}  /* end of IsStepCall2() */

/***************************************************************************
**
**  Name: IsStrOpCode(Code)
**
**  Description:
**     determine if code is a string-manipulation instruction
**
**  Input:  code
**
**  Return: TRUE or FALSE
**
****************************************************************************/
PRIVATE BOOLEAN IsStrOpCode(U8 Code)
{
static U8 StrOpCodes[] = {
   0x6C,0x6D,0x6E,0x6F,0xA4,0xA5,0xA6,0xA7,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF
};
int CodeCmp(U8*, U8*);
int (*fcmp)();

   fcmp = CodeCmp;
   return (bsearch(&Code, StrOpCodes, COUNT(StrOpCodes),
           sizeof(StrOpCodes[0]), fcmp) != NULL);
}  /* end of IsStrOpCode() */

/***************************************************************************/
PRIVATE int CodeCmp(U8 *c1, U8 *c2)
{
   return (*c1 - *c2);
}

/******************************* End of File *******************************/
