/***************************************************************************
**
**  File name : DSMTYPE.C
**
**  Description : type processing routines of disassembler
**
**  Date : 11/13/1992  by C. H. Lin
**    1. Initial version (include symbol processing)
**
**  Date : 11/17/1992  by C. H. Lin
**    1. Rename Addr2Sym() as dsmAddr2Sym()
**       Change  static BOOLEAN Addr2Sym()
**       to      BOOLEAN dsmAddr2Sym()
**
**  Date : 11/25/1992  by C. H. Lin
**    1. In declaration, change AddrToSym() to AddrToSym(int *itemno)
**    2. In dsmAddr2Sym(), set comsym.mask = MSK_PHY
**
**  Date : 12/22/1992  by C. H. Lin
**    1. Add ChkSegOverride()
**
**  Date : 12/23/1992  by C. H. Lin
**    1. Modify AddDataInfo() to arrange output format
**
**  Date : 3/16/1993  by C. H. Lin
**    1. In dsmAddr2Sym(), call AddrToSym() using absolute address
**
**
**  Copyright (C) 1993 Microtek International Inc.
**  All Rights Reserved
**
****************************************************************************/


/****************************** Include File *******************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "usym1.h"
#include "dsmdef.h"

/**************************** Macro Definition *****************************/

#define SEGID(x) (((x) & 0x18) >> 3)    /* xxxSGxxx */
#define MODID(x) (((x) & 0xC0) >> 6)    /* MOxxxxxx */
#define REGID(x) (((x) & 0x38) >> 3)    /* xxREGxxx */
#define RMID(x)  ( (x) & 0x07 )         /* xxxxxR_M */

/*************************** Enumerated Constants **************************/

/* The order must be arranged correctly according to
   PtrTable in section Local Variable Definition */
enum MEMPTR {BYTEPTR, WORDPTR, DWORDPTR, QWORDPTR, TBYTEPTR};

/*********************** External Function Prototype ***********************/

/* defined in WUSYM1.C */
EXTERN AddrToSym(int *itemno);

/********************** External Variable Declaration **********************/

/* defined in DSM.C */
EXTERN U16     dsmCS;
EXTERN BOOLEAN dsmIsAbsAddr, dsmAddrGiven;
EXTERN long    dsmAddr;
EXTERN U8      dsmSegOverride;
EXTERN BOOLEAN dsmSegUsed;

/* defined in OLDGBL.H */
EXTERN int sym_flag;

/* defined in USYM2.H */
EXTERN struct common_symbol comsym;

/************************ Local Function Prototype *************************/

PRIVATE Proc_Xw_modXrm(U8 *Code,  STR Operand, int *Len);
PRIVATE Proc_X_modXrm (U8 *Code,  STR Operand, int *Len);
PRIVATE GetPtr        (U8 *Code,  STR PtrStr);
PRIVATE SignbyteToStr (U8 Data,   STR Buf);
PRIVATE ChkSegOverride(char *mem);
PRIVATE AddDataInfo   (STR Buf, U8 Data);
PRIVATE ConvertAddr   (long Addr,  STR Buf);

/************************ Local Variable Definition ************************/

/* The order in PtrTable must be arranged correctly
   according to MEMPTR constants in section Enumerated Constants */
LOCAL char PtrTable[5][6] = {
   "BYTE", "WORD", "DWORD", "QWORD", "TBYTE"
};

LOCAL char SegTable[4][3] = {
   "ES", "CS", "SS", "DS"
};
LOCAL char RegTable[16][3] = {
   "AL", "CL", "DL", "BL", "AH", "CH", "DH", "BH",
   "AX", "CX", "DX", "BX", "SP", "BP", "SI", "DI"
};
LOCAL char STtable[8][6] = {
   "ST(0)", "ST(1)", "ST(2)", "ST(3)", "ST(4)", "ST(5)", "ST(6)", "ST(7)"
};
LOCAL char *MemTable[8] = {
   "BX+SI", "BX+DI", "BP+SI", "BP+DI", "SI", "DI", "BP", "BX",
};


                           /*********************
                            *  Executable Code  *
                            *********************/


/***************************************************************************
**
**  Name: Dtype_*(Code,  Buf, Len)
**
**  Description:
**     derive operands of instruction for its type
**
**  Input:  code data
**
**  Output: operands, instruction length
**
**  Return: none
**
****************************************************************************/

/***************************************************************************
**  undefined codes
****************************************************************************/
PUBLIC RETCODE Dtype_DB(U8 *Code,  STR Buf, int *Len)
{
   *Len = 1;
   sprintf(Buf,"%02X", Code[0]);
}

/***************************************************************************
**  no operands
**  obj codes: | ** *** ***|
****************************************************************************/
PUBLIC RETCODE Dtype_X(U8 *Code,  STR Buf, int *Len)
{
   *Len = 1;
   Buf[0] = '\0';
}

/***************************************************************************
**  PUSH/POP seg
**  obj codes: | ** 0sg ***|
****************************************************************************/
PUBLIC RETCODE Dtype_X0sgX(U8 *Code,  STR Buf, int *Len)
{
   *Len = 1;
   strcpy(Buf, SegTable[ SEGID(Code[0]) ] );
}

/***************************************************************************
**  shift/rotate reg/mem,1/CL
**  obj codes: | ** *** *cw|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_Xcw_modXrm(U8 *Code,  STR Buf, int *Len)
{
char PtrStr[20], Operand[80];

   GetPtr(Code, PtrStr);
   Proc_Xw_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s%s,%s", PtrStr, Operand, (Code[0] & 0x02) ? "CL" : "1");
}

/***************************************************************************
**  MOV seg,reg/mem
**  MOV reg/mem,seg
**  obj codes: | ** *** *d0|mod 0sg r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_Xd0_mod0sgrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_X_modXrm(Code,  Operand, Len);
   if (Code[0] & 0x02)   /* destination is segment */
      sprintf(Buf,"%s,%s", SegTable[ SEGID(Code[1]) ], Operand);
   else
      sprintf(Buf,"%s,%s", Operand, SegTable[ SEGID(Code[1]) ]);
}

/***************************************************************************
**  OUT DX,acc
**  IN  acc,DX
**  obj codes: | ** *** *dw|
****************************************************************************/
PUBLIC RETCODE Dtype_Xdw(U8 *Code,  STR Buf, int *Len)
{
   *Len = 1;
   if (Code[0] & 0x02)   /* destination is DX */
      sprintf(Buf,"DX,%s", RegTable[(Code[0] & 0x01) << 3]);
   else
      sprintf(Buf,"%s,DX", RegTable[(Code[0] & 0x01) << 3]);
}

/***************************************************************************
**  MOV memdirect,acc
**  MOV acc,memdirect
**  obj codes: | ** *** *dw| addr_low  | addr_high |
****************************************************************************/
PUBLIC RETCODE Dtype_Xdw_laddr_haddr(U8 *Code,  STR Buf, int *Len)
{
char DirectAddr[50];
char Symbol[50];

   *Len = 3;
   /* check whether direct memory address can be converted into symbol */
   if (dsmAddr2Sym(Code[1] + Code[2]*256L,  Symbol) == TRUE)
      sprintf(DirectAddr,"[%s]", Symbol);
   else
      sprintf(DirectAddr,"[%02X%02X]", Code[2], Code[1]);

   ChkSegOverride(DirectAddr);
   if (Code[0] & 0x02)   /* destination is memory */
      sprintf(Buf,"%s,%s", DirectAddr, RegTable[(Code[0] & 0x01) << 3]);
   else
      sprintf(Buf,"%s,%s", RegTable[(Code[0] & 0x01) << 3], DirectAddr);
}

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/MOV
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** *dw|mod reg r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_Xdw_modregrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_Xw_modXrm(Code,  Operand, Len);
   if (Code[0] & 0x02)   /* destination is register */
       sprintf(Buf,"%s,%s",
               RegTable[ ((Code[0] & 0x01)<<3) | REGID(Code[1]) ],Operand);
   else
       sprintf(Buf,"%s,%s", Operand,
               RegTable[ ((Code[0] & 0x01)<<3) | REGID(Code[1]) ]);
}

/***************************************************************************
**  OUT port,acc
**  IN  acc,port
**  obj codes: | ** *** *dw|   port    |
****************************************************************************/
PUBLIC RETCODE Dtype_Xdw_port(U8 *Code,  STR Buf, int *Len)
{
   *Len = 2;
   if (Code[0] & 0x02)   /* destination is port */
      sprintf(Buf,"%02X,%s", Code[1], RegTable[(Code[0] & 0x01) << 3]);
   else
      sprintf(Buf,"%s,%02X", RegTable[(Code[0] & 0x01) << 3], Code[1]);
}

/***************************************************************************
**  INC/DEC/PUSH/POP reg
**  XCHG reg,AX
**  obj codes: | ** *** reg|
****************************************************************************/
PUBLIC RETCODE Dtype_Xreg(U8 *Code,  STR Buf, int *Len)
{
   *Len = 1;
   strcpy(Buf, RegTable[8 + (Code[0] & 0x07)]);
   if ((Code[0] & 0xF8) == 0x90)   /* XCHG */
      strcat(Buf, ",AX");
}

/***************************************************************************
**  PUSH data
**  obj codes: | ** *** *s0|   data    |data if s=0|
****************************************************************************/
PUBLIC RETCODE Dtype_Xs0_data(U8 *Code,  STR Buf, int *Len)
{
   if (Code[0] & 0x02) {   /* data is signed byte */
      *Len = 2;
      SignbyteToStr(Code[1], Buf);
   }
   else {
      *Len = 3;
      sprintf(Buf,"%02X%02X", Code[2], Code[1]);
   }
}

/***************************************************************************
**  JMP address
**  obj codes: | ** *** *s1|   disp    |
****************************************************************************/
PUBLIC RETCODE Dtype_Xs1_disp(PCTYPE PC, U8 *Code,  STR Buf, int *Len)
{
   if (Code[0] & 0x02) {   /* displacement is signed byte */
      *Len = 2;
      dsmAddr = PC + *Len + (int) ((char)Code[1]);
   }
   else {
      *Len = 3;
      dsmAddr = PC + *Len + (int) (Code[1]+Code[2]*256);
   }
   dsmAddrGiven = TRUE;
   /* try converting address into symbol */
   ConvertAddr(dsmAddr,  Buf);
}

/***************************************************************************
**  IMUL reg,reg/mem,data
**  obj codes: | ** *** *s1|mod reg r/m|   data    |data if s=0|
****************************************************************************/
PUBLIC RETCODE Dtype_Xs1_modregrm_data(U8 *Code,  STR Buf, int *Len)
{
char Operand[80], DataStr[20];

   Proc_X_modXrm(Code,  Operand, Len);
   if (Code[0] & 0x02) {   /* data is signed byte */
      SignbyteToStr(Code[*Len], DataStr);
      (*Len)++;
   }
   else {
      sprintf(DataStr,"%02X%02X", Code[*Len + 1], Code[*Len]);
      *Len += 2;
   }
   sprintf(Buf,"%s,%s,%s", RegTable[ 8 + REGID(Code[1]) ], Operand, DataStr);
}

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR reg/mem,data
**  obj codes: | ** *** *sw|mod *** r/m|   data    |data if sw=01|
****************************************************************************/
PUBLIC RETCODE Dtype_Xsw_modXrm_data(U8 *Code,  STR Buf, int *Len)
{
char PtrStr[20], Operand[80], DataStr[20];
U8 Data, ID;

   GetPtr(Code, PtrStr);
   Proc_Xw_modXrm(Code,  Operand, Len);
   if ((Code[0] & 0x01) == 0) {   /* byte data */
      Data = Code[*Len];
      sprintf(DataStr,"%02X", (U16)Data);
      (*Len)++;
   }
   else if (Code[0] & 0x02) {     /* word data, sign extended from byte */
      SignbyteToStr(Code[*Len], DataStr);
      (*Len)++;
   }
   else {                         /* word data */
      sprintf(DataStr,"%02X%02X", Code[*Len + 1], Code[*Len]);
      *Len += 2;
   }
   sprintf(Buf,"%s%s,%s", PtrStr, Operand, DataStr);

   /* try adding extra information */
   if ((Code[0] & 0x01) == 0) {   /* byte data */
      ID = (Code[1] & 0x38) >> 3;
      if (ID == 7)   /* CMP */
         AddDataInfo(Buf, Data);
   }
}

/***************************************************************************
**  INT data
**  obj codes: | ** *** **v|   data    |
****************************************************************************/
PUBLIC RETCODE Dtype_Xv_type(U8 *Code,  STR Buf, int *Len)
{
   if (Code[0] & 0x01) {
      *Len = 2;
      sprintf(Buf,"%02X", Code[1]);
   }
   else {   /* INT 3 */
      *Len = 1;
      strcpy(Buf,"3");
   }
}

/***************************************************************************
**  MOV reg,data
**  obj codes: | ** **w reg|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Dtype_Xwreg_data(U8 *Code,  STR Buf, int *Len)
{
   if (Code[0] & 0x08) {   /* word data */
      *Len = 3;
      sprintf(Buf,"%s,%02X%02X", RegTable[Code[0] & 0x0F], Code[2], Code[1]);
   }
   else {                  /* byte data */
      *Len = 2;
      sprintf(Buf,"%s,%02X", RegTable[Code[0] & 0x0F], Code[1]);
      AddDataInfo(Buf, Code[1]);
   }
}

/***************************************************************************
**  ADC/ADD/AND/CMP/OR/SBB/SUB/XOR/TEST acc,data
**  obj codes: | ** *** **w|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Dtype_Xw_data(U8 *Code,  STR Buf, int *Len)
{
U8 Opcode;

   if (Code[0] & 0x01) {   /* word data */
      *Len = 3;
      sprintf(Buf,"%s,%02X%02X", RegTable[(Code[0] & 0x01) << 3],
              Code[2], Code[1]);
   }
   else {                  /* byte data */
      *Len = 2;
      sprintf(Buf,"%s,%02X", RegTable[(Code[0] & 0x01) << 3], Code[1]);
      Opcode = Code[0];
      if (Opcode == 0x3C)   /* CMP */
         AddDataInfo(Buf, Code[1]);
   }
}

/***************************************************************************
**  TEST/XCHG
**     reg,reg/mem  or  reg/mem,reg
**  obj codes: | ** *** **w|mod reg r/m|
***************************************************************************/
PUBLIC RETCODE Dtype_Xw_modregrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_Xw_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s,%s", RegTable[ ((Code[0] & 0x01) << 3) | REGID(Code[1]) ],
           Operand);
}

/***************************************************************************
**  NOT/NEG/MUL/IMUL/DIV/IDIV/INC/DEC reg/mem
**  obj codes: | ** *** **w|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_Xw_modXrm(U8 *Code,  STR Buf, int *Len)
{
char PtrStr[20], Operand[80];

   GetPtr(Code, PtrStr);
   Proc_Xw_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s%s", PtrStr, Operand);
}

/***************************************************************************
**  shift/rotate reg/mem,count
**  obj codes: | ** *** **w|mod *** r/m|   count   |
****************************************************************************/
PUBLIC RETCODE Dtype_Xw_modXrm_count(U8 *Code,  STR Buf, int *Len)
{
char PtrStr[20], Operand[80];

   GetPtr(Code, PtrStr);
   Proc_Xw_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s%s,%02X", PtrStr, Operand, Code[*Len]);
   (*Len)++;
}

/***************************************************************************
**  MOV/TEST reg/mem,data
**  obj codes: | ** *** **w|mod *** r/m|   data    |data if w=1|
****************************************************************************/
PUBLIC RETCODE Dtype_Xw_modXrm_data(U8 *Code,  STR Buf, int *Len)
{
char PtrStr[20], Operand[80];

   GetPtr(Code, PtrStr);
   Proc_Xw_modXrm(Code,  Operand, Len);
   if (Code[0] & 0x01) {     /* word data */
      sprintf(Buf,"%s%s,%02X%02X", PtrStr, Operand,
              Code[*Len + 1], Code[*Len]);
      *Len += 2;
   }
   else {                    /* byte data */
      sprintf(Buf,"%s%s,%02X", PtrStr, Operand, Code[*Len]);
      if (Code[0] == 0xC6)   /* MOV */
         AddDataInfo(Buf, Code[*Len]);
      (*Len)++;
   }
}

/***************************************************************************
**  AAM/AAD
**  obj codes: | ** *** ***|   data    |
****************************************************************************/
PUBLIC RETCODE Dtype_X_data(U8 *Code,  STR Buf, int *Len)
{
   *Len = 2;
   if (Code[1] == 0x0A)   /* exact AAM or AAD */
      Buf[0] = '\0';
   else
      sprintf(Buf,"%02X", Code[1]);
}

/***************************************************************************
**  conditional branch
**  obj codes: | ** *** ***|   disp    |
****************************************************************************/
PUBLIC RETCODE Dtype_X_disp(PCTYPE PC, U8 *Code,  STR Buf, int *Len)
{
char Symbol[40];

   *Len = 2;
   dsmAddr = PC + *Len + (int) ((char)Code[1]);
   dsmAddrGiven = TRUE;
   /* try converting address into symbol */
   ConvertAddr(dsmAddr,  Buf);
}

/***************************************************************************
**  RET/RETF data
**  obj codes: | ** *** ***| data_low  | data_high |
****************************************************************************/
PUBLIC RETCODE Dtype_X_ldata_hdata(U8 *Code,  STR Buf, int *Len)
{
   *Len = 3;
   sprintf(Buf,"%02X%02X", Code[2], Code[1]);
}

/***************************************************************************
**  ENTER data,data
**  obj codes: | ** *** ***| data_low  | data_high |   data    |
****************************************************************************/
PUBLIC RETCODE Dtype_X_ldata_hdata_data(U8 *Code,  STR Buf, int *Len)
{
   *Len = 4;
   sprintf(Buf,"%02X%02X,%02X", Code[2], Code[1], Code[3]);
}

/***************************************************************************
**  CALL address
**  obj codes: | ** *** ***| disp_low  | disp_high |
****************************************************************************/
PUBLIC RETCODE Dtype_X_ldisp_hdisp(PCTYPE PC, U8 *Code,  STR Buf, int *Len)
{
   *Len = 3;
   dsmAddr = PC + *Len + (int) (Code[1]+Code[2]*256);
   dsmAddrGiven = TRUE;
   /* try converting address into symbol */
   ConvertAddr(dsmAddr,  Buf);
}

/***************************************************************************
**  CALL/JMP segment:offset
**  obj codes: | ** *** ***|offset_low |offset_high|  seg_low  | seg_high  |
****************************************************************************/
PUBLIC RETCODE Dtype_X_loff_hoff_lseg_hseg(U8 *Code,  STR Buf, int *Len)
{
long Segment, Offset;

   *Len = 5;
   Segment = ((long)Code[4] << 8) + (long)Code[3];
   Offset  = ((long)Code[2] << 8) + (long)Code[1];
   dsmAddr = (Segment << 4) + Offset;
   dsmAddrGiven = TRUE;
   /* try converting address into symbol */
   if (dsmAddr2Sym((Segment << 16) | Offset,  Buf) == FALSE)
      sprintf(Buf,"%02X%02X:%02X%02X", Code[4], Code[3], Code[2], Code[1]);
}

/***************************************************************************
**  BOUND/LEA/LES/LDS reg,mem
**  obj codes: | ** *** ***|mod reg r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_X_modregrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_X_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s,%s", RegTable[ 8 + REGID(Code[1]) ], Operand);
}

/***************************************************************************
**  PUSH/POP/CALL/JMP reg/mem
**  obj codes: | ** *** ***|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_X_modXrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];
int ID;

   Proc_X_modXrm(Code,  Operand, Len);
   ID = Code[1] & 0x38;
   if ((Code[0] == 0xFE || Code[0] == 0xFF) && (ID == 0x18 || ID == 0x28))
      sprintf(Buf, "FAR %s", Operand);
   else
      strcpy(Buf, Operand);
}

/***************************************************************************
**  ESC data,reg/mem
**  obj codes: | 11 011 ***|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCxxx_modXrm(U8 *Code,  STR Buf, int *Len)
{
int ESCnum;
char Operand[80];

   ESCnum = ((Code[0] & 0x07)<<3) | (Code[1] & 0x38)>>3;
   Proc_Xw_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%02X,%s", ESCnum, Operand);
}

/***************************************************************************
**  operands: ST,ST(i)  /  ST(i),ST
**  obj codes: | 11 011 d**| 11 *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCdxx_11xxxrm(U8 *Code,  STR Buf, int *Len)
{
   *Len = 2;
   if (Code[0] & 0x04)   /* destination is ST(i) */
      sprintf(Buf,"%s,ST", STtable[ RMID(Code[1]) ]);
   else
      sprintf(Buf,"ST,%s", STtable[ RMID(Code[1]) ]);
}

/***************************************************************************
**  operands: QWORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 q**|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCqxx_modXrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_X_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s PTR %s",
           Code[0] & 0x04 ? PtrTable[QWORDPTR]: PtrTable[DWORDPTR], Operand);
}

/***************************************************************************
**  operands: WORD PTR mem / DWORD PTR mem
**  obj codes: | 11 011 w**|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCwxx_modXrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_X_modXrm(Code,  Operand, Len);
   sprintf(Buf,"%s PTR %s",
           Code[0] & 0x04 ? PtrTable[WORDPTR]: PtrTable[DWORDPTR], Operand);
}

/***************************************************************************
**  operands: mem / TBYTE PTR mem
**  obj codes: | 11 011 *t*|mod *** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCxtx_modXrm(U8 *Code,  STR Buf, int *Len)
{
char Operand[80];

   Proc_X_modXrm(Code,  Operand, Len);
   if (Code[0] & 0x02)   /* TBYTE PTR */
      sprintf(Buf,"%s PTR %s", PtrTable[TBYTEPTR], Operand);
   else
      strcpy(Buf, Operand);
}

/***************************************************************************
**  operands: ST(i)
**  obj codes: | 11 011 ***| 11 0** r/m|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCxxx_110xxrm(U8 *Code,  STR Buf, int *Len)
{
   *Len = 2;
   strcpy(Buf, STtable[ RMID(Code[1]) ]);
}

/***************************************************************************
**  FPinst with no operands
**  obj codes: | 11 011 ***| ** *** ***|
****************************************************************************/
PUBLIC RETCODE Dtype_ESCxxx_X(U8 *Code,  STR Buf, int *Len)
{
   *Len = 2;
   Buf[0] = '\0';
}


/***************************************************************************
**               Procedures used by type processing routines
****************************************************************************/

/***************************************************************************
**
**  Name: Proc_Xw_modXrm(Code,  Operand, Len)
**
**  Description:
**     derive operand according to the codes: xxxxxxxw mod_xxx_rm
**
**  Input:  code data
**
**  Output: operand, number of bytes involved
**
****************************************************************************/
PRIVATE Proc_Xw_modXrm(U8 *Code,  STR Operand, int *Len)
{
int mod, rm, wreg;
char Symbol[50], DispStr[20];

   *Len = 2;               /* 2 bytes at least */
   mod = MODID(Code[1]);
   rm  = RMID(Code[1]);
   switch (mod) {
   case 0:                 /* no displacement  memory mode */
      if (rm == 6) {       /* direct address */
         *Len += 2;
         if (dsmAddr2Sym(Code[2] + Code[3]*256L,  Symbol))
            sprintf(Operand,"[%s]", Symbol);
         else
            sprintf(Operand, "[%02X%02X]", Code[3], Code[2]);
      }
      else
         sprintf(Operand,"[%s]", MemTable[rm]);
      break;
   case 1:                 /* 1 displacement byte  memory mode */
      (*Len)++;
      SignbyteToStr(Code[2], DispStr);
      sprintf(Operand,"[%s%s]", MemTable[rm], DispStr);
      break;
   case 2:                 /* 2 displacement bytes  memory mode */
      *Len += 2;
      sprintf(Operand,"[%s+%02X%02X]", MemTable[rm], Code[3],Code[2]);
      break;
   case 3:                 /* register mode */
      wreg = ((Code[0] & 0x01)<<3) | rm;
      strcpy(Operand, RegTable[wreg]);
      break;
   }  /* end of switch */
   if (mod != 3)   /* not register */
      ChkSegOverride(Operand);
}  /* end of Proc_Xw_modXrm() */

/***************************************************************************
**
**  Name: Proc_X_modXrm(Code,  Operand, Len)
**
**  Description:
**     derive operand according to the codes: xxxxxxxx mod_xxx_rm
**
**  Input:  code data
**
**  Output: operand, number of bytes involved
**
****************************************************************************/
PRIVATE Proc_X_modXrm(U8 *Code,  STR Operand, int *Len)
{
U8 Save;

   Save = Code[0];
   Code[0] |= 0x01;   /* force to set word ptr */
   Proc_Xw_modXrm(Code,  Operand, Len);
   Code[0] = Save;
}  /* end of Proc_X_modXrm() */

/***************************************************************************
**
**  Name: GetPtr(Code,  PtrStr)
**
**  Description:
**     derive access ptr according to the codes: xxxxxxxw mod_xxx_rm
**
**  Input: code data
**
**  Output: "BYTE PTR" or "WORD PTR" or ""
**
****************************************************************************/
PRIVATE GetPtr(U8 *Code,  STR PtrStr)
{
   if (MODID(Code[1]) == 3)   /* register mode */
      *PtrStr = '\0';
   else
      sprintf(PtrStr,"%s PTR ", PtrTable[ Code[0] & 0x01 ]);
}  /* end of GetPtr() */

/***************************************************************************
**
**  Name: SignbyteToStr(Data,  Buf)
**
**  Description:
**     convert signed byte data to hex-number string
**
**  Input:  byte data
**
**  Output: hex-number string
**
****************************************************************************/
PRIVATE SignbyteToStr(U8 Data,  STR Buf)
{
   if (Data < 0x80)   /* positive */
      sprintf(Buf,"+%02X", Data);
   else               /* negative */
      sprintf(Buf,"-%02X", 256-Data);
}  /* end of SignbyteToStr() */

/***************************************************************************
**
**  Name: ChkSegOverride(mem)
**
**  Description:
**     add segment override of memory operand if necessary
**
**  Input:  memory operand
**
**  Output: memory operand added with segment prefix
**
****************************************************************************/
PRIVATE ChkSegOverride(char *mem)
{
char temp[20];

   if (dsmSegOverride != 0) {
      strcpy(temp, mem);
      sprintf(mem, "%s:%s", SegTable[SEGID(dsmSegOverride)], temp);
      dsmSegUsed = TRUE;
   }
}  /* end of ChkSegOverride() */

/***************************************************************************
**
**  Name: AddDataInfo(Buf, Data)
**
**  Description:
**     append information of data to buffer
**
**  Input:  buffer, data
**
**  Output: buffer with information
**
****************************************************************************/
PRIVATE AddDataInfo(STR Buf, U8 Data)
{
char temp[50];

   return;
   if (isprint(Data)) {
      strcpy(temp, Buf);
      if (strlen(temp) < 12)
         sprintf(Buf,"%-12s;'%c'", temp, Data);
      else
         sprintf(Buf,"%-28s;'%c'", temp, Data);
   }
}  /* end of AddDataInfo() */

/***************************************************************************
**
**  Name: ConvertAddr(Addr,  Buf)
**
**  Description:
**     convert branch address to ASCII string or symbol
**
**  Input:  address which is offset or absolute address
**
**  Output: in buffer
**
****************************************************************************/
PRIVATE ConvertAddr(long Addr,  STR Buf)
{
U32 SymAddr;

   SymAddr = ((U32)dsmCS << 16) + ((U32)Addr & 0x0000FFFFL);
   if (dsmAddr2Sym(SymAddr,  Buf) == FALSE)
      if (dsmIsAbsAddr)
         sprintf(Buf,"%05lX", dsmAddr & 0x000FFFFFL);
      else
         sprintf(Buf,"%04lX", dsmAddr & 0x0000FFFFL);
}  /* end of ConvertAddr() */

/***************************************************************************
**
**  Name: dsmAddr2Sym(Addr,  Symbol)
**
**  Description:
**     convert address to symbol
**
**  Input:  address which consists of segment and offset
**
**  Output: symbol
**
**  Return: TRUE if successful conversion, FALSE otherwise
**
****************************************************************************/
PUBLIC BOOLEAN dsmAddr2Sym(U32 Addr,  STR Symbol)
{
U32 absaddr;
int ItemNo;
int i;

   if (!sym_flag)  return (FALSE);
   absaddr = ((Addr >> 16) << 4) + (Addr & 0x0000FFFF);
   for (i = 0; i < 4; i++)
      comsym.addr[i] = (U8) ( (absaddr >> (8 * (3-i))) & 0x000000FF );
   comsym.mask = MSK_ABS;
   if (AddrToSym(&ItemNo) == FALSE)  return (FALSE);
   sprintf(Symbol,"%%%s",comsym.name);
   return (TRUE);
}  /* end of dsmAddr2Sym() */

/******************************* End of File *******************************/
