/******************************************************************************
** File name : sldext.c
**
** Description :  The routines consistant of :
**                . C expressions analysis.
**                   . Arithmetic calculating
**                   . Recognizes hexadecimal
**                   . Variable evaluation.
**                      . variables
**                      . pointers
**                      . members of array
**                      . members of structure, union, enumeration ...
**
** Copyright (C) 1994 Microtek International, Inc.
** All Rights Reserved
**
** Programmed by James Wang
**
****************************************************************************/

/***************************************************************************
**
**    Include files
**
***************************************************************************/
#include  "system.h"
#include  <stdlib.h>
#include  <errno.h>
#include  "usd3.h"
#include  "gblext.h"
#include  "usym1.h"
#include  "usym3.h"
#ifdef IEEE
#include  "reg68k.h"
#else
#include  "reg86.h"
#endif
#include  "database.h"
#include  "oldext.h"
#include  "cmdid.h"
#ifndef _SD_ABI_DEF_
#include "sdabidef.h"
#endif
#ifndef _LINUMDEF_
#include "linumdef.h"
#endif

/**************************************************************************
**
** Local Define
**
**************************************************************************/

#define LINE_BUF   512
#define TOKEN_BUF   80

// Token type definition
typedef enum {
   // Delimiter type
   DOT       = 0x2E,       //  '.'
   L_BRACKET = 0x5B,       //  '['
   R_BRACKET = 0x5D,       //  ']'
   L_PARENTHESIS = 0x28,   //  '('
   R_PARENTHESIS = 0x29,   //  ')'
   PLUS = 0x2B,            //  '+'
   DEC = 0x2D,             //  '-'
   MUL = 0x2A,             //  '*'
   DIV = 0x2F,             //  '/'
   MOD = 0x24,             //  '$'  replaced of '%'
   SYMBOL = 0x25,          //  '%'
   LOGIC_NEGATIVE = 0x21,  //  '!'
   BIT_AND = 0x26,         //  '&'
   BIT_OR = 0x7C,          //  '|'
   EXPONENT = 0x5E,        //  '^'
   BIT_NEGATIVE = 0x7E,    //  '~'
// L_ARRAW = 0x3C,         //  '<'
   R_ARRAW = 0x3E,         //  '>'
   UNDERSCORE = 0x5F,      //  '_'
   /*-----------------------------*/
// PLUSPLUS,               //  "++"
// DECDEC,                 //  "--"
// LOGIC_AND,              //  '&&'
// LOGIC_OR,               //  "||"
// SHIFT_LEFT,             //  "<<"
// SHIFT_RIGHT,            //  ">>"
   VARIABLE = 0xFE,
   CONSTANT = 0xFF,
} TOKEN_TYPE;

// storages sub informations such as structure item's name, dimenssion pos.
typedef struct {
   U8    ItemName[40];
   S16   DimLen;
} SUBINFO;

// for return level
typedef enum {
   _BYTE  =  1,
   _WORD  =  2,
   _DWORD =  4,
   _SINGLE,
   _DOUBLE = 8,
} RET_LEVEL;

// access structure mode.
typedef enum {
   NO_ACCESS,
   ACC_BY_ADDR,
   ACC_BY_POINTER,
} ACC_MODE;

#define   WHITE_SPACE    0x20
#define   TAB_KEY        0x09
#define   ON             1
#define   OFF            0

#define   isOperator(x)  ( x == PLUS || x == DEC || x == MUL || x == DIV   \
                || x == MOD || x == EXPONENT || x == L_PARENTHESIS || x == R_PARENTHESIS )
#define   isAlpha(x)     ((x >= 0x61 && x <= 0x7A) || (x >= 0x41 && x <= 0x5A)) ? 1 : 0
#define   isDigit(x)     ( x >= 0x30 && x <= 0x39) ? 1 : 0
#define   isHexAlpha(x)  ((x >= 0x61 && x <= 0x66) || (x >= 0x41 && x <= 0x46)) ? 1 : 0
#define   isDot(x)       ( x == DOT ) ? 1 : 0
#define   isSymbolic(x)  ( x == SYMBOL ) ? 1 : 0
#define   isUnderScore(x) ( x == UNDERSCORE ) ? 1 : 0

/**************************************************************************
**
** Global or Local Declarations
**
***************************************************************************/

LOCAL  U8          asc[LINE_BUF], *ptr, *tail, token[TOKEN_BUF], val[80];
LOCAL  TOKEN_TYPE  token_type;
LOCAL  RET_LEVEL   ret_level, old_level = _BYTE, cur_level;
LOCAL  S16         hex_flag = OFF, hex_length = 0, var_len = 0;
LOCAL  SUBINFO     subitem;
LOCAL  ACC_MODE    acc_mode = NO_ACCESS;
GLOBAL double      sum, maxdouble = 1e+208;

/**************************************************************************
**
** Prototype Declarations
**
***************************************************************************/

/**************************************************************************
**
** External Declarations
**
***************************************************************************/
EXTERN double Str2Float(U8 *);
EXTERN double Str2Double(U8 *);

/**************************************************************************
**
** Execution codes ( Public )
**
***************************************************************************/
/***************************************************************************
**
**  Name: MathCmd()
**
**  Description: The routine for calculating C expressions.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PUBLIC RETCODE MathCmd(VOID)
{

   if (cmd_syntax.argc < 2)   return(FAIL);
   if (strlen(cmd_syntax.asc) > LINE_BUF)
   {
      DisplayStr("Expression length is too long !\r\n");
      return(FAIL);
   }
   strcpy(asc, cmd_syntax.asc);
   tail = &asc[strlen(asc)-1];
   ptr = asc;
   memset(token, NULL, sizeof(token));
   if (Cparser() == FAIL) return (FAIL);
   return(OK);

}

/**************************************************************************
**
** Execution codes ( Private )
**
***************************************************************************/
/***************************************************************************
**
**  Name: Cparser()
**
**  Description: The routine for analyzing C expressions.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE Cparser(VOID)
{
U8       tmpBuf[80];
S16      ret;

   hex_length = 0;
   if ( CheckParenthesis() != OK )   {
      err_exit(1, "");
      return(FAIL);
   }
   if ( (ret = expr2(&sum) ) == OK)   {
      // print expression value.
      sum = (double)sum;
      sprintf(tmpBuf,">%lg\r\n",sum);
      DisplayStr(tmpBuf);
      return (OK);
   }
   else
      return(FAIL);

}
/***************************************************************************
**
**  Name: expr2()
**
**  Description: The routine for processing "+","-" operator in C expression
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE expr2(double *sum)
{
S16     ret;
double  temp;

   if (expr3(sum) != OK)
      return (FAIL);
   while (1) {
      if ((ret=get_token()) != OK)
         break;
      switch (token_type) {
         case PLUS:
               if (expr3(&temp) != OK) return(FAIL);
               *sum += temp;    break;
         case DEC:
               if (expr3(&temp) != OK) return(FAIL);
               *sum -= temp;    break;
         default:
               unget_token();
               return (ret);
           }
   }
   return (ret);

}
/***************************************************************************
**
**  Name: expr3()
**
**  Description: The routine for processing "*","/","$"
**               operator in C expression
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE expr3(double *sum)
{
S16     ret;
double  temp;

   if (expr31(sum) != OK)
      return (FAIL);
   while (1) {
      if ((ret=get_token()) != OK)
         break;
      switch (token_type) {
         case MUL:
               if (expr31(&temp) != OK) return(FAIL);
               if ( *sum >= maxdouble || temp >= maxdouble )
                  { err_exit(10,""); return(FAIL); }
               if (*sum == 0 || temp == 0)   *sum = 0;
               else  *sum *= temp;
               continue;
         case DIV:
               if (expr31(&temp) != OK) return(FAIL);
               if (temp ==  0)   { err_exit(5,""); return(FAIL); }
               *sum /= temp;    continue;
         case MOD:
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if (expr31(&temp) != OK) return(FAIL);
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if ( isFloat(*sum) || isFloat(temp) ) { err_exit(4,""); return(FAIL); }
               if (temp ==  0)   { err_exit(5,""); return(FAIL); }
               *sum = (S32)*sum%(S32)temp;    continue;
         default:
               unget_token();
               return(ret);
            }
      }
   return (ret);
}

/***************************************************************************
**
**  Name: expr31()
**
**  Description: The routine for processing "|","&","^"
**               operator in C expression
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE expr31(double *sum)
{
S16     ret;
double  temp, ex;

   if (expr4(sum) != OK)
      return (FAIL);
   while (1) {
      if ((ret=get_token()) != OK)
         break;
      switch (token_type) {
         case BIT_AND:
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if (expr4(&temp) != OK) return(FAIL);
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if ( isFloat(*sum) || isFloat(temp) ) { err_exit(4,""); return(FAIL); }
               if (*sum < 0 && temp < 0)  *sum = (S32)*sum&(S32)temp;
               else *sum = (U32)*sum&(U32)temp;
               continue;
         case BIT_OR:
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if (expr4(&temp) != OK) return(FAIL);
               if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
               if ( isFloat(*sum) || isFloat(temp) ) { err_exit(4,""); return(FAIL); }
               if (*sum < 0 || temp < 0 ) *sum = (S32)*sum|(S32)temp;
               else *sum = (U32)*sum|(U32)temp;
               continue;
         case EXPONENT:
               if (expr4(&temp) != OK) return(FAIL);
               if (old_level > _BYTE || temp < 0) {err_exit(6,"");return(FAIL);}
               if ( isFloat(temp) ) { err_exit(4,""); return(FAIL); }
               if (temp == 0) { *sum = 1; continue; }
               ex = *sum;
               while (--temp)
                 if ( (*sum *= ex) >= maxdouble ) { err_exit(10,""); return(FAIL); }
               continue;
         default:
               unget_token();
               return(ret);
            }
      }
   return (ret);
}

/***************************************************************************
**
**  Name: expr4()
**
**  Description: The routine for processing unary "-","+","!", "~",
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE expr4(double *sum)
{
S16        ret;
double     temp;
TOKEN_TYPE type;

     if ((ret = get_token()) != OK) return (FAIL);
     if ( (type = token_type) == PLUS || type == DEC ||
            type == LOGIC_NEGATIVE || type == BIT_NEGATIVE) {
         if ((ret = get_token()) != OK) return(FAIL);
         switch (type) {
            case  PLUS: if (expr5(&temp) != OK) return (FAIL);
                        *sum = temp;   break;
            case  DEC:  if (expr5(&temp) != OK) return (FAIL);
                        *sum = -temp;  break;
            case  LOGIC_NEGATIVE:
                        if (expr5(&temp) != OK) return (FAIL);
                        if (!temp) *sum = 1; else *sum = 0; break;
            case BIT_NEGATIVE:
                        if (expr5(&temp) != OK) return (FAIL);
                        if (old_level > _DWORD) { err_exit(4,""); return(FAIL); }
                        if ( isFloat(temp) ) { err_exit(4,""); return(FAIL); }
                        *sum = -temp-1; break;
            default:    err_exit(0,""); break;
                  }
         } else {
               if ((ret = expr5(&temp)) != OK) return (FAIL);
               *sum = temp;  return (ret);
            }
    return(ret);
}
/***************************************************************************
**
**  Name: expr5()
**
**  Description: The routine for processing parethesis.
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE expr5(double *sum)
{
S16     ret;
double  temp;

     if (token_type == L_PARENTHESIS) {
         if ((ret = expr2(sum)) != OK) return (FAIL);
         if ((ret = get_token()) != OK || token_type != R_PARENTHESIS)
               return (FAIL);
         } else {
         unget_token();
         if ((ret = atom(&temp)) != OK) return (FAIL);
         *sum = temp;
            }
     return (ret);

}

/***************************************************************************
**
**  Name: atom()
**
**  Description: The routine for processing the atom.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE atom(double *sum)
{
S16     ret;
double  temp;

     if ((ret = get_token()) != OK) return (FAIL);
     if (token_type == CONSTANT) *sum = atof(token);
      else if (token_type == VARIABLE) {
#ifdef IEEE
             if (fetch_var_695(token) != OK)
#else
             if (fetch_var_86(token) != OK)
#endif
                   return (FAIL);
             *sum = atof(val);
             ChgLev();
         }
         else { err_exit(0, ""); return(FAIL); }
     hex_length = 0;
     return (ret);

}
/***************************************************************************
**
**  Name: GetSubName()
**
**  Description: The routine for getting a sub item's name.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE GetSubName()
{
S16  i, do_flag = 0;

   if ( ptr < tail  && *ptr == DOT) {
      ptr++; var_len++;
      acc_mode = ACC_BY_ADDR; do_flag = 1;
   }
   else if ( ptr < tail && *ptr==DEC && *(ptr+1)==R_ARRAW ){
        ptr += 2; var_len += 2;
        acc_mode = ACC_BY_POINTER, do_flag = 1;
   }
   if (do_flag) {
        if ( !isAlpha(*ptr) ) { err_exit(0,""); return(FAIL); }
        i = 0;
        do {
         subitem.ItemName[i++] = *ptr++;
         var_len++;
        } while ( isAlpha(*ptr) || isDigit(*ptr) || isUnderScore(*ptr) );
        subitem.ItemName[i] = '\0';
     }
   return(OK);
}
/***************************************************************************
**
**  Name: GetDimLen()
**
**  Description: The routine for getting dimession's length.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE GetDimLen()
{
S16 i=0, ret, dim;
U8  tmpBuf[80];

   if ( ptr < tail )
   if ( *ptr == L_BRACKET ) {
      ptr++;
      var_len++;
      if ( !isDigit(*ptr) )  { err_exit(0, ""); return(FAIL); }
      while (isDigit(*ptr)) {
      tmpBuf[i++] = *ptr++;
      var_len++;
      }
      tmpBuf[i] = '\0';
      dim = abs((S16)atoi(tmpBuf));
      subitem.DimLen = dim;
      if ( *ptr == R_BRACKET ) { ptr++; var_len++; }
      else { err_exit(0, ""); return(FAIL); }
   }
   return(OK);

}
/***************************************************************************
**
**  Name: get_token()
**
**  Description: The routine for getting a token from input stream.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE get_token( VOID )
{
U8    ch, *tmp;
S16   i = 0, HexCnt = 0;

   memset(token, NULL, sizeof(token));
   if (!ptr) return (FAIL);
   if (*ptr == NULL) {   // Is normal finished ?
       if (token_type == VARIABLE || token_type == CONSTANT
         || token_type == R_PARENTHESIS || token_type == R_BRACKET)
         return (OK);
       else {
         err_exit(0,"");
         return (FAIL);
          }
       }
   while ((*ptr == WHITE_SPACE) || (*ptr == TAB_KEY))   ptr++;

   if ((tmp = strchr("[]()+-*/$!&|^~", *ptr)) != NULL) {
      token[0] = ch = *tmp;
      token[1] = '\0';
      switch (ch) {
         case PLUS: token_type = PLUS;
                    ptr++;    return (OK);
         case  DEC: token_type = DEC;
                    ptr++;    return (OK);
         case  MUL: token_type = MUL;
                    ptr++;    return (OK);
         case  DIV: token_type = DIV;
                    ptr++;    return (OK);
         case  MOD: token_type = MOD;
                    ptr++;    return (OK);
         case  LOGIC_NEGATIVE:
                    token_type = LOGIC_NEGATIVE;
                    ptr++;    return (OK);
         case  BIT_NEGATIVE:
                    token_type = BIT_NEGATIVE;
                    ptr++;    return (OK);
         case  BIT_AND:
                    token_type = BIT_AND;
                    ptr++;    return (OK);
         case  BIT_OR:
                    token_type = BIT_OR;
                    ptr++;    return (OK);
         case  EXPONENT:
                    token_type = EXPONENT;
                    ptr++;    return (OK);
         case  L_BRACKET:
                    token_type = L_BRACKET;
                    ptr++;    return (OK);
         case  R_BRACKET:
                    token_type = R_BRACKET;
                    ptr++;    return (OK);
         case  L_PARENTHESIS:
                    token_type = L_PARENTHESIS;
                    if ( (ptr > asc) && ( isAlpha(*(ptr-1))||isDigit(*(ptr-1))
                     || isSymbolic(*(ptr-1)) ) )
                    {  err_exit(0,""); return(FAIL); }
                    ptr++;    return (OK);
         case  R_PARENTHESIS:
                    token_type = R_PARENTHESIS;
                    if ( (ptr > asc) && ( *(ptr+1) == L_PARENTHESIS ||
                        isAlpha(*(ptr+1)) || isDigit(*(ptr+1)) || isSymbolic(*(ptr+1))
                      || *(ptr+1) == LOGIC_NEGATIVE || *(ptr+1) == BIT_NEGATIVE))
                    {  err_exit(0,""); return(FAIL); }
                    ptr++;    return (OK);
         default:   return (FAIL);
            }  // end of switch
         }  // end of delimiter.
   else if ( isSymbolic(*ptr) ) {
         if ( !isAlpha(*(ptr+1)) ) { err_exit(0,""); return(FAIL); }
         // initializes var sub information
         memset(subitem.ItemName, NULL, sizeof(subitem.ItemName));
         subitem.DimLen = -1;
         acc_mode = NO_ACCESS;
         var_len = 0;
         // gets variable name.
         token[i++] = *ptr++;
         var_len++;
         while ( isAlpha(*ptr) || isDigit(*ptr) || isUnderScore(*ptr) ) {
            var_len++;
            token[i++] = *ptr++;
         }
         token[i] = '\0';
         if (GetSubName() != OK) return(FAIL);
         if (GetDimLen() != OK) return(FAIL);
         token_type = VARIABLE;
         if (isAlpha(*ptr) || isDigit(*ptr) || isDot(*ptr) || isSymbolic(*ptr)
            || *ptr == LOGIC_NEGATIVE || *ptr == BIT_NEGATIVE ) {  // previews next token.
             err_exit(0,"");
             return(FAIL);
         }
         if ( *ptr == L_BRACKET || *ptr == R_BRACKET )
            { err_exit(16,""); return(FAIL); }
//       ChgLev();
         return (OK);
         } else if (isDigit(*ptr) || isDot(*ptr)) {
            if (*ptr == '0') {
               ptr++;
               if ((ch = *ptr) == 'x' || ch == 'X') {  // Is HexDecimal number
                  ptr++;
                  hex_flag = ON;
                  }
               else // Is Decimal number
                  ptr--;
               }
            while ( isDigit(*ptr) || isDot(*ptr)
                  || ( hex_flag == ON && isHexAlpha(*ptr) ))
               token[i++] = *ptr++;
            token[i] = '\0';
            if (hex_flag == ON) {
               hex_flag = OFF;
               if (Hex2Dec(token) == FAIL)  return (FAIL);
               }
            else { // read exponent expression.
               if ( *ptr == 'E' || *ptr == 'e' ) {
                  token[i++] = *ptr++;
                  if ( ptr > tail ) { err_exit(0,""); return(FAIL); }
                  if ( *ptr != PLUS && *ptr != DEC && !isDigit(*ptr) )
                        { err_exit(0,""); return(FAIL); }
                  if ( (*ptr == PLUS || *ptr == DEC) && !isDigit(*(ptr+1)) )
                        { err_exit(0,""); return(FAIL); }
                  token[i++] = *ptr++;
                  while ( ptr <= tail && isDigit(*ptr) )
                     token[i++] = *ptr++;
                  token[i] = '\0';
               // check next token.
               if ( ptr <= tail && !isOperator(*ptr) )
                  { err_exit(0,""); return(FAIL); }
               }
            }
            token_type = CONSTANT;
            ChgLev();
            if (isAlpha(*ptr) || *ptr == LOGIC_NEGATIVE || isSymbolic(*ptr)
                  || *ptr == BIT_NEGATIVE ) {  // previews next token.
                err_exit(0,"");
                return(FAIL);
            }
            return (OK);
         } // end of digital.
         else {
            err_exit(0,"");
            return (FAIL);
            }
}

/****************************************************************************
**
**  Name: ChgLev()
**
**  Description: The routine for decision of current data level.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE ChgLev(VOID)
{
     /*----- decision data type level --------*/
     if (strchr(token,'.') != NULL)
         cur_level = _SINGLE;
     else
     {
         if ( strlen(token) < 4  && atol(token) <= 0XFF ) cur_level = _BYTE;
         else  if ( atol(token) <= 0XFFFF ) cur_level = _WORD;
         else  if ( atol(token) <= 0XFFFFFFFF ) cur_level = _DWORD;
         else  cur_level = _SINGLE;
     }
     ret_level = cur_level >= old_level? cur_level : old_level;
     old_level = cur_level;

}
/****************************************************************************
**
**  Name: unget_token()
**
**  Description: The routine for ungetting a token into asc.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE unget_token( VOID )
{
S16   i;

   if (hex_length != 0) {
      i = hex_length + 2;
      hex_length = 0;
   } else if (token_type == VARIABLE && var_len) {
      i = var_len;
      var_len = 0;
   } else
      i = strlen(token);
   while ( i-- > 0 )
      ptr--;
}
/****************************************************************************
**
**  Name: CheckParenthesis()
**
**  Description: The routine for checking if the parenthesis matched ?
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE CheckParenthesis( VOID )
{
S16   LparCnt = 0, RparCnt = 0;
U8    *CurPtr;

      CurPtr = ptr;
      while ( CurPtr <= tail ) {
         if ( *CurPtr == L_PARENTHESIS )
            LparCnt++;
         else if ( *CurPtr == R_PARENTHESIS )
                  RparCnt++;
         CurPtr++;
      }
      if ( LparCnt != RparCnt )
         return(FAIL);
      else
         return(OK);
}

/****************************************************************************
**
**  Name: isFloat()
**
**  Description: The routine for checking if the Decimal number is float ?
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE isFloat( double val )
{
U8 tmpBuf[80];

   sprintf(tmpBuf, "%lg", val);
   if ( strchr(tmpBuf,'.') != NULL && strchr(tmpBuf,'e') == NULL )
      return(YES);
   else
      return(NO);

}
/****************************************************************************
**
**  Name: Hex2Dec()
**
**  Description: The routine for translating a number from Hexdecimal
**               to Decimal.
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE Hex2Dec(U8 *buf)
{
U8    tmpBuf[80], hex = 16, *ptr, *buffer, pos;
static U8 hex_upper[16] = "0123456789ABCDEF";
static U8 hex_lower[16] = "0123456789abcdef";
U16    i, j, ignore = 0;
U32   tmpval = 0, bitval;

   if (strchr(buf,'.') != NULL)
   {
      err_exit(7,buf);
      return (FAIL);
   }
   if ((hex_length = strlen(buf)) == 0)
   {
      err_exit(8,buf);
      return (FAIL);
   }
   buffer = buf;
   i = 0;
   while ( i < hex_length ) {
      if ( *buffer++ == '0' )
         ignore++;
      else break;
      i++;
   }
   ignore = (ignore == hex_length) ? ignore-1 : ignore;
   if ( (hex_length-ignore) > 8) { err_exit(9,""); return(FAIL); }

   buffer = buf;
   for (i=hex_length; i>0; i--)
   {
      // find position.
      if ((ptr = strchr(hex_upper, *buffer)) != NULL)
         pos = ptr - hex_upper;
      else if ((ptr = strchr(hex_lower, *buffer)) != NULL)
         pos = ptr - hex_lower;
           else  err_exit(0,"");
      bitval = (U32)pos;
      for (j=i-1; j>0; j--)
         bitval *= hex;
      tmpval += bitval;
      buffer++;
   }
   sprintf(tmpBuf, "%lu", tmpval);
   strcpy(buf, tmpBuf);

   return (OK);
}
/****************************************************************************
**
**  Name: err_exit()
**
**  Description: The routine for processing error message.
**
**
**
**  Input:
**
**  Output:
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE err_exit(int errno, char *err_msg)
{
U8 tmpBuf[256];
static  U8 *err_str[] = {
            "Syntax error",                              // 0:
            "Unmatched parenthesis",                     // 1:
            "Unmatched bracketes",                       // 2:
            "Symbol not found",                          // 3:
            "Operand types incorrect for this operator", // 4:
            "Divided by zero ",                          // 5:
            "Exponent's type is unsigned char",          // 6:
            "Invalid float pointer number",              // 7:
            "Null hexdecimal number",                    // 8:
            "Hexdecimal number is too big",              // 9:
            "Floating point error ",                     // 10:
            "Structure or union member not found",       // 11:
            "Null member of structure or union ",        // 12:
            "Nested structure or union",                 // 13:
            "Symbol unknown type",                       // 14:
            "Array read error",                          // 15:
            "Changes into one dimension array",          // 16:
            "Null pointer"                               // 17:
            };

   sprintf(tmpBuf,">%s %s\r\n", err_msg, err_str[errno]);
   DisplayStr(tmpBuf);
}

#ifdef IEEE
//==================== IEEE 695 ============================================

// Find a symbol value : According to IEEE 695 object format.
// 1. Symbol is a local variable, a static variable or a global variable ?
// 2. The type of symbol is simple or multiple ?
//    (such as array, structure ... )

//==========================================================================

#define   MASK_STRUC 0x81
#define   MASK_UNION 0x82
#define   MASK_ENUM  0x84
#define   MASK_ARRAY 0x40
#define   MASK_POINT 0x20

typedef struct TYPINFO{
   U16 typeIdx;
   U8  type;
   S16 offset; // register index or offset
   S8  method; // variable attribute.
   U32 addr;
   U16 len;
   U16 dim_len;
   U8  ptr[10];
   U8  msg[256];
   struct TYPINFO *SubInfo[20];
   U8  ItemName[20][40];
   S16 ItemCnt;
   S16 PosOffset;
}TYPEINFO;
EXTERN TYPEINFO type_Info, *type_ptr;
EXTERN signed long direct_flag;
EXTERN S16      stack_size, short_flag;

/****************************************************************************
**
**  Name: fetch_var_695
**
**  Description: The routine for finding a variable value.
**
**
**
**  Input:   token buffer.
**
**  Output:  changed token buffer.
**           return value : OK - successly, FAIL - not found.
**
**  Notes:
**
****************************************************************************/
PRIVATE RETCODE fetch_var_695( VOID )
{
U8    flag;
U16   type, i, curMod, RegIdx;
S16   ret;
U32   curPC, curFP, addr, dummy, offset;

   emuGetReg( I68K_REG, REG_PC, &curPC);
   memset(&type_Info , NULL, sizeof( TYPEINFO ));
   memset(type_Info.msg, NULL, sizeof(type_Info.msg));
   memset(type_Info.ptr, NULL, sizeof(type_Info.ptr));
   type_ptr = &type_Info;

  /***********************************************************
   *
   *  Searching symbol in table & filling type infomation.
   *
   ***********************************************************/
   if ((ret = FindLocalVariable( 0, &token[1], &addr, &type, &dummy,
         &flag )) != OK) {
            strcpy( comsym.name, &token[1] );
            ret = MISS;
            for ( i=0; i<modcnt; i++ )
            if ( find_mod_byn(i) == FOUND) {
               curMod = i;
               ret += FOUND;
               if ( curPC >= gblmod[i].modStart && curPC <= gblmod[i].modEnd )
                  break;
               }
            if ( ret >= FOUND ) {
               Str2Abs695( comsym.addr, &(type_Info.addr) );
               dummy = gblmod[curMod].modStart;
               type = ( ((U16)comsym.type & 0X00FF) << 8 )
                        + ( ((U16)comsym.type & 0XFF00) >> 8 );
                     }
            else  { err_exit(3,""); return( FAIL ); }
      }
      if ( GetTypeInfo695( type, dummy ) == FAIL )
         { err_exit(3,""); return( FAIL ); }

     /*******************************************************
      *
      *  Getting address of symbol, if local variable.
      *
      *******************************************************/
      if (!type_Info.addr)  // register var or frame offset.
      {
         switch (type_ptr->method)
         {
            case  1: // REG_FP or REG_SSP? frame offset
                     if (stack_size > 0)
                        emuGetReg(I68K_REG, REG_FP, &addr);
                     else
                        emuGetReg(I68K_REG, REG_SSP, &addr);
                     if (type_ptr->offset <= 0x7f)
                        offset = (U32)type_ptr->offset;
                     else
                        offset = (0xffffff00 | (U32)type_ptr->offset);
                     addr += offset ;
                     break;
            case  2: case 10:
                     direct_flag = 1;
                     RegIdx = type_ptr->offset;
                     emuGetReg(I68K_REG, RegIdx, &addr); // register
                     break;
            default: err_exit(14,""); free_subitem(); return(FAIL);
         }
         addr = (U32)addr&0xFFFFFFFFL;
      }
      else addr = type_Info.addr;

     /*************************************************************
      *
      *  Fetching a symbol variable.
      *
      *************************************************************/
      // pointer index
      if ( (type_ptr->type & MASK_POINT) == MASK_POINT )
         find_var_index(&addr);

      // structure or union
      if ( ((type_ptr->type & MASK_STRUC) == MASK_STRUC) ||
            ((type_ptr->type & MASK_UNION) == MASK_UNION) )
      {
         if ( (acc_mode == ACC_BY_ADDR  && type_ptr->ptr[0] == '*')
            || (acc_mode == ACC_BY_POINTER && type_ptr->ptr[0] != '*') )
            { err_exit(4,""); return(FAIL); }
         if ( subitem.ItemName[0] == '\0' ) // NULL item.
            { err_exit(12,""); free_subitem(); return(FAIL); }
         for (i=0; i<type_ptr->ItemCnt; i++)
         {
            if (( caseFlag && (!strcmp(subitem.ItemName, type_ptr->ItemName[i])) )
              || (!caseFlag && (!strcmpi(subitem.ItemName, type_ptr->ItemName[i])) ) )
            {
               addr += type_ptr->SubInfo[i]->PosOffset;
               type_ptr = type_ptr->SubInfo[i];
               break;
            }
         }
         if (type_ptr == &type_Info)  // subItem not found
            { err_exit(11,""); return(FAIL); }
         /*------------ sub item index --------------------*/
         if ((type_ptr->type & MASK_POINT) == MASK_POINT )
            find_var_index(&addr);
         /*------------ nested structure ------------------*/
         if (((type_ptr->type & MASK_STRUC) == MASK_STRUC) ||
               ((type_ptr->type & MASK_UNION) == MASK_UNION) )
            { err_exit(13,""); free_subitem(); return(FAIL); }
         /*------------ array element ---------------------*/
         if ((type_ptr->type & MASK_ARRAY) == MASK_ARRAY) {
            if (fetch_var_array(addr, direct_flag ) != OK)
               { err_exit(15,""); free_subitem(); return(FAIL); }
            }
         else if ( fetch_var_atom(addr, direct_flag ) != OK )
               { err_exit(3,""); free_subitem(); return(FAIL); }
      direct_flag = 0;
      free_subitem();
      return( OK );
      }  // end of structure

      // Array data
      if ((type_ptr->type & MASK_ARRAY) == MASK_ARRAY) {
         if ( fetch_var_array(addr, direct_flag ) != OK )
         {  err_exit(15,""); return(FAIL); }
         direct_flag = 0;
         return(OK);
      }
      // Simple type
      if (fetch_var_atom(addr, direct_flag) != OK) return(FAIL);
      direct_flag = 0;
      return(OK);

}
/***************************************************************************
**
**  Name: fetch_var_array()
**
**  Description:  for IEEE-695 format.
**
**
**  Input:
**
**  Output:
**
**  Return:
**
****************************************************************************/
LOCAL RETCODE fetch_var_array( U32 addr, S16 flag)
{
S16 i;

      if ( subitem.DimLen > type_ptr->dim_len )
         return(FAIL);
      for (i=0; i<=type_ptr->dim_len; i++)
      {
         if ( i == subitem.DimLen )  {
            if (fetch_var_atom(addr, flag) != OK) return (FAIL);
            break;
         }
         else {
            if ( type_ptr->len == 1 ) addr++;
            else if ( type_ptr->len == 8 ) addr +=8; else addr += 4;
         }
      }
      if (short_flag) short_flag = 0;
      return(OK);
}
/***************************************************************************
**
**  Name: fetch_var_atom()
**
**  Description:  for IEEE-695 format.
**
**
**  Input:
**
**  Output:
**
**  Return:
**
****************************************************************************/
LOCAL RETCODE fetch_var_atom( U32 addr, S16 flag)
{
U8     var_u8, arr[8], tmp[8];
S8     var_s8;
U32    var_u32;
U16    var_u16;
S16    var_s16, i;
S32    var_s32;
float  var_float;
double var_double;
long double var_long_double;

      if (!flag)
         emuGetMemN(addr, arr, 8);
      else
      {  // directly register number.
         memcpy(tmp, &addr, 8);
         if (type_ptr->len == 1)
            arr[0] = tmp[0];
         else if (type_ptr->len == 8)
                for (i=0; i<8; i++)
                  arr[7-i] = tmp[i];
              else
                  for (i=0; i<4; i++)
                     arr[3-i] = tmp[i];
      }

      switch (type_ptr->len)
      {
         case 1: if ((type_ptr->type & SCHAR) == SCHAR)
                 {
                     var_s8 = (S8)arr[0];
                     sprintf( val, "%d", var_s8);
                 }
                 else
                 {
                     var_u8 = (U8)arr[0];
                     sprintf( val, "%u", var_u8);
                 }
                 break;
         case 2: if ((type_ptr->type & UNINT) == UNINT)
                 {
                     if (short_flag)
                        var_u16 = ((U16)arr[0] << 8) + (U16)arr[1];
                     else
                        var_u16 = ((U16)arr[2] << 8) + (U16)arr[3];
                     sprintf( val, "%u", var_u16);
                 }
                 else
                 {
                     if (short_flag)
                        var_s16 = ((S16)arr[0] << 8) + (S16)arr[1];
                     else
                        var_s16 = ((S16)arr[2] << 8) + (S16)arr[3];
                     sprintf( val, "%d", var_s16);
                 }
                 break;
         case 4: if ((type_ptr->type & SINGLE) == SINGLE)
                 {
                     var_float = Str2Float(arr);
                     sprintf( val, "%g", var_float);
                 }   else
                 if ((type_ptr->type & UNLONG) == UNLONG)
                 {
                     var_u32 = ((U32)arr[0]<<24) + ((U32)arr[1]<<16)
                        + ((U32)arr[2]<<8) + (U32)arr[3];
                     sprintf( val, "%lu", var_u32);
                 }   else
                 if ((type_ptr->type & SLONG) == SLONG)
                 {
                     var_s32 = ((S32)arr[0]<<24) + ((S32)arr[1]<<16)
                        + ((S32)arr[2]<<8) + (S32)arr[3];
                     sprintf( val, "%ld", var_s32);
                 }
                 break;
         case 8: if ((type_ptr->type & DOUBLE) == DOUBLE)
                 {
                     var_double = Str2Double(arr);
                     sprintf( val, "%lg", var_double);
                 }
                 break;
        default: return (FAIL);
      }
      if (short_flag) short_flag = 0;
      return(OK);

}
/***************************************************************************
**
**  Name: free_subitem()
**
**  Description:  The routine for free all sub type information.
**
**
**  Input:
**
**  Output:
**
**  Return:
**
****************************************************************************/
LOCAL RETCODE free_subitem( VOID )
{
S16   i;

   for (i=0; i<type_Info.ItemCnt; i++)
      if ( type_Info.SubInfo[i] )    free(type_Info.SubInfo[i]);

}

#else
//  OMF86 format ...
EXTERN TYPE_INFO typeInfo;
EXTERN ARRAY_INFO arrInfo;
EXTERN STRUCT_INFO struInfo;
EXTERN STR typeArr[];
LOCAL U8 scalarLen[] = { 0, 8, 8, 16, 16, 32, 32, 32, 64, 80 };
/****************************************************************************
**
**  Name: fetch_var_86
**
**  Description: The routine for finding a variable value.
**
**
**
**  Input:   token buffer.
**
**  Output:  changed token buffer.
**           return value : OK - successly, FAIL - not found.
**
**  Notes:
**
****************************************************************************/
LOCAL RETCODE fetch_var_86( VOID )
{
U16 level, type;
U32 addr, dummy, u32,len, new;
U8 flag, name[40];
S16 ret, i, j;
U8  *buffer=NULL, ptr[80], temp[100], c, *tmpPtr, str[80], dim[40], arr[4];
STRUCT_INFO *tmpStruct;

   /*****************************************************************
    *
    *  A variable from token found in local or global.
    *
    *****************************************************************/
    strcpy( name, &token[1]);
    if ((ret = FindLocalVariable(0, &token[1], &addr, &type, &dummy,
          &flag)) != OK) {
        strcpy( comsym.name, name );
        if ( SymToAddr() == FOUND ) {
           addr = 0L;
           for (addr = 0, i = 0; i < 4; i++) addr = (addr << 8) + comsym.addr[i];
           type = comsym.type;
           ret = OK;
        }
        else if (ret != 1) {
           err_exit(3,"");
           return(FAIL);
        }
   }
  /*******************************************************
   *
   *  Initializing of typeInfo, arrayInfo, struInfo &
   *  getting typeInfo.
   *
   *******************************************************/
   memset( &typeInfo , NULL, sizeof( TYPE_INFO ));
   memset( &arrInfo , NULL, sizeof( ARRAY_INFO ));
   memset( &struInfo , NULL, sizeof( STRUCT_INFO ));
   if (GetTypeInfo( type ) == FAIL) { err_exit(3,""); return(FAIL); }
   else {
  /********************************************************
   *
   *   Change OMF86 PrintVar() old version to ...
   *
   ********************************************************/
   if (((c=typeInfo.type) < UNCHAR && c > LDOUBLE) && !typeInfo.scope) {
      len = typeInfo.len / 8;
      if ((buffer= malloc(len)) == NULL) return(FAIL);
      if (len > 0 && emuGetMemN( addr, buffer, len) != OK) {
         free(buffer);
         err_exit(4,"");
         return(FAIL);
      }
   }
   memset(ptr, NULL, sizeof(ptr) );
   if (typeInfo.scope) {
      new = addr;
      for (i=0; i < typeInfo.scope; i++) {
         ptr[i] = '*';
         len = typeInfo.ptrLen[i]/8;
         if ((len != sizeof(U32) && len != sizeof(U16)) ||
             emuGetMemN( new, arr, len) != OK) return(FAIL);
         if (len < sizeof(U32)) {
            emuGetReg(I86_REG, DS, &u32);
            new =  ((u32&0x0FFFFL) << 16) + *(U16 *)&arr[0];
         }
         else new = *(U32 *)&arr[0];
         if (!new) break;
      }
      if (!new)
         { err_exit(17,""); return(FAIL); }
      addr = new;
   }
   switch (typeInfo.type) {
     case  STRUCTURE:
        if ( (acc_mode == ACC_BY_ADDR  && typeInfo.ptrLen[0] == '*')
           || (acc_mode == ACC_BY_POINTER && typeInfo.ptrLen[0] != '*') )
           { err_exit(4,""); return(FAIL); }
        if ( subitem.ItemName[0] == '\0' ) // NULL item.
           { err_exit(12,""); return(FAIL); }
        if ((tmpStruct=malloc(sizeof(STRUCT_INFO))) == NULL) {
           free(buffer);
           return(FAIL);
        }
        memcpy( tmpStruct, &struInfo, sizeof( STRUCT_INFO ) );
        if (typeInfo.scope) {
           err_exit(17,"");
           return(FAIL);
        }
        for ( i = 0; i < tmpStruct->itemCnt; i++ ) {
         memset( &typeInfo , NULL, sizeof( TYPE_INFO ));
         memset( &arrInfo , NULL, sizeof( ARRAY_INFO ));
         memset( &struInfo , NULL, sizeof( STRUCT_INFO ));
         if (GetTypeInfo( tmpStruct->itemType[i] ) == FAIL) {
         if (tmpStruct->itemType[i]) free( tmpStruct->itemName[i] );
         for (j=0; j < struInfo.itemCnt; j++)
            if (struInfo.itemName[j]) free(struInfo.itemName[j]);
         }
         if (( caseFlag && (!strcmp(subitem.ItemName, tmpStruct->itemName[i])) )
           || (!caseFlag && (!strcmpi(subitem.ItemName, tmpStruct->itemName[i])) ) )
            break;
         len = (typeInfo.len/8);
         addr += len;
        }
        if ( i == tmpStruct->itemCnt ) { err_exit(11,""); return(FAIL); }
        if ( typeInfo.type == STRUCTURE ) { err_exit(13,""); return(FAIL); }
        else if ( typeInfo.type == ARRAY ) {
         len = typeInfo.len / 8;
         strcpy( name, subitem.ItemName );
         for (i=0, new=1; i < arrInfo.dim; i++)
            new *= arrInfo.size[i];
         if (Omf86Array(&addr, 0, temp, name, typeInfo, arrInfo, struInfo, len, new)
            != OK) {
            free(buffer);
            return(FAIL);
           }
         }
         else
         if ( typeInfo.type >= UNCHAR && typeInfo.type <= LDOUBLE) {
            if ( Omf86Atom(FALSE, FALSE, addr, typeInfo.type, typeInfo.len,
                   str)==OK)
                   strcpy(val, str);
            else {
                 free(buffer);
                 return(FAIL);
            }
         }
        // free typeInfo, strucInfo, tmpStruct ...
        for (j=0; j < struInfo.itemCnt; j++)
            if (struInfo.itemName[j]) free(struInfo.itemName[j]);
        struInfo.itemCnt = 0;
        for (i=0; i < tmpStruct->itemCnt; i++)
            free( tmpStruct->itemName[i] );
        free(tmpStruct);
        break;
     case ARRAY:
        memset( dim, NULL, sizeof(dim) );
        for (i=0, new=1; i < arrInfo.dim; i++)
           new *= arrInfo.size[i];
        u32 = addr;
        len = typeInfo.len / 8;
        if (Omf86Array(&u32, 0, temp, name, typeInfo, arrInfo, struInfo, len, new)
            != OK) {
         free(buffer);
         return(FAIL);
      }
        break;
     default :
       if ( typeInfo.type >= UNCHAR && typeInfo.type <= LDOUBLE) {
           if ( Omf86Atom(FALSE, FALSE, addr, typeInfo.type, typeInfo.len,
                str)==OK)
                strcpy(val, str);
           else {
              free(buffer);
              return(FAIL);
           }
        break;
         }
      }
   free(buffer);
   return(OK);
  }
  for (i=0; i < struInfo.itemCnt; i++)
     if (struInfo.itemName[i]) free(struInfo.itemName[i]);

}
/***************************************************************************
**
**  Name: Omf86Array()
**
**  Description:  for OMF86 format.
**
**
**  Input:
**
**  Output:
**
**  Return:
**
****************************************************************************/
LOCAL RETCODE Omf86Array( U32 *addr, U16 dim, STRING temp, STRING name,
             TYPE_INFO tmpType, ARRAY_INFO tmpArr, STRUCT_INFO tmpSt, U32 len, U32 new)
{
 U8 str[100], *buf, ptr[60], indent[NAMELEN];
 U16 i;

   if (dim < tmpArr.dim ) {
      strcpy( str, temp );
      if ((buf = malloc(tmpArr.size[dim]*50)) == NULL) return(FAIL);
      buf[0] = NULL;
      if ( subitem.DimLen >= new )
         { err_exit(15,""); free(buf); return(FAIL); }
      for (i=0; i < new ; i++) {
         if ( i != subitem.DimLen ) { *addr += len;  continue; }
         if (Omf86Atom( TRUE, FALSE, *addr, tmpArr.subType,
              (U32)scalarLen[tmpArr.subType], ptr) == FAIL) {
            free(buf);
            return(FAIL);
         }
         strcpy(val, ptr);
      }
      if (buf) free(buf);
      return(OK);
   }
   return(FAIL);
}
/***************************************************************************
**
**  Name: Omf86Atom()
**
**  Description:  for OMF86 format.
**
**
**  Input:
**
**  Output:
**
**  Return:
**
****************************************************************************/
LOCAL RETCODE Omf86Atom(FLAG ptrFlag, FLAG isProc, U32 addr, U8 type,
                             U32 len, STRING str)
{
U8  *buffer, c, arr[4];
U32 new;
U32 u32;
U16 i;

   len /= 8;
   if ((buffer= malloc(len)) == NULL) return(FAIL);
   if (len > 0 && emuGetMemN( addr, buffer, len) != OK) {
      free(buffer);
      return(FAIL);
   }
   if (ptrFlag && typeInfo.scope) {
      new = addr;
      for (i=0; i < typeInfo.scope; i++) {
         len = typeInfo.ptrLen[i]/8;
         if ((len != sizeof(U32) && len != sizeof(U16)) ||
             emuGetMemN( new, arr, len) != OK) {
             free(buffer);
             return(FAIL);
         }
         if (len < sizeof(U32)) {
            emuGetReg(I86_REG, DS, &u32);
            new =  ((u32&0x0FFFFL) << 16) + *(U16 *)&arr[0];
         }
         else new = *(U32 *)&arr[0];
         if (!new) break;
      }
      if (!new) { err_exit(17,""); return(FAIL); }
   }
   switch( type ) {
      case SCHAR:
        if (!isProc)   sprintf(str, "%d", *buffer);
        else return(FAIL);
        break;
      case UNCHAR:
        if (!isProc)   sprintf(str, "%u", *buffer);
        else return(FAIL);
        break;
      case UNINT:
        if (isProc) return(FAIL);
        else sprintf(str, "%u", *(U16 *)&buffer[0]);
        break;
      case SINT:
        if (isProc) return(FAIL);
        else sprintf(str,"%d", *(S16 *)&buffer[0]);
        break;
      case UNLONG:
        if (isProc) return(FAIL);
        else sprintf(str,"%lu", *(U32 *)&buffer[0]);
        break;
      case SLONG:
        if (isProc) return(FAIL);
        else sprintf(str,"%ld", *(S32 *)&buffer[0]);
        break;
      case SINGLE:
        sprintf( str, "%g", *(float *)&buffer[0]);
        break;
      case DOUBLE:
        sprintf( str, "%lg", *(double *)&buffer[0]);
        break;
      case LDOUBLE:
        sprintf( str, "%lg", *(long double *)&buffer[0]);
        break;
      default :
        free(buffer);
        return(FAIL);
   }
   if (buffer) free(buffer);
   return(OK);
}
#endif
