/******************************************************************************
** File name : synlib.c for MICE-III
**
** Description :  This module includes the subroutines called by syntax.c
**
** Modifications :
**
**  0. Date : 09/23/1993  by C. Joyce Lin
**     0. Initial Version.
**     1. this module is distributed from syntax.c
**
**    Copyright (C) 1993  Microtek International, Inc.
**    All Rights Reserved
**
****************************************************************************/
/***************************************************************************
**
**    Include files
**
***************************************************************************/
#include  "system.h"
#include  "cmdid.h"
#include  "reg86.h"
#include  "usym1.h"
#include  "usym3.h"
#include  "database.h"
#ifndef _LINUMDEF_
#include "linumdef.h"
#endif
#ifndef _SD_ABI_DEF_
#include "sdabidef.h"
#endif

/**************************************************************************
**
** Global  Declarations : The structures for parameter dispatchings
**                        will be used by the cases of CmdProcess().
**                        The strings are used in complex.c
***************************************************************************/
GLOBAL BUS_EVENT ev_def;
//GLOBAL EXEC_EVENT0 exec_ev0;
GLOBAL QUALIFY_LIST quaList;
//GLOBAL TRIGGER trigger;
GLOBAL QUALIFY qualifyRec;  // added by frank, 4/18/1994

GLOBAL U8 **ctlName;
GLOBAL U8 *ctlNameEB[]= { "PEREQ", "", "INT0", "INT1", "INT2", "INT3",
                          "INT4" , "", "RESET", "HOLD", "NMI"};
GLOBAL U8 *ctlNameDefault[11]= { "PEREQ","", "INT0", "INT1", "INT2", "INT3",
                               "DRQ0", "DRQ1", "RESET", "HOLD","NMI"};
GLOBAL U8 *ctl8086Min[]={ "Nmi", "Reset", "Intr", "Hold",""};
GLOBAL U8 *ctl8086Max[]={ "Nmi", "Reset", "Intr", "RQ0", "RQ1"};
GLOBAL U8 *ctlNEC8086Min[] = { "Nmi", "Reset", "Int", "Hldrq",""};
GLOBAL U8 *ctlNEC8086Max[] = { "Nmi", "Reset", "Int", "RQ0", "RQ1"};
GLOBAL U8 *actionStr[] = { "Run"};
GLOBAL S16 flagTotal;
GLOBAL S8 *statusPtr[] = { "S", "R", "W", "I", "O", "AK", "H" };
GLOBAL U8 singleIntReg[10];
/**************************************************************************
**
** Externals
**
**************************************************************************/
EXTERN S16 cmdIdx;  // Command Index #, given in cmdline.c
EXTERN S16 synIdx;  // Syntax Index #, ie. case no. of CheckSyntax(); given in
                    // cmdline.c
EXTERN CURRENT_MODULE curModule; // .modNum   : current module index
                                 // .lastLine : # of the last line
EXTERN FLAG syntaxErrFlag;
EXTERN FLAG isLocal;  // flag of local symbol used by MONitor command
EXTERN U32  localBlkAddr;  // used by MONitor command
EXTERN S16 SymLoaded;
EXTERN S16 MICE, LAM2;
/**************************************************************************
**
** Definitions
**
***************************************************************************/
#define  ERROR          0
#define  CORRECT        1
#define  DQ     0x22 /* " */
#define  NS     0x23 /* # */
#define  SQ     0x27 /* ' */
#define  PS     0x25 /* % */
#define  LP     0x28 /* ( */
#define  RP     0x29 /* ) */
#define  AT     '@'
#define  HEX    16
#define  DEC    10

/**************************************************************************
**
** Name : InitStruct
**
** Function : Initializes Structures. To ensure the correctness of dispatched
**            values, it's important to initialize the structures before
**            giving values. It might be verbose though to initialize all
**            structures instead of the exact structures being used.
**
**        Input  : global structures
**        Output : global structures
**
**************************************************************************/
PUBLIC VOID InitStruct()
{
   S16 i;

   memset( &cmd_syntax,  NULL, sizeof(cmd_syntax) );
   memset( &addr_form,   NULL, sizeof(addr_form) );
   memset( &ev_def,      NULL, sizeof(ev_def) );
   memset( &qualifyRec,  NULL, sizeof(qualifyRec) );
   memset( &addr_block,  NULL, sizeof(addr_block) );
//   memset( &trigger,     NULL, sizeof(trigger) );
}

/**************************************************************************
**
** Name : IsInNumberRange
**
** Function : Converts a string to a decimal number and checks if the number
**            is within the limited range.
**
**    Input  : string, radix, limits (min,max )
**    Output : a U32 value(number).
**
** RETCODE : CORRECT or ERROR.
**
**************************************************************************/
PUBLIC RETCODE IsInNumberRange(STR str, S16 radix, S16 min, U32 max, U32 *num )
{
S8  *endPtr;

    *num = strtoul( str, &endPtr, radix);
    if ( *endPtr || *num > max || *num < min ) return(ERROR);
    return(CORRECT);

}   /* end of IsInNumberRange */


/*******************************************************************************
**
** Name : SyntaxStepCmd
**
** Function : Checks the syntactic correctness of command Step.
** Syntax: [[Over] [count|FOrever|addr1 addr2]|Till {Ret|Call|
**         {@reg|addr}=value}]
**
**    Input  : argc, argv
**    Output : structures cmd_syntax
**
** RETCODE : CORRECT or ERROR.
**
** Notes:  The parameters dispatched :
**  Step                  : argc =1; argv[0] = 1;
**  Step count            : argc =1; argv[0] = count;
**  Step FOrever          : argc =1; argv[0] = -1;
**  Step Over             : argc =2; argv[0] = 1;
**  Step Over count       : argc =2; argv[0] = count;
**  Step Over FOrever     : argc =2; argv[0] = -1;
**  Step addr1 addr2      : argc =3; fill addr_block
**  Step Over addr1 addr2 : argc =4; fill addr_block
**  Step Till Ret         : argc =5;
**  Step Till Call        : argc =6;
**  Step Till addr=value  : argc =7; argv[0]=value; fill addr_form
**  Step Till @reg =value : argc =8; argv[0]=value; argv[1] = reg_id; argv[2] = regType;
**  Step Till @PC = addr  : argc =9; argv[0]=abs;
**
*********************************************************************/
PUBLIC RETCODE SyntaxStepCmd( S16 argc, S8 argv[90][80] )
{
S16  i, valueLen, regNo, regType;
U32  addr1, addr2, num;
FLAG flag1, flag2;
S8   variable[80], value[20], *ptr;
U16 dummy;

   if (argc < 1) {
      cmd_syntax.argv[0] = 1L;
      cmd_syntax.argc = 1;
      return(CORRECT);
   }
   if ((argc < 5) && (stricmp(argv[0],"TILL" ) == 0 )) {
      if ((stricmp(argv[1],"RET" ) == 0 ) && (argc ==2) ) {
         cmd_syntax.argc = 5;
         return(CORRECT);
      }
      if ((stricmp(argv[1],"CALL" )== 0) && (argc ==2)) {
         cmd_syntax.argc = 6;
         return(CORRECT);
      }
      memset ( variable, NULL, sizeof( variable ) );
      memset ( value, NULL, sizeof( value ) );
      if (ptr = strchr(argv[1], '=')) {
         strncpy( variable, argv[1], ptr - argv[1] );
         valueLen = strlen(argv[1]) - (ptr - argv[1]) - 1;
         if (valueLen) {
            if (argc==2) strncpy( value, ptr + 1, valueLen ); // variable=value
            else return(ERROR);
         }
         else {
            if (argc==3) strcpy( value, argv[2] );  // variable= value
            else return(ERROR);
         }
      }
      else {
         strcpy( variable, argv[1] );
         if (argv[2][0] == '=') {
            if (!(valueLen = strlen(argv[2]) -1) && (argc==4))
               strcpy( value, argv[3] );      // variable = value
            else {
               if (argc==3) strncpy( value, argv[2]+1, valueLen ); // variable =value
               else return(ERROR);
            }
         }
         else return(ERROR);
      }
      if (stricmp(variable,"@PC" )== 0) cmd_syntax.argc = 9; // @PC = value
      else {
         if ((variable[0] == AT ) && IsRegister( variable+1, &regNo, &regType )) {
            cmd_syntax.argc = 8;
            cmd_syntax.argv[1] = regNo;  // @reg = value
            cmd_syntax.argv[2] = regType;
         }
         else {
            if (IsAddr( variable, &flag1, &addr1, &dummy)) {
               cmd_syntax.argc = 7;      // addr = value
               addr_form.csFlag = flag1;
               addr_form.addr = addr1;
            }
            else return(ERROR);
         }
      }
      if (cmd_syntax.argc == 9 ) {   // @PC = abs or cs:ip
         if (!IsAbs( value, &num)) {
            if (!IsAddr( value, &flag1, &addr1, &dummy)) return(ERROR);
            num = ADDR2ABS( addr1 );
         }
         cmd_syntax.argv[0] = num;
         return(CORRECT);
      }
      if (IsInNumberRange( value, HEX, 0, 0xFFFFFFFFL, &num )) {
         cmd_syntax.argv[0] = num;
         return(CORRECT);
      }
      return(ERROR);
   }
   i = 0;
   if ( stricmp(argv[0],"OVER" ) == 0 ) {
      if (argc > 1) i= 1;
      else {
         cmd_syntax.argv[0] = 1L;
         cmd_syntax.argc = 2;
         return(CORRECT);
      }
   }
   if ((argc== (2+i)) && (IsAddr(argv[i], &flag1, &addr1, &dummy)) &&
      (IsAddr(argv[1+i],&flag2,&addr2, &dummy)) && ( addr1 <= addr2 )) {
      if (!IsAddrLength(&num,addr1,addr2,flag1,flag2)) return(ERROR);
      cmd_syntax.argc = i ? 4 : 3;
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      addr_block.CSflag2 = -1;
      addr_block.count = 0;
      return(CORRECT);
   }
   if (argc == (1+i)) {
      if (stricmp(argv[i], "FOREVER") == 0 ) num= -1L;
      else
         if (!IsInNumberRange( argv[i], HEX, 1, 0x10000L,&num)) return(ERROR);
      cmd_syntax.argc = argc;
      cmd_syntax.argv[0] = num;
      return(CORRECT);
   }
   return(ERROR);
}



/*******************************************************************************
**
** Name : SyntaxListCmd
**
** Function : Checks the syntactic correctness of command LIST.
** Syntax: [Number|{SO|IN} [frame]|{frame} [abs1 abs2] [status]
**
**    Input  : argc, argv
**    Output : structures cmd_syntax
**
** RETCODE : CORRECT or ERROR.
**
** Notes:  The parameters dispatched :
**         no argument : cmd_syntax.argc = 0;
**         Number      : cmd_syntax.argc = 1;
**         INstruction frame : cmd_syntax.argc=2; argv[0]=1;
**                             argv[1]=frame;     argv[2]=sign;
**         Mixed frame       : cmd_syntax.argc=2; argv[0]=2;
**                             argv[1]=frame;      argv[2]=sign;
**         SOurce frame      : cmd_syntax.argc=2; argv[0]=3;
**                             argv[1]=frame;      argv[2]=sign;
**         MOdule frame      : cmd_syntax.argc=2; argv[0]=4;
**                             argv[1]=frame;      argv[2]=sign;
**		   frame [abs1 abs2] [status][EXternal trace_bits]
**            cmd_syntax.argc = 3;
**            argv[0] = frame;
**            argv[1] = abs1; argv[2] = abs2; argv[3] = status;
**            argv[4] = sign;
*******************************************************************************/
PUBLIC SyntaxListCmd( S16 argc, S8 argv[90][80] )
{
 U32 num, frame, mask, abs1=0L, abs2= 0L;
 FLAG sign=0, flag = 0;
 S16 i, j=1, retCode=0;
 S16 gate;

   if ( argc < 1 ) {
      cmd_syntax.argc = argc;
      return(CORRECT);
   }
   if ( argc==1 && (stricmp(argv[0],"NUMBER")==0) ){
      cmd_syntax.argc = 1;
      return(CORRECT);
   }

   /*  {INstruction|SOurce|MIxed|MOdule}  [frame] */
   if ( argc< 3 && ( (!stricmp(argv[0],"INSTRUCTION"))
        || (!stricmp(argv[0],"MIXED")) || (!stricmp(argv[0],"SOURCE"))
        || (!stricmp(argv[0],"MODULE")) ) ) {
      if (!stricmp(argv[0],"INSTRUCTION") ) cmd_syntax.argv[0]= 1;
      else if (!stricmp(argv[0],"MIXED") )  cmd_syntax.argv[0]= 2;
      else if (!stricmp(argv[0],"SOURCE")) cmd_syntax.argv[0]= 3;
      else  cmd_syntax.argv[0]= 4;  /* MOdule */
      if ( argc > 1 && !IsFrame( argv[1], &frame, &sign ) ) return(ERROR);
      else{
         cmd_syntax.argc = 2;
         cmd_syntax.argv[1]= (argc >1)? frame : -1;
         cmd_syntax.argv[2]= sign;
         return(CORRECT);
      }
   }

   /*  frame [abs1 abs2] [status] [trace_bits] */
   if (!IsFrame( argv[0], &frame, &sign )) return(ERROR);
   cmd_syntax.argc = 3;
   cmd_syntax.argv[0]= frame;
   cmd_syntax.argv[1]= -1;
   cmd_syntax.argv[2]= -1;
   cmd_syntax.argv[3]= -1;
   if ( argc ==1)  return(CORRECT);
   if ((argc >= 3) && IsAbs( argv[1], &abs1 ) && IsAbs( argv[2], &abs2)) {
      cmd_syntax.argv[1]= abs1;
      cmd_syntax.argv[2]= abs2;
      if (argc > 3)  j = 3;
      else return( CORRECT );
   }
   for ( i = j; i < argc; i++ ){
	  if ( !IsStatus( cmdIdx, argv[i], &flag ) ){
                 if ( i != argc -1 ) return(ERROR);
                 strcpy(cmd_syntax.asc, argv[i]);
				 cmd_syntax.argv[4] = 1;
	  }
	  else cmd_syntax.argv[3]= flag;
   }
   return(CORRECT);
}

/*******************************************************************************
**
** Name : IsBusEvent
**
** Function : Checks the syntax of EV1 and EV2, Event 1 and Event 2
**        or  Checks the syntax of EV1 to EV4 , Event 1 to  Event 4
**  For LAM :
**     EV1 Syntax  : [adx [datum] [status] [COunt count]|CLear]
**     EV2 Syntax  : [adx [datum] [status] |CLear]
**  For LAM2 :
**	   EV1-EV4 Syntax: [{RAnge abs1 abs2|adx} [datum] [status]
**                         [EXternal trace_bits] [COunt count]|CLear]
**
**    Input  : argc, argv
**    Output : structures cmd_syntax, ev_def.
**
** RETCODE : CORRECT or ERROR.
**
** Notes: 1.The parameters dispatched :
**          no argument : cmd_syntax.argc = 0;
**          CLear       : cmd_syntax.argc = 1;
**          otherwise   : cmd_syntax.argc = 2; and fill ev_def:
**          . Wildcard addr   :  defined = 1;
**                              addrSpec = 3;  addrLo,        addHi = adx;
**                              dataSpec=ret1; dataLo = mask; dataHi = num;
**                              stat = flag;   count = num2;
**        2. address is allowed to input in cs:ip form( except wildcard.)
**        3. For 80188 CPU, datum can be 1 byte, else 2 bytes.
**        4. For EV1, .count=count; for EV2, count nosense.
**
*******************************************************************************/
PUBLIC RETCODE IsBusEvent( S16 argc, S8 argv[90][80] )
{
   S16  i,j,end=0;
   U32  abs1, abs2, limit, dataLo, dataHi, num2;
   FLAG flag = 0, csFlag1;
   EXTERN S16 MICE;
   RETCODE ret1 = 0, ret;
   U16 dummy;

   if ((argc< 1 ) || ((argc==1) && ( stricmp(argv[0], "CLEAR") == 0))){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ((argc > 2) && (stricmp(argv[0], "RANGE") == 0)) {
      if (!LAM2) return(ERROR);
      if (!IsAbs(argv[1], &abs1) || (!IsAbs(argv[2], &abs2))) return(ERROR);
      if (abs1 > abs2 ) return(ERROR);
      ev_def.addrSpec = 2;
      j = 3;
   }
   else
   if (( ret= IsAdx(argv[0], &abs1, &abs2)) > 0 ) {
      ev_def.addrSpec = ret;
      j = 1;
   }
   else return(ERROR);

   ev_def.count =1;
   cmd_syntax.asc[0] = '\0'; //for EXternal trace_bits
   if (argc == j) ev_def.dataSpec = 0;
   else {
	  if (MICE < 5 || MICE == 9 || MICE == 10 || MICE == 13 || MICE == 14 )
		  limit = 0xffL;
	  else limit = 0xffffL;
      if ( (ret1=IsWildCard(argv[j], limit ,&dataLo, &dataHi)) !=0) j++;
      for ( i= j;  i < argc; i++ ){
         if (!IsStatus(cmdIdx, argv[i], &flag ) ){
			if ( stricmp(argv[i], "EXTERNAL") != 0 ) {
                if( (synIdx == 1 || synIdx == 76) && i==argc-2 &&
                   !stricmp(argv[i],"COUNT")){
                   if(!IsInNumberRange(argv[i+1], HEX, 1, 0xffffl, &num2) )
                       return(ERROR);
                   else{
                       ev_def.count= (U16) num2;
                       break;
                   }
                }   //end of COUNT
                else return(ERROR);
            }
            else {
				  if ( !LAM2 ) return(ERROR);
				  else strcpy ( cmd_syntax.asc, argv[++i] );
             } //end if of EXTERNAL
         }
         else ev_def.stat |= flag;
      }
      ev_def.dataSpec = ret1;

      if (ret1) {
         ev_def.dataLo = dataLo;
         ev_def.dataHi = dataHi;
      }
   }
   cmd_syntax.argc = 2;
   ev_def.defined = 1;
   ev_def.addrLo = abs1;
   ev_def.addrHi = abs2;
   return (CORRECT);
}

/**************************************************************************
**
** Name : IsExtEvent
**
** Function : Checks the syntax of EV3 of LAM
**        or  check the syntax of EV5 of LAM2
**    Input  :  argc, argv
**    Output :  cmd_syntax structure
**
** RETCODE : CORRECT or ERROR.
**
** Notes:
**     no argument       : cmd_syntax.argc = 0;
**     High              : cmd_syntax.argc = 2; cmd_syntax.argv[0]=1;
**     Low               : cmd_syntax.argc = 2; cmd_syntax.argv[0]=0;
**
**************************************************************************/
PUBLIC RETCODE IsExtEvent(S16 argc, S8 argv[90][80])
{
U16 number;
U32 num;

   if (argc< 1 ){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ((argc == 1 ) && (stricmp( argv[0],"HIGH") == 0 )) {
      cmd_syntax.argc=2;
      cmd_syntax.argv[0]=1;
      return(CORRECT);
   }
   if ((argc == 1 ) && (stricmp( argv[0],"LOW") == 0)){
      cmd_syntax.argc=2;
      cmd_syntax.argv[0]=0;
      return(CORRECT);
   }
   return(ERROR);
}

/**************************************************************************
**
** Name : IsExecEvent
**
** Function : Checks the syntax of EV4, EV5 and EV6  //186LAM
**         or Checks the syntax of EV6, EV7 and EV8  //186LAM2
**		   or Checks the syntax of EV5 , EV6 and EV7 //8086LAM && V20
** Syntax : [addr|CLear]
**
**    Input  : argc, argv
**    Output : cmd_syntax, addr_form structures.
**
** RETCODE : CORRECT or ERROR.
**
** Notes: no argument       : cmd_syntax.argc = 0;
**        CLear             : cmd_syntax.argc = 1;
**        addr              : cmd_syntax.argc = 2, fill addr_form structure;
**
**************************************************************************/
PUBLIC RETCODE IsExecEvent( S16 argc, S8 argv[90][80] )
{
FLAG flag;
U32 num;
U16 dummy;

   if (  argc < 1 ) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( argc < 2 )  {
      if ( stricmp( argv[0],"CLEAR" ) == 0 ) {
         cmd_syntax.argc=argc;
         return(CORRECT);
      }
      else
         if ( IsAddr ( argv[0], &flag, &num, &dummy )  ) {
            cmd_syntax.argc= 2;
            addr_form.csFlag = flag;
            addr_form.addr = num;
            strcpy(addr_form.sym, argv[0]);  // not used?
            return(CORRECT);
         }
   }
   return(ERROR);
}

/**************************************************************************
**
** Name : IsExecEvent2
**
** Function : Checks the syntax of EV4 //8086LAM
** Syntax : [[Run] addr [COunt count]|CLear]
**
** Notes: no argument       : cmd_syntax.argc = 0;
**        CLear             : cmd_syntax.argc = 1;
**        addr  (count)     : cmd_syntax.argc = 2; argv[0] = 0;
**                   if exist COunt count , argv[1]= count;else 0;
**        Run addr (count)  : cmd_syntax.argc = 2; argv[0] = 1;
**                   if exist COunt count , argv[1]= count;else 0;
**                            fill addr_form structure;
**************************************************************************/
PUBLIC RETCODE IsExecEvent2( S16 argc, S8 argv[90][80] )
{
FLAG flag;
U32 num;
U16 dummy;
U32 count;
U16 j=0;

   if (  argc < 1 || argc ==1 && stricmp(argv[0],"CLEAR") == 0) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if (stricmp(argv[0],"RUN") == 0) j = 1;
   if (j == 1 && argc == 1) return(ERROR);
   if (!IsAddr ( argv[j], &flag, &num, &dummy ) )   return(ERROR);
   if ( argc > j+1 ){  //exist COunt
         if ( stricmp(argv[j+1], "COUNT") ||
            !IsInNumberRange(argv[j+2],HEX,0L, 0xFFFFL, &count))
                              return(ERROR);
         if ( argc >= j+4) return(ERROR);
      cmd_syntax.argv[1] = count;
   }
   else cmd_syntax.argv[1] = 0;

   if (!j) cmd_syntax.argv[0]= 0;
   else cmd_syntax.argv[0]= 1;
   cmd_syntax.argc= 2;
   addr_form.csFlag = flag;
   addr_form.addr = num;
   strcpy(addr_form.sym, argv[0]);
   return(CORRECT);
}

/**************************************************************************
**
** Name : IsData
**
** Function : Checks if the input is an ASCII string or a
**            hexadecimal datum, and fills the contents into dest and
**            counts the length of the dest.
**
**    Input  : a string, cmdIdx( will determine the data length.)
**    Output : dest, cnt
**
** RETCODE : CORRECT or ERROR
**
** Notes: 1. the filled data length is based on the command type, for example,
**           if the command is WORD, the data length will be 2 bytes. Any inputed
**           data length less than the designate length will be filled in 0 at
**           the higher significant bytes; so is ASCII input.
**
**************************************************************************/
PUBLIC RETCODE IsData ( S8 *str, U8 *dest, S16 *cnt, const S16 cmd )
{
  S16 i,len, size, remainder, extra;
  union {
       U32  l;
       U8  ch[4];
  }  num;
  S8 k, *endPtr;
  U32 range= 0L;
  double d, ff;
  union {
     double dd;
     U8 ch[8];
  } doubleNum;
  union {
     float dd;
     U8 ch[4];
  } floatNum;

  size = strlen(str);
  InitArray( dest, 80, NULL );
  switch ( cmd ) {
        case C_WORD   : range = 0xffff;
                        len = 2;
                        break;
        case C_FLOAT  : d = strtod(str,&endPtr);
                        ff = fabs(d);
                        if (*endPtr || ((ff < 1.175e-38) && (ff != 0)) ||
                            (ff>3.4e38 ) ) return(ERROR);
                        *cnt = 4;
                        floatNum.dd = (float) d;
                        StrmnCopy( dest, floatNum.ch, 0, 4 );
                        return( CORRECT );
        case C_DOUBLE : d = doubleNum.dd = strtod(str,&endPtr);
                        if ((d== HUGE_VAL) || (d==-HUGE_VAL) || *endPtr) return(ERROR);
                        *cnt = 8;
                        StrmnCopy( dest, doubleNum.ch, 0, 8 );
                        return( CORRECT );
        case C_DWORD  :
        case C_LONG   : range = 0xffffffff;
                        len = 4;
                        break;
        case C_BYTE   :
        default       : range = 0xff;
                        len = 1;
                        break;
  }
  if ( IsInNumberRange( str, HEX, 0, range, &num.l ) ) {
      *cnt = len;
      StrmnCopy( dest, num.ch, 0, *cnt );
      return(CORRECT);
  }
  else {
     if (((k=str[0]) == str[size-1]) && ((k == DQ ) || ( k == SQ ))) {
		if (size <= 2) return(ERROR);
        for ( i=1; i < size-1; i++ )
           if ( !isascii( str[i] ) || ( str[i] == k ) ) return (ERROR);
        StrmnCopy( dest, str, 1, size-2 );
        remainder = (size-2) % len;
        extra = remainder ? (len - remainder) : 0 ;
        *cnt = size-2 + extra;
        for ( i= size-2; i < *cnt; i++ ) dest[i] = 0;
        return( CORRECT );
     }
  }
  return(ERROR);
} /* end of IsData() */

/**************************************************************************
**
** Name : IsAddr
**
** Function : Checks if the input is a logical address [seg:]offset. If yes,
**            converts the string to a U32 number with [seg] as upper half
**            byte and [offset] as lower half. If the input were a symbol,
**            the address will be obtained if available, and converted in the
**            same way described as above.
**
**    Input  : a string
**    Output : address, csFlag = 0 (IP) ; csFlag = 1 (CS:IP),
**             address string (CS:IP) (only if input is a symbol.)
**
** RETCODE : CORRECT or ERROR
**
** Notes: Special case : Watch and MONitor commands
**
**************************************************************************/
PUBLIC RETCODE IsAddr( S8 *address, FLAG *csFlag, U32 *adr, U16 *type)
{
S8  seg[10], offs[10], newAddr[20], *addr;
S16 i,len;
U32 num=0L, number=0L;
FLAG wascolon = 0;

   *type = 0;
   memset( newAddr, NULL, sizeof( newAddr ));
   if (((address[0] == NS ) && IsLineNumAddr( address, newAddr )) ||
       ((address[0] == PS ) && IsSymbolicAddr( address, newAddr, type)))
      addr = newAddr;
   else addr = address;
   InitArray(seg,sizeof(seg), NULL);
   InitArray(offs,sizeof(offs), NULL);
   len = strlen( addr );
   for ( i=0; i< len; i++ ) {
      if ( !isxdigit( addr[i] ) )  {
         if ( addr[i] != ':' ) return (ERROR);
         else {
            if ( wascolon ) return (ERROR);
            wascolon = 1;
            if ((i<1) || (len == (i + 1)))  return(ERROR);
            StrmnCopy(seg, addr, 0, i );
            StrmnCopy(offs, addr, i+1, len - i - 1 );
         }
      }
   }
   if (wascolon) {
      if( !IsInNumberRange( seg, HEX, 0, 0x0ffffL, &num )  ) return(ERROR);
      number = num * 0x10000;
      if( !IsInNumberRange(offs, HEX, 0, 0x0ffffL, &num )  ) return(ERROR);
      number += num;
      *adr = number;
      *csFlag = 1;   /* CS:IP */
      return(CORRECT);
   }
   else {
      if( !IsInNumberRange( addr, HEX, 0, 0x0ffffL,&num )  ) return(ERROR);
      *adr = num;
      *csFlag = 0;
      return (CORRECT);
   }
}          /* end of IsAddr() */

/**************************************************************************
**
** Name : IsAbs
**
** Function : Checks if the input is an absolute address between 0 and 0FFFFFH
**
**    Input  : a string
**    Output : absolute address
**
** RETCODE : CORRECT or ERROR
**
** Notes : If the input were a symbol, the routine will check whether it's a
**         valid symbol and get the logical address, then convert it to an
**         absolute address.
**
**************************************************************************/
PUBLIC RETCODE IsAbs( S8 *str, U32 *abs )
{
U32 num, addr;
FLAG csFlag;
U8 tempBuf[80];
U16 dummy;

   if (str[0] == PS ) {
      strcpy(tempBuf, str);
      if (!IsAddr( tempBuf, &csFlag, &addr, &dummy)) return(ERROR);
      num = ADDR2ABS( addr );
   }
   else
      if (!IsInNumberRange( str, HEX, 0, 0xfffff, &num ))  return(ERROR);
   *abs = num;
   return(CORRECT);
}  /* end of IsAbs() */

/**************************************************************************
**
** Name : IsLength
**
** Function : Checks if the input is a valid length between 1 and 0100000H.
**
**    Input  : a string
**    Output : length
**
** RETCODE : CORRECT or ERROR
**
**************************************************************************/
PUBLIC RETCODE IsLength( STR str, U32 *length )
{
U32 num = 0L;

   if (  IsInNumberRange( str, HEX, 1, 0x100000, &num ) ) {
      *length = num;
      return(CORRECT);
   }
   return(ERROR);
}         /* end of IsLength() */

/**************************************************************************
**
** Name : IsAddrLength
**
** Function : Checks the range of addr1 and addr2 and get the difference as length.
**            Addr2 has to be greater than addr1; if the 2 addresses were at
**            different segments, they are converted to absolute addresses and
**            then get the difference.
**
**    Input  : addr1, addr2, csFlag1,  csFlag2
**    Output : length
**
** RETCODE : CORRECT or ERROR
**
** Notes: 1. address type : IP (csFlag=0), CS:IP (csFlag=1)
**        2. The valid combinations of addr1, addr2 are :
**         Ŀ
**          addr1   addr2   
**         Ĵ
**             ip      ip   
**          cs:ip      ip   
**          cs:ip   cs:ip   
**         
**
**************************************************************************/
PUBLIC RETCODE IsAddrLength(U32 *len, U32 addr1, U32 addr2, FLAG flag1, FLAG flag2 )
{
  U32 abs1, abs2, num;

  if (!flag1 && flag2) {
     prn_ferr(50); // Invalid address range!
     return(ERROR);
  }
  if ( ((flag1 && !flag2) || (!flag1 && !flag2)) && (( addr1 & 0xffff) > addr2)) {
     prn_ferr(50); // Invalid address range!
     return(ERROR);
  }
  if (flag1 && flag2) {
     abs1 = ADDR2ABS( addr1 );
     abs2 = ADDR2ABS( addr2 );
     num = 1+ abs2 - abs1;
  }
  else num = (flag1 && !flag2) ? 1+(addr2 - (addr1 & 0x0ffff)) : 1+addr2-addr1;
  if ((num <=0) || (num > 0x100000))  {
     prn_ferr(50); // Invalid address range!
     return(ERROR);
  }
  *len = num;
  return(CORRECT);
}

/**************************************************************************
**
** Name : IsFrame()
**
** Function : Checks if the input is a valid signed number.
**
**    Input  : a string
**
**    Output : frame number, sign
**
** RETCODE : CORRECT or ERROR
**
**************************************************************************/
PUBLIC RETCODE IsFrame( S8 *str, U32 *frame, FLAG *sign )
{
U32 num=0L;
S8 *ptr;
U32 maxFrame;

   if (!LAM2) maxFrame = 0x7FFL;
   else maxFrame = 0x7FFFL;

   *sign = 0; // for MICE-III, this has no defination.
   ptr = str;
      if (str[0] == '-') return(ERROR);
      if (str[0] == '+') ptr = str + sizeof(char);

   if ( IsInNumberRange( ptr, HEX, 0, maxFrame, &num ) ){
      *frame = num;
      return(CORRECT);
   }
   return(ERROR);
} /* end of IsFrame() */

/**************************************************************************
**
** Name : IsWildCard
**
** Function : Checks if the input is a hexadecimal with(out)
**            wild card nibbles/bits.
**
**    Input  : a string, limit( the allowed maximum value )
**
**    Output : num (wildcard value), mask
**
** RETCODE : 3: WildCard, 1: hexadecimal, 0: Syntax Error, -1 : beyond limit.
**
** Notes: Eg. Input value is F2(01X0)1X :
**            num  =  F2410 (the bit or byte of X is considered as 0. )
**            mask =  FFDF0 (the bit or byte of X is masked with 0, otherwise 1.)
**
**************************************************************************/
PUBLIC RETCODE IsWildCard( S8 *str, U32 limit, U32 *num, U32 *mask )
{
S16   i, len, bits,value;
S8 c;
FLAG isBit=0, isZero=0;
U32  data=0L, hByte = 0L,mmask=0xFFFFFFFF;

   len = strlen(str);
   for ( i=0; i< len; i++ ) {
      c = ch_toupper(str[i]);
      switch ( c ) {
         case LP :
            if (isBit) return(ERROR);
            isBit = 1;
            bits = 0;
            hByte = 0L;
            break;
         case RP :
            if ((!isBit) || (bits % 4) || (hByte > limit)) return(ERROR);
            data = 0x10 * data + hByte;
            isBit = 0;
            bits = 0;
            break;
         case 'X' :
            if (bits == 4){
                data = 0x10 * data + hByte; //Frank , 7/12/94
                bits = 0;
                hByte = 0L;
            }
            if (isBit) {
               mmask = _lrotl( mmask, 1 );
               mmask &= 0xfffffffe;
               hByte = 0x02 * hByte;
               ++bits;
            }
            else {
               mmask = _lrotl( mmask, 4 );
               mmask &= 0xfffffff0;
               data = 0x10 * data;
            }
            break;
         case '0' : isZero = 1;
         case '1' :
            if (bits == 4){
                data = 0x10 * data + hByte; //Frank , 7/12/94
                bits = 0;
                hByte = 0L;
            }
            value = c - '0';
            if (isBit) {
               mmask = _lrotl( mmask, 1 );
               hByte = 0x02 * hByte + value;
               ++bits;
            }
            else {
               mmask = _lrotl( mmask, 4 );
               data = 0x10 * data + value;
            }
            break;
         default  :
            if ( isxdigit(c) && !isBit ) {
               value = ((c >= '2') && (c <= '9')) ? c -'0' : 10 + c - 'A';
               mmask = _lrotl( mmask, 4 );
               data = 0x10 * data + value;
            }
            else return( ERROR );
            break;
      }
   }
   *num = data;
   *mask = mmask;
   if ( data <= limit ) {
      if ( !data && (!isZero) && (mmask != 0xffffffff)) *mask = ~limit;
      return( mmask == 0xffffffff ? 1 : 3 );
   }
//   else return -1; //5/25/1994
   return(ERROR);
}  /* end of IsWildCard */

/**************************************************************************
**
** Name : IsAdx
**
** Function : To check if the input address is in valid address form (
**            ( symbol, absolute, or wildcard nibbles. )
**
**    Input  : str
**
**    Output : adx, mask
**
** RETCODE : 1: Single address;  3: Wildcard nibbles; ERROR(0);
**
** Notes: Symbol address will be converted to absolute address and returned.
**
**************************************************************************/
PUBLIC RETCODE IsAdx( S8 *str, U32 *adx, U32 *mask )
{
  U32 abs;

  if (IsAbs(str, &abs)) {
     *adx = abs;
     *mask = 0xFFFFFFFF;
     return( 1 );
  }
  return( IsWildCard( str, 0x0FFFFFL, adx, mask));
}

/**************************************************************************
**
** Name : IsStatus
**
** Function : Checks if the input is in the status sets --
** S8 *statusPtr[] = { "S", "R", "W", "I", "O", "AK", "H" };
**
**    Input  : cmdIdx, a string
**
**    Output : status Flag
**
** RETCODE : CORRECT or ERROR
**
** Notes: bit-0  S: Fetch cycle        bit-1  R: Read cycle
**        bit-2  W: Write cycle        bit-3  I: Input cycle
**        bit-4  O: Output cycle       bit-5  AK: Interrupt Acknowledge cycle
**		  bit-6  H: Halt cycle		  //		bit-7  DMA: DMA cycle
**
**************************************************************************/
PUBLIC RETCODE IsStatus(S16 cmdId, S8 *str, FLAG *status )
{
S16 i;
S8  c;
U8 flag;

   flag = *status;
   for (i=0; i < 7; i++ ) {
      if ( stricmp( statusPtr[ i ], str ) == NULL ) {
         flag = flag | _rotl( 0x01, i);
         break;
      }
   }
//   if ((flag & 0x80) && flag != 0x80) {//if(flag==0)
//      syntaxErrFlag = 4; // DMA can not be set with other statuses
//      return(ERROR);
//   }
   if (i < 7) {
      *status = flag;
      return(CORRECT);
   }
   return(ERROR);
}   /* end of IsStatus */

/**************************************************************************
**
** Name : IsRegister
**
** Function : Checks if the input is a valid 80186 register name, and
**            gets the register number.
**
**    Input  : a string
**    Output : register number.
**
** RETCODE : CORRECT or ERROR
**
**************************************************************************/
PUBLIC RETCODE IsRegister ( S8 *str, S16 *reg, S16 *regType )
{
EXTERN S8 *cpuRegTable[];
EXTERN S8 *copRegTable[];
EXTERN S8 *intRegMaster[];
EXTERN S8 *intRegSlave[];
EXTERN S8 *intReg186EB[];
EXTERN S8 *intReg186EC[];
S16 cnt, mode, intRegTableSize;
S8 **intRegTable;
U16 relReg;

   for (cnt = 0; cnt < SIZE_CPUREG; cnt++) {
       if ( !stricmp( cpuRegTable[ cnt ], str ) ) {
          *reg = cnt;
          *regType = I86_REG;
          return(CORRECT);
       }
   }
   for (cnt = 0; cnt < SIZE_COPREG; cnt++) {
       if ( !stricmp( copRegTable[ cnt ], str ) ) {
          *reg = cnt;
          *regType = I87_REG;
          return(CORRECT);
       }
   }
   GetCpuMode(&mode);
   switch ( mode ) {
      case 2 :    // EB
         intRegTable = intReg186EB;
         intRegTableSize = SIZE_186EB_INTREG;
         *regType = INT_REG_186EB;
         break;
      case 3 :    // EC
         intRegTable = intReg186EC;
         intRegTableSize = SIZE_186EC_INTREG;
         *regType = INT_REG_186EC;
         break;
      default :   // XL/EA
         GetRegister(INT_REG_MASTER, RELREG, &relReg);
         if (relReg & 0x4000L) {             // Slave mode
            intRegTable = intRegSlave;
            intRegTableSize = SIZE_XLEA_INTREG;
            *regType = INT_REG_SLAVE;
         }
         else {                              // Master mode
            intRegTable=intRegMaster;
            intRegTableSize = SIZE_XLEA_INTREG;
            *regType = INT_REG_MASTER;
         }
   }
   for (cnt = 0; cnt < intRegTableSize; cnt++) {
       if ( !stricmp( intRegTable[ cnt ], str ) ) {
          *reg = cnt;
          strcpy(singleIntReg, intRegTable[cnt]);
          return(CORRECT);
       }
   }
   return(ERROR);
} /* end of IsRegister */


PUBLIC RETCODE IsV20Reg ( S8 *str, S16 *reg )
{
EXTERN S8 *REGV20_TAB[]; //total 24
int cnt;

   for (cnt = 0; cnt < 24; cnt++) {
	   if ( !stricmp( REGV20_TAB[ cnt ], str ) ) {
          *reg = cnt;
          return(CORRECT);
       }
   }
   return(ERROR);
}
/**************************************************************************
**
** Name : IsAction
**
** Function : Checks if the syntax matches the sequence of action.
** action: {[EXTIn] [Seq|BReak|Reset] [Troff|Next] [INC0|RST0] [INC1|RST1] [EXTOut]}
**
**    Input  : trNo( trigger level #), evNo(event #), index(index of arguments),
**             argc, argv.
**    Output : index, trigger structure.
**
** RETCODE : CORRECT or ERROR
**
** Notes: 1. At least one member of the action sequence has to be included in
**           an event/cnt.
**        2. The meanings of action flag bits:
**       Ŀ
**        0  BReak    4  Next    8  RST1   
**        1  Reset    5  INC0    9  EXTIn       0 BReak
**        2  Seq      6  RST0   10  EXTOut 
**        3  Troff    7  INC1              
**       
**        3. Bits (1,2,3), bits (3,4), bits(5,6), bits(7,8) are exclusive within
**           their groups.
**        4. While the event set(isTimer == 1) is TImer, bits(5,6,7,8) are not
**           allowed to set.
**
**************************************************************************/
/*************************
PUBLIC RETCODE IsAction( U8 evNo, S16 *index, S16 argc, S8 argv[90][80],
                          FLAG isTimer)
{
S16  idx,  i;
U16  actionFlag = 0, x, y;
FLAG isAct = 1;

   idx = 1 + *index;
   while ( isAct ) {
      isAct = 0;
      for ( i = 0; i < 1; i++ ) {
         if ( stricmp( argv[ idx ], actionStr[i] ) == 0 )  {
     //    if (isTimer && (i > 4 ) && (i < 9)) return(ERROR);
            x = _rotl(0x01, i);
            if (actionFlag & x ) return (ERROR);
            actionFlag |= x;
            if ((((y = actionFlag & 0x07) >2) && (y!=4)) ||
                ((actionFlag & 0x18) == 0x18) || ((actionFlag & 0x60) == 0x60) ||
                ((actionFlag & 0x180) == 0x180) ) return(ERROR);
            ++idx;
            isAct = 1;
            break;
         }
      }
      if (!actionFlag) return(ERROR);
      trigger.act[evNo] = actionFlag;
      *index = idx;
      if (argc == idx) return(CORRECT);
   }
   return(CORRECT);
}
**************/
/**************************************************************************
**
** Name :   IsSymbolicAddr
**
** Function : Check if a symbolic address is in the symbol table. If yes,
**            get the address.
**
**    Input  : symbol
**    Output : address
**
** RETCODE :   CORRECT( found ), ERROR( not found )
**
** Notes:
**
**************************************************************************/
int nModuleFlag = FALSE;
PUBLIC RETCODE IsSymbolicAddr( STRING aWord, STRING addrStr, U16 *type )
{
S8   *ptr,*ptr1, modName[42], *ptr2;
S16  c, len, oldMod;
U32 addr, curFP;
EXTERN U16 backLevel;          // current stack level on display

   if (((c=cmdIdx) == C_QUERY) || (c==C_CREATE) || (c==C_DELETE) ||
        (c==C_SYMBOL)) return( ERROR );
   memset(modName, NULL, sizeof(modName) );
   ptr = aWord;
   if ((len = strlen(aWord)) < 2) return ERROR;
   if ((len > 4) && (aWord[1] == PS) && (aWord[2] != PS)) {  /* with module */
      ptr1 = strchr( ptr + 2, PS );
      ptr2 = strchr( ptr + 2, NS );
      if ( (ptr1 && ptr2) || (!ptr1 && !ptr2) ) return( ERROR );
      else if (ptr1) {   // "%%module%symbol"
         StrmnCopy( modName, aWord, 2, ptr1 - ptr - 2 );
         StrmnCopy( comsym.name, ptr1, 1, sizeof(comsym.name)-1);
         if (NameToMod(modName) == FOUND) { // NameToMod() is defined in wusym1.c
// Chen 05/23/94
//             comsym.modx = commod.mod_inx;  // comsym is a global structure for symbol
//             comsym.mask |= MSK_MOD;
         }
         else {
            syntaxErrFlag = 72;
            return( ERROR );
         }
      }
      else {            //  "%%module#line_number"
        // added by Chen, 08/10/94  ; for Go %%prime2#6 at COMVPMax == TRUE
         nModuleFlag = TRUE;
         StrmnCopy( modName, aWord, 2, ptr2 - ptr - 2 );
         if (NameToMod(modName) == FOUND) {
            oldMod = curModule.modNum ;
            curModule.modNum = commod.mod_inx;
            c = IsLineNumAddr( ptr2, addrStr );
            curModule.modNum  = oldMod;
            return( c );
         }
         else {
            syntaxErrFlag = 72;
            return ERROR;
         }
      }
   }
   else {   // straight symbol, i.e. %symbol
      if ( strchr(ptr+1, PS ) ) return(ERROR);
      else StrmnCopy( comsym.name, ptr, 1, sizeof(comsym.name)-1);
   }


   c=0;

// Chen 05/23/94; 05/26/94
//   if ( !modName[0] && (c=FindLocalVariable(backLevel, comsym.name, &addr,
//        type, &localBlkAddr, &isLocal)) == OK ) {
  if ( modName[0]==NULL || stricmp(modName, curModule.modName)==0 ) {
    if ( ( c = FindLocalVariable(backLevel, comsym.name, &addr,
                                type, &localBlkAddr, &isLocal) ) == OK ) {
      if (cmdIdx == C_MONITOR && isLocal == 2) { // stacked var
         emuGetReg(I86_REG, REG_FP, &curFP);
         addr -= curFP;
      }
      sprintf( addrStr, "%04X:%04X", (U16)(addr >> 16), (U16)addr);
      return(CORRECT);
    }
  }

   if ( SymToAddr() == FOUND ) {
      addr2str( comsym.addr, addrStr);
      *type = comsym.type;
      return(CORRECT);
   }
   if (c==1) syntaxErrFlag = 26;  // local variables not alive!
   else if (SymLoaded != OK) syntaxErrFlag = 27; // Symbol file not loaded
   else syntaxErrFlag = 74;    // symbol not found
   return(ERROR);
}

/**************************************************************************
**
** Name :   IsLineNumAddr
**
** Function : Check if a line number is in the linnumInfo table. If yes,
**            get the address.
**
**    Input  : line number string
**    Output : address string
**
** RETCODE :   CORRECT( found ), ERROR( not found )
**
** Notes:
**
**************************************************************************/
PUBLIC RETCODE IsLineNumAddr( S8 *aWord, S8 *addrStr)
{
EXTERN FLAG langMode;
U32 lineNum, addr;
S8 *endPtr;

U32 dummy, curPC;
int modNum;
char modName[41];
extern U16 cpuReg[];
extern  int COMVPMax;


   if (SymLoaded != OK) {
      syntaxErrFlag = 27;  // symbol table not loaded!
      return(ERROR);
   }
   if (langMode==ASM) {
      syntaxErrFlag = 76; // line number is not available for assembly mode.
      return(ERROR);
   }
   if (cmdIdx == C_QUERY) return( ERROR );
   if (!IsInNumberRange(aWord+1, DEC, 1, 0x0ffffL, &lineNum)) {
      syntaxErrFlag = 65; // Invalid line number
      return(ERROR);
   }

// added by Chen, 08/09/94
    if ( COMVPMax && nModuleFlag == FALSE ) {
        emuGetAllReg(cpuReg);
        emuGetReg(I86_REG, REG_PC, &curPC);
        if ( IdentifyCurrentModule(TRUE, curPC, &modNum, modName, &dummy, &dummy) == FAIL ||
            LinNum2Addr( (U16)lineNum, &addr ) == FAIL ) {
            syntaxErrFlag = 65; // Invalid line number
            return(ERROR);
        }
    }
    else {
        nModuleFlag = FALSE;
        if ( LinNum2Addr( (U16)lineNum, &addr ) == FAIL ) {
            syntaxErrFlag = 65; // Invalid line number
            return(ERROR);
        }
    }

   sprintf( addrStr, "%04X:%04X", (U16)(addr >> 16), (U16)addr);
   return(CORRECT);
}




/*** SET TRIGGER *****

Trigger [RUn] {level#[ <THen level#>] [Reset|If state# <ELse state#>]
               |If state# <ELse state#>} [FOrward|BAckward|CEnter|DElay count]

 1. trigger run and mode omit to think over.
 2.
    1. trigger level#[ <THen level#>] [Reset|If state# <ELse state#>]
       a. trigger level#
       b. trigger level# <THen level#> ...   // do "THen level# " loop
       c. trigger level# Reset
       d. trigger level# If state# <ELse state#> // do "THen level# " loop
    2. trigger If state# <ELse state#>}

    state# : level# [RESet]
             |(level# {<Then level#> [RESet|If state# <ELse state#>]
                       |If state# <ELse state#>})

*****  end   ******/

/******************************** End of File *****************************/
