/******************************************************************************
** File name : syntax.c for MICE-III
**
** Description :  Module of syntax checkings and parameter dispatchings.
**
** Modifications :
**
** 36. Date : 9/23/1993   by C. Joyce Lin
**    1. Split into two: all functions called by CheckSyntax() are putted
**       onto synlib.c.
**
** 35. Date : 8/06/1993   by C. Joyce Lin
**    1. Modify command MOde to CPU.
**    2. Modify command LAnguage to MOde, and change syntax (case 57,
**       added "Mix" option.)
**    3. Add syntax checking code for WATch command. (case 60)
**    4. Add syntax for FINd, WATch, BACK commands.
**
** 34. Date : 7/30/1993   by C. Joyce Lin
**    1. Modify syntax for VIew command from case 41 to case 58(newly added).
**
** 33. Date : 7/22/1993   by C. Joyce Lin
**    1. Modify the syntax of LAnguage command(case 57).
**    2. Move CheckSymbolicAddr() from cmdline.c, and remove "showErr" flag.
**    3. Modify CheckSymbolicAddr() and add CheckLineNumber()
**    4. Modify the syntax checking routine of Query command for Line# query
**
** 32. Date : 7/12/1993   by C. Joyce Lin
**    1. Add command LAnguage for Hi/Lo level language toggle.
**
** 31. Date : 6/22/1993   by C. Joyce Lin
**    1. Modify IsRegister(), CheckControl() for EC type CPU.
**
** 30. Date : 5/14/1993   by C. Joyce Lin
**    1. Modify CheckControl() to use different control flag tables when CPU
**       model varies.
**    2. Alter command TC to SUspend, and also change the syntax checking.
**
** 29. Date : 4/28/1993   by C. Joyce Lin
**    1. Modify the syntax of case 52( for Command TC).
**
** 28. Date : 4/20/1993   by C. Joyce Lin
**    1. Modify syntax for Log and Step commands.
**
** 27. Date : 4/16/1993   by C. Joyce Lin
**    1. Modify syntax for DOUble, Go and List commands.
**
** 26. Date : 4/12/1993   by C. Joyce Lin
**    1. Add a command: STOp.
**
** 25. Date : 3/23/1993   by C. Joyce Lin
**    1. Fix a bug in IsAddrLength() about crossing segment problem.
**    2. Fix a bug in InitStruc() about improper initializing for struc trigger.
**    3. Fix a bug in IsAddr() for Monitor command malfunctions when address
**       has module.
**    4. Modify case 43 to fill struc addr_form when input has address range.
**
** 24. Date : 3/19/1993   by C. Joyce Lin
**    1. Fix a bug in CheckControl().
**
** 23. Date : 3/17/1993   by C. Joyce Lin
**    1. Add "start" and "stop" modes to CheckListCmdSyntax()
**    2. Modify the size of argv[20][80] to argv[90][80].
**    3. Fix  a bug in CheckListCmdSyntax().
**
** 22. Date : 3/15/1993   by C. Joyce Lin
**    1. Modify IsBusEvent() to allow addresses in cs:ip form.
**
** 21. Date : 3/12/1993   by C. Joyce Lin
**    1. fix a bug in IsAddrLength() for failing to check error conditions.
**    2. The total length for data1 to data16 is not greater than 80.
**
** 20. Date : 3/11/1993   by C. Joyce Lin
**    1. Modify IsAction() for not checking "BReak"
**    2. Modify IsData() to fill 0 to an odd number count of ASCII input.
**    3. Modify the size of argv[12][80] to argv[20][80], because the meximum
**       number of parameters is 19.
**
** 19. Date : 3/8/1993   by C. Joyce Lin
**    1. Fix a bug in IsAction() to check the special case of overlapped input
**       -- Troff.
**    2. Modify IsAddr() to allow dynamic length of cs:id.
**
** 18. Date : 3/4/1993   by C. Joyce Lin
**    1. Fix a bug in case 22 about failing to detect "no argument" condition.
**
** 17. Date : 2/26/1993   by C. Joyce Lin
**    1. Modify IsAction(), if "BReak" Enable, "Troff" Enable, "Next" Disable.
**    2. Modify case 44 to set symbol if a symbol address were input.
**    3. Remove IsAlphaNum().
**
** 16. Date : 2/23/1993   by C. Joyce Lin
**    1. Fix a bug in IsWildCard(). It failed to distinguish 0X or X0 from X.
**
** 15. Date : 2/19/1993   by C. Joyce Lin
**    1. Add TRIgger Command Checking (case 50) and move the module of case 11,
**       47 from CheckSyntax() to CheckTriggerLevel().
**    2. Modify CheckTriggerLevel() and IsAction to check timer options.
**
** 14. Date : 2/17/1993   by C. Joyce Lin
**    1. Fix a bug in IsData() about floating point checking .
**    2. Modify syntax checking modules for case 40, 9, 25, 47; do not check
**       "number".
**    3. Modify the syntax of Timer and CNT0, CNT1 commands.
**    4. Remove case 45, 49; Change Case number 51->45, 50->49.
**
** 13. Date : 2/10/1993   by C. Joyce Lin
**    1. Add the syntax checking codes of Save and Recall commands( case 51).
**    2. Modify CheckListCmdSyntax() to initialize status with 0x7f.
**    3. Modify the syntax of List command.
**
** 12. Date : 2/05/1993   by C. Joyce Lin
**     1. Modify the syntax of List command ( add trace-bits ), and move the
**        module from CheckSyntax() to CheckListCmdSyntax().
**     2. Modify IsFrame() to accept negative numbers.
**
** 11. Date : 2/04/1993   by C. Joyce Lin
**     1. Fix a bug in IsExtEvent().
**
** 10. Date : 2/03/1993   by C. Joyce Lin
**     1. Fix bugs about status settings (modify IsStatus() and all modules
**        call to this function; an initial value is given.)
**     2. Add DMA(D) to IsStatus(), and seperate List command and
**        Event command status( add cmdIdx checking in IsStatus ).
**     3. Modify the syntax and checking codes of EV9 command.
**     4. Modify the paramaters dispatching codes of EV10 command.
**     5. Add VIew command to case 41.
**
**  9. Date : 2/02/1993   by C. Joyce Lin
**     1. Modify the syntax of List command.
**     2. Revise the function IsWildCard() to allow user keyin only 1 nibble,
**        eg. X intead of XXXX
**
**  8. Date : 1/20/1993   by C. Joyce Lin
**     1. Modify the syntax of Map command.
**
**  7. Date : 1/19/1993   by C. Joyce Lin
**     1. Fix a bug in IsLength(). The maximum of length is 10000H instead of
**        0FFFFFH.
**     2. Modify syntax #12 : allowed 16 input data.
**     3. Modify syntax of CNT0, CNT1 : remove "Disable".
**     4. Modify syntax of Timer : [Enable] (syntax # 48).
**     5. Add BUffer command.
**     6. Modify the syntax of Patch command.
**
**  6. Date : 1/12/1993    by C. Joyce Lin
**     1. Add IsAddrLength() to check addr1 and addr2 range and convert logical
**        addresses to absolute addresses if they crossed segment.
**     2. Modify the syntax of Quit command.
**
**  5. Date : 1/11/1993    by C. Joyce Lin
**     1. Add "Program", "Data","External" for Map command syntax checking.
**
**  4. Date : 1/7/1993     by C. Joyce Lin
**     1. Fix a bug in IsAbs() symbol settings.
**     2. Fix a bug in IsData at case C_FLOAT to check the range before assigned
**        to the variable.
**
**  3. Date : 1/5/1993    by C. Joyce Lin
**     1. Modify the syntax of Map command.
**     2. Add addr2abs() to convert logical address to absolute address
**     3. Add IsAdx() and modify IsAbs() to check symbols and do the conversions.
**
**  2. Date : 1/4/1993    by C. Joyce Lin
**     1. Modify the syntax of TR1 to TR4 commands.
**
**  1. Date : 12/30/1992  by C. Joyce Lin
**     1. Modify the syntax of CNT0, CNT1, Timer commands .
**     2. Modify codes of addr1, addr2 range checking.
**
**  0. Date : 12/23/1992  by C. Joyce Lin
**     0. Initial Version.
**     1. Cmdexp1.c was splitted into two ( cmdline.c and syntax.c ) because
**        of impractical size.
**
**    Copyright (C) 1992, 1993  Microtek International, Inc.
**    All Rights Reserved
**
**    Programmed by Chung-Yee Joyce Lin
**
****************************************************************************/

/***************************************************************************
**
**    Include files
**
***************************************************************************/
#include  "system.h"
#include  "cmdid.h"
#include  "reg86.h"

/**************************************************************************
**
** Externals
**
**************************************************************************/
EXTERN S16 cmdIdx;  // Command Index #, given in cmdline.c
EXTERN S16 synIdx;  // Syntax Index #, ie. case no. of CheckSyntax(); given in
                    // cmdline.c
EXTERN FLAG langMode;
extern LAM2;
extern MICE;
extern HEMM;
/**************************************************************************
**
** Global  Declarations : the structures for parameter dispatchings
**                        will be used by the modules in CmdProcess().
**
***************************************************************************/
GLOBAL S8 pathName[4][50];
GLOBAL S8 extName[4][5];
GLOBAL FLAG syntaxErrFlag;
GLOBAL FLAG isLocal;  // flag of local symbol used by MONitor command
GLOBAL U32  localBlkAddr;  // used by MONitor command

struct syntax {     // for parameter dispatchings
  S16 argc;
  S32 argv[5];
  S8  asc[80];
  S8  input[80];
} cmd_syntax;

struct addr_format {   // used for one address
  S16 symFlag;
  S16 csFlag;
  U32 addr;
  U8  sym[80];
} addr_form;

struct block_attr {  // used for 2 addresses
  S16 CSflag1;
  U32 addr1;
  U32 length;
  S16 CSflag2;
  U32 addr2;
  S16 count;
  S8   buff[64];
} addr_block;

struct patch_address {
  S16 CSflag1;
  U32 addr1;
  U32 addr2;
  S16 CSflag2;
  U32 addr3;
  U32 addr4;
} patch_addr;

/**************************************************************************
**
** Definitions
**
***************************************************************************/
#define  ERROR          0
#define  CORRECT        1
#define  DQ     0x22 /* " */
#define  NS     0x23 /* # */
#define  SQ     0x27 /* ' */
#define  PS     0x25 /* % */
#define  LP     0x28 /* ( */
#define  RP     0x29 /* ) */
#define  AT     '@'
#define  CNT_NUMBER(arr)   ( sizeof( arr ) / sizeof( arr[0] )  )
#define  SYNTAX_ERROR   25
#define  HEX    16
#define  DEC    10
#define  ASM     0

/**************************************************************************
**
** Execution codes ( PUBLIC )
**
***************************************************************************/
/*****************************************************************************
**
** Name : CheckSyntax
**
** Function : Checks the syntactic correctness of user's input line. if correct,
**            fills the global structures and returns CORRECT, otherwise
**            returns ERROR.
**
**    Input  : cmdIdx, synIdx, argc, argv[]
**    Output : global structures.
**
** RETCODE : CORRECT or ERROR
**
******************************************************************************/
PUBLIC RETCODE CheckSyntax(S16 argc, S8 argv[90][80])
{
   InitStruct();
   syntaxErrFlag = SYNTAX_ERROR;
 /*----------------------------------------------------------------
   synIdx of Event or Trigger command was assigned the index of
   the event # or trigger # in CMDLINE.C; but now it has to be
   recovered for syntax checking.
  ---------------------------------------------------------------*/
   if (cmdIdx == C_EVENT){
      if (MICE <= 8 && LAM2) synIdx = 42;
	  else if ( (MICE <=8 || MICE == 17) && !LAM2) synIdx = 74; //186LAM
	  else if ( (MICE >= 9) && !LAM2) synIdx = 75; //8086LAM
   }

 /*----------------------------------------------------------------
   In this switch block, each case does the checking of one of the
   syntax types with respect to "synIdx". synIdx was assigned
   by
  ---------------------------------------------------------------*/
   switch ( synIdx ) {
      case 0 : return(IsNoArgument      (argc, argv));
      case 1 : return(IsBusEvent        (argc, argv));
      case 2 : return(IsExtEvent        (argc, argv));
      case 3 : return(IsExecEvent       (argc, argv));
      case 4 : return(SyntaxGoCmd       (argc, argv));
      case 5 : return(SyntaxHelpCmd     (argc, argv));
      case 6 : return(IsOnOff           (argc, argv));
      case 7 : return(SyntaxIsAddr      (argc, argv));
      case 8 : return(IsIntExt          (argc, argv));
      case 9 : return(IsNumber          (argc, argv));
     case 10 : return(SyntaxSyncCmd     (argc, argv));
     case 11 : return(CORRECT);
     case 12 : return(IsAddrData        (argc, argv));
	 case 13 : return(CORRECT);
	 case 14 : return(IsAddrRangeDest	(argc, argv));
	 case 15 : return(IsOptAddrRange	(argc, argv));
     case 16 : return(IsAddrRangeData   (argc, argv));
     case 17 : return(SyntaxInputCmd    (argc, argv));
     case 18 : return(SyntaxListCmd     (argc, argv));
     case 19 : return(SyntaxOutputCmd   (argc, argv));
     case 20 : return(SyntaxResetCmd    (argc, argv));
     case 21 : return(SyntaxSizeCmd     (argc, argv));
	 case 22 : return(SyntaxCopyCmd 	(argc, argv));
	 case 23 : return(CORRECT);
	 case 24 : return(CORRECT);
     case 25 : return(IsExecEvent2      (argc, argv));
     case 26 : return(SyntaxRegCmd      (argc, argv));
     case 27 : return(IsNo1No2          (argc, argv));
     case 28 : return(SyntaxCreateCmd   (argc, argv));
     case 29 : return(SyntaxDeleteCmd   (argc, argv));
     case 30 : return(IsString          (argc, argv));
     case 31 : return(IsString          (argc, argv));
     case 32 : return(IsString          (argc, argv));
     case 33 : return(SyntaxJournalCmd  (argc, argv));
     case 34 : return(SyntaxOptionsCmd  (argc, argv));
     case 35 : return(SyntaxQueryCmd    (argc, argv));
     case 36 : return(SyntaxSoftkeyCmd  (argc, argv));
     case 37 : return(SyntaxSymbolCmd   (argc, argv));
     case 38 : return(SyntaxUpdateCmd   (argc, argv));
     case 39 : return(IsWindowSoftkey   (argc, argv));
     case 40 : return(SyntaxMonitorCmd  (argc, argv));
     case 41 : return(SyntaxIsAddr      (argc, argv));
     case 42 : return(SyntaxEventCmd    (argc, argv));
     case 43 : return(SyntaxStepCmd     (argc, argv));
     case 44 : return(SyntaxBkptCmd     (argc, argv));
     case 45 : return(IsString          (argc, argv));
	 case 46 : return(CORRECT);
     case 47 : return(IsNumber          (argc, argv));
     case 48 : return(SyntaxQuitCmd     (argc, argv));
     case 49 : return(SyntaxPatchCmd    (argc, argv));
	 case 50 : return(CORRECT);
	 case 51 : return(CORRECT);
     case 52 : return(IsBusEvent        (argc, argv));  //Modify Suspend to EV2
     case 53 : return(IsAddrData        (argc, argv));
     case 54 : return(SyntaxLogCmd      (argc, argv));
	 case 55 : return(CORRECT);
	 case 56 : return(CORRECT);
     case 57 : return(SyntaxModeCmd     (argc, argv));
     case 58 : return(SyntaxViewCmd     (argc, argv));
     case 59 : return(SyntaxFindCmd     (argc, argv));
     case 60 : return(SyntaxWatchCmd    (argc, argv));
     case 61 : return(SyntaxExtenCmd    (argc, argv));
     case 62 : return(SyntaxPathCmd     (argc, argv));
     case 63 : return(CORRECT);     // Matthew

	 case 65 : return(SyntaxCpuCmd		(argc, argv));	 //changed
     case 66 : return(CORRECT);     // Matthew
     case 67 : return(CORRECT);   // 7/08/94, FRANK
     case 68 : return(CORRECT);     // Matthew
	 case 69 : return(CORRECT);
	 case 70 : return(CORRECT);
	 case 71 : return(CORRECT);
	 case 72 : return(CORRECT);
     case 73 : return(CORRECT);
     case 74 : return(SyntaxEventCmd    (argc, argv));
     case 75 : return(SyntaxEventCmd    (argc, argv));
     case 76 : return(IsBusEvent        (argc, argv));
     case 78 : return(IsString          (argc, argv));  // Chen 07/12/94
	 case 79 : return(SyntaxCpuCmd		(argc, argv));	//changed
	 case 80 : return(CORRECT);
     default : return( ERROR );

   }
}  /* end of CheckSyntax */
/**************************************************************************
**
** Execution codes (Private)
**
**************************************************************************/

/*******************************************************************************
**
** Name : IsNoArgument()
**
** Function : Check and dispatch "No argument" syntax type
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax.
**
** RETCODE : CORRECT or ERROR
**
**  Commands: BAck, HAlt, IDentify, DIagnostic, MODule, REcall, SAve,
**            REPEATDIAG, SETup
**  0:  no argument
**
*******************************************************************************/
PRIVATE RETCODE IsNoArgument( S16 argc,  S8 argv[90][80])
{
   if (argc < 1 ) {
      cmd_syntax.argc=0;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsNoArgument()

/*****************************************************************************
**
**  Name : SyntaxCpuCmd()
**
**  Coded by Frank
******************************************************************************/
PRIVATE RETCODE SyntaxCpuCmd( S16 argc,  S8 argv[90][80])
{
 int temp;

   if (argc < 1 ) {
      cmd_syntax.argc=0;
      return(CORRECT);
   }
   if (argc==1){
	  if ( MICE <= 8 || MICE == 17 ){ // 80186/80188 & 80286 mode
		  if ( !stricmp(argv[0],"COMMAND"))     temp = 0;
		  else if ( !stricmp(argv[0],"QUEUE"))  temp = 1;
		  else return(ERROR);
	  }
	  else { //V20/V30 mode
		  if ( !stricmp(argv[0],"INTEL"))     temp = 0;
		  else if ( !stricmp(argv[0],"NEC"))  temp = 1;
          else return(ERROR);
          if (argc > 1) return(ERROR);
	  }
      cmd_syntax.argc= 1;
	  cmd_syntax.argv[0] = temp;
      return(CORRECT);
   }
   return(ERROR);
}  //end of syntaxCpuCmd()

/*******************************************************************************
**
** Name : SyntaxGoCmd()
**
** Function : Check syntax and dispatch arguments for Go command
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  4: [@n|[Run] [From addr]|[From addr] [Till] addr1]
** .Go                 : argc = 0;
** .Go Run             : argc = 1;
** .Go From addr       : argc = 2;  fill addr in addr_block;
** .Go Run From addr   : argc = 3;  fill addr in addr_block;
**  Go Till addr1      : argc = 4;  fill addr1 in addr_block;
**  Go From addr addr1 : argc = 5;  fill addr, addr1 in addr_block;
**  Go From addr Till addr1 : argc = 5;  fill addr, addr1 in addr_block;
** .Go @n              : argc = 6;  argv[0] = n;
*******************************************************************************/
PRIVATE RETCODE SyntaxGoCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr1, jj, ii;
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;

   if (!argc) {
      cmd_syntax.argc=0;
      return (CORRECT);  // Go
   }
   if (argc == 1 && argv[0][0] == AT) { // @n
      num = strtoul( argv[0]+1, &ptr1, 10);
      if ( *ptr1 || num > 0xFFFF ) return(ERROR);
      cmd_syntax.argc = 6;
      cmd_syntax.argv[0] = num;
      return(CORRECT);   // Go @n
   }
   ii = 0;
   if ( stricmp( argv[0],"RUN" ) == 0 ) {
      cmd_syntax.argc = 1;
      if (argc == 1) return( CORRECT ); //  Go Run
      ii = 1;
   }
   jj = 0;
   if ( argc >= 2+ii && stricmp( argv[ii], "FROM" ) == 0 &&
        IsAddr( argv[1+ii], &flag1, &addr1, &type)) {
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      if ((ii==1) && (argc == 3)) {
         cmd_syntax.argc = 3;
         return(CORRECT); // Go Run From Addr
      }
      if (argc == 2) {
         cmd_syntax.argc = 2;
         return( CORRECT ); // Go From Addr
      }
      jj = 2;
   }
   if ( stricmp( argv[jj],"TILL" ) == 0) jj++; // 3 or 2  or 1
   if ((argc == jj+1 ) && IsAddr( argv[jj], &flag2, &addr2, &type)) {
      cmd_syntax.argc = (jj >= 2)? 5 : 4;
      addr_block.CSflag2 = flag2;
      addr_block.addr2 = addr2;
      return( CORRECT ); // Go [From addr] [Till] addr1
   }
   return(ERROR);
}    // end of SyntaxGoCmd()

/*************************************************************************
**
** Name : SyntaxGoCmd2()
**
** Function : Check syntax and dispatch arguments for Go command FOR 8086
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  4: [Run] [addr]
**  Go                 : argc = 0;
**  Go Run             : argc = 1;
**  Go addr            : argc = 2;  fill addr1 in addr_block;
**  Go Run addr        : argc = 3;  fill addr1 in addr_block;
************************************************************************/
/**PRIVATE RETCODE SyntaxGoCmd2( S16 argc, S8 argv[90][80] )
{

}
 **/
/*******************************************************************************
**
** Name : SyntaxHelpCmd()
**
** Function : Check syntax and dispatch arguments for Help command
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  5:  [command_keyword]
**  no argument  : argc = 0;
**  Help Keyword : argc = 1; cmd_syntax.asc = command_keyword;
*******************************************************************************/
PRIVATE RETCODE SyntaxHelpCmd( S16 argc, S8 argv[90][80] )
{
   if (  argc < 2  ) {
      cmd_syntax.argc=argc;
      if ( !argc ) return(CORRECT);
      else if ( argc > 0 ) {
         strcpy( cmd_syntax.asc, argv[0]);
         return(CORRECT);
      }
   }
   return(ERROR);
}  // end of SyntaxHelpCmd()

/*******************************************************************************
**
** Name : IsOnOff()
**
** Function : Check syntax and dispatch arguments for commands
**            INterval and Verify.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: INterval, Verify
**  6: [On|OFf]
**  no argument: argc = 0;
**  On/OFf     : argc = 1; argv[0] = 1 / 0;
*******************************************************************************/
PRIVATE RETCODE IsOnOff( S16 argc, S8 argv[90][80] )
{
U8 number;

   if ( argc < 1 ) {
      cmd_syntax.argc=argc;
      return (CORRECT);
   }
   if ( argc < 2 ) {
      if  ( stricmp( argv[0],"ON" ) == 0 )  number=1;
      else if ( stricmp( argv[0],"OFF" ) == 0 ) number = 0;
      else return(ERROR);
      cmd_syntax.argc=argc;
      cmd_syntax.argv[0]= number;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsOnOff()

/*******************************************************************************
**
** Name : SyntaxIsAddr()
**
** Function : Check syntax and dispatch argument for Assemble and
**            Jump commands
**
**    Input  : argc, argv[90][80], cmdIdx
**    Output : struct cmd_syntax, addr_form
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Jump
**  7: addr
**  Commands: Assemble
**  41: [addr]
**  no argument : argc = 0;
**  addr        : argc = 1; fill addr_form
**  Notes :     ip => csFlag = 0;
**           cs:ip => csFlag = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxIsAddr( S16 argc, S8 argv[90][80] )
{
U32 num;
U16 type;
FLAG flag;

   if (cmdIdx == C_ASSEMBLE && argc < 1) {
      cmd_syntax.argc=argc;
      return (CORRECT);
   }
   if ( argc == 1 && IsAddr(argv[0], &flag, &num, &type) ) {
      cmd_syntax.argc=argc;
      addr_form.csFlag = flag;
      addr_form.symFlag = 0; /* hex addr */
      addr_form.addr = num;
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxIsAddr()

/****************************************************************
**
** Name : IsIntExt()
**
** Function : Check syntax and dispatch arguments for REAdy cmd
**            and CLock cmd.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: REAdy , CLock
**  8: [Internal|External]
**
**  no argument       : argc = 0;
**  Internal/External : argc = 1; argv[0] = 0 / 1;
****************************************************************/
PRIVATE RETCODE IsIntExt( S16 argc, S8 argv[90][80] )
{
U8 number;

   if (argc < 1 ) {
      cmd_syntax.argc=argc;
      return (CORRECT);
   }
   if ( argc < 2 ) {
      if ( stricmp( argv[0],"INTERNAL" ) == 0 ) number=0;
      else if  ( stricmp( argv[0],"EXTERNAL" )  == 0 ) number=1;
      else return(ERROR);
      cmd_syntax.argc=argc;
      cmd_syntax.argv[0]= number;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsIntExt()

/*******************************************************************************
**
** Name : IsNumber()
**
** Function : Check syntax and dispatch arguments for syntax "[number]"
**
**    Input  : argc, argv[90][80], cmdIdx
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands:  HIstory, BUffer
**  9: [number]
**  Commands:  CNT0, CNT1
**  25: [count]
**  Commands:  WAIt
**  47: [0|1|2|3|4|5]
**  Commands:  TImer
**  51: [timer]
**  no argument : cmd_syntax.argc = 0;
**  number      : cmd_syntax.argc = 1;  cmd_syntax.argv[0] = number
*******************************************************************************/
PRIVATE RETCODE IsNumber( S16 argc, S8 argv[90][80] )
{
U32 num;

   if ( argc > 1 ) return(ERROR);
   if (argc == 1) {
      if (cmdIdx == C_WAIT) {
         if (!IsInNumberRange(argv[0], HEX, 0, 5L, &num )) return(ERROR);
      }
      else if (!IsInNumberRange(argv[0], HEX, 0, 0xFFFFFFFF, &num))
         return(ERROR);
      cmd_syntax.argv[0]= num;
   }
   cmd_syntax.argc=argc;
   return(CORRECT);
}  // end of IsNumber()

/*******************************************************************************
**
** Name : SyntaxSyncCmd()
**
** Function : Check syntax and dispatch arguments for SYnc command
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: SYnc
**  10: [{Input|OUtput} {On|OFf}]
**  no argument : argc = 0;
**  else        : argc = 2; Input/OUtput : argv[0] = 0 / 1;
**                          On/OFf       : argv[1] = 1 / 0;
*******************************************************************************/
PRIVATE RETCODE SyntaxSyncCmd( S16 argc, S8 argv[90][80] )
{
U8 i, j;

   if (argc < 1 ){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( argc  == 2 ) {
      if ( stricmp( argv[0],"INPUT" ) == 0 ) i= 0;
      else if ( stricmp( argv[0],"OUTPUT" )  == 0 ) i=1;
      else return(ERROR);
      if ( stricmp( argv[1],"ON" ) == 0 ) j=1;
      else if ( stricmp( argv[1],"OFF" )  == 0 ) j=0;
      else return(ERROR);
      cmd_syntax.argc=argc;
      cmd_syntax.argv[0]= i;
      cmd_syntax.argv[1]= j;
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxSyncCmd()

/*******************************************************************************
**
** Name : IsAddrData()
**
** Function : Check syntax and dispatch arguments for Byte, DWord
**            FLoat, LOng, WOrd, DOUble commands.
**
**    Input  : argc, argv[90][80], cmdIdx
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Byte, DWord, FLoat, LOng, WOrd
**  12: [addr [data1 [...data16]]]
**  Commands: DOUble
**  53: [addr [data1 [...data8]]]
**  no argument  : argc = 0;
**  addr         : argc = 1; fill addr_block
**  addr data... : argc = 2; fill addr_block
**
**  Notes : The total length of data is limited up to 64 bytes.
*******************************************************************************/
PRIVATE RETCODE IsAddrData( S16 argc, S8 argv[90][80] )
{
U32 addr;
U16 type;
FLAG flag;
U8 temp[80], dest[80];
S16 i, j, len, cnt;

   if ( argc < 1 ) {
       cmd_syntax.argc=argc;
       return(CORRECT);
   }
   if ( IsAddr(argv[0], &flag, &addr, &type) &&  argc < 18 ) {
      if (synIdx == 53 && argc > 9) return(ERROR);
      memset(temp, NULL, sizeof(temp));
      memset(dest, NULL, sizeof(dest));
      addr_block.CSflag1 = flag;
      addr_block.addr1 = addr;
      addr_block.length = 1;
      addr_block.CSflag2 = -1;
      if ( argc > 1 ) {
         for ( i= 1, cnt=0; i< argc; i++ ) {
            if ( !IsData(argv[i], dest, &len, cmdIdx) ) return(ERROR);
            for (j=0; j < len; j++ ) temp[ cnt++ ] = dest[ j ];
            if (cnt > 64 ) return(ERROR);
         }
         cmd_syntax.argc = 2;
         addr_block.count = cnt;
         StrmnCopy( addr_block.buff, temp, 0, cnt );
      }
      else {
         cmd_syntax.argc = 1;
         addr_block.count = 0;
      }
      return( CORRECT );
   }
   return(ERROR);
}  // end of IsAddrData()

/*******************************************************************************
**
** Name : IsAddrRange()
**
** Function: Check syntax and dispatch arguments for CHecksum and Test commands.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  Commands: CHecksum, Test
**  13: addr1 {addr2|Length length}
**
*******************************************************************************/
/*****
PRIVATE RETCODE IsAddrRange( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
U32 currentSeg;
FLAG flag1, flag2;
U16 type;

   if (argc < 1) return(ERROR);

   if (IsAddr(argv[0], &flag1, &addr1, &type) ) {
      if (IsAddr( argv[1],&flag2, &addr2, &type) && ( argc == 2 ) ) {
         if (!IsAddrLength(&num,addr1,addr2,flag1,flag2))
                return(ERROR);
      }
      else
         if (stricmp( argv[1],"LENGTH" ) == 0 && IsLength( argv[2], &num ) &&
             argc == 3);
      else return(ERROR);
      if ( num >0x10000) return(ERROR);

      cmd_syntax.argc = 1;
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsAddrRange()
*****/

/*******************************************************************************
**
** Name : IsAddrRangeDest()
**
** Function: Check syntax and dispatch arguments for Compare command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  Commands: COMpare
**  14: addr1 {addr2|Length length} addr3
**  CLear       : argc = 1;
**  otherwise   : argc = 3; fill struc addr_block.
*******************************************************************************/
PRIVATE RETCODE IsAddrRangeDest( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;

   if ( IsAddr(argv[0], &flag1, &addr1, &type) ) {
      if ( IsAddr( argv[1],&flag2, &addr2, &type) && ( argc ==3 ) ) {
         if (!IsAddrLength(&num,addr1,addr2,flag1,flag2)) return(ERROR);
         if (!IsAddr(argv[2],&flag2,&addr2,&type)) return(ERROR);
      }
      else
         if ( ( stricmp( argv[1],"LENGTH" ) == 0 ) &&
           IsLength( argv[2], &num ) &&
           IsAddr( argv[3], &flag2, &addr2, &type ) && ( argc ==4 ) ) ;
         else return(ERROR);

      cmd_syntax.argc = 3;
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      addr_block.CSflag2 = flag2;
      addr_block.addr2 = addr2;
      addr_block.count = 0;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsAddrRangeDest()

/*******************************************************************************
**
** Name : IsOptAddrRange()
**
** Function: Check syntax and dispatch arguments for Disassemble and Memory
**           commands.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Disassemble, Memory
**  15: [addr1 [addr2|Length length]]
**  no argument : argc = 0;
**  addr1       : argc = 1; fill struc addr_block; addr_block.length = 0.
**  addr1 addr2 : argc = 2; fill struc addr_block;
**                addr_block.length = addr2-addr1+1.
**  addr1 Leng..: argc = 2; fill struc addr_block; addr_block.length = length.
*******************************************************************************/
PRIVATE RETCODE IsOptAddrRange( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;

   if ( argc  < 1 ) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( IsAddr(argv[0], &flag1, &addr1, &type))  {
      if ( argc < 2) num=0;
      else
         if ( IsAddr( argv[1],&flag2, &addr2, &type) && ( argc == 2 ) ) {
            if (!IsAddrLength(&num,addr1,addr2,flag1,flag2)) return(ERROR);
         }
         else
            if ( ( stricmp( argv[1],"LENGTH" ) == 0 ) &&
			   IsLength( argv[2], &num ) && ( argc == 3 ) ) argc=2;
            else return(ERROR);
	  if (cmdIdx == C_DISASSEMBLE && num > 0x10000) return(ERROR);
      cmd_syntax.argc=argc;
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      addr_block.CSflag2 = -1;
      addr_block.count = 0;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsOptAddrRange()

/*******************************************************************************
**
** Name : IsAddrRangeData()
**
** Function: Check syntax and dispatch arguments for Fill and SEarch
**           commands.
**
**    Input  : argc, argv[90][80], cmdIdx
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Fill, SEarch
**  16: addr1 {addr2|Length length} data1 [...data16]
**      cmd_syntax.argc = 3; fill addr_block.
**
**  Notes : The total length of data is limited up to 64 bytes.
*******************************************************************************/
PRIVATE RETCODE IsAddrRangeData( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;
S16 i, j, k, cnt, len;
U8 temp[80], dest[80];

   if ( IsAddr(argv[0],&flag1,&addr1, &type)) {
      memset(temp, NULL, sizeof(temp));
      memset(dest, NULL, sizeof(dest));
      if ( IsAddr( argv[1],&flag2, &addr2, &type) && ( argc >= 3 )) {
         if (!IsAddrLength(&num,addr1,addr2,flag1,flag2)) return(ERROR);
         i =  2;
      }
      else
         if (( stricmp(argv[1],"LENGTH") == 0 ) && ( argc >= 4 ) &&
            IsLength( argv[2], &num ) ) i=3;
         else return(ERROR);
      if ((argc - i ) > 16 ) return(ERROR);
      for ( j = i, cnt=0; j < argc; j++ ) {
          if ( !IsData(argv[j], dest, &len, cmdIdx) ) return(ERROR);
          for (k=0; k < len; k++ ) temp[ cnt++ ] = dest[ k ];
          if (cnt > 64 ) return(ERROR);
      }
      cmd_syntax.argc = 3;
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      addr_block.CSflag2 = -1;
      addr_block.count = cnt;
      StrmnCopy( addr_block.buff, temp, 0, cnt );
      return(CORRECT);
   }
   return(ERROR);
}   // end of IsAddrRangeData()

/*******************************************************************************
**
** Name : SyntaxInputCmd()
**
** Function: Check syntax and dispatch arguments for Input command
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Input
**  17: [Byte|Word] port
**   port      :   cmd_syntax.argc = 1, argv[0] = port
**   Byte port :   cmd_syntax.argc = 2, argv[0] = port; argv[1] = 0;
**   Word port :   cmd_syntax.argc = 2, argv[0] = port; argv[1] = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxInputCmd( S16 argc, S8 argv[90][80] )
{
U32 num;

   if (argc == 1 && IsInNumberRange(argv[0],HEX,0,0x0ffffL,&num)) ;
   else {
      if (argc == 2 && IsInNumberRange(argv[1],HEX,0,0x0ffffL,&num)) {
         if ( stricmp( argv[0],"BYTE" ) == 0 ) cmd_syntax.argv[1]=0;
         else {
            if ( stricmp( argv[0],"WORD" ) == 0 ) cmd_syntax.argv[1]=1;
            else return(ERROR);
         }
      }
      else return(ERROR);
   }
   cmd_syntax.argc = argc;
   cmd_syntax.argv[0] = num;
   return(CORRECT);
}   // end of SyntaxInputCmd()

/*******************************************************************************
**
** Name : SyntaxOutputCmd()
**
** Function: Check syntax and dispatch arguments for Output command.
**
**    Input  : argc, argv[90][80], cmdIdx
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: OutPut
**  19: [Byte|Word] port data
**   port data     : .argc = 2, argv[0] = port; argv[1] = data;
**   Byte port data: .argc = 3, argv[0] = port; argv[1] = data; argv[2] = 0;
**   Word port data: .argc = 3, argv[0] = port; argv[1] = data; argv[2] = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxOutputCmd( S16 argc, S8 argv[90][80] )
{
U32 num, number;
S16 len;
U8 dest[80];

   if ( ( argc == 2) && IsInNumberRange( argv[0], HEX, 0, 0x0ffffL,&num ) &&
      IsData( argv[1], dest, &len, C_BYTE  ) ) ;
   else
      if ( ( argc == 3 ) && IsInNumberRange( argv[1], HEX, 0, 0x0ffffL,&num )) {
         if (( stricmp( argv[0],"BYTE" ) == 0 ) &&
            IsData( argv[2], dest, &len, C_BYTE )) cmd_syntax.argv[2]=0;
         else
            if ( ( stricmp( argv[0],"WORD" ) == 0 ) &&
               IsData( argv[2], dest, &len, C_WORD )) cmd_syntax.argv[2]=1;
            else return(ERROR);
      }
      else return(ERROR);
   if ( len > 2 ) return(ERROR);
   number = dest[1] * 0x100 + dest[0];
   cmd_syntax.argc = argc;
   cmd_syntax.argv[0] = num;
   cmd_syntax.argv[1] = number ;
   return(CORRECT);
}   // end of SyntaxOutputCmd()

/*******************************************************************************
**
** Name : SyntaxResetCmd()
**
** Function: Check syntax and dispatch arguments for Reset command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: REset
**  20: [cs[:ip]]
**  no argument : argc = 0;
**  cs          : argc = 1; argv[0] = cs;
**  cs:ip       : argc = 2; argv[0] = cs, argv[1] = ip;
*******************************************************************************/
PRIVATE RETCODE SyntaxResetCmd( S16 argc, S8 argv[90][80] )
{
U32 addr;
FLAG flag;
U16 type;

   if ( argc < 1 ) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if (( argc < 2) && IsAddr(argv[0],&flag, &addr, &type )) {
      if (flag) {
         cmd_syntax.argc=2;
         cmd_syntax.argv[1]= addr&0x0ffffL;
         cmd_syntax.argv[0]= (addr>>16)&0x0ffffL;
      }
      else {
         cmd_syntax.argc=1;
         cmd_syntax.argv[0]= addr&0x0ffffL;
      }
      return(CORRECT);
   }
   return(ERROR);
}   // end of SyntaxResetCmd()

/*******************************************************************************
**
** Name : SyntaxSizeCmd()
**
** Function: Check syntax and dispatch arguments for Size command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Size
**  21: [Byte|Word]
**  no argument : argc = 0;
**  Byte        : argc = 1; argv[0] = 0;
**  Word        : argc = 1; argv[0] = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxSizeCmd( S16 argc, S8 argv[90][80] )
{
   if (argc < 1 ) ;
   else
      if ( argc < 2 ) {
         if ( stricmp( argv[0],"BYTE" ) == 0 ) cmd_syntax.argv[0]=0;
         else
            if ( stricmp( argv[0],"WORD" ) == 0 ) cmd_syntax.argv[0]=1;
            else return(ERROR);
      }
      else return(ERROR);
   cmd_syntax.argc=argc;
   return(CORRECT);
}   // end of SyntaxSizeCmd()

/*******************************************************************************
**
** Name : SyntaxCopyCmd()
**
** Function: Check syntax and dispatch arguments for Copy command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_block
**
** RETCODE : CORRECT or ERROR
**
** Commands: Copy
** 22: addr1 {addr2|Length length} [addr3]
** addr1 {addr2|Length length}      : cmd_syntax.argc=2, fill struct addr_block
** addr1 {addr2|Length length} addr3: cmd_syntax.argc=3, fill struct addr_block
*******************************************************************************/
PRIVATE RETCODE SyntaxCopyCmd( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;

   if ( argc > 1 && IsAddr(argv[0], &flag1, &addr1, &type) ) {
      if (IsAddr(argv[1], &flag2, &addr2, &type) && argc < 4) {
         if (!IsAddrLength(&num,addr1,addr2,flag1,flag2)) return(ERROR);
         if (argc == 2 ) flag2 = -1;
         else if (!IsAddr( argv[2],&flag2, &addr2, &type)) return(ERROR);
         cmd_syntax.argc = argc;
      }
      else {
        if (stricmp( argv[1],"LENGTH" ) == 0 && argc < 5 &&
            IsLength( argv[2], &num )) {
           if (argc == 3 ) flag2 = -1;
           else if (!IsAddr(argv[3], &flag2, &addr2, &type)) return(ERROR);
           cmd_syntax.argc = argc-1;
        }
        else return(ERROR);
      }
      addr_block.CSflag1 = flag1;
      addr_block.addr1 = addr1;
      addr_block.length = num;
      addr_block.CSflag2 = flag2;
      if ( flag2 >= 0 ) addr_block.addr2 = addr2;
      addr_block.count = 0;
      return(CORRECT);
   }
   return(ERROR);
}   // end of SyntaxCopyCmd()

/*******************************************************************************
**
** Name : SyntaxMapCmd()
**
** Function: Check syntax and dispatch arguments for Map command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: MAp
**  24: [abs1 abs2 {IR|Internal|ER|External|Guard}|All External]
**  no argument : argc = 0;
**  all external: argc = 1;
**  otherwise   : argc = 2; argv[0] = abs1; argv[1] = abs2;
**                argv[2]=  IR:0, I:1, ER:2, E:3, G:4;
*******************************************************************************/
/*********************************
PRIVATE RETCODE SyntaxMapCmd( S16 argc, S8 argv[90][80] )
{
U32 addr1, addr2;
U8  number;

    if (argc < 1 ) {
       cmd_syntax.argc = argc;
       return(CORRECT);
    }
    if ( argc == 2 && stricmp( argv[0],"ALL" ) == 0 &&
         stricmp( argv[1],"EXTERNAL" ) == 0 ) {
       cmd_syntax.argc = 1;
       return(CORRECT);
    }
    if ( argc == 3 && IsAbs( argv[0], &addr1 ) && IsAbs( argv[1], &addr2 ) ) {
       if (addr1 > addr2 ) return (ERROR);
       if ( stricmp( argv[2],"IR" ) == 0 )              number = 0;
       else if ( stricmp( argv[2],"INTERNAL" ) == 0 )   number = 1;
       else if ( stricmp( argv[2],"ER" )  == 0 )        number = 2;
       else if ( stricmp( argv[2],"EXTERNAL" )  == 0 )  number = 3;
       else if ( stricmp( argv[2],"GUARD" ) == 0 )      number = 4;
       else return(ERROR);
       if (!HEMM && (number == 2 || number == 4)) return(ERROR);
       cmd_syntax.argc = 2;
       cmd_syntax.argv[0] = addr1;
       cmd_syntax.argv[1] = addr2;
       cmd_syntax.argv[2] = number;
       return(CORRECT);
    }
    return(ERROR);
}   // end of SyntaxMapCmd()
************************/

/*******************************************************************************
**
** Name : SyntaxRegCmd()
**
** Function: Check syntax and dispatch arguments for Register command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Register
**  26: [reg_id [data] |INternal|COprocessor]
**  no argument: argc = 0;
**  INternal   : argc = 1, argv[0] = 1
**  COprocessor: argc = 1, argv[0] = 2
**  reg_id     : argc = 2; argv[0] = reg. ID, argv[1] = reg. type
**  reg_id data: argc = 3, argv[0] = reg. ID, argv[1] = reg. type, argv[2]=data
**  reg_id     : argc = 6; argv[0] = reg. ID, argv[1] = Internal
**  reg_id     : argc = 7; argv[0] = reg. ID, argv[1] = Coprocessor
*******************************************************************************/
extern int NECflag;
PRIVATE RETCODE SyntaxRegCmd( S16 argc, S8 argv[90][80] )
{
S16 number, type;
U32 num;

   if ( argc < 1 ) return(CORRECT);

   if (NECflag == 1) { //for V20/V30
	   if ( !IsV20Reg( argv[0], &number) ) return(ERROR);
		   cmd_syntax.argv[0] = number;   // reg. ID
	   if (argc == 1) cmd_syntax.argc = 4;
	   if (argc == 2) {
			if ( !IsInNumberRange(argv[1], HEX, 0, 0xFFFFL,&num) )
				return(ERROR);
			if (number >= 14 && (num != 0 && num != 1) )
			   return(ERROR);
			 cmd_syntax.argv[2] = num;
			 cmd_syntax.argc = 5;
	   }
	   if (argc > 2 ) return(ERROR);
	   return(CORRECT);
   }

   if (argc == 1 && stricmp(argv[0], "INTERNAL") == 0) cmd_syntax.argv[0]=1;
   else  {
     if (argc ==1 && stricmp(argv[0], "COPROCESSOR") == 0) {
            cmd_syntax.argv[0] = 2;
        }
        // added by Chen, 09/02/94
        else if ( argc == 2 && !stricmp(argv[0], "INTERNAL") ) {
            cmd_syntax.argc = 6;
            return(CORRECT);
        }
        else if ( argc == 2 && !stricmp(argv[0], "COPROCESSOR") ) {
            cmd_syntax.argc = 7;
            return(CORRECT);
        }
	 else {
		if (argc < 3 && IsRegister(argv[0], &number, &type) ) {
		   cmd_syntax.argv[0] = number;   // reg. ID
		   cmd_syntax.argv[1] = type;	  // reg. type
		   cmd_syntax.argc = 2;
		   if (argc == 2) {
			  if (IsInNumberRange(argv[1], HEX, 0, 0xFFFFL,&num)) {
				 cmd_syntax.argv[2] = num;
				 cmd_syntax.argc = 3;
			  }
			  else return(ERROR);
		   }
		   return(CORRECT);
		}
		else return(ERROR);
     }
   }
   cmd_syntax.argc=argc;
   return(CORRECT);
}  // end of SyntaxRegCmd()

/*******************************************************************************
**
** Name : IsNo1No2()
**
** Function: Check syntax and dispatch arguments for CLear and Nomonitor
**           commands.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: CLear, Nomonitor
**  27: [no1 [no2]]
**  no argument: argc = 0;
**  no1        : argc = 1, argv[0] = no1
**  no2        : argc = 2, argv[0] = no1, argv[1] = no2
*******************************************************************************/
PRIVATE RETCODE IsNo1No2( S16 argc, S8 argv[90][80] )
{
U32 num, no;

   if ( argc < 1 );
   else
      if ( ( argc < 3 ) && IsInNumberRange(argv[0], HEX, 1, 0x100L,&num) ) {
         if (argc < 2) ;
         else
           if ( IsInNumberRange(argv[1], HEX, 1, 0x100L,&no ) && ( no > num))
              cmd_syntax.argv[1]= no;
           else return(ERROR);
         cmd_syntax.argv[0]= num;
      }
      else return(ERROR);
   cmd_syntax.argc=argc;
   return(CORRECT);
}  // end of IsNo1No2()

/*******************************************************************************
**
** Name : SyntaxCreateCmd()
**
** Function: Check syntax and dispatch arguments for CReate command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_form
**
** RETCODE : CORRECT or ERROR
**
**  Commands: CReate
**  28: %symbol hex_address
**  cmd_syntax.argc =2, fill addr_form
*******************************************************************************/
PRIVATE RETCODE SyntaxCreateCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr;
U32 addr;
FLAG flag;
U16 type;

   ptr = argv[0];
   if ( argv[0][0] == PS && strlen(argv[0]) > 1 &&
        strchr(argv[0]+1,'%')==NULL &&      // Chen 06/02/94
        strchr(argv[0]+1,'#')==NULL &&      // don't support the symbol name
        strchr(argv[0]+1,'@')==NULL &&      // contains %, # or @
        IsAddr( argv[1], &flag, &addr, &type ) && argc == 2 ) {
       cmd_syntax.argc=argc;
       addr_form.symFlag = 2; /* symbol and hex */
       addr_form.csFlag = flag;
       addr_form.addr = addr;
       strcpy( addr_form.sym, ptr+1 );
       return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxCreateCmd()

/*******************************************************************************
**
** Name : SyntaxDeleteCmd()
**
** Function: Check syntax and dispatch arguments for DElete command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, addr_form
**
** RETCODE : CORRECT or ERROR
**
**  Commands: DElete
**  29: %symbol
**  cmd_syntax.argc =1, fill addr_form.symFlag and addr_form.sym
*******************************************************************************/
PRIVATE RETCODE SyntaxDeleteCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr;

   ptr = argv[0];
   if ((argv[0][0] == PS) && (strlen(argv[0]) > 1) &&
        strchr(argv[0]+1,'%')==NULL &&      // Chen 06/02/94
        strchr(argv[0]+1,'#')==NULL &&      // don't support the symbol name
        strchr(argv[0]+1,'@')==NULL &&      // contains %, # or @
        ( argc < 2 ) ) {
      cmd_syntax.argc=argc;
      addr_form.symFlag = 1; /* symbol  */
      strcpy( addr_form.sym, ptr+1 );
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxDeleteCmd()

/*******************************************************************************
**
** Name : IsString()
**
** Function: Check string and dispatch arguments.
**
**    Input  : argc, argv[90][80], cmdIdx, synIdx
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: DOS
**  30: [dos_command]
**
**  Commands: DOSA      ; added by Chen, 07/12/94
**  78: dos_command
**
**  Commands: DOwnload, LOad, SRecall, SSave, Upload
**  31: filename
**
**  Commands: INClude
**  32: filename [(arg0,...,arg9)]
**
**  Commands: RECall, SAve
**  45: [filename]
**
**  Commands: ICe
**  67: ICE_command
**
*******************************************************************************/
PRIVATE RETCODE IsString( S16 argc, S8 argv[90][80] )
{
    // added by Chen, 07/12/94
    if ( cmdIdx == C_DOSA && argc == 0 ) {
        return(ERROR);
    }
   if (cmdIdx == C_INCLUDE && argc == 2) {
      if (argv[1][0] == LP && argv[1][strlen(argv[1])-1] == RP)
         strcpy( cmd_syntax.input, argv[1] );
      else return(ERROR);
   }
   if ((synIdx == 31 || synIdx == 67) && argc < 1 ) return(ERROR);
   if ( argc < 3  ) {
      if (cmdIdx != C_INCLUDE && argc > 1 ) return(ERROR);
      if (argc > 0) strcpy( cmd_syntax.asc, argv[0]);
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   return(ERROR);
}  // end of IsString()

/*******************************************************************************
**
** Name : SyntaxJournalCmd()
**
** Function: Check syntax and dispatch arguments for JOurnal command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: JOurnal
**  33: [filename [Append]|Close]
**
**  no argument: cmd_syntax.argc =0;
**  Close      : cmd_syntax.argc =1;
**  Log Close                   : argc =3; argv[0] = 2;
**  filename   : cmd_syntax.argc =2, .argv[0] = 0, .asc = filename
**  Log filename                : argc =1; argv[0] = 1; cmd_syntax.asc = filename
**  filename  Append : cmd_syntax.argc =2, .argv[0] = 1, .asc = filename
**  Log filename Append         : argc =2; argv[0] = 2; cmd_syntax.asc = filename
*******************************************************************************/
PRIVATE RETCODE SyntaxJournalCmd( S16 argc, S8 argv[90][80] )
{  
   if ( argc < 1 || (argc < 2 && stricmp( argv[0],"CLOSE" ) == 0 )) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( stricmp(argv[1],"APPEND" ) == 0 && argc == 2 ) cmd_syntax.argv[0] = 1;
   else
      if ( argc < 2 ) cmd_syntax.argv[0]=0;
      else return(ERROR);
   cmd_syntax.argc = 2;
   strcpy ( cmd_syntax.asc, argv[0] );
   return(CORRECT);
}  // end of SyntaxJournalCmd()

/*******************************************************************************
**
** Name : SyntaxOptionsCmd()
**
** Function: Check syntax and dispatch arguments for OPtions command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: OPtions
**  34: [{Symbol|Log|Journal|Breakpoint|Case} {On|OFf}]
**  no argument: cmd_syntax.argc =0;
**  otherwise  : cmd_syntax.argc =2:
**  {Symbol|Log|Journal|Breakpoint|Case} : cmd_syntax.argv[0] = {0|1|2|3|4}
**  {On|OFf} : cmd_syntax.argv[1] = { 0 | 1 }
*******************************************************************************/
PRIVATE RETCODE SyntaxOptionsCmd( S16 argc, S8 argv[90][80] )
{
U8 number, no;

   if  (argc < 1 ) ;
   else
      if  ( argc < 3 ) {
         if ( stricmp( argv[0],"SYMBOL" ) == 0 ) number = 0;
         else if  ( stricmp( argv[0],"LOG" )  == 0 )   number = 1;
         else if  ( stricmp( argv[0],"JOURNAL" ) == 0 ) number = 2;
         else if  ( stricmp( argv[0],"BREAKPOINT" )  == 0 )  number = 3;
         else if  ( stricmp( argv[0],"CASE" )  == 0 )  number = 4;
         else return(ERROR);
         if ( stricmp( argv[1],"ON" ) == 0 ) no = 0;
         else if ( stricmp( argv[1],"OFF" )  == 0 ) no = 1;
         else return(ERROR);
         cmd_syntax.argv[0] = number;
         cmd_syntax.argv[1] = no;
      }
      else return(ERROR);
   cmd_syntax.argc=argc;
   return(CORRECT);
}  // end of SyntaxOptionsCmd()

/**************************************************************************
**
** Name : SyntaxQueryCmd()
**
** Function: Check syntax and dispatch arguments for QUEry command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: QUEry
**  35: [%%module]{%symbol|#line_number}|hex_address
**
**  hex_address :  argc=1, addr_form.symFlag=0; addr_form.addr = addr
**  %%module%sym:  argc=1, addr_form.symFlag=1, addr_form.sym = %%module%sym;
**  %symbol :      argc=1, addr_form.symFlag=1, addr_form.sym = %%module%sym;
**  #line_number : argc=2, argv[0]=line_number;
**  %%module#line: argc=3, argv[0]=line_number, cmd_syntax.asc = module;
***************************************************************************/
PRIVATE RETCODE SyntaxQueryCmd( S16 argc, S8 argv[90][80] )
{
S16 len;
U32 addr, num;
FLAG flag;
S8 *ptr, *ptr1, *ptr2;
U16 type;

   if ( IsAddr(argv[0],&flag,&addr, &type ) &&  argc == 1)  {
       addr_form.symFlag = 0; /* hex */
       addr_form.csFlag = flag;
       addr_form.addr = addr;
       cmd_syntax.argc = argc;
       return(CORRECT);
   }
   ptr = argv[0];
   if ( *ptr == NS ) {
      if (!IsInNumberRange(ptr+1, DEC, 1, 0x0ffffL, &num)) return(ERROR);
      cmd_syntax.argv[0] = num;
      cmd_syntax.argc = 2;
      return(CORRECT);
   }
   if ( argv[0][0] == PS && (len=strlen(argv[0])) > 1 && argc == 1) {
      if ( argv[0][1] == PS && argv[0][2] != PS) {
         /* with module */
         ptr1 = strchr(ptr+2,PS);
         ptr2 = strchr(ptr+2,NS);
         if ( (!ptr1 && !ptr2) || (ptr1 && ptr2) ) return(ERROR);
         if (ptr2) {
            if (!IsInNumberRange(ptr2+1, DEC, 1, 0x0ffffL, &num))
               return(ERROR);
            cmd_syntax.argv[0] = num;
            StrmnCopy( cmd_syntax.asc, ptr, 2, ptr2-ptr-2);
            cmd_syntax.argc = 3;
            return(CORRECT);
         }
         if ( ptr1 && (ptr1-ptr == strlen(argv[0])-1)) return(ERROR);
      }
      else if( strchr(ptr+1, PS ) ) return(ERROR);
      addr_form.symFlag = 1; /* symbol */
      strcpy( addr_form.sym, argv[0] );
      cmd_syntax.argc = argc;
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxQueryCmd()

/*******************************************************************************
**
** Name : SyntaxSoftkeyCmd()
**
** Function: Check syntax and dispatch arguments for Softkey command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: SOftkey
**  36: [{K1|K2|K3|K4|K5|K6|K7|K8|K9|K10} "command_description"]
*******************************************************************************/
PRIVATE RETCODE SyntaxSoftkeyCmd( S16 argc, S8 argv[90][80] )
{
S8 k, *ptr;
S16 len, number;

   if ( argc < 1 ) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( ((k = argv[0][0]) == 'K' || k=='k' ) && argc == 2 )  {
      ptr = argv[0];
      len = strlen( argv[1] );
      if ((argv[1][0] != DQ ) || (argv[1][len-1] != DQ)) return(ERROR);
      StrmnCopy( cmd_syntax.asc, argv[1], 1, len-2);
      if ( ((number= atoi( 1 + ptr )) > 0) && (number < 11 ) )
          cmd_syntax.argv[0] = number;
      else return(ERROR);
      cmd_syntax.argc = argc;
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxSoftkeyCmd()

/*******************************************************************************
**
** Name : SyntaxSymbolCmd()
**
** Function: Check syntax and dispatch arguments for SYMbol command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: SYMbol
**  37: [All|Global|%%module]
*******************************************************************************/
PRIVATE RETCODE SyntaxSymbolCmd( S16 argc, S8 argv[90][80] )
{
U8 number;
S16 len;

   if ( argc == 0 ) ;
   else if ( argc < 2 ) {
      if ( stricmp( argv[0],"ALL" ) == 0 ) number=0;
      else
         if ( stricmp(argv[0],"GLOBAL" )==0 ) number=1;
         else {
            if ( strlen(argv[0]) > 2 && (argv[0][0] == PS) &&
                 (argv[0][1] == PS ) ) {
               number = 2;
               len = strlen(argv[0] );
               StrmnCopy ( cmd_syntax.asc, argv[0], 2, len );
            } else  return(ERROR);
         }
   } else return(ERROR);
   cmd_syntax.argc = argc;
   cmd_syntax.argv[0] = number;
   return(CORRECT);
} // end of SyntaxSymbolCmd()

/*******************************************************************************
**
** Name : SyntaxUpdateCmd()
**
** Function: Check syntax and dispatch arguments for UPDate command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: UPDate
**  38: [{Breakpoint|CODe|Register|Stack|Trace|Data} {On|OFf}]
**  no argument: cmd_syntax.argc =0;
**  otherwise  : cmd_syntax.argc =2:
**  {Breakpoint|CODe|Register|Stack|Trace|Data}:.argv[0] = {0|1|2|3|4|5}
**  {On|OFf} : cmd_syntax.argv[1] = { 0 | 1 }
*******************************************************************************/
PRIVATE RETCODE SyntaxUpdateCmd( S16 argc, S8 argv[90][80] )
{
U8 number, no;

    if  (argc < 1 ) ;
    else if  ( argc < 3 ) {
       if  ( stricmp( argv[0],"BREAKPOINT" )  == 0 )  number = 0;
       else if ( stricmp( argv[0],"CODE" ) == 0 ) number = 1;
       else if  ( stricmp( argv[0],"REGISTER" )  == 0 )   number = 2;
       else if  ( stricmp( argv[0],"STACK" ) == 0 ) number = 3;
       else if  ( stricmp( argv[0],"TRACE" ) == 0 ) number = 4;
       else if  ( stricmp( argv[0],"DATA" ) == 0 ) number = 5;
       else return(ERROR);
       if ( stricmp( argv[1],"ON" ) == 0 ) no = 0;
       else if ( stricmp( argv[1],"OFF" )  == 0 ) no = 1;
       else return(ERROR);
       cmd_syntax.argv[0] = number;
       cmd_syntax.argv[1] = no;
    }
    else return(ERROR);
    cmd_syntax.argc=argc;
    return(CORRECT);
}  // end of SyntaxUpdateCmd()

/*******************************************************************************
**
** Name : IsWindowSoftkey()
**
** Function: Check syntax and dispatch arguments for USave and URecall commands.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: URecall, USave
**  39: [Window|Softkey]
**  no argument: cmd_syntax.argc =0;
**  otherwise  : cmd_syntax.argc =1;
**  [Window|Softkey] : cmd_syntax.argv[0] = [ 0 | 1 ]
******************************************************************************/
PRIVATE RETCODE IsWindowSoftkey( S16 argc, S8 argv[90][80] )
{
   if ( argc < 1 ) ;
   else if ( argc < 2 )  {
      if (stricmp( argv[0],"WINDOW") ==0 )  cmd_syntax.argv[0] = 0;
      else if (stricmp( argv[0],"SOFTKEY") ==0 ) cmd_syntax.argv[0] = 1;
      else return(ERROR);
   }
   else return(ERROR);
   cmd_syntax.argc=argc;
   return(CORRECT);
} // end of IsWindowSoftkey()

/*******************************************************************************
**
** Name : SyntaxMonitorCmd()
**
** Function: Check syntax and dispatch arguments for Monitor command.
**
**    Input  : argc, argv[90][80], isLocal, localBlkAddr
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: MONitor
**
**   40: [[%%module] %symbol|addr [number] [Byte|Word|Long]]
**
**  no argument : argc = 0;
**  argc=3; argv[0] = number, argv[1] = Byte/Word/Long(0/1/2), argv[2] = type,
**          argv[3] = isLocal(0/1/2), argv[4] = localBlkAddr, asc = %%module
**          addr filled in struct addr_form
*******************************************************************************/
PRIVATE RETCODE SyntaxMonitorCmd( S16 argc, S8 argv[90][80] )
{
S16 len;
S8 *ptr1, *ptr2, *ptr, chk, j;
U32 num, addr1;
U16 type;
FLAG flag1;

    if ( argc < 1 ) {
       cmd_syntax.argc=argc;
       return(CORRECT);
    }

    isLocal =  chk = 0;
    localBlkAddr = 0L;

    /*  MONitor [%%module]%symbol */
    if ( argv[0][0] == PS && ((len = strlen(argv[0])) > 1)) {
       ptr = argv[0];
       if ( argv[0][1] == PS ) {
          ptr1 = strchr(ptr+2,PS);
          if ( !(ptr1) || ( (ptr1-ptr) == ( strlen(argv[0]) - 1)) )
             return(ERROR);
       }
       else if( strchr(ptr+1, PS ) ) return(ERROR);
       chk = 1;
       strcpy(addr_form.sym, 1+argv[0] );

    }


    /* MONitor [addr] [number] [BYTE|WORD|LONG] */

    if (!IsAddr(argv[0],&flag1, &addr1, &type)) return(ERROR);
    num = 1;        // default data number
    len = 0;        // default size: byte


// added by Chen
    if ( argv[0][0]==PS && argc!=1 ) {      // MONitor %%module%var Byte ...
        return (ERROR);
    }

if ( argv[0][0] != PS ) {
    if ( argc == 2 ) {  // MONitor [addr] [number] or MONitor [addr] [B|W|L]
        if ( stricmp(argv[1], "BYTE") == 0 ) {
            len=0;
        }
        else if ( stricmp(argv[1], "WORD") == 0 ) {
            len = 1;
        }
        else if ( stricmp(argv[1], "LONG" ) == 0 ) {
            len = 2;
        }
        else if ( IsInNumberRange(argv[1],HEX,1,0xFFFFL,&num)==CORRECT ) {
            len = 0 ;
        }
        else {
            return (ERROR);
        }
    }
    else if ( argc == 3 ) {     // MONitor [addr] [number] [Byte|Word|Long]
        if ( IsInNumberRange(argv[1], HEX, 1, 0xFFFFL, &num) == CORRECT ) {
            if ( stricmp(argv[2], "BYTE") == 0 ) {
                len=0;
            }
            else if ( stricmp(argv[2], "WORD") == 0 ) {
                len = 1;
            }
            else if ( stricmp(argv[2], "LONG" ) == 0 ) {
                len = 2;
            }
            else {
                return (ERROR);
            }
        }
        else {
            return (ERROR);
        }
    }
    else if ( argc != 1 ) {
        return (ERROR);
    }
}


/*
// marked by Chen 05/26/94
    if ( (argc > 1) && (argc < 4) ) {
       j= 1;
       if ( ( IsInNumberRange(argv[1], HEX, 1, 0xFFFFFFFF,&num) ) ) {
          j = (argc < 3) ? 0 : 2;
       }
       else num = 1;
       if (j) {
          if ( stricmp( argv[j],"BYTE" )  == 0 ) len=0;
          else if ( stricmp( argv[j],"WORD" ) == 0 ) len = 1;
          else if ( stricmp( argv[j],"LONG" )  == 0 ) len = 2;
          else return(ERROR);
       }
    }
*/



    cmd_syntax.argc = 3;
    cmd_syntax.argv[0] = num;
    cmd_syntax.argv[1] = len;
    cmd_syntax.argv[2] = type;
    cmd_syntax.argv[3] = isLocal;
    cmd_syntax.argv[4] = localBlkAddr;
    addr_form.symFlag = chk;
    addr_form.addr = addr1;
    addr_form.csFlag = flag1;
    return(CORRECT);
}  // end of SyntaxMonitorCmd

/*******************************************************************************
**
** Name : SyntaxEventCmd()
**
** Function: Check syntax and dispatch arguments for Event command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, cmdIdx
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Event
**  42:  [1|2|3|4|5|6|7|8|CLear]
**  74:  [1|2|3|4|5|6|CLear]
**  75:  [1|2|3|4|5|6|7|CLear]
**  no argument : argc = 0;
**  CLear       : argc = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxEventCmd( S16 argc, S8 argv[90][80] )
{
S16 i;
S8 *pStr;
S16 diff;

   if (argc < 1 || (argc==1 && stricmp(argv[0], "CLEAR") == 0)){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if (!LAM2){
	  if (MICE <= 8 || MICE == 17)	diff = 6; //80186LAM
	  if (MICE >= 9)  diff = 7; //8086/88 LAM
   }
   else{
      if (MICE <= 8) diff = 8;
      else return(ERROR);
   }

   if ( (i=atoi(argv[0])) > 0 && i <= diff ) {
	   pStr = argv[1];
	   if (!LAM2){
		   switch(i){
			   case 1: synIdx = 1;	break;
			   case 2: synIdx = 52; break;
			   case 3: synIdx = 2;	break;
			   case 4: if (MICE >= 9 && MICE != 17) synIdx = 25;
					   else synIdx = 3;
					   break;
			   case 5:
			   case 6: synIdx = 3;	break;
               case 7: if (MICE >=9 && MICE != 17) synIdx = 3;
					   else return(ERROR);
					   break;
		   }
       }
	   else {
		  switch(i){
			   case 1:
			   case 2:
			   case 3:
			   case 4: synIdx = 76; break;
			   case 5: synIdx = 2;	break;
			   case 6:
			   case 7:
			   case 8: synIdx = 3;	break;
		   }
	   }
   }
   else return(ERROR);

    cmdIdx = C_EV1 + i - 1;
    if ( !(i < 3+LAM2 && IsBusEvent( argc-1, pStr)) &&
         !(i == 3+LAM2 && IsExtEvent( argc-1, pStr)) &&
         !(i > 3+LAM2 && i <= diff && synIdx != 25
             && IsExecEvent(argc-1, pStr)) &&
         !(synIdx ==25 && IsExecEvent2(argc-1, pStr)))
            return (ERROR);
    return( CORRECT );

}  // end of SyntaxEventCmd

/*******************************************************************************
**
** Name : SyntaxBkptCmd()
**
** Function: Check syntax and dispatch arguments for BReakpoint command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands:  BReakpoint
**  44: [addr [count]]
**  no argument : argc = 0;
**  addr        : argc = 1; fill addr_form (see IsAddr for details)
**  addr count  : argc = 2; argv[0] = count; fill addr_form
*******************************************************************************/
PRIVATE RETCODE SyntaxBkptCmd( S16 argc, S8 argv[90][80] )
{
U32 addr1, num;
U16 type;
FLAG flag;

   if ( argc < 1 );
   else  {
      addr_form.symFlag = 0; /* hex addr */
      if (argv[0][0] == PS) {
         addr_form.symFlag = 1; /* symbol  */
         strcpy( addr_form.sym, argv[0] );
      }
      if ( argc < 3 && IsAddr( argv[0], &flag, &addr1, &type ) ) {
         if (argc < 2) ;
         else
            if (IsInNumberRange(argv[1], HEX, 1, 0xFFFFFFFF,&num))
                cmd_syntax.argv[0]= num;
            else return(ERROR);
         addr_form.csFlag = flag;
         addr_form.addr = addr1;
      }
      else return(ERROR);
   }
   cmd_syntax.argc=argc;
   return(CORRECT);
}  // end of SyntaxBkptCmd

/*******************************************************************************
**
** Name : SyntaxQuitCmd()
**
** Function: Check syntax and dispatch arguments for Quit command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Quit
**  48: [Y]
**  no argument : cmd_syntax.argc = 0;
**  Y           : cmd_syntax.argc = 1;
*******************************************************************************/
PRIVATE RETCODE SyntaxQuitCmd( S16 argc, S8 argv[90][80] )
{
   if (argc < 1 || (argc ==1 && stricmp(argv[0],"Y")== 0 )) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxQuitCmd

/*******************************************************************************
**
** Name : SyntaxPatchCmd()
**
** Function: Check syntax and dispatch arguments for Patch command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, patch_addr
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Patch
**  49: [addr1 {[addr2] [With addr3 addr4]}|CLear]
**  no argument : argc = 0;
**  CLear       : argc = 1;
**  addr1 addr2 : argc = 2; fill struc patch_addr.
**  addr1 With addr3 addr4       : argc = 3; fill struc patch_addr.
**  addr1 addr2 With addr3 addr4 : argc = 4; fill struc patch_addr.
*******************************************************************************/
PRIVATE RETCODE SyntaxPatchCmd( S16 argc, S8 argv[90][80] )
{
U32 num, addr1, addr2;
FLAG flag1, flag2;
U16 type;
U8 i;

   if ( !argc || (argc == 1  && stricmp( argv[0], "CLEAR" ) == 0 )){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( IsAddr(argv[0], &flag1, &addr1, &type) ) {
      i = 1;
      patch_addr.addr1 = addr1;
      patch_addr.CSflag1= flag1;
      if ( argc > 1 && IsAddr( argv[1], &flag2, &addr2, &type) ) {
         if (!IsAddrLength(&num,addr1,addr2,flag1,flag2) || num > 0x10000)
            return(ERROR);
         patch_addr.addr2 = addr2;
         if (argc == 2) {
            cmd_syntax.argc = 2;
            return(CORRECT);
         }
         i = 2;
      }
      if ( argc == i+3 && stricmp( argv[i],"WITH" ) == 0 &&
           IsAddr( argv[i+1], &flag1, &addr1, &type ) &&
           IsAddr( argv[i+2], &flag2, &addr2, &type ) &&
           IsAddrLength(&num,addr1,addr2,flag1,flag2) && num <= 0x10000) {
         patch_addr.CSflag2 = flag1;
         patch_addr.addr3 = addr1;
         patch_addr.addr4 = addr2;
         cmd_syntax.argc = (i < 2) ? 3 : 4;
         return(CORRECT);
      }
      return(ERROR);
   }
   return(ERROR);
}  // end of SyntaxPatchCmd

/*******************************************************************************
**
** Name : SyntaxSuspendCmd()
**
** Function: Check syntax and dispatch arguments for SUspned command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands:  SUspend
**  52: [{0|1|2} Enable|Disable]
**  SUspend            : argc = 0;
**  SUspend Disable    : argc = 1;
**  SUspend {0|1|2} Enable : argc = 2;  argv[0]= {0|1|2}
*******************************************************************************/
/*****PRIVATE RETCODE SyntaxSuspendCmd( S16 argc, S8 argv[90][80] )
{
U32 num;

   if (!argc || (argc == 1 && stricmp( argv[0],"DISABLE" ) == 0)) {
      cmd_syntax.argc = argc;
      return(CORRECT);
   }
   if (argc==2 && IsInNumberRange(argv[0], HEX, 0, 2L, &num) &&
       stricmp( argv[1],"ENABLE" )  == 0 ) {
      cmd_syntax.argc = argc;
      cmd_syntax.argv[0] = num;
      return(CORRECT);
   }
   return(ERROR);
} *****/
 // end of SyntaxSuspendCmd

/*******************************************************************************
**
** Name : SyntaxLogCmd()
**
** Function: Check syntax and dispatch arguments for Log command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Log
**  54: [[Command|Menu] {filename [Append]|Close}]
**  no argument                 : argc =0;
**  Log filename                : argc =1; argv[0] = 1; cmd_syntax.asc = filename
**  Log filename Append         : argc =2; argv[0] = 2; cmd_syntax.asc = filename
**  Log Close                   : argc =3; argv[0] = 2;
**  Log Command filename        : argc =1; argv[0] = 1; cmd_syntax.asc = filename
**  Log Command filename Append : argc =2; argv[0] = 1; cmd_syntax.asc = filename
**  Log Command Close           : argc =3; argv[0] = 1;
**  Log Menu filename           : argc =1; argv[0] = 0; cmd_syntax.asc = filename
**  Log Menu filename Append    : argc =2; argv[0] = 0; cmd_syntax.asc = filename
**  Log Menu Close              : argc =3; argv[0] = 0;
********************************************************************************/
PRIVATE RETCODE SyntaxLogCmd( S16 argc, S8 argv[90][80] )
{
   if ( argc < 1){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if ( argc < 2 && stricmp( argv[0],"CLOSE" ) == 0 ) {
      cmd_syntax.argc = 3;
      cmd_syntax.argv[0] = 2;
      return(CORRECT);
   }

   if ( stricmp(argv[1],"APPEND" ) == 0 && argc == 2 ){
       cmd_syntax.argc = 2;
       cmd_syntax.argv[0] = 2;
       strcpy ( cmd_syntax.asc, argv[0] );
       return(CORRECT);
   }
   if ( argc < 2 ){
       cmd_syntax.argc = 1;
       cmd_syntax.argv[0]=1;
       strcpy ( cmd_syntax.asc, argv[0] );
       return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxLogCmd


/****
U8 i;

   if (  argc < 1 ) {
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   i = 0;
   if ( stricmp( argv[0], "COMMAND" ) == 0 ) cmd_syntax.argv[0] = i = 1;
   else if ( stricmp( argv[0],"MENU" ) == 0 ) {
      i = 1;
      cmd_syntax.argv[0] = 0;
   }
   if (argc >= i+1 ) {
      if (stricmp( argv[i], "CLOSE" ) == 0)  {
         cmd_syntax.argc = 3;
         if (!i) cmd_syntax.argv[0] = 2;      // Log Close
         if (argc < 3 ) return( CORRECT );    // Log [Command|Menu] Close
         return( ERROR );
      }
      strcpy ( cmd_syntax.asc, argv[i] );
      if (!i) cmd_syntax.argv[0] = 1;         // Log filename
      if (argc == i + 1) cmd_syntax.argc = 1; // Log [Command|Menu] filename
      else
         if ( stricmp( argv[i+1],"APPEND" ) == 0 ) {
            if (!i) cmd_syntax.argv[0] = 2;      // Log filename Append
            cmd_syntax.argc = 2;    // Log [Command|Menu] filename Append
         }
         else return( ERROR );
      return( CORRECT );
   }
   return( ERROR );
}  // end of SyntaxLogCmd
****/


/*******************************************************************************
**
** Name : SyntaxModeCmd()
**
** Function: Check syntax and dispatch arguments for MOde command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: MOde
**  57: [Source|Mixed|Assembly]
**  no argument     :argc =0;
**  Assembly       : argc =1; argv[0] = 0;
**  Source         : argc =1; argv[0] = 1;
**  Mixed          : argc =1; argv[0] = 2;
*******************************************************************************/
PRIVATE RETCODE SyntaxModeCmd( S16 argc, S8 argv[90][80] )
{
   if ( argc < 1 ) ;
   else if ( argc < 2 )  {
      if      (stricmp( argv[0],"ASSEMBLY") ==0 ) cmd_syntax.argv[0] = 0;
      else if (stricmp( argv[0],"SOURCE") ==0 ) cmd_syntax.argv[0] = 1;
      else if (stricmp( argv[0],"MIXED") ==0 )  cmd_syntax.argv[0]  = 2;
      else return(ERROR);
   }
   else return(ERROR);
   cmd_syntax.argc=argc;
   return(CORRECT);
}   // end of SyntaxModeCmd

/*******************************************************************************
**
** Name : SyntaxViewCmd()
**
** Function: Check syntax and dispatch arguments for VIew command.
**
**    Input  : argc, argv[90][80], langMode
**    Output : struct cmd_syntax, addr_form, syntaxErrFlag
**
** RETCODE : CORRECT or ERROR
**
**  Commands: VIew
**  58: [@n|addr|[%%module][#line_number]]
**  no argument : argc=0;
**  addr        : argc=1; fill struc addr_form
**  #line_number: argc=2, argv[0] = line_number;
**  %%module    : argc=3, argv[0] = 1, cmd_syntax.asc = module_name;
**  %%module#line_number: argc=3, argv[0] = line_number, cmd_syntax.asc = module_name;
**  @n          : argc=4, argv[0] = n;
******************************************************************************/
PRIVATE RETCODE SyntaxViewCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr, *ptr1, *ptr2;
U32 num;
S16 len;
FLAG flag;
U16 type;

   if ( argc < 1 ) {
      cmd_syntax.argc=argc;
      return (CORRECT);
   }
   if (argc == 1 ) {
      ptr = argv[0];
      if (argv[0][0] == AT) {
         if (!IsInNumberRange(ptr+1, DEC, 0, 0x0ffffL, &num))
            return(ERROR);

// added by Chen 05/26/94
        if ( argv[0][1]=='0' ) {        // VIew @0 = VIew
            cmd_syntax.argc = 0;
            return(CORRECT);
        }

         cmd_syntax.argc = 4;
         cmd_syntax.argv[0] = num;
         return(CORRECT);
      }
      if ( *ptr == NS ) { // #line_number
         if (langMode==ASM) {
            syntaxErrFlag = 76; // Line number is not available
                                // for assembly mode!
            return(ERROR);
         }
         if (!IsInNumberRange(ptr+1, DEC, 1, 0x0ffffL, &num))
            return(ERROR);
         cmd_syntax.argv[0] = num;
         cmd_syntax.argc = 2;
         return(CORRECT);
      }
      if ((len=strlen(argv[0])) > 1 && argv[0][0] == PS &&
           argv[0][1] == PS) {   /* with module */
         ptr1 = strchr(ptr+2,NS);
         ptr2 = strchr(ptr+2,PS);
         if (ptr1 && ptr2) return (ERROR);
         if (ptr1) {  /*  %%module#line_number  */
            if (langMode==ASM) {
               syntaxErrFlag = 76; // Line number is not available
                                   // for assembly mode!
               return(ERROR);
            }
           if (!IsInNumberRange(ptr1+1, DEC, 1, 0x0ffffL, &num))
               return(ERROR);
            cmd_syntax.argv[0] = num;
            StrmnCopy( cmd_syntax.asc, ptr, 2, ptr1-ptr-2);
            cmd_syntax.argc = 3;
            return(CORRECT);
         }
         else if (!ptr2) {  /*  %%module */
            cmd_syntax.argv[0] = 1L;
            strcpy( cmd_syntax.asc, ptr+2 );
            cmd_syntax.argc = 3;
            return(CORRECT);
         }
      }
      if ( IsAddr(argv[0], &flag, &num, &type ) ) {
         addr_form.symFlag = 0;
         addr_form.csFlag = flag;
         addr_form.addr = num;
         cmd_syntax.argc = argc;
         return(CORRECT);
      }
   }
   return(ERROR);
}  // end of SyntaxViewCmd

/*******************************************************************************
**
** Name : SyntaxFindCmd()
**
** Function: Check syntax and dispatch arguments for FINd command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: FINd
**  59: ["string"] [%%module][#line_number] [Forward|Backward]
**  no argument         : argc = 1, argv[0] = 0, argv[1] = 0;
**  #line_num           : argc = 1, argv[0] = line, argv[1] = 0;
**  #line_num For|Back  : argc = 1, argv[0] = line, argv[1] = 1|2;
**  Forward|Backward    : argc = 1, argv[0] = 0, argv[1] = 1|2;
**  "string"            : argc = 2, argc[0] = 0, argv[1] = 0, .asc=string;
**  "string" Forward    : argc = 2, argv[0] = 0, argv[1] = 1, .asc= string;
**  "string" Backward   : argc = 2, argv[0] = 0, argv[1] = 2, .asc= string;
**  "string" #line_num  : argc = 2, argv[0] = line, argv[1] = 0, .asc= string;
**  "string" #line Forw : argc = 2, argv[0] = line, argv[1] = 1, .asc= string;
**  "string" #line Back : argc = 2, argv[0] = line, argv[1] = 2, .asc= string;
**  %%mod#line_num      : argc = 3, argv[0] = line, argv[1] = 0, .input= module;
**  %%mod#line For|Back : argc = 3, argv[0] = line, argv[1] = 1|2, .input= module;
**  %%mod Forward|Back  : argc = 3, argv[0] = 1, argv[1] = 1|2, .input= module;
**  "string" %%mod       : argc = 4, argc[0] = 1, argv[1] = 0, .asc=string, .input= module;
**  "string" %%mod For   : argc = 4, argv[0] = 1, argv[1] = 1, .asc= string, .input= module;
**  "string" %%mod Back  : argc = 4, argv[0] = 1, argv[1] = 2, .asc= string, .input= module;
**  "str" %%mod#line_num : argc = 4, argv[0] = line, argv[1] = 0, .asc= string, .input= module;
**  "str" %%mod#line Forw: argc = 4, argv[0] = line, argv[1] = 1, .asc= string, .input= module;
**  "str" %%mod#line Back: argc = 4, argv[0] = line, argv[1] = 2, .asc= string, .input= module;
******************************************************************************/
PRIVATE RETCODE SyntaxFindCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr, *ptr1, *ptr2, i;
U32 num;
S16 len;

   cmd_syntax.argc = 1;
   cmd_syntax.argv[0] = cmd_syntax.argv[1] = 0L;
   if (argc < 1) return(CORRECT);
   len = strlen(argv[0]);
   i=0;
   if (argv[0][0] == DQ && argv[0][len-1] == DQ) {
      cmd_syntax.argc = 2;
      StrmnCopy( cmd_syntax.asc, argv[0], 1, len-2 );
      if (argc == 1) return(CORRECT);
      i++;
   }
   if (argc >= i+1 && argv[i][0] == NS) {
      if (!IsInNumberRange(argv[i]+1, DEC, 1, 0x0ffffL, &num))
         return(ERROR);
      cmd_syntax.argv[0] = num;
      if (argc == i+1) return(CORRECT);
      i++;
   }
   else if (argc >= i+1 && (len=strlen(argv[i])) > 1 &&
            argv[i][0] == PS && argv[i][1] == PS) {   /* with module */
      ptr = argv[i];
      ptr1 = strchr(ptr+2,NS);
      ptr2 = strchr(ptr+2,PS);
      if (ptr1 && ptr2) return (ERROR);
      if (ptr1) {  /* [string] %%module#line_number  */
         if (!IsInNumberRange(ptr1+1, DEC, 1, 0x0ffffL, &num))
            return(ERROR);
         cmd_syntax.argv[0] = num;
         StrmnCopy( cmd_syntax.input, ptr, 2, ptr1-ptr-2);
         cmd_syntax.argc = i > 0 ? 4 : 3;
         if (argc == i+1) return(CORRECT);
         i++;
      }
      else if (!ptr2) {  /* [string] %%module */
         cmd_syntax.argv[0] = 1L;
         strcpy( cmd_syntax.input, ptr+2 );
         cmd_syntax.argc = i > 0 ? 4 : 3;
         if (argc == i+1) return(CORRECT);
         i++;
      }
   }
   if (argc == i+1) {
      if      (!stricmp( argv[i],"FORWARD"))  cmd_syntax.argv[1] = 1;
      else if (!stricmp( argv[i],"BACKWARD")) cmd_syntax.argv[1] = 2;
      else return(ERROR);
      return(CORRECT);
   }
   return(ERROR);
}  // end of SyntaxFindCmd

/*******************************************************************************
**
** Name : SyntaxWatchCmd()
**
** Function: Check syntax and dispatch arguments for Watch command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**  Commands: Watch
**  60: [@n][%variable_name]
**  no argument      : argc = 0;
**  @n               : argc = 1, argv[0] = n;
**  %variable_name   : argc = 2, cmd_syntax.asc = variable_name;
**  @n%variable_name : argc = 3, argv[0] = n, .asc= variable_name;
*******************************************************************************/
PRIVATE RETCODE SyntaxWatchCmd( S16 argc, S8 argv[90][80] )
{
S8 *ptr1, *ptr2;
U32 num;
U8 temp[80];

   if (!argc) {
      cmd_syntax.argc = argc;
      return(CORRECT);
   }
   if (argc == 1) {
      ptr1 = strchr(argv[0], PS);
      ptr2 = strchr(argv[0], AT);
      if ((!ptr1 && !ptr2) || (ptr2 && ptr2 != argv[0])) return(ERROR);
      if (ptr1 && ptr2) {              // @n%variable_name
         StrmnCopy( temp, ptr2, 1, ptr1 - ptr2 -1 );
         if (!IsInNumberRange(temp, DEC, 0, 0x0ffffL, &num)) return(ERROR);
         cmd_syntax.argc = 3;
         cmd_syntax.argv[0] = num;
         strcpy( cmd_syntax.asc, ptr1+1 );
         return(CORRECT);
      }
      if ( ptr1 ) {                     // %variable_name
         cmd_syntax.argc = 2;
         strcpy( cmd_syntax.asc, ptr1+1 );
         return(CORRECT);
      }
      if (!IsInNumberRange(ptr2+1, DEC, 0, 0x0ffffL, &num)) return(ERROR);
      cmd_syntax.argc = 1;
      cmd_syntax.argv[0] = num;
      return(CORRECT);
   }
   return(ERROR);
} // end of SyntaxWatchCmd

/*******************************************************************************
**
** Name : SyntaxExtenCmd()
**
** Function: Check syntax and dispatch arguments for EXtension command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, extName[]
**
** RETCODE : CORRECT or ERROR
**
**  Commands: EXtension
**  61 : [Clear|.ext[ .ext[ .ext[ .ext]]]]
**    no arguments,  cmd_syntax.argc= 0 ; Clear,  cmd_syntax.argc= 5 ;
**            .ext,  cmd_syntax.argc= 1 ; .ext .ext,            .argc= 2 ;
**  .ext .ext .ext,  cmd_syntax.argc= 3 ; .ext .ext .ext .ext,  .argc=4;
**
******************************************************************************/
PRIVATE RETCODE SyntaxExtenCmd( S16 argc, S8 argv[90][80] )
{
S16 i;

   if (argc > 4) return(ERROR);
   if(argc==1 && !stricmp(argv[0],"CLEAR")){
      cmd_syntax.argc=5;
      return(CORRECT);
   }
   else if (argc) {
      for ( i=0; i < argc; i++)
         if (argv[i][0] != '.' || strlen(argv[i]) > 4 ) return(ERROR);
      memset( extName, NULL, sizeof(extName) );
      for ( i=0; i < argc; i++) StrmnCopy(extName[i], argv[i], 0, 4);
   }
   cmd_syntax.argc = argc;
   return(CORRECT);
}  // end of SyntaxExtenCmd

/*******************************************************************************
**
** Name : SyntaxPathCmd()
**
** Function: Check syntax and dispatch arguments for PATH command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax, pathName[]
**
** RETCODE : CORRECT or ERROR
**
**  Commands: PATH
**  62 : [Clear|path[ path[ path[ path]]]]
*******************************************************************************/
PRIVATE RETCODE SyntaxPathCmd( S16 argc, S8 argv[90][80] )
{
S16 i;

   if (argc > 4) return(ERROR);
   if(argc==1 && !stricmp(argv[0],"CLEAR")){
      cmd_syntax.argc= 5;
      return(CORRECT);
   }
   else if (argc) {
      memset( pathName, NULL, sizeof(pathName) );
      for ( i=0; i < argc; i++) StrmnCopy(pathName[i], argv[i], 0, 49);
   }
   cmd_syntax.argc = argc;
   return(CORRECT);
}  // end of SyntaxPathCmd

/***********************************************************
** Name: CYcle
**
** Function: Check syntax and dispatch arguments for CYcle command.
**
**    Input  : argc, argv[90][80]
**    Output : struct cmd_syntax
**
** RETCODE : CORRECT or ERROR
**
**   Command: CYcle
**   63: [Wait|count]
**   0: no argument   1: Wait  2: count
***********************************************************/
PRIVATE RETCODE SyntaxCycleCmd(S16 argc,S8 argv[90][80])
{
U32 num;
   if(!argc ){
      cmd_syntax.argc=argc;
      return(CORRECT);
   }
   if(argc >1) return(ERROR);
   if (stricmp(argv[0],"WAIT") == 0 ){
      cmd_syntax.argc = 1;
      return(CORRECT);
   }

   if(IsInNumberRange(argv[0],HEX,0,0x7ffl,&num)){
      cmd_syntax.argc = 2;
      cmd_syntax.argv[0] = (U16)num;
      return(CORRECT);
   }
   else return(ERROR);
} /* end of CYcle() */

/************************************************************
**  Name : SyntaxTbaseCmd
**
**
**
**
**
**
*************************************************************/
PRIVATE RETCODE SyntaxTbaseCmd(S16 argc, U8 argv[90][80])
{
   int i;

   if( argc < 1 )  return(CORRECT);
   if( argc > 1 )  return(ERROR);

   if (!LAM2){
        if( stricmp(argv[0],"1") && stricmp(argv[0],"10") &&
            stricmp(argv[0],"100")  && stricmp(argv[0],"1000") &&
            stricmp(argv[0],"10000") ) return(ERROR);
        cmd_syntax.argc= 1;
        strcpy(cmd_syntax.asc,argv[0]);
   }
   else {
        if( stricmp(argv[0],"1") && stricmp(argv[0],"10") &&
            stricmp(argv[0],"100")  && stricmp(argv[0],"1000") &&
            stricmp(argv[0],"0.1") ) return(ERROR);
        cmd_syntax.argc= 1;
        strcpy(cmd_syntax.asc,argv[0]);
   }
        return(CORRECT);
}


/************************************************************
**  Name : SyntaxMathCmd()
**
**
**
**
**
**
*************************************************************/
// 06/06/1994 James Wang
PRIVATE RETCODE SyntaxMathCmd(int argc, char argv[90][80])
{
   int i;

   cmd_syntax.argc = 1;
   for( i= 0; i < argc; i++){
      if( strlen(cmd_syntax.asc) > 512 ) return(ERROR);
      strcat(cmd_syntax.asc, argv[i]);
      cmd_syntax.argc++;
   }

   return(CORRECT);
}


/***************************************************************************
**
**  Name : SyntaxCoverage  (this function only for MICE-IIIs)
**
**  Function : check the syntax of command COVerage and INItialize
**
**  Syntax : [adr1 adr2[ CLear]|Enable|Disable|CLear|Reset|{I|O|S|R|W}]
**
**  Input :
**
**  Output :
**
**  Notes : no argument : argc = 0;
**          adr1 adr2 [CLear] : argc = 1, fill addr_block,
**                              argv[0]=0 or 1 (CLear);
**          enable  : argc = 2 ;
**          disable : argc = 3 ;
**          clear   : argc = 4 ;
**          reset   : argc = 5 ;
**          {I|O|S|R|W} : argc = 6; argv[0] = 0|1|2|3|4
**          list    : argc = 7; fill addr_form
****************************************************************************/
PRIVATE RETCODE SyntaxCoverage ( S16 argc, U8 argv[90][80])
{
   U32 addr1, addr2;
   S16 dummy1, dummy2;
   FLAG flag1, flag2;

   if (argc < 1) return(CORRECT);

   /* addr1 addr2 [CLear] */
   if (IsAddr(argv[0], &flag1, &addr1, &dummy1) &&
       IsAddr(argv[1], &flag2, &addr2, &dummy2)){
         if (argc == 3 && !stricmp(argv[2], "CLEAR") )
            cmd_syntax.argv[0] = 1;
         else if (argc == 2) cmd_syntax.argv[0] = 0;
         else return(ERROR);
         cmd_syntax.argc = 1;
         addr_block.CSflag1 = flag1;
         addr_block.CSflag2 = flag2;
         addr_block.addr1 = addr1;
         addr_block.addr2 = addr2;
         return(CORRECT);
   }

   if (argc == 1 && !stricmp(argv[0], "ENABLE"))
       cmd_syntax.argc = 2;
   else if (argc == 1 && !stricmp(argv[0], "DISABLE"))
       cmd_syntax.argc = 3;
   else if (argc == 1 && !stricmp(argv[0], "CLEAR"))
       cmd_syntax.argc = 4;
   else if (argc == 1 && !stricmp(argv[0], "RESET"))
       cmd_syntax.argc = 5;
   else if (argc == 1 && strlen(argv[0]) == 1 &&
            strpbrk(argv[0], "IOSRW") != NULL){
            switch(argv[0][0]){
               case 'I': cmd_syntax.argv[0] = 0; break;
               case 'O': cmd_syntax.argv[0] = 1; break;
               case 'S': cmd_syntax.argv[0] = 2; break;
               case 'R': cmd_syntax.argv[0] = 3; break;
               case 'W': cmd_syntax.argv[0] = 4; break;
         }
         cmd_syntax.argc = 6;
   }
   else if (argc <= 2 && !stricmp(argv[0], "LIST") ){
       cmd_syntax.argc = 7;
       if (argc == 1) return(CORRECT);
       if (!IsAddr(argv[1], &flag1, &addr1, &dummy1)) return(ERROR);
       addr_form.csFlag = flag1;
       addr_form.addr =  addr1;
       cmd_syntax.argv[0] = 1;
   }

   if (cmd_syntax.argc != 0) return(CORRECT);
   else return(ERROR);
}


/************************** End of File ***************************************/

