
/* Name: addr.h
**
** Description:   provide the address convertion routines to other servers
**
** Status: PRELIMINARY
**
** Copyright (C) 1991 Microtek International.  All rights reserved.
**
**
*/

#define _ADDR_


#define ADDR_BUFF_SZ 20    /* used for allocating strings to be passed back*/

/* used for allocating string for AdrConvAddressToTextWithParams when output
   is symbolic */
#define ADDR_BUFF_SZ_SYMBOLIC 256

typedef enum { /* types of addresses that can be referred to */
   ADDR_LOGICAL, /* Motorola logical space (like Intel's linear) */
   ADDR_VIRTUAL, /* Intel's segment/offset (real/v86 mode) or selector/offset
                    (protected modes) */
   ADDR_LINEAR, /* Intel linear space (real/v86 is segment shifted 4 bits and
                   added to offset); protected uses GDT entry base + offset*/
   ADDR_PHYSICAL /* Intel/Motorola output on address pins (after page tables,
                    if used or same as linear/logical if no page tables) */
} ADDR_TYPE;

typedef enum { /* different processor modes that affect address calculation*/
   MODE_NONE, /* no mode, this (or current) is used for non-x86 processors */
   MODE_CURRENT,  /* use the current processor state */
   MODE_REAL, /* x86 real mode (segment:offset) */
   MODE_PROTECT16, /* x86 16 bit protected mode (selector:offset) */
   MODE_PROTECT32, /* x86 32 bit protected mode (selector:offset) */
   MODE_VIRTUAL_86 /* x86 virtual 86 mode (segment:offset) */
} ADDR_MODE;

#ifndef _ADDR_SPACE_
typedef enum { /* Motorola address spaces */
   SPACE_UNDEF0, SPACE_UD, SPACE_UP, SPACE_UNDEF3, SPACE_UNDEF4,
   SPACE_SD, SPACE_SP, SPACE_CPU, SPACE_DONT_CARE, SPACE_END = 0x7fff
} ADDR_SPACE;
#define _ADDR_SPACE_
#endif

typedef enum { /* results from the address compare routine */
   FIRST_ADDR_GREATER, SECOND_ADDR_GREATER, EQUAL_ADDRS
} ADDR_COMPARE;

typedef enum { /* how the segment (real/v86 mode) or selector (prot16/32
                  modes) should be interpreted */
   ADDR_USE_VALUE, /* use the value supplied by the caller */
   ADDR_USE_CS, /* use the current code segment value whenever referenced */
   ADDR_USE_DS, /* current data segment */
   ADDR_USE_SS /* current stack segment */
} ADDR_SEGSEL_TYPE;

                       /****************************
                        *                          *
                        *          ERRORS          *
                        *                          *
                        ****************************/

/* errors are in errtext/mkerrors.h */

#ifdef __cplusplus
extern "C" {
#endif
                        /****************************
                         *                          *
                         *        PROTOTYPES        *
                         *                          *
                         ****************************/

/*
**  Initialization of Server
*/

/***************************************************************************
**
**  AdrCreateAddress
**
**  Description:
**     This routine creates an opaque address descriptor.  After receiving
**     the descriptor, the other routines that follow are called to set and
**     manipulate the descriptor.  When done with descriptor,
**     AddrDestroyAddress eliminates the descriptor.
**
**     Sets the address to the default state.  The defaults are:
**
**     FAMILY_UNKNOWN (assumes straight 32 bit linear space):
**        address type:       ADDR_LOGICAL
**        address mode:       MODE_NONE
**        address space:      unused
**        ldt selector:       unused
**        symbol usage:       FALSE
**        segment/selector:   unused
**
**     FAMILY_68K:
**        address type:       ADDR_LOGICAL
**        address mode:       MODE_NONE
**        address space:      SPACE_UD
**        ldt selector:       unused
**        symbol usage:       FALSE
**        segment/selector:   unused
**
**     FAMILY_X86:
**        address type:       ADDR_VIRTUAL
**        address mode:       MODE_CURRENT
**        address space:      unused
**        ldt selector:       0
**        symbol usage:       FALSE
**        segment/selector:   type=ADDR_USE_DS, value=0
**
**  Parameters:
**     input:
**        none
**     output:
**        desc:  opaque address descriptor
**
****************************************************************************/
RETCODE EXPORT AdrCreateAddress(DESCRIPTOR FAR *desc);

/***************************************************************************
**
**  AdrDestroyAddress
**
**  Description:
**     This routine removes an address descriptor from use.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        none
**
****************************************************************************/
RETCODE EXPORT AdrDestroyAddress(DESCRIPTOR desc);

/***************************************************************************
**
**  AdrSetAddrSpace
**
**  Description:
**     This routine sets the address space.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        space:  space to set address to
**     output:
**        none
**
****************************************************************************/
RETCODE EXPORT AdrSetAddrSpace(DESCRIPTOR desc, ADDR_SPACE space);

/***************************************************************************
**
** AdrConvAddressToTextWithParams
**    - supercedes AdrConvAddressToText and AdrConvAddressToTextNoFill
**
** Description:
**    This routine converts an opaque address descriptor into an ASCII
**    text string.  The text string will be in a standardized form which
**    depends on the processor family, the type, mode, etc.
**
**    If symbolUsage == OFF, the address will be converted to hex.
**    Use ADDR_BUFF_SZ for allocating a string for the return text.
**
**    If symbolUsage == ON, the address will be converted to its symbol
**    match.  Use ADDR_BUFF_SZ_SYMBOLIC (much larger) for return string
**    allocation.
**
** Input parameters:
**    desc:  opaque address descriptor
**    with0X: if TRUE, 0x prepended to hex output
**    leadZeroFill: if TRUE, leading 0's will be used to make the hex output
**       a fixed width equal to the return parameter from proc.dll
**       ProcReturnMaxOutputAddrDigits call.
**    
**    output:
**       str: ASCII text of the address in hex or symbolic form
**
****************************************************************************/
RETCODE EXPORT
AdrConvAddressToTextWithParams(DESCRIPTOR desc,
                               BOOLEAN    with0X,
                               BOOLEAN    leadZeroFill,
                               LPSTR      str);

/***************************************************************************
**
**  AdrConvTextToAddress
**
**  Description:
**     This routine converts an ASCII text string to an opaque address
**     descriptor.  To use, first call AdrCreateAddress to get an address
**     structure pointed to by your descriptor.  Then call this routine.
**     This function fills in the address structure pointed to by the
**     passed-in descriptor.  When done with descriptor, call the
**     AdrDestroyAddress function.
**
**     The symbol usage flag (and AdrSetAddrSymbolUsage) are no longer used.
**
**     Leading white space (using isspace()) is skipped over.
**     The symbolic input must start with #.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        text:  ASCII text of the address
**     output:
**        desc:  the contents of the address descriptor are modified
**
****************************************************************************/
RETCODE EXPORT AdrConvTextToAddress(DESCRIPTOR desc, LPSTR text);

/*
**  Address Manipulation Routines
*/

/***************************************************************************
**
**  AdrAddToAddress
**
**  Description:
**     This routine adds an unsigned long number to an address.  This will
**     be interpreted differently depending on the family processor, address
**     type, mode, etc.  However, this routine takes care of all these messy
**     details, so you don't have to worry about them!
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        value:  value to add to address
**     output:
**        desc:  the contents of the address descriptor are modified
**
****************************************************************************/
RETCODE EXPORT AdrAddToAddress(DESCRIPTOR desc, U32 value);

/***************************************************************************
**
**  AdrSubtractFromAddress
**
**  Description:
**     This routine subtracts an unsigned long number from an address.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        value:  value to subtract from address
**     output:
**        desc:  the contents of the address descriptor are modified
**
****************************************************************************/
RETCODE EXPORT AdrSubtractFromAddress(DESCRIPTOR desc, U32 value);

/***************************************************************************
**
**  AdrCompareAddresses
**
**  Description:
**     This routine compares two addresses and reports which is higher
**     than the other (or they are equal).  If the addresses are not the
**     same type, mode, etc., a conversion to a common denominator type
**     (e.g. physical) will be made before the comparison is done.  The
**     conversion will not change the input parameters; it is only an
**     temporary (internal) representation.  Some combinations will not be
**     possible to compare and an error will be returned.
**
**  Parameters:
**     input:
**        desc1:   opaque descriptor of first address
**        desc2:   opaque descriptor of second address
**     output:
**        result:  result of comparison
**
****************************************************************************/
RETCODE EXPORT AdrCompareAddresses(DESCRIPTOR desc1, DESCRIPTOR desc2,
                                   ADDR_COMPARE FAR *result);

/***************************************************************************
**
**  AdrIsAddrInRange
**
**  Description:
**     This routine compares one address to see if it is included within the
**     range of the second input address.  "Included" means:
**       inputAddr >= rangeAddr start address  AND
**       inputAddr <= rangeAddr end address
**
**  Parameters:
**     input:
**        inputAddr: descriptor of single address to be compared with range
**        rangeAddr: descriptor of address range
**     output:
**        isInRange: TRUE if inputAddr in range of rangeAddr; FALSE if not
**
**  NOTE: error is returned if rangeAddr is not set as a range address
****************************************************************************/
RETCODE EXPORT AdrIsAddrInRange(DESCRIPTOR inputAddr, DESCRIPTOR rangeAddr,
                                BOOLEAN FAR *isInRange);

/***************************************************************************
**
**  AdrDuplicateAddress
**
**  Description:
**     This routine takes one address and makes an exact copy of it.  This
**     can be used if an original copy of an address needs to be saved.
**
**  Parameters:
**     input:
**        desc1:  opaque descriptor of source address (the one to duplicate)
**     output:
**        desc2:  opaque descriptor of destination address
**
****************************************************************************/
RETCODE EXPORT AdrDuplicateAddress(DESCRIPTOR desc1, DESCRIPTOR FAR *desc2);

/***************************************************************************
**
**  AdrSetEndAddrOffset
**
**  Description:
**     This routine sets the offset of the end address and sets the range
**     active flag to TRUE.  Generates error if the end address is greater
**     than the max address allowed for the processor.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        offset:  offset of the end address
**     output:
**        none
**
****************************************************************************/
RETCODE EXPORT AdrSetEndAddrOffset(DESCRIPTOR desc, U32 offset);

/***************************************************************************
**
**  AdrGetEndAddrOffset
**
**  Description:
**     This routine gets the offset of the end point of a range.  Use the
**     "is range active" to see if the end address is being used.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        offset:  offset of the end address
**
****************************************************************************/
RETCODE EXPORT AdrGetEndAddrOffset(DESCRIPTOR desc, U32 FAR *offset);

/*
**  Low Level Routines... these are needed by the symbol table server, but
**  should not be needed by other servers.
*/
/***************************************************************************
**
**  AdrSetAddrOffset
**
**  Description:
**     This routine sets the address offset.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**        offset:  offset to set address to
**     output:
**        none
**
****************************************************************************/
RETCODE EXPORT AdrSetAddrOffset(DESCRIPTOR desc, U32 offset);

/***************************************************************************
**
**  AdrGetAddrOffset
**
**  Description:
**     This routine gets the address offset.
**
**  Parameters:
**     input:
**        desc:  opaque address descriptor
**     output:
**        offset:  current setting of address offset
**
****************************************************************************/
RETCODE EXPORT AdrGetAddrOffset(DESCRIPTOR desc, U32 FAR *offset);

/******************************** E O F ***********************************/


/****************************************************************************
**
**    Name:    trace.h
** 
**    Description: 
**       Routines to maintain information on available traces
**
**
****************************************************************************/

#define _TRACE_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASETYPE_
#include "basetype.h"
#endif

#ifndef _DASM_
#include "dasm.h"
#endif

#ifndef _EVTTMPLT_
#include "evttmplt.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _SDPROBE_
#include "sdprobe.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define  NUM_HDR_LINES  3
#define  MAX_HDR_LINE   128

typedef struct {
   S32 frameNum;
   XILINX_GROUP frameInfo;
} RAW_FRAME_INFO;

typedef enum {START_ALIGN, TRIGGER_ALIGN, NO_ALIGN, FDM_DUMMY=0x7fff 
} FRAME_DISPLAY_MODE;

typedef enum { RELATIVETOFRAME, DELTA, TSM_DUMMY=0x7fff
} TIMESTAMP_MODE;

#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************************
** 
**    TrcReadBufferGet
** 
**    Description:
**       Returns the current value of the read trace buffer found in the
**       state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          buffer:        current starting trace buffer
**
****************************************************************************/
RETCODE EXPORT TrcReadBufferGet(DESCRIPTOR id, U16 FAR *buffer);
                                         
/****************************************************************************
** 
**    TrcFrameDisplayModeSet
** 
**    Description:
**       This routine basically sets up the frame numbering scheme using
**       within each trace buffer.  START_ALIGN means the first frame in
**       each trace buffer is numbered frame zero(0) and each following
**       frame is sequentially numbered.  TRIGGER_ALIGN means the trigger
**       frame is numbered frame zero(0) and each frame found before the
**       trigger has a negative number and each frame following the trigger
**       is a positive number. And finally NO_ALIGN means the frame numbers
**       correspond directly to the hardware frame number no adjustment 
**       is made this is an internal selection (to be used for diagnostics
**       purposes).
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          mode:          how to number frames within a trace buffer.
**       output:
**          none
**
****************************************************************************/
RETCODE EXPORT TrcFrameDisplayModeSet(DESCRIPTOR id,FRAME_DISPLAY_MODE mode);
      
/****************************************************************************
** 
**    TrcReadBufferGet
** 
**    Description:
**       Returns the current value of the read trace buffer found in the
**       state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          buffer:        current starting trace buffer
**
****************************************************************************/
RETCODE EXPORT TrcReadBufferGet(DESCRIPTOR id, U16 FAR *buffer);
      
/****************************************************************************
** 
**    TrcReadFrameGet
** 
**    Description:
**       Returns the current value of the read trace buffer frame found in
**       the state table for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frame:         current starting trace buffer frame
**
****************************************************************************/
RETCODE EXPORT TrcReadFrameGet(DESCRIPTOR id, S32 FAR *frame);
      
/****************************************************************************
** 
**    TrcReadFrameSet
** 
**    Description:
**       Sets the read trace buffer frame to 'frame' in the state table 
**       for the given calling routine.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frame:         new starting trace buffer frame
**       output:
**          none
**
****************************************************************************/
RETCODE EXPORT TrcReadFrameSet(DESCRIPTOR id, S32 frame);
      
/****************************************************************************
** 
**    TrcNumFramesGet
** 
**    Description:
**       Returns the number of trace frames in the active trace buffer. This
**       will return the number of trace collected not the physical number
**       of frames per buffer.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          frames:        number of frames
**
****************************************************************************/
RETCODE EXPORT TrcNumFramesGet(DESCRIPTOR id, U32 FAR *frames);
      
/****************************************************************************
** 
**    TrcTraceBufferInfoGet
** 
**    Description:
**       Returns the starting, ending and trigger frame number of the current
**       read trace buffer.  This is used by the presenter to display for 
**       the user.  The frame display mode affects the frame numbers.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          oldest:        frame number of first frame within trace buffer
**          youngest:      frame number of last frame within trace buffer
**          trigger:       frame number of trigger frame within trace buffer
**
****************************************************************************/
RETCODE EXPORT TrcTraceBufferInfoGet(DESCRIPTOR id, S32 FAR *oldest,
                                     S32 FAR *youngest, S32 FAR *trigger);
      
/****************************************************************************
** 
**    TrcGetCliId
** 
**    Description:
**       Returns the descriptor id from the trace cli commands.  If trace
**    hasn't been opened yet this call will "open" a trace.
** 
**    Parameters:
**       input:
**          pType:         Which processor to setup trace for.  Used only
**                         if trace isn't currently opened.
**       output:
**          trcCliId:      trace command id
**
****************************************************************************/
RETCODE EXPORT TrcGetCliId(DESCRIPTOR FAR *trcCliId);

/****************************************************************************
** 
**    TrcTimestampGet
** 
**    Description:
**       Returns TRUE if timestamp is enabled.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**       output:
**          value:         Value of timestampOn flag
**
****************************************************************************/
RETCODE EXPORT TrcTimestampGet(DESCRIPTOR id, BOOLEAN FAR *value);

/****************************************************************************
** 
**    TrcGetTimestampText
** 
**    Description:
**       Gets the text for the timestamp for the current frame. !!!
**    eventually we will need to pass the previous frame number to
**    support the relative timestamp mode.
** 
**    Parameters:
**       input:
**          id:            id number of calling routine.
**          frameNumber:   current frame we need timestamp text for
**          prevFrameNum:  frame number of previous frame used for relative
**                         TS
**          buffer:        caller allocate pointer for text
**       output:
**          none
**
****************************************************************************/
RETCODE EXPORT TrcGetTimestampText(DESCRIPTOR id, S32 frameNumber,
                                   S32 prevFrameNum, LPSTR buffer);

#ifdef __cplusplus
}
#endif
/*********************************** E O F *********************************/
