/****************************************************************************
**
**  Name:  sshared.h
**
**  Description:
**     Shared Data Server (used for both host and firmware)
**
**  Status:  PRELIMINARY
**
**  $Log:   S:/tbird/arcmmcf/inc/sshared.h_v  $
** 
**    Rev 1.2   17 Oct 1996 17:56:00   gene
** added SdsReadResetResult
** 
**    Rev 1.1   02 Oct 1996 17:22:22   gene
** added SdsSetSrcStepFlag prototype
** 
**    Rev 1.0   03 Jun 1996 11:25:32   gene
** Initial revision.
** 
**    Rev 1.0   14 Sep 1995 14:22:26   kevin
** Initial revision.
**  
**     Rev 1.15   28 Jun 1993 12:25:48   marilyn
**  Added if defs to handle conflicts between windows.h 3.0 vs. 3.1. and the
**  CALLBACK typedef.
**  
**     Rev 1.14   26 Jun 1993 12:14:58   ernie
**  Added prorotypes to support asynchronous I/O for packet transfers
**  
**     Rev 1.13   25 May 1993 11:23:28   ernie
**  Added SdnData() and SdnWriteMemberIfChanged() functions
**  
**     Rev 1.12   12 Mar 1993 09:48:18   mindy
**  added cplusplus ifdef support
**  
**     Rev 1.11   04 Dec 1992 12:38:32   doug
**  com port is now read in .ini and passed in to shared data
**  
**     Rev 1.10   04 Dec 1992 11:33:58   ernie
**  Added Sd(n)WritePartialMemberNoCallback function for PowerScope loader
**  
**     Rev 1.9   03 Dec 1992 14:48:32   ernie
**  Added Sd(n)WriteMemberNoCallback functions
**  
**     Rev 1.8   02 Nov 1992 13:04:06   ernie
**  Added Sd(n)WriteCmdChkAbortReadResponse entry points
**  
**     Rev 1.7   27 Aug 1992 06:34:10   mindy
**  added a read member routine that doesn't check input
**  
**     Rev 1.6   13 Aug 1992 11:10:32   doug
**  There is now always a time out in the write command read response
**  functions.  (So the system can never hang waiting for the emulator.)
**  The default time out is 5 seconds.
**  
**     Rev 1.5   28 Jul 1992 15:29:18   doug
**  added NULLPTR since it is not in sdtempl.h anymore
**  
**     Rev 1.4   08 Jun 1992 09:42:14   ernie
**  Added member index access routines to improve performance
**  
**     Rev 1.3   11 May 1992 09:06:02   mindy
**  added SdTerminateServer to kill timer before unloading shared data
**  
**     Rev 1.2   30 Oct 1991 16:00:30   doug
**  new enum names since name conflict
**  
**     Rev 1.1   29 Jul 1991 14:29:02   nghia
**  changes for basic system.
 *
 *    Rev 1.1   09 Jul 1991 15:39:02   nghia
 * Debugged and cleaned up to compile with the Trace Server.
**
**  $Header:   S:/tbird/arcmmcf/inc/sshared.h_v   1.2   17 Oct 1996 17:56:00   gene  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/

#define _SSHARED_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif


                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define NULLPTR NULL
/* !!! get rid of the one in sharedat.h */
typedef VOID EXPORT(*CALLBACKFNC)(DESCRIPTOR);

/* If included with the 3.0 windows.h, define compatible aliases */
#if !defined(WINVER) || (WINVER < 0x030a)
typedef FARPROC CALLBACK;
#endif //win3.0

typedef enum {
   MEMBER_READ_ONLY, MEMBER_WRITE_ONLY, MEMBER_READ_WRITE
} MEMBER_ACCESS;

typedef U32 MEMBER_BITS;
typedef U32 MEMBER_INDEX;
typedef U32 MEMBER_OFFSET;
typedef enum { MEMBER_HOST, MEMBER_FIRMWARE } MEMBER_RESIDENCE;
typedef U32 MEMBER_SIZE;
typedef char MEMBER_NAME;
typedef U32 TIMEOUT;  /* time out in number of seconds */

#ifdef __cplusplus
extern "C" {
#endif
                        /****************************
                         *                          *
                         *        PROTOTYPES        *
                         *                          *
                         ****************************/

/****************************************************************************
**
**  SdInitialize
**
**  Description:
**     Routine to call to initialize the shared data server.  This must be
**     called before any of the other calls can be made.
**
**  Parameters:
**     input:
**        residence:  inform whether this code will run on host or in
**                    firmware; this is used because code is common to both
**                    sides and there are a few subtle differences in the code
**        hWnd:       handle of main window
**        comPort:    comm port to initialize
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdInitialize(U16 hWnd, MEMBER_RESIDENCE residence,
   S16 *comPort);
                            /* !!! hWnd : HWND */

/****************************************************************************
**
**  SdsInputAvailable
**
**  Description:
**     Processes any input which has arrived since the last shared data call.
**     Used when busy-waiting for a callback to occur.
*****************************************************************************/
RETCODE EXPORT SdsInputAvailable(VOID);

/****************************************************************************
**
**  SdRegister
**
**  Description:
**     Get a descriptor to gain access to a shared data member.  Optionally,
**     register a callback function to be invoked whenever the value of the
**     member changes.  (Use NULLPTR to indicate no callback requested.)  If
**     the member name is not valid, an error is returned.
**
**  Parameters:
**     input:
**        name:  member name; TBird will maintain a list of members which will
**               depend on the system configuration.  See shared data template
**               for list of names.
**        updateFunc:  routine to call when the value of this member changes
**                     (or NULLPTR if no update notification requested)
**     output:
**        desc:  opaque descriptor which can be used to get the value, get
**               information about the member, unregister, etc.
**
*****************************************************************************/
#if !defined(WINVER) || (WINVER < 0x030a)  //win3.0
RETCODE EXPORT SdRegister(MEMBER_NAME FAR *name,
                          CALLBACK updateFunc,
                          DESCRIPTOR FAR *desc);
#else
RETCODE EXPORT SdRegister(MEMBER_NAME FAR *name,
                          FARPROC updateFunc,
                          DESCRIPTOR FAR *desc);
#endif

/****************************************************************************
**
**  SdUnRegister
**
**  Description:
**     Unregister a callback for a particular member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor receive when registered
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdUnRegister(DESCRIPTOR desc);

/****************************************************************************
**
**  SdRegisterSemaphore
**
**  Description:
**     This is a special case of registration.  It will search for an
**     available semaphore by name.  If there are more than one (quantity
**     field in shared data template is greater than 1 and a particular index
**     is not specified in the name), it will search for the first available
**     one.
**
**  Parameters:
**     input:
**        name:  member name
**        updateFunc:  routine to call when the value of this member changes
**                     (or NULLPTR if no update notification requested)
**     output:
**        desc:  opaque descriptor which can be used to get the value, get
**               information about the member, unregister, etc.
**
*****************************************************************************/
#if !defined(WINVER) || (WINVER < 0x030a)  //win3.0
RETCODE EXPORT SdRegisterSemaphore(MEMBER_NAME FAR *name,
                                   CALLBACK updateFunc,
                                   DESCRIPTOR FAR *desc);
#else
RETCODE EXPORT SdRegisterSemaphore(MEMBER_NAME FAR *name,
                                   FARPROC updateFunc,
                                   DESCRIPTOR FAR *desc);
#endif

/****************************************************************************
**
**  SdUnRegisterSemaphore
**
**  Description:
**     Unregister a semaphore.  Used when semaphore resource no longer needed.
**     This will free the semaphore for someone else to use.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor receive when registered
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdUnRegisterSemaphore(DESCRIPTOR desc);

/****************************************************************************
**
**  SdReadMember
**
**  Description:
**     Get the current value (byte stream) of a member.  The number of bytes
**     in the data returned will be the full size of the member.  Use the get
**     size routine to find the size of a member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        data:  caller provides the buffer
**     output:
**        data:  the byte stream for the member
**
*****************************************************************************/
RETCODE EXPORT SdReadMember(DESCRIPTOR desc, U8 FAR *data);

/****************************************************************************
**
**  SdWriteMember
**  SdWriteMemberNoCallback
**
**  Description:
**     Change the value of a member.  The number of bytes in the data stream
**     is for the full member.  Use the get size routine to find the size of a
**     member.  Use NULLPTR for the data if only the return code is to be set
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        data:  new value (byte stream) for the member (or NULLPTR if no
**               data)
**        returnCode:  the return code that will be given when this member is
**                     read
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdWriteMember(DESCRIPTOR desc, U8 FAR *data,
                             RETCODE returnCode);
RETCODE EXPORT SdWriteMemberNoCallback(DESCRIPTOR desc, U8 FAR *data,
                                        RETCODE returnCode);

/****************************************************************************
**
**  SdReadPartialMember
**
**  Description:
**     Get only certain bytes of a member.  The "partial" function is useful
**     for very large members like memory arrays and trace buffers where
**     reading the entire member would be impractical.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        offset:  the index (from 0) of where to start
**        numBytes:  the number of bytes to read
**        data:  caller provides the buffer
**     output:
**        data:  the byte stream for that part of the member
**
*****************************************************************************/
RETCODE EXPORT SdReadPartialMember(DESCRIPTOR desc, MEMBER_OFFSET offset,
                                   MEMBER_SIZE numBytes, U8 FAR *data);

/****************************************************************************
**
**  SdWritePartialMember
**  SdWritePartialMemberNoCallback
**
**  Description:
**     Change the value of a section of a member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        offset:  the index (from 0) of where to start
**        numBytes:  the number of bytes to read
**        data:  data to be written (or NULLPTR if only return code to be set)
**        returnCode:  the return code that will be given when this member is
**                     read
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdWritePartialMember(DESCRIPTOR desc, MEMBER_OFFSET offset,
                 MEMBER_SIZE numBytes, U8 FAR *data, RETCODE returnCode);
RETCODE EXPORT SdWritePartialMemberNoCallback(DESCRIPTOR desc,
   MEMBER_OFFSET offset,MEMBER_SIZE numBytes,U8 FAR *data,RETCODE returnCode);

/****************************************************************************
**
**  SdReadBitsOfMember
**
**  Description:
**     Get only certain bits of a member.  This can be used to read only
**     a portion of a member (e.g. individual flags in the CPU flag register).
**     This function can be used to get any bits in a double-word, located
**     anywhere in a member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        offset:  the byte index (from 0) of where to start
**        bitMask:  the bits of interest are set to 1, others are 0
**        bitData:  caller provides the buffer
**     output:
**        bitData:  the value read (only bits in mask are returned, others are
**                  zeroed)
**
*****************************************************************************/
RETCODE EXPORT SdReadBitsOfMember(DESCRIPTOR desc, MEMBER_OFFSET offset,
                                  MEMBER_BITS bitMask, MEMBER_BITS FAR *bitData);

/****************************************************************************
**
**  SdWriteBitsOfMember
**
**  Description:
**     Change the value of only certain bits of a member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        offset:  the byte index (from 0) of where to start
**        numBytes:  the number of bytes to read
**        bitMask:  the bits of interest are set to 1, others are 0
**        bitData:  data to write (only those bits associated with bitData
**                  are set)
**        returnCode:  the return code that will be given when this member is
**                     read
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdWriteBitsOfMember(DESCRIPTOR desc, MEMBER_OFFSET offset,
                                   MEMBER_BITS bitMask, MEMBER_BITS bitData,
                                   RETCODE returnCode);

/****************************************************************************
**
**  SdGetMemberSize
**
**  Description:
**     Get the number of bytes for a member.  This routine is intended to be
**     used when allocating memory.  It should be used instead of hard coding
**     values so that routines can be written more generically.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**     output:
**        numBytes:  the number of bytes for the member associated with the
**                   descriptor
**
*****************************************************************************/
RETCODE EXPORT SdGetMemberSize(DESCRIPTOR desc, MEMBER_SIZE FAR *numBytes);

/****************************************************************************
**
**  SdGetMemberAccess
**
**  Description:
**     Get the read/write access for the member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        residence:  get the access for the host or the firmware
**     output:
**        access:  the access for the member associated with the descriptor
**
*****************************************************************************/
RETCODE EXPORT SdGetMemberAccess(DESCRIPTOR desc, MEMBER_RESIDENCE residence,
                                 MEMBER_ACCESS FAR *access);

/****************************************************************************
**
**  SdGetMemberName
**
**  Description:
**     Convert a descriptor back to a name.  This is useful if a table
**     contains the ASCII names, but descriptors are being passed around.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (has information to get name)
**        memberName:  user supplies buffer (use SD_MAX_NAME as buffer length)
**     output:
**        memberName:  ASCII name string
**
*****************************************************************************/
RETCODE EXPORT SdGetMemberName(DESCRIPTOR desc, MEMBER_NAME FAR *memberName);

/****************************************************************************
**
**  SdGetMemberIndex
**
**  Description:
**     Use a descriptor to get the index of a multiple count member.  The
**     index can then be used to get other members that need an index in
**     their name.  See the "Thunderbird Firmware Specification" read memory
**     example.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (has information to get index)
**        index:  caller supplies storage
**     output:
**        index:  the index of the member (from 0).  NOTE:  if only one
**                member, 0 will be returned.
**
*****************************************************************************/
RETCODE EXPORT SdGetMemberIndex(DESCRIPTOR desc, MEMBER_INDEX FAR *index);

/****************************************************************************
**
**  SdGetMemberReturnCode
**
**  Description:
**     Get the current return code value for a member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (has information to get index)
**        retCode:  caller supplies storage
**     output:
**        retCode:  the current return code stored for the member
**
*****************************************************************************/
RETCODE EXPORT SdGetMemberReturnCode(DESCRIPTOR desc, RETCODE FAR *retCode);

/****************************************************************************
**
**  SdTestAndSetSemaphore
**
**  Description:
**     This function acts as a semaphore between functions.  It will test a
**     semaphore (BIT member) to see if it is not in use.  If not in use, it
**     will get it and return success.  
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (has information about member)
**        setOk:  caller supplies the buffer
**     output:
**        setOk:  true if semaphore is gotten, else false (bit already set by
**                someone else)
**
*****************************************************************************/
RETCODE EXPORT SdTestAndSetSemaphore(DESCRIPTOR desc, BOOLEAN FAR *setOk);

/****************************************************************************
**
**  SdWriteCmdReadResponse
**  SdWriteCmdChkAbortReadResponse
**
**  Description:
**     Write a value to a member and then wait for another member to be
**     changed before returning.  This routine will act as a command that
**     blocks and waits for the response to come back.  After it returns,
**     the caller can use the read member function to get the response data.
**     NOTE:  Writing is to an entire member, not partial, which is how this
**     function is intended to be used.
**
**  Parameters:
**     input:
**        cmdBuff:  buffer containing bytes for the new value of the member
**                  described by cmdDesc
**        returnCode:  the return code that will be given when this member is
**                     read
**        respDesc:  opaque descriptor of member to block on, waiting for
**                   update
**        timeout:  number of seconds to wait blocking for the response to be
**                  updated before returning timeout (set to 0 for the default
**                  time out of 5 seconds... use this normally so that the
**                  default can be changed in one place; use a higher value
**                  if you have a longer operation).
**        abortDesc: member to write when CHECK_ABORT is true
**        abortData: pointer to data to write to abortDesc.
**        abortReturnCode: retcode to write to abortDesc.
**     output:
**        timedOut:  true if the command timed out, else false if not
**
*****************************************************************************/
RETCODE EXPORT SdWriteCmdReadResponse(DESCRIPTOR cmdDesc, VOID FAR *cmdBuff,
                                      RETCODE cmdReturnCode,
                                      DESCRIPTOR respDesc, TIMEOUT timeout,
                                      BOOLEAN FAR *timedOut);
RETCODE EXPORT SdWriteCmdChkAbortReadResponse(DESCRIPTOR cmdDesc,
                                   VOID FAR *cmdBuff,
                                   RETCODE cmdReturnCode,
                                   DESCRIPTOR abortDesc, VOID FAR *abortData,
                                   RETCODE abortReturnCode,
                                   DESCRIPTOR respDesc, TIMEOUT timeout,
                                   BOOLEAN FAR *timedOut);


/****************************************************************************
**
**  SdTerminateServer
**
**  Description:
**     Need to kill the timer to avoid getting timer interrupts after
**     shared data server has been unloaded from the system.
**
**  Parameters:
**     input:
**        none
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdTerminateServer(VOID);

/****************************************************************************
**
**  SdReadMemberNoInputCheck
**
**  Description:
**     Exactly the same as SdReadMember except we don't first make a call
**     to check for any new input on the comm port.  This is done to allow
**     callback routines to get the original value of their member.
**
**  Parameters:
**     input:
**        desc:  opaque descriptor (contains member information)
**        data:  caller provides the buffer
**     output:
**        data:  the byte stream for the member
**
*****************************************************************************/
RETCODE EXPORT SdReadMemberNoInputCheck(DESCRIPTOR desc, U8 FAR *data);

/****************************************************************************
**
**  SdnRegister
**
**  Description:
**     Get a descriptor to gain access to a shared data member.  Optionally,
**     register a callback function to be invoked whenever the value of the
**     member changes.  (Use NULLPTR to indicate no callback requested.)  If
**     the member name is not valid, an error is returned.
**
**  Parameters:
**     input:
**        index: member index
**        updateFunc:  routine to call when the value of this member changes
**                     (or NULLPTR if no update notification requested)
**     output:
**        desc:  opaque descriptor which can be used to get the value, get
**               information about the member, unregister, etc.
**
*****************************************************************************/
#if !defined(WINVER) || (WINVER < 0x030a)    //win3.0
RETCODE EXPORT SdnRegister(MEMBER_INDEX index,
                          CALLBACK updateFunc,
                          DESCRIPTOR FAR *desc);
#else     //win3.1
RETCODE EXPORT SdnRegister(MEMBER_INDEX index,
                          FARPROC updateFunc,
                          DESCRIPTOR FAR *desc);
#endif


/****************************************************************************
**
**  SdnData
**
**  Description:
**     Return pointer to shared data memory for given member
**
**  Parameters:
**     input:
**        index: member index
**     <retval>:
**        pointer to shared data memory
**
*****************************************************************************/
U8 * EXPORT SdnData(MEMBER_INDEX index);

/****************************************************************************
**
**  SdnReadMember
**
**  Description:
**     Get the current value (byte stream) of a member.  The number of bytes
**     in the data returned will be the full size of the member.  Use the get
**     size routine to find the size of a member.
**
**  Parameters:
**     input:
**        index: member index
**        data:  caller provides the buffer
**     output:
**        data:  the byte stream for the member
**
*****************************************************************************/
RETCODE EXPORT SdnReadMember(MEMBER_INDEX index, U8 FAR *data);

/****************************************************************************
**
**  SdnWriteMember
**  SdnWriteMemberNoCallback
**  SdnWriteMemberIfChanged
**  SdnWriteMemberAsync
**
**  Description:
**     Change the value of a member.  The number of bytes in the data stream
**     is for the full member.  Use the get size routine to find the size of a
**     member.  Use NULLPTR for the data if only the return code is to be set
**
**  Parameters:
**     input:
**        index: member index
**        data:  new value (byte stream) for the member (or NULLPTR if no
**               data)
**        returnCode:  the return code that will be given when this member is
**                     read
**        func:  (Async only): Function to call when transmission completed.
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdnWriteMember(MEMBER_INDEX index, U8 FAR *data,
                              RETCODE returnCode);
RETCODE EXPORT SdnWriteMemberNoCallback(MEMBER_INDEX index, U8 FAR *data,
                              RETCODE returnCode);
RETCODE EXPORT SdnWriteMemberIfChanged(MEMBER_INDEX index, U8 FAR *data,
                              RETCODE returnCode);
RETCODE EXPORT SdnWriteMemberAsync(MEMBER_INDEX index, U8 FAR *data,
                              RETCODE returnCode, FARPROC func);

/****************************************************************************
**
**  SdsAsyncWait
**
**  Description:
**     Wait for any asynchronous transmission to complete and return the
**     last transmission error.
**
**  Parameters:
**     none
**
*****************************************************************************/
RETCODE EXPORT SdsAsyncWait(VOID);

/****************************************************************************
**
**  SdnReadPartialMember
**
**  Description:
**     Get only certain bytes of a member.  The "partial" function is useful
**     for very large members like memory arrays and trace buffers where
**     reading the entire member would be impractical.
**
**  Parameters:
**     input:
**        index:   member index
**        offset:  the index (from 0) of where to start
**        numBytes:  the number of bytes to read
**        data:  caller provides the buffer
**     output:
**        data:  the byte stream for that part of the member
**
*****************************************************************************/
RETCODE EXPORT SdnReadPartialMember(MEMBER_INDEX index, MEMBER_OFFSET offset,
                                   MEMBER_SIZE numBytes, U8 FAR *data);

/****************************************************************************
**
**  SdnWritePartialMember
**  SdnWritePartialMemberNoCallback
**  SdnWritePartialMemberAsync
**
**  Description:
**     Change the value of a section of a member.
**
**  Parameters:
**     input:
**        index:   member index
**        offset:  the index (from 0) of where to start
**        numBytes:  the number of bytes to read
**        data:  data to be written (or NULLPTR if only return code to be set)
**        returnCode:  the return code that will be given when this member is
**                     read
**        func:  (Async only): Function to call when transmission completed.
**     output:
**        none
**
*****************************************************************************/
RETCODE EXPORT SdnWritePartialMember(MEMBER_INDEX index, MEMBER_OFFSET offset,
                 MEMBER_SIZE numBytes, U8 FAR *data, RETCODE returnCode);
RETCODE EXPORT SdnWritePartialMemberNoCallback(MEMBER_INDEX index,
   MEMBER_OFFSET offset,MEMBER_SIZE numBytes,U8 FAR *data,RETCODE returnCode);
RETCODE EXPORT SdnWritePartialMemberAsync(MEMBER_INDEX index,
   MEMBER_OFFSET offset,MEMBER_SIZE numBytes,U8 FAR *data,RETCODE returnCode,
   FARPROC func);

/****************************************************************************
**
**  SdnGetMemberSize
**
**  Description:
**     Get the number of bytes for a member.  This routine is intended to be
**     used when allocating memory.  It should be used instead of hard coding
**     values so that routines can be written more generically.
**
**  Parameters:
**     input:
**        index:  member index
**     output:
**        numBytes:  the number of bytes for the member associated with the
**                   descriptor
**
*****************************************************************************/
RETCODE EXPORT SdnGetMemberSize(MEMBER_INDEX index,MEMBER_SIZE FAR *numBytes);

/****************************************************************************
**
**  SdnGetMemberAccess
**
**  Description:
**     Get the read/write access for the member.
**
**  Parameters:
**     input:
**        index:     member index
**        residence:  get the access for the host or the firmware
**     output:
**        access:  the access for the member associated with the descriptor
**
*****************************************************************************/
RETCODE EXPORT SdnGetMemberAccess(MEMBER_INDEX index,
                   MEMBER_RESIDENCE residence, MEMBER_ACCESS FAR *access);

/****************************************************************************
**
**  SdnGetMemberName
**
**  Description:
**     Convert a sd index back to a name.
**
**  Parameters:
**     input:
**        index:     member index
**        memberName:  user supplies buffer (use SD_MAX_NAME as buffer length)
**     output:
**        memberName:  ASCII name string
**
*****************************************************************************/
RETCODE EXPORT SdnGetMemberName(MEMBER_INDEX index,
                               MEMBER_NAME FAR *memberName);

/****************************************************************************
**
**  SdnGetMemberIndex
**
**  Description:
**     Use a descriptor to get the index of a multiple count member.  The
**     index can then be used to get other members that need an index in
**     their name.  See the "Thunderbird Firmware Specification" read memory
**     example.  The return value is based at 0 for the group requested.
**
**  Parameters:
**     input:
**        index:      opaque descriptor (has information to get index)
**        basedIndex: caller supplies storage
**     output:
**        basedIndex: the index of the member (from 0).  NOTE:  if only one
**                    member, 0 will be returned.
**
*****************************************************************************/
RETCODE EXPORT SdnGetMemberIndex(MEMBER_INDEX index,
                                MEMBER_INDEX FAR *basedIndex);

/****************************************************************************
**
**  SdnGetMemberReturnCode
**
**  Description:
**     Get the current return code value for a member.
**
**  Parameters:
**     input:
**        index:    member index
**        retCode:  caller supplies storage
**     output:
**        retCode:  the current return code stored for the member
**
*****************************************************************************/
RETCODE EXPORT SdnGetMemberReturnCode(MEMBER_INDEX index,
                                      RETCODE FAR *retCode);

/****************************************************************************
**
**  SdnWriteCmdReadResponse
**  SdnWriteCmdChkAbortReadResponse
**
**  Description:
**     Write a value to a member and then wait for another member to be
**     changed before returning.  This routine will act as a command that
**     blocks and waits for the response to come back.  After it returns,
**     the caller can use the read member function to get the response data.
**     NOTE:  Writing is to an entire member, not partial, which is how this
**     function is intended to be used.
**
**  Parameters:
**     input:
**        cmdIndex: command member index
**        cmdBuff:  buffer containing bytes for the new value of the member
**                  described by cmdDesc
**        returnCode:  the return code that will be given when this member is
**                     read
**        respIndex: response member index
**        timeout:  number of seconds to wait blocking for the response to be
**                  updated before returning timeout (set to 0 for the default
**                  time out of 5 seconds... use this normally so that the
**                  default can be changed in one place; use a higher value
**                  if you have a longer operation)
**        abortDesc: member to write when CHECK_ABORT is true
**        abortData: pointer to data to write to abortDesc.
**        abortReturnCode: retcode to write to abortDesc.
**     output:
**        timedOut:  true if the command timed out, else false if not
**
*****************************************************************************/
RETCODE EXPORT SdnWriteCmdReadResponse(MEMBER_INDEX cmdIndex,VOID FAR*cmdBuff,
                                      RETCODE cmdReturnCode,
                                      MEMBER_INDEX respIndex, TIMEOUT timeout,
                                      BOOLEAN FAR *timedOut);
RETCODE EXPORT SdnWriteCmdChkAbortReadResponse(MEMBER_INDEX cmdIndex,
                                VOID FAR*cmdBuff,
                                RETCODE cmdReturnCode,
                                MEMBER_INDEX abortIndex, VOID FAR *abortData,
                                RETCODE abortReturnCode,
                                MEMBER_INDEX respIndex, TIMEOUT timeout,
                                BOOLEAN FAR *timedOut);

VOID EXPORT SdsSetSrcStepFlag(VOID);
RETCODE EXPORT SdsReadResetResult(VOID);

#ifdef __cplusplus
}
#endif

/******************************** E O F ***********************************/
