/****************************************************************************
**
**  Name:  basetbl.h
**
**  Description:
**     Prototypes for the functions to manipulate BASE symbols.
**
**  Status:  PRELIMINARY
**
**  $Log:   S:/tbird/arcmmcf/symbol/basetbl.h_v  $
** 
**    Rev 1.1   11 Jun 1996 18:48:44   gene
** 
**    Rev 1.0   07 Sep 1995 11:16:30   gene
** Initial revision.
** 
**    Rev 1.11   22 Oct 1992 18:24:54   brucea
** Initialize maxIndex to MAX_BASE_INDEX_SIZE so that it cannot exceed 4095
**    is the 12 bit restriction on the base index value since it is packed
**    with the 4 bit type field.
** 
**    Rev 1.10   25 Jul 1992 16:31:38   brucea
** Cosmetic cleanup
** 
**    Rev 1.9   10 Jul 1992 18:38:10   brucea
** General cleanup
** Added: baseCount to private BaseIndexTable variables
**        Init var in constructor
**        BaseGetBaseCount()
** 
**    Rev 1.8   20 Apr 1992 09:10:26   brucea
** Added: BasetblResetCacheVars
** 
**    Rev 1.7   11 Dec 1991 17:04:36   brucea
** Added MAX_BASE_INDEX_SIZE
** 
**    Rev 1.6   11 Dec 1991 16:31:12   john
** Added function to get base offsets given a base name
** 
**    Rev 1.5   06 Dec 1991 17:06:42   john
** Moved some inline code into the cpp file.  The code was too involved
** to really be left inline.
** Added prototype for ObliterateTable.  This is used to free dynamically
** allocated memory in the base index table.
** Added prototype for a function to free the dynamically allocated 
** by-address table memory.
** 
**    Rev 1.4   29 Nov 1991 19:45:56   brucea
** changed logicalAddr to offsetAddr
** changed NULL to NULL_SYMBOL
** 
**    Rev 1.3   18 Nov 1991 15:21:24   brucea
** Added #include symutil.h
** 
**    Rev 1.2   01 Nov 1991 10:46:56   brucea
** Added CompareAddrVars
** Changed return parameter of GetRootBaseSymbolOffset to TABLE_OFFSET
** 
**    Rev 1.1   11 Oct 1991 14:43:44   brucea
** Large changes: made BaseSymbol a class, created constructor and operator new
** Added access methods to class variables
** Created class BaseIndexTable, inherited from IndexTable
** Added (outside) functions AddAddrToByAddrTable and BaseSetAddr
** 
**    Rev 1.0   03 Oct 1991 11:42:50   john
** Initial revision.
** 
**  $Header:   S:/tbird/arcmmcf/symbol/basetbl.h_v   1.1   11 Jun 1996 18:48:44   gene  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/

#define _BASETBL_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _INDEXTBL_
#include "indextbl.h"
#endif

#ifndef _SYMADD_
#include "symadd.h"
#endif

#ifndef _SYMUTIL_
#include "symutil.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define MAX_BASE_INDEX_SIZE 4095L

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

//-----------------------------------------------------------------------
//  baseSymbol
//
//  Description:
//     Methods for adding BASE information to the symbol pool.
//-----------------------------------------------------------------------

class BaseSymbol {

protected:
   TABLE_OFFSET baseNameOffset;     // st offset to name
   TABLE_OFFSET nextBaseOffset;     // linked list pointer to next base object
   U32          addressMin;         // smallest addr of syms in by-addr table
   U32          addressMax;         // largest addr of syms in by-addr table
   U32          byAddressTableOffset;  // st offset to by-address object
   BASE_ADDRESS baseAddress;        // address to make sym logical address
                                    // into physical addresses
   U8           baseType;           // CODE, DATA, CODE-DATA, UNKNOWN
   BOOLEAN      addrTableDirty;     // TRUE ==> by-address table not sorted

//NOTE: baseType is not spec'ed as BASE_TYPE because enums are size int and we
//      want the baseType size to be as small as possible.

public:
//-----------------------------------------------------------------------
// Parameterized Constructor; returns error code via <err> reference var
//-----------------------------------------------------------------------
BaseSymbol(LPSTR baseName,
           BASE_INDEX baseIndex,
           BASE_ADDRESS baseAddr,
           BASE_TYPE baseTypeParam,
           RETCODE& err);


//-----------------------------------------------------------------------
// operator new
//
// Purpose: allocates memory in st for a new BaseSymbol
//
// Input parameter:
//    size: amount of space to allocate
//
// Output parameter:
//    err: RETCODE from memory allocation
//
// Return parameter:
//    TABLE_OFFSET: offset into st where memory was allocated
//--------------------------------------------------------------------------
VOID *operator new(size_t size);


//--------------------------------------------------------------------------
// CompareAddrVars
//
// Returns TRUE-FALSE whether address in range or not
//--------------------------------------------------------------------------
BOOLEAN CompareAddrVars(OFFSET_ADDR_TYPE offsetAddr)  {
   if ((offsetAddr < addressMin) ||
       (offsetAddr > addressMax)) return FALSE;
   else return TRUE;
   }


//--------------------------------------------------------------------------
// GetAddressMax
//
// Returns addressMax
//--------------------------------------------------------------------------
U32 GetAddressMax(VOID)  { return addressMax; }


//--------------------------------------------------------------------------
// GetAddressMin
//
// Returns addressMin
//--------------------------------------------------------------------------
U32 GetAddressMin(VOID)  { return addressMin; }


//--------------------------------------------------------------------------
// GetAddrTableDirty
//
// Returns addrTableDirty
//--------------------------------------------------------------------------
BOOLEAN GetAddrTableDirty(VOID)  { return addrTableDirty; }


//--------------------------------------------------------------------------
// GetBaseAddress
//
// Returns baseAddress
//--------------------------------------------------------------------------
BASE_ADDRESS GetBaseAddress(VOID)  { return baseAddress; }


//--------------------------------------------------------------------------
// GetBaseNameOffset
//
// Returns baseNameOffset
//--------------------------------------------------------------------------
U32 GetBaseNameOffset(VOID)  { return baseNameOffset; }


//--------------------------------------------------------------------------
// GetBaseType
//
// Returns baseType
//--------------------------------------------------------------------------
U8 GetBaseType(VOID)  { return baseType; }


//--------------------------------------------------------------------------
// GetByAddressTableOffset
//
// Returns the byAddressTableOffset class variable
//--------------------------------------------------------------------------
U32 GetByAddressTableOffset(VOID) { return byAddressTableOffset; }


//--------------------------------------------------------------------------
// GetNextBaseOffset
//
// Returns nextBaseOffset
//--------------------------------------------------------------------------
TABLE_OFFSET GetNextBaseOffset(VOID)  { return nextBaseOffset; }


//--------------------------------------------------------------------------
// SetAddrTableDirty
//
// Sets the address dirty flag
//--------------------------------------------------------------------------
VOID SetAddrTableDirty(BOOLEAN b) { addrTableDirty = b; }


//--------------------------------------------------------------------------
// SetBaseAddr
//
// Sets the base address of the base object
//--------------------------------------------------------------------------
VOID SetBaseAddr(BASE_ADDRESS baseAddr) { baseAddress = baseAddr; }


//--------------------------------------------------------------------------
// UpdateAddrVars
//
// update the min and max address vars (uses macros)
//--------------------------------------------------------------------------
VOID UpdateAddrVars(U32 minAddr, U32 maxAddr) {

   addressMin = min(addressMin, minAddr);
   addressMax = max(addressMax, maxAddr);
   }
};  // end of BaseSymbol class functions


//--------------------------------------------------------------------------
// BaseIndexTable CLASS
//--------------------------------------------------------------------------
class BaseIndexTable : public IndexTable {

private:
   TABLE_OFFSET rootBaseSymbolOffset;
   U32          baseCount;  // number of bases loaded so far

public:
   // initialize the starting BaseSymbol "pointer" to null since there are no
   // BaseSymbol objects created until the loader creates one or more

//--------------------------------------------------------------------------
// BaseIndexTable - constructor
//--------------------------------------------------------------------------
BaseIndexTable() {
   rootBaseSymbolOffset = NULL_SYMBOL;
   baseCount = 0L;  // initialize number of bases loaded
   maxIndex = MAX_BASE_INDEX_SIZE;  // reduce default value to restrict
                                    // index size to 12 bits for bases
}

//--------------------------------------------------------------------------
// GetRootBaseSymbolOffset
//--------------------------------------------------------------------------
TABLE_OFFSET GetRootBaseSymbolOffset() { return rootBaseSymbolOffset; }


//--------------------------------------------------------------------------
// BaseAddIndex
//
// Description:
//    Calls TableIndex AddIndex, then if successful,
//    sets the root offset to the first base symbol created
//    It only sets the root offset if it is NULL, otherwise just returns
//--------------------------------------------------------------------------
RETCODE BaseAddIndex(TABLE_INDEX index, TABLE_OFFSET itemOffset);


//---------------------------------------------------------------------------
// ObliterateTable
//
// Purpose:
//    Unlocks, then frees the table to Windows memory.
//---------------------------------------------------------------------------
VOID ObliterateTable(VOID);

//--------------------------------------------------------------------------
// BaseGetBaseCount
//
// Description:
//    returns count of number of bases added.
//--------------------------------------------------------------------------
U32 BaseGetBaseCount(VOID) {return baseCount;}

};  // end of BaseIndexTable class definition and functions

                       /*********************************
                        *                               *
                        * NON-CLASS FUNCTION PROTOTYPES *
                        *                               *
                        *********************************/

//--------------------------------------------------------------------------
// Description:
//    Add a symbol offset to the by-address sorted table and update base min
//    and max address values
//--------------------------------------------------------------------------
RETCODE
AddAddrToByAddrTable(BASE_INDEX baseIndex, TABLE_OFFSET symOffset,
                     U32 minAddr, U32 maxAddr);

//--------------------------------------------------------------------------
// Description:
//    Returns the base address of <baseIndex>
//--------------------------------------------------------------------------
RETCODE BaseGetAddr(BASE_INDEX baseIndex, BASE_ADDRESS *baseAddr);


//--------------------------------------------------------------------------
// Description:
//    Modifies the base address of <baseIndex>
//--------------------------------------------------------------------------
RETCODE BaseSetAddr(BASE_INDEX baseIndex, BASE_ADDRESS baseAddr);

//--------------------------------------------------------------------------
// GetBaseOffsetByName
//
// Description:  Given a basename, find the offset in the symbol table of
//               the base.
//
// Inputs:
//       baseName: the name of the base to find.
// Outputs:
//       baseOffset: the offset of the base found.
// Errors:
//       ER_SYMBOL_NOT_FOUND
//--------------------------------------------------------------------------
RETCODE GetBaseOffsetByName(LPSTR baseName, TABLE_OFFSET &baseOffset);

//--------------------------------------------------------------------------
// RemoveAllByAdrTables
//
// Description: Frees the memory that was allocated by the by-address-tables.
//              This allows symbols to be cleanly deleted and all memory 
//              used by bases/base indexes/by address tables to be freed.
//
// Input Parameters:
//    none:
// Output Parameters:
//    none:
// Errors:
//    ER_OUT_OF_MEMORY:
// NOTE: no BaseSymbol is created on initialization.  This means that any
//    loader must first create a base (and its index) before using it.
//--------------------------------------------------------------------------
VOID RemoveAllByAdrTables(VOID);

//--------------------------------------------------------------------------
//  ResetCacheVars
//
//  Description:
//    Re-initialize all cached vars after removal of symbols
//--------------------------------------------------------------------------
VOID BasetblResetCacheVars(VOID);

/******************************** E O F ***********************************/
