/****************************************************************************
** Name: tbirdmem.h
**
** Description:  Routines to fulfill the memory or I/O related operations on
**               the target memory or I/O spaces.
**
** Status: CODED
**
** Copyright (C) 1991 Microtek International.  All rights reserved.
**
** $Log:   S:/tbird/arcmmcf/inc/tbirdmem.h_v  $
** 
**    Rev 1.0   03 Jun 1996 11:25:44   gene
** Initial revision.
** 
**    Rev 1.4   28 Nov 1995 08:49:18   kevin
** added errmsgdisplay()
** 
**    Rev 1.3   22 Sep 1995 11:35:14   kevin
** addedcompare
** 
**    Rev 1.2   18 Sep 1995 16:38:50   gene
** added include abi.h
** 
**    Rev 1.1   14 Sep 1995 14:24:22   kevin
** changed MemChecksum prototype
** 
**    Rev 1.0   14 Sep 1995 14:22:36   kevin
** Initial revision.
** 
**    Rev 1.29   06 Oct 1993 16:44:04   tom
** 8802, 8929: make entry point for processing verify error available
** 
**    Rev 1.28   14 Jul 1993 11:37:00   marilyn
** Added interface for endian conversion routines.
** 
**    Rev 1.27   13 Jul 1993 19:04:14   doug
** Errors consolidated to errtext/mkerrors.h
** 
**    Rev 1.26   25 May 1993 16:37:04   paul
** Added memory copy functions and error codes
** 
**    Rev 1.25   25 May 1993 13:40:20   ernie
** Added CACHE_USE_AND_LOCK option to CACHE_OP used by ReadMem()
** 
**    Rev 1.24   25 May 1993 11:24:02   ernie
** 1. Added memory caching capability to MemRead.... functions.
** 2. Added MemReadSizedNoAlloc() and MemReadQuick() entry points for
**      memory reads for performance reasons.
** 3. Added functions MemCacheInvalidate() and MemCacheInvalidateRange()
**      to support memory caching.
** 
**    Rev 1.23   29 Mar 1993 09:55:24   ernie
** Added ifdef'ed extern C stuff to support c++
** 
**    Rev 1.22   16 Dec 1992 11:35:08   ernie
** Added MemXXXXSized() functions as part of memory access size impl.
** 
**    Rev 1.21   02 Dec 1992 10:24:40   john
** added error message for word size transfers
** 
**    Rev 1.20   20 Oct 1992 11:13:32   john
** added prototype for MemGetVerifyErrorString
** 
**    Rev 1.19   08 Sep 1992 09:44:30   john
** removed EVENT_ definitions
** 
**    Rev 1.18   20 Aug 1992 20:36:42   brucea
** Added: end dummy value to MEM_SRC_ID enum
** 
**    Rev 1.17   20 Aug 1992 11:22:08   doug
** added no access size error
** 
**    Rev 1.16   19 Aug 1992 16:19:02   brucea
** Changed: MemSetSourceRanges to MemSetSourceRange and its input parameters
** Added: typedef MEM_SRC_ID used for defining passed param
** 
**    Rev 1.15   18 Aug 1992 18:25:04   brucea
** Added: MemInitOnce
** 
**    Rev 1.14   17 Aug 1992 17:15:04   brucea
** Added: two events to report mem change to source presenter windows
** Added: new exported function MemSetSourceRanges
** 
**    Rev 1.13   29 Jul 1992 17:32:32   brucea
** Added: MemCallback as exported function
** 
**    Rev 1.12   23 Jul 1992 06:53:10   doug
** added include file for access size definition
** 
**    Rev 1.11   21 Jul 1992 19:42:38   doug
** added routines to get/set access size and write verification
** 
**    Rev 1.10   21 Jul 1992 11:48:58   brucea
** Removed: all functions that were in present.c that were removed
** 
**    Rev 1.9   15 May 1992 14:39:40   brucea
** Removed: several error messages and renumbered remaining
** 
**    Rev 1.8   24 Mar 1992 15:16:54   tom
** Corrected description of MemSearch's not parameter.
** 
**    Rev 1.7   28 Feb 1992 11:24:18   doug
** added error messages
** 
**    Rev 1.6   28 Feb 1992 07:54:28   doug
** when returning a pointer, the caller must give a pointer to the pointer
** 
**    Rev 1.5   28 Feb 1992 07:31:10   doug
** added error messages
** 
**    Rev 1.4   27 Feb 1992 23:38:36   brucea
** Added: #defines for event notification and errors
** 
**    Rev 1.1   25 Feb 1992 10:10:44   brucea
** General comment cleanup
** Changed LPSTR buffer pointer to LPU8
** 
**    Rev 1.0   15 Jan 1992 08:45:04   tom
** Initial revision.
**  
** $Header:   S:/tbird/arcmmcf/inc/tbirdmem.h_v   1.0   03 Jun 1996 11:25:44   gene  $
**
*****************************************************************************/

#define _TBIRDMEM_

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _HOSTERRS_
#include "hosterrs.h"
#endif

#ifndef _SDTEMPL_
#include "sdtempl.h"
#endif

#ifndef _ABI_
#include "abi.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif
		       /**************************
			*                        *
			*    TYPE DEFINITIONS    *
			*                        *
			**************************/

typedef enum {
   MEM_OCTAL,
   MEM_DECIMAL,
   MEM_HEX
} MEM_RADIX_TYPE;

typedef enum {
   CARRY_IGNORE,
   CARRY_PROPOGATED
} SUM_MODE;

typedef enum {
   MEM_U8 = 8,
   MEM_U16 = 16,
   MEM_U32 = 32
} MEM_DISPLAY_SIZE_TYPE;


typedef enum {
   MEM_SRC_ID_0 = 0,
   MEM_SRC_ID_1 = 1,
   MEM_SRC_DUMMY = 0x7FFF
} MEM_SRC_ID;

/* Options for handling shared data read memory cache */
typedef enum {
   CACHE_USE,          /* Return data from cache if present */
   CACHE_BYPASS,       /* Reread from hardware even if already in cache */
   CACHE_USE_AND_LOCK, /* Return data from cache; lock buffer */
} CACHE_OP;

		       /****************************
			*                          *
			*          ERRORS          *
			*                          *
			****************************/

/* errors are now in errtext/mkerrors.h */

			/***********************
			 *                     *
			 *      PROTOTYPES     *
			 *                     *
			 ***********************/

/*--------------------------------------------------------------------------
**          ONE-TIME (SINGLE SHOT) CALL INTERFACE
/*------------------------------------------------------------------------*/

/***************************************************************************
**
** MemGetVerifyErrorString
**
** Description:
**    Interface to report the addr, expected data, and actual data
**    when a memory command returns ER_MEMORY_VERIFY
**
** Input:
**         None
**
** Output:
**         errString: the formatted text string containing the error data.
**                    the caller is expected to allocate this buffer.
**
*****************************************************************************/
RETCODE EXPORT MemGetVerifyErrorString(LPSTR errString);

/*--------------------------------------------------------------------------
**  MemRead
**  MemReadSizedNoAlloc
**  MemReadSized
**  MemReadQuick
**  MemReadAhead
**
**  Purpose:
**     Read a block of memory specified by <startAddr> and <length>
**     and put the contents into <buffer>.
**
**  Input:
**     startAddr:    starting address of memory block to be read;
**                     NOT to be consumed by MemRead.
**     length:       length of the memory block in bytes
**     size:         access size to be used for this access.
**     startOffset:  (MemReadQuick) starting offset of memory block
**     space:        (MemReadQuick) address space for read
**     cacheOp:      (MemReadQuick) Cache treatment option
**
**  Output:
**     buffer: pointer to data.
**         MemRead(), MemReadSized():
**            Memory is TMalloc'ed.  Caller is responsible for TFree'ing
**            buffer when done.
**         MemReadSizedNoAlloc(), MemReadQuick():
**            Caller provides memory for return data.  If buffer is NULL,
**            no data is returned (used to fill up memory cache).
**
**  Note:  The data is returned in ascending byte order, regardless of
**         endianness of the processor.  To interpret the data stream as
**         a size other than byte, the endianness of the processor must be
**         used to convert between the byte stream to word or dword stream.
----------------------------------------------------------------------------*/
RETCODE EXPORT MemRead(DESCRIPTOR  startAddr,
		       U16         length,
		       LPU8        *buffer,
		       CACHE_OP    cacheOp);
		    
RETCODE EXPORT MemReadSized(DESCRIPTOR  startAddr,
			    U16         length,
			    LPU8        *buffer,
			    ACCESS_SIZE size,
			    CACHE_OP    cacheOp);

RETCODE EXPORT MemReadSizedNoAlloc(DESCRIPTOR  startAddr,
			    U16         length,
			    U8          *buffer,
			    ACCESS_SIZE size,
			    CACHE_OP    cacheOp);

RETCODE EXPORT MemReadQuick(U32         startOffset,
			    ADDR_SPACE  space,
			    U16         length,
			    U8          *buffer,
			    ACCESS_SIZE size,
			    CACHE_OP    cacheOp);

/*--------------------------------------------------------------------------
**  MemWrite
**  MemWriteSized
**
**  Purpose:
**     Write a block of memory of size <length> to <startAddr>
**
**  Input:
**     startAddr: starting address of write; CALLER MUST FREE ITS MEMORY
**     length: number of bytes in buffer to write
**     buffer: pointer to data to be written.  Memory is allocated by caller
**             with TMalloc.  MemWrite will TFree this buffer when done.
**     size:   access size to use for this access.
**
**  Output: None
**
**  Note:  MemWrite always uses the global access size and always frees the
**         buffer using TFree.
**
**  Note:  The data is supplied in ascending byte order, regardless of
**         endianness of the processor.  To interpret the data stream as
**         a size other than byte, the endianness of the processor must be
**         used to convert between the byte stream to word or dword stream.
----------------------------------------------------------------------------*/
RETCODE EXPORT MemWrite(DESCRIPTOR  startAddr,
			U16         length,
			LPU8        buffer);

RETCODE EXPORT MemWriteSized(DESCRIPTOR  startAddr,
			     U16         length,
			     LPU8        buffer,
			     ACCESS_SIZE size);


/*--------------------------------------------------------------------------
**  MemFill
**  MemFillSized
**
**  Purpose:
**     fill a block of memory with the specified pattern
**
**  Input:
**     startAddr: starting address of write                   
**     fillLength: amount of target memory to fill
**     pattern: pointer to pattern buffer; caller will create and destroy
**              this buffer.  This allows caller to use local or heap memory.
**     patternLength: length of pattern
**     size:    access size to be used for this access.
**
**  Output: <none>
**
**  Note: the fill pattern will be repeated into the target memory as many
**        times as possible to satisfy the fillLength.  It is possible for
**        a partial pattern to exist at the end of the fill because
**        the fillLength was not an even multiple of patternLength.
**
**  Note:  The data is supplied in ascending byte order, regardless of
**         endianness of the processor.  To interpret the data stream as
**         a size other than byte, the endianness of the processor must be
**         used to convert between the byte stream to word or dword stream.
----------------------------------------------------------------------------*/
RETCODE EXPORT MemFill(DESCRIPTOR  startAddr,
		       U32         fillLength,
		       LPU8        pattern,
		       U16         patternLength);

RETCODE EXPORT MemFillSized(DESCRIPTOR  startAddr,
			    U32         fillLength,
			    LPU8        pattern,
			    U16         patternLength,
			    ACCESS_SIZE size);


/*--------------------------------------------------------------------------
**  MemSearch
**  MemSearchSized
**
**  Purpose:
**     match (or unmatch) the specified pattern within a block of target
**     memory
**
**  Input:
**     firstAddr:    starting address of memory block to be searched
**     secondAddr:   length of the memory block
**     not:          TRUE = invert the match (look for mismatch)
**                   FALSE = don't invert match (look for match)
**     pattern:      long pointer to pattern buffer
**     patLen:       length of the pattern
**     size:         Access size to use for memory operations (MemSearch
**                      uses global accessSize)
**
**  Output:
**     firstAddr: the start address of matched block or the first address
**                which is unmatched to the pattern
**     found:     TRUE : if pattern matched (if not == FALSE) or
**                                unmatched (if not == TRUE)
**                FALSE: opposite conditions
----------------------------------------------------------------------------*/
RETCODE EXPORT MemSearch(DESCRIPTOR  firstAddr,
			 DESCRIPTOR  secondAddr,
			 BOOLEAN     not,
			 LPU8        pattern,
			 U16         patLen,
			 BOOLEAN FAR *found);

RETCODE EXPORT MemSearchSized(DESCRIPTOR  firstAddr,
			      DESCRIPTOR  secondAddr,
			      BOOLEAN     not,
			      LPU8        pattern,
			      U16         patLen,
			      BOOLEAN FAR *found,
			      ACCESS_SIZE size);
		     

/*--------------------------------------------------------------------------
**  MemCopy
**  MemCopySized
**
**  Purpose:
**     copy memory image from one place to another, even to another
**     memory space.
**
**  Input:
**     srcDesc:      starting address of source memory block
**     length:       length of the memory block
**     srcTarget:    TRUE = copy and force using target
**                   FALSE = use current map
**     srcDesc:      starting address of source memory block
**     srcTarget:    TRUE = copy and force using target
**                   FALSE = use current map
**     size:         Access size to use for memory operations (MemCopy
**                      uses global "accessSize")
**
**  Output:       none
----------------------------------------------------------------------------*/
RETCODE EXPORT MemCopy(DESCRIPTOR  srcDesc,
		       U32         length,
		       BOOLEAN     srcTarget,
		       DESCRIPTOR  destDesc,
		       BOOLEAN     destTarget);

RETCODE EXPORT MemCopySized(DESCRIPTOR  srcDesc,
			    U32         length,
			    BOOLEAN     srcTarget,
			    DESCRIPTOR  destDesc,
			    BOOLEAN     destTarget,
			    ACCESS_SIZE size);


/*--------------------------------------------------------------------------
**  MemCompare
**  MemCompareSized
**
**  Purpose:
**     compare the identity of the specified two memory blocks
**
**  Input:
**     src:    starting address of the 1st block
**     dest:   starting address of the 2nd block
**     length: length
**     size:   Access size to use for memory operations.  MemCompare uses
**                global accessSize.
**
**  Output:
**     src:    address in the 1st block which differ from the 2nd
**     dest:   corresponding address in the 2nd block where differed from 1st
**     result: TRUE if the blocks are identical; FALSE otherwise
----------------------------------------------------------------------------*/
RETCODE EXPORT MemCompare(DESCRIPTOR src,
			  DESCRIPTOR dest,
			  U32 length,
			  BOOLEAN *result,
			  U32 *retAddr1,
			  U32 *retAddr2,
			  U16 *retData1,
			  U16 *retData2);

RETCODE EXPORT MemCompareSized(DESCRIPTOR src,
			       DESCRIPTOR dest,
			       U32 length,
			       BOOLEAN *result,
			       ACCESS_SIZE size);


/*--------------------------------------------------------------------------
**  MemMove
**  MemMoveSized
**
**  Purpose:
**     move a block of memory to another
**
**  Input:
**     src:    starting address of the 1st block
**     dest:   starting address of the 2nd block
**     length: length
**     size:   Access size to use for memory operations
**
**  Output:  <none>
----------------------------------------------------------------------------*/
RETCODE EXPORT MemMove(DESCRIPTOR src,
		       DESCRIPTOR dest,
		       U32 length);

RETCODE EXPORT MemMoveSized(DESCRIPTOR src,
			    DESCRIPTOR dest,
			    U32 length,
			    ACCESS_SIZE size);


/*--------------------------------------------------------------------------
**  MemChecksum
**  MemChecksumSized
**
**  Purpose:
**     calculate the checksum of the specified block
**
**  Input:
**     src:      starting address
**     length:   length
**     sumSize:  size of the sum
**     size:     Access size to use for memory operations
**
**  Output:
**     sum: the result checksum
----------------------------------------------------------------------------*/
RETCODE EXPORT MemChecksum(DESCRIPTOR            src,
			   U32                   length,
			   U16                   sumSize,
			   U32                   *sum);

RETCODE EXPORT MemChecksumSized(DESCRIPTOR            src,
				U32                   length,
				MEM_DISPLAY_SIZE_TYPE itemSize,
				U16                   sumSize,
				SUM_MODE              mode,
				U32                   *sum,
				ACCESS_SIZE           size);


/*--------------------------------------------------------------------------
**  MemTest
**  MemTestSized
**
**  Purpose:
**     test the function of the specified block memory
**
**  Input:
**     src:    starting address
**     length: length
**     size:   Access size to use for memory operations
**
**  Output:
**     result: TRUE if the blocks test o.k.
----------------------------------------------------------------------------*/
RETCODE EXPORT MemTest(DESCRIPTOR src,U32 length,ADDR *retAddress);

RETCODE EXPORT MemTestSized(DESCRIPTOR src,
			    U32        length,
			    BOOLEAN    *result,
			    ACCESS_SIZE size);

/*--------------------------------------------------------------------------
**          CONNECTION-BASED CALL INTERFACES
/*------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
** InitServer
**
**      Purpose:
**    Initialize all the PRIVATE variables used in this module; only called
**    once when DLL loaded.  Also registers to receive event notification
----------------------------------------------------------------------------*/
RETCODE EXPORT InitServer(VOID);


/*--------------------------------------------------------------------------
**  MemOpenSession
**
**  Purpose:
**     opens a memory access session for client to use.  This module
**     maintains static state information for this session.
**
**  Input: <none>
**
**  Output:
**     memSessionId: opaque descriptor "key" to get and set parameters
**                   and call operations
----------------------------------------------------------------------------*/
RETCODE EXPORT MemOpenSession(DESCRIPTOR FAR *memSessionId);


/*--------------------------------------------------------------------------
**  MemCloseSession
**
**  Purpose:
**     removes any static session state and disables <memSessionId>
**
**  Input:         
**     memSessionId: opaque descriptor "key" to access or modify state and
**                   to call specific memory functions
**
**  Output: none
----------------------------------------------------------------------------*/
RETCODE EXPORT MemCloseSession(DESCRIPTOR memSessionId);


/*--------------------------------------------------------------------------
**          GLOBAL SETTINGS
/*------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
** MemSetAccessSize
**
**      Purpose:
**    Set the read/write access size
----------------------------------------------------------------------------*/
RETCODE EXPORT MemSetAccessSize(ACCESS_SIZE size);

/*------------------------------------------------------------------------
** MemGetAccessSize
**
**      Purpose:
**    Get the current read/write access size
----------------------------------------------------------------------------*/
RETCODE EXPORT MemGetAccessSize(ACCESS_SIZE FAR *size);

/*------------------------------------------------------------------------
** MemSetVerifyWrites
**
**      Purpose:
**    Set whether to verify memory writes
----------------------------------------------------------------------------*/
RETCODE EXPORT MemSetVerifyWrites(BOOLEAN verify);

/*------------------------------------------------------------------------
** MemGetVerifyWrites
**
**      Purpose:
**    Get current state of whether to verify memory writes
----------------------------------------------------------------------------*/
RETCODE EXPORT MemGetVerifyWrites(BOOLEAN FAR *verify);

/*------------------------------------------------------------------------
** MemCallback
**
** Purpose:
**    Is called when event that the memory server has registered on occurs
**
** Input parameter:
**    eventNum: indicates which event occurred that was previously
**       registered on
----------------------------------------------------------------------------*/
RETCODE EXPORT MemCallback(U32 eventNum);

/*--------------------------------------------------------------------------
** MemSetSourceRanges
**
** Purpose: set up two source range addresses (for two source windows)
**          used to determine if memory changed falls within any of these
**          ranges.  If it does, an event is broadcast to tell src presenter
**          to update its window.
**
** Input parameters:
**    addrRange" Addr descriptor of a range covering the code in each
**       source window.  The addr is destroyed by this function.
**       If there is no valid address to be compared (window not open),
**       simply pass descriptor from AdrCreateAddress.  It will compare
**       FALSE and not generate errors.
----------------------------------------------------------------------------*/
RETCODE EXPORT
MemSetSourceRange(DESCRIPTOR addrRangeDesc, MEM_SRC_ID srcID);


/*------------------------------------------------------------------------
** MemInitOnce
**
**      Purpose:
**    This is kludge code which initializes address descriptors; it should 
**    be in MemInitServer but Windows cannot load DLLs in the right order.
**    This should only be called by Actor once.
----------------------------------------------------------------------------*/
RETCODE EXPORT MemInitOnce(VOID);

/*--------------------------------------------------------------------------
** MemCacheInvalidate
** MemCacheInvalidateRange
**
** Flush memory cache
**
** inputs:
**    addr:    Range of memory to invalidate
** 
----------------------------------------------------------------------------*/
RETCODE EXPORT MemCacheInvalidate(VOID);
RETCODE EXPORT MemCacheInvalidateRange(DESCRIPTOR addr);

/*--------------------------------------------------------------------
**          ENDIAN CONVERSION ROUTINES
--------------------------------------------------------------------*/

/*--------------------------------------------------------------------------
** MemSwapBytesInWord
**
**  Swapping only occurs when the processor is BIG_ENDIAN.
**
** inputs:
**    theWord: word to swap the bytes in    
** 
----------------------------------------------------------------------------*/
U16 EXPORT MemSwapBytesInWord(U16 theWord);

/*--------------------------------------------------------------------------
** MemSwapBytesInLong
**
**  Swapping only occurs when the processor is BIG_ENDIAN.
**
** inputs:
**    theLong: long to swap the bytes in    
** 
----------------------------------------------------------------------------*/
U32 EXPORT MemSwapBytesInLong(U32 theLong);

/*--------------------------------------------------------------------------
** MemSwapWordsInLong
**
**  Swapping only occurs when the processor is BIG_ENDIAN.
**
** inputs:
**    theLong: long to swap the bytes in    
** 
----------------------------------------------------------------------------*/
U32 EXPORT MemSwapWordsInLong(U32 theLong);

/*--------------------------------------------------------------------------
** MemChooseByteInWord
**
** inputs:
**    theWord: word to choose the byte from 
**    position: byte position, 0 or 1
** 
----------------------------------------------------------------------------*/
U8  EXPORT MemChooseByteInWord(U16 theWord, U32 position);

/*--------------------------------------------------------------------------
** MemChooseByteInLong
**
** inputs:
**    theLong: long to choose the byte from 
**    position: byte position, 0,1,2 or 3
** 
----------------------------------------------------------------------------*/
U8  EXPORT MemChooseByteInLong(U32 theLong, U32 position);

/*--------------------------------------------------------------------------
** MemChooseWordInLong
**
** inputs:
**    theLong: long to choose the word from 
**    position: byte position, 0,1,2 or 3
** 
----------------------------------------------------------------------------*/
U16 EXPORT MemChooseWordInLong(U32 theLong, U32 position);

/***************************************************************************
**
**  MemProcessVerifyError
**
**  Description:
**    Send additional memory verification error information to the
**    error text server if the
**
**  Parameters:
**    None.
**
*****************************************************************************/
RETCODE EXPORT MemProcessVerifyError(VOID);

VOID MemDisplayError(HWND hwnd, RETCODE errCode);

#ifdef __cplusplus
}
#endif
/******************************** E O F ***********************************/
