/****************************************************************************
**
**  Name:  dbcentrl.h
**
**  Description:
**     Provide access to DB Central Server.
**
**  Status:  PRELIMINARY
**
**  $Log:   S:/tbird/mt2_amd/inc/dbcentrl.h_v  $
** 
**    Rev 1.0   20 Mar 1998 10:06:54   Eric
** Initial revision.
** 
**    Rev 1.0   16 Dec 1996 13:53:54   Judy
** Initial revision.
** 
**    Rev 1.0   27 Aug 1996 15:21:18   Judy
** Initial revision.
   
      Rev 1.0   23 Aug 1996 17:44:26   Judy
   Initial revision.
**  
**     Rev 1.3   21 Aug 1991 11:22:30   tom
**  Updated types, include file format.
**  
**     Rev 1.2   11 Jul 1991 12:48:18   jim
**  Fixed the comment prefix for Rev.
** 
**    Rev 1.1   10 Jul 1991 17:24:18   gary
**  Changed the includes around
** 
**    Rev 1.0   14 Jun 1991 13:35:24   jim
**  Initial revision.
**
**  $Header:   S:/tbird/mt2_amd/inc/dbcentrl.h_v   1.0   20 Mar 1998 10:06:54   Eric  $
**
*****************************************************************************/

#define _DBCENTRL_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

typedef U32 DESIGNATOR;

/* action to take when error encountered */
typedef enum {
   ABORT_ON_ERROR,
   CONTINUE_ON_ERROR
} ERROR_ACTION;

/* action to take when ticket agent reaches established maximum */
typedef enum {
   ERROR_ON_MAX,
   ROLLOVER_ON_MAX
} TICKET_MAX_ACTION;

/* how was the component loaded? */
typedef enum {
   BY_INSTRUMENT_DESCRIPTOR,
   BY_PROJECT_DESCRIPTOR,
   BY_SESSION
} COMPONENT_LOAD_AGENT;

/* information associated with a loaded component */
typedef struct {
   DESIGNATOR designator;
   DESCRIPTOR descriptor;
   HANDLE handle;
   COMPONENT_LOAD_AGENT loadedBy;
} COMPONENT_INFO;

#define MODULE_NAME_SIZE 128
/* information on a module */
typedef struct MODULE_ID_STRUCT {
   U8 moduleName[MODULE_NAME_SIZE];
   U16 moduleVersion;
   U16 moduleRevision;
   U8 moduleRelease;
   U16 compatabilityVersion;
   U16 compatabilityRevision;
   U8 compatabilityRelease;
} MODULE_ID;

                        /****************************
                         *                          *
                         *        PROTOTYPES        *
                         *                          *
                         ****************************/

/****************************************************************************
**
**  DbcCliDoStringCommand
**
**  Description:
**     Lex command string (statement or statement block) and execute valid
**     component operations.  If executing a statement within a block of
**     statements when a failure occurs, either ABORT_ON_ERROR or continue
**     execution at the next statement as directed by the abortOrContinue
**     parameter.
**
**  Parameters:
**     input:
**        returnLocation:  CLI results returned here.
**        stringCommand:  CLI string command(s).
**        abortOrContinue:  abort or continue on error.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcCliDoStringCommand (HWND returnLocation,
   STRING * stringCommand, ERROR_ACTION abortOrContinue);

/****************************************************************************
**
**  DbcComponentAddInfo
**
**  Description:
**     Adds information for a software component to the loaded component
**        database.
**
**  Parameters:
**     input:
**        componentDesignator:  a component designator.
**        componentDescriptor:  a component descriptor.
**        componentHandle:  a component handle.
**        loadAgent:  who loaded the component.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcComponentAddInfo (DESIGNATOR componentDesignator,
   DESCRIPTOR componentDescriptor, HANDLE componentHandle,
   COMPONENT_LOAD_AGENT loadAgent);

/****************************************************************************
**
**  DbcComponentDeleteInfo
**
**  Description:
**     Deletes sw component information from the loaded component data base.
**
**  Parameters:
**     input:
**        componentDesignator:  a component designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcComponentDeleteInfo (DESIGNATOR componentDesignator);

/****************************************************************************
**
**  DbcComponentGetInfo
**
**  Description:
**     Returns component information for the designated component.
**
**  Parameters:
**     input:
**        componentDesignator:  a component designator.
**     output:
**        componentInfo:  component info.  Caller must pre-allocate space.
**
*****************************************************************************/
RETCODE EXPORT DbcComponentGetInfo (DESIGNATOR componentDesignator,
   COMPONENT_INFO * componentInfo);

/****************************************************************************
**
**  DbcComponentGetSet
**
**  Description:
**     Returns a set of component descriptors.
**
**  Parameters:
**     input:
**        None.
**     output:
**        componentSet:  an array of component descriptors.  Must be cast as
**           type "DESCRIPTOR *" prior to use.  Note:  structure
**           must be deallocated by DbcFreeComponentSet.
**        componentCount:  number of component descriptors in componentSet.
**
*****************************************************************************/
RETCODE EXPORT DbcComponentGetSet (FAR_HANDLE * componentSet, 
   U32 * componentCount);

/****************************************************************************
**
**  DbcErrorMapCode
**
**  Description:
**     Given an error code originating with this component, return a
**     designator of the correct string table and index to obtain the
**     corresponding error message text string.
**
**  Parameters:
**     input:
**        errorCode:  error ID of failed operation.
**     output:
**        stringTableId:  error message index within component's string table.
**        stringTableIndex:  allows access to component's string table.
**
*****************************************************************************/
RETCODE EXPORT DbcErrorMapCode (RETCODE errorCode, U32 * stringTableId,
   U32 * stringTableIndex);

/****************************************************************************
**
**  DbcFlagDelete
**
**  Description:
**     Deletes a flag variable.
**
**  Parameters:
**     input:
**        flagDescriptor:  flag descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcFlagDelete (DESCRIPTOR flagDescriptor);

/****************************************************************************
**
**  DbcFlagGet
**
**  Description:
**     Given a flag variable descriptor, returns its value (true or false).
**
**  Parameters:
**     input:
**        flagDescriptor:  flag descriptor.
**     output:
**        flagValue:  Flag's current value.
**
*****************************************************************************/
RETCODE EXPORT DbcFlagGet (DESCRIPTOR flagDescriptor, BOOLEAN * flagValue);

/****************************************************************************
**
**  DbcFlagGetDescriptor
**
**  Description:
**     Given the name of a flag variable, returns its descriptor.
**
**  Parameters:
**     input:
**        flagName:  flag name string.
**     output:
**        flagDescriptor:  flag descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcFlagGetDescriptor (STRING * flagName,
   DESCRIPTOR * flagDescriptor);

/****************************************************************************
**
**  DbcFlagNew
**
**  Description:
**     Given a name, creates a globally accessible flag variable.  Note that
**     flag variables are booleans and totally separate from local and
**     global environment based, string valued, variables.
**
**  Parameters:
**     input:
**        flagName:  flag name string.
**     output:
**        flagDescriptor:  flag descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcFlagNew (STRING * flagName, DESCRIPTOR * flagDescriptor);

/****************************************************************************
**
**  DbcFlagSet
**
**  Description:
**     Given a flag variable's descriptor, sets it value to true or false.
**
**  Parameters:
**     input:
**        flagDescriptor:  flag descriptor.
**        flagValue:  Flag's new value.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcFlagSet (DESCRIPTOR flagDescriptor, BOOLEAN flagValue);

/****************************************************************************
**
**  DbcFreeComponentSet
**
**  Description:
**     Deallocates a variable-length structure allocated and returned to the
**     user by DB Central.
**
**  Parameters:
**     input:
**        componentSet:  an array of type DESCRIPTOR component descriptors.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcFreeComponentSet (FAR_HANDLE * componentSet);

/****************************************************************************
**
**  DbcFreeInstrumentSet
**
**  Description:
**     Deallocates a variable-length structure allocated and returned to the
**     user by DB Central.
**
**  Parameters:
**     input:
**        instrumentSet:  an array of type DESCRIPTOR instrument descriptors.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcFreeInstrumentSet (FAR_HANDLE * instrumentSet);

/****************************************************************************
**
**  DbcFreeModuleList
**
**  Description:
**     Deallocates a variable-length structure allocated and returned to the
**     user by DB Central.
**
**  Parameters:
**     input:
**        moduleList:  an array of type MODULE_ID.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcFreeModuleList (FAR_HANDLE * moduleList);

/****************************************************************************
**
**  DbcHWGetAdvertisementID
**
**  Description:
**     Given a hardware descriptor, returns the associated
**     provides/requires advertisement.
**
**  Parameters:
**     input:
**        hardwareDescriptor:  hardware descriptor.
**     output:
**        hwAdvertisementID:  hardware advertisement ID for
**           provides/requires checking.
**
*****************************************************************************/
RETCODE EXPORT DbcHWGetAdvertisementID (DESCRIPTOR hardwareDescriptor,
   U32 * hwAdvertisementID);

/****************************************************************************
**
**  DbcInstdefAddComponents
**
**  Description:
**     Adds component descriptors to instrument definition.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**        componentSet:  an array of type DESCRIPTOR component descriptors.  
**        componentCount:  number of elements in componentSet.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefAddComponents (DESCRIPTOR instrumentDescriptor,
   FAR_HANDLE * componentSet, U32 componentCount);

/****************************************************************************
**
**  DbcInstdefDelete
**
**  Description:
**     Removes an instrument definition from the instrument data base.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefDelete (DESCRIPTOR instrumentDescriptor);

/****************************************************************************
**
**  DbcInstdefGetComponentSet
**
**  Description:
**     Returns a set of component descriptors for components comprising the
**     designated instrument.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        componentSet:  a set of component descriptors.  Must be cast as
**           type "DESCRIPTOR *" prior to use.  Note:  structure must be
**           deallocated by DbcFreeComponentSet.
**        componentCount:  number of elements in componentSet.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefGetComponentSet (DESCRIPTOR instrumentDescriptor,
   FAR_HANDLE * componentSet, U32 * componentCount);

/****************************************************************************
**
**  DbcInstdefGetEnv
**
**  Description:
**     Gets the local environment associated with an instrument.  The
**     environment is used in provides/requires macro expansion.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        localEnv:  local environment descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefGetEnv (DESCRIPTOR instrumentDescriptor,
   DESCRIPTOR * localEnv);

/****************************************************************************
**
**  DbcInstdefGetInstrumentSet
**
**  Description:
**     Returns set of instrument descriptors for all defined instruments.
**
**  Parameters:
**     input:
**        None.
**     output:
**        instrumentSet:  a set of instrument descriptors.  Must be cast
**           as type "DESCRIPTOR *" prior to use.  Note:  structure must
**           be deallocated by DbcFreeInstrumentSet().
**        instrumentCount:  number of elements in instrumentSet.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefGetInstrumentSet (FAR_HANDLE * instrumentSet,
   U32 * instrumentCount);

/****************************************************************************
**
**  DbcInstdefGetName
**
**  Description:
**     Given an instrument descriptor, returns the associated instrument name.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        instrumentName:  instrument name string.  Caller pre-allocates space.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefGetName (DESCRIPTOR instrumentDescriptor,
   STRING * instrumentName);

/****************************************************************************
**
**  DbcInstdefNew
**
**  Description:
**     Creates a new instrument definition.
**
**  Parameters:
**     input:
**        instrumentName:  instrument name string.
**        localEnv:  local environment descriptor.
**        componentSet:  an array of type DESCRIPTOR component descriptors.
**        componentCount:  number of elements in componentSet.
**     output:
**        instrumentDescriptor:  designates a user-selected instrument.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefNew (STRING * instrumentName, DESCRIPTOR localEnv,
   FAR_HANDLE * componentSet, U32  componentCount,
   DESCRIPTOR * instrumentDescriptor);

/****************************************************************************
**
**  DbcInstdefRemoveComponents
**
**  Description:
**     Removes the indicated component descriptors from the instrument
**        definition.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**        componentSet:  an array of type DESCRIPTOR component descriptors.
**        componentCount:  number of elements in componentCount;
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstdefRemoveComponents (DESCRIPTOR instrumentDescriptor,
   FAR_HANDLE * componentSet, U32 componentCount);

/****************************************************************************
**
**  DbcInstloadLoadInstrument
**
**  Description:
**     Load loadmodules associated with designated instrument.  DB Central
**     does bookkeeping for loadmodules.  Due to general lack of one-to-one
**     mapping from loadmodules to software components, components must
**     register themselves.  {Note:  loadmodules loaded by other components
**     are managed by the loading component}.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstloadLoadInstrument (DESCRIPTOR instrumentDescriptor);

/****************************************************************************
**
**  DbcInstloadLoadProject
**
**  Description:
**     Delegate operation to Project Management Server.
**
**  Parameters:
**     input:
**        projectDescriptor:  designates a user project.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstloadLoadProject (DESCRIPTOR projectDescriptor);

/****************************************************************************
**
**  DbcInstloadUnloadInstrument
**
**  Description:
**     Unload and unregister load modules associated with designated
**     instrument.
**
**  Parameters:
**     input:
**        instrumentDescriptor:  designates a user-selected instrument.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstloadUnloadInstrument (DESCRIPTOR instrumentDescriptor);

/****************************************************************************
**
**  DbcInstloadUnloadProject
**
**  Description:
**     Delegate operation to Project Management Server.
**
**  Parameters:
**     input:
**        projectDescriptor:  designates a user project.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcInstloadUnloadProject (DESCRIPTOR projectDescriptor);

/****************************************************************************
**
**  DbcModuleAddInfo
**
**  Description:
**     Adds load module information to the loaded module data base.
**
**  Parameters:
**     input:
**        moduleID:  contains module information.
**        pathname:  module pathname.
**     output:
**        moduleIDdescriptor:  a moduleID descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcModuleAddInfo (MODULE_ID moduleID, STRING * pathname,
   DESCRIPTOR * moduleIDdescriptor);

/****************************************************************************
**
**  DbcModuleDeleteInfo
**
**  Description:
**     Removes load module information from the loaded module data base.
**
**  Parameters:
**     input:
**        moduleIDdescriptor:  a moduleID descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcModuleDeleteInfo (DESCRIPTOR moduleIDdescriptor);

/****************************************************************************
**
**  DbcModuleGetInfo
**
**  Description:
**     Returns a set of module information for all loaded modules.
**
**  Parameters:
**     input:
**        None.
**     output:
**        modulesInfo:  module information list.  Must be cast as type
**           "MODULE_ID *" prior to use.  Note:  structure must
**           be deallocated by DbcFreeModuleList.
**        moduleCount:  number of elements in modulesInfo.
**
*****************************************************************************/
RETCODE EXPORT DbcModuleGetInfo (FAR_HANDLE * modulesInfo,
   U32 * moduleCount);

/****************************************************************************
**
**  DbcRefmapBindDesignatorDescriptor
**
**  Description:
**     Creates association for referent map.
**
**  Parameters:
**     input:
**        designator:  a designator.
**        descriptor:  a descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapBindDesignatorDescriptor (DESIGNATOR designator,
   DESCRIPTOR descriptor);

/****************************************************************************
**
**  DbcRefmapBindDesignatorHandle
**
**  Description:
**     Creates association for referent map.
**
**  Parameters:
**     input:
**        designator:  a designator.
**        handle:  a Windows-issued descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapBindDesignatorHandle (DESIGNATOR designator,
   HANDLE handle);

/****************************************************************************
**
**  DbcRefmapBindHandleDescriptor
**
**  Description:
**     Creates association for referent map.
**
**  Parameters:
**     input:
**        handle:  a Windows-issued descriptor.
**        descriptor:  a descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapBindHandleDescriptor (HANDLE handle,
   DESCRIPTOR descriptor);

/****************************************************************************
**
**  DbcRefmapDescriptor2designator
**
**  Description:
**     Returns the designator associated with the descriptor.
**
**  Parameters:
**     input:
**        descriptor:  a descriptor.
**     output:
**        designator:  a designator.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapDescriptor2designator (DESCRIPTOR descriptor,
   DESIGNATOR * designator);

/****************************************************************************
**
**  DbcRefmapDescriptor2handle
**
**  Description:
**     Returns the handle associated with the descriptor.
**
**  Parameters:
**     input:
**        descriptor:  a descriptor.
**     output:
**        handle:  a Windows-issued descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapDescriptor2handle (DESCRIPTOR descriptor,
   HANDLE * handle);

/****************************************************************************
**
**  DbcRefmapDesignator2descriptor
**
**  Description:
**     Returns the descriptor associated with the designator.
**
**  Parameters:
**     input:
**        designator:  a designator.
**     output:
**        descriptor:  a descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapDesignator2descriptor (DESIGNATOR designator,
   DESCRIPTOR * descriptor);

/****************************************************************************
**
**  DbcRefmapDesignator2handle
**
**  Description:
**     Returns the handle associated with the designator.
**
**  Parameters:
**     input:
**        designator:  a designator.
**     output:
**        handle:  a Windows-issued descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapDesignator2handle (DESIGNATOR designator,
   HANDLE * handle);

/****************************************************************************
**
**  DbcRefmapHandle2descriptor
**
**  Description:
**     Returns the descriptor associated with the handle in the referent map.
**
**  Parameters:
**     input:
**        handle:  a Windows-issued descriptor.
**     output:
**        descriptor:  a descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapHandle2descriptor (HANDLE handle,
   DESCRIPTOR * descriptor);

/****************************************************************************
**
**  DbcRefmapHandle2designator
**
**  Description:
**     Returns the designator associated with the handle in the referent map.
**
**  Parameters:
**     input:
**        handle:  a Windows-issued descriptor.
**     output:
**        designator:  a designator.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapHandle2designator (HANDLE handle,
   DESIGNATOR * designator);

/****************************************************************************
**
**  DbcRefmapUnbindDescriptor
**
**  Description:
**     Removes association from the referent map.
**
**  Parameters:
**     input:
**        descriptor:  a descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapUnbindDescriptor (DESCRIPTOR descriptor);

/****************************************************************************
**
**  DbcRefmapUnbindDesignator
**
**  Description:
**     Removes association from the referent map.
**
**  Parameters:
**     input:
**        designator:  a designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapUnbindDesignator (DESIGNATOR designator);

/****************************************************************************
**
**  DbcRefmapUnbindHandle
**
**  Description:
**     Removes association from the referent map.
**
**  Parameters:
**     input:
**        handle:  a Windows-issued descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcRefmapUnbindHandle (HANDLE handle);

/****************************************************************************
**
**  DbcSetupRestore
**
**  Description:
**     Restore previously saved setup (state) for this component.
**
**  Parameters:
**     input:
**        fileDescriptor:  C file descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcSetupRestore (FILE * fileDescriptor);

/****************************************************************************
**
**  DbcSetupSave
**
**  Description:
**     Save component state as a setup for possible later restoration.
**     Saved state consists of consistency marks header, global environment,
**     and instrument definitions.  {design issue:  save other local
**     environments?}
**
**  Parameters:
**     input:
**        fileDescriptor:  C file descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcSetupSave (FILE * fileDescriptor);

/****************************************************************************
**
**  DbcTicketDestroyAgent
**
**  Description:
**     Deletes the designated ticket agent.
**
**  Parameters:
**     input:
**        ticketAgentDescriptor:  ticket agent descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcTicketDestroyAgent (DESCRIPTOR ticketAgentDescriptor);

/****************************************************************************
**
**  DbcTicketNew
**
**  Description:
**     Causes the indicated ticket agent to produce a new ticket.
**
**  Parameters:
**     input:
**        ticketAgentDescriptor:  ticket agent descriptor.
**     output:
**        ticketValue:  ticket value returned.
**
*****************************************************************************/
RETCODE EXPORT DbcTicketNew (DESCRIPTOR ticketAgentDescriptor,
   U32 * ticketValue);

/****************************************************************************
**
**  DbcTicketNewAgent
**
**  Description:
**     Creates a new ticket agent (instance).
**
**  Parameters:
**     input:
**        maxTicketValue:  the largest ticket value to hand out.
**        maxAction:  what to do after maxTicketValue handed out.
**     output:
**        ticketAgentDescriptor:  ticket agent descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcTicketNewAgent (U32 maxTicketValue,
   TICKET_MAX_ACTION maxAction, DESCRIPTOR * ticketAgentDescriptor);

/****************************************************************************
**
**  DbcValueChangeRegister
**
**  Description:
**     Causes DB Central to issue a message to the designated component with
**     the variables' new value upon value changes.  If a local environment
**     is specified, the variable is searched for in the local environment
**     (only), otherwise the global environment is searched.  If the variable
**     is not found and must be created, a warning is returned.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**        whoToNotify:  notification window used when a specific variable's
**           value changes.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcValueChangeRegister (DESCRIPTOR env, STRING * var,
   HWND whoToNotify);

/****************************************************************************
**
**  DbcValueChangeUnregister
**
**  Description:
**     Causes DB Central to forget notification for this variable (for the
**     designated component only).
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**        whoToNotify:  notification window used when a specific variable's
**           value changes.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcValueChangeUnregister (DESCRIPTOR env, STRING * var,
   HWND whoToNotify);

/****************************************************************************
**
**  DbcVarCopyIntoEnv
**
**  Description:
**     Copies variable and their values from one environment to another.
**
**  Parameters:
**     input:
**        srcEnv:  source environment descriptor.
**        destEnv:  destination environment descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarCopyIntoEnv (DESCRIPTOR srcEnv, DESCRIPTOR destEnv);

/****************************************************************************
**
**  DbcVarCreateAndSetGlobalVal
**
**  Description:
**     Creates a new value in the shared, global environment and gives it a
**        value.
**
**  Parameters:
**     input:
**        var:  variable name.
**        val:  value designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarCreateAndSetGlobalVal (STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarCreateAndSetLocalVal
**
**  Description:
**     Creates a new value in a local environment and gives it a value.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**        val:  value designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarCreateAndSetLocalVal (DESCRIPTOR env, STRING * var,
   STRING * val);

/****************************************************************************
**
**  DbcVarCreateLocalEnv
**
**  Description:
**     Creates a new local environment.
**
**  Parameters:
**     input:
**        None.
**     output:
**        localEnvDescriptor:  local environment descriptor.
**
*****************************************************************************/
RETCODE EXPORT DbcVarCreateLocalEnv (DESCRIPTOR * localEnvDescriptor);

/****************************************************************************
**
**  DbcVarDestroyGlobalVal
**
**  Description:
**     Makes the value of the global variable equal to the empty string.
**
**  Parameters:
**     input:
**        var:  variable name.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarDestroyGlobalVal (STRING * Var);

/****************************************************************************
**
**  DbcVarDestroyLocalEnv
**
**  Description:
**     Deletes a local environment.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarDestroyLocalEnv (DESCRIPTOR env);

/****************************************************************************
**
**  DbcVarDestroyLocalVal
**
**  Description:
**     Makes the value of a local variable equal to the empty string.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarDestroyLocalVal (DESCRIPTOR env, STRING * Var);

/****************************************************************************
**
**  DbcVarDestroyLocalVariables
**
**  Description:
**     Removes all local variables (empties the local environment).
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarDestroyLocalVariables (DESCRIPTOR env);

/****************************************************************************
**
**  DbcVarForgetLocalValues
**
**  Description:
**     Sets the values of all variables in the designated local environment to
**     the empty string.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarForgetLocalValues (DESCRIPTOR env);

/****************************************************************************
**
**  DbcVarGetGlobalVal
**
**  Description:
**     Gets the value of the indicated variable from the global environment.
**
**  Parameters:
**     input:
**        var:  variable name.
**     output:
**        val:  value designator.  Caller allocates space.
**
*****************************************************************************/
RETCODE EXPORT DbcVarGetGlobalVal (STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarGetLocalVal
**
**  Description:
**     Given a local environment and a variable, returns the value of the
**     variable contained in the environment.
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**     output:
**        val:  value designator.  Caller allocates space.
**
*****************************************************************************/
RETCODE EXPORT DbcVarGetLocalVal (DESCRIPTOR env, STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarGetVal
**
**  Description:
**     Looks first for the value of the local variable in the specified local
**     environment, then in the global environment.  Returns the first value
**     it finds (i.e. local definitions shadow global ones).
**
**  Parameters:
**     input:
**        env:  the local environment descriptor.
**        var:  variable name.
**     output:
**        val:  value designator.  Caller allocates space.
**
*****************************************************************************/
RETCODE EXPORT DbcVarGetVal (DESCRIPTOR env, STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarSetGlobalVal
**
**  Description:
**     Sets the value of a global variable.  It is an error if the variable is
**     undefined (see DbcVarCreateAndSetGlobalVal).
**
**  Parameters:
**     input:
**        var:  variable name.
**        val:  value designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarSetGlobalVal (STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarSetLocalVal
**
**  Description:
**     Given a local environment and a variable name, sets the local value.
**     It is an error if the variable is undefined (see createAndSetLocalVal).
**
**  Parameters:
**     input:
**        env:  environment descriptor.
**        var:  variable name.
**        val:  value designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarSetLocalVal (DESCRIPTOR env, STRING * var, STRING * val);

/****************************************************************************
**
**  DbcVarSetVal
**
**  Description:
**     If the variable is found in the specified local environment it is set,
**     else if the variable exists in the global environment it is set,
**     else an error has occurred (see DbcVarCreateAndSetLocalVal,
**     DbcVarCreateAndSetGlobalVal).
**
**  Parameters:
**     input:
**        env:  the local environment descriptor.
**        var:  variable name.
**        val:  value designator.
**     output:
**        None.
**
*****************************************************************************/
RETCODE EXPORT DbcVarSetVal (DESCRIPTOR env, STRING * var, STRING * val);

/******************************** E O F ***********************************/
