/****************************************************************************
**
**  Name: LOADER.H
**
**  Description:
**     Contains the declaration of the generic loader interfaces.
**
**  Status:  CODED
**
**  $Log:   S:/tbird/mt2_amd/inc/loader.h_v  $
** 
**    Rev 1.0   20 Mar 1998 10:06:58   Eric
** Initial revision.
** 
**    Rev 1.0   16 Dec 1996 13:54:02   Judy
** Initial revision.
** 
**    Rev 1.0   27 Aug 1996 15:21:20   Judy
** Initial revision.
   
      Rev 1.0   23 Aug 1996 17:44:58   Judy
   Initial revision.
** 
**    Rev 1.6   30 Aug 1994 17:18:00   nghia
** Added LOAD_REGISTERS option to support OMF386 system initialize register
** values.
** 
**    Rev 1.5   14 Jul 1994 11:05:56   nghia
** Loader UI changes to support X86.
** Added load address space to interfaces.
** 
**    Rev 1.4   15 Jun 1994 18:21:00   nghia
** Added LdrWarning() to handle reporting errors for all loaders.
** 
**    Rev 1.3   14 Jun 1994 17:31:22   nghia
** Added TerminateServer() to handle Server termination.
** 
**    Rev 1.2   03 Jun 1994 14:58:24   nghia
** Exported LdrSendMessageToCLI() so that specific loader can use it to output
** message to the Shell.
** 
**    Rev 1.1   20 May 1994 12:55:38   nghia
** Revised from code review.
** Changed to text string for StartPC, StackBase, and StackSize to support
** processor independent.
** 
**    Rev 1.0   18 May 1994 16:39:50   nghia
** Initial revision.
**
**  $Header:   S:/tbird/mt2_amd/inc/loader.h_v   1.0   20 Mar 1998 10:06:58   Eric  $
**
**  Copyright (C) 1994 Microtek International.  All rights reserved.
**
*****************************************************************************/
#ifndef _LOADER_
#define _LOADER_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/
#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _ADDR_
#include "addr.h"
#endif

#ifndef _SYMBLSVR_
#include "symblsvr.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/
/*
** Loader Load Options
*/
#define LOAD_NONE        0x00000000L
#define LOAD_CODE        0x00000001L
#define LOAD_SYMBOL      0x00000002L
#define LOAD_ONDEMAND    0x00000004L
#define LOAD_ASM         0x00000008L
#define LOAD_STATUS      0x00000010L
#define LOAD_WARNING     0x00000020L
#define LOAD_DEMANGLE    0x00000040L
#define LOAD_UPDATE_BASE 0x00000080L
#define LOAD_RELOAD      0x00000100L
#define LOAD_MODULE      0x00000200L
#define LOAD_REGISTERS   0x00000400L
#define LOAD_IGNORE_CASE 0x00000800L

#define LOAD_FROM_CLI    0x10000000L

/* Default Load Options */
#define DEF_LOAD_OPTIONS (LOAD_CODE | LOAD_SYMBOL | LOAD_ONDEMAND \
| LOAD_STATUS | LOAD_WARNING | LOAD_DEMANGLE)

/* Macros to determine the specific load options */
#define IsLoadCode(x)      ((x & LOAD_CODE) > 0)
#define IsLoadSymbol(x)    ((x & LOAD_SYMBOL) > 0)
#define IsLoadOnDemand(x)  ((x & LOAD_ONDEMAND) > 0)
#define IsLoadAsm(x)       ((x & LOAD_ASM) > 0)
#define IsReportStatus(x)  ((x & LOAD_STATUS) > 0)
#define IsReportWarning(x) ((x & LOAD_WARNING) > 0)
#define IsReload(x)        ((x & LOAD_RELOAD) > 0)
#define IsDeMangle(x)      ((x & LOAD_DEMANGLE) > 0)
#define IsLoadModule(x)    ((x & LOAD_MODULE) > 0)
#define IsLoadFromShell(x) ((x & LOAD_FROM_CLI) > 0)
#define IsUpdateBase(x)    ((x & LOAD_UPDATE_BASE) > 0)
#define IsLoadReg(x)       ((x & LOAD_REGISTERS) > 0)
#define IsIgnoreCase(x)    ((x & LOAD_IGNORE_CASE) > 0)

/*
 * The maximum size of the string to hold the current module name
 */
#define MAX_MODULE_NAME_LEN   64 
#define MAX_LOADFILE_NAME_LEN 64 
#define MAX_NUM_LEN           30

/*
 * Here is where the callback functions called by the routine in LDRPROG
 * progress reporting dialog put the information about the size and 
 * status of the load.
 */
typedef struct {
   /* PUBLIC */
   U32 loadFileSize;/* size, in bytes, of the current loadfile */
   U32 curLocation; /* ftell offset within loadfile */
   CHAR startPC[MAX_NUM_LEN];     /* NULL if unknown */
   CHAR stackBase[MAX_NUM_LEN];   /* NULL if unknown */
   CHAR stackSize[MAX_NUM_LEN];   /* NULL if unknown */
   CHAR curModule[MAX_MODULE_NAME_LEN];
   CHAR curLoadFile[MAX_LOADFILE_NAME_LEN];
   /* PRIVATE ONLY - DO NOT TOUCH THESE */
   U32 numSymbols;  /* number of symbols loaded so far */
   U32 numBytes;    /* number of bytes loaded so far */
   U16 numModules;  /* number of modules loaded so far */
   U32 numLines;    /* number of lines loaded so far */
   U32 numFunctions;/* number of functions loaded so far */
   U32 numTypes;    /* number of types loaded so far */
   U32 lastNumModules;
   U32 lastNumSymbols;
   U32 lastNumBytes; 
   U32 lastLocation;
   U16 percentInc;
} LDRSTATBLOCK;

typedef LDRSTATBLOCK FAR *LPLDRSTATBLOCK;

#ifndef HFILE
   // from windows.h for Windows 3.1 
   typedef int HFILE; /* Polymorphic with C runtime file handle type */
#endif

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/
#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************
**
** LdrLoad
**
**	Description:
**		Load the specified load file, using the input load flags. If openDlg is
**    TRUE the opend the LoadFile dialog for the user to select a loadfile.
**
**	Parameters:
**		input:
**			loadFileName   File name (including path) of the loadfile.
**                      this must be a string of MAX_STRING_SIZE.   
**       loadFlags      Load options (combinations of one or more load
**                      options: LOAD_CODE, LOAD_SYMBOL, etc.,
**       parentHwnd     Parent Window handle (HWND).
**       loadSpace      Load address space - See ADDR.H for more info.
**       openDlg        TRUE = open the LoadFile dialog, FALSE = direct load.
**
**		output:
**			none.
**
**	Returns:
**       GOOD on successful load operation
**		   if error occurs, an error code is returned to caller.
**
*****************************************************************************/
RETCODE EXPORT LdrLoad(LPSTR loadFileName, U32 loadFlags, U32 parentHwnd,
                       ADDR_SPACE loadSpace, BOOLEAN openDlg);

/*****************************************************************************
**
**  LdrLoadModuleByDesc
**
**	Description:
**		Load local symbols for the specified module.  This function is used by
**    the Symbol Server and the Shell's LOAD command in response to the
**    ondemand loading request.  Only local symbols and line number records
**    will be loaded.  All global symbols - including module, function, and
**    type information has already been loaded).
**
**	Parameters:
**		input:
**       fileName       Filename where module is found
**       moduleName     Name of module to load
**       moduleDesc     Module Descriptor of the specified module
**                      (assigned by the Symbol Server).
**       loadAddrSpace  Load address space.
**
**		output:
**			none.
**
**	Returns:
**       GOOD on successful load of required module
**		   if an error occurred, an error code is returned to report.
**
*****************************************************************************/
RETCODE EXPORT LdrLoadModuleByDesc(LPSTR Filename, LPSTR moduleName,
      SYM_DESCRIPTOR moduleDesc, ADDR_SPACE loadAddrSpace);


/*****************************************************************************
**
**  LdrGetStartPC
**
**  Description:
**     Function to retrieve the loaded loadfile start PC.  This function will
**     allocate the returned address descriptor, Caller is responsible for
**     destroying the descriptor when done.  When error occurs, this function
**     does not return the start PC.
**
**  Parameters:
**     input:
**        none.
**
**     output:
**        lpStartPC     Pointer to an Address Descriptor.
**
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned to caller.
**
******************************************************************************/
RETCODE EXPORT LdrGetStartPC(DESCRIPTOR FAR *lpStartPC);

/******************************************************************************
**
**  LdrGetStackInfo
**
**  Description: Function to retrieve the loadfile's stack information.
**     This function will allocate the returned address descriptor, Caller is
**     responsible for destroying the descriptor when done.  When error occurs,
**     this function does not return the descriptor.
**
**  Parameters:
**     input:
**        none.
**
**     output:
**        lpStkTop   Pointer to address descriptor.
**        lpStkSize  Pointer to Stack size.
**
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned to caller.
**
*****************************************************************************/
RETCODE EXPORT LdrGetStackInfo(DESCRIPTOR FAR *lpStkTop, U32 *lpStkSize);

/*****************************************************************************
**
**  LdrGetLoadInfo
**
**  Description: Funtion to retrieve information about the last load
**     operation.
**
**  Parameters:
**     input:
**        none.
**
**     output:
**        lpFileName    Pointer to loadfile name.
**        lpLoadFlags   Pointer to load flags.
**        lpLoadSpace   Pointer to load address space - See ADDR.H.
**        
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned.
**
*****************************************************************************/
RETCODE EXPORT LdrGetLoadInfo(LPSTR lpFileName, U32 FAR *lpLoadFlags,
                              ADDR_SPACE FAR *lpLoadAddrSpace);

/******************************************************************************
**
**  LdrCheckTimestamp
**
**  Description:
**     Compare the Source timestamp and the current loadfile.
**
**  Parameters:
**     input:
**       none.
**
**     output:
**       lpSourceFile   Source file name (disk file)
**       lpValid        Pointer to status:
**                      TRUE if source <= loadfile timestamp, else FALSE.
**
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned to caller.
**
******************************************************************************/
RETCODE EXPORT LdrCheckTimestamp(LPSTR lpSourceFile, BOOLEAN FAR *lpValid);

/******************************************************************************
**
**  LdrGetTimestamp
**
**  Description:
**     Compare the Source timestamp and the current loadfile.
**
**  Parameters:
**     input:
**       fileHandle     File handle of a Source file - returned by fopen().
**       lpTS           pointer to TIMESTAMP_TYPE structure. 
**
**     output:
**       none.
**
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned to caller.
**
******************************************************************************/
RETCODE EXPORT LdrGetTimestamp(HANDLE fileHandle, TIMESTAMP_TYPE FAR* lpTS);

/******************************************************************************
**
**  LdrCompareTimestamps
**
**  Description:
**     Compare the Source timestamp and the current loadfile.
**
**  Parameters:
**     input:
**       lpt1           TimeStamp of the first file.
**       lpt2           TimeStamp of the second file.
**
**     output:
**       resultEqual    TRUE if lpt1 == lpt2, else FALSE.
**
**	Returns:
**       GOOD on successful operation
**		   if an error occurred, an error code is returned to caller.
**
******************************************************************************/
RETCODE EXPORT LdrCompareTimestamps(TIMESTAMP_TYPE FAR* lpt1,
                                    TIMESTAMP_TYPE FAR* lpt2,
                                    BOOLEAN FAR *resultEqual);

/*****************************************************************************
**
** LdrCliLoad
**
**	Description:
**		Load the specified input load file from the Shell command line.
**
** Syntax:
**    load "<filename>" [addrspace][symbols|nosymbols] [code|nocode]
**                      [progress|noprogress] [demand|nodemand]
**                      [module <module path name>] [reload] [warn|nowarn]
**                      [demangle|nodemangle] [asm|noasm]
**
**	Parameters:
**		input:
**			cmdString      Concatenated list of null-terminated strings indicating
**			               various options to loader
**			argc           Number of arguments found in argv parameter
**			argv           Offsets in cmdString where strings live
**
**		output:
**			none.
**
**	Returns:
**       GOOD on successful load operation
**		   if an error occurred, an error code is returned to report.
**
*****************************************************************************/
RETCODE EXPORT LdrCliLoad(LPSTR cmdString, U32 argc, U32 argv[]);

/******************************************************************************
**
**  LdrCliSetToolUse
**
**  Description:
**     Allow user to set the <CompilerUsed> from the Shell window.
**
**  Syntax:
**     compilerUsed [none|<compiler>]
** 
**  Parameters:
**     input:
**        cmdString     CLI command string.
**        argc          number of arguments in the command string.
**        argv          String contains command arguments.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT LdrCliSetToolUse(LPSTR cmdString, U32 argc, U32 argv[]);

/******************************************************************************
**
**  LdrCliSetSecNames
**
**  Description:
**     Allow user to set section names from the Shell window.
**
**  Syntax:
**    _sectionNames [<code section>,<data section>]
**  
**  Parameters:
**     input:
**        cmdString     CLI command string.
**        argc          number of arguments in the command string.
**        argv          String contains command arguments.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT LdrCliSetSecNames(LPSTR cmdString, U32 argc, U32 argv[]);

/******************************************************************************
**
**  LdrCliMergeSections
**
**  Description:
**     Allow user to set the <MergeSection> from the Shell window.
**
**  Syntax:
**     mergeSections [on|off]
**
**  Parameters:
**     input:
**        cmdString     CLI command string.
**        argc          number of arguments in the command string.
**        argv          String contains command arguments.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT LdrCliMergeSections(LPSTR cmdString, U32 argc, U32 argv[]);

/******************************************************************************
**
**  LdrCliResetLoaders
**
**  Description:
**     Allow user to reset the LoaderServer from the Shell window.
**
**  Syntax:
**    resetLoaders "[path to the LOADER.INI file]"
**
**  Parameters:
**     input:
**        cmdString     CLI command string.
**        argc          number of arguments in the command string.
**        argv          String contains command arguments.
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT LdrCliResetLoaders(LPSTR cmdString, U32 argc, U32 argv[]);

/******************************************************************************
**
**  LdrGetToolUse
**
**  Description:
**     Get the current selected tool or open the ToolChain dialog box to
**     let the user select a new tool.  If the received buffer is specified,
**     copy the current selected tool name into the returned buffer.
**
**  Parameters:
**     input:
**        parentWnd     parent Windows handle (HWND) - U32 for Actor.
**        toolName      String buffer of MAX_STRING_LENGTH to receive the
**                      user selected tool.  If not specified (NULL) then
**                      toolName will not receive the returned selected tool.
**        selectNewTool Flag indicates whether or not to select a new tool
**                         TRUE = open ToolChains dialog box.
**                    
**     output:
**        none.
**
******************************************************************************/
RETCODE EXPORT LdrGetToolUse(U32 parentWnd, LPSTR toolName,
                             BOOLEAN selectNewTool);

/*****************************************************************************
**
**  LdrGetStatBlock
**
**  Purpose:
**     Get the pointer to the loader status block for the specific loader.
**     The status block is PRIVATE data in the LoaderLauncher DLL, so
**     the pointer will not change during PowerViews loading operation.
**
**     NOTES:  These are the steps to use the common progress indicator:
**             1. Call LdrGetStatBlock() to get the pointer to the status
**                block.  Save the information for later use.
**             2. Initialize the status block loadFileSize and curLocation.
**             3. To update the progress, update the curLocation with the
**                current loadfile file pointer and call LdrProgressInc().
**             4. Call LdrProgressDone() to report done status.  
**
**  Input:
**     none
**  Output:
**     lpstb: the pointer to the loader status block.
**
*****************************************************************************/
RETCODE EXPORT LdrGetStatBlock(LPLDRSTATBLOCK FAR *lpstb);

/*****************************************************************************
**
**  LdrProgressInc
**
**  Purpose:
**     Update the load progress dialog status based on the current
**  values in the loader status block.
**
**  Input:
**     none
**  Output:
**     none
*****************************************************************************/
RETCODE EXPORT LdrProgressInc(VOID);

/*****************************************************************************
**
**  LdrProgressDone
**
**  Purpose:
**     Signals the load progress dialog that the load operation is about
**     to be terminated.  For normal termination, the caller is responsible
**     to set the StartPC, StackBase and StackSize values (if any) before
**     calling this routine.
**
**  Input:
**     loadErr    Error condition of the termination - error code.
**  Output:
**     none
*****************************************************************************/
RETCODE EXPORT LdrProgressDone(RETCODE loadErr);

/****************************************************************************
**
**  LdrInfoDialog
**
**  Purpose:
**     Create the Load Information Dialog for PowerViews, and wait until
**     the OK button is pressed.
**
**  Input:
**     parentHwnd:  The parent window handle (HWND) - U32 for Actor.
**
**  Output:
**     none
**
*****************************************************************************/
RETCODE EXPORT LdrInfoDialog(U32 parentHwnd);

/****************************************************************************
**
**  LdrSendMessageToCLI
**
**  Purpose:
**     Routine to handle sending loader message to the CLI.
**
**  Input:
**     strMsg:  Pointer to the string message.
**
**  Output:
**     none
**
*****************************************************************************/
RETCODE EXPORT LdrSendMessageToCLI(LPSTR strMsg);

/****************************************************************************
**
**  TerminateServer
**
**  Purpose:
**     Routine to cleanup the DLL before the application is terminated.
**
**  Input:
**     none
**
**  Output:
**     none
**
*****************************************************************************/
RETCODE EXPORT TerminateServer(VOID);

/****************************************************************************
**
**  LdrWarning
**
**  Purpose:
**     Routine to handle reporting error for loader (IEEE695, SREC, OMF86).
**     Return GOOD if successfully report the error.
**     Return ER_LDR_ABORT if the user pressed CANCEL button to abort
**     the load operation.  Caller of this routine should cleanup itself
**     and calls LdrProgressDone(ER_LDR_ABORT) to terminate the load.
**
**  Input:
**     err        error code to report
**     infoStr1   first optional string argument  - use NULL if not available 
**     infoStr2   second optional string argument - use NULL if not available 
**     infoStr3   third optional string argument  - use NULL if not available 
**
**  Output:
**     none
**
*****************************************************************************/
RETCODE EXPORT LdrWarning(RETCODE err, LPSTR infoStr1, LPSTR infoStr2,
                          LPSTR infoStr3);

#ifdef __cplusplus
}
#endif

#endif /* _LOADER_ */
/******************************** E O F *************************************/
