/****************************************************************************
**
**  Name:  basetbl.h
**
**  Description:
**     Prototypes for the functions to manipulate BASE symbols.
**
**  Status:  COMPLETE
**
**  $Log:   S:/tbird/mt2_amd/symbol/basetbl.h_v  $
** 
**    Rev 1.0   20 Mar 1998 10:21:42   Eric
** Initial revision.
** 
**    Rev 1.1   26 Feb 1997 11:42:22   Judy
** 
**    Rev 1.1   14 Jan 1997 15:30:30   Judy
** No change.
** 
**    Rev 1.0   14 Jun 1996 16:41:54   Judy
** Initial revision.
** 
**    Rev 1.15   05 Jan 1995 16:31:38   nghia
** Fixed GPF bug - GE reports
** Revised BaseSymbol constructor to handle initialize baseName from the caller.
** Added 2 member to set baseNameOffset and byAddressTableOffset.
** 
**    Rev 1.14   26 Sep 1994 10:15:12   nghia
** Revised symbol base class declaration to store the segment with LDTSel 
** instead of gdtSel.  This will allow the symbol server to use the same
** Address server interfaces for Real and Protected mode symbols.
** 
**    Rev 1.13   22 Aug 1994 18:33:18   brucea
** Changed method GetBaseType interface to return BASE_TYPE rather than U8.
** 
**    Rev 1.12   14 Jul 1994 01:47:54   brucea
** Added: more protected variables to hold segment/selector, ldt, and modes for
**        Intel architectures
**        Get and Set functions for these variables
** 
**    Rev 1.11   22 Oct 1992 18:24:54   brucea
** Initialize maxIndex to MAX_BASE_INDEX_SIZE so that it cannot exceed 4095
**    is the 12 bit restriction on the base index value since it is packed
**    with the 4 bit type field.
** 
**    Rev 1.10   25 Jul 1992 16:31:38   brucea
** Cosmetic cleanup
** 
**    Rev 1.9   10 Jul 1992 18:38:10   brucea
** General cleanup
** Added: baseCount to private BaseIndexTable variables
**        Init var in constructor
**        BaseGetBaseCount()
** 
**    Rev 1.8   20 Apr 1992 09:10:26   brucea
** Added: BasetblResetCacheVars
** 
**    Rev 1.7   11 Dec 1991 17:04:36   brucea
** Added MAX_BASE_INDEX_SIZE
** 
**    Rev 1.6   11 Dec 1991 16:31:12   john
** Added function to get base offsets given a base name
** 
**    Rev 1.5   06 Dec 1991 17:06:42   john
** Moved some inline code into the cpp file.  The code was too involved
** to really be left inline.
** Added prototype for ObliterateTable.  This is used to free dynamically
** allocated memory in the base index table.
** Added prototype for a function to free the dynamically allocated 
** by-address table memory.
** 
**    Rev 1.4   29 Nov 1991 19:45:56   brucea
** changed logicalAddr to offsetAddr
** changed NULL to NULL_SYMBOL
** 
**    Rev 1.3   18 Nov 1991 15:21:24   brucea
** Added #include symutil.h
** 
**    Rev 1.2   01 Nov 1991 10:46:56   brucea
** Added CompareAddrVars
** Changed return parameter of GetRootBaseSymbolOffset to TABLE_OFFSET
** 
**    Rev 1.1   11 Oct 1991 14:43:44   brucea
** Large changes: made BaseSymbol a class, created constructor and operator new
** Added access methods to class variables
** Created class BaseIndexTable, inherited from IndexTable
** Added (outside) functions AddAddrToByAddrTable and BaseSetAddr
** 
**    Rev 1.0   03 Oct 1991 11:42:50   john
** Initial revision.
** 
**  $Header:   S:/tbird/mt2_amd/symbol/basetbl.h_v   1.0   20 Mar 1998 10:21:42   Eric  $
**
**  Copyright (C) 1991 Microtek International.  All rights reserved.
**
*****************************************************************************/

#define _BASETBL_

                       /****************************
                        *                          *
                        *       INCLUDE FILES      *
                        *                          *
                        ****************************/

#ifndef _BASEWIND_
#include "basewind.h"
#endif

#ifndef _INDEXTBL_
#include "indextbl.h"
#endif

#ifndef _SYMADD_
#include "symadd.h"
#endif

#ifndef _SYMUTIL_
#include "symutil.h"
#endif

                       /****************************
                        *                          *
                        *        DEFINITIONS       *
                        *                          *
                        ****************************/

#define MAX_BASE_INDEX_SIZE 4095L

                       /****************************
                        *                          *
                        *        PROTOTYPES        *
                        *                          *
                        ****************************/

//-----------------------------------------------------------------------
//  baseSymbol
//
//  Description:
//     Methods for adding BASE information to the symbol pool.
//-----------------------------------------------------------------------

class BaseSymbol {

protected:
   TABLE_OFFSET baseNameOffset;     // st offset to name
   TABLE_OFFSET nextBaseOffset;     // linked list pointer to next base object
   U32          addressMin;         // smallest addr of syms in by-addr table
   U32          addressMax;         // largest addr of syms in by-addr table
   U32          byAddressTableOffset;  // st offset to by-address object
   BASE_ADDRESS baseAddress;        // logical address offset from base value
   U16          GDTsel;             // GDT selector in prot.
   U16          segOrLDTsel;        // segment in real; LDT selector in prot.
   BASE_INDEX   baseIndex;          // store index from bit pointing to this
   U8           baseType;           // CODE, DATA, CODE-DATA, UNKNOWN
   BOOLEAN      addrTableDirty;     // TRUE ==> by-address table not sorted
   U8           segAddrSpace;       // SEG_SPACE_USER, SEG_SPACE_SMM
   U8           symMode;            // SYM_NONE, SYM_REAL, SYM_PROTECTED         
   
//NOTE: baseType is not spec'ed as BASE_TYPE because enums are size int and we
//      want the baseType size to be as small as possible.
//      Same is true for segAddrSpace and symMode

public:
//-----------------------------------------------------------------------
// Parameterized Constructor; returns error code via <err> reference var
//-----------------------------------------------------------------------
BaseSymbol(BASE_INDEX baseIndex,
           BASE_ADDRESS baseAddr,
           BASE_TYPE baseTypeParam,
           RETCODE& err);


//-----------------------------------------------------------------------
// operator new
//
// Purpose: allocates memory in st for a new BaseSymbol
//
// Input parameter:
//    size: amount of space to allocate
//
// Output parameter:
//    err: RETCODE from memory allocation
//
// Return parameter:
//    TABLE_OFFSET: offset into st where memory was allocated
//--------------------------------------------------------------------------
VOID *operator new(size_t size);


//--------------------------------------------------------------------------
// CompareAddrVars
//
// Returns TRUE-FALSE whether address in range or not
//--------------------------------------------------------------------------
BOOLEAN CompareAddrVars(OFFSET_ADDR_TYPE offsetAddr)  {
   if ((offsetAddr < addressMin) ||
       (offsetAddr > addressMax)) return FALSE;
   else return TRUE;
   }


//--------------------------------------------------------------------------
// GetBaseNameOffset
//
// Returns baseNameOffset
//--------------------------------------------------------------------------
U32 GetBaseNameOffset(VOID)  { return baseNameOffset; }


//--------------------------------------------------------------------------
// GetNextBaseOffset
//
// Returns nextBaseOffset
//--------------------------------------------------------------------------
TABLE_OFFSET GetNextBaseOffset(VOID)  { return nextBaseOffset; }


//--------------------------------------------------------------------------
// GetAddressMin
//
// Returns addressMin
//--------------------------------------------------------------------------
U32 GetAddressMin(VOID)  { return addressMin; }


//--------------------------------------------------------------------------
// GetAddressMax
//
// Returns addressMax
//--------------------------------------------------------------------------
U32 GetAddressMax(VOID)  { return addressMax; }


//--------------------------------------------------------------------------
// GetByAddressTableOffset
//
// Returns the byAddressTableOffset class variable
//--------------------------------------------------------------------------
U32 GetByAddressTableOffset(VOID) { return byAddressTableOffset; }


//--------------------------------------------------------------------------
// GetBaseAddress
//
// Returns baseAddress
//--------------------------------------------------------------------------
BASE_ADDRESS GetBaseAddress(VOID)  { return baseAddress; }


//--------------------------------------------------------------------------
// GetSegOrLDTsel
//
// Returns segOrLDTsel
//--------------------------------------------------------------------------
U16 GetSegOrLDTsel(VOID)  { return segOrLDTsel; }


//--------------------------------------------------------------------------
// GetGDTsel
//
// Returns GDTsel
//--------------------------------------------------------------------------
U16 GetGDTsel(VOID)  { return GDTsel; }


//--------------------------------------------------------------------------
// GetBaseType
//
// Returns baseType
//--------------------------------------------------------------------------
BASE_TYPE GetBaseType(VOID)  { return (BASE_TYPE)baseType; }


//--------------------------------------------------------------------------
// GetBaseIndex
//
// Returns baseIndex
//--------------------------------------------------------------------------
BASE_INDEX GetBaseIndex(VOID)  { return baseIndex; }


//--------------------------------------------------------------------------
// GetAddrTableDirty
//
// Returns addrTableDirty
//--------------------------------------------------------------------------
BOOLEAN GetAddrTableDirty(VOID)  { return addrTableDirty; }


//--------------------------------------------------------------------------
// GetSegAddrSpace
//
// Returns segAddrSpace
//--------------------------------------------------------------------------
SEG_ADDR_SPACE GetSegAddrSpace(VOID)  { return (SEG_ADDR_SPACE)segAddrSpace; }


//--------------------------------------------------------------------------
// GetSymMode
//
// Returns symMode
//--------------------------------------------------------------------------
SYM_MODE GetSymMode(VOID)  { return (SYM_MODE)symMode; }


//--------------------------------------------------------------------------
// SetBaseAddr
//
// Sets the base address of the base object
//--------------------------------------------------------------------------
VOID SetBaseAddr(BASE_ADDRESS baseAddr) { baseAddress = baseAddr; }


//--------------------------------------------------------------------------
// SetAddrTableDirty
//
// Sets the address dirty flag
//--------------------------------------------------------------------------
VOID SetAddrTableDirty(BOOLEAN b) { addrTableDirty = b; }


//--------------------------------------------------------------------------
// SetGDTsel
//
// Sets the GDTsel value
//--------------------------------------------------------------------------
VOID SetGDTsel(U16 passedGDTsel)
   { GDTsel = passedGDTsel; }


//--------------------------------------------------------------------------
// SetSegOrLDTsel
//
// Sets the SegOrLDTsel value
//--------------------------------------------------------------------------
VOID SetSegOrLDTsel(U16 passedSegOrLDTsel)
   { segOrLDTsel = passedSegOrLDTsel; }


//--------------------------------------------------------------------------
// SetSegAddrSpace
//
// Sets segAddrSpace
//--------------------------------------------------------------------------
VOID SetSegAddrSpace(SEG_ADDR_SPACE s)  { segAddrSpace = (U8)s; }


//--------------------------------------------------------------------------
// SetSymMode
//
// Sets symMode
//--------------------------------------------------------------------------
VOID SetSymMode(SYM_MODE s)  { symMode = (U8)s; }


//--------------------------------------------------------------------------
// UpdateAddrVars
//
// update the min and max address vars (uses macros)
//--------------------------------------------------------------------------
VOID UpdateAddrVars(U32 minAddr, U32 maxAddr) {

   addressMin = min(addressMin, minAddr);
   addressMax = max(addressMax, maxAddr);
   }

//--------------------------------------------------------------------------
// SetBaseNameOffset
//
// Sets baseNameOffset
//--------------------------------------------------------------------------
VOID SetBaseNameOffset(TABLE_OFFSET offset)  { baseNameOffset = offset; }

//--------------------------------------------------------------------------
// SetByAddressTableOffset
//
// Sets byAddressTableOffset
//--------------------------------------------------------------------------
VOID SetByAddressTableOffset(TABLE_OFFSET offset)  {
   byAddressTableOffset = offset;
   }

};  // end of BaseSymbol class functions


//--------------------------------------------------------------------------
// BaseIndexTable CLASS
//--------------------------------------------------------------------------
class BaseIndexTable : public IndexTable {

private:
   TABLE_OFFSET rootBaseSymbolOffset;
   U32          baseCount;  // number of bases loaded so far

public:
   // initialize the starting BaseSymbol "pointer" to null since there are no
   // BaseSymbol objects created until the loader creates one or more

//--------------------------------------------------------------------------
// BaseIndexTable - constructor
//--------------------------------------------------------------------------
BaseIndexTable() {
   rootBaseSymbolOffset = NULL_SYMBOL;
   baseCount = 0L;  // initialize number of bases loaded
   maxIndex = MAX_BASE_INDEX_SIZE;  // reduce default value to restrict
                                    // index size to 12 bits for bases
}

//--------------------------------------------------------------------------
// GetRootBaseSymbolOffset
//--------------------------------------------------------------------------
TABLE_OFFSET GetRootBaseSymbolOffset() { return rootBaseSymbolOffset; }


//--------------------------------------------------------------------------
// BaseAddIndex
//
// Description:
//    Calls TableIndex AddIndex, then if successful,
//    sets the root offset to the first base symbol created
//    It only sets the root offset if it is NULL, otherwise just returns
//--------------------------------------------------------------------------
RETCODE BaseAddIndex(TABLE_INDEX index, TABLE_OFFSET itemOffset);


//---------------------------------------------------------------------------
// ObliterateTable
//
// Purpose:
//    Unlocks, then frees the table to Windows memory.
//---------------------------------------------------------------------------
VOID ObliterateTable(VOID);

//--------------------------------------------------------------------------
// BaseGetBaseCount
//
// Description:
//    returns count of number of bases added.
//--------------------------------------------------------------------------
U32 BaseGetBaseCount(VOID) {return baseCount;}

};  // end of BaseIndexTable class definition and functions
      
                       /*********************************
                        *                               *
                        * NON-CLASS FUNCTION PROTOTYPES *
                        *                               *
                        *********************************/

//--------------------------------------------------------------------------
// Description:
//    Add a symbol offset to the by-address sorted table and update base min
//    and max address values
//--------------------------------------------------------------------------
RETCODE
AddAddrToByAddrTable(BASE_INDEX baseIndex, TABLE_OFFSET symOffset,
                     U32 minAddr, U32 maxAddr);

//--------------------------------------------------------------------------
// Description:
//    Returns the base address of <baseIndex>
//--------------------------------------------------------------------------
RETCODE BaseGetAddr(BASE_INDEX baseIndex, BASE_ADDRESS *baseAddr);


//--------------------------------------------------------------------------
// Description:
//    Modifies the base address of <baseIndex>
//--------------------------------------------------------------------------
RETCODE BaseSetAddr(BASE_INDEX baseIndex, BASE_ADDRESS baseAddr);

//--------------------------------------------------------------------------
// GetBaseOffsetByName
//
// Description:  Given a basename, find the offset in the symbol table of
//               the base.
//
// Inputs:
//       baseName: the name of the base to find.
// Outputs:
//       baseOffset: the offset of the base found.
// Errors:
//       ER_SYMBOL_NOT_FOUND
//--------------------------------------------------------------------------
RETCODE GetBaseOffsetByName(LPSTR baseName, TABLE_OFFSET &baseOffset);

//--------------------------------------------------------------------------
// RemoveAllByAdrTables
//
// Description: Frees the memory that was allocated by the by-address-tables.
//              This allows symbols to be cleanly deleted and all memory 
//              used by bases/base indexes/by address tables to be freed.
//
// Input Parameters:
//    none:
// Output Parameters:
//    none:
// Errors:
//    ER_OUT_OF_MEMORY:
// NOTE: no BaseSymbol is created on initialization.  This means that any
//    loader must first create a base (and its index) before using it.
//--------------------------------------------------------------------------
VOID RemoveAllByAdrTables(VOID);

//--------------------------------------------------------------------------
//  ResetCacheVars
//
//  Description:
//    Re-initialize all cached vars after removal of symbols
//--------------------------------------------------------------------------
VOID BasetblResetCacheVars(VOID);

/******************************** E O F ***********************************/
