/***************************************************************************
**
**  Name:  emusetup.c
**
**  Description:
**     Michelle setup-group routines.
**     include:
**     "Get the identity of the Michelle FW version, HW configuration
**      and HW model type",
**     "Set access size to be byte/word/dword-accessing",
**     "Set or read the memory-write verify-flag",
**     "Control ready using external/internal",
**     "Set or read the memory mapping",
**     "Read all setup information",
**     "Set control signals",
**    * "set or read the wait-state" --- This is a issue to be discussed.
**      We will decide this latter.
**
**  Status:  preliminary
**
**  $Log:   S:/tbird/arcmtat2/am186/emusetup.c_v  $
** 
**    Rev 1.2   12 Aug 1998 10:12:30   Eric
** Modify attrib of SetMap.
** 
**    Rev 1.1   30 Apr 1998 09:16:36   Eric
** 
**    Rev 1.0   20 Mar 1998 09:20:54   Eric
** Initial revision.
** 
**    Rev 1.0   03 Dec 1996 09:27:36   gene
** Initial revision.
** 
**    Rev 1.0   10 May 1996 10:05:14   jacky
** Get file from ATL. V1.3
** 
** 
**  $Header:   S:/tbird/arcmtat2/am186/emusetup.c_v   1.2   12 Aug 1998 10:12:30   Eric  $
**
** Copyright (C) 1992 Microtek International, Inc.
**
****************************************************************************/

                       /****************************
			*                          *
			*       INCLUDE FILES      *
			*                          *
                        ****************************/

#ifndef _EMU_SETUP_
#include "emusetup.h"
#endif

#ifndef _EMU_LIB_
#include "emulib.h"
#endif

#ifndef _EMU_TRACE_
#include "emutrace.h"
#endif

#ifndef _EMU_LLFW_
#include "emullfw.h"
#endif

#ifndef _EMU_EXTERNAL_
#include "emuext.h"
#endif

#include <string.h>
		       /****************************
			*                          *
			*     LOCAL DEFINITIONS    *
			*                          *
			****************************/


BANK_INFO bankInfo[MAX_BANK+1];
U32 bankSize;
MAP_INFO mapSetInfo[MAX_BANK*2-1];
MAP_TABLE_INFO mapTable[MAX_MAP+1];
SLICE slice[MAX_BANK];
U8 bankInfoCount,mapSetCount;
enum BANK_NO {BANK1, BANK2, BANK3, BANK4, ER_BANK, NO_BANK = 0xff};


			/****************************
			 *                          *
			 *    EXTERNAL VARIABLES    *
			 *                          *
			 ****************************/

			/****************************
			 *                          *
			 *     LOCAL PROTOTYPES     *
			 *                          *
			 ****************************/

VOID PRIVATE RealMapSet(U16 attr, U32 mapEnd, U32 mapStart);
VOID PRIVATE VirMapSet(U16 attr, U32 mapEnd, U32 mapStart);
VOID PRIVATE L_FwMapSet(VOID);
VOID PRIVATE clearMapEntry(U8 entryPtr);
VOID PRIVATE setTableInfoBankAddr(MAP_TABLE_INFO *mapTableInfo, SLICE *slice,U8 bankNo);
STATUS PRIVATE addMapTable(MAP_TABLE_INFO *mapInfo);
STATUS PRIVATE FindEqualMap(U32 startAddr,U32 endAddr,U16 attr,U8 mapTablePtr);
STATUS PRIVATE DeleteMap(MAP_TABLE_INFO *mapInfo);
BOOLEAN PRIVATE checkMapEqu(U32 startAddr, U32 endAddr, U16 attr);

			/****************************
			 *                          *
			 *      EXECUTABLE CODE     *
			 *                          *
			 ****************************/


/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS InitMapTable(VOID) {
STATUS status;
U16 lp,lp2;

   status = GOOD;
   DisEmStop();
   if (MapSet(0xff70,0xfffff,0)) status = EMM_SET_ERROR;
   EnEmDisViol();
   bankInfoCount = 0;
   for (lp = 0; lp <= MAX_BANK; lp++) {
      bankInfo[lp].status = NO_BANK;   // bank no use
      bankInfo[lp].bottom = BUTTOM_NULL;
      bankInfo[lp].top    = TOP_NULL;
      slice[lp].addr1 = BUTTOM_NULL;
      slice[lp].addr2 = TOP_NULL;
      for (lp2 = 0; lp2 < BLOCK_NO; lp2++)
	 bankInfo[lp].table[lp2] = 0xff00;  // external
   }
   for (lp = 0 ; lp <= MAX_MAP ; lp++) {
      mapTable[lp].startAddr = BUTTOM_NULL;
      mapTable[lp].endAddr = TOP_NULL;
      mapTable[lp].attr = 0;
      for (lp2=0 ; lp2 < MAX_BANK+1 ; lp2++) {
	 mapTable[lp].bank[lp2] = 0;
	 mapTable[lp].bankSA[lp2] = BUTTOM_NULL;
	 mapTable[lp].bankEA[lp2] = TOP_NULL;
      }
   }
   return(status);
}
/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS SliceNewAddr(U32 startAddr, U32 endAddr) {
U8 lp,lp2;

   if ((endAddr - startAddr) > (bankSize*4 - 1)) return(ADDR_RANGE_OVERFLOW);
   else {
      lp = 0;
      slice[lp].addr1 = startAddr;
      slice[lp].addr2 = ((startAddr/bankSize)+1)*bankSize - 1;
      while (slice[lp].addr2 < endAddr) {
	 if (lp >= 4) return(ADDR_RANGE_OVERFLOW);
	 slice[lp+1].addr1 = slice[lp].addr2 + 1;
	 slice[lp+1].addr2 =
	    ((slice[lp+1].addr1/bankSize)+1)*bankSize - 1;
	 lp++;
      }
      slice[lp].addr2 = endAddr;
      for (lp2 = (U8)(lp+1); lp2 < MAX_BANK; lp2++) {   /* CLEAR ANOTHER addr */
	 slice[lp2].addr1 = BUTTOM_NULL;
	 slice[lp2].addr2 = TOP_NULL;
      }
   }
   return(GOOD);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U8 SearchBank(U32 newAddr,U8 *bankNo) {
U8  lp,findBank;

   findBank = 0;
   if (bankInfoCount > 0) {
      for (lp = 0; lp < MAX_BANK; lp++)
	 if ((newAddr >= bankInfo[lp].bottom) && (newAddr <= bankInfo[lp].top)) {
	    bankNo[findBank] = lp;
	    findBank++;
	 }
   }
   return(findBank);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
VOID VirMapSet(U16 attr, U32 addrEnd, U32 addrStart) {
   mapSetInfo[mapSetCount].startAddr = addrStart;
   mapSetInfo[mapSetCount].endAddr = addrEnd;
   mapSetInfo[mapSetCount].attr = attr;
   mapSetCount++;
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
VOID SetMapTable(U32 bankAddr1, U32 bankAddr2, U16 attr,U8 bankNo) {
U16 startBlock,endBlock,lp,tempAttr;


   if (bankNo != ER_BANK) {
      tempAttr = attr ;
      if ( tempAttr >= 0xff00 ) tempAttr = 0xff70;
      startBlock = (U16)((bankAddr1 - bankInfo[bankNo].bottom) / BLOCK_SIZE);
      endBlock = (U16)(((bankAddr2 - bankInfo[bankNo].bottom) / BLOCK_SIZE) + 1);
      for (lp = startBlock; lp < endBlock ; lp++) {
	 bankInfo[bankNo].table[lp] = tempAttr;
      }
    }
   tempAttr |= (bankNo << 10);//NBSEL0,NBSEL1
   attr = tempAttr;
   VirMapSet(attr,bankAddr2,bankAddr1);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID RealMapSet(U16 attr, U32 mapEnd, U32 mapStart) {
//U32 addrEnd,addrStart;
//   addrEnd = (mapEnd >> 11);
//   addrStart = (mapStart >> 11);
   DisEmStop();
   MapSet(attr,mapEnd,mapStart);
   EnEmDisViol();
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U8 CreateBank(U32 bankAddr1, U32 bankAddr2, U16 attr) {
U8 lp,bank ;
S16 lp2;

   bank = 0xFF;
   lp = 0 ;
   for (lp = 0; lp < MAX_BANK; lp++) {
      if (bankInfo[lp].status == NO_BANK) {
	 bank = lp;
	 bankInfo[lp].status = lp;
	 bankInfo[lp].bottom = (bankAddr1 / bankSize) * bankSize;
	 bankInfo[lp].top = bankInfo[lp].bottom + bankSize - 1;
	 for (lp2 = 0; lp2 < BLOCK_NO; lp2++)
	    bankInfo[lp].table[lp2] = 0xff00;  // external
	 SetMapTable(bankAddr1, bankAddr2, attr,lp);
	 bankInfoCount++ ;
	 break;
      }
   }
   return (bank);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS DeleteBank(U8 bankNo) {
S16 lp;

   if (bankNo > MAX_BANK) return(!GOOD);
   else {
      bankInfo[bankNo].status = NO_BANK;
      bankInfo[bankNo].bottom = BUTTOM_NULL;
      bankInfo[bankNo].top = TOP_NULL;
      for (lp = 0; lp < BLOCK_NO; lp++)
         bankInfo[bankNo].table[lp] = 0xff70;  // external
      bankInfoCount--;
   }
   return(GOOD);
}

VOID ClearEmptyBank(U8 lp) {
   S16 lp2,blockNo;

   blockNo = (S16) (bankSize/BLOCK_SIZE);
   for (lp2 = 0; lp2 < blockNo; lp2++)
      if ((bankInfo[lp].table[lp2] & 0xff00) != 0xff00)
	 break; // not external

   if (lp2 >= blockNo) {
      VirMapSet(0xff70,bankInfo[lp].top,bankInfo[lp].bottom);
      DeleteBank(lp); // All blocks of bank are external or ER.
   }
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrSw2Fw(U16 attr) {
U16 tempAttr = 0;

   //Bank No.
   // tempAttr = (attr & 0x18) << 5; // copy bit 3,4 to bit 8,9
   // Allocation Type.
   //if ((attr & 0x07) < 4) tempAttr += ((attr & 0x03) << 10);// copy bit 0,1 to 10,11
   //else tempAttr += 0xf00;

   //tempAttr += ((attr & 0x0002) << 7); // copy bit 1 to bit 8;  // I/E

   // Allocation Type.
   if ((attr & 0x07) < 4) {
      tempAttr += ((attr & 0x0001) << 9); // copy bit 0 to bit 9;  // R/W
      if (!(attr &0x0002)) tempAttr += (((attr & 0x0018) << 7) | 0xf070); // copy bit 3,4 to bit 10,11 //Bank No.
      else tempAttr += 0xfd70;
   }
   else tempAttr = 0xef70; //G
   return(tempAttr);
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
U16 AttrFw2Sw(U16 attr) {
U16 tempAttr = 0;

   // Bank No.
   // tempAttr = (attr & 0x300) >> 5; // copy bit8,9 to bit 3,4
   // Allocation Type.
   //if ((attr & 0xf00) == 0x0f00) tempAttr += 0x04;// copy bit 10,11 to 0,1
   //else tempAttr += (attr & 0x0c00) >> 10;

   // Allocation Type.
   if (!attr) return(tempAttr);  //for map initial attr=0 12/15/97
   if ((attr & 0xf000) == 0xf000) {
      tempAttr += ((attr & 0x0c00) >> 7); // copy bit 10,11 to bit 3,4 // Bank No.
      tempAttr += ((attr & 0x0200) >> 9); // copy bit 9 to bit 0;  // R/W
      tempAttr += ((attr & 0x0100) >> 7); // copy bit 8 to bit 1;  // I/E
   }
   else tempAttr = 0x0064;// G
   return(tempAttr);
}

STATUS CheckValidBank(MAP_INFO mapInfo,U8 bankNo) {
U16 startBlock,endBlock,lp;

  startBlock = (U16)((mapInfo.startAddr - bankInfo[bankNo].bottom) / BLOCK_SIZE);
  endBlock = (U16) (((mapInfo.endAddr - bankInfo[bankNo].bottom) / BLOCK_SIZE) + 1);
  for (lp =startBlock; lp < endBlock; lp++)
     if ((bankInfo[bankNo].table[lp] & 0xff00) != 0xff00)
	return(FALSE);
  return(TRUE);
}

STATUS PRIVATE FindEqualMap(U32 startAddr,U32 endAddr,U16 attr,U8 mapTablePtr) {
   //U8 spaceType;

   //spaceType = (U8) (attr & 0xff);
   if ( (mapTable[mapTablePtr].startAddr == startAddr) &&
	(mapTable[mapTablePtr].endAddr == endAddr) )
	//&& ( (U8)(mapTable[mapTablePtr].attr & 0xff) == spaceType) )
      return(TRUE);
   return(FALSE);
}

VOID setTableInfoBankAddr(MAP_TABLE_INFO *mapTableInfo, SLICE *slice,U8 bankNo) {

    mapTableInfo->bank[bankNo] = 1;
    mapTableInfo->bankSA[bankNo] = slice->addr1;
    mapTableInfo->bankEA[bankNo] = slice->addr2;
}

/**************************************************************************
**
** Name :
**
** Function
**
**    Input  :
**
**    Output :
**
** Notes:
**
**************************************************************************/
STATUS MapGuard(MAP_TABLE_INFO *mapInfo) {
   if (((mapInfo->endAddr+1)%ER_BOUND_SIZE) || (mapInfo->startAddr % ER_BOUND_SIZE))
      return(ERROR_BOUN);
   else {
      mapInfo->bank[ER_BANK] = 1;
      mapInfo->bankSA[ER_BANK] = mapInfo->startAddr;
      mapInfo->bankEA[ER_BANK] = mapInfo->endAddr;
      RealMapSet(mapInfo->attr,mapInfo->endAddr,mapInfo->startAddr);
      addMapTable(mapInfo);
      return(GOOD);
   }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID CopyBankInfo(BANK_INFO *desBank, BANK_INFO *sourceBank) {
U8 lp;
U16 lp2;
   for (lp = 0; lp < MAX_BANK; lp++) {
      desBank[lp].top = sourceBank[lp].top;
      desBank[lp].bottom = sourceBank[lp].bottom;
      desBank[lp].status = sourceBank[lp].status;
      for ( lp2 = 0; lp2 < BLOCK_NO; lp2++)
	 desBank[lp].table[lp2] = sourceBank[lp].table[lp2];
   }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS addMapTable(MAP_TABLE_INFO *mapInfo) {
   U8 lp,lp2;

   for ( lp=0 ; lp < MAX_MAP ; lp++)
      if (mapTable[lp].attr == 0) break;
   mapTable[lp].startAddr = mapInfo->startAddr;
   mapTable[lp].endAddr = mapInfo->endAddr;
   mapTable[lp].attr = mapInfo->attr;
   for ( lp2=0 ; lp2 < MAX_BANK+1 ; lp2++) {
      mapTable[lp].bank[lp2] = mapInfo->bank[lp2];
      if (mapInfo->bank[lp2] != 0) {
	 mapTable[lp].bankSA[lp2] = mapInfo->bankSA[lp2];
	 mapTable[lp].bankEA[lp2] = mapInfo->bankEA[lp2];
      }
   }
   return GOOD;
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID clearMapEntry(U8 entryPtr) {
   U8 lp;

   mapTable[entryPtr].startAddr = BUTTOM_NULL;
   mapTable[entryPtr].endAddr = TOP_NULL;
   mapTable[entryPtr].attr = 0;
   for (lp=0 ; lp < MAX_BANK+1 ; lp++)
      if ( mapTable[entryPtr].bank[lp] != 0) {
	 mapTable[entryPtr].bank[lp] = 0;
	 mapTable[entryPtr].bankSA[lp] = BUTTOM_NULL;
	 mapTable[entryPtr].bankEA[lp] = TOP_NULL;
      }
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
BOOLEAN checkMapEqu(U32 startAddr, U32 endAddr, U16 attr) {
   U8 lp;

   for (lp=0 ; lp < MAX_MAP ; lp++)
      if ( mapTable[lp].startAddr == startAddr &&
           mapTable[lp].endAddr == endAddr &&
           mapTable[lp].attr == attr)
	 return (TRUE);
   return (FALSE);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS DeleteMap(MAP_TABLE_INFO *mapInfo) {
   U8 mapPtr, bankPtr;

   for (mapPtr = 0; mapPtr < MAX_MAP ; mapPtr++) {
      if ( FindEqualMap(mapInfo->startAddr,mapInfo->endAddr,mapInfo->attr,mapPtr) )
	 break;
   }
   if (mapPtr >= MAX_MAP) return(BANK_FULL);
   for ( bankPtr=0 ; bankPtr < MAX_BANK+1 ; bankPtr++) {
      if (mapTable[mapPtr].bank[bankPtr] != 0) {
	 SetMapTable(mapTable[mapPtr].bankSA[bankPtr],mapTable[mapPtr].bankEA[bankPtr],
		     mapInfo->attr,bankPtr);
         //if (bankPtr < MAX_BANK)
         //   ClearEmptyBank(bankPtr);//Eric 7/8/98
      }
   }
   clearMapEntry(mapPtr);
   L_FwMapSet();
   return(GOOD);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
VOID L_FwMapSet(VOID) {
U8 lp;
   for (lp = 0; lp < mapSetCount; lp++)
      RealMapSet(mapSetInfo[lp].attr,mapSetInfo[lp].endAddr,
		 mapSetInfo[lp].startAddr);
}

/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
/*STATUS MapEorER(MAP_TABLE_INFO *mapTableInfo) {
   U8 NumOfBank,lp,lp2;
   U8 bankNo[MAX_BANK],oneBankNo;
   SLICE sliceER[2];
   BANK_INFO bankCopy[MAX_BANK];
   MAP_INFO mapInfo;

   if ( (mapTableInfo->attr & 0x0066) != 0x0066 )
      return (FALSE);

   CopyBankInfo(bankCopy, bankInfo);

   for (lp = 0; lp < 2; lp++) {
      sliceER[lp].addr1 = BUTTOM_NULL;
      sliceER[lp].addr2 = TOP_NULL;
   }

   //slice address to sliceER[0],mapStart--mapEnd,sliceER[1]
   //sliceER[0] and sliceER[1] are not full bank boundary
   // the range from mapStart to mapEnd is full bank boundary

   sliceER[0].addr1 = mapTableInfo->startAddr;
   sliceER[0].addr2 = ((mapTableInfo->startAddr/ER_BOUND_SIZE)+1)*ER_BOUND_SIZE - 1;
   if (mapTableInfo->endAddr < sliceER[0].addr2)
      return (FALSE);

   if ((sliceER[0].addr1 % ER_BOUND_SIZE) == 0) {
      while (mapTableInfo->endAddr >= (sliceER[0].addr2+ER_BOUND_SIZE)) {
	 sliceER[0].addr2 += ER_BOUND_SIZE;
      }
      sliceER[0].addr2 = mapTableInfo->endAddr;
   }

   if (mapTableInfo->endAddr == sliceER[0].addr2) {
      if ((sliceER[0].addr1 % ER_BOUND_SIZE) != 0)
         return (FALSE);
   } else {
     sliceER[1].addr1 = sliceER[0].addr2 + 1;
     sliceER[1].addr2 = mapTableInfo->endAddr ;
   }

   if ((sliceER[0].addr1 % ER_BOUND_SIZE) == 0) {
      SetMapTable(mapTableInfo->startAddr,mapTableInfo->endAddr,mapTableInfo->attr,ER_BANK);
      setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),ER_BANK);
   }
   else if ( (sliceER[1].addr2-sliceER[1].addr1+1) >= ER_BOUND_SIZE) {
      mapInfo.startAddr = slice[lp].addr1;
      mapInfo.endAddr = slice[lp].addr2;
      mapInfo.attr = mapTableInfo->attr;
      NumOfBank = SearchBank(sliceER[0].addr1,bankNo);
      if (NumOfBank == 0) { // not found
         oneBankNo = CreateBank(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr);
	 if (oneBankNo == 0xFF) {
            if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
            return(BANK_FULL);
         }
	 setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),oneBankNo);
      } else {
	 for (lp2 = 0; lp2 < NumOfBank; lp2++) {
            if (CheckValidBank(mapInfo,bankNo[lp2])) {
	       SetMapTable(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr,bankNo[lp2]);
	       setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),bankNo[lp2]);
               break;
	    }
         }
         if (lp2 >= NumOfBank) {
	    oneBankNo = CreateBank(sliceER[0].addr1,sliceER[0].addr2,mapTableInfo->attr);
            if (oneBankNo == 0xFF) {
	       if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
	       return(BANK_FULL);
            }
	    setTableInfoBankAddr(mapTableInfo,&(sliceER[0]),oneBankNo);
         }
      }
      SetMapTable(sliceER[1].addr1,sliceER[1].addr2,mapTableInfo->attr,ER_BANK);
      setTableInfoBankAddr(mapTableInfo,&(sliceER[1]),ER_BANK);
   } else return (FALSE);

   addMapTable(mapTableInfo);
   L_FwMapSet();
   return(TRUE);
}
*/
/**************************************************************************
** Name :
** Function
**    Input  :
**    Output :
** Notes:
**************************************************************************/
STATUS SetMap(U32 startAddr, U32 endAddr, U16 attr) {
   STATUS status;
   U8 NumOfBank,lp,lp2,allocType;
   U8 bankNo[MAX_BANK],oneBankNo;
   MAP_INFO mapInfo;
   MAP_TABLE_INFO mapTableInfo;
   BANK_INFO bankCopy[MAX_BANK];

   if (bankSize == 0) return(NO_EMM);
   attr = AttrSw2Fw(attr);

   if (checkMapEqu(startAddr,endAddr,attr)) return(GOOD);
   
   allocType = (U8)((attr & 0xff00) >> 8); // attribute
   mapTableInfo.startAddr = startAddr;
   mapTableInfo.endAddr = endAddr;
   mapTableInfo.attr = attr;
   for ( lp=0 ; lp < MAX_BANK+1 ; lp++) {
      mapTableInfo.bank[lp] = 0;
      mapTableInfo.bankSA[lp] = BUTTOM_NULL;
      mapTableInfo.bankEA[lp] = TOP_NULL;
   }
   mapSetCount = 0;
   if ((startAddr == 0) && (endAddr == 0)) return(InitMapTable());
   else {
      if (((startAddr >= INTER_REG_ADDR) || (endAddr >= INTER_REG_ADDR)) && (attr & 0x0020))
          return(MICE_LIMIT); //the address is in internal register(0xfffff000,0xffffffff),and attribute is SD.

      switch(allocType) {
         case EXTERNAL_MAP : return(DeleteMap(&mapTableInfo));
	 case GUARD_MAP : return(MapGuard(&mapTableInfo));
	 default : break;
      }

      if (status = SliceNewAddr(startAddr,endAddr))
            return(ADDR_RANGE_OVERFLOW);
      CopyBankInfo(bankCopy, bankInfo);
      lp = 0;
      while ((slice[lp].addr1 != BUTTOM_NULL) &&
             (slice[lp].addr2 != TOP_NULL) && (lp < 4)) {
         mapInfo.startAddr = slice[lp].addr1;
         mapInfo.endAddr = slice[lp].addr2;
         NumOfBank = SearchBank(slice[lp].addr1,bankNo);
         if (NumOfBank == 0) { // not found
            oneBankNo = CreateBank(slice[lp].addr1,slice[lp].addr2,attr);
            if (oneBankNo == 0xFF) {
	       if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
	       return(BANK_FULL);
	    }
	    setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),oneBankNo);
	 } else {
	    for (lp2 = 0; lp2 < NumOfBank; lp2++) {
	       if (CheckValidBank(mapInfo,bankNo[lp2])) {
		  SetMapTable(slice[lp].addr1,slice[lp].addr2,attr,bankNo[lp2]);
                  setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),bankNo[lp2]);
		  break;
               }
	    }
            if (lp2 >= NumOfBank) {
               oneBankNo = CreateBank(slice[lp].addr1,slice[lp].addr2,attr);
               if (oneBankNo == 0xFF) {
                  if (lp != 0) CopyBankInfo(bankInfo, bankCopy);
                  return(BANK_FULL);
               }
               setTableInfoBankAddr(&mapTableInfo,&(slice[lp]),oneBankNo);
	    }
         }
	 lp++;
      }
      addMapTable(&mapTableInfo);
      L_FwMapSet();
   }
   return(GOOD);
}

/***************************************************************************
**
**  EmuSetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**               "addr" is the starting address of this mapping memory.
**               It should be on the address boundary of 64KB.
**               "map" is the mapping of current 64K memory bank.
**               If memoryresolution is 4KB and map is 0xFFCF. The emulation
**               memory will be mapped as two segments. 0..0x3fff and
**               0x6000..0xffff.
**               The definition of "attr" as below:
**               BIT Meaning
**               --- --------------------------------------------
**                0  "0" = Internal Memory, "1" = External memory
**                1  "0" = Read/Write, "1" = Read-Only
**                2  "0" = Exist memory, "1" = Gard memory
**                3  Reserved
**                4  Segments(with bit 5) -- "00" = ES, "01" = CS
**                5  Segments(with bit 4) -- "10" = SS, "11" = DS
**                6  Reserved
**                7  Reserved
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetMap(U32 addr1, U32 addr2, U16 attrib) {
STATUS status;

   status = GOOD;
   StopRun();
   status = SetMap(addr1,addr2,attrib);
   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

//VOID EmuSetMap(U32 addr1, U32 addr2, U16 attrib) {
//STATUS status;
//U16 addrStart,addrEnd;
//
//   if ((addr1 == 0) && (addr2 == 0)) {
//      mapBuf[0] = MAP_END_RECORD; /* set map data to map buffer */
//      status = InitMapCtr();
//   }
//   else {
//      addrStart = (U16)(addr1 >> 11);
//      addrEnd = (U16)(addr2 >> 11);
//      status = MapSet(attrib,addrEnd,addrStart);
//   }
//
//   status = status & 0x00ff;
//   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
//}


/***************************************************************************
**
**  EmuGetMap
**
**  Description: Michelle setup-group routine, to set the emulation
**               memory map.
**               "addr" is the starting address of this mapping memory.
**               It should be on the address boundary of 64KB.
**               "map" is the mapping of current 64K memory bank.
**               If memoryresolution is 4KB and map is 0xFFCF. The emulation
**               memory will be mapped as two segments. 0..0x3fff and
**               0x6000..0xffff.
**               The definition of "attr" as below:
**               BIT Meaning
**               --- --------------------------------------------
**                0  "0" = Internal Memory, "1" = External memory
**                1  "0" = Read/Write, "1" = Read-Only
**                2  "0" = Exist memory, "1" = Gard memory
**                3  Reserved
**                4  Segments(with bit 5) -- "00" = ES, "01" = CS
**                5  Segments(with bit 4) -- "10" = SS, "11" = DS
**                6  Reserved
**                7  Reserved
**
**  Parameters:
**     input:
**        "addr" -- unsigned long,
**        "map"  -- unsigned int,
**        "attr" -- unsigned int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMap(VOID) {
U16 count, tempEnd;
//U32 endLongData;

   StopRun();
   OutputStatus(GOOD,OFF);
   for (count=0; count<MAX_MAP; count++) {
      mapData.addr1 = mapTable[count].startAddr;
      mapData.addr2 = mapTable[count].endAddr;
      mapData.attr  = AttrFw2Sw(mapTable[count].attr);
      OutData(10,&mapData,OFF);
   }
   tempEnd = 0xaa55;
   OutData(2,&tempEnd,OFF);
   OutEnd();

}

/***************************************************************************
**
**  EmuGetMapSize
**
**  Description:
**
**  Parameters:
**     input:
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetMapSize(VOID) {
U16 emmSize;
U8 dataLen;

   OutputStatus(GOOD,OFF);
   AccessIceFlag(READ_ONE,EMM_SIZE,&emmSize);
   dataLen = 2;
   OutData(dataLen,&emmSize,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuSetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetControl(U16 signals) {

   AccessSignals(WRITE_ALL,noUse,&signals);
   OutputStatus(GOOD,ON);
}


/***************************************************************************
**
**  EmuGetControl
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        signals -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetControl(VOID) {
U16 signals;
U8 dataLen;

   AccessSignals(READ_ALL,noUse,&signals);
   OutputStatus(GOOD,OFF);
   dataLen = sizeof(signals);
   OutData(dataLen,&signals,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuSetTimer
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        timers -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuSetTimer(U16 timers) {

   AccessTimers(WRITE_ALL,noUse,&timers);
   OutputStatus(GOOD,ON);
}


/***************************************************************************
**
**  EmuGetTimer
**
**  Description: Michelle setup-group routine,
**
**  Parameters:
**     input:
**        timers -- U16
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
****************************************************************************/
VOID EmuGetTimer(VOID) {
U16 timers;
U8 dataLen;

   AccessTimers(READ_ALL,noUse,&timers);
   OutputStatus(GOOD,OFF);
   dataLen = sizeof(timers);
   OutData(dataLen,&timers,ON);
   OutEnd();

}

/***************************************************************************
**
**  EmuGetStatus
**
**  Description: Michelle setup-group routine, to read the current setting
**               of the Michelle.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." be Normal return
**        "Emulator Fatal Error" for HW error
**        return data -- "length" + "events" + "qualify: + "trigger" +
**                       "MAP" + control signals" +
**                       "ready setting"
**
**                     * "ID" is a issue to be discussed later
**
****************************************************************************/
VOID EmuGetStatus(VOID) {
U16 flyFlag;

  AccessIceFlag(READ_ONE,FLYING,&flyFlag);
  if (flyFlag) OutputStatus(CPU_FLY,ON);
  else if (!CheckEpStop()) OutputStatus(CPU_RUN,ON);
  else OutputStatus(BROKE,ON);
}

/***************************************************************************
**
**  EmuGetID
**
**  Description: Michelle setup-group routine, to get the emulator's ID to
**               make sure that driver program has connected to a correct
**               emulator.
**
**  Parameters:
**     input:
**        none
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**        return data -- "length" + "ID"
**
****************************************************************************/
VOID EmuGetID(VOID) {
U8 dataLen;

   OutputStatus(GOOD,OFF);
   dataLen = MAX_ID;
   // get probe,trace,cp board info from llfw and save to idData
   OutData(dataLen,idData,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuGetMode()
**
**  Description: Michelle setup-group routine, to select the Mode
**               "src" = 0 --> command mode
**               "src" = 1 --> queue status mode
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetMode(VOID) {
U8 dataLen,modeFlag;
U16 intReg[8],phyCpu;

   AccessIceFlag(READ_ONE,PHY_CPU_TYPE,&phyCpu);
   switch (phyCpu) {
      case XL_EA :
         //modeFlag = ModeStatus();
         break;
      case EB :
	 modeFlag = 0;
         break;
      case EC :
         modeFlag = 0;
         break;
      default :
         //modeFlag = ModeStatus();
         break;
   }
   if ((cpuFlag & 0x0f) == 1) { //cpu = 801xxXL
      IntRegister(0xfff0,0,01,intReg);
      if ((intReg[0] & 0x7ffc) != 0) modeFlag |= 0x02; // bit2 equal 1
   }
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = 1;
   OutData(dataLen,&modeFlag,ON);
   OutEnd();

}


/****************************************************************************
**
**  EmuSetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetReady(U16 readyFlag) {
U16 signals;
   AccessIceFlag(WRITE_ONE,READY_FLAG,&readyFlag);
   AccessSignals(READ_ALL,noUse,&signals);
   if (readyFlag == 0) {
      AccessSignals(WRITE_ONE,S_RDEXT,&C_OFF);
   } //ReadyInternal();
   else {
      AccessSignals(WRITE_ONE,S_RDEXT,&signals);
   } //ReadyExternal();
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetReady()
**
**  Description: Michelle setup-group routine, to select the READY signal
**               source.
**               "src" = 0 --> Internal
**               "src" = 1 --> External
**
**  Parameters:
**     input:
**        src --  (Or int)
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetReady(VOID) {
STATUS status;
U16 readyFlag;
U8 dataLen;

   status = GOOD;
   AccessIceFlag(READ_ONE,READY_FLAG,&readyFlag);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(readyFlag);
   OutData(dataLen,&readyFlag,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetAccessSize(SIZE size) {
STATUS status;

   status = GOOD;
   status = AccessIceFlag(WRITE_ONE,SIZE_FLAG,&size);

   OutputStatus(status,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetAccessSize()
**
**  Description: Michelle setup-group routine, to set the bus access size.
**               "size" = 0 --> Byte-access (8-Bit),
**               "size" = 1 --> Byte-access (16-Bit),
**               "size" = 2 --> Byte-access (32-Bit)
**
**  Parameters:
**     input:
**        size -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetAccessSize(VOID) {
STATUS status;
SIZE size;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,SIZE_FLAG,&size);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(size);
   OutData(dataLen,&size,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
/*VOID EmuSetWait(U16 count) {

   AccessIceFlag(WRITE_ONE,WAIT_FLAG,&count);

   WaitStates(count);

   OutputStatus(GOOD,ON);// Before next data add to outputStream. len - 1 

}
*/

/****************************************************************************
**
**  EmuGetWait()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetWait(VOID) {
U8 dataLen;
U16 count;

   AccessIceFlag(READ_ONE,WAIT_FLAG,&count);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(count);
   OutData(dataLen,&count,ON);
   OutEnd();

}

/****************************************************************************
**
**  EmuSetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetSync(U16 state) {

   AccessIceFlag(WRITE_ONE,SYNC_FLAG,&state);
   switch (state) {
      case 0:
         DisableSSI();
         AccessIceFlag(WRITE_ONE,SYNC_IN,&C_OFF);
        break;
      case 1:
         EnableSSI();
	 AccessIceFlag(WRITE_ONE,SYNC_IN,&C_ON);
        break;
      case 2:
         DisableSSO();
        break;
      case 3:
         EnableSSO();
        break;
   }
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}

/****************************************************************************
**
**  EmuGetSync()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetSync(VOID) {
U16 sync;
U8 dataLen;

   AccessIceFlag(READ_ONE,SYNC_FLAG,&sync);
   OutputStatus(GOOD,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(sync);
   OutData(dataLen,&sync,ON);
   OutEnd();
}

/****************************************************************************
**
**  EmuSetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuSetVerify(U16 verifyFlag) {

   AccessIceFlag(WRITE_ONE,VERIFY_FLAG,&verifyFlag);

   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */
}

/****************************************************************************
**
**  EmuGetVerify()
**
**  Description: Michelle setup-group routine, to insert wait state for
**               Michelle. Insert wait state, "count" id the number of
**               clocks of the wait-state.
**
**  Parameters:
**     input:
**        count -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuGetVerify(VOID) {
STATUS status;
U16 verifyFlag;
U8 dataLen;

   status = GOOD;
   status = AccessIceFlag(READ_ONE,VERIFY_FLAG,&verifyFlag);
   OutputStatus(status,OFF);/* Before next data add to outputStream. len - 1 */
   dataLen = sizeof(verifyFlag);
   OutData(dataLen,&verifyFlag,ON);
   OutEnd();
}

/***************************************************************************
**
**  EmuEpTimer
**
**  Description:
**
**
**
**
**  Parameters:
**     input:
**        flag -- int
**
**     output:
**        return status code(error-code) in to the output processor.
**        "O.K." -- Normal return
**        "Fatal error on emulator" -- Fatal HW error; check status-code
**
*****************************************************************************/
VOID EmuEpTimer(U16 timerFlag) {
U16 baseReg[MAX_BASE_REG_NO],timerValue;

   AccessBaseReg(READ_ALL,noUse,baseReg);
   switch (timerFlag) {
      case 0 :
         AccessInternalReg(READ_ONE,T0CON,&timerValue);
	 baseReg[R_TIMER0] = timerValue;
         break;
      case 1 :
	 AccessInternalReg(READ_ONE,T1CON,&timerValue);
         baseReg[R_TIMER1] = timerValue;
         break;
      case 2 :
         AccessInternalReg(READ_ONE,T2CON,&timerValue);
         baseReg[R_TIMER2] = timerValue;
	 break;
      default : // 0
         break;

   }
   AccessBaseReg(WRITE_ALL,noUse,baseReg);
   stpTimFlg = timerFlag;
   AccessIceFlag (WRITE_ONE,TIMER_FLAG,&timerFlag);
   OutputStatus(GOOD,ON);/* Before next data add to outputStream. len - 1 */

}

//Eric 3/16/98
VOID EmuGetHWMode() {
   U8 tmp;
   OutputStatus(GOOD,OFF);
   tmp = 2;//Am186 type=2
   OutData(1,&tmp,ON);
   OutEnd();
}

VOID EmuSwitchFlash() {
          FlashRomMp();
          OutputStatus(GOOD,OFF);
      OutEnd();
}

VOID EmuWriteBlaster(U8 setting,U8 data) {
   WriteByteBlaster(setting,data);
   OutputStatus(GOOD,ON);
}

VOID EmuReadBlaster(VOID) {
   U8 data ;
   U8 dataLen;

   data = ReadByteBlaster();
   OutputStatus(GOOD,OFF);
   dataLen = sizeof(data);
   OutData(dataLen,&data,ON);
   OutEnd();
}

/******************************** E O F ***********************************/

