/* Launches Presenters

   Logically, the presenters should be class variables here rather than
   global variables.  We should do this move when we get time to do so.
*/!!

inherit(Object, #PreLauncher, nil, 2, nil)!!

setClassVars(PreLauncher, #($cliRegistered /* boolean flag */
$windowsMenu /* shared menu */
$presenterDict /* launchable presenters */
$inBoot?          /* during boot process? */
$startupScriptFileName
$windowRestoreCompleted
    /* true when all saved windows have been restored */))!!

now(class(PreLauncher))!!

/* 8/2/1996 9:24 */
/* Launch off the requested presenter */
Def launchSpaPresenter(self, rect, showVal )
{

  if (exists(SpaPresenter) cand TheSpaPresenter) then
    Call BringWindowToTop(getHWnd(TheSpaPresenter));
    show( TheSpaPresenter, SW_SHOWNORMAL);
  else
    /* Open the Spa Presenter; retrieve location and show status if
       they are not already specified. */
    TheSpaPresenter :=
      openWithPosAndState(SpaPresenter,
        rect cor getPos(self, "TheSpaPresenter"),
        getShowState(self, "TheSpaPresenter") cor showVal);
  endif;
}!!

/* 4/19/1994 12:08 - PUBLIC
  Cycles from one active window to the next.
  Multiple memory window enhancements added. 
*/
Def nextWindow(self | actWind actvWindows presenterDict curPos nextPos
      index memObject) {
  /* find out which window is active */
  actWind := Call GetActiveWindow();
  actvWindows := new(OrderedCollection, 1);
  presenterDict := getPresenterDict(self);

  /* search the presenter dictionary and match handle to presenter */
  keysDo(presenterDict,
  {using(key | info)
    /* find out which presenters are open */
    info := at(Actor, key);
    /* special processing for multiple memory windows */
    if (key = #TheMemoryPresenter)
      index := 0;
      loop
      while index < memoryWindowMax(MultMem)
      begin
        if (memObject := memoryWindowObject(MultMem, index))
          if (memObject cand (Call IsIconic(getHWnd(memObject)) = 0))
            /* window is not iconized */
            /* get the handle to the open windows */
            add(actvWindows, getHWnd(memObject));
          endif;
        endif;
        index := index + 1;
      endLoop;
    else
      /* 04/18/94 - Nghia
      ** only call if info is not nil - Fixed GPF bug 
      */
      if ((info <> nil) cand (Call IsIconic(getHWnd(info)) = 0))
        /* window is not iconized */
        /* get the handle to the open windows */
        add(actvWindows, getHWnd(info));
      endif;
    endif;
  });
  
  /* get the entry number for the currently active window (if its */
  /* handle matches one of the open windows (safety)) */
  if actWind in actvWindows
    curPos := find(actvWindows,actWind);
  else
    curPos := 0;
  endif;
  /* if the active window is the last element in the set then */
  /* set the next window to display to the first element in the set */
  if curPos = (size(actvWindows)-1)
    nextPos := 0;
  else
    nextPos := curPos + 1;
  endif;
  Call BringWindowToTop(actvWindows[nextPos]);
}

!!

/* special processing for opening saved mult memory windows. */
Def openAllSavedMemWindows(self |
      tmpRect showVal tuple launchMethod index label symbol)
{
  tuple := at(getPresenterDict(PreLauncher), #TheMemoryPresenter);
  launchMethod := tuple[3];
  
  index := 0;
  loop
  while index < memoryWindowMax(MultMem)
  begin
    label := memoryWindowLabel(MultMem, index);
    symbol := asSymbol(label);
    tmpRect := getIniWindowPos(self, symbol);
    showVal := getIniWindowState(self, symbol);
      
    /* if tmpRect, window has saved coordinates, and should be opened. */
    if (tmpRect)
      perform(PreLauncher, tmpRect, showVal, launchMethod);
    endif;
    index := index + 1;
  endLoop;
  
  Call BringWindowToTop(getHWnd(TheToolBarPresenter));
}
!!

/* special processing for saving multiple memory presenter's position. */
Def saveAllMemPresenterPos(self | index memObject)
{
  index := 0;
  loop
  while index < memoryWindowMax(MultMem)
  begin
    memObject := memoryWindowObject(MultMem, index);
    if (memObject)
      saveWindowPos(memObject, memoryWindowLabel(MultMem, index));
    else
      /* don't need to test for toolbar; only memory windows here. */
      clearLayoutEntry(MPresenter, index);
    endif;
    index := index + 1;
  endLoop;

  ^#ok
}
!!

/* Return true if presenter symbol passed in is the memory presenter. */
Def memoryPresenter?(self, presenterSymbol)
{
  ^(presenterSymbol = #TheMemoryPresenter);
}
!!

/* Special processing for the multiple memory windows. */
Def closeAllMemPresenters(self | saveOnExit index memObject)
{
  saveOnExit := getIniSaveLayoutOnExit(PreLauncher);
  
  index := 0;
  loop
  while index < memoryWindowMax(MultMem)
  begin
    memObject := memoryWindowObject(MultMem, index);
    if (memObject)
      if (saveOnExit = 1)
        saveWindowPos(memObject, memoryWindowLabel(MultMem, index));
      endif;
      close(memObject);
    else
      clearLayoutEntry(MPresenter, index);
    endif;
    index := index + 1;
  endLoop;

  nilifyWindowsMenu(self);

  ^#ok
}
!!

/* Show presenters using  "ShowWindow"
 *  get the state they were in before we hid them.
 *  $presenterDict[4] is showValue
 */
Def showMemPresenter(self, presenter, index | hwnd iconic showVal)
{
  showVal := getShowStatus(MultMem, index);
  if (showVal = nil)
    showVal := 1;      /* force as visible */
    setShowStatus(MultMem, index, showVal);
  endif;

  hwnd := getHWnd(presenter);
  if (Call IsWindow(hwnd) <> 0)
    show(presenter, showVal);  /* was showval */
    invalidate(presenter);
  endif;

  ^0;
}
!!

/* hide a presenters using  "ShowWindow"
 *  keep a copy of what state they were in before we hid them.
 *  setShowState value replaces $presenterDict[4] for mult memory windows.
 */
Def hideMemPresenter(self, presenter, index | hwnd iconic )
{
  hwnd := hWnd(presenter);
  if (Call IsWindow(hwnd) <> 0)
    iconic :=  Call IsIconic(hwnd);
    if (iconic<>0)
      setShowStatus(MultMem, index, SW_SHOWMINIMIZED);
    else
      setShowStatus(MultMem, index, SW_SHOWNORMAL);
    endif;

    show(presenter, 0);
  endif;
   
  ^0;
}
!!

/* do "ShowWindow" over all memory presenters */
Def hideAllMemPresenters(self | index memObject)
{
  index := 0;
  loop
  while index < memoryWindowMax(MultMem)
  begin
    memObject := memoryWindowObject(MultMem, index);
    if (memObject)
      hideMemPresenter(self, memObject, index);
    endif;
    index := index + 1;
  endLoop;
}
!!

/* do "ShowWindow" over all memory presenters */
Def showAllMemPresenters(self | index memObject)
{
  index := 0;
  loop
  while index < memoryWindowMax(MultMem)
  begin
    memObject := memoryWindowObject(MultMem, index);
    if (memObject)
      showMemPresenter(self, memObject, index);
    endif;
    index := index + 1;
  endLoop;
}
!!

/* return show state from profile file.  Returns SW_MINIMIZE if the
   window should be minimized, and SW_SHOWNORMAL otherwise. */
Def getShowState(self, presenter | iniInfo showValue)
{
  iniInfo :=
    getProfileEntry(TheProfileInfoObj, "DefaultLayout", presenter, ", ");
  if not(iniInfo)
    ^SW_SHOWNORMAL;
  endif;  
  showValue := asInt(iniInfo[4],10); 
  if (showValue = 1)
    ^SW_MINIMIZE;
  endif;
  
  ^SW_SHOWNORMAL;
}
!!

/* return position rectangle from the profile file, or nil if not found. */
Def getPos(self, presenter | iniInfo iniRect)
{
  iniInfo :=
    getProfileEntry(TheProfileInfoObj, "DefaultLayout", presenter, ", ");

  if not(iniInfo)
    ^nil;
  endif;

  /* there is an entry for the window location */
  iniRect := new(Rect);
  iniRect := init(iniRect, asInt(iniInfo[0],10), asInt(iniInfo[1],10),
                asInt(iniInfo[2],10), asInt(iniInfo[3],10));

  ^iniRect;
}
!!

/* PRIVATE
   Save Shell startup file name.  No value implies there is no
   startup file.
*/
Def startupFileNameClear(self)
{
  $startupScriptFileName := nil;
}!!

/* PRIVATE
   Save Shell startup file name.  No value implies there is no
   startup file.
*/
Def startupFileNameGet(self)
{
  ^$startupScriptFileName;
}!!

/* PRIVATE
   Save Shell startup file name.  No value implies there is no
   startup file.
*/
Def startupFileNameSave(self, fileString)
{
  $startupScriptFileName := fileString;
}!!

/* This is code factored from openButtonBarPresenter to work around
   3500 byte limit for actor methods */
Def backOutOfPV(self)
{
  if (CLIULibraryClass$Inst) then
     f1unhook(CLIULibraryClass$Inst);
     closeLogin(CLIULibraryClass$Inst);
  endif;
  removeAllLibs(ProvidedLibs);
  setOkToClose(TheApp:late);
  ^nil;
}
!!

/* do "ShowWindow" over all presenters */
Def showAllPresenters(self)
{
  do(presenterNames(self),
  {using(sym | presenter)
    if memoryPresenter?(self, sym)
      showAllMemPresenters(self);
    else
      if presenter := at(Actor, sym)
        if not(sym = #TheToolBarPresenter)
          showPresenter(self, presenter, sym);
        endif;
      endif;
    endif;
  });

  ^#ok
}
!!

/* do "ShowWindow" over all presenters */
Def hideAllPresenters(self)
{
  do(presenterNames(self),
  {using(sym | presenter)
    if memoryPresenter?(self, sym)
      hideAllMemPresenters(self);
    else
      if presenter := at(Actor, sym)
        if not(sym = #TheToolBarPresenter)
          hidePresenter(self, presenter, sym);
        endif;
      endif;
    endif;
  });

  ^#ok
}
!!

/* Show presenters using  "ShowWindow"
 *  get the state they were in before we hid them.
 *  $presenterDict[4] is showValue
 */
Def showPresenter(self presenter  preSym | dictTuple hwnd iconic showVal  )
{
     dictTuple := at($presenterDict,  preSym);
     showVal := dictTuple[4];    /* param for ShowWindow call */
      if(showVal=nil)
         showVal := 1;      /* force as visible */
         dictTuple[4] := showVal;
         put($presenterDict, dictTuple,  preSym);
     endif;

     if not(preSym = #TheToolBarPresenter) then
        hwnd := getHWnd(presenter);
        if( Call IsWindow(hwnd) <> 0)
            show(presenter , showVal);  /* was showval */
            invalidate(presenter);
        endif;
     endif;

^#ok
}
!!

/* hide a presenters using  "ShowWindow"
 *  keep a copy of what state they were in before we hid them.
 *  $presenterDict[4] is showValue
 */
Def hidePresenter(self presenter preSym | dictTuple hwnd iconic )
{
  dictTuple := at($presenterDict,  preSym);

  if not(presenter = #TheToolBarPresenter)
    hwnd := hWnd(presenter);
    if (Call IsWindow(hwnd) <> 0)
      iconic :=  Call IsIconic(hwnd);
      if (iconic<>0)
        dictTuple[4] := SW_SHOWMINIMIZED;
      else
        dictTuple[4] := SW_SHOWNORMAL;
      endif;

      put($presenterDict, dictTuple,  preSym);
      show(presenter, 0);
    endif;
  endif;
  
  ^#ok
}
!!

/* Launch off the requested presenter */
Def launchPeripheralPresenter(self, rect, showVal )
{

  if (exists(PeriphPresenter) cand ThePeripheralPresenter) then
    Call BringWindowToTop(getHWnd(ThePeripheralPresenter));
    show( ThePeripheralPresenter, SW_SHOWNORMAL);
  else
    /* Open the Peripheral Presenter; retrieve location and show status if
       they are not already specified. */
    ThePeripheralPresenter :=
      openWithPosAndState(PeriphPresenter,
        rect cor getPos(self, "ThePeripheralPresenter"),
        getShowState(self, "ThePeripheralPresenter") cor showVal);
  endif;
}
!!

/* PRIVATE */
Def $cliRegistered(self)
{
  ^$cliRegistered
}
!!

/* PUBLIC (Application Exit) */
Def closeAllPresenters(self | saveOnExit )
{
  saveOnExit := getIniSaveLayoutOnExit(PreLauncher);

  do(presenterNames(self),
  {using(sym | val)
    if memoryPresenter?(self, sym)
      closeAllMemPresenters(self);
    else
      if val := at(Actor, sym)
        if (saveOnExit=1)
          saveWindowPos(val, asString(sym));
        endif;
          close(val);
      else
        if not(sym = #TheToolBarPresenter)
          clearLayoutEntry(Window, asString(sym));
        endif;
      endif;
    endif;
  });

  nilifyWindowsMenu(self);

  ^#ok
}
!!

/* saving each presenter's position. */
Def saveAllPresenterPos(self)
{
  do(presenterNames(self),
  {using(sym | val)
    if memoryPresenter?(self, sym)
      saveAllMemPresenterPos(self);
    else
      if val := at(Actor, sym)
        saveWindowPos(val, asString(sym));
      else
        if not(sym = #TheToolBarPresenter)
          clearLayoutEntry(Window, asString(sym));
        endif;
      endif;
    endif;
  });

  ^#ok
}
!!


/* pull window state from ini file at position defined by preSym
 *  for example,  preSym = #TheMemoryPresenter.
 *   if no entry is present, do the sensible thing and display anyway.
 */
Def getIniWindowState(self preSym | iniInfo tmpRect temp  state)
{
  tmpRect := new(Rect);

    iniInfo := getProfileEntry(TheProfileInfoObj, "DefaultLayout",
               asString(preSym), ", ");

    if (iniInfo = nil) then
        ^SW_SHOWNORMAL;   /* display if not defined in ini file */
    endif;

      state := asInt(iniInfo[4],10);

      if(state=0)    /* not minimized, as defined by saveWindowPos */
        ^SW_SHOWNORMAL ;     /* 1 */
      else
        ^SW_SHOWMINIMIZED;    /* 2 */
      endif;

}
!!

/* pull window position from ini file at position defined by preSym
 *  for example,  preSym = #TheMemoryPresenter.
 *   if no entry is present, do the sensible thing pass nil.
 *   this causes create() to use CW_USEDEFAULT
 */
Def getIniWindowPos(self preSym | iniInfo tmpRect cwUseDefault)
{
    tmpRect := new(Rect);

    iniInfo := getProfileEntry(TheProfileInfoObj, "DefaultLayout",
               asString(preSym), ", ");


    if not(iniInfo = nil) then
      /* there is an entry for the window location */
       init(tmpRect, asInt(iniInfo[0],10), asInt(iniInfo[1],10),
                    asInt(iniInfo[2],10), asInt(iniInfo[3],10));
      ^tmpRect;
    else
     ^nil
    endif;

}
!!


/* val is an integer */
Def getIniSaveLayoutOnExit(self | saveLayout )
{
   saveLayout  := getIniValueNum(TheProfileInfoObj,
                      "ToolBarInfo", "SaveLayoutOnExit");
   if not(saveLayout)
      ^0;
   endif;
   ^saveLayout;
}
!!

/* val must be an integer */
Def putIniSaveLayoutOnExit(self, val)
{
  saveProfileEntry(TheProfileInfoObj,"ToolBarInfo","SaveLayoutOnExit", asString(val) );
}
!!

Def openSavedWindow(self preSym | dictTuple tmpRect temp launchMethod showVal)
{
  tmpRect := new(Rect);
  tmpRect := getIniWindowPos(self, preSym);
  showVal := getIniWindowState(self, preSym);

  dictTuple := at($presenterDict, preSym);

  launchMethod := dictTuple[3];
  if(tmpRect)
     perform(PreLauncher, tmpRect, showVal, launchMethod);
  endif;
}
!!

Def openAllSavedWindows(self | presDictionary)
{
  $windowRestoreCompleted := nil;
  
  presDictionary := getPresenterDict(PreLauncher);

  /* open all saved windows except the toolbar (already open) and status
     window (opened by toolbar); special processing for memory windows. */
  keysDo(presDictionary,
  {using(preSym)
     if not(preSym = #TheToolBarPresenter) cand
        not(preSym = #TheStatusPresenter)
          if memoryPresenter?(self, preSym)
            openAllSavedMemWindows(self)
          else
            openSavedWindow(self, preSym);
          endif;
     endif;
  });
  Call BringWindowToTop(getHWnd(TheToolBarPresenter));
  
  $windowRestoreCompleted := 0;
}
!!

/* PUBLIC  mod 19-Jul-93 to pass rect, showVal */
Def doAction(self, actionID | aVec)
{
  if (actionID >= 0x3F00) cand (actionID < (0x3F00+size($presenterDict)))
  then
    perform( self, nil, SW_NORMAL, presenterActionVec(self)[actionID-0x3f00] ) ;
  else
    ^nil
  endif;
}
!!

/* PUBLIC */
Def doActionOld(self, actionID | aVec)
{
  if (actionID >= 0x3F00) cand (actionID < (0x3F00+size($presenterDict)))
  then
    perform( self, presenterActionVec(self)[actionID-0x3f00] ) ;
  else
    ^nil
  endif;
}
!!


/* 7/10/1992 15:37 */
Def getPresenterDict(self)
{
  ^$presenterDict;
}
!!

/* PUBLIC */
Def inBoot?(self)
{ ^$inBoot? }!!

/* Launch off the requested presenter */
Def launchBreakPresenter(self, rect, showVal)
{
  if TheBreakpointPresenter then
    Call BringWindowToTop(getHWnd(TheBreakpointPresenter));
    show(TheBreakpointPresenter, SW_NORMAL);
  else
    /* Open the Breakpoint Presenter; retrieve location and show status if
       they are not already specified. */
    TheBreakpointPresenter := openWithPosAndState(BkptPresenter,
        rect cor getPos(self, "TheBreakpointPresenter"),
        showVal cor getShowState(self, "TheBreakpointPresenter"));
  endif;
}
!!



/* Launch off the requested presenter */
Def launchButtonBarPresenter(self, rect, showVal)
{
  if TheToolBarPresenter
  then
    Call BringWindowToTop(getHWnd(TheToolBarPresenter));
    show(TheToolBarPresenter, SW_NORMAL); /* redisplay */
  endif ;
}!!

/* Get the name of the disassembler DLL */
Def getDasmDLLName(self)
{
   if (getCpuType(ProcLibClass$Inst) = PROC_CPU_CPU16) then
      ^asString("dasm16.dll");
   else
     if (getCpuType(ProcLibClass$Inst) = PROC_CPU_80386 cor
         getCpuType(ProcLibClass$Inst) = PROC_CPU_80186)
       ^asString("dasm86.dll");
     endif;
     ^asString("dasm32.dll");    
   endif;
}!!

/* Launch off the requested presenter */
Def launchCLIPresenter(self, rect, showVal | cliServer)
{
  if TheShellPresenter
    Call BringWindowToTop(getHWnd(TheShellPresenter));
    show(TheShellPresenter, showVal); /* redisplay */
  else
    /* Open the Shell Presenter; retrieve location and show status if
       they are not already specified. */
    TheShellPresenter :=
      openWithPosAndState(CliPresenterWindow,
        rect cor getPos(self, "TheShellPresenter"),
        showVal cor getShowState(self, "TheShellPresenter"));

    if TheShellPresenter cand not($cliRegistered) then /* add CLI servers */
      cliServer := getCliServerObject(CliServer);
      /*                    library    svr prefix   dll name    */
      initDLLLib(cliServer, SpaLib,      "spa",    "spa.dll");
      initDLLLib(cliServer, ErrorTextLib, "err",  "errtext.dll"  );
      initDLLLib(cliServer, EvNoteLib,   "enlib",  "enlib.dll"   );
      initDLLLib(cliServer, MallocLib,   "malloc", "malloc.dll"  );
      initDLLLib(cliServer, SharedDataLib, "sds",  "sdserver.dll"   );
      initDLLLib(cliServer, SymbolLib,   "symbol", "symbl.dll"   );
      initDLLLib(cliServer, AddressLib,  "addr",   "addr.dll"    );
      initDLLLib(cliServer, MemServLib,  "mem",    "mem.dll"     );
      initDLLLib(cliServer, CpuLib,      "cpu",    "cpu.dll"     );
      initDLLLib(cliServer, LoaderLib,   "loader", "loader.dll"  );
      initDLLLib(cliServer, StackLib,    "stk",    "stkservr.dll");
      initDLLLib(cliServer, VarServLib,  "var",    "varservr.dll");
      initDLLLib(cliServer, MapLib,      "map",    "map.dll"     );
      initDLLLib(cliServer, EvttmpltLib, "tmplt",  "evttmplt.dll");
      initDLLLib(cliServer, EventLib,    "evt",    "event.dll"   );
      initDLLLib(cliServer, TriggerLib,  "trig",   "trig.dll"    );
      initDLLLib(cliServer, DisAsmLib,   "dasm",   getDasmDLLName(self)    );
      initDLLLib(cliServer, TraceLib,    "trc",    "trace.dll"   );
      initDLLLib(cliServer, HLBreakLib,  "bkpt",   "bkptexec.dll");
      initDLLLib(cliServer, CLIULibrary, "cliulib","cliulib.dll");
      initDLLLib(cliServer, PeriphLib,   "peri",   "peri.dll");
      initDLLLib(cliServer, ProcLib,     "proc",   "proc.dll");
      initDLLLib(cliServer, PvtestLib,   "pvtest", "pvtest.dll");
      $cliRegistered := #true;
    endif;
    processStartupFile(TheShellPresenter);
  endif;
}!!

/* Launch off the requested presenter */
Def launchCPUPresenter(self, rect, showVal)
{ new(ATParser, " "); /* reset text color */
  if TheCPUPresenter
  then
    Call BringWindowToTop(getHWnd(TheCPUPresenter));
    show( TheCPUPresenter, SW_SHOWNORMAL ) ; /* redisplay */
  else
    showWaitCurs() ;
    requireWithPath(ProcLib,     "proc.dll"      );
    requireWithPath(SymbolLib,   "symbl.dll"     );
    requireWithPath(AddressLib,  "addr.dll"      );
    requireWithPath(MemServLib,  "mem.dll"       );
    requireWithPath(CpuLib,      "cpu.dll"       );
    requireWithPath(LoaderLib,   "loader.dll"    );
    requireWithPath(StackLib,    "stkservr.dll"  );
    requireWithPath(VarServLib,  "varservr.dll"  );
    requireWithPath(MapLib,      "map.dll"       );
    showOldCurs();
    /* Open the CPU Presenter; retrieve location and show status if
       they are not already specified. */
    TheCPUPresenter :=
      openWithPosAndState(CpuPresenter,
        rect cor getPos(self, "TheCPUPresenter"),
        showVal cor getShowState(self, "TheCPUPresenter"));
  endif;
}
!!

Def launchEventPresenter(self, rect, showVal)
{
  if (getSystemType(ProcLibClass$Inst) = PROC_POWERSCOPE)
    ^nil;
  endif;

  showWaitCurs();
  if (exists(EventWindow) cand TheEventPresenter) then
  /* if TheEventPresenter */
    Call BringWindowToTop(getHWnd(TheEventPresenter));
    show(TheEventPresenter, SW_SHOWNORMAL);
  else
    /* Open the Event Presenter; retrieve location and show status if
       they are not already specified. */
    TheEventPresenter :=
      openWithPosAndState(EventWindow,
        rect cor getPos(self, "TheEventPresenter"),
        showVal cor getShowState(self, "TheEventPresenter"));
  endif;
  showOldCurs();
}!!

/* Launch off the requested presenter */
Def launchMemoryPresenter(self, rect, showVal |
      index memoryObject label theDlg)
{
  if ($windowRestoreCompleted) cand (memoryWindowCount(MultMem) > 0) then
    if not(theDlg := new(MultMemDlg)) cor
       (runModal(theDlg, DLG_MULTMEM, ThePort) <> IDOK) then
      /* 04/25/94 - Nghia
      ** exit if cannot create dialog object or user press CANCEL 
      */
      ^nil;
    endif;
    memoryObject := getSelection(theDlg);  /* handle, or nil if new selected */
  endif; 

  if memoryObject
    Call BringWindowToTop(getHWnd(memoryObject));
    show(memoryObject, SW_SHOWNORMAL);
  else
    /* retrieve available index; nil indicates error */
    if not(index := getFreeMemoryWindowIndex(MultMem)) then
      errorBox("Error", "too many memory windows");
      ^nil;
    endif;
    
    showWaitCurs();
    requireWithPath(ProcLib,     "proc.dll"      );
    requireWithPath(SymbolLib,   "symbl.dll"     );
    requireWithPath(AddressLib,  "addr.dll"      );
    requireWithPath(MemServLib,  "mem.dll"       );
    requireWithPath(CpuLib,      "cpu.dll"       );
    requireWithPath(LoaderLib,   "loader.dll"    );
    requireWithPath(StackLib,    "stkservr.dll"  );
    requireWithPath(VarServLib,  "varservr.dll"  );
    requireWithPath(MapLib,      "map.dll"       );
    requireWithPath(EvttmpltLib, "evttmplt.dll"  );
    requireWithPath(EventLib,    "event.dll"     );
    requireWithPath(DisAsmLib,   getDasmDLLName(self));
    requireWithPath(HLBreakLib,  "bkptexec.dll"  );
    requireWithPath(TraceLib,    "trace.dll"     );
    showOldCurs();
    /* Open the Memory Presenter; retrieve location and show status if
       they are not already specified. */
    label := memoryWindowLabel(MultMem, index);
    memoryObject := openWithPosAndState(MPresenter,
                rect cor getPos(self, label),
                showVal cor getShowState(self, label),
                index);
    if memoryObject then
      allocateMemoryIndex(MultMem, index, memoryObject);
    endif;
  endif;
}
!!

/* Launch off the requested presenter */
Def launchOverlayPresenter(self, aWindow | Dlg result)
{
  showWaitCurs() ;
  requireWithPath(ProcLib,     "proc.dll"      );
  requireWithPath(SymbolLib,   "symbl.dll"     );
  requireWithPath(AddressLib,  "addr.dll"      );
  requireWithPath(MemServLib,  "mem.dll"       );
  requireWithPath(CpuLib,      "cpu.dll"       );
  requireWithPath(LoaderLib,   "loader.dll"    );
  requireWithPath(StackLib,    "stkservr.dll"  );
  requireWithPath(VarServLib,  "varservr.dll"  );
  requireWithPath(MapLib,      "map.dll"       );
  showOldCurs();
  result := mapPresenter(MapLibClass$Inst);
  displayError(ErrorTextLibClass$Inst, result, FORCE_POPUP);
}
!!

/* Launch off the requested presenter */
Def launchSourcePresenter(self, rect, showVal  | modInfo)
{
  if TheSourcePresenter then
    Call BringWindowToTop(getHWnd(TheSourcePresenter));
    show( TheSourcePresenter, SW_SHOWNORMAL );
    ^nil;
  else
    /* If emulation is running then do not perform the requested function */
    if not(processorHalted?(HLBrkRootLibClass$Inst)) then
      ^nil;
    endif;
    /* Open the Source Presenter; retrieve location and show status if
       they are not already specified. */
    TheSourcePresenter :=
      openWithPosAndState(SourcePresenter,
        rect cor getPos(self, "TheSourcePresenter"),
        showVal cor getShowState(self, "TheSourcePresenter"));
  endif;
}
!!

/* Launch off the requested presenter */
Def launchStackPresenter(self, rect, showVal | curLoadFile, stkBase, stkSize)
{
  if TheStackPresenter
  then
    Call BringWindowToTop(getHWnd(TheStackPresenter));
    show(TheStackPresenter, SW_SHOWNORMAL); /* redisplay */
  else
    /* Open the Stack Presenter; retrieve location and show status if
       they are not already specified. */
    TheStackPresenter :=
      openWithPosAndState(StackPresenter,
        rect cor getPos(self, "TheStackPresenter"),
        showVal cor getShowState(self, "TheStackPresenter"));
  endif;
}

!!

/* Launch off the requested presenter
 *  rect, showVal and showVal are not used.  They are placholders
 *  to make this method match the standard launch syntax.
 */
Def launchStatusPresenter(self, rect, showVal)
{
  if TheStatusPresenter then
    bringToTop(TheStatusPresenter);
    show(TheStatusPresenter, SW_SHOWNORMAL);
  else
    showWaitCurs() ;
    requireWithPath(ProcLib,     "proc.dll"      );
    requireWithPath(SymbolLib,   "symbl.dll"     );
    requireWithPath(AddressLib,  "addr.dll"      );
    requireWithPath(MemServLib,  "mem.dll"       );
    requireWithPath(CpuLib,      "cpu.dll"       );
    requireWithPath(LoaderLib,   "loader.dll"    );
    requireWithPath(StackLib,    "stkservr.dll"  );
    requireWithPath(VarServLib,  "varservr.dll"  );
    requireWithPath(MapLib,      "map.dll"       );
    requireWithPath(EvttmpltLib, "evttmplt.dll"  );
    requireWithPath(EventLib,    "event.dll"     );
    requireWithPath(DisAsmLib,   getDasmDLLName(self)    );
    requireWithPath(HLBreakLib,  "bkptexec.dll"  );
    requireWithPath(TraceLib,    "trace.dll"     );
    requireWithPath(TriggerLib,  "trig.dll"      );
/*  requireWithPath(SWATLib,   "swat.dll"    ); */
    showOldCurs();
    /* Open the Status Presenter; retrieve location and show status if
       they are not already specified. */
    TheStatusPresenter :=
      openWithPosAndState(StatusWindow,
        rect cor getPos(self, "TheStatusPresenter"),
        showVal cor getShowState(self, "TheStatusPresenter"));
  endif;
}
!!

/* Launch off the requested presenter */
Def launchTracePresenter(self, rect, showVal )
{
  if (getSystemType(ProcLibClass$Inst) = PROC_POWERSCOPE)
    ^nil;
  endif;

  if TheTracePresenter
  then
    Call BringWindowToTop(getHWnd(TheTracePresenter));
    show( TheTracePresenter, SW_SHOWNORMAL ) ;
  else
    /* Open the Trace Presenter; retrieve location and show status if
       they are not already specified. */
    /*
    */
    TheTracePresenter :=
      openWithPosAndState(TracePresenter,
        rect cor getPos(self, "TheTracePresenter"),
        showVal cor getShowState(self, "TheTracePresenter"));
    actionTraceStop(TheToolBarPresenter);
  endif ;
}
!!

/* Launch off the requested presenter */
Def launchTriggerPresenter(self, rect, showVal )
{
  if (getSystemType(ProcLibClass$Inst) = PROC_POWERSCOPE)
    ^nil;
  endif;
  if (exists(TriggerPresenter) cand TheTriggerPresenter) then
/*  if TheTriggerPresenter
  then */
    Call BringWindowToTop(getHWnd(TheTriggerPresenter));
    show( TheTriggerPresenter, SW_SHOWNORMAL ) ;
  else
    /* Open the Trigger Presenter; retrieve location and show status if
       they are not already specified. */
    TheTriggerPresenter :=
      openWithPosAndState(TriggerPresenter,
        rect cor getPos(self, "TheTriggerPresenter"),
        showVal cor getShowState(self, "TheTriggerPresenter"));
  endif;
}
!!

/* Launch off the requested presenter */
Def launchVariablePresenter(self, rect, showVal )
{
  if TheVariablePresenter
  then
    Call BringWindowToTop(getHWnd(TheVariablePresenter));
    show( TheVariablePresenter, SW_SHOWNORMAL ) ;
  else
    /* Open the Variable Presenter; retrieve location and show status if
       they are not already specified. */
    TheVariablePresenter :=
      openWithPosAndState(VarPresenter,
        rect cor getPos(self, "TheVariablePresenter"),
        showVal cor getShowState(self, "TheVariablePresenter"),
        nil, nil);
  endif ;
}
!!

/* PRIVATE -
  @@@ Load all dlls we will ever need - only way to prevent getting a UAE
  when we exit.  @@@
 */
Def loadAllDlls(self)
{
  /* decide processor */
  showWaitCurs() ;
  if not(requireWithPath(SpaLib,      "spa.dll"       )) then ^nil; endif;
  if not(requireWithPath(SymbolLib,   "symbl.dll"     )) then ^nil; endif;
  if not(requireWithPath(AddressLib,  "addr.dll"      )) then ^nil; endif;
  if not(requireWithPath(MemServLib,  "mem.dll"       )) then ^nil; endif;
  if not(requireWithPath(CpuLib,      "cpu.dll"       )) then ^nil; endif;
  if not(requireWithPath(LoaderLib,   "loader.dll"    )) then ^nil; endif;
  if not(requireWithPath(StackLib,    "stkservr.dll"  )) then ^nil; endif;
  if not(requireWithPath(VarServLib,  "varservr.dll"  )) then ^nil; endif;
  if not(requireWithPath(MapLib,      "map.dll"       )) then ^nil; endif;
  if not(requireWithPath(EvttmpltLib, "evttmplt.dll"  )) then ^nil; endif;
  if not(requireWithPath(EventLib,    "event.dll"     )) then ^nil; endif;
  if not(requireWithPath(DisAsmLib,   getDasmDLLName(self))) then ^nil; endif;
  if not(requireWithPath(AsmLib,      getDasmDLLName(self))) then ^nil; endif;
  if not(requireWithPath(DqLib,       getDasmDLLName(self))) then ^nil; endif;
  if not(requireWithPath(HLBreakLib,  "bkptexec.dll"  )) then ^nil; endif;
  if not(requireWithPath(TraceLib,    "trace.dll"     )) then ^nil; endif;
  if not(requireWithPath(TriggerLib,  "trig.dll"      )) then ^nil; endif;
  if not(requireWithPath(PeriphLib,   "peri.dll"      )) then ^nil; endif;
  if not(requireWithPath(Abi186Lib,   "abi186.dll"    )) then ^nil; endif;
  if not(requireWithPath(Sds2abiLib,  "sds2abi.dll"   )) then ^nil; endif;
  if not(requireWithPath(PvtestLib,   "pvtest.dll"    )) then ^nil; endif;
  /*  requireWithPath(SWATLib,   "swat.dll"    ); */
  showOldCurs();
}
!!

/* PRIVATE -
  Load the system Bootup DLLs (Malloc, Event Notification, ShareData)
 */
Def loadBootDlls(self | commOK)
{
  showWaitCurs();
  /* modify requireWithpath to requireWithpath1 */
   /* borlands container class library's DLLs */
  if not(requireWithPath1(Bc30rtlLib,"bc30rtl.dll")) then ^nil; endif;
  if not(requireWithPath1(TclassLib,"tclass31.dll")) then ^nil; endif;
  if not(requireWithPath1( CLIULibrary,  "cliulib.dll" )) then ^nil; endif;
  /* Bring up login screen and hook up windows hook proc for F1 */
  openLogin(CLIULibraryClass$Inst);
  f1hook(CLIULibraryClass$Inst);
  if not(requireWithPath( ErrorTextLib, "errtext.dll" )) then ^nil; endif;
  if not(requireWithPath( EvNoteLib,    "enLib.dll"   )) then ^nil; endif;
  if not(requireWithPath( MallocLib,    "malloc.dll"  )) then ^nil; endif;
  showOldCurs();
  commOK := loadComm( self );
  if (not(commOK)) then
    ^commOK;
  endif;
  if not(requireWithPath( ProcLib,      "proc.dll"    )) then ^nil; endif;
  ^commOK;
}
!!

/* PRIVATE */
Def loadComm(self | commOK commType typeString libString)
{
  /* decide ether/rs232 */
  commType := getProfileEntry(TheProfileInfoObj, "comm", "type", "");
  
  commOK := 0;  /* initially OK */
  libString := "wscom.dll";  /* default library string */

  showWaitCurs();
  
  /* load the proper dll; serial -> wscom.dll, pcnfs -> ws_pcnfs.dll,
     lanserver -> wslansvr.dll, parallel -> wspar.dll */
  if (commType cand (size(commType) = 1))
    typeString := asLowerCase(commType[0]);
    libString := "wscom.dll";  /* default library string */
    select
      case typeString = "pcnfs"
        libString := "ws_pcnfs.dll";
      endCase
      case typeString = "lanserver"
        commOK := requireWithPath(WinsockLib, "winsock.dll");
        loop /* allow repaint */
        while not(checkMessage())
        begin /* skip */ ;
        endLoop;
        libString := "wslansvr.dll";
      endCase
      case typeString = "microtekparallel"
        libString := "wspar.dll";
      endCase
    endSelect;
  endif;
  
  if commOK
    commOK := requireWithPath(CommLib, libString);
    loop /* allow repaint */
    while not(checkMessage())
    begin /* skip */ ;
    endLoop;
  endif;

  /* talk to probe */
  if commOK
    commOK := requireWithPath(SharedDataLib, "sdserver.dll");
    loop /* allow repaint */
    while not(checkMessage())
    begin /* skip */ ;
    endLoop;
  endif;

  if commOK
    commOK := requireWithPath(HLBrkRootLib,  "bkroot.dll");
    loop /* allow repaint */
    while not(checkMessage())
    begin /* skip */ ;
    endLoop;
  endif;

  showOldCurs();

  ^commOK  /* nil if bad */
}
!!

/* DEBUG -- clean up all statics */
Def nilifyPowerViews(self)
{
  nilifyPresenterVars( self ) ;
  nilifyLibs( ProvidedLibs ) ;

  ^#ok
}
!!

/* DEBUG -- set presenter global vars to value nil */
Def nilifyPresenterVars(self)
{
  do(presenterNames(self),
  {using(sym)
    add( Actor, sym, nil ) ;
  });

  TheStatusPresenter := nil ;
  $cliRegistered := nil ;
  nilifyWindowsMenu( self ) ;

  ^#ok
}
!!

/* PRIVATE */
Def nilifyWindowsMenu(self)
{
  $windowsMenu := nil ;
}
!!

/* Launch the toolbar */
Def openButtonBarPresenter(self, parent |
      AOK version major minor sizeRect cOld)
{
  if TheToolBarPresenter
    Call BringWindowToTop(getHWnd(TheToolBarPresenter));
    show(TheToolBarPresenter, SW_NORMAL); /* redisplay */
    ^0;
  endif;

  $inBoot? := #true;
  removeAllLibs(self);

  nilifyPowerViews(self);
  TheProfileInfoObj := new(ProfileControl);
  if findIni(ProfileControl) = ERR_FILE_NOT_FOUND
    /* return the error and don't come up */
    Call MessageBox(0, asciiz("Can't find MP186.INI"),
      asciiz("FATAL: MP/SLD"), MB_OK);
    removeAllLibs(ProvidedLibs);
    setOkToClose(TheApp:late);
    ^nil  /* fatal */
  endif;

  /* get Windows version; split of major and minor version number */
  version := Call GetVersion();
  major := version bitAnd 0xff;
  minor := (version bitAnd 0xff00) / 0xff;

  /* croak if this is not at least windows 3.1 (major 3, minor 10) */
  if (major < 3) cor (major = 3 cand minor < 10)
    Call MessageBox(0, asciiz("Must use Windows version 3.1 or above"),
      asciiz("FATAL: MP/SLD"), MB_OK);
    removeAllLibs(ProvidedLibs);
    setOkToClose(TheApp:late);
    ^nil  /* fatal */
  endif;

  if not(loadBootDlls(self))
    backOutOfPV(self);
    ^nil  /* die */
  endif;

  TheHelpObject := new(Help, "pwrviews.hlp");

  if not(isValidProcessor(ProcLibClass$Inst))
    backOutOfPV(self);
    ^nil  /* die */
  endif;

  AOK := loadAllDlls(self);
  /* take down boot logo */
  if (CLIULibraryClass$Inst)
    closeLogin(CLIULibraryClass$Inst);
  endif;

  if not(AOK)
    backOutOfPV(self);
    ^nil  /* die */
  endif;

  $inBoot? := nil;
   if (getDemo(ProcLibClass$Inst)==1)
     TheToolBarPresenter := newStyle(ButtonWindow, parent, nil,
       "MP/SLD Toolbar (Demo Version)", nil, nil, nil);
  else
    TheToolBarPresenter := newStyle(ButtonWindow, parent, nil,
      "MP/SLD Toolbar", nil, nil, nil);
    saveToolBarHandle(CLIULibraryClass$Inst, getHWnd(TheToolBarPresenter));
  endif;

  /* retrieve saved position */
  sizeRect := getPos(PreLauncher, "TheToolBarPresenter");
  if sizeRect
    /* use position only if it is the proper proportion */
    cOld := cRect(TheToolBarPresenter);
    if ((width(sizeRect) = width(cOld)) cand (height(sizeRect) = height(cOld)))
      setCRect(TheToolBarPresenter, sizeRect);
      moveWindow(TheToolBarPresenter);
    endif;
  endif;

  show(TheToolBarPresenter, SW_NORMAL);
  
  initEvents(TheToolBarPresenter);

  bringToTop(TheToolBarPresenter);
  registerF1Help(CLIULibraryClass$Inst, HI_ENTRY_TOOLBAR,
    getHWnd(TheToolBarPresenter), HELP_ENTRY_TOOLBAR);
  setMainWnd(SharedDataLibClass$Inst);
  if not(initTimer(TheToolBarPresenter))
    Call MessageBox(0,
      asciiz("Timer is required for communication timeout"
      + CR_LF + "Close CLOCK or other timer users"),
      asciiz("FATAL: MP/SLD can't get timer"), MB_OK);
    if (CLIULibraryClass$Inst)
      f1unhook(CLIULibraryClass$Inst);
    endif;
    removeAllLibs(ProvidedLibs);
    setOkToClose(TheApp:late);
    ^nil;  /* fatal */
  endif;
  init(SourcePresenter);
  launchStatusPresenter(self, nil, nil);
}!!

/* PUBLIC */
Def presenterActionVec(self | aVec)
{
 aVec := new( Array, size($presenterDict) ) ;
 keysDo($presenterDict,
 {using(key | info) /* info is #(menuStr, noSelect?, Order, action) */
   info := at($presenterDict, key) ;
   aVec[info[2]] := info[3] ;  /* aVec[Order] := actionSymbol */
 });

 ^aVec
}
!!

/* PUBLIC */
Def presenterNames(self | aSet)
{
 aSet :=
  newWithCompareBlock(
                      SortedSet,
                       size($presenterDict),
                       {using(a,b) (a < b);}/* sort block */
                      );
 keysDo($presenterDict,
 {using(key)
   add( aSet, key ) ;
 });

 ^aSet
}
!!

/* DEBUG -- print window handles to transcript */
Def printWindowHandles(self)
{
  do(presenterNames(self),
  {using(sym)
    if Actor[sym]
    then
      printNewLine( asString(sym) + ": " + asHex(hWnd(Actor[sym])) ) ;
    endif ;
  });

  ^#ok
}!!

/* PUBLIC
   Free all libraries--call before loading libs to clean up from last time.
*/
Def removeAllLibs(self | libList, libHandle)
{
  libList := /* DLLs in load order */
    tuple( "bc30rtl.dll",  "tclass31.dll", 
           "cliulib.dll",  "errtext.dll",  "enLib.dll",
           "malloc.dll",   "wscom.dll",
           "winsock.dll",
           "ws_pcnfs.dll", "wslansvr.dll", "wspar.dll",
           "sdserver.dll", "bkroot.dll",   "proc.dll",
           "symbl.dll",    "addr.dll",     "mem.dll",
           "cpu.dll",      "loader.dll",
           "stkservr.dll", "varservr.dll", "map.dll",
           "evttmplt.dll", "event.dll",    "bkptexec.dll",
           "trace.dll",    "dasm16.dll",   "dasm32.dll", "dasm86.dll",
           "trig.dll", 
           "peri.dll",
           "abi186.dll", "sds2abi.dll",
           "pvtest.dll",
           "spa.dll"
           /* , "swat.dll" */
         );
  /* remove all libraries in reverse order */
  do(reverse(libList),
      {using(libName)
       /* Decrement library usage count to 0 to remove it */ 
       loop
       while ((libHandle := Call GetModuleHandle(asciiz(libName))) <> 0)
       begin
         Call FreeLibrary(libHandle);
       endLoop;      
     }
   );
}!!

/* PRIVATE
   If startup file exists in the boot directory, bring up
   the CLI Presenter and execute the script.
*/
Def startupScript(self | scriptFile fileDesc fileString commandTextCol state)
{
  /* exit if no startup file specified */
  fileString :=
    getProfileEntry(TheProfileInfoObj, "InitScript", "script", "");
  if not(fileString)
    ^nil;
  endif;
  
  fileString := fileString[0];
  scriptFile := new(File);
  setName(scriptFile, fileString);
  if (fileDesc := open(scriptFile, 0))  /* file exists? */
    close(scriptFile);
    startupFileNameSave(self, fileString);
    launchCLIPresenter(self, nil, SW_NORMAL);
    
    /* now that shell has run, minimize it if minimized in save file */  
    state := getShowState(PreLauncher, "TheShellPresenter");
    if state = SW_MINIMIZE
      show(TheShellPresenter, state);
    endif;
  endif;
}!!

/* PRIVATE */
Def windowsMenu(self)
{
 if not($windowsMenu)
 then
   $windowsMenu := Call CreateMenu();
/*@@Call AppendMenu($windowsMenu, 0, 0x3FFF, name := asciiz("Bogus"));*/
 endif ;

 ^$windowsMenu
}
!!

/* Initialize menu contents when pull-down menu requested. */
Def WM_INITMENU(self, wp, lp | p items windowNames systemType menuText procType mt2Flag)
{
  if (getTraceModule(ProcLibClass$Inst) <> nil) then
     mt2Flag:=1;
  else
     mt2Flag:=0;
  endif;
  /* kill off the old */
  p := windowsMenu(self);
  items := Call GetMenuItemCount(p);
  do(items,
     {using(ignored)
        Call DeleteMenu(p, 0, MF_BYPOSITION);
     });

  windowNames := presenterNames(self);

  systemType := getSystemType(ProcLibClass$Inst);
  /* Create Windows menu with all menu items in windowNames */
  do(windowNames,
     {using(key | info)
        info := at($presenterDict, key) ;
        if not(info[1]) /* selectable? */
       
           /* append "..." to Memory menu text if needed */
           if (info[2] = 5)
              menuText := info[0] + memoryMenuSuffix(MultMem);
           else
              menuText := info[0];
           endif;
         
           Call AppendMenu(p, 0, info[2]+0x3F00, asciiz(menuText));
           if ( (systemType = PROC_POWERSCOPE) cand
               ( (info[2] = 6) cor (info[2] = 7) cor (info[2] = 9)) )
               Call EnableMenuItem(p, info[2]+0x3f00, MF_GRAYED);
           endif;
           if ((mt2Flag=0) and ((info[2]=6) or (info[2]=7) or (info[2]=12))) then
               Call EnableMenuItem(p, info[2]+0x3f00, MF_GRAYED);
           endif;
           
           /* get processor type */
           procType := getSpecificProcessor(ProcLibClass$Inst);
           
           /* gray out peri menu selection if cx or sx */
           select
             case procType = PROC_I80386CX_TB cor
                  procType = PROC_I80386SX_TB cor
                  procType = PROC_I80386CX_SH cor
                  procType = PROC_I80386SX_SH is
                    if (info[2] = 12)
                      Call EnableMenuItem(p, info[2]+0x3f00, MF_GRAYED);
                    endif;
             endCase;
           endSelect;
        endif;
     });
}!!

now(PreLauncher)!!

$startupScriptFileName := nil;
/* PreLauncher Class Initialization Code */
$presenterDict := new(Dictionary, 13);
                      /* symbol = varID */
/* DICTIONARY definition */
/* presenterDict = [presenter->#(menuStr, noSelect?, Order, action, show)],... */

add( $presenterDict, #TheSourcePresenter,
     tuple("&Source",        nil, 0, #launchSourcePresenter, nil) ) ;
add( $presenterDict, #TheStackPresenter,
     tuple("S&tack",         nil, 1, #launchStackPresenter, nil) ) ;
add( $presenterDict, #TheVariablePresenter,
     tuple("&Variable",      nil, 2, #launchVariablePresenter, nil ) ) ;
add( $presenterDict, #TheStatusPresenter,
     tuple("Stat&us",        nil, 3, #launchStatusPresenter, nil ) ) ;
add( $presenterDict, #TheCPUPresenter,
     tuple("&CPU",           nil, 4, #launchCPUPresenter, nil ) ) ;
add( $presenterDict, #TheMemoryPresenter,
     tuple("&Memory",        nil, 5, #launchMemoryPresenter, nil ) ) ;
add( $presenterDict, #TheTracePresenter,
     tuple("Tr&ace",         nil, 6, #launchTracePresenter, nil ) ) ;
add( $presenterDict, #TheTriggerPresenter,
     tuple("Tri&gger",       nil, 7, #launchTriggerPresenter, nil ) ) ;
add( $presenterDict, #TheShellPresenter,
     tuple("S&hell",         nil, 8, #launchCLIPresenter, nil ) ) ;
/* add( $presenterDict, #TheEventPresenter,
     tuple("&Event",         nil, 9, #launchEventPresenter, nil ) ) ; */
add( $presenterDict, #TheBreakpointPresenter,
     tuple("&Breakpoint",    nil, 9, #launchBreakPresenter, nil ) ) ;
add( $presenterDict, #TheToolBarPresenter,
     tuple("T&oolBar",  nil, 10, #launchButtonBarPresenter, nil ) ) ;
add( $presenterDict, #ThePeripheralPresenter,
     tuple("&Peripheral",  nil, 11, #launchPeripheralPresenter, nil ) ) ;
add( $presenterDict, #TheSpaPresenter,
     tuple("Spa(&z)",  nil, 12, #launchSpaPresenter, nil ) ) ;     
